"""
Tests for NEXIUM core application.
Author: Juste Elysée MALANDILA
"""

import pytest
from nexium import Nexium


def test_app_creation():
    """Test creating a NEXIUM app."""
    app = Nexium(title="Test App", version="1.0.0")
    
    assert app.title == "Test App"
    assert app.version == "1.0.0"
    assert app.router is not None


def test_route_decorator():
    """Test route decorator."""
    app = Nexium()
    
    @app.get("/test")
    async def test_route():
        return {"message": "test"}
    
    assert len(app.router.routes) == 1
    route = app.router.routes[0]
    assert route.path == "/test"
    assert "GET" in route.methods


def test_multiple_methods():
    """Test route with multiple methods."""
    app = Nexium()
    
    @app.route("/test", methods=["GET", "POST"])
    async def test_route():
        return {"message": "test"}
    
    route = app.router.routes[0]
    assert "GET" in route.methods
    assert "POST" in route.methods


def test_path_parameters():
    """Test route with path parameters."""
    app = Nexium()
    
    @app.get("/users/{id}")
    async def get_user(id: int):
        return {"id": id}
    
    route = app.router.routes[0]
    assert "{id}" in route.path


def test_middleware():
    """Test middleware decorator."""
    app = Nexium()
    
    @app.middleware()
    async def test_middleware(request, call_next):
        response = await call_next(request)
        return response
    
    assert len(app.middlewares) == 1


def test_cors_enabled():
    """Test CORS middleware enabled by default."""
    app = Nexium(cors=True)
    
    assert len(app._middleware_stack) > 0


def test_cors_disabled():
    """Test CORS can be disabled."""
    app = Nexium(cors=False)
    
    assert len(app._middleware_stack) == 0


def test_debug_mode():
    """Test debug mode."""
    app = Nexium(debug=True)
    assert app.debug is True
    
    app = Nexium(debug=False)
    assert app.debug is False
