"""
NEXIUM Desktop App Example
Transform your web app into a native desktop application!

Requirements:
    pip install nexium[desktop]
"""

from nexium import Nexium, Model, DesktopApp, DesktopAPI

# Create app
app = Nexium(
    title="My Desktop App",
    version="1.0.0",
    description="A NEXIUM desktop application",
    debug=True,
)

# Desktop API for window control
desktop_api = DesktopAPI()


# Models
class Task(Model):
    """Task model."""
    __table__ = "tasks"


# Routes
@app.get("/")
async def home():
    """Home page."""
    return {
        "message": "Welcome to NEXIUM Desktop App! 🖥️",
        "mode": "desktop",
        "framework": "NEXIUM",
        "features": [
            "Native window",
            "Cross-platform (Windows, macOS, Linux)",
            "Full web app capabilities",
            "Desktop APIs (file dialogs, notifications, etc.)"
        ]
    }


@app.get("/tasks")
async def list_tasks():
    """List all tasks."""
    tasks = await Task.all()
    return {"tasks": [t.dict() for t in tasks]}


@app.post("/tasks")
async def create_task(title: str, description: str = ""):
    """Create a new task."""
    task = Task(title=title, description=description, completed=False)
    await task.save()
    return {"message": "Task created", "task": task.dict()}


# Desktop-specific endpoints
@app.get("/window/info")
async def window_info():
    """Get window information."""
    return {
        "desktop_mode": desktop_api.available,
        "message": "Desktop API is available!" if desktop_api.available else "Running in web mode"
    }


@app.post("/window/minimize")
async def minimize_window():
    """Minimize the window."""
    if desktop_api.available:
        desktop_api.minimize()
        return {"status": "minimized"}
    return {"error": "Not in desktop mode"}, 400


@app.post("/window/maximize")
async def maximize_window():
    """Maximize the window."""
    if desktop_api.available:
        desktop_api.maximize()
        return {"status": "maximized"}
    return {"error": "Not in desktop mode"}, 400


@app.post("/window/fullscreen")
async def toggle_fullscreen():
    """Toggle fullscreen mode."""
    if desktop_api.available:
        desktop_api.toggle_fullscreen()
        return {"status": "fullscreen toggled"}
    return {"error": "Not in desktop mode"}, 400


@app.get("/window/dialog/open")
async def open_file_dialog():
    """Show file open dialog."""
    if not desktop_api.available:
        return {"error": "Not in desktop mode"}, 400
    
    file_path = desktop_api.show_dialog(
        dialog_type="open",
        file_types=("Text Files (*.txt)", "All Files (*.*)")
    )
    
    if file_path:
        return {"file": file_path}
    return {"message": "No file selected"}


@app.get("/window/dialog/save")
async def save_file_dialog():
    """Show file save dialog."""
    if not desktop_api.available:
        return {"error": "Not in desktop mode"}, 400
    
    file_path = desktop_api.show_dialog(
        dialog_type="save",
        save_filename="untitled.txt",
        file_types=("Text Files (*.txt)", "All Files (*.*)")
    )
    
    if file_path:
        return {"file": file_path}
    return {"message": "Save cancelled"}


if __name__ == "__main__":
    # Option 1: Run as desktop app (RECOMMENDED)
    print("🖥️  Running as DESKTOP APPLICATION...")
    print("💡 To run as web app instead, use: app.run()")
    
    desktop = DesktopApp(
        app,
        title="NEXIUM Desktop App",
        width=1000,
        height=700,
        resizable=True,
        min_size=(600, 400),
    )
    desktop.run(debug=True)
    
    # Option 2: Run as regular web app
    # app.run(port=8000, reload=True)
