"""
OMNIWEB Admin Example - Auto-generated admin panel
"""

from omniweb import OmniWeb, Model
from omniweb.admin import AutoAdmin, AdminConfig

app = OmniWeb(title="Admin Example", debug=True)

# Define models
class User(Model):
    """User model."""
    __table__ = "users"

class Post(Model):
    """Post model."""
    __table__ = "posts"

class Comment(Model):
    """Comment model."""
    __table__ = "comments"

# Configure admin
admin_config = AdminConfig(
    title="My Admin Panel",
    theme="light",  # or "dark", "auto"
    language="en",
    items_per_page=25,
    enable_export=True,
    enable_charts=True,
)

# Create admin
admin = AutoAdmin(app, config=admin_config, mount_path="/admin")

# Register models
admin.register(User)
admin.register(Post)
admin.register(Comment)

# API routes
@app.get("/")
async def home():
    """Homepage with admin info."""
    return {
        "message": "Welcome to OMNIWEB Admin Example!",
        "admin": {
            "url": "/admin",
            "models": list(admin.models.keys()),
            "features": [
                "CRUD operations",
                "Modern UI",
                "Charts & Analytics",
                "Export to CSV/Excel",
                "Bulk actions",
                "Search & Filter",
                "Mobile responsive",
                "Dark mode support"
            ]
        }
    }

@app.get("/users")
async def list_users():
    """List all users."""
    users = await User.all()
    return {"users": [u.dict() for u in users]}

@app.post("/users")
async def create_user(name: str, email: str):
    """Create new user."""
    user = User(name=name, email=email)
    await user.save()
    return {"user": user.dict()}, 201


if __name__ == "__main__":
    print("""
╔════════════════════════════════════════╗
║  📊 OMNIWEB Admin Example             ║
╚════════════════════════════════════════╝

Admin Panel Features:
  ✅ Auto-generated CRUD
  ✅ Modern UI (Tailwind + React)
  ✅ Charts & Analytics
  ✅ Export to CSV/Excel
  ✅ Bulk actions
  ✅ Search & Filter
  ✅ Mobile responsive
  ✅ Dark mode

Admin available at:
  http://localhost:8000/admin

Models registered:
  - User
  - Post
  - Comment

Admin endpoints:
  GET    /admin                    - Admin home
  GET    /admin/{model}            - List view
  GET    /admin/{model}/{id}       - Detail view
  POST   /admin/{model}            - Create
  PUT    /admin/{model}/{id}       - Update
  DELETE /admin/{model}/{id}       - Delete
  GET    /admin/{model}/export     - Export data
  GET    /admin/stats              - Statistics

Starting server...
    """)
    
    app.run(host="127.0.0.1", port=8000, reload=True)
