"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const cdk = require("@aws-cdk/core");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured.
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *    Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time.
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value.
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value.
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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