"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 *
 * @stability stable
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
    LustreDeploymentType["PERSISTENT_2"] = "PERSISTENT_2";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 * @stability stable
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_fsx_LustreFileSystemProps(props);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_b = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _b === void 0 ? void 0 : _b.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     *
     * @stability stable
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_fsx_FileSystemAttributes(attrs);
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1 && deploymentType !== LustreDeploymentType.PERSISTENT_2) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1/PERSISTENT_2 deployment types, received: ' + deploymentType);
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_1) {
            if (![50, 100, 200].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB for PERSISTENT_1 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_2) {
            if (![125, 250, 500, 1000].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 125, 250, 500 or 1000 MB/s/TiB for PERSISTENT_2 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "@aws-cdk/aws-fsx.LustreFileSystem", version: "1.143.0" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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