![Github](https://img.shields.io/github/tag/essembeh/rapid-api-client.svg)
![PyPi](https://img.shields.io/pypi/v/rapid-api-client.svg)
![Python](https://img.shields.io/pypi/pyversions/rapid-api-client.svg)
![CI](https://github.com/essembeh/python-helloworld/actions/workflows/poetry.yml/badge.svg)


# Rapid Api Client

Library to rapidly develop API clients based on [Pydantic](https://docs.pydantic.dev/) and [Httpx](https://www.python-httpx.org/) using *decorators* and *annotations*.

This project is largely inspired by [FastAPI](https://fastapi.tiangolo.com/).


# Usage 

Install the project

```sh
pip install rapid-api-client
```

Declare your API using decorators and annotations (the method does not need any code, it will be generated by the decorator)

```python
class GithubIssuesApi(RapidApi):

    @get("/repos/{owner}/{repo}/issues", response_class=RootModel[List[Issue]])
    def list_issues(self, owner: Annotated[str, Path()], repo: Annotated[str, Path()]): ...

```

Use it 

```python
    for issue in GithubIssuesApi().list_issues("essembeh", "rapid-api-client").root:
        print("Issue:", issue)
```

# Features

## Http method

Any HTTP method can be used with `http` decorator

```python
class MyApi(RapidApi)

    @http("/anything") # default is GET
    def get(self): ...

    @http("/anything", method="POST")
    def post(self): ...

    @http("/anything", method="DELETE)
    def delete(self): ...
```

Convenient decorators are available like `get`, `post`, `delete`, `put`, `patch`

```python
class MyApi(RapidApi)

    @get("/anything")
    def get(self): ...

    @post("/anything")
    def post(self): ...

    @delete("/anything")
    def delete(self): ...
```


## Response class

By default methods return a `httpx.Response` object and the http return code is not tested (you have to call `resp.raise_for_status()` if you need to ensure the response is OK).

But you can also specify a *Pydantic model class* to automatically parse the response.

> Note: When a `response_class` is given, the `raise_for_status()` is always called to ensure the http response is OK

```python
class User(BaseModel): ...

class MyApi(RapidApi)

    # this method return a httpx.Response
    @get("/user/me")
    def get_user_resp(self): ...

    # this method returns a User class
    @get("/user/me", response_class=User)
    def get_user(self): ...
```


## Path parameters

Like `fastapi` you can use your method arguments to build the api path to call.

```python
class MyApi(RapidApi)

    @get("/user/{user_id}")
    def get_user(self, user_id: Annotated[int, Path()]): ...

    # Path parameters dans have a default value
    @get("/user/{user_id}")
    def get_user(self, user_id: Annotated[int, Path()] = 1): ...

```

## Query parameters

You can add `query parameters` to your request using the `Query` annotation.

```python
class MyApi(RapidApi)

    @get("/issues")
    def get_issues(self, sort: Annotated[str, Query()]): ...

    # Query parameters can have a default value
    @get("/issues")
    def get_issues_default(self, sort: Annotated[str, Query()] = "date"): ...

    # Query parameters can have an alias to change the key in the http request
    @get("/issues")
    def get_issues_alias(self, sort: Annotated[str, Query(alias="sort-by")] = "date"): ...
```


## Header parameter

You can add `headers` to your request using the `Header` annotation.

```python
class MyApi(RapidApi)

    @get("/issues")
    def get_issues(self, version: Annotated[str, Header()]): ...

    # Headers can have a default value
    @get("/issues")
    def get_issues(self, version: Annotated[str, Header()] = "1"): ...

    # Headers can have an alias to change the key in the http request
    @get("/issues")
    def get_issues(self, version: Annotated[str, Header(alias="X-API-Version")] = "1"): ...
```

## Body parameter

