"use strict";
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'creating a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // WHEN
        new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_10_X],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Content: {
                S3Bucket: stack.resolve(bucket.bucketName),
                S3Key: 'ObjectKey',
            },
            CompatibleRuntimes: ['nodejs10.x'],
        }));
        test.done();
    },
    'granting access to a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const layer = new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_10_X],
        });
        // WHEN
        layer.addPermission('GrantUsage-123456789012', { accountId: '123456789012' });
        layer.addPermission('GrantUsage-o-123456', { accountId: '*', organizationId: 'o-123456' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '123456789012',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '*',
            OrganizationId: 'o-123456',
        }));
        test.done();
    },
    'creating a layer with no runtimes compatible'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // THEN
        test.throws(() => new lambda.LayerVersion(stack, 'LayerVersion', { code, compatibleRuntimes: [] }), /supports no runtime/);
        test.done();
    },
    'asset metadata is added to the cloudformation resource'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
        // WHEN
        new lambda.LayerVersion(stack, 'layer', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'layer-code')),
        });
        // THEN
        assert_1.expect(assert_1.canonicalizeTemplate(assert_1.SynthUtils.toCloudFormation(stack))).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Metadata: {
                'aws:asset:path': 'asset.Asset1Hash',
                'aws:asset:property': 'Content',
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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