"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        props.retryAttempts !== undefined && cdk.withResolved(props.retryAttempts, (attempts) => {
            if (attempts < 0 || attempts > 10000) {
                throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${attempts}`);
            }
        });
        props.parallelizationFactor !== undefined && cdk.withResolved(props.parallelizationFactor, (factor) => {
            if (factor < 1 || factor > 10) {
                throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${factor}`);
            }
        });
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB or Kinesis stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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