"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = false;
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias. Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options
     */
    addAutoScaling(options) {
        var _a;
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_a = options.minCapacity) !== null && _a !== void 0 ? _a : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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