"""Parse bel xls output files"""

import logging

logger = logging.getLogger('pygaps')
import re
from itertools import product

import dateutil.parser
import xlrd

_RSPACE_REGEX = re.compile(r'\s+')

_FIELDS = {
    'sample:': {
        'text': ['comment1'],
        'name': 'material',
        'row': 0,
        'column': 2,
        'type': 'string'
    },
    'adsorbate': {
        'text': ['adsorptive'],
        'name': 'adsorbate',
        'row': 0,
        'column': 2,
        'type': 'string'
    },
    'temperature': {
        'text': ['adsorption temp'],
        'name': 'temperature',
        'row': 0,
        'column': 2,
        'type': 'number'
    },
    'user': {
        'text': ['comment2'],
        'name': 'user',
        'row': 0,
        'column': 2,
        'type': 'string'
    },
    'date': {
        'text': ['date of measurement'],
        'name': 'date',
        'row': 0,
        'column': 2,
        'type': 'date'
    },
    'sample mass': {
        'text': ['sample weight'],
        'name': 'mass',
        'row': 0,
        'column': 2,
        'type': 'number'
    },
    'comments': {
        'text': ['comment3'],
        'name': 'comment',
        'row': 0,
        'column': 2,
        'type': 'string'
    },
    'isotherm data': {
        'text': ['no'],
        'type': 'isotherm data',
        'labels': {
            'No': 'measurement',
            'pi/': 'pressure_internal',
            'pe/': 'pressure',
            'pe2/': 'pressure2',
            'p0/': 'pressure_saturation',
            'p/p0': 'pressure_relative',
            'Va/': 'loading',
            'na/': 'loading',
        }
    },
    'primary data': {
        'text': ['primary data'],
        'type': 'error',
        'row': 1,
        'column': 0,
        'name': 'errors'
    },
    'cell_value': {
        'text': [],
        'header': {
            'row': 0
        },
        'datapoints': {
            'row': 1
        }
    }
}


def read_bel_report(path):
    """
    Parse an xls file generated by micromeritics software.

    Parameters
    ----------
    path: str
        The location of an xls file generated by a belsorp instrument.

    Returns
    -------
    dict
        A dictionary containing report information.
    """
    workbook = xlrd.open_workbook(path, encoding_override='latin-1')
    sheet = workbook.sheet_by_name('AdsDes')
    meta = {}
    data = {}
    errors = []
    for row, col in product(range(sheet.nrows), range(sheet.ncols)):
        cell_value = str(sheet.cell(row, col).value).lower()
        try:
            field = next(
                v for k, v in _FIELDS.items()
                if any([cell_value.startswith(n) for n in v.get('text')])
            )
        except StopIteration:
            continue
        if field['type'] == 'number':
            val = sheet.cell(row + field['row'], col + field['column']).value
            meta[field['name']] = val
        if field['type'] == 'date':
            day = sheet.cell(row + field['row'], col + field['column']).value
            time = sheet.cell(
                row + 1 + field['row'], col + field['column']
            ).value
            meta[field['name']] = _handle_date(sheet, day + time)
        elif field['type'] == 'string':
            val = sheet.cell(row + field['row'], col + field['column']).value
            meta[field['name']] = _handle_string(val)
        elif field['type'] == 'isotherm data':
            (ads_start, ads_end, des_start,
             des_end) = _find_datapoints(sheet, row, col)

            for i, item in enumerate(_get_data_labels(sheet, row, col)):
                ads_points = [
                    sheet.cell(r, i).value for r in range(ads_start, ads_end)
                ]
                des_points = [
                    sheet.cell(r, i).value for r in range(des_start, des_end)
                ]
                _assign_data(item, field, meta, data, ads_points, des_points)
        elif field['type'] == 'error':
            errors += _get_errors(sheet, row, col)
    if errors:
        meta['errors'] = errors

    _check(meta, data, path)

    # Set extra metadata
    meta['apparatus'] = 'BEL'
    meta['date'] = dateutil.parser.parse(meta['date']).isoformat()
    meta['loading_key'] = 'loading'
    meta['pressure_key'] = 'pressure'
    meta['other_keys'] = sorted([
        a for a in data if a not in ['loading', 'pressure', 'branch']
    ])
    meta['pressure_mode'] = 'absolute'
    meta['loading_basis'] = 'molar'
    meta['material_basis'] = 'mass'

    return meta, data


def _handle_date(sheet, val):
    """
    Convert date to string.

    Input is a cell of type 'date'.
    """
    if val:
        return xlrd.xldate.xldate_as_datetime(val, sheet.book.datemode
                                              ).strftime("%Y-%m-%d %H:%M:%S")
    else:
        return None


def _handle_string(val):
    """
    Replaces Comments: and any newline found.

    Input is a cell of type 'string'.
    """
    return val.replace('Comments: ', '').replace('\r\n', ' ')


def _get_data_labels(sheet, row, col):
    """Locate all column labels for data collected during the experiment."""
    final_column = col
    header_row = _FIELDS['cell_value']['header']['row']
    # Abstract this sort of thing
    header = re.sub(
        _RSPACE_REGEX, '',
        sheet.cell(row + header_row, final_column).value
    )
    while any(
        header.startswith(label)
        for label in _FIELDS['isotherm data']['labels']
    ):
        final_column += 1
        header = re.sub(
            _RSPACE_REGEX, '',
            sheet.cell(row + header_row, final_column).value
        )
    return [
        re.sub(_RSPACE_REGEX, '',
               sheet.cell(row + header_row, i).value)
        for i in range(col, final_column)
    ]


def _find_datapoints(sheet, row, col):
    """Return start and stop points for adsorption and desorption."""
    rowc = _FIELDS['cell_value']['datapoints']['row']

    # Check for adsorption branch
    if sheet.cell(row + rowc, col).value == 'ADS':
        ads_start_row = row + rowc + 1
        ads_final_row = ads_start_row

    point = sheet.cell(ads_final_row, col).value

    while point != 'DES':
        ads_final_row += 1
        point = sheet.cell(ads_final_row, col).value

    if sheet.cell(ads_final_row, col).value == 'DES':
        des_start_row = ads_final_row + 1
        des_final_row = des_start_row

    if des_final_row < sheet.nrows:

        point = sheet.cell(des_final_row, col).value

        while str(point).strip():
            des_final_row += 1
            if des_final_row < sheet.nrows:
                point = sheet.cell(des_final_row, col).value
            else:
                point = None

    return (ads_start_row, ads_final_row, des_start_row, des_final_row)


def _assign_data(item, field, meta, data, ads_points, des_points):
    """
    Add data to the data dictionary.

    For each column of data collected
    in a form depending on the label of the column.
    """
    name = next(f for f in field['labels'] if item.startswith(f))

    if field['labels'][name] == 'measurement':
        data['branch'] = [False] * \
            len(ads_points) + [True] * len(des_points)

    elif field['labels'][name] == 'loading':
        data['loading'] = ads_points + des_points
        for (u, c) in (
            ('/mmol', 'mmol'),
            ('/mol', 'mol'),
            ('/cm3(STP)', 'cm3(STP)'),
        ):
            if u in item:
                meta['loading_unit'] = c
        for (u, c) in (
            ('g-1', 'g'),
            ('kg-1', 'kg'),
        ):
            if u in item:
                meta['material_unit'] = c

    elif field['labels'][name] == 'pressure':
        data['pressure'] = ads_points + des_points
        for (u, c) in (
            ('/mmHg', 'torr'),
            ('/torr', 'torr'),
            ('/kPa', 'kPa'),
            ('/bar', 'bar'),
        ):
            if u in item:
                meta['pressure_unit'] = c

    elif field['labels'][name] in [
        'pressure2',
        'pressure_saturation',
        'pressure_relative',
        'pressure_internal',
    ]:
        data[field['labels'][name]] = ads_points + des_points

    else:
        raise ValueError(
            f"Label name '{field['labels'][name]}' not recognized."
        )


def _get_errors(sheet, row, col):
    """
    Look for all cells that contain errors.

    (are below a cell labelled primary data).
    """
    field = _FIELDS['primary data']
    val = sheet.cell(row + field['row'], col + field['column']).value
    if not val:
        return []
    final_row = row + field['row']
    error = sheet.cell(final_row, col + field['column']).value
    while error:
        final_row += 1
        error = sheet.cell(final_row, col + field['column']).value
    return [
        sheet.cell(i, col + field['column']).value
        for i in range(row + field['row'], final_row)
    ]


def _check(meta, data, path):
    """
    Check keys in data and logs a warning if a key is empty.

    Also logs a warning for errors found in file.
    """
    if 'loading' in data:
        empties = (k for k, v in data.items() if not v)
        for empty in empties:
            logger.info(f"No data collected for {empty} in file {path}.")
    if 'errors' in meta:
        logger.warning('\n'.join(meta['errors']))
