"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const ka = require("@aws-cdk/aws-kinesisanalytics");
const logs = require("@aws-cdk/aws-logs");
const core = require("@aws-cdk/core");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /**
     * (experimental) Implement the convenience {@link IApplication.addToPrincipalPolicy} method.
     *
     * @experimental
     */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        this.applicationName = attrs.applicationName;
    }
}
/**
 * (experimental) The L2 construct for Flink Kinesis Data Applications.
 *
 * @experimental
 * @resource AWS::KinesisAnalyticsV2::Application
 */
class Application extends ApplicationBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, { physicalName: props.applicationName });
        validation_1.validateFlinkApplicationProps(props);
        this.role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        const resource = new ka.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: (_c = props.snapshotsEnabled) !== null && _c !== void 0 ? _c : true,
                },
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new ka.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * (experimental) Import an existing Flink application defined outside of CDK code by applicationName.
     *
     * @experimental
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn, applicationName });
    }
    /**
     * (experimental) Import an existing application defined outside of CDK code by applicationArn.
     *
     * @experimental
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const applicationName = core.Stack.of(scope).parseArn(applicationArn).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${applicationArn}) must include resource name`);
        }
        return new Import(scope, id, { applicationArn, applicationName });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink.Application", version: "1.117.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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