"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends constructs_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        this.addResourceMethod(namedShadowResource, props, 'things/{thingName}/shadow?name={shadowName}', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        if (props.apiGatewayCreateApiKey === true) {
            // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
            defaults.addCfnSuppressRules(apiMethod, [
                {
                    id: "W59",
                    reason: "When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE",
                },
            ]);
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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