use anyhow::Result;
use std::time::Duration;

/// 连接池配置
#[derive(Clone, Debug)]
pub struct PoolConfig {
    pub idle_timeout: Option<Duration>,
    pub max_idle_per_host: usize,
    pub max_size: Option<u32>,
}

impl Default for PoolConfig {
    fn default() -> Self {
        Self {
            // 减少空闲超时，避免使用已被服务器关闭的连接
            // 很多服务器的 keep-alive 超时较短（Apache 5-15s, Nginx 75s）
            // 设置为 30s 可以在大多数情况下避免"瞬间连接失败"
            idle_timeout: Some(Duration::from_secs(30)),
            // 限制每主机最大空闲连接数，避免资源浪费和状态管理复杂性
            // 对于大多数应用场景，10 个连接足够
            max_idle_per_host: 10,
            max_size: None,
        }
    }
}

impl PoolConfig {
    /// 验证配置的有效性
    pub fn validate(&self) -> Result<()> {
        if let Some(max) = self.max_size {
            if max == 0 {
                return Err(anyhow::anyhow!("pool_max_size cannot be 0"));
            }
        }
        Ok(())
    }

    /// 应用到 wreq ClientBuilder
    pub fn apply(&self, builder: wreq::ClientBuilder) -> wreq::ClientBuilder {
        let mut builder = builder.pool_max_idle_per_host(self.max_idle_per_host);

        if let Some(timeout) = self.idle_timeout {
            builder = builder.pool_idle_timeout(timeout);
        }
        if let Some(max) = self.max_size {
            builder = builder.pool_max_size(max);
        }

        builder
    }
}
