use anyhow::Result;
use std::time::Duration;

/// HTTP/2 配置
#[derive(Clone, Debug)]
pub struct Http2Config {
    /// PING 间隔 - 设置 HTTP/2 PING 帧发送间隔以保持连接活跃
    /// None 表示禁用 keep-alive (默认)
    pub keep_alive_interval: Option<Duration>,

    /// PING 超时 - 接收 PING 确认的超时时间
    /// 如果在超时内未收到确认，连接将被关闭
    /// 默认 20 秒
    pub keep_alive_timeout: Duration,

    /// 空闲时是否发送 PING
    /// false: 仅在有活动流时发送 PING (默认)
    /// true: 即使没有活动流也发送 PING
    pub keep_alive_while_idle: bool,

    /// 连接级初始窗口大小 (字节)
    /// 默认 5MB (wreq 默认值)
    pub initial_connection_window_size: Option<u32>,

    /// 流级初始窗口大小 (字节)
    /// 默认 2MB (wreq 默认值)
    pub initial_stream_window_size: Option<u32>,

    /// 是否启用自适应窗口 (覆盖上面两个窗口设置)
    /// 默认 false
    pub adaptive_window: bool,

    /// 最大并发流数量
    pub max_concurrent_streams: Option<u32>,

    /// 最大帧大小
    pub max_frame_size: Option<u32>,

    /// 最大发送缓冲区大小
    pub max_send_buffer_size: Option<usize>,

    /// 初始最大发送流数量
    pub initial_max_send_streams: Option<usize>,

    /// 最大头列表大小 (字节)
    pub max_header_list_size: Option<u32>,

    /// HPACK 压缩表大小 (字节)
    pub header_table_size: Option<u32>,

    /// 是否启用服务器推送 (默认禁用)
    pub enable_push: Option<bool>,
}

impl Default for Http2Config {
    fn default() -> Self {
        Self {
            // 优化: 启用 keep-alive 以保持长连接健康
            // 关键修复: 30秒 PING 间隔防止服务器关闭空闲连接
            keep_alive_interval: Some(Duration::from_secs(30)),
            keep_alive_timeout: Duration::from_secs(20),
            // ✅ 关键优化: 设为 true 以在空闲时也发送 PING
            // 这可以防止服务器因空闲超时关闭连接，避免"瞬间连接失败"
            keep_alive_while_idle: true,

            // ✅ 使用 Chrome 140+ 的窗口大小 (来源: wreq-util)
            // 连接级窗口: 15MB，流级窗口: 6MB
            initial_connection_window_size: Some(15728640), // 15MB (Chrome 140+)
            initial_stream_window_size: Some(6291456),      // 6MB (Chrome 140+)
            adaptive_window: false,  // 不启用自适应 (使用固定值更稳定)

            // 优化: 增加并发流和缓冲区
            max_concurrent_streams: Some(1000),  // Chrome 默认值
            max_frame_size: Some(16384),         // 16KB (HTTP/2 规范默认)
            max_send_buffer_size: Some(1048576), // 1MB

            // 使用现代浏览器优化值
            initial_max_send_streams: Some(100),
            max_header_list_size: Some(262144),  // 256KB (Chrome 默认)
            header_table_size: Some(65536),      // 64KB (Chrome 默认)
            enable_push: Some(false),            // 禁用服务器推送
        }
    }
}

impl Http2Config {
    /// 验证配置的有效性
    pub fn validate(&self) -> Result<()> {
        if let Some(interval) = self.keep_alive_interval {
            if interval.is_zero() {
                return Err(anyhow::anyhow!(
                    "keep_alive_interval cannot be zero, use None to disable"
                ));
            }
        }

        if self.keep_alive_timeout.is_zero() {
            return Err(anyhow::anyhow!(
                "keep_alive_timeout cannot be zero"
            ));
        }

        if let Some(size) = self.initial_connection_window_size {
            if size == 0 {
                return Err(anyhow::anyhow!(
                    "initial_connection_window_size cannot be zero"
                ));
            }
        }

        if let Some(size) = self.initial_stream_window_size {
            if size == 0 {
                return Err(anyhow::anyhow!(
                    "initial_stream_window_size cannot be zero"
                ));
            }
        }

        Ok(())
    }

    /// 应用到 wreq ClientBuilder
    pub fn apply(&self, builder: wreq::ClientBuilder) -> wreq::ClientBuilder {
        let mut http2_builder = wreq::http2::Http2Options::builder();

        // Keep-alive 配置
        if let Some(interval) = self.keep_alive_interval {
            http2_builder = http2_builder
                .keep_alive_interval(interval)
                .keep_alive_timeout(self.keep_alive_timeout)
                .keep_alive_while_idle(self.keep_alive_while_idle);
        }

        // 窗口大小配置
        if !self.adaptive_window {
            if let Some(size) = self.initial_connection_window_size {
                http2_builder = http2_builder.initial_connection_window_size(size);
            }
            if let Some(size) = self.initial_stream_window_size {
                http2_builder = http2_builder.initial_window_size(size);
            }
        } else {
            http2_builder = http2_builder.adaptive_window(true);
        }

        // 其他配置
        if let Some(max) = self.max_concurrent_streams {
            http2_builder = http2_builder.max_concurrent_streams(max);
        }

        if let Some(size) = self.max_frame_size {
            http2_builder = http2_builder.max_frame_size(size);
        }

        if let Some(size) = self.max_send_buffer_size {
            http2_builder = http2_builder.max_send_buf_size(size);
        }

        // 新增参数
        if let Some(size) = self.initial_max_send_streams {
            http2_builder = http2_builder.initial_max_send_streams(size);
        }

        if let Some(size) = self.max_header_list_size {
            http2_builder = http2_builder.max_header_list_size(size);
        }

        if let Some(size) = self.header_table_size {
            http2_builder = http2_builder.header_table_size(size);
        }

        if let Some(enabled) = self.enable_push {
            http2_builder = http2_builder.enable_push(enabled);
        }

        builder.http2_options(http2_builder.build())
    }
}
