#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
基础使用示例 - never_primp
演示基本的 HTTP 请求方法和参数
"""

import sys
from never_primp import Client

# 设置 UTF-8 输出（Windows 兼容性）
if sys.platform == 'win32':
    import codecs
    sys.stdout = codecs.getwriter('utf-8')(sys.stdout.buffer, 'strict')
    sys.stderr = codecs.getwriter('utf-8')(sys.stderr.buffer, 'strict')

print("=" * 80)
print("基础使用示例")
print("=" * 80)

# ============================================================================
# 1. 创建客户端
# ============================================================================
print("\n[示例 1] 创建客户端")
print("-" * 80)

# 基础客户端
client = Client()
print("✓ 基础客户端创建成功")

# 带浏览器模拟的客户端
client_chrome = Client(impersonate="chrome_142")
print("✓ Chrome 142 模拟客户端创建成功")

# ============================================================================
# 2. GET 请求
# ============================================================================
print("\n[示例 2] GET 请求")
print("-" * 80)

# 简单 GET
response = client.get("https://httpbin.org/get")
print(f"状态码: {response.status_code}")
print(f"内容长度: {len(response.content)} bytes")

# 带查询参数的 GET
response = client.get("https://httpbin.org/get", params={
    "key1": "value1",
    "key2": "value2",
    "中文": "测试"
})
data = response.json()
print(f"查询参数: {data['args']}")

# ============================================================================
# 3. POST 请求
# ============================================================================
print("\n[示例 3] POST 请求")
print("-" * 80)

# JSON POST
response = client.post("https://httpbin.org/post", json={
    "name": "张三",
    "age": 25,
    "skills": ["Python", "Rust"]
})
data = response.json()
print(f"发送的 JSON: {data['json']}")

# Form POST
response = client.post("https://httpbin.org/post", data={
    "username": "user123",
    "password": "pass456"
})
data = response.json()
print(f"表单数据: {data['form']}")

# ============================================================================
# 4. 自定义 Headers
# ============================================================================
print("\n[示例 4] 自定义 Headers")
print("-" * 80)

response = client.get("https://httpbin.org/headers", headers={
    "User-Agent": "MyCustomAgent/1.0",
    "X-Custom-Header": "CustomValue",
    "Authorization": "Bearer token123"
})
data = response.json()
print(f"User-Agent: {data['headers'].get('User-Agent')}")
print(f"X-Custom-Header: {data['headers'].get('X-Custom-Header')}")

# ============================================================================
# 5. Cookies 管理
# ============================================================================
print("\n[示例 5] Cookies 管理")
print("-" * 80)

# 设置 cookie
client.cookies["session_id"] = "abc123"
client.cookies["user_token"] = "xyz789"

response = client.get("https://httpbin.org/cookies")
data = response.json()
print(f"发送的 Cookies: {data['cookies']}")

# ��看响应的 cookies
response = client.get("https://httpbin.org/cookies/set?new_cookie=test_value")
print(f"客户端 Cookies: {dict(client.cookies)}")

# ============================================================================
# 6. 其他 HTTP 方法
# ============================================================================
print("\n[示例 6] 其他 HTTP 方法")
print("-" * 80)

# PUT
response = client.put("https://httpbin.org/put", json={"data": "updated"})
print(f"PUT 状态码: {response.status_code}")

# DELETE
response = client.delete("https://httpbin.org/delete")
print(f"DELETE 状态码: {response.status_code}")

# PATCH
response = client.patch("https://httpbin.org/patch", json={"field": "patched"})
print(f"PATCH 状态码: {response.status_code}")

# HEAD
response = client.head("https://httpbin.org/get")
print(f"HEAD 状态码: {response.status_code}, Content-Length: {response.headers.get('Content-Length')}")

# ============================================================================
# 7. 响应处理
# ============================================================================
print("\n[示例 7] 响应处理")
print("-" * 80)

response = client.get("https://httpbin.org/json")

# 状态码
print(f"状态码: {response.status_code}")

# Headers
print(f"Content-Type: {response.headers.get('Content-Type')}")

# JSON 解析
data = response.json()
print(f"JSON 键: {list(data.keys())}")

# 文本内容
print(f"文本长度: {len(response.text)} 字符")

# 二进制内容
print(f"字节长度: {len(response.content)} bytes")

# ============================================================================
# 8. URL 编码
# ============================================================================
print("\n[示例 8] URL 编码")
print("-" * 80)

# 自动编码特殊字符
response = client.get("https://httpbin.org/get", params={
    "query": "hello world",
    "filter": "a&b=c",
    "中文参数": "测试值"
})
data = response.json()
print(f"URL: {data['url']}")
print(f"参数: {data['args']}")

# ============================================================================
# 9. 重定向处理
# ============================================================================
print("\n[示例 9] 重定向处理")
print("-" * 80)

# 默认跟随重定向
response = client.get("https://httpbin.org/redirect/2")
print(f"最终 URL: {response.url}")
print(f"状态码: {response.status_code}")

# 限制重定向次数
client_limited_redirect = Client(max_redirects=1)
try:
    # 尝试访问需要 2 次重定向的 URL（会失败）
    response = client_limited_redirect.get("https://httpbin.org/redirect/3")
    print(f"重定向限制测试: {response.status_code}")
except Exception as e:
    print(f"超出重定向限制: {type(e).__name__}")

# ============================================================================
# 10. 错误处理
# ============================================================================
print("\n[示例 10] 错误处理")
print("-" * 80)

# 超时错误
try:
    response = client.get("https://httpbin.org/delay/10", timeout=2.0)
except Exception as e:
    print(f"超时错误: {type(e).__name__}")

# HTTP 错误
response = client.get("https://httpbin.org/status/404")
if response.status_code == 404:
    print(f"处理 404 错误: {response.status_code}")

response = client.get("https://httpbin.org/status/500")
if response.status_code >= 500:
    print(f"处理服务器错误: {response.status_code}")

print("\n" + "=" * 80)
print("基础示例完成")
print("=" * 80)
