"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolIdentityProviderGoogle = exports.UserPoolIdentityProviderFacebook = exports.UserPoolIdentityProviderAmazon = exports.ProviderAttribute = exports.UserPoolIdentityProvider = void 0;
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
/**
 * User pool third-party identity providers.
 */
class UserPoolIdentityProvider {
    /**
     * Import an existing UserPoolIdentityProvider.
     */
    static fromProviderName(scope, id, providerName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.providerName = providerName;
            }
        }
        return new Import(scope, id);
    }
    constructor() { }
}
exports.UserPoolIdentityProvider = UserPoolIdentityProvider;
/**
 * An attribute available from a third party identity provider.
 */
class ProviderAttribute {
    constructor(attributeName) {
        this.attributeName = attributeName;
    }
    /**
     * Use this to specify an attribute from the identity provider that is not pre-defined in the CDK.
     *
     * @param attributeName the attribute value string as recognized by the provider.
     */
    static other(attributeName) {
        return new ProviderAttribute(attributeName);
    }
}
exports.ProviderAttribute = ProviderAttribute;
/**
 * The user id attribute provided by Amazon.
 */
ProviderAttribute.AMAZON_USER_ID = new ProviderAttribute('user_id');
/**
 * The email attribute provided by Amazon.
 */
ProviderAttribute.AMAZON_EMAIL = new ProviderAttribute('email');
/**
 * The name attribute provided by Amazon.
 */
ProviderAttribute.AMAZON_NAME = new ProviderAttribute('name');
/**
 * The postal code attribute provided by Amazon.
 */
ProviderAttribute.AMAZON_POSTAL_CODE = new ProviderAttribute('postal_code');
/**
 * The user id attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_ID = new ProviderAttribute('id');
/**
 * The birthday attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_BIRTHDAY = new ProviderAttribute('birthday');
/**
 * The email attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_EMAIL = new ProviderAttribute('email');
/**
 * The name attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_NAME = new ProviderAttribute('name');
/**
 * The first name attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_FIRST_NAME = new ProviderAttribute('first_name');
/**
 * The last name attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_LAST_NAME = new ProviderAttribute('last_name');
/**
 * The middle name attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_MIDDLE_NAME = new ProviderAttribute('middle_name');
/**
 * The gender attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_GENDER = new ProviderAttribute('gender');
/**
 * The locale attribute provided by Facebook.
 */
ProviderAttribute.FACEBOOK_LOCALE = new ProviderAttribute('locale');
/**
 * The name attribute provided by Google.
 */
ProviderAttribute.GOOGLE_NAMES = new ProviderAttribute('names');
/**
 * The gender attribute provided by Google.
 */
ProviderAttribute.GOOGLE_GENDER = new ProviderAttribute('gender');
/**
 * The birthday attribute provided by Google.
 */
ProviderAttribute.GOOGLE_BIRTHDAYS = new ProviderAttribute('birthdays');
/**
 * The birthday attribute provided by Google.
 */
ProviderAttribute.GOOGLE_PHONE_NUMBERS = new ProviderAttribute('phoneNumbers');
/**
 * The email attribute provided by Google.
 */
ProviderAttribute.GOOGLE_EMAIL = new ProviderAttribute('email');
/**
 * The name attribute provided by Google.
 */
ProviderAttribute.GOOGLE_NAME = new ProviderAttribute('name');
/**
 * The email attribute provided by Google.
 */
ProviderAttribute.GOOGLE_PICTURE = new ProviderAttribute('picture');
/**
 * The email attribute provided by Google.
 */
ProviderAttribute.GOOGLE_GIVEN_NAME = new ProviderAttribute('given_name');
/**
 * The email attribute provided by Google.
 */
ProviderAttribute.GOOGLE_FAMILY_NAME = new ProviderAttribute('family_name');
/**
 * Options to integrate with the various social identity providers.
 */
class UserPoolIdentityProviderBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        props.userPool.registerIdentityProvider(this);
    }
    /**
     *
     */
    configureAttributeMapping() {
        if (!this.props.attributeMapping) {
            return undefined;
        }
        let mapping = {};
        mapping = Object.entries(this.props.attributeMapping)
            .filter(([k, _]) => k !== 'custom') // 'custom' handled later separately
            .reduce((agg, [k, v]) => {
            return { ...agg, [attr_names_1.StandardAttributeNames[k]]: v.attributeName };
        }, mapping);
        if (this.props.attributeMapping.custom) {
            mapping = Object.entries(this.props.attributeMapping.custom).reduce((agg, [k, v]) => {
                return { ...agg, [k]: v.attributeName };
            }, mapping);
        }
        if (Object.keys(mapping).length === 0) {
            return undefined;
        }
        return mapping;
    }
}
/**
 * Represents a identity provider that integrates with 'Login with Amazon'.
 *
 * @resource AWS::Cognito::UserPoolIdentityProvider
 */
class UserPoolIdentityProviderAmazon extends UserPoolIdentityProviderBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        const scopes = (_a = props.scopes) !== null && _a !== void 0 ? _a : ['profile'];
        const resource = new cognito_generated_1.CfnUserPoolIdentityProvider(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            providerName: 'LoginWithAmazon',
            providerType: 'LoginWithAmazon',
            providerDetails: {
                client_id: props.clientId,
                client_secret: props.clientSecret,
                authorize_scopes: scopes.join(' '),
            },
            attributeMapping: super.configureAttributeMapping(),
        });
        this.providerName = super.getResourceNameAttribute(resource.ref);
    }
}
exports.UserPoolIdentityProviderAmazon = UserPoolIdentityProviderAmazon;
/**
 * Represents a identity provider that integrates with 'Facebook Login'.
 *
 * @resource AWS::Cognito::UserPoolIdentityProvider
 */
class UserPoolIdentityProviderFacebook extends UserPoolIdentityProviderBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        const scopes = (_a = props.scopes) !== null && _a !== void 0 ? _a : ['public_profile'];
        const resource = new cognito_generated_1.CfnUserPoolIdentityProvider(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            providerName: 'Facebook',
            providerType: 'Facebook',
            providerDetails: {
                client_id: props.clientId,
                client_secret: props.clientSecret,
                authorize_scopes: scopes.join(','),
                api_version: props.apiVersion,
            },
            attributeMapping: super.configureAttributeMapping(),
        });
        this.providerName = super.getResourceNameAttribute(resource.ref);
    }
}
exports.UserPoolIdentityProviderFacebook = UserPoolIdentityProviderFacebook;
/**
 * Represents a identity provider that integrates with 'Google'.
 *
 * @resource AWS::Cognito::UserPoolIdentityProvider
 */
class UserPoolIdentityProviderGoogle extends UserPoolIdentityProviderBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        const scopes = (_a = props.scopes) !== null && _a !== void 0 ? _a : ['profile'];
        const resource = new cognito_generated_1.CfnUserPoolIdentityProvider(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            providerName: 'Google',
            providerType: 'Google',
            providerDetails: {
                client_id: props.clientId,
                client_secret: props.clientSecret,
                authorize_scopes: scopes.join(' '),
            },
            attributeMapping: super.configureAttributeMapping(),
        });
        this.providerName = super.getResourceNameAttribute(resource.ref);
    }
}
exports.UserPoolIdentityProviderGoogle = UserPoolIdentityProviderGoogle;
//# sourceMappingURL=data:application/json;base64,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