"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Attributes', () => {
    describe('mutable', () => {
        test('default', () => {
            // GIVEN
            const allAttributes = [
                new lib_1.StringAttribute(),
                new lib_1.NumberAttribute(),
                new lib_1.BooleanAttribute(),
                new lib_1.DateTimeAttribute(),
            ];
            // WHEN
            const bounds = allAttributes.map((attr) => attr.bind());
            // THEN
            bounds.forEach((bound) => {
                expect(bound.mutable).toBeUndefined();
            });
        });
        describe('mutable is set to true when specified', () => {
            // GIVEN
            const allTrueProps = {
                mutable: true,
            };
            const allAttributeTypes = [
                new lib_1.StringAttribute(allTrueProps),
                new lib_1.NumberAttribute(allTrueProps),
                new lib_1.BooleanAttribute(allTrueProps),
                new lib_1.DateTimeAttribute(allTrueProps),
            ];
            // WHEN
            const bounds = allAttributeTypes.map((attr) => attr.bind());
            // THEN
            bounds.forEach((bound) => {
                test(`in attribute of type ${bound.dataType}:`, () => {
                    expect(bound.mutable).toEqual(true);
                });
            });
        });
        describe('mutable is set to false', () => {
            // GIVEN
            const allFalseProps = {
                mutable: false,
            };
            const allAttributeTypes = [
                new lib_1.StringAttribute(allFalseProps),
                new lib_1.NumberAttribute(allFalseProps),
                new lib_1.BooleanAttribute(allFalseProps),
                new lib_1.DateTimeAttribute(allFalseProps),
            ];
            // WHEN
            const bounds = allAttributeTypes.map((attr) => attr.bind());
            // THEN
            bounds.forEach((bound) => {
                test(`in attribute of type ${bound.dataType}`, () => {
                    expect(bound.mutable).toEqual(false);
                });
            });
        });
    });
    describe('StringAttribute', () => {
        test('default', () => {
            // GIVEN
            const attr = new lib_1.StringAttribute();
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.dataType).toEqual('String');
            expect(bound.stringConstraints).toBeUndefined();
            expect(bound.numberConstraints).toBeUndefined();
        });
        test('specified constraints are recognized', () => {
            // GIVEN
            const attr = new lib_1.StringAttribute({ minLen: 10, maxLen: 60 });
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.stringConstraints).toEqual({
                minLen: 10,
                maxLen: 60,
            });
            expect(bound.numberConstraints).toBeUndefined();
        });
        test('throws error when crossing limits', () => {
            expect(() => new lib_1.StringAttribute({ minLen: -10 }))
                .toThrow(/minLen cannot be less than/);
            expect(() => new lib_1.StringAttribute({ maxLen: 5000 }))
                .toThrow(/maxLen cannot be greater than/);
        });
        test('validation is skipped when minLen or maxLen are tokens', () => {
            const stack = new core_1.Stack();
            const parameter = new core_1.CfnParameter(stack, 'Parameter', {
                type: 'Number',
            });
            expect(() => new lib_1.StringAttribute({ minLen: parameter.valueAsNumber }))
                .not.toThrow();
            expect(() => new lib_1.StringAttribute({ maxLen: parameter.valueAsNumber }))
                .not.toThrow();
        });
    });
    describe('NumberAttribute', () => {
        test('default', () => {
            // GIVEN
            const attr = new lib_1.NumberAttribute();
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.dataType).toEqual('Number');
            expect(bound.stringConstraints).toBeUndefined();
            expect(bound.numberConstraints).toBeUndefined();
        });
        test('specified constraints are recognized', () => {
            // GIVEN
            const attr = new lib_1.NumberAttribute({ min: 5, max: 600 });
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.numberConstraints).toEqual({
                min: 5,
                max: 600,
            });
            expect(bound.stringConstraints).toBeUndefined();
        });
    });
    describe('BooleanAttribute', () => {
        test('default', () => {
            // GIVEN
            const attr = new lib_1.BooleanAttribute();
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.dataType).toEqual('Boolean');
            expect(bound.stringConstraints).toBeUndefined();
            expect(bound.numberConstraints).toBeUndefined();
        });
    });
    describe('DateTimeAttribute', () => {
        test('default', () => {
            // GIVEN
            const attr = new lib_1.DateTimeAttribute();
            // WHEN
            const bound = attr.bind();
            // THEN
            expect(bound.dataType).toEqual('DateTime');
            expect(bound.stringConstraints).toBeUndefined();
            expect(bound.numberConstraints).toBeUndefined();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWF0dHIudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzZXItcG9vbC1hdHRyLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQW9EO0FBQ3BELGdDQUF3STtBQUV4SSxRQUFRLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO0lBRXBDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO1FBQ3ZCLElBQUksQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO1lBQ25CLFFBQVE7WUFDUixNQUFNLGFBQWEsR0FBdUI7Z0JBQ3hDLElBQUkscUJBQWUsRUFBRTtnQkFDckIsSUFBSSxxQkFBZSxFQUFFO2dCQUNyQixJQUFJLHNCQUFnQixFQUFFO2dCQUN0QixJQUFJLHVCQUFpQixFQUFFO2FBQ3hCLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxNQUFNLEdBQTRCLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBRSxDQUFDO1lBRWxGLE9BQU87WUFDUCxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7Z0JBQ3ZCLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDeEMsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyx1Q0FBdUMsRUFBRSxHQUFHLEVBQUU7WUFDckQsUUFBUTtZQUNSLE1BQU0sWUFBWSxHQUFHO2dCQUNuQixPQUFPLEVBQUUsSUFBSTthQUNkLENBQUM7WUFDRixNQUFNLGlCQUFpQixHQUF1QjtnQkFDNUMsSUFBSSxxQkFBZSxDQUFDLFlBQVksQ0FBQztnQkFDakMsSUFBSSxxQkFBZSxDQUFDLFlBQVksQ0FBQztnQkFDakMsSUFBSSxzQkFBZ0IsQ0FBQyxZQUFZLENBQUM7Z0JBQ2xDLElBQUksdUJBQWlCLENBQUMsWUFBWSxDQUFDO2FBQ3BDLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxNQUFNLEdBQTRCLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFFLENBQUM7WUFFdEYsT0FBTztZQUNQLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRTtnQkFDdkIsSUFBSSxDQUFDLHdCQUF3QixLQUFLLENBQUMsUUFBUSxHQUFHLEVBQUUsR0FBRyxFQUFFO29CQUNuRCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDdEMsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtZQUN2QyxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUc7Z0JBQ3BCLE9BQU8sRUFBRSxLQUFLO2FBQ2YsQ0FBQztZQUNGLE1BQU0saUJBQWlCLEdBQXVCO2dCQUM1QyxJQUFJLHFCQUFlLENBQUMsYUFBYSxDQUFDO2dCQUNsQyxJQUFJLHFCQUFlLENBQUMsYUFBYSxDQUFDO2dCQUNsQyxJQUFJLHNCQUFnQixDQUFDLGFBQWEsQ0FBQztnQkFDbkMsSUFBSSx1QkFBaUIsQ0FBQyxhQUFhLENBQUM7YUFDckMsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLE1BQU0sR0FBNEIsaUJBQWlCLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUUsQ0FBQztZQUV0RixPQUFPO1lBQ1AsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFO2dCQUN2QixJQUFJLENBQUMsd0JBQXdCLEtBQUssQ0FBQyxRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUU7b0JBQ2xELE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUN2QyxDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxHQUFHLEVBQUU7UUFDL0IsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7WUFDbkIsUUFBUTtZQUNSLE1BQU0sSUFBSSxHQUFHLElBQUkscUJBQWUsRUFBRSxDQUFDO1lBRW5DLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFFMUIsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3pDLE1BQU0sQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUNoRCxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDbEQsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1lBQ2hELFFBQVE7WUFDUixNQUFNLElBQUksR0FBRyxJQUFJLHFCQUFlLENBQUMsRUFBRSxNQUFNLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBRTdELE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFFMUIsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxPQUFPLENBQUM7Z0JBQ3RDLE1BQU0sRUFBRSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxFQUFFO2FBQ1gsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQ2xELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtZQUM3QyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxxQkFBZSxDQUFDLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztpQkFDL0MsT0FBTyxDQUFDLDRCQUE0QixDQUFDLENBQUM7WUFDekMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUkscUJBQWUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2lCQUNoRCxPQUFPLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUM5QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7WUFDbEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLFNBQVMsR0FBRyxJQUFJLG1CQUFZLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtnQkFDckQsSUFBSSxFQUFFLFFBQVE7YUFDZixDQUFDLENBQUM7WUFFSCxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxxQkFBZSxDQUFDLEVBQUUsTUFBTSxFQUFFLFNBQVMsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO2lCQUNuRSxHQUFHLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDakIsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUkscUJBQWUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQztpQkFDbkUsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ25CLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO1FBQy9CLElBQUksQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO1lBQ25CLFFBQVE7WUFDUixNQUFNLElBQUksR0FBRyxJQUFJLHFCQUFlLEVBQUUsQ0FBQztZQUVuQyxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUN6QyxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDaEQsTUFBTSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQ2xELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtZQUNoRCxRQUFRO1lBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUV2RCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUN0QyxHQUFHLEVBQUUsQ0FBQztnQkFDTixHQUFHLEVBQUUsR0FBRzthQUNULENBQUMsQ0FBQztZQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUNsRCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtRQUNoQyxJQUFJLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRTtZQUNuQixRQUFRO1lBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxzQkFBZ0IsRUFBRSxDQUFDO1lBRXBDLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFFMUIsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQzFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUNoRCxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDbEQsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7UUFDakMsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7WUFDbkIsUUFBUTtZQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksdUJBQWlCLEVBQUUsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBRTFCLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUMzQyxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDaEQsTUFBTSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQ2xELENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgQ2ZuUGFyYW1ldGVyLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQm9vbGVhbkF0dHJpYnV0ZSwgQ3VzdG9tQXR0cmlidXRlQ29uZmlnLCBEYXRlVGltZUF0dHJpYnV0ZSwgSUN1c3RvbUF0dHJpYnV0ZSwgTnVtYmVyQXR0cmlidXRlLCBTdHJpbmdBdHRyaWJ1dGUgfSBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnVXNlciBQb29sIEF0dHJpYnV0ZXMnLCAoKSA9PiB7XG5cbiAgZGVzY3JpYmUoJ211dGFibGUnLCAoKSA9PiB7XG4gICAgdGVzdCgnZGVmYXVsdCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBhbGxBdHRyaWJ1dGVzOiBJQ3VzdG9tQXR0cmlidXRlW10gPSBbXG4gICAgICAgIG5ldyBTdHJpbmdBdHRyaWJ1dGUoKSxcbiAgICAgICAgbmV3IE51bWJlckF0dHJpYnV0ZSgpLFxuICAgICAgICBuZXcgQm9vbGVhbkF0dHJpYnV0ZSgpLFxuICAgICAgICBuZXcgRGF0ZVRpbWVBdHRyaWJ1dGUoKSxcbiAgICAgIF07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGJvdW5kczogQ3VzdG9tQXR0cmlidXRlQ29uZmlnW10gPSBhbGxBdHRyaWJ1dGVzLm1hcCgoYXR0cikgPT4gYXR0ci5iaW5kKCkgKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgYm91bmRzLmZvckVhY2goKGJvdW5kKSA9PiB7XG4gICAgICAgIGV4cGVjdChib3VuZC5tdXRhYmxlKS50b0JlVW5kZWZpbmVkKCk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCdtdXRhYmxlIGlzIHNldCB0byB0cnVlIHdoZW4gc3BlY2lmaWVkJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGFsbFRydWVQcm9wcyA9IHtcbiAgICAgICAgbXV0YWJsZTogdHJ1ZSxcbiAgICAgIH07XG4gICAgICBjb25zdCBhbGxBdHRyaWJ1dGVUeXBlczogSUN1c3RvbUF0dHJpYnV0ZVtdID0gW1xuICAgICAgICBuZXcgU3RyaW5nQXR0cmlidXRlKGFsbFRydWVQcm9wcyksXG4gICAgICAgIG5ldyBOdW1iZXJBdHRyaWJ1dGUoYWxsVHJ1ZVByb3BzKSxcbiAgICAgICAgbmV3IEJvb2xlYW5BdHRyaWJ1dGUoYWxsVHJ1ZVByb3BzKSxcbiAgICAgICAgbmV3IERhdGVUaW1lQXR0cmlidXRlKGFsbFRydWVQcm9wcyksXG4gICAgICBdO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBib3VuZHM6IEN1c3RvbUF0dHJpYnV0ZUNvbmZpZ1tdID0gYWxsQXR0cmlidXRlVHlwZXMubWFwKChhdHRyKSA9PiBhdHRyLmJpbmQoKSApO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBib3VuZHMuZm9yRWFjaCgoYm91bmQpID0+IHtcbiAgICAgICAgdGVzdChgaW4gYXR0cmlidXRlIG9mIHR5cGUgJHtib3VuZC5kYXRhVHlwZX06YCwgKCkgPT4ge1xuICAgICAgICAgIGV4cGVjdChib3VuZC5tdXRhYmxlKS50b0VxdWFsKHRydWUpO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ211dGFibGUgaXMgc2V0IHRvIGZhbHNlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGFsbEZhbHNlUHJvcHMgPSB7XG4gICAgICAgIG11dGFibGU6IGZhbHNlLFxuICAgICAgfTtcbiAgICAgIGNvbnN0IGFsbEF0dHJpYnV0ZVR5cGVzOiBJQ3VzdG9tQXR0cmlidXRlW10gPSBbXG4gICAgICAgIG5ldyBTdHJpbmdBdHRyaWJ1dGUoYWxsRmFsc2VQcm9wcyksXG4gICAgICAgIG5ldyBOdW1iZXJBdHRyaWJ1dGUoYWxsRmFsc2VQcm9wcyksXG4gICAgICAgIG5ldyBCb29sZWFuQXR0cmlidXRlKGFsbEZhbHNlUHJvcHMpLFxuICAgICAgICBuZXcgRGF0ZVRpbWVBdHRyaWJ1dGUoYWxsRmFsc2VQcm9wcyksXG4gICAgICBdO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBib3VuZHM6IEN1c3RvbUF0dHJpYnV0ZUNvbmZpZ1tdID0gYWxsQXR0cmlidXRlVHlwZXMubWFwKChhdHRyKSA9PiBhdHRyLmJpbmQoKSApO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBib3VuZHMuZm9yRWFjaCgoYm91bmQpID0+IHtcbiAgICAgICAgdGVzdChgaW4gYXR0cmlidXRlIG9mIHR5cGUgJHtib3VuZC5kYXRhVHlwZX1gLCAoKSA9PiB7XG4gICAgICAgICAgZXhwZWN0KGJvdW5kLm11dGFibGUpLnRvRXF1YWwoZmFsc2UpO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnU3RyaW5nQXR0cmlidXRlJywgKCkgPT4ge1xuICAgIHRlc3QoJ2RlZmF1bHQnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYXR0ciA9IG5ldyBTdHJpbmdBdHRyaWJ1dGUoKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgYm91bmQgPSBhdHRyLmJpbmQoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGJvdW5kLmRhdGFUeXBlKS50b0VxdWFsKCdTdHJpbmcnKTtcbiAgICAgIGV4cGVjdChib3VuZC5zdHJpbmdDb25zdHJhaW50cykudG9CZVVuZGVmaW5lZCgpO1xuICAgICAgZXhwZWN0KGJvdW5kLm51bWJlckNvbnN0cmFpbnRzKS50b0JlVW5kZWZpbmVkKCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzcGVjaWZpZWQgY29uc3RyYWludHMgYXJlIHJlY29nbml6ZWQnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYXR0ciA9IG5ldyBTdHJpbmdBdHRyaWJ1dGUoeyBtaW5MZW46IDEwLCBtYXhMZW46IDYwIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBib3VuZCA9IGF0dHIuYmluZCgpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoYm91bmQuc3RyaW5nQ29uc3RyYWludHMpLnRvRXF1YWwoe1xuICAgICAgICBtaW5MZW46IDEwLFxuICAgICAgICBtYXhMZW46IDYwLFxuICAgICAgfSk7XG4gICAgICBleHBlY3QoYm91bmQubnVtYmVyQ29uc3RyYWludHMpLnRvQmVVbmRlZmluZWQoKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Rocm93cyBlcnJvciB3aGVuIGNyb3NzaW5nIGxpbWl0cycsICgpID0+IHtcbiAgICAgIGV4cGVjdCgoKSA9PiBuZXcgU3RyaW5nQXR0cmlidXRlKHsgbWluTGVuOiAtMTAgfSkpXG4gICAgICAgIC50b1Rocm93KC9taW5MZW4gY2Fubm90IGJlIGxlc3MgdGhhbi8pO1xuICAgICAgZXhwZWN0KCgpID0+IG5ldyBTdHJpbmdBdHRyaWJ1dGUoeyBtYXhMZW46IDUwMDAgfSkpXG4gICAgICAgIC50b1Rocm93KC9tYXhMZW4gY2Fubm90IGJlIGdyZWF0ZXIgdGhhbi8pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndmFsaWRhdGlvbiBpcyBza2lwcGVkIHdoZW4gbWluTGVuIG9yIG1heExlbiBhcmUgdG9rZW5zJywgKCkgPT4ge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHBhcmFtZXRlciA9IG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2ssICdQYXJhbWV0ZXInLCB7XG4gICAgICAgIHR5cGU6ICdOdW1iZXInLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdCgoKSA9PiBuZXcgU3RyaW5nQXR0cmlidXRlKHsgbWluTGVuOiBwYXJhbWV0ZXIudmFsdWVBc051bWJlciB9KSlcbiAgICAgICAgLm5vdC50b1Rocm93KCk7XG4gICAgICBleHBlY3QoKCkgPT4gbmV3IFN0cmluZ0F0dHJpYnV0ZSh7IG1heExlbjogcGFyYW1ldGVyLnZhbHVlQXNOdW1iZXIgfSkpXG4gICAgICAgIC5ub3QudG9UaHJvdygpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnTnVtYmVyQXR0cmlidXRlJywgKCkgPT4ge1xuICAgIHRlc3QoJ2RlZmF1bHQnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYXR0ciA9IG5ldyBOdW1iZXJBdHRyaWJ1dGUoKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgYm91bmQgPSBhdHRyLmJpbmQoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGJvdW5kLmRhdGFUeXBlKS50b0VxdWFsKCdOdW1iZXInKTtcbiAgICAgIGV4cGVjdChib3VuZC5zdHJpbmdDb25zdHJhaW50cykudG9CZVVuZGVmaW5lZCgpO1xuICAgICAgZXhwZWN0KGJvdW5kLm51bWJlckNvbnN0cmFpbnRzKS50b0JlVW5kZWZpbmVkKCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzcGVjaWZpZWQgY29uc3RyYWludHMgYXJlIHJlY29nbml6ZWQnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgYXR0ciA9IG5ldyBOdW1iZXJBdHRyaWJ1dGUoeyBtaW46IDUsIG1heDogNjAwIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBib3VuZCA9IGF0dHIuYmluZCgpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoYm91bmQubnVtYmVyQ29uc3RyYWludHMpLnRvRXF1YWwoe1xuICAgICAgICBtaW46IDUsXG4gICAgICAgIG1heDogNjAwLFxuICAgICAgfSk7XG4gICAgICBleHBlY3QoYm91bmQuc3RyaW5nQ29uc3RyYWludHMpLnRvQmVVbmRlZmluZWQoKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ0Jvb2xlYW5BdHRyaWJ1dGUnLCAoKSA9PiB7XG4gICAgdGVzdCgnZGVmYXVsdCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBhdHRyID0gbmV3IEJvb2xlYW5BdHRyaWJ1dGUoKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgYm91bmQgPSBhdHRyLmJpbmQoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGJvdW5kLmRhdGFUeXBlKS50b0VxdWFsKCdCb29sZWFuJyk7XG4gICAgICBleHBlY3QoYm91bmQuc3RyaW5nQ29uc3RyYWludHMpLnRvQmVVbmRlZmluZWQoKTtcbiAgICAgIGV4cGVjdChib3VuZC5udW1iZXJDb25zdHJhaW50cykudG9CZVVuZGVmaW5lZCgpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnRGF0ZVRpbWVBdHRyaWJ1dGUnLCAoKSA9PiB7XG4gICAgdGVzdCgnZGVmYXVsdCcsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBhdHRyID0gbmV3IERhdGVUaW1lQXR0cmlidXRlKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGJvdW5kID0gYXR0ci5iaW5kKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChib3VuZC5kYXRhVHlwZSkudG9FcXVhbCgnRGF0ZVRpbWUnKTtcbiAgICAgIGV4cGVjdChib3VuZC5zdHJpbmdDb25zdHJhaW50cykudG9CZVVuZGVmaW5lZCgpO1xuICAgICAgZXhwZWN0KGJvdW5kLm51bWJlckNvbnN0cmFpbnRzKS50b0JlVW5kZWZpbmVkKCk7XG4gICAgfSk7XG4gIH0pO1xufSk7XG4iXX0=