# -*- coding: utf-8 -*-
#
#   DIM-SDK : Decentralized Instant Messaging Software Development Kit
#
#                                Written in 2019 by Moky <albert.moky@gmail.com>
#
# ==============================================================================
# MIT License
#
# Copyright (c) 2019 Albert Moky
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# ==============================================================================

"""
    Session Server
    ~~~~~~~~~~~~~~

    for login user
"""

import traceback
from threading import Thread
from typing import Optional, List

from dimsdk import Station

from startrek import Docker, DockerStatus
from startrek import Arrival

from ..utils import Logging
from ..common import SessionDBI
from ..conn import BaseSession
from ..conn import MTPStreamArrival


class ClientSession(BaseSession, Logging):
    """
        Session for Connection
        ~~~~~~~~~~~~~~~~~~~~~~

        'key' - Session Key
                A random string generated by station.
                It will be set after handshake success.

        'ID' - Local User ID
                It will be set before connecting to remote station.
                When it's empty, the session state would be 'Default'.

        'active' - Session Status
                It will be set to True after connected to remote station.
                When connection broken, it will be set to False.
                Only send message when it's True.

        'station' - Remote Station
                Station with remote IP & port, its ID will be set
                when first handshake responded, and we can trust
                all messages from this ID after that.
    """

    def __init__(self, station: Station, database: SessionDBI):
        super().__init__(remote=(station.host, station.port), sock=None, database=database)
        self.__station = station
        self.__key: Optional[str] = None
        self.__thread: Optional[Thread] = None

    @property
    def station(self) -> Station:
        return self.__station

    @property
    def key(self) -> Optional[str]:
        return self.__key

    @key.setter
    def key(self, session_key: str):
        self.__key = session_key

    def start(self):
        self.__force_stop()
        t = Thread(target=self.run, daemon=True)
        self.__thread = t
        t.start()

    def __force_stop(self):
        t: Thread = self.__thread
        if t is not None:
            # waiting 2 seconds for stopping the thread
            self.__thread = None
            t.join(timeout=2.0)

    # Override
    def stop(self):
        super().stop()
        self.__force_stop()

    # Override
    def setup(self):
        self.set_active(active=True)
        super().setup()

    # Override
    def finish(self):
        self.set_active(active=False)
        super().finish()

    #
    #   Docker Delegate
    #

    # Override
    def docker_status_changed(self, previous: DockerStatus, current: DockerStatus, docker: Docker):
        # super().docker_status_changed(previous=previous, current=current, docker=docker)
        if current is None or current == DockerStatus.ERROR:
            self.warning(msg='connection lost, waiting for reconnecting: %s' % docker)
            # TODO: clear session ID and handshake again

    # Override
    def docker_received(self, ship: Arrival, docker: Docker):
        # super().docker_received(ship=ship, docker=docker)
        all_responses = []
        messenger = self.messenger
        # 1. get data packages from arrival ship's payload
        packages = get_data_packages(ship=ship)
        for pack in packages:
            try:
                # 2. process each data package
                responses = messenger.process_package(data=pack)
                for res in responses:
                    if res is None or len(res) == 0:
                        # should not happen
                        continue
                    all_responses.append(res)
            except Exception as error:
                source = docker.remote_address
                self.error(msg='parse message failed (%s): %s, %s' % (source, error, pack))
                traceback.print_exc()
                # from dimsdk import TextContent
                # return TextContent.new(text='parse message failed: %s' % error)
        gate = self.gate
        source = docker.remote_address
        destination = docker.local_address
        # 3. send responses separately
        for res in all_responses:
            gate.send_response(payload=res, ship=ship, remote=source, local=destination)


def get_data_packages(ship: Arrival) -> List[bytes]:
    assert isinstance(ship, MTPStreamArrival), 'arrival ship error: %s' % ship
    payload = ship.payload
    # check payload
    if payload is None or len(payload) == 0:
        return []
    elif payload.startswith(b'{'):
        # JsON in lines
        return payload.splitlines()
    else:
        # TODO: other format?
        return [payload]
