"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let fn;
beforeEach(() => {
    stack = new core_1.Stack();
    fn = new lambda.Function(stack, 'Fn', {
        code: lambda.Code.fromInline('hello'),
        handler: 'index.hello',
        runtime: lambda.Runtime.PYTHON_2_7,
    });
});
test('Invoke lambda with default magic ARN', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn, {
            payload: {
                foo: 'bar'
            },
            invocationType: tasks.InvocationType.REQUEST_RESPONSE,
            clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            qualifier: '1',
        })
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke',
                ],
            ],
        },
        End: true,
        Parameters: {
            FunctionName: {
                Ref: 'Fn9270CBC0'
            },
            Payload: {
                foo: 'bar'
            },
            InvocationType: 'RequestResponse',
            ClientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            Qualifier: '1'
        },
    });
});
test('Lambda function can be used in a Task with Task Token', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: {
                token: sfn.Context.taskToken
            }
        })
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            FunctionName: {
                Ref: 'Fn9270CBC0'
            },
            Payload: {
                'token.$': '$$.Task.Token'
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in payLoad', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunLambdaTask(fn, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Task Token is missing in payload/i);
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunLambdaTask(fn, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call Lambda./i);
});
//# sourceMappingURL=data:application/json;base64,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