"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Add Step with static ClusterId and Step configuration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar'
                }
            }
        },
    });
});
test('Terminate cluster with ClusterId from payload and static Step configuration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: sfn.Data.stringAt('$.ClusterId'),
            name: 'StepName',
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
            'Step': {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar'
                }
            }
        },
    });
});
test('Add Step with static ClusterId and Step Name from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: sfn.Data.stringAt('$.StepName'),
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                'Name.$': '$.StepName',
                'ActionOnFailure': 'CONTINUE',
                'HadoopJarStep': {
                    Jar: 'Jar'
                }
            }
        },
    });
});
test('Add Step with static ClusterId and Step configuration and FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar'
                }
            }
        },
    });
});
test('Add Step with static ClusterId and Step configuration with TERMINATE_CLUSTER', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.TERMINATE_CLUSTER,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'TERMINATE_CLUSTER',
                HadoopJarStep: {
                    Jar: 'Jar'
                }
            }
        },
    });
});
test('Add Step with static ClusterId and Step configuration with Args', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            args: ['Arg1'],
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                    Args: ['Arg1']
                }
            }
        },
    });
});
test('Add Step with static ClusterId and Step configuration with Properties', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrAddStep({
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            properties: {
                PropertyKey: 'PropertyValue'
            },
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                    Properties: [{
                            Key: 'PropertyKey',
                            Value: 'PropertyValue'
                        }]
                }
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.EmrAddStep({
                clusterId: 'ClusterId',
                name: 'StepName',
                jar: 'Jar',
                actionOnFailure: tasks.ActionOnFailure.CONTINUE,
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call AddStep./i);
});
//# sourceMappingURL=data:application/json;base64,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