"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A StepFunctions Task to send messages to SQS queue.
 *
 * A Function can be used directly as a Resource, but this class mirrors
 * integration with other AWS services via a specific class instance.
 */
class SendToQueue {
    constructor(queue, props) {
        this.queue = queue;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SQS.`);
        }
        if (props.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.messageBody)) {
                throw new Error('Task Token is missing in messageBody (pass Context.taskToken somewhere in messageBody)');
            }
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sqs', 'sendMessage', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    actions: ['sqs:SendMessage'],
                    resources: [this.queue.queueArn]
                })],
            parameters: {
                QueueUrl: this.queue.queueUrl,
                MessageBody: this.props.messageBody.value,
                DelaySeconds: this.props.delay && this.props.delay.toSeconds(),
                MessageDeduplicationId: this.props.messageDeduplicationId,
                MessageGroupId: this.props.messageGroupId,
            }
        };
    }
}
exports.SendToQueue = SendToQueue;
//# sourceMappingURL=data:application/json;base64,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