import datetime
from unittest.mock import Mock

import freezegun
import yaml

from security_constraints.main import main


@freezegun.freeze_time(time_to_freeze=datetime.datetime(1986, 4, 9, 12, 11, 10, 9))
def test_main_flow(tmp_path, github_token, monkeypatch, requests_mock) -> None:
    mock_version = Mock(return_value="x.y.z")
    monkeypatch.setattr("security_constraints.main.version", mock_version)
    output_file = str(tmp_path / "output")
    config_file = str(tmp_path / "sc-conf.yaml")
    ignored_ids = [3, 5]
    with open(config_file, mode="w") as fh:
        yaml.safe_dump(
            {
                "ignore_ids": [f"GHSA-X{i}" for i in ignored_ids],
                "min_severity": "CRITICAL",
            },
            stream=fh,
        )

    fake_argv = [
        "security-constraints",
        "--output",
        output_file,
        "--min-severity",
        "high",
        "--config",
        config_file,
        "--ignore-ids",
        "GHSA-X7",
    ]
    ignored_ids.append(7)
    monkeypatch.setattr("sys.argv", fake_argv)

    nr_of_vulnerabilities = 9
    requests_mock.post(
        "https://api.github.com/graphql",
        json={
            "data": {
                "securityVulnerabilities": {
                    "totalCount": nr_of_vulnerabilities,
                    "pageInfo": {
                        "endCursor": "Y3Vyc29yOnYyOpK5MjAyMi0wMy0yM",
                        "startCursor": "Y3Vyc29yOnYyOpK5MjAyMC0wOS0yN",
                        "hasNextPage": False,
                    },
                    "nodes": [
                        {
                            "advisory": {
                                "ghsaId": f"GHSA-X{i}",
                                "identifiers": [
                                    {
                                        "value": f"GHSA-X{i}",
                                        "type": "GHSA",
                                    },
                                    {"value": "CVE-2020-12345", "type": "CVE"},
                                ],
                            },
                            "vulnerableVersionRange": f"< 1.{i}",
                            "package": {"name": f"package_{i}"},
                        }
                        for i in range(nr_of_vulnerabilities)
                    ],
                }
            }
        },
        request_headers={"Authorization": f"bearer {github_token}"},
    )

    exit_code = main()

    assert exit_code == 0
    with open(output_file) as fh:
        output = fh.readlines()

    assert output == [
        "# Generated by security-constraints x.y.z on 1986-04-09T12:11:10.000009Z\n",
        "# Data sources: Github Security Advisory\n",
        (
            "# Configuration: {"
            f"'ignore_ids': {[f'GHSA-X{j}' for j in ignored_ids]}, "
            "'min_severity': 'HIGH'"
            "}\n"
        ),
    ] + [
        f"package_{i}>=1.{i}  # CVE-2020-12345 (ID: GHSA-X{i})\n"
        for i in range(nr_of_vulnerabilities)
        if i not in ignored_ids
    ]
