# coding=utf-8
from __future__ import absolute_import, division, print_function, \
  unicode_literals

from unittest import TestCase

from iota import Hash, TryteString
from iota.crypto.signing import KeyGenerator, SignatureFragmentGenerator
from iota.crypto.types import PrivateKey


# noinspection SpellCheckingInspection
class KeyGeneratorTestCase(TestCase):
  """
  Unit tests for KeyGenerator.

  Strap in, folks; SigningKeys are multiples of 2187 trytes long!

  Note to developers: to pretty-format signing keys so that this file
  doesn't exceed 80 chars per line, run the byte string through this
  code and then copy-paste the result into your test:

  .. code-block:: python

     print('\n'.join(repr(s[i:i+66]) for i in range(0, len(s), 66)))

  References:
    - http://stackoverflow.com/a/1751478/
  """
  def test_get_keys_single(self):
    """
    Generating a single key.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999ZTRFNBTRBSDIHWKOWCFBOQYQTENWL',
    )

    self.assertListEqual(
      kg.get_keys(start=0),

      # Note that the result is always a list, even when generating a
      # single key.
      [
        PrivateKey(
          b'BWFTZWBZVFOSQYHQFXOPYTZ9SWB9RYYHBOUA9NOYSWGALF9MSVNEDW9A9FLGBRWKED'
          b'MPEIPRKBMRXRLLFJCAGVIMXPISRGXIJQ9BOBHKJEUKDEUUWYXJGCGAWHYBQHBPMRTZ'
          b'FPBGNLMKPZYXZPXFSPFUWZNRWYXUEWMP9URKVVJOSWEPJKSMPLWZPIZGOTVAA9QQOC'
          b'YISMGHSBU9YCXZCMSTPJVASDKEVZCSPNSPYOUUWWFTNWZTTZBKGZ9PDNAKNSGNODSB'
          b'IRKUGFYCZXIFHQCDTXQNLMKRVKIFJS9XARBNMJQOTDL9CAOKEXQTMWCKWRNHLLMLYP'
          b'QGTDFNTDBHNAFRBEUWTKPKPECAADKRPEAFDHABMYYXQPQYDSGFRSRFNHFHHHTAH9YF'
          b'OXKRZOTKAHZPRISHZRR9YBVSOZUSKKU9HTCXPTPZFAHFMOQJBKZIACZB9ZRXFPPPMY'
          b'RBCWPBAPRFXLQZOTGXJGMZUUEZIAVWXUEN9UIFLEESVCCGNKDISMEPYWTXDQHOSUWZ'
          b'OEHOCZQJKDCJJNRZVODVNNUOV9FZQEXFGAMDKMV9PVUYMWTFNISYGYKQG9OKNOQUEK'
          b'YDEJ9EGHUXQFCPHTTVBCRTZJLOAWHGDEQHLPLHWTWVOBCCQTCWCNLYDGUV9FKFZENU'
          b'NOCOYNU9CYQDSAQDSMZGRQYB9YOCFSOHQXANMSPYFVCTPTZKUGIGUZMPJIJRZVN9VX'
          b'ADLIVJYJGQWXBOBBAROGNIOIJVRUHWMFLVCGTMZISADLTXVEZLQDYAVQ9OCYQDPCYL'
          b'DD9SUPNTUMUESC9VRSCAYBPXAPTYXZODUUMBNCSLOWJYJA9JBITZLZNHPZFGSPRURJ'
          b'HYFLSFTMEPAEG9DWRFOTFWWPGGDGZWFVEPOHDGNMOXUSR9AVQLNDUMGPYWVN9LKEIZ'
          b'Q9MNIUPJXPTRYMSXRA9GTSZFMNZZT9Z9HJOKVBCHRRIZZN9UYTVRDNHOXYSFRO9FRK'
          b'HWNNZ9DXTLV9D9PNJLGWJXAFUOJZTRVJOLYGSPNVCXYWMTOEEUBLNRGAJPK9HIWGZM'
          b'HMBTHLABTACRYLQIDPOYEFNSYQ9YAQPOYYDCJAAAVDUWCHSS9OKQYH9CQNUPYRTCWK'
          b'CXYDKTAIJKWOQIHSGBZMFJXGOQDODBDZNBOPFYCBLSU9RJYVGXINUDODGHNAGFDAEG'
          b'LPDVSCJPCIZHOFNHCZUTRLQXEZUFDZVROFXVHUNWMYRSZHBZAFCWIY9ULTBTEDSKEC'
          b'CLDGAU9MZXYRAXVY9NIQUYHATJCZXDSAELMCXQMALHNFMEAWHIQAZMQOO9QPEPDOYC'
          b'OJXTUWEJHMPXGZBXFPNXUPOSDINZJNIREYDFZMESFQUPBKSDGTAJHEZSCOVSLYUAUK'
          b'DIWNNLQJTPYYTPRGGN9IXIORWXHJBPYINQZIUXKLKXCTQZYJIRH9MHYBQIFQZCFAKZ'
          b'DUUZTYIMTNNVNKVMFIW9UYRTVRQHMYR9Y9VYFTPBJGSB9VINGTMBKVZJEZUE9RMBDZ'
          b'CQGDHNPW9YIJHLGFOG9YAPXZECSFVXAMPILBIHC9DBGMIE99YEPGTAALOHBUKXSGFZ'
          b'YHHWFOIHMEDXFHIYSUHADOCKFNGHKTNPZHINAWG9YGRJBQGECRCVPXXOG9CNVJNFLC'
          b'LMGC9I9HAGTAGGVRKCXDJWDNHYZBFNQSKH99MFAMLGRSBMIBBMHBDJTVSQ99ZHYPSS'
          b'XLUNFCNOJXITUETNBHIGXFLLEHUKEXGJLO9BBALXMNGJKTETFIZHSSKLOQXPXOSZRW'
          b'QP9A9RIHWEHATMSVMZEQPGAUQBCIAQXZSUUFSU9HYK9RAVASYCVNALKJJXAJF9RLTD'
          b'ZEIIYCFLQVMHBPBFHHQNXVEKPHOOFTQEIVB9IXZMTOFBHTGLPWDYVPO9HHBPVWZYEG'
          b'IDMK9UPWEJDLIPJSIGFKKCZFRJVDN9ENWADNOTFWZGUDJRRUMFPVXHNAJBBCI9WEDK'
          b'RKCQUHRQTCFYFHXPOFBC9BCENMI9HRSIUAKLWEAOUXRBWMHWLGEOCP9NWIJAXODJDS'
          b'P9SKEXEDVUGHZAFPNMR9PXD9THOWNWWTDTWTYMDINGC9EBBVUYZRUQDVSIOXAEVGFP'
          b'XS9CLTHUESMTDWUJNCZSOIOEJG9WKNAZMDJGMRBXGVMLUAN9IGDVFAESJMXNTMNFND'
          b'CAXEBRAU9'
        ),
      ],
    )

    self.assertListEqual(
      kg.get_keys(start=1),

      [
        PrivateKey(
          b'WEVFRAOIRJBSBKQWF9JOTQXWUDLOIJRAC9WOOJNW99UAXVMUMSCMAABBXI99PRTAQL'
          b'UWJKVMM9DPSZSU9SAUN9URDWGGXIHWJJCDBAY9OQMQURNHZBD9E9CGERZC9RSUQVMZ'
          b'VYUTYXLH9CCEQPQLVDICQD9UCH9RPP9NSNZEBYERXHDEBEZUZOKNNEMXBYZPAYIIOB'
          b'ED9LRBWBBKGEPSONFKHAZPKSUKRDZABHEJNNWXAVTVVENGQRQSH9Y9XMGMAU9BDPBT'
          b'MLTVKCIQESSKRDYKELSAPDODV9HY9YKSWGKHOQGMOQUDYOXKDXERZ9EWCZPLCBRCRE'
          b'AGOZHPYMUTZEOODHUGROGONAAGAETOIEJVRLXXQMXYAZZNKUBQEOMPNYREMGWTAWSX'
          b'XJX9DWXLMCGJM99SGKRLSJTXHBCFQRDGHCVSXNSDTDKP99KLZRJLRWTTYXA9CLROYZ'
          b'OONLHBPVLECRWSVVLJPKPAAOICLSRBZIVSNTQGTXRYYDWEBIOLVOXCVFWYIKAQMNJT'
          b'BLGRBYWMNK9MD9FAAKDWGMRQQ9ZELXGMYJF9WH9HEZEYKPHGDCVCPEIEXMDQMGIOSS'
          b'9KCVPCQHSHALBYJRSRNJNJXMSBOIURZAABSRESLVLYHJINSNGDGQHGOLLMEBR9PQND'
          b'KZCYRTQTPZQHPZLVDCHIKEIARRUYREQBUXYHVZGTPNAUZCVCTDQUDAZHBRCWXKBIIN'
          b'IAMU9MUEIDDZRVYPOAKWYUFULQGYXMC9PCBLATKZROJSKVSQIXZOTKWIWBII9XHZYU'
          b'PJVPEHCCOPUFNTYDMEEJOEYB9RYZAO9TTDKUMKBQGIQDTSKQLDXB9ADYOQ9KTIQLNE'
          b'XVO9QOGLNEPNYKUUKDPHBABJLODZXBKUCUUXDPHA9L9ZCCDUMWOJIWRLDAO9YISWMB'
          b'GGUKKMNTLBUUASGSNYOUTDNSZJTOKZPLZXBUYTLNVLIXVDKPSVUK9GYIPYMXALZLWN'
          b'TVQHOYRTCNQQWYAUVSVAUYXXJMBESFVIJZTLEOICAUHZTOSJLSSLRELXZXIQYMQIG9'
          b'OSZFTTYAWIRZYWBBVAZDERDFMVDOCSCXMHJQYWPD9N9MKRZTZVTSTDETHMOACATA9V'
          b'UIFBALVQILCEZZIHPQGYUVHUQVGERPFFTITFEISHIRPLEPQBNNNWFUXVBZSDNMOPGA'
          b'ETRVNI99YDQBYMUUHGGXDTHTKTQUGKXBELCTDGCQRRAPKEFQJUTXDO9QDYWJCYPAQW'
          b'GDAOCNVEGLEICLXPTCSFUGYPNBBBABFSJ9JKTPEXQFAXIBSLHTKWSS9KYPFFDKMZZX'
          b'CWKDJQZ9ZTWAKYHTVFPENX9EJBYLC9PJCNKYFMJJSIA9NNEQVTLCPDESHEUTKZBHDF'
          b'SGBNRAWFVYXHXQKNOTSPULDRSQEZHVDCIOMJB9XWDUFJMBHMXQRHNWHKYYDDPOLC9Z'
          b'XASGTHOEVRJVESTRZZLJQNYKCNHBHIHCAZCAFHEGXUDFCBAJLLWLJPZFXMHXBXUOCN'
          b'GCIRTIASUQTDZQUAXHOOYICWFDK9VTBQOMUXJICSXUQQPFSTDYITFAXWFHMVPMCUWZ'
          b'ZJAV9HIKFWXHCFXNNNXIREOBYDELGZTZTOKXVEQTMKSMQGXHEPWYOBASQGYCDZYCZX'
          b'IJRPVGPTXOCRPAPPMBNQDVQBPSWHSQVALOBQGILEAUWIPCCXPTZRRQHAGW9SRCODAH'
          b'FVCW9TYQLVTPMUDWYBXKGFFDBMDIYOICLBRJPAUKQVTVQEJUWMCMBJVZZFVUUJQKW9'
          b'ADITOSRWTUHUBWMRZFBLV9VV9TYLOKVLJXVPNEOMQFFVKN9AFGREGUXFNVBZPAMRLT'
          b'RRUNOPMYCWJLTMVJDEHJEUAQPFMZLDJSUHOYVISZTYGQN9ZHUBMMKFKSFGI9XXVBRJ'
          b'JUHPEAQZANTOTTSQRPWRVLORFC99ATMGSOBLIV9JLDWPOHCIYLLFYKSLCVEXHNPXFG'
          b'JUPNCKZEVL9JXMCGAJJDATXWSLHILXC9MWHMBWTIJG9ZWXFGMDAZOFCNMMBGPJLMRS'
          b'MIZSVNFVNWGZHIMCXRPEIGIZ9RPNYVSAF9HTDX9BUDPIJTKYCBMAJPAYHG9OMOUYEE'
          b'ZT9WPVUWXCIONWGPQAEWAMHQFDWVSHACZIEJOSDTEVRYAPZCHFHSYKEAHTPAGABIHV'
          b'TLMWMCVUE'
        ),
      ],
    )

    # You can request a key at any arbitrary index, and the result will
    # always be consistent (assuming the seed doesn't change).
    # Note: this can be a slow process, so we'll keep the numbers small
    # so that tests don't take too long.
    self.assertListEqual(
      kg.get_keys(start=13),

      [
        PrivateKey(
          b'ZIGSOMJRQUXQHMNP9NCEGWCNXXVMJW9BXYRTMVUWRVTFQ9GCMJOOENTBSJDKPQTWML'
          b'FGEMPNODQWJ9BIZJSFWOOQNYLHIGUAJJXGISEMZKVPQOLQIMKESDECLJLDJFTRSBCQ'
          b'UEAJTCMUDYYWWENQZAPI9B9B9RBOCEAUAQHQNCZTWDOCYCRXZNHYTTTTNUNROHJGCF'
          b'VRPV9RWZFFQOIWCIYSTWIQOZFYMESMFYPQTZNLFTUPGROJSQRCSOPRDYQFM9UBJDJZ'
          b'CIV9UQDPZWRAEELURWQWZVM9BMZL9YMVLFBYXRXOJMVVAWOCQQDVGDGXMKOAQWU9YP'
          b'UUXSGYZBSLQEVLLSMLIHISX9UGOPRKIVYMCLJXA9CRUSBKEHVDHPEBELEDIMT9VDVP'
          b'KVZLZJCWKQQQTWQOWLYYHNVYFOIMLLXIZFMYBPZ9E9U9ZZIRNGPP9YYLPSDHKDOMQB'
          b'PZAKGCVBQCOCERITODGEQUISDBUQSAASVCJRFOPIAQRAWMSMWLWFQ9GGRNFCCXZELB'
          b'UAISJVCMDEWMWQPAWTAXKHSKBGWCZKHIBQCIUUBLGPVHAEVLMFBGQZXFY9AGCYWCBI'
          b'JKDULRYBMAHXXBBTDMXZMWK9VMJKGXL9WSQCDSVZSDAKYETPBUZHVJOCZSDKQVBNF9'
          b'PHETFF99VSCZGHUZGJPGLDLGTWHWRAZTBTWONDXVNDAUMGZAUKQTIVNTMQHTCHPDBL'
          b'JWHOXITFMDDLF9UZ9QVMJPHQAOKJVJNICLSHHFOEOLEPJJBRREWMGGZKOSLLIXOXRM'
          b'XSXWBHPIFQGZDXWIANKUMERQQKITASHTK9UTWYIVBCADTHTCLPAQOKSWOCTVBPDISR'
          b'TXFOX9JJGSSIMUJNBLCATMROIYTULKMSDSAOXHIFOEQXORNQBJIQXTKWKHONKWWGLL'
          b'IECZX9QGVZKURJ9GQJNJWMRMZQAJALNICBUQOWVPOLHYHLCDHNMGNBQPJDGGXFDQVZ'
          b'GUOILR9MPAPCGJ9ZZ9YWBJKQGDYJSGK9TFBIJWSMIWWCUFJ9QCGUNEQMDBQDODDWJN'
          b'FMCZOIJREHRNFSIQIBLLDGPHWGZB9MPKBGHFTRRFZRZJYOZRAYVB9AUOGXLGHHDGBE'
          b'XOVAYLBAVUEAQCBIWNVJJQSCIE9TOL9RQSYLYAZQCZGGOIIDOSHFZDJDZITGVHMZYI'
          b'Q9WGIQDGCKUUGCQAZYCSMNRPXKPDIUUHBFNGPRORGLXSGEDQHHYIJKBKCFZFTRQCYY'
          b'BKRETMZWKQWFMKZGETJSLZFBTA9JETYEYN9VJUBJKM9SEJROWFPRB9NAPMDMIZOCUB'
          b'BUJWFMBBSSUSDKLXVMDR9HXZGYJIEX9JJXTUBMROYERCUNFQUDUDGHHFH9CRTDPTAY'
          b'PPFHIJTBGPQHMAJWHMPDCPBBJLEULBBMWKCDPHQZE99JWZGJWYIZKWIZIIXLWTYVTB'
          b'LLPHSLMXRLBFTMRPPXADNVFRLFYCJZNLRFZXTR9MVDANWPSNZBVFXPKVBQJXATDCBQ'
          b'VSCXJZCQSMIEPDJARJF9NKDVXZAY9EPTDGTCGMEWMCPGOKOZDYZAYTTOEDGNGSZUIH'
          b'CQWQXMGTXFGKUQ9EQFMOUWXBTEIJZUYFTTFFFJBKYURQKTXCNIZOIONWOBKRFMRVSB'
          b'QP9ACSWHOAPYIWNCDSLZGXQAKSBBUZPDNQEBWFRISIZIXLHIHCBTQT9SONHRHRGERF'
          b'RJGBHWSVUUGQAEABDGWXBRIQIQHSINHXNI9BDJUCQKHTPDRNMVD9GFBIUIMCJSEFYV'
          b'DEUUQFF9RRMZPAPRJUVCGIGMJEHPYMUMASUROXAYQVTVEB9ATIYKRSECWLZLWWGIAX'
          b'VGCOXFIIYTME99PT9CCKQWHJYMSIMCAUPN9PXGCHBCTWBJD9JGSZPOTWYRWQOYEES9'
          b'GZJHTDQVAWFKJYHW9NOMLXDXYWUCER9KJVDAKEVWCUBHSQWIWVHVZLQACWHNSPELZA'
          b'UBGQHVZPVKRIAZSNYWTEQSXAJDNKNXTFH9QBOXLRZJRMJSSCMWHQWJCTZ9YNDJORJF'
          b'APQQSJALFLUGIBFBRPYKKM9DP9ZKPGVIBGIVVPXFQQURTFRBAVNWWF9DXIDEDCXMDE'
          b'NAEDEEZGLLWGYKXYEGLYC9FKYNKARWYIGVWBLGROXSEBAOIVARNLWTUEJUYYNHIQXS'
          b'DDAKAUN9T'
        ),
      ],
    )

  def test_get_keys_long_seed(self):
    """
    Generating a PrivateKey from a seed longer than 1 hash.

    This catches a regression caused by
    :py:meth:`iota.crypto.pycurl.Curl.squeeze` processing the wrong
    number of trits.
    """
    kg = KeyGenerator(
      b'TESTSEED9DONTUSEINPRODUCTION99999ZTRFNBTRBSDIHWKOWCFBOQYQTENWLTG9S'
      b'IGVTKTOFGZAIMWHNQWWENEFBAYZXBYWK9QBIWKTMO9MFZIEQVJULQILER9GRDCBLEY'
      b'OPLCYJALVJESQMIEZOVOPYYAOLJMIUCGAJLIUFKHTIHZSEOYYLTPHKSURQSWPQEESV'
      b'99QM9DUSKSMLSCCDYMDAJIAPGJIHWBROISBLAA9GZFGPPRPHSTVNJMPUWGLTZEZEGQ'
      b'HIHMCRZILISRFGVOJMXOYRALR9ZOUAMQXGW9XPFID'
    )

    self.assertListEqual(
      kg.get_keys(start=0),

      [
        PrivateKey(
          b'V9ZUOXOFKFSYMYRUKLXFTMLKBTDGXJWZZSEQINISGCXCPDQ9MFLTIEXXOEKPJPU9TW'
          b'WQAOWDNBVHODJSMTULXCDKTBPEBQMWNXRUWY9WPOFQZDOLYXWEABWU9DNFIUEETSHR'
          b'RNXXAGENAIQQYDBWIVXFZIKLNSEHVNKZGKJPAFTICTRNXCRDBXBEKBQF9XKS9LSVZU'
          b'EOEUSSHWVQBLBUO9CZRNWHFDBPMUSWLZUUZXXFNZMAUVOPWSVWELDLEBORRPMPWWNU'
          b'HRBIERMHKKE99ECTJGTGIY9SDSJITHAQVJNABAQYJGKODTDRPVODJFEBR9HBBOKGTU'
          b'XQHVGWMIULQN9QTDGZCYCVRH9ULEOWKLQCZXAQCAQWRGNHWFXDXOOIMRAKARXQWYBP'
          b'HZGK9ZXRPERLYLNFOCZIQOM9KFMRTXK9CVYITDLW9WHFYSVEPKZCNKZURNLLPCHYKX'
          b'OKPOARQN9EIGPMUWSXRBCDGG9UMMJDUGIPPSUZACIASWZ9PFTBHIJKG9YBQXWNZAHA'
          b'AANAOMQGMYSVBGFKHMJBHGPGZSRMKCMYBVTG9DCJCUXAPWLITOAKWOSIMTORUIYRAI'
          b'9EAUSYORSHMEVAAVPXBDHQKIMLPQU9WGBQLFLXVIAHQWWQJFACPBQYYSTGHEQCVJZQ'
          b'CZNQLHIROHBNFKBCDLTOWIGDKMPUBMLTSWKNF9KRYKE9GFFOWBAJTB9VRGRJGSFTRZ'
          b'DEMYHHB9ZDMNUPRNKITM9RRDPQCCLHGVKUMRJNJIIOGK9DWZRPUTBA9FLELUCFLMLR'
          b'PZNCOVRFOV9EXQHBUYNBHXIWOEE9UJIBEIST9UCMRHIBYROHCTVMFGYYSRLKXRSWFY'
          b'XLYGDDI9VLXEAAZE9LNUYSFLQULELTZKE9ZMAAZRZE9WUTDKHPFFILGKYWBRJDOJMQ'
          b'XQRQSHMHCUFZFWVPDBXWATDLJYAKDGXXAJBGMOQABEGKADPBRQINBRFSUUMCTYTC9V'
          b'TMVTXTCYIGNQWHAESSGDUNKBTJEVXUUJ9OELTSGITAQDLUXJ9WVNVXBHPHVHTQDFUY'
          b'QXZRJZTBBHWZPKBFA9NRT9BGDALRCFOGJE9SGKXMHNTHJGJH9AYBBCMTEKGEGASVXM'
          b'9S9ZMFIZXADIMFFZHKNWLG9PECBUH9DWIUEYWCSFQXQBAMGNAOXDDOMQA9RSVYGFET'
          b'SVNJKYUQEOPDNPJCNUFO9OIHNMBWDIJBSULTWHTHHXY9GECMMMIMVDYMSYCGGZEXDR'
          b'NOZJBTZHIEECAJACURD9Z9YGDZIXDAYLJIZCCX9NHYMJIUPLYHOYRQHXXCVOBJOIMD'
          b'ROZFIJOBYAFVNN9PKNAHTMYODHQLFBOEDGFAEITDHZXTEMLGQBCWDRZI9PNECLNZEN'
          b'BRPWRQGHDI9VJNKUQIWPIZZ99XKLTIQABPWNQZCRUL9LJIBPTFQRAOSJVAATQOONLM'
          b'9KBDO9QTSFS99GFKMSOMAIXADL9ISLMJJPB9ECZPWSARYBJFJQQBALFYEPIYLFC99B'
          b'JH9IRDFOYR9VSUJEALRAFNIPYFHUKVAZHYKKGTVGHVMKVMTSHSRKKIVKQJHDOAIVIW'
          b'JYWCOFPONADIYTQBRUAVNEIYDFTH9FINVATCSBEEVEALIRBXFPZNWZZTHRYLMRPPZJ'
          b'SEUQBMSHA9FVSYRRSAGVDT9WWNIHKKRVTHPOZOVUHGGBGBJWXDEYHQMWJMCEMTVYWQ'
          b'RLCYUQFJOZVDRAXUVZUPHBBSQBPF9GJGVFCINMQNLIZ9UTIMCDBEHSLYXFGULQSLCT'
          b'VLSSZIOOBKMHRXJX9YHDUBYGJVIBGKGCGBCLUFVQIPDFSTYLD9AEUVUQZPDWDIUVZG'
          b'FGJFPRXBDBBBFFUFT9XLJLTOXIIYIBMPPVSJZFUJSDOAPBHJOSURGYTMHDOIB9AACW'
          b'NPJPYIHWCEFW9GTWLAQSOIBVAUFWSSVLAOCKNBLVTDDQUGSVZYPVZIYDUVZZSOZH9P'
          b'AWJJZQ9CY9BQGWXQADPXVZTJHGM9UHZQGYAUIICUX9BKZPLYOYZWIYHNBORMBKNCQA'
          b'CIGBDKXAJLEISKOJVFMGCIQIAZBEOBCFXLBLACGYRSTE9YNAPJPPLQZQSIEKOFFWVM'
          b'CNRFTNWRGTDQYN9PNJSPBHFQZXKGERMZOEETWSDPOTSRNUYSYQVZUYLMICCUUKJVBE'
          b'9PUHIMNYD'
        ),
      ],
    )

  def test_get_keys_multiple(self):
    """
    Generating multiple keys in one go.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999TPXGCGPRTMI9QQNCW9PKWTAAOPYHU',
    )

    self.assertListEqual(
      kg.get_keys(start=1, count=2),

      [
        PrivateKey(
          b'VBGOUVYVPMMVMNYIASWCSDOECOVM9BLBAWECSHKURDUVTFBRMUCIADFORQXPPCDVGM'
          b'QAGEPRS9IVKLZTJOVOBFXWC9PDXBL9WCRYKCNNFWOAJCKKV9KZICCQKUOFUAUYTRJH'
          b'DTCRDLWCXKSPK9ZMVVFMGXBPQRCVAFMRAGVHWFKQEDPXPR9UTYOSKPMGMZZWLW9SBZ'
          b'PIQKJAPCAVMOWRURRUKYP99ILMGYGNUTCKXEHXTYHPJXUJJBATFLBJTZZSKUZZVTRU'
          b'BNJJJCBXTVNUVFQQPOYIIRVJGJQGPIDDSPNRTAQLHBKJNJXSGSUOHXSWRTDOSLB9OG'
          b'WPRBDCZMRCFEUHQMYZGLRSPCSMCYHZQLRPYTFFLVJDYTEETUVIAJLKVPWIEGKJ99LO'
          b'LHSDQNDTFIXTRY9RCZTOTCYJEDIK9YOQLEEYIAMTSOENXLAALCHWJUSYFJZEZPVBUH'
          b'NSIQRUDTVQUKJZWYJDNMSMFDVDTFSAODYOGFYK9OTEIHQGSSUVINXURWBAAYVNRYVR'
          b'YIHQSSPKQKFKVGJRTHMRMSBLIWKHSCFIMAJIBTKMVADBOEWTANSJZCUGQRCLVIB9ZY'
          b'PYPIEJPQAXOPEEGWYHJJTPZRHOSTSWIPQMC9T99IMALJONRAOLSJVLGHAQSXNCCIBF'
          b'XBJZA9FVP9TBQDHNJPSAWSJ9HMCKDX9RGQEUHOHLXPVJHCFBEQTLDVXYOMLRNUGFWK'
          b'UGXNYDKXKAKNHFYVO9DXNALJLUSROLNMPSOOAQKPDHGXHCOAVMYRQDPQQQRBQESBOW'
          b'L99CCPSYYYSMTLDLVEWUFWLAAZMINKRRAYIZRENIKFSYFROVZFCYQIH9BYSTUEWNHT'
          b'SA9LQDET9XLMYAKAR9MJILMHKMZAZPXSGVMKMZWCNO9RGGPJBGYRDEIZWSALUVVYEZ'
          b'VKJKCIOWBHLGUICAPMSCOKHGYVGUTKFCEPZNFQPUIQTKCLQIFKDQYESLKYERQPQDLS'
          b'YXIUAZVHWGFCAPVQOITULEJEVQRQXFYKZPUEVKWKQENGSGFLMFSPPBBPZL9NPFYILD'
          b'MWCRVHYYBEIJPLGSLDJ9ATXIWHPM9VSBZBOLAVMLRUHLAPDVHVAMAOWPQMP9BUYUIG'
          b'PWJCCXSTCNABWECAVAJGBBSJXIDOOZKWMXTENEXHBLJAZHNFTTORCTHELRFDYRCCWI'
          b'YJXUL9BWE9YPVTHVFCSUGRPAOTTKIFJAAS9MDRENZYAGEHARCOVARJHUWJCKLHOKQA'
          b'ZPRXOGKOBVCGCQFUJAMQZMKQOZPXZJZKJCHADNKZV9ZKQ9DUHZOCNTU9ZGKWHJODYH'
          b'JVSOOCFPKSAE9EFOLSNWYXYGVONYLFBHIKGRBMSS9URVCITGYCHXKVWYRQYTLWZLOU'
          b'QXMTDADHDZGWYKWQVZLSUPMTPBHWAUWQLLKXHY99QNSMQXAMHHWLVL9WHTALBLVMAP'
          b'WVFFCPXMUZTQCBPXNRQMKDEIZBIOCOJVUYGXFKMZGTCIYMX99RGMGKMWTRJ9N9ONLG'
          b'XJDPMYECS9AIJSEDALMOMSHBLDRI9AYRWDEZYNRBOEB9DIDXXEBOPBDZPPEQTKYRMJ'
          b'AEAZWUYQXROVBY9JZZGBRDUQFEKHH9TZMTLNNFEZBFFPEXQIHDZE9TBHDYN9SKGMA9'
          b'SYZXUSBOOXREZAQMFFKPFGTWUGUDV9SKSNH9ONNWDJYKZCBYCZIDZCFOMBNGNCSQDI'
          b'RCCEMYSHNWKVWJFWFPI9UTYEZRZHECQKGBGKUJEEGQQGSVNOVMIKGBRPXACAS9VNZW'
          b'CIYAXUHYOFJCXTAFZVWFXVVELGHCUNISAMCVJDNW9XXNLSFOXHWPBNYCUDZHHDITG9'
          b'WETROSQWYMRIIAAMLGNJSR9HMSMICDEWBHLXDJNMGVLTWOCEUPTFCNCIZZ9OGIANLO'
          b'WKFGGKLXODJDYNIDSECWHBIYSYHVUDROUGW9V9KBQCGMZVSUIGYMKAZSMYTLBKDNYH'
          b'ZLPFTCJZCDAPGNWJVQBLYVEWAEQKGGJMLEQPUDXPXPHZPYGTHZCGPUGTXNGUNLZJZV'
          b'QOCSNRTMVYANLYHVBRZXPYAGSQYDGZFASTYTXKATWLJZHGUF9MH9VWAGGWEVRCKUCC'
          b'PYSXYKHFRVHGLZVTXYJZ9WPOPBHLPYMPQCZNOKCNDNKDNQOQTYCRAPCJDQGRCBEHCM'
          b'REONEIFHT'
        ),

        PrivateKey(
          b'H9VZYRIEGPFTUOMXYHPDLGNNWEXKHOMNHI9YIEOCBXRQJTZMYW9LHXTZDQEYMWUAGY'
          b'SJASVKHVXRJBMZINYZX9WBUERAH9EV9XDESZESVMYMQRNCNERUUGCYLBKHZDDLAPGU'
          b'MIEMROYBKAIETO9JVWLFPOHJZTFOTU9FNMOECNZEOFG9DKVSQNQBXWLVDTBCPMRAUN'
          b'AOCMVLMNYRHFCONSTS9BGEKBHTVMEBQCWKQVHYADOPGS9DILCOQSZPLJTSFVCRWDVZ'
          b'FPSPMUDDGQGZLFYWLXKXWUKDNGVMDSJ9WQMDGNMCQUXHDORCRQUVSTUHDPDQOVMRJP'
          b'JGALZNEEVZQWEYELKZATPCZK9TLJOSUCBUEPPOZAGKYKMSKQDGHTDFZMAGJEBMXC9F'
          b'HMZQODXKGGXFYZ9JQ9TPO9ED9EASXDU9NNLIUZMKCJNZVGMJCXCATGLABBR9TBTPNA'
          b'LUQS9FICSSWFQWZCSMPECJZOQKIUBOCHFBGJIMIAXCBMLQ9DRNJLE9LRBA9ZLAKVWM'
          b'BVLGARBWLX99GZVWOPGE9JXVATWKLWDIGDZYWKMRWSVCAUKCFZANFDEBEDOXRCMNSK'
          b'JELOAPHRLVEZOLICVBEXFYUVIMPAY9EBAXZCVNUZPE9IGGONTDGKDLXRQNLTSJDQIC'
          b'EWAINBIIFTWIDYNZGSP9ZMMDBIMAWA9GVZOAENXYYRRFTEI9CUTDJ9RMJCVK9QDH9C'
          b'KFMROPLYAJHOLO9KDJSXFZZBEXC9XIDJCCDELTQIUG9FBMNKNOEMGMFGLEFJIOM9R9'
          b'ZTKIKAJGJHRHONUZAJGJONYTGXUEYBAXRJHJ9PMYQBMNAZZKMHWHHLTYUVDGNEGIDO'
          b'AYAAUMHLYOTEXFHTCGSVQRYKGOXECJSEZFZMLFFFXO9WWQHIEXXJZWRGVQUQQRSCKO'
          b'ORUCSQFUCQYO9MQW9XADTARGZYFUXAAJXSSGPHLWVDLOIYUNVGAENJWQLSMNKCBVPZ'
          b'XCWBWO9NTUAOUMWGFPAEHCPXNTSYIUMRKMYMTZDVQMN9QC9GF9RSAWDIYJUXVELERK'
          b'UCKHJUUCGP9DQRVMKVJMPNONGTOVXQYBTEFHTBQALQAMDWLJWYQAPNWIONSAVYTGQZ'
          b'SSTQ9KHRZKU9GVWXCMSTTBKRILDDDADFIPYSKHEYNPVXOCNATUQGWUWYPDSCEWIXCO'
          b'UGFCWRVRBSQRLPXLCVWJDAVXOCAXLYBMQUQEVKPWSNPBXPKZPLXUBKKTINVMZFUMXY'
          b'EQIURPJXBSNAZERRDWMQARORPS9SVKSOIQIEXGNEQD9ZJJGFNYXFPA99BAMK9TLDLT'
          b'QTJHSFKEXRFCPIJPNXZVXPQDDBOVGQDLD9E9JPDNNTISHFTEICSCSGWRPNCICTZRDU'
          b'XKL9PZDJXYDUITTHTKTQRXBQHSYKIDVBFQJLE9SA9SEJHWAJRGXRTZSKJTNDBFQLPB'
          b'LRHPPQUQPD9INBFAWOJUYCQG9ALWQKVAJCGAKGBWZLOCCEPRCJMSJIYLMGUPTUDDTY'
          b'9QWLWCPTMPRYTBJHKWQNALSLRAZVWUFJULOHRVPGGAPST9LDLIVVWXG9MOXLJQEAMY'
          b'UJNZVOHHBATSQMLKDCOXHJPJAJEYFRDYSKHVIDLQYDLRJLUBPJLZRQOEOCMBPYQIM9'
          b'PRTOHDATAUWGDFAJBPJQSFPCGPIZOMNTXWGPKUTETANJUAGDZOBBBMOPJAASNVUC9J'
          b'DOXSGNHCUCFICUBYKPKUJJCXCFXCZCWWCGGJAHDNIGCFEBMQCBDIUIURDX9FMBVCY9'
          b'IYFDRWVYIABFNSR9MBQGISPSBCRVCLWFMY99SYNYNUCRGAWKPGJXV9LOYWSEHREKRQ'
          b'WFKGHBREVCQYESRHGQMDHOPPNHZWR9TMVXUXEKORNWQANFWMPFQ9FJHSBFJGPCGUFX'
          b'ROODDYB9SEI9BENQMPDKLMUUXB9LTTTGOYUIJLGIIDIEJBYAPFYUGTEEVKKBVMQNOY'
          b'OTBKGRLTMLZMOAAEGCGRNMABTOJFKYQYDAYNSJSNXGKDEVKXAVYHBBSJIEFPOD9JMD'
          b'CEROCHTBQPNUYRAHRTVOUPAMVY9MLSWCMSJYM9FWJFEHKSFUVOACFBVXKQVRXEMANB'
          b'XDGGYBSACXOLLZ9EEZGFXLSINKTWLUGQEABYPOFXVMCQSYRJBHGQJRCNTNWQJXRAFH'
          b'SHWFKWSBP'
        ),
      ],
    )

  def test_get_keys_error_start_too_small(self):
    """
    Providing a negative ``start`` value to ``get_keys``.

    :py:class:`KeyGenerator` can potentially generate an infinite
    number of keys, so there is no "end" to offset against.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=-1)

  def test_get_keys_error_count_too_small(self):
    """
    Providing a ``count`` value less than 1 to ``get_keys``.

    :py:class:`KeyGenerator` can potentially generate an infinite
    number of keys, so there is no "end" to offset against.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=42, count=0)

  def test_get_keys_error_step_zero(self):
    """
    Providing a ``step`` value of 0 to ``get_keys``.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=42, step=0)

  def test_get_keys_step_negative(self):
    """
    Providing a negative ``step`` value to ``get_keys``.

    This is probably a weird use case, but what the heck.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999JKOJGZVGHCUXXGFZEMMGDSGWDCKJX',
    )

    self.assertListEqual(
      kg.get_keys(start=1, count=2, step=-1),

      # This is the same as ``kg.get_keys(start=0, count=2)``, except
      # the order is reversed.
      [
        PrivateKey(
          b'ODJGKFVKDKETMOUH9OCDBIDQCEMEDVKMEOIKSDTIQFECONPDCZUITROKXYCTNEMVQI'
          b'KZWKIJQZVYEAEUFGWAZUJEQMZLBPPYLJYEBOEZNYDFWKLYB9S9GTVZOXGGQ9CVYVVX'
          b'ZNIPHGIARTYMUXOJVTQZSASWVK9CSRUJODBJRPCCDDFEPZRNYHIGMFRFXAFGPMACLZ'
          b'WTNBVFISOJGBYRBONVFSLUWNZGERXLOCWRSNDFSZHKZMKVQDUXALLWDDOZZAEEXDDK'
          b'SXBZC9NDTIKMEZE9ZMDLUVFXKFPEZGPSFVHLQXLVZNCQGITVJKOABSRXALRMECHAZS'
          b'EXXDKEKLHXYVG9U9FZXWQHRGZKDUIA9XACNYPPWWLTXTGEAGXCQV9ZRA9OTGQVVWTG'
          b'OMHKSOA9YYBHZFNUGX9YF9LBXHK9DFFNIESPRMVESGMIXPNGOJLWMBAXIRIDQNPSLQ'
          b'XFELX9QDANRNJYKQLSOFUFQKSXWFGOB9HZQZDND9TKWYONT9LPWMUSYBZNRXJQLWDH'
          b'JMDODMYPHNWACOQG9ROBLJJCDDEJHWFXAJAGTILEO9BODJJMFYJPOK9DMTXCFAGSEJ'
          b'BMJQTPCETXDRKKUGWHDND9SGZZZDNFNVWYJYRDWPGIQZJMKQAEZPHIEZUVXIHHUWQN'
          b'TI9YMTVWZBHKZJVJJEPUNEACXMOKZXDKDSRMLYSETAHODDZGFDW9BQRQ9ULXSSQRQN'
          b'RYCRL9YJQZYHBSTXHGVQLNIXBETRHZQNYZPDNPMHXIQZVLHZUVYWXHPNWVMAIAJYHF'
          b'DSF9XKYLGPEOQCVEFBIMBNSMUVLEWOMBOTPNGOLWKFUHQWLFTEBKUPZSAXHLK9JIKY'
          b'CNPLLDTLGNJUNELVRIKPXPONORRXUHKIA9BWFRKATZZHHISN9GWBEKCZXLLXVAZPVT'
          b'OBGGUYPAGLPCD9HADLEIBWGWALONELBMZU9HGDBRXPHL9TFU9EPRGUUZQKPPHJRFBC'
          b'C9LNXCKGDRZMWJSGSMWI9VDQGKXEULXWTMROCADMYBFWBEGQXTANPOPAWEQAKYNHLO'
          b'LGALTNSIEZJXHIMRQPBJKDOCMZSNQMDKMOQMXRVULJCPEBNK9ELRAZ9XTCKCKUJKLH'
          b'FNBVFKYCTKELJSKCDVOOACVEAFPFUIZNXUES9KRVPSVMXVFLYVN9UQCNKWMJAAJA9E'
          b'DSYFEUUJTNTAOUZXVAUDNZBC9WCUJ9QNWMSWWIQXKUNNFUUUJXWAPOQDBKVXDUVERI'
          b'AXMCSUFLSRIFAYBNU9FDPVGOZOU9HYK9NUUV9WOLHVETGNASJPRHKBTFFURRIXWRPP'
          b'RXBOPZZYUFCVHZYBR9DXKSPVAKJKMSSBHOXKTAGTEUUCGDAZWWXQT9QJZQEASYJKTF'
          b'SBCXRJQBSACBPNCTAHEZQSNYCEJGVEDK9VTSDR9NYIBMQDUSCMMARWBWTVTYUMTVMU'
          b'TCIRIYGSGSLUUABTVBNAGQ99HSFTMJDMEUFNTDUKBCXWVLPEAHDIQREFBWKZXRUTRV'
          b'OARCTDNCSPLVP9YAJKFJZG9MVMNIKMBNKT9NEPNRMCAYNPNSYDVOJEFJOMAIEUOPUA'
          b'DOFCJBZBULOMFHQCQUDTPNYC9GV9LHSUYFNAEWCIUCNWERVIAKBMPNFKOXGMOMWUV9'
          b'9DCI9HFOGWELJUEBHKOBT9DFXRUNJIFIHJWSUIMRBOUGUVR9QUIJOZ9FKABGGKOOFS'
          b'QATVJEJVUTXIU9NXLUHFCOVHRV9AGBUMBGGBJQKVFZFCDWOVGPZAIAKMKQYQFXLNNS'
          b'PPYNGFVYPZZFCCCF9FPJMDBCCMQHANYURFWEZQIAYCCRIGQETJLUOFKPVC9ESXMGAC'
          b'SUQZNWNLQTTKRJ9QMAXPFFGNYHALLPIVNCEPMJFEYPVPXYGKYMJNQESUUWRICVFPTV'
          b'VOOGJFJB9T9KZRAPYHIHLXA9HGKSRTEGNHCUWZPUSIQOTFWEMHCHBEYFXXFWJTSQIC'
          b'RKRJIBBVEGGNILSTXROYHEXKQBWALNKPMAFSTVCOSOTOKVFRVMFTXGQKOZJRDICBRF'
          b'TOMK9FDMKAWUSKJDEVHCSHNLK9IHODD9ZIHXBCKV9QDEIRTKYPLEZLPHCBGCNXUJCL'
          b'EQFHFJDLXIL9P9WRSBKVFTZXMZEIHWJWJJVSEJZTUDAOXULUB9OJZQHEVD9DTFKDTF'
          b'KVEPFJDTR'
        ),

        PrivateKey(
          b'TIOBFKKFAELHQOCLGJBGJZWVNSZMBPK9D9GTBZWDJBFFIKJWJBKIAAAAPDMYOCPDRN'
          b'BMK9QGJZAMIQZKSBNVVDLIVNRYVWQSHSJZZNGJGWNHGDMYCDZLIIUAPIPBKYSWIFNP'
          b'ZLVUGCN9MIQTYFYJWVIKXOUOFHAMCYZEVBHYABYPEQHXPHIPM9NWENCXRLUYQFILME'
          b'HZYQUAOCIT9HZUDRLNOJPZGWURENUPQFG9CJCAKZRRNKICSUCLYWVEFDLMXIVQCBDP'
          b'RQXLSCFVRIZBDCWENUPAIEWKZLTIUWLWFGOAQRVANDCTVTDWKJOGNICUOUGVQAHKRR'
          b'EGIDORXMBADYNBXLEUMP9PK9TAMVBJEFGFGCXQZQSJNUBOUUCKDYMUAVOJIWMDA9LF'
          b'WPDZHSOMXJFUFCXUIVHRMDCOVSRO9ACUHNEAWCPQCEYVRTMGJWARYYNEYUXQQLQDPS'
          b'RAABZ9WYQUQDCOOYBZYBXDEMINQHLWKXLLOUCRTAWLGSCZTASUVIJJBVDUPTNMOBIY'
          b'BC9CVDPJNGTBOZZHLDWIGLAHYQVKNWKQQXBY9GPSD9R9KZWNNUDDYXYXJHNDURRTJF'
          b'9NXPMVGVSEEHNOKESYRVFBMJGHQLGHGTTEBEPMWW9JNOMKFTVJR9VHHWEXEOAIBNQI'
          b'CFNUXLTZCYHHLADHHHAQHSWDTEVUGXHPMUDSIKGZHRMDDLBPZYNNHUXTSUBWUWZMDF'
          b'999AKZSRZLAKVDQVWWTN9DPNAMHCJENMKMS9ISTDLB9PBBJVPTHAEX99JZQQXPPCS9'
          b'PIOHNVUWINMCLVZVAO9SDEKIQOLZFDOCIQKNPLR9EEEVX9TSNWIP9GOBLWUCUVGTJI'
          b'WTZKLWRQTCXZNDFSMJTHSEGNDRMTJP9UQBLZSNWWI9DPI9XPPCDBW9SJIKHODBMJNM'
          b'HIEEHLJXOXVUTSWXJ9IPGGCVKLSNVVVNJC9Q9CJYHGGCMGUCBMBIVAODZVANGATWI9'
          b'GJFJBTBGFGIDKJUHRXGNVFJCMCADBYKKGAOOCDIXTBKBRQFVCVGPTSEZFTXIMT9PUX'
          b'QECAOKJZOPDHXZTFVNNUKUGGZMLVNZYFSHSYWKNXNIIHRBCRGMHVWSWMTS9DCTNLMG'
          b'ONAP9EIAUDDZ9HZLITHGENXAWZMXJWUARORECXLWSNWCBOSEUMAQGGIISHXWUZ9A9M'
          b'WATL9IZNLH9JJCGAZGJMNEAOWRODWE9DXGPCOVCQGGJOGCH9AA9NFKHYJLCDWUFCNF'
          b'WQNZC9PYPGHNPSWT9DZEFUZFPOTNLDLQSRCTSG9YHFWVNDVTGPGGPC9OYOGEUDAGFV'
          b'OMVZDQXSWYFAYRSHEZ9PNZMIJPQMYXLIXHYEHZTFSDMCUCKMABZKAMZFUXOGIGHRUG'
          b'XABWWL9GRWPATDHEEMOJEMJSSVWKNVOYLQHXUVHIAMMQDEWNWAOPJXBAQVKCBIXDYO'
          b'DSPKQHOBVSJNHALPEYCJAZCGWKSXLZZQXYD9ZXAJOOXNB9WEKKVLFFISCBIEIEHJMU'
          b'OLMZP9VTHGXDYPQVQO9FYTPMAYDWRRAOQGBDFCEWBXQGLYXGBHCDNDKEVJPWDCYDCR'
          b'HZVXTSOSNFXCVZWFJMCSMMUTWMBVDLIUDYTELXUKFSHXH9VKBQAXFPZOKDJBF9LJAF'
          b'TR9MYBJLRADSLKU9VVAAPYYEZBRQYKPHGQQHFTYPZUIFIWMWVESRZXPIAIAFQIBKEZ'
          b'DIIVTJMHDTPZBQVXSLPTPZBCYVJFYKETYIJQSURIDGRLYZHQKNWKMWKTVTSBFVUZFS'
          b'QXUAKQE9AGO9JHBDYCMTGDIRKWJCACCJLWKFOKLAIYBXMPF9XHCT9SWETORWSDLBPY'
          b'HSIFZBREQLVLSARPYAFRQUDXYRB999FZFOILTR9GZDTVEFXOQZCAX9FQJRXOSXF9S9'
          b'KWJJSRIBDKKPPLWWXPJDLXYUTTRPZBCJRRWLIUEZKOBEAWUVVHKVEEQUKK9MMWWCLI'
          b'GEC9LMBXBJ9HSJR9OBWOLUKSXRJAJAIPXXASTICFAYELQEOIRQZKXHPWOSXGEUNQED'
          b'VUZNFZ9T9CSECMDQ9ENSXLFNONKZNCNCCQGIJQAKSJSKFUXXSYWVNBFTGKLDMALNHD'
          b'WYGBJ99Y9OSENWWKYPWP9IKFSMRFI9QCEFTUZKZTZRHHNXXMEPBGBWHBKYRSCHQ9U9'
          b'JJTUIQBXZ'
        )
      ],
    )

  def test_iterations(self):
    """
    Using more iterations to generate longer, more secure keys.
    """
    kg = KeyGenerator(
      # Using the same seed as the previous test, just to make sure the
      # key generator doesn't cheat.
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999TPXGCGPRTMI9QQNCW9PKWTAAOPYHU',
    )

    self.assertListEqual(
      kg.get_keys(start=1, iterations=2),

      [
        # 2 iterations = key is twice as long!
        PrivateKey(
          b'VBGOUVYVPMMVMNYIASWCSDOECOVM9BLBAWECSHKURDUVTFBRMUCIADFORQXPPCDVGM'
          b'QAGEPRS9IVKLZTJOVOBFXWC9PDXBL9WCRYKCNNFWOAJCKKV9KZICCQKUOFUAUYTRJH'
          b'DTCRDLWCXKSPK9ZMVVFMGXBPQRCVAFMRAGVHWFKQEDPXPR9UTYOSKPMGMZZWLW9SBZ'
          b'PIQKJAPCAVMOWRURRUKYP99ILMGYGNUTCKXEHXTYHPJXUJJBATFLBJTZZSKUZZVTRU'
          b'BNJJJCBXTVNUVFQQPOYIIRVJGJQGPIDDSPNRTAQLHBKJNJXSGSUOHXSWRTDOSLB9OG'
          b'WPRBDCZMRCFEUHQMYZGLRSPCSMCYHZQLRPYTFFLVJDYTEETUVIAJLKVPWIEGKJ99LO'
          b'LHSDQNDTFIXTRY9RCZTOTCYJEDIK9YOQLEEYIAMTSOENXLAALCHWJUSYFJZEZPVBUH'
          b'NSIQRUDTVQUKJZWYJDNMSMFDVDTFSAODYOGFYK9OTEIHQGSSUVINXURWBAAYVNRYVR'
          b'YIHQSSPKQKFKVGJRTHMRMSBLIWKHSCFIMAJIBTKMVADBOEWTANSJZCUGQRCLVIB9ZY'
          b'PYPIEJPQAXOPEEGWYHJJTPZRHOSTSWIPQMC9T99IMALJONRAOLSJVLGHAQSXNCCIBF'
          b'XBJZA9FVP9TBQDHNJPSAWSJ9HMCKDX9RGQEUHOHLXPVJHCFBEQTLDVXYOMLRNUGFWK'
          b'UGXNYDKXKAKNHFYVO9DXNALJLUSROLNMPSOOAQKPDHGXHCOAVMYRQDPQQQRBQESBOW'
          b'L99CCPSYYYSMTLDLVEWUFWLAAZMINKRRAYIZRENIKFSYFROVZFCYQIH9BYSTUEWNHT'
          b'SA9LQDET9XLMYAKAR9MJILMHKMZAZPXSGVMKMZWCNO9RGGPJBGYRDEIZWSALUVVYEZ'
          b'VKJKCIOWBHLGUICAPMSCOKHGYVGUTKFCEPZNFQPUIQTKCLQIFKDQYESLKYERQPQDLS'
          b'YXIUAZVHWGFCAPVQOITULEJEVQRQXFYKZPUEVKWKQENGSGFLMFSPPBBPZL9NPFYILD'
          b'MWCRVHYYBEIJPLGSLDJ9ATXIWHPM9VSBZBOLAVMLRUHLAPDVHVAMAOWPQMP9BUYUIG'
          b'PWJCCXSTCNABWECAVAJGBBSJXIDOOZKWMXTENEXHBLJAZHNFTTORCTHELRFDYRCCWI'
          b'YJXUL9BWE9YPVTHVFCSUGRPAOTTKIFJAAS9MDRENZYAGEHARCOVARJHUWJCKLHOKQA'
          b'ZPRXOGKOBVCGCQFUJAMQZMKQOZPXZJZKJCHADNKZV9ZKQ9DUHZOCNTU9ZGKWHJODYH'
          b'JVSOOCFPKSAE9EFOLSNWYXYGVONYLFBHIKGRBMSS9URVCITGYCHXKVWYRQYTLWZLOU'
          b'QXMTDADHDZGWYKWQVZLSUPMTPBHWAUWQLLKXHY99QNSMQXAMHHWLVL9WHTALBLVMAP'
          b'WVFFCPXMUZTQCBPXNRQMKDEIZBIOCOJVUYGXFKMZGTCIYMX99RGMGKMWTRJ9N9ONLG'
          b'XJDPMYECS9AIJSEDALMOMSHBLDRI9AYRWDEZYNRBOEB9DIDXXEBOPBDZPPEQTKYRMJ'
          b'AEAZWUYQXROVBY9JZZGBRDUQFEKHH9TZMTLNNFEZBFFPEXQIHDZE9TBHDYN9SKGMA9'
          b'SYZXUSBOOXREZAQMFFKPFGTWUGUDV9SKSNH9ONNWDJYKZCBYCZIDZCFOMBNGNCSQDI'
          b'RCCEMYSHNWKVWJFWFPI9UTYEZRZHECQKGBGKUJEEGQQGSVNOVMIKGBRPXACAS9VNZW'
          b'CIYAXUHYOFJCXTAFZVWFXVVELGHCUNISAMCVJDNW9XXNLSFOXHWPBNYCUDZHHDITG9'
          b'WETROSQWYMRIIAAMLGNJSR9HMSMICDEWBHLXDJNMGVLTWOCEUPTFCNCIZZ9OGIANLO'
          b'WKFGGKLXODJDYNIDSECWHBIYSYHVUDROUGW9V9KBQCGMZVSUIGYMKAZSMYTLBKDNYH'
          b'ZLPFTCJZCDAPGNWJVQBLYVEWAEQKGGJMLEQPUDXPXPHZPYGTHZCGPUGTXNGUNLZJZV'
          b'QOCSNRTMVYANLYHVBRZXPYAGSQYDGZFASTYTXKATWLJZHGUF9MH9VWAGGWEVRCKUCC'
          b'PYSXYKHFRVHGLZVTXYJZ9WPOPBHLPYMPQCZNOKCNDNKDNQOQTYCRAPCJDQGRCBEHCM'
          b'REONEIFHTYTHN9XKG9OCEPEFEVTPCKRTVYOTBCKBAEEUDDNWINAHQTWWREQAHW9SIB'
          b'WLZJSTEHDVGJYLYHBODMSDWGOQIBM9HCQNMHAGGYF9O9QWBXSPBOFCOGYBMKXXKQEA'
          b'9YDRIKBKROKSBPYIDQNIHAHNOMPTOEJVTLVIAVGZFRPQLDDMDKCIYATYUYGPVAMHOT'
          b'JEDXYGTOZHQALLKTBBJKHRBGVBBSSQSAGFABDMJCOAPFXLNPWPV9DTYLIWMEWIH9AW'
          b'WQMVCGFOFRTDRTDNIW9DJUUISRFEWPLOCSVGPEMXUJXQXBHVFYPFPAHWRKIWZXQCRC'
          b'NZCSNDY9EBDEGARIXIVTSONP9GZLATF9TLWLESEXRTWIUGMAJFTCJLNBSHXEPNNWJA'
          b'HNWSFBTHD9BFYIN9C9WEJBLNJNSQTGJ9H9ZUSAIARGZSLFGXH9SMALFIMNIIWHTJOB'
          b'A9NDZICTFRPXQGMLGCZ9JCGLRFBZHUVBIWGUDWGLBROTUQWNPSD9RVC9ALPBJWEYIV'
          b'ZRFWUUBNUPHDLFOAT99YMZNBYTX9XWWUSBZGPUP9LPKNPJOMQJEVTXSXRJACTLPIGS'
          b'XFFUUICWM9TQDIBVTDEIROVSTAJXSJLKJZGCCC9JCPTKKBVTEYMDLJMLAUPZGQCWWA'
          b'ZMMUGZPSWLZTJTGSSWHTVDWHHV99FKK9PTVXQSTRFKMMH9QX9IWLALNSXITGTHQOCF'
          b'PLVEDVDBCUY9XCJQZIKV9NS9LIYNQTVYFUSYMPZTS9PUQHVBHNTVPZCKIDHWNOGHO9'
          b'XMEWBE9HMBNCCMYGBOVDJO99YV9SARRENSNOG9JHYHIFNBIFYZPWIRWZGSOTPFYOFX'
          b'GVRFPVUBNERKXGKDYENHECPC99JOCOFNDULLNDMDREJWLBWFEUIDJCUQVDERFDYDYC'
          b'FID9VDWWDBECVFRDSZGPMTRNGQHAMRXADYFJJBGKEQEOVXSDYICOGUYDRBHOAXRSHJ'
          b'XPIJIDWAVSSMGDAIDUCDKIIOZEDDKSSZSFOPRHRNPYAXAQOMVNDSKJIP9KTQZBFVOZ'
          b'ZZMDLDLQCCOQNKQEA9FCCGDMZGEDJIWGZZS9GIUOHEAPVDQTSTWTMVMJA9SHZRHZCS'
          b'9KOF9ZSSJWKA9LLK9TBWAZYARJJQVWRKZQXSMRGUOXFWLAIWERAVIWAHPXAKPCRYTX'
          b'XMYXDGSVG9SBPY9IWVAUS9VWXVLGTDDEWMFLKQTIRDKNTNJRNALMKJMMAESGODAUWK'
          b'WUNJZNG9OLZCZRCQDFGVSR9UXUOVDJBOXMKFB9FWFSTQHVILADJHEAGHWBICNZZGXK'
          b'B9SDZPECJXXXXMEHPTISE99MOCHFUDJLKSZOMWIAFYQQY9YCSMZC9RUFPNBIHUMXLI'
          b'FBEDUDFDOKQGKLMWTILEEXOQLAYEPKHZQWRTXEKXJYCGIYWZHWVH9PYZRYXLVMUEBT'
          b'MPSUAAHVSBQPXBGSAUUWDFNRJJIJLBZBFBQZGQKHKFQUY9FKPHTOIUZAKUKUQAOXTA'
          b'VXFZYJQZL9YMAYEEOMPYAILPEPTJMRGBWNAARSSINLJGD9ZFQELMDUBNHPZWYFZQGW'
          b'QQXCNGOQGMTXRFSXYOXZFCBTSTHBQ9FWYEXHKZEWQNMLOZSHJVKQY9FC9FZQKUTKYF'
          b'FLMMNXXPKOMOOQL9F9HSANCTLOQT9YSTGQCFBEANXUGFKGKQQJFYRPEBBVOFZCGHFJ'
          b'HKFGISYFIXDZACBIDGOUCMLVZQOWXFARMNEWJTNBSXYSZNYYXYRTZAQFVSHAHR9LPP'
          b'9VFBLSQWFRCGVUKTVXYZSGFRZDQGUYEWOBNJGVDGVVQDOQIHSWNTHOURYPDVZDHTYL'
          b'VMLSLCENHG9AR9OQFHRFODEHOGVGTYLGUYG9MZK9VPOTPVOINJRJSNVVDK9WEBCHJL'
          b'FR9PAWCEJGPQOYCRQCN9DWHWDBXBRXUIXVMQJCFRNLRGZOQEDQCSRFUNLMKNEFQDVV'
          b'CGFF9VWUEXGYYUFRJPJNKQHXXGKRTUNTCCFSMDMDYU9PLQEJHSLFX9XQ9KEFNLIFDU'
          b'TENVG9VGPXLZPJCLIYXYMZIJFXWCBXQJDFSBTLHGMHLJHUCSNWDVLKIWNPTCEYCNOD'
          b'UFKOJCHLELHMNRPEXOKTDE9VQUIRCDZBUBKKCDTQOKTKJGRSJJEJS9XUSASFJSQWMR'
          b'PMWYTXQHXYUOMVQCVH'
        ),
      ],
    )

  def test_error_iterations_zero(self):
    """
    Attempting to generate a key with a number of iterations < 1.
    """
    kg = KeyGenerator(seed=b'')

    with self.assertRaises(ValueError):
      kg.get_keys(start=0, iterations=0)

  def test_generator(self):
    """
    Creating a generator.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999IPKZWMLYYOLWBJGINLSO9EEYQMCUJ',
    )

    generator = kg.create_generator()

    self.assertEqual(
      next(generator),

      PrivateKey(
        b'LZDQHHPRICEESIHX9VVYCHLYSJDMXPLFOMMVOMUZKHSXLQYSQNDFHWLNKTCAJYPWUM'
        b'RMILEVQCHTGILDRABOOXPNRQHWBXVBRIZELVFEHTMYITSVBBBUQCZMDEUFGCTKHISF'
        b'MI9XQOBAIOVMCKNIGJQANHDMCWHYFJLDQLPMHLULLZZXYNZUNDLMRITVORDUEGNWKH'
        b'VOFM9YRMGEOM99DOJICXVVFUHFHTZGFROK9LWRPZZMB9GIVUSEKB9KEZOHRSRMVPOS'
        b'IGFM9INOXQSKTBI9O9PIDRUUWG9SDWJPGVK9SZMDZ9XJDHCIRMYBVZSXLMYZOWKZEX'
        b'TRCY99JKTTESVPWPTXCAOEKIFC9IRCXTRTIOCDUCJZR9VQMHUXSZTSKFHTABGZBLSY'
        b'CMKUHEASAVH9NBPAIJO9QITSPIEGJNCINILWFZU9BCXOEAZ9SGXKDUNOJZQPCRVWEA'
        b'QBNYTYAPSEFGQYAXYNIEFVOQZIYPDZHHGKANGQXBT9GCCZKRFUJMQLCDOBXWLQWYYQ'
        b'KOYXPNYHJWVGYPKLW9XIESVOX9MSOBTMJHTYEYGERFLUCPCJJLGULVEVGQRB9EATRW'
        b'ZBFLGVMAORSIWAIWLMSEYK9OIBOHLOXLSSIOSRTMLHLY9HGKYEOSVNDLDNKYFPVUYF'
        b'FCYEATDCRRZSYAJRSJTZ9HYEATYKORKQIMHND9SALEMSBSJPLLAIHK9QLWTSKPQUC9'
        b'RLSBBHMEA99HCISABTHXDVBEZYQYWJLAFHAOGFBJKPJNVCEBGAYTYCONOWOCFZTQOF'
        b'NUFWRTPWMEWPV9RGPOCGUDJGDGDJKKWTONTYXTM9GHQBPGOBYKTRGMBKRFUBSSGGYZ'
        b'GIEKBNLUJKVFFSIVSVCL9FGLWZOGFUBZWWWQROZQEPUZDC9GUHJBBUDHATQDBIFKOP'
        b'VNMQAFMTYWUXXJSL9XFRZCJDRNKWWUBYVCQLTATEL9VXIQUO9CQOKL9XFKEYQBGFSC'
        b'GPVEVAMR9NMVHKIHAGVVXXAMNSFKI9ZEJXIFCMNPTJKDARSIFKIGUNVNADIGFHQEZM'
        b'IBVFTEQQJPKZYFGYHVJOYJUCXSSBEXNQXIOEZVPOTSKRGRVMAFIRMH9LDTSQVOHO9N'
        b'SUAMRAGYIVCUZMWUGJWTLISSHBXXRNTNTPGRGFTXEPBWGOFOVSZXTZLNZEKQQNRELZ'
        b'XXJQMIZRLIKQDSQIUYBXTRNTUJJKIRRVCFWMJHOXQXRSRJMESICNXYW9QYEM9JNBBK'
        b'CSFEKPQIABSMLMXROFOHUNBZBOHNQVOIMERBZMNKXTMUZWRYRFJISOGYEDTBYCDKOJ'
        b'TAJGAWI9SSRQUPUDCDMYVFUKXSUHXROTHUSLDZGBACEANJAJMGLLKCKXSDZQPSWDHN'
        b'HDIAEIYIODUBOSE9ALOJONOIDS9ZIKFAGWXJBJDSBJEJK9IYUFY9CPVAEVOVPVF9PF'
        b'YHZAGYUGWMDDOWCUSISJEZM9BFFJTOPONR9XZKV9OLVXLPEQALITBGNMBJSGYORIH9'
        b'MKTFYCYZD9XTKQ9RBCPDTGFJRGWDIWMSHDSSKGWJVUYWO9BCETIUHJUFYFDYADILCU'
        b'CRSQYGBTWNAYEYIWBXLINOAUWGLSKBRIWGLPPX9FYWQHSSBJCNBXIYBQWBEIQAQCAO'
        b'GSNXKYKDRNCIENDGHAGAQVETYLLQKZSYBWNCFWK99UEIDHXZFJLRKHJZXIO9ZFITBI'
        b'BVFPXJHCEFTECVTXWDJTYFTNIUITGPJLBA9BTAJBYGCCDYXTXIWZXDFFHKJINAJCGX'
        b'ZGLKEWRJBFOFWOD9OJEYSOOFGFWQHREKQCWLQWTSOFN9EMIBO9RWZLDYMXEUTPXZVQ'
        b'KWPMJQTQDUPTFCGWYGJITQRYFZXWYWQ9VYILAU9RPPF9CGFIYNLJNNJLKVQGK9WJVN'
        b'NQTGRIKPDFVEWUSKDZKYPUYXXYSKGEXI9DIDIJPGTRWYAKUQHLEVMJXBACKZNPRIQE'
        b'PIHONHAKNNODBZZUGISKCFFAUBPLFLDNPDTIGYRSKLZJAIXWSPBWQBGKWRPOIJCCZR'
        b'JLPXRDNSSHULCXCXIXATHFG9MWLZ9FKTORJHTZCX9BZWBBMPJOCGOVQWL9BEIOVSXJ'
        b'TAPYQCOSOUTFNQU9RVTCOZNSALMPKRXRPKQDRVLPOPBWMCBQBVERXIIEPYGKUWODGP'
        b'TZBGMVDVT'
      ),
    )

    self.assertEqual(
      next(generator),

      PrivateKey(
        b'BTQKIPUTHFDMCKGVLBCFGEFLJHCHOANY9GBIRSODCOWTJGUWGFMDXWEB9HDNP9M9ZU'
        b'HVZQULQHEMXV9GMXDNYDHEJQZ9BSPIPHIVWNZDJWWDWONDKNVRFPZWNXZAVVTZCCMF'
        b'DBOYJZCGKIFJDKLMTCGGFUJAZ9FPOIBDSHDZCUYNVXCCDAFZLF9IMKZWTZFUMZSVCT'
        b'SFKUFVTZRLORSY9QNBYGEFLVITWYYJAIXGLNCMOEUJRVUHAKFFPRFGFLBXFAXJWKGE'
        b'XZOGHLFWAPV9ISJMBPXCNXZDEQTLKMWVYFJBDKEZJVS9IX9RAKNEQCUQWEZZXKLXYF'
        b'F9KBZZUOSVQ9FADRSBH9MQEOODDUMYJSZFQMUYNEYWWGSVBGCQLRDAJHFPCQEMKBG9'
        b'DAKQZZCDGHOWXV9ZAZYQDGQAVTAUWNMKNBBJ9NGFBRWBWUEUJTHYF9AWSAYBRIOBRQ'
        b'VSWAAYGGUGOEARZHSNBCFM9FMDICLOVEUDINON9YDPYNTKFBPGYSDEBYKYQCNDCCLT'
        b'MKGYNKJFEHEUXENVNXVJUMI9IFHKMDPOXNPWQAUVJLHAWJMDFSZXIPJQPWHSJG9WDR'
        b'R9IFTZXDHXSRKYISISAR9PBXL9R9UWRDLDL9MHKAOENNL9XXHJZWFGYRX9AJ9NIKLX'
        b'9NHJRKEZCQCBZZJUREZHWLQY99OJ9MRYAWCVLL9PJDEMIMMPVSUHUD9ZIDDRETZBOV'
        b'DOLHJDCRHCZBHDMUCXFBLJQQSHSBGWNGJOTRRXMRADROBTLBIDUBLBJESPZXKTKMUN'
        b'9BDUTXPBZMZEFLEVOPIDMAGLLJEEOORHDTMAZBCLDKNOCGZLK9BOEWVAVEGMBBQK9E'
        b'DKJQOQWHVVYRPEJFMBULFXROSVEZJPUTKOENCMGVPQWDZCVVJEFIAYSVQWLHFBEJRY'
        b'TRRYL9FDTYRXJXGMCFPXBFXL9BJJTTBQXMRCZOSEPCCUX9XQKDGRQYHA9PRAGHYOLU'
        b'TXSWWXWGEIHNW9JDZCQWVDQQSMQNAR99OYUFI9BHXZZLPFJXOFAULIJWZKCDTBYBYV'
        b'PJYUHINJSODSHRLYZLV9BRLNUTBFRNMEQWIBJHLGYDNPAOCPBQVIPSDDDOACISQB9O'
        b'QRRLKEHCHAGGDFUHEWUW9AHKHJIMMQHXYFNPEEBKLPFAMLGITBCHBAAWUAQXZXGRPM'
        b'BKCRZNZQSBZTUYXXFEVDOKXXYYYLIUABRDAWYANEJZNOVUQJVNGYSLCYJKDIPUWK9F'
        b'JVWHQAFLYMDYQBBFJRSQULZ9AAPVQAMHPNLQR9CULQYGTIMUWBTWKSWDETYDGORKVL'
        b'AMTDFSVLKIZDQONCK9HXMYDREBYYWUUZFUJV9WSIM9USEEOZPLNSBBBOFTDRDFQTDJ'
        b'QDTLOGRRRK9KBSXBCSJF9FXRELGUOND9OXVWQSVKWEUPPSSDFZIOWQUZRMDRJWJUZB'
        b'9VUEGUQBX9JJKHDW9EABVQIZNTLK9TSZDLPINFYEOD9CCPGDPUPYMUUEPVXQRGHBBX'
        b'ZYDDRGYKTPVXLNLDFNSMQTYGRHCZNYJQPZCOTGGYCAGGPBRUIVDHPBSGACFUMOZJGM'
        b'JOVKWUOHNAZVSXBTZUKCHCRYQCFNUDDMN9GQOWDPPOVNAQOFAZVIEHZOYAJJBSIL9L'
        b'BJIHVRELXSNHHWNPKPKAMRVZXH9GACFNXLLEAEBEGIGRGHTI9YONIRCMAXIVJUGRLW'
        b'DWFDKBMPFPOAPAJ9ROCVVHHZOIREVFYSDXC9UKQIEHVSZJO9CTDPRADZBDWNUIWGRO'
        b'FFRFYAMRNWLGSGZNYUJNEBNWJQNYFMKOKDXCJOB99QEHWZLCBXMIGOWROUFHJATSKY'
        b'OYKMPVA9TXMM9Y9KOHUJZ9XRKAJ9TPTJYOWBNKXQFKLMWNVFZRXGBDSQRAGGFVSABR'
        b'MDEQMHKWWEJMBLVBKVHHUFOVBSLMXAYTREGXECZTHOSDOQLFXOCCZTKHVLWBEYUIEA'
        b'JBRSTMFHGQK9UCESUMUBFZGR9MQ9LIZFYNVXOTLJJERNSMUWMUWDCWGOGR9YLPO9UK'
        b'QJWLOKEHIBGPGSD9CY9FQKUBVABDL9ODRPOZDWIEQBEKRNZDYWREIOOBU9XXLLDDFQ'
        b'KFEIESAVXZIGTHLYZRTHLRSLKEEMTWFFLZFUVEJFEKZUEIZWBYADEKBUIQLJGTORCO'
        b'SGXJHVGAB'
      ),
    )

  def test_generator_with_offset(self):
    """
    Creating a generator that starts at an offset greater than 0.
    """
    kg = KeyGenerator(
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999FFRFYAMRNWLGSGZNYUJNEBNWJQNYF',
    )

    generator = kg.create_generator(start=3, step=2)

    self.assertEqual(
      next(generator),

      PrivateKey(
        b'CBKFYOTDYHFWSIOUYAUAHQWOOQDNNQBTSSJPHREUWFBXZFYFHPHZJN9ILAMZYOXLBQ'
        b'MCN9T9F9VSCUSFVQDZROPWBKZ9ENENGFVDLMBNDOAAKZXVBXVAYJLSGVD9ZZSMDCRT'
        b'JUUJPTNRVYYTOALRFOIGAXHUIXWCMXEUGTH9UJKYWTWBEOVWPMNVHJSXXKBAUPYCLS'
        b'XATLJYLFNZ9NLRTPCCMAEFGLIONLXPTDCCAWCWNZNCUG9HKCRXPQRCUWBUDSIBZNCY'
        b'VXSSHCKBBDSHJWZSUHQPVWJGYEEJCPYUDX99U9NBEJXKFIJDELKSLJOJ9ORMHA9FRU'
        b'YQVGDKRCQZ9CKRNISKMRNWKJ9RGDNKFFESKBPQMIRDCYDQKRPYJZGMIENVBADUSDIX'
        b'AECNYIBPXLJDKYCMAEZHF9JMIJZMFZCABOJZFAAGWKFEZVOCTUIEGWBWYDYUCGAQI9'
        b'LRQOJVJUPATKDBKPWTMOYADYCCSARRPDMUXDNFZTXB9KLWCAINGFKENGUJPHVIPLLL'
        b'CVTLESCANFQDBFOPKZTBRXDQYNTWDGMFHUIDJFEPBGMKWFRXOTBDVPSPFXWEMBXBEO'
        b'LLGVWJYJ9HLIT9OQSAWWPFSQKQHJJXUKLVYSRQCLXIVUXWBUXAHV9SU9VOOUOSMVXC'
        b'BQSMUVDN9WWOQXIGKTJPSEKFBLGLRQH9OEO9HFYDFDLZ9PIWCEYIINHHWXYZHESAWK'
        b'XICDRDHDZMPDETWQNMPLGARJTMYTPMZT9BNZFNC9HMDCCNCKZPXOHZFLLCVPVARNTI'
        b'QYAIGLC9SAHOKVELOXUVCVIFU9ZPVLNLSDTSKQJZKMYRXTD9STQEFDLYDQILBTISLV'
        b'99EAWHJSNDBPPLKRPMPLYKPXEILA9XTVTKDAUYDVTYYJHWVMGRVEKGNMYDDWNLJPUU'
        b'PBUNCZXTFIEWZDNGGSDLZZIOLM99TBHEFBJ9GTFDXNSMWFIYKXELWQYAQBLDNYSTTL'
        b'YGLETFJJRTKRFVKRFNDTJVVQDEJTVWQWTKDIQOZTUHIAWJNSNCWNYHOCEZBITEMI9V'
        b'NDCMPQ9KXQEUOTIHBJVTJHLSWRPOHQWKJDI9GCFOSABBKCDRHGV9UOHKEORUJVPFSU'
        b'RKKXXXIWNEEEABGYNELGAKVUUPYMXTSSFT9Q9KCJTEAASIEJRHHJJIKJEUVM9A9JYZ'
        b'YTZW9QZR9IASSDZGGBMOTZLOMCGEXPFUTDKXFLNJJRNMIURDSNRQEPHCVFCQONMCXX'
        b'QIYEHOWB9WOCBZVSVKARB9MP9DQ9WAAZDLLZNAPLPAZEWDECSNLBKECVVWFXGZFQB9'
        b'OUSIFYLSXQMVFCCLDJVXGHH9LGLNW9UZWVAYYAMGGOVGTNXWBGSQAIEAQNL9EJACRF'
        b'QWHUVKDHKRLVAFWFX9HMHLKGWIYKOTVMRNIQTARCNMRCIXPYJJGOHVC9LBWCTFSCRH'
        b'AEKUC9BJHXIVFFPGEEGBWZNCSRSOMSUOLNSUXUQLOPWZDTRXIHVZZLBCCKYNXSRIGH'
        b'YEACLIEIANAYUT9JCLOGUPMRQRXPTOPKYBENALUUWFU9DXMGECFULSQKQFRRKLSAR9'
        b'IUXGNYJZUIMBARYLYTSM9KQKGJQ9LOUVSURPPAILTNOFMKWTPNCXOWVYWNDDBXEESZ'
        b'LEDTETJVMXOOHUZPCQRFNKWVDGOCTOTHBOMWWMOMAOEUIGVBRSMBBJPZASDARKOBFJ'
        b'UKQSLUBQPHRXJNO9JYBNMHHTWCHYEICB9GGJIQQFTFPLOGVJGDXEZBN9XYZS9BKPXQ'
        b'QHXEYE9HISINR9DHDFGKLSO9TCDRUAGCRFZIQVCPJWPQMZGUXXXBOYHXZLRWEIZJDR'
        b'WYLQSPBTJQEBUBHVXKOCQUKOSIUTTAGWBBHTQTQRXOMVIALUWAEQMQXNYPAHJABDIW'
        b'VGHPBGDFNUTOWQOFZQDYVDKDO9NZHFVLFMCXZGVEXYGIKJLGYOGXVXRIMBBCQOAF9H'
        b'LISJHCTKSXTCQXCPLGSCZKQTE9CLWAIVPKGVMECLAAHFEHAXAZI9AZRCFVKX9SGPCL'
        b'ALDSDAL9KXQLYDYEPTXQXSTNZLRLMXCSVSGJGNAHGKJLKMZDPXMHSOONBWIU9EGTIV'
        b'9Q9DHAXLMSOEG9BXCNERYWTSHCEQBOM9USPXLKXTGQUDKYSITRTMIF9VPJGXGOEXSJ'
        b'GBWSNGI9O'
      ),
    )

    self.assertEqual(
      next(generator),

      PrivateKey(
        b'OTOFWEAUQFRSWBTJPPAACBIPCTMYAESBAGVMPVMH9IQAEEKEXVUCSOWORDIQBRZZLD'
        b'BXNAKQAQ9XMVRKMBQTBDFGJZEQBVRLEOUGHZNCJTSZQDDICMLCWZWF9FGAIFDNWUNV'
        b'XMVVXKNCCMRD9NNAKEMILISEKDBYTLPEUXTIGFRWCXKNEOEWBZPHLKPVSNYCEZVNVL'
        b'YZTZSCETSIPLBV9JMQDGZJVQHPVRDGFFAE9UYL99GIEXKPJXHXJFKHAPXBNN9QSNUC'
        b'IHMHFIVNLBXM9VODQJWK9OQTUI9KIAMPHSOUGXDLJ9HDMCKWRORNKJV9YZYEGSXQZO'
        b'VMICUMDOYJNRSYZQALQMLLUIKTEKOHYVZILBDERRROLRXYNIRWZYDOGYDKSCUJWDZB'
        b'HHQUGZBCSRUCLOLDBHUNKASWQOT99Y9UMAZALVQPNQGYNKAEPIJJIXFVZJZATWXLYP'
        b'JBQSFDCSOD9MUMYTFAIDDGMCFCY9GUGCDRXIGRDOMSTBOEKCKFBWGFTGJCMOTNSPJY'
        b'UARTXPHERZIGQKRAHIMJCXMWPR9AA9UHXNKKHQSUFSFSTGMQTHDCYBNVLXEPHIJLUZ'
        b'IECWNNI9BZQWQUMACUDXITELAKFQHUXOBCXKRE9J9EHGYCKGKB9RGGXVBALUDGPUOG'
        b'XCSMVBDPOCGTXQZZSSYKBMYIJCKVAG99BBCZ9VAFQSZBICEEKAWNLUDUNDSPFVBWAH'
        b'BMACURTREUIOYBJIFACYVPYXDWJXROXDKPGAYHNIFJVPDACYQXFVPCCIAGWK9WMKOS'
        b'IQN9KBEOKIKVPAD9GJOR9KWI99NOSXSJUOLFPDNFNX9EBQ9QOFFPCSUA99UIDEHLG9'
        b'PPUO9ZSZFTITOJTQYYDNAGJNMUUHKIS9QVYRRULFRKYYQYQNQEWOXHDUG9GKMAWIUJ'
        b'JVCXGQOBVJYMTODRGSLTFFLWMXZFLWSYXTJBFVREXEAZHWIXEABFBLMREPGDHAQRNY'
        b'AQ9ETQZOCVPP9QWKBGDOZVMMHVOZLSXOYVFNVSDYUDIQSMOJJOSHSU9HUMOGQGOJXP'
        b'HPAQWUCDM9SQEZSFUTNGROJYTYNEAQMKVSAHAFIWCC9AAZIQSZLEBTIAEFCENIRXCK'
        b'FEXPFUMQEJKZGCWQVDUPRAPLXCWNKUEWYVSYAYQSYSGLBZFX9NTRTSFCYLENLBPDDP'
        b'JMRVW9RHNTLDMLFZXPYGBRUSEWJN9WTEOSFOFZLMHSWXINXLGNGFDJBWFKRDVYUUX9'
        b'ZNLIRJZTKQVBWRJJQ9AAGSMCPBOINBGZSATBSUEIIBNUAMQQUTXCWXWMYQFZSSVNAM'
        b'BACN9ISAHFKS9VPZCZX9CKAZGC9WYTBBFZQLPOFK9QHIMXFCXJULKFUYIOTTRPFNUK'
        b'GEGUQMKDZREYWBLLPAVLAWXGFYXTXOVMEADBFESCSYHFJGHBVZEJHP9SDNOPMWHXKA'
        b'DTOWPBNYNOPZGTTPZSWYXLTATCZHXZXMYMQBWPTGEQEGZM9NJTIFMDJBSCISTPKTWA'
        b'APCFNTIDK9LMFGM9EQE9LYHF9POZEHUVSMIUOAGVKOJZDXUIQUGXGMOGJEWEBOXVFL'
        b'JP9HMDMYOOMXGQYTPPWXDOCTDAXFALT9ESVOIIADLGAXOOHXVVEHCMVWOVLHOKMDTB'
        b'ILNAZPFRCGTWJM9M9DL9YDKUGOXKJFHSOSLKZPZVGGDAHQZTODPPGAHMGZJISGVQMB'
        b'ITZUALOLSCDQYOIWVKQOURQBAVEADUCIYIVQPKPUJLGZQJFU9VOK9DIVFMYIRNOBQF'
        b'NNFIMND9PDKWDM9MKLEVXOQHC9CGOFOVAKHIZXDP9TIGMJCLHFGPWXKUDYWOGQMFFF'
        b'SMRAMJQXWIXGOQTTA9UPIZIDRCWPMNVXHOKHWNQTTXHLQOZXFKTYVCYDZXAEMGVIHQ'
        b'ODIIVSVGDNTSTXHS9DLFAEIGCLBCGLBMJ9HCSLIGKAGHGKMPCMXBWYRVBKSMNPEUIG'
        b'HKZ9Y9ZNBSHWJFRGMYHMFLGHWJDZDMYZKTXLLGBVRYCYKBNHCJQBHAUSVCSVUJJAWV'
        b'YYJ9KJJZHHYRROSFIARDIMYMDMSDUPZAHRRXWEGPH9KIPTYYJFVEYNMEZHOAEPMGBN'
        b'SLTAJAVLVUOJNNXNPFQZMWDDFXORWRBPXGQSNAPNUMBSLFRUPK9AJVPWYPPBPIGQMP'
        b'RONQOHEYR'
      ),
    )

  def test_generator_with_iterations(self):
    """
    Creating a generator that uses multiple iterations in order to
    create longer keys.
    """
    kg = KeyGenerator(
      # Using the same seed as the previous test, just to make sure the
      # key generator doesn't cheat.
      seed = b'TESTSEED9DONTUSEINPRODUCTION99999FFRFYAMRNWLGSGZNYUJNEBNWJQNYF',
    )

    generator = kg.create_generator(start=3, iterations=2)

    self.assertEqual(
      next(generator),

      PrivateKey(
        b'CBKFYOTDYHFWSIOUYAUAHQWOOQDNNQBTSSJPHREUWFBXZFYFHPHZJN9ILAMZYOXLBQ'
        b'MCN9T9F9VSCUSFVQDZROPWBKZ9ENENGFVDLMBNDOAAKZXVBXVAYJLSGVD9ZZSMDCRT'
        b'JUUJPTNRVYYTOALRFOIGAXHUIXWCMXEUGTH9UJKYWTWBEOVWPMNVHJSXXKBAUPYCLS'
        b'XATLJYLFNZ9NLRTPCCMAEFGLIONLXPTDCCAWCWNZNCUG9HKCRXPQRCUWBUDSIBZNCY'
        b'VXSSHCKBBDSHJWZSUHQPVWJGYEEJCPYUDX99U9NBEJXKFIJDELKSLJOJ9ORMHA9FRU'
        b'YQVGDKRCQZ9CKRNISKMRNWKJ9RGDNKFFESKBPQMIRDCYDQKRPYJZGMIENVBADUSDIX'
        b'AECNYIBPXLJDKYCMAEZHF9JMIJZMFZCABOJZFAAGWKFEZVOCTUIEGWBWYDYUCGAQI9'
        b'LRQOJVJUPATKDBKPWTMOYADYCCSARRPDMUXDNFZTXB9KLWCAINGFKENGUJPHVIPLLL'
        b'CVTLESCANFQDBFOPKZTBRXDQYNTWDGMFHUIDJFEPBGMKWFRXOTBDVPSPFXWEMBXBEO'
        b'LLGVWJYJ9HLIT9OQSAWWPFSQKQHJJXUKLVYSRQCLXIVUXWBUXAHV9SU9VOOUOSMVXC'
        b'BQSMUVDN9WWOQXIGKTJPSEKFBLGLRQH9OEO9HFYDFDLZ9PIWCEYIINHHWXYZHESAWK'
        b'XICDRDHDZMPDETWQNMPLGARJTMYTPMZT9BNZFNC9HMDCCNCKZPXOHZFLLCVPVARNTI'
        b'QYAIGLC9SAHOKVELOXUVCVIFU9ZPVLNLSDTSKQJZKMYRXTD9STQEFDLYDQILBTISLV'
        b'99EAWHJSNDBPPLKRPMPLYKPXEILA9XTVTKDAUYDVTYYJHWVMGRVEKGNMYDDWNLJPUU'
        b'PBUNCZXTFIEWZDNGGSDLZZIOLM99TBHEFBJ9GTFDXNSMWFIYKXELWQYAQBLDNYSTTL'
        b'YGLETFJJRTKRFVKRFNDTJVVQDEJTVWQWTKDIQOZTUHIAWJNSNCWNYHOCEZBITEMI9V'
        b'NDCMPQ9KXQEUOTIHBJVTJHLSWRPOHQWKJDI9GCFOSABBKCDRHGV9UOHKEORUJVPFSU'
        b'RKKXXXIWNEEEABGYNELGAKVUUPYMXTSSFT9Q9KCJTEAASIEJRHHJJIKJEUVM9A9JYZ'
        b'YTZW9QZR9IASSDZGGBMOTZLOMCGEXPFUTDKXFLNJJRNMIURDSNRQEPHCVFCQONMCXX'
        b'QIYEHOWB9WOCBZVSVKARB9MP9DQ9WAAZDLLZNAPLPAZEWDECSNLBKECVVWFXGZFQB9'
        b'OUSIFYLSXQMVFCCLDJVXGHH9LGLNW9UZWVAYYAMGGOVGTNXWBGSQAIEAQNL9EJACRF'
        b'QWHUVKDHKRLVAFWFX9HMHLKGWIYKOTVMRNIQTARCNMRCIXPYJJGOHVC9LBWCTFSCRH'
        b'AEKUC9BJHXIVFFPGEEGBWZNCSRSOMSUOLNSUXUQLOPWZDTRXIHVZZLBCCKYNXSRIGH'
        b'YEACLIEIANAYUT9JCLOGUPMRQRXPTOPKYBENALUUWFU9DXMGECFULSQKQFRRKLSAR9'
        b'IUXGNYJZUIMBARYLYTSM9KQKGJQ9LOUVSURPPAILTNOFMKWTPNCXOWVYWNDDBXEESZ'
        b'LEDTETJVMXOOHUZPCQRFNKWVDGOCTOTHBOMWWMOMAOEUIGVBRSMBBJPZASDARKOBFJ'
        b'UKQSLUBQPHRXJNO9JYBNMHHTWCHYEICB9GGJIQQFTFPLOGVJGDXEZBN9XYZS9BKPXQ'
        b'QHXEYE9HISINR9DHDFGKLSO9TCDRUAGCRFZIQVCPJWPQMZGUXXXBOYHXZLRWEIZJDR'
        b'WYLQSPBTJQEBUBHVXKOCQUKOSIUTTAGWBBHTQTQRXOMVIALUWAEQMQXNYPAHJABDIW'
        b'VGHPBGDFNUTOWQOFZQDYVDKDO9NZHFVLFMCXZGVEXYGIKJLGYOGXVXRIMBBCQOAF9H'
        b'LISJHCTKSXTCQXCPLGSCZKQTE9CLWAIVPKGVMECLAAHFEHAXAZI9AZRCFVKX9SGPCL'
        b'ALDSDAL9KXQLYDYEPTXQXSTNZLRLMXCSVSGJGNAHGKJLKMZDPXMHSOONBWIU9EGTIV'
        b'9Q9DHAXLMSOEG9BXCNERYWTSHCEQBOM9USPXLKXTGQUDKYSITRTMIF9VPJGXGOEXSJ'
        b'GBWSNGI9OQGDVCIVNBM9FALTM9SNTBFSDWZCROFVSEUEDMTQZZJATYAOFPTHWQLSHJ'
        b'WCQMIXRMICXGKOWT9VYKYTXXSIEVLMIHBREXFDIHPQOQHZYBURPHBDQSAWCNTDSADJ'
        b'DYGVVUTVDWLXFWN9AENFDAJNQHHXZHCTLQSRMFS9MSJQQWAANCISFBDPXLERSETCQX'
        b'TKXNVCIIOZWT9OLRVPGTKMLSMXYNJDPEVQFYAADMLXC9VXLTJTSVDBGFLVRLMYQAOR'
        b'MJG9EYCQQENFCRH9JYTEWMMZTUASOOKXGYGDWKG9MCUNBGNEOYVXEOMAHMNNPDDJI9'
        b'UKZOUVCMSEDSY9SOVEHHQ9ZCCWOXTXXGUYNLKMMBLRQCM9BBDZPUTPM9GBO9UTEEJA'
        b'XSMQXAKFTCGPHQJQX9GMTZKSK9SIQOMBHSZRTPMPLKLOKKJQCCWVVHCHNCAHSQLNFF'
        b'99A9WXKINJV9JHCAEFSQY9UIBVQPBUKWDAAEEZTPSFQNUXOYXHD9LYXWMBBABVNJYH'
        b'DZMOJDYUBL9ZHILQDZJLKCYBBUHFOHETRMJURXODDPVKTWUUFEPANILESYJXOKBYAN'
        b'OTUQSNOZUZQZFGGHKWHY9EAMUE9N9ZDMVGZKWQVXXTDOSOTBMJEMLQTMXUQPEV9HNO'
        b'OHNI9MMTOWDJMCFUUNQEAPIOHFLLNTWGKPJGWWDJRJNFTUFLLHKXWCWXNZITFRLJRF'
        b'PEUDOGNNTTSHTXCBXVFILDRRJVYTOPAGGSIPQDJGIMBZQ9XYCFYRMHOMPHRM9LTRMF'
        b'MWESTEIJFCPGOFKPRTIHPUGRMOOR9FBCFAXPYSDZ9DIBSRLBZPMPOSQZKZXEDAXBKV'
        b'I9IHSLIFTQEXIYMMIPZHYEKGNE9KJBXATVYCXLOX9FVXWWBVGPLCWYXJYMXSQPFYJT'
        b'MSQZPSVMHFJTHHWRJLNGHLWPXRIEOQLLLFTOAIKAPLPNVDSHLWTHNKRPTWEKHQIKZL'
        b'XCWQAKKLIPHYBJFPVNWWWGLDKTARWSIFRQBVJMYETVLUSRAPNLPCHACXXCGHLXCSVR'
        b'BH9RTPQZFW9FNQAS9VGHAQBCGKUQKFG9OTYYTJAWXANALWFPAQWYEWQMILAPUICZUL'
        b'ADQQUAQEKTVUIEECFCCGPYKIUCYJEIQFFHAKZWJZMTOFWQUZHINXNIHWYKGDLJ9AUE'
        b'GKVQAXFTESQENO9TCSTQFATFBDCYCWHKPXNPJFR9ZAWCKZRKHPOMRQQVUHCLALSXFW'
        b'QNKOVUPFRAOZNRQISOODHBBJPJOASUIFLLFLLRUDGPXY9XXHDIDSHZADBPBEWDGTXA'
        b'PBQOXLRIX9VSINBJPKKIHKNIWDVQVEURNTLNMEGB9YETRAZKI9R9XNKKPNKEJIUP9G'
        b'KPWEELKPQRJFOBHNZOQWNQXUTPPIEGWOYINXEFCVCXTU9BNSMQPWYHTZPSJOQZNAWR'
        b'RS9TEQGZJIOOUWRTTNTCFWINUWQYXUCWXOYBCVLQXOVBCVLPEYKCIOXMNDPCFPPSCS'
        b'UTQRDLXZDZIPQCLCOJQ9ASWVLVUVZWWIKQHQBYHEZYATKOZZKGYG9HIXNPAZINWBAP'
        b'JNU99OLYNDFCNYQIAKYZDRFQEO9T9K9EWOIIXNUCLRVHWKOGXSXXKCRDLBLWGBCQFQ'
        b'WIYHNCITMTCRFBHAGFLDFVQVLPRSBUXSKLY9DAZTKIXKVRVTSALFEOOFCTBTSBSCIL'
        b'DVTFFJQRVFJBCJJAZGGPZB9DDF9XLIDDXQESOZKEZWILNVAYHSGCNPJRMATZTOBKJO'
        b'FOUM9UUJXZUDDAAHBZSPW9LE9DEUGVJSCJG9EMFOOLPCNVGNFCR9IRGBBQVTADFBZA'
        b'9GKQBJWLNOEEEUSOZLTCLITLKHAIWOQWZUWKIRNODXTGVDXJXOIUYAUNKLXRJLXSCE'
        b'QKNPAJVVQCWFKWXSTWQMOTDVLSZVDJDBJUKQTQOIHTUTYMZABYEZJMPYZSKJAIXSYG'
        b'DX9NXGJIYADQGVZCXMMIUGUY9CAU9YSNHLVFAIQU9MUPPHIKVTSDLEO9ADVTQLGSRD'
        b'KXZGKH9YBNMGDIBEM9QSIOJEWGPNDLPJEXBMOUCED9VDNIWQXSPSAACPQNOGPWMVKQ'
        b'FQGYYYXWZHTLYXQBIXHHIXOIRDPXWMUQDWIIRYNLZORRNAJOLLXU9WHJPTKLLWRFKW'
        b'I9K9NDMQVNOWBHTZL9'
      ),
    )


# noinspection SpellCheckingInspection
class SignatureFragmentGeneratorTestCase(TestCase):
  def test_single_fragment(self):
    """
    Creating signature fragments from a PrivateKey exactly 1 fragment
    long.
    """
    generator = SignatureFragmentGenerator(
      private_key = PrivateKey(
        b'QFBNTRFTTBHNIPZQNQPXOFQWQCP9KMMMVEPLHYQMVMCWSVSWS9HEEJHNRGKELXEEKI'
        b'DBREGBJTIKHBMODRBXEOKYOXTPPYGVSROOMLBHQKFALHNUYEVUAMEYPNRLFFNCDGBF'
        b'YZQRLGLVQEPCYGP9HRUWRVIRZYFTFJNEJKCLUMFDXULSOWKVSEMVFXHAHLXBZITIHR'
        b'BEDHBAOG9YYKCQDMZZJRDEYTTGVCDPMYUBEAIQUUV9KWPJBNJFFVXTKFJOWXD9RRJF'
        b'HBPDMVVQMDTVCRTNFNHVCOLHOLXHIWKJW9ICHUBHCSMDKHVXGEQKFRI9DROBXBPWYV'
        b'PDUOEOKOKSYEHKQSDLQIPT9JLKRSZIAGJPBTAMQOADXCZDRFMLTUB9UDYQXCIIYDDX'
        b'OMIKLEJKZU9KGJHQLYAWFPPLRADBQDOAEZNENKTPJUQFEWPERZWJNYVSSVNQOBWZAT'
        b'WMZOVPPRKBAISZLIDVBZECDULBGKAINVFPAI9QGAGMSQFMSPSNFSKXKNCNRHUMSQPA'
        b'CPAPMNSDXAPHQXLFJZKLFGLWNZKUSACXTYKAPKLCGDMRIVGGZSAGLXSYDAKXVEHVPV'
        b'VVLGUNACNOVJGOFGZTNILHJTRGSHCLBPJPPRYZMRLKC9EWINJRREXTOIIPJ9MQKHNM'
        b'SKCNGLILLJZFZKXSLSLCBBCYSFIJPFPTCB9ALKC9VAZHG9URISFCWTVVUJSMDRZVKP'
        b'9AIZJTEUPFEXFHCFLDL9NEVZH9QDC9WXDBXXAWIHQWIZQVXIHL9C9LMGCAGPJZKESY'
        b'KADPDLYUHAFQVUJEDFQOSATEZTPIVRQTEMCHTJEODWAJHH9QQAFAUMISXHOLXRWOIZ'
        b'GLSDWIOUYZA9SQWFUKEUGGUQUYWXYVKPINSCEPQXQETIOHXLQPHSPNHRBYPOQMWHQA'
        b'PXAHZMAPPGCYZIHFQLOIYVOYKRXKRCPYHZMITNYQHVELAUUOELPWGVQNLPIDYNSCMN'
        b'QWPSSOGIRHJMEONAIU9ZYRK9HWCKGINSEYVEKLBWCTRUZJFAVBLIWASYZYJFONCRVE'
        b'GIAAJFFFOXTRLITRSJSSFGPNJPKNEBJZLQKIV9QPDSMZQIAVZJSKQMBAYIUCKTJMWP'
        b'UJULPOCPMMNJ9OYSPPSYRQSRDLVNDDXLWYDEXDGXJOOLXYKBJWBOWSTQVBVHR9OXCA'
        b'OOKIW9JCSKHGGSESTAUQIRCIMUPJAMMOEEQYHCHOSTKFVAHYNZ9NJSVWZHGWBRPNQK'
        b'ZTHSV9MPBSZHEWCXFXUZLJFOPTPGTXUWPPXXJQMCPOAI9CYTNYCLS9O9CFLXPVCFID'
        b'TSL9YT9SNSROTNSYCTKTICYZYAJTTPI9YILQSXZQXPTRTPOGALQMHZIRHPRTRXLT99'
        b'9ICOCEUZDXBENFISVLBMUYMVQHQMPLEDCOSAFPVLLORRVJJPLBCZDSKQJXHKXHVVOE'
        b'DG9OSKILVSWRHZL9WDKFR9HDEUTDTZQDPEWQQMXCRQSAXPNHHYDWGXHJCPEKJF9SAA'
        b'FLENDPJFHKSYBDZKZXXBKUKKAEBBYWKEYSSURNWTQRAYJ9TZ9FFXNA9LXNZAQVCRWQ'
        b'KNNAYGPIKUTCJZAMCMJNTRBIQNBMJVTPVMOTTNSGYABZECIOBBIDCKLRSJDESXGVNY'
        b'HACSAVTMFICFHLBEBGDMWKEBEICRUFYGMXWXJJ9MHMCRLTAOM9GPOYDNWLYLOQCVKR'
        b'HXZFRZKBRFCF9LOGAIGQLCZR9OIRL9HYSAMMJQEXJZMMKMCGGTYOKXVXFBDXDQOQKK'
        b'IZG9TMO9CAPGRQSUWE9WKHAIERQRJRZFKS9AHYNNGDNVLXPLLARQHWCFZKATLJNEGG'
        b'JAYVDGKSVQDYCIUEELIQCSVTCCFWVTYOMYIHSMN9JMTFSUBOQGCPBZYAFMCQMMQUTN'
        b'OOPVKCGRVOEJMIPZKSOLGNBXHTKLWHMTOZRCKHRAIBOAPPSMCLBGXWHBYSTYPGXETK'
        b'FHALEOMABXFV9MLUMZBVOFRI9UASHROBDN9EFCTUFZFYLBRUOVZWJENH9QHUJTX9EL'
        b'NBKYYTDZSEHJCPORZDIHYABRLZRQZPECYFFNRVCRSNENPFCRJWZPMAQKYUXVPDFFVX'
        b'EHIPLPZSFDJCALFZNPAKSLPD9QEBZIZNCAMKBZQFWRCFSVIEBDDSVMQIYDCWLQQGPC'
        b'HAXQWELNH'
      ),

      hash_ = Hash(
        b'ZHAGCBCERH9FBGEFUDXGNGCAKCCCFEBHUCYGRFNB'
        b'ZFFHEHSEEDDBGBDIMFEHYCGDBFPGVAIEH9A9VBNEJ'
      ),
    )

    self.assertEqual(
      next(generator),

      TryteString(
        b'XJASWJN9LHHCZFUBPFXNWZFCPCLYQRLCMEATXQUEUCVDIRFOVGXXCGTAMRXNGCQZEB'
        b'VQ9QUFGTBWRBYWCQFWUDNGNZLVRE9EQYTTTQAVZCSSNPTFIRWMNRHEXIUZTM9TTRFY'
        b'GYQQBSQQQEAVFMKSAWLQVDPRGJ9NFBKRYAACGXBKDCPCEVEO9SCDJXDDKDWIJFVEPD'
        b'EXLEAWLIRZHRJGIUXS99CEDNEVDB9EVEDFOP9PGTDUVZXRJVSLARUBRFHNRQQTXNSO'
        b'WUSSZABBZGPDNVNK9UUQZNSKFWJAJZVUGKKUIW9JSRIEQHUFSIIGOBX9IUFNBNBKYG'
        b'JXVAG9GZZXETMH9LCSHWEVCMPWIHTUOPQCFFJSXQWSNHVYT9GEPTSAECAITTXREMLR'
        b'HMFBMCESO9YAIRGJOELUVNPMUMNJMAYGLNL9AWIZWCSIQEEQT9I9CBWURNICTELPYL'
        b'RWFRGCIWBUGXFPVRVBOFSSDSQDBUHNPDOYRDHJKZEKRLBYCBQMCSIUNRPIBAYZSJNW'
        b'LHOVZXSRAHKHTXEXZGTHQLKVZZQURRWPVYLXK9DYTDNXIEJBZSAICUPYNSFKPZWQOA'
        b'PHQTXXPCNQGBYYZBEGHD9MJBLNVREJAFXCKWRXHSPSZPZVLMHEEGEPQ9JEYCKJRUSQ'
        b'MVCDGQQVUCUWTOYYPOKWHGKPEWLUQCTRCKAOEEORR9WKKVFCAVMQLRZHHJDMYABBMK'
        b'TTCTKZFLTLDBJ9AXDLCANUOBABM9YVOUSPCDFFKBUFPZDJWENLHTSNYJGNBFCOMI9Z'
        b'AWPRUCGDHXGSUERBUXNLOSQDCQLUIBM9UCRBPNEB9WZGZKWBNNJYODUWORKO9ZWURY'
        b'DXXHELMBRXKHXGIGLJV9MLEKCGCSFROTM9QJDKZPHGHXOLGZHUNXTQEUTGKIIQ9OSA'
        b'ENUOSRYCFKMLOPAJLJIEFZDCHILJFEXQCRGRJLTOHLKDLPUKYHDJJCYXJHHCYIHVTU'
        b'LQHEKEIHMQNCYDTKCHAAOOMPXGPJUTXNOAMUFVIABPDKWMSSFNEZKTQJIVZISPVZAE'
        b'K9BPSVWNCWYPCQDNLFUWFYRFTZMVRPJJBSMWWERIPSGCNJJCOBIGWNHV9NVUIJXFVZ'
        b'UKCD9WBVZDYGVGKXCDXUHGCDYMUNWLUPSUVXABHHDMACCNVGSJAHHSFJ9IOEPPURVZ'
        b'NLDFOSNRASVRIWLINGEOZPXRMZMFMLWALBSMILWNHGRGYPAHSMPPMPZTMHMEJWHESL'
        b'MU9NZTWRAATVGSJQMS99WYCXPURWOSSZISEJZKW9AJJAKNDAYBZOLUNAAZOUGSWQAH'
        b'OYKBDGBXBWRMTEJXMOAVSSRVYBIHWCJZSTZ9EFWQZANHVIKSMBTQRNJOAYFNCGLIUL'
        b'TWCDQBJVFKIBXSJOTCWMHRVQAOHFQFGFGPMDMIRHIGY9XYSHPGIIMT9SPDHDRDVSJV'
        b'MLILZQWDJCRGNCRSGJWLUKNTTGIH9SNJGYDZAFQVMHTTRWKVZFIHPVIXDSTOANVTHB'
        b'HDHOWE9RTE9SPJXVVPCJFEPZRQERUTY9PDADTKLXIFGKGRQFZTTTNPMNTPSJQYPPUP'
        b'PJTWVJC9EJFFGIXMZGHTNKPGCZALIPTCRR9DWXZEBMZRFKHNYBBPOVISABQYMLLVSA'
        b'PJKFDVCRHAMNMEQSWPWQYMXNPZRBDFUWOOLAIZLFQXGWNCQN9KGQHEIQPNFOKKBUJU'
        b'WMXCQAXYYSHZNDOWWMGHBMEYAAPMBYXTHZTQASRGXDJFKGBSXGVWIXFBFBWIKBDENB'
        b'THWHEHJHHALBSVAEGYFXVDRHFMPQPIWPGLXGOGFYEMZK9FN9MYSXYCMMUXMUSLKWZP'
        b'9BNQKOJMONDFNZWIYQEJNCBSLGWSHYKLQCUK9ERBLQDSZ9XUTAXXIUCWHTHBBRBQRX'
        b'QAWNRSHAKKJLBFNEDGWBA9AKPQYHTOZHVHRUNUXXMPPWQRXGPA9XFFTQFVPMXHWJMK'
        b'TPHUHLWHCODHWTLEIOENJLUSD9URLOLCQOKYSDAGDOJDXIFLSUKGVWYKHVQN9UIFWK'
        b'RHECRUMGJJYIPFMAAJXBBBBVGFXXRFOAFXEGHRFCSRENWNT9EXGKMA9WD9REIUZUJJ'
        b'KAVBSMAEJRBGPKCHQTN9CPZBDKZYYYRNQCQZTUKDSRMZDZLABGSLJU9VBRTGVIZC9F'
        b'BPEKYDOEB'
      ),
    )

    # A generator can only generate one signature fragment per fragment
    # in its private key.
    with self.assertRaises(StopIteration):
      next(generator)

  def test_multiple_fragments(self):
    """
    Creating signature fragments from a PrivateKey longer than 1
    fragment.
    """
    generator = SignatureFragmentGenerator(
      private_key = PrivateKey(
        b'MPRSBNPSAZRHZSBXSJASXSEHLYVLBXHCUNNQUNYB9CYEVWNBJKZUODPCZGOBRJ9V9C'
        b'CARPTZEQRXGYDBEZISGMMXGHHHATAUDRTWUHEKJAMKUUKCK9NUUEWEFHPTVAARNLJB'
        b'XRTFHQDXAVAOWYGJJNLCRVWQFHXPTMLFPGDKEBZCSSOQZEPNRTUKUJ9TRHLWQKOCBZ'
        b'WBXBQVKUDTKV9IKCDQZRXRGQGIKCQ9BUNYMDFALF9QUFVKHQEHFICORMLLELVXM9VM'
        b'ZIBUFKHLFGHDCPD9CBZVU9PMXWTAKWCBPBRBUG9PIAYYHKTP9VGDQIP9GUPKK9DVZI'
        b'UYYTBTBYCDDF9LPBRGNGW9GHHFGRQPYZKKLCUDUAPAKZVIWL9RSJFQO9L9WKMBSWTY'
        b'LCJJGWTWNLLKMHOUQVKTOUQGEOYNLEHTHJQIPV9ACBEBU9DN9KZWCEZIDEGQYQEYVY'
        b'WUMPHMJAXOVEXOUMJ9MANMTGBOJAHUGHEYDPUQLNBIRUCLTXRFQIBTEWGUAHIJL9RK'
        b'LMGKURCYGJLXGFLDF9HPSQUCGSZQDUQGZSYKRKHCCPBAPEBWJUYOBBGUXVSWHRAKWT'
        b'TMHGJAWM9RPCLMVVNOWNXPBNTPFYWQIPQLNOJJYNRDTLAUBZPLVUACUSPNBHCOPO9Z'
        b'XSIETGTFDLMRESLN9WXUNQ9SFQFD9CNIETCSVDMNLGNMKOOTOCODTDCPLZPUYON9HD'
        b'QLDGSCWHKDLQOGDHZHPURB9TSUGVSPKDIZE9ETVMLVMNDRJOOUUFDTSKDDTOGXEVFN'
        b'BDSBRDZPZTUGYZNVKOIFZYRUIZI9LRIFCCXPHJMHCDOUPKJVUGFPTFI9XPGRDHRZUQ'
        b'UXCK9MTBINIJFFCQTECFTVTILTKRYJTI9ADGPHFBKNLAPLPKKAOOIEKQLBCDGTZZEF'
        b'KDBRICXLWZQXXCYVDFAJGKJSS9PADMDCJGJSYSIAOUFHW9CHTDXLVE9PBFCMBZKIHZ'
        b'9VBCXUDBUUEFNRHI9FXC9KREKFVDGHBFDHNEGCIM9CVSJEQSU9FUKVYSVWCTLM9LAB'
        b'JCWMPTUHULMGQBVRFCYSGXEQJHMZIVGTIS9QVPTTULFKLSMIPMBUTBXTUQRBOJVUMP'
        b'BGWEHIFRMNWLHQY9XKFCSGISDWSDILBOSYYWUSICMRWYUWTJY9GW9CC9ZQOKWMIYYD'
        b'ZRZEKKKJVD9DWTTBEPAJQOVOYPVDGRILVPYXRJZSUXKHCWGVPQNDWKCXJO99AUPVZB'
        b'VOFELMUKYCRETU9QVUYXXEWHBFFWQOLIHFLRKLGECUMUMHWYESXOUPWLEFMJZVXWJH'
        b'EBSYVSRSIFXRWBCAEDNUZWEHAKUPAIGIQKCVNURYELBRSKENYWEVPEORWIQHHCSGYW'
        b'XADSSJGDGTVDUHUNJBPTOTKOS9KKSWSJPRGUB9AFTBBUAMEIBBGIQCCZDRHCCYADMI'
        b'ATAOUOJOWNUWUKKMYVHNLJSZMTBHYODMJW9RFZNUJCCGTKMVPTBLLUOKXWVJEKSBQW'
        b'TGNJBEU99LIUBZIHIMX9GHTEXPDMH9OXPFFGMYQFGEJV9TQJPI9ZFVCFZHMRLDVLUG'
        b'MALNTWOMCNQGHQKZRHSLKFCHDFTOBEOQZRAPDOFEKTQZVRUDCYVZCLONUKMUYIIJRD'
        b'BVHOZINOYECRMGGIXZNSAGZUOYCNSCEB9EPMZDVMIIBHHLEKSWIDGBOMJRMU9KXKPX'
        b'DTPULQZNSFKNIFWNPWBRSAQHESXMNCMPIMQLPOKEDFWPYCVYHMXLFFMIQGGBTAM9CP'
        b'COUJBMZWLYIXOGVFYGLRKSYMPMQBNUMSBV9JXWB9BLJNXOORUTLGI9WSAGFDUVL9DE'
        b'YGVGYVWUCXXLJQFDGXJWYWNHIUJXFCRWJLVUD9CTLXKZSLGGG9UOKYUX9ESVPNUYLV'
        b'LGSBVCA99HYBFYZYBEKVSRBFWDURRA9OVXQLKSUAEMKNT9FVYPPRQHCIVPXUS9KP9R'
        b'OEZ9NOZHYLIURCFY9XFYUEKWQHYKNZEJWKJMFFDJCEWDGVSQLNSQRMJALTGQJIFNED'
        b'PSYXCIRKYURDVHTNWGRZLDETFXQGWAJTGWAIXXCBNSMMVFCJELLMLEFFYOGNISGWKJ'
        b'KLAXTGLAYNWEMXUZBKTCFNRRHWMCXMIDPATZCYQFFGNAKTGSIZYPQXDIYUGFCO9YBR'
        b'LAOEGJUJNL9SHU9NPYQGOLUTDTNQHZPNQGWXXNWYUHLNTCRCOZTVZJQGNSFQLKSNEU'
        b'LEEYHDMWAWVVHCXVVYNRDLPWDLRGHNPEAOCYESSMABTSMDMPNJHMAKWVEZSQCATTPD'
        b'IMRA9JOZPVKAZWMEEKTQNQVCRQBVLPCJCKBPNSWFJBM9JBNCUETNUVOHHCZZLMBIOE'
        b'QEWUOGNIETNOKGFLLKKZUIRDLBAHMZQ9JUWXO9QHZ9XZNAFGQTCJYJUYNKIYTTNQXQ'
        b'DSUWKEZACQBYRWLTTINDLBRSZPESTWV9KKYCBEEVPKFQIK9RGIADVNDWWMPQOWLYAR'
        b'VK9BNZLFEZGUHZW9KFZAEDUTVTSGSTGFPCJNX9MGLDFXMTYWXEHOYBTQCVCEXNQNRI'
        b'YEUGEUJ9AUXMNUTIVYB9VRNVTAAEQTQRPEKYOLECFGPZWWOTIEZMSA9P9QCMCSDMUS'
        b'HDJOQEUUKFCSHQTWHEHJWQPEXQNKVPNUEMBA9CVOEVJEGJORYWZAOIVWHAMCRVPVTW'
        b'IMEJNHZ9PBSIPNEUWAQBNSOAWYA9GLPWCNTSAIHLSQD9LJJIVLLVXWSNQHWLYQMQFB'
        b'WTAUYUGAWOICQJFWTJXN9BLVQDZSTTTGJCCWGWEBGTGODNDCGCSXZMBTSFEZIKMCOZ'
        b'WJU9IVFAEIGHLCLUNERARLHWQICKGYYEUBBPEEHHHGPTDPEMHMNRYKIXOASAVPN9TT'
        b'YDEYZHVDGVVTICNCXRMJKJLWJDVPNPQCQESACMEZRDVCFTGMEEHEUFAWZDLNFT9PNI'
        b'Q9WTWZAMDRYSBSCXIOOYAMQBAHJEFLDUQHPZYDQRPA9MEKAMRBZGTUBRTNYYGBGAY9'
        b'YIERRWFRMNKGTZSXBONWLGSQVBL9HYSUSUHTVPTIHFFXWDCZSGTBDZQMLPUCYWWVBF'
        b'9JNJPCQODJYOHXRUTEPRJZW9MJHKQLGIFGQRQWRSBFPASG9YLKOHOKRZQQZPM9MNQC'
        b'AWAZIU9JRPDXBNMBIYFJUYAEWQSMLSZUPLBKHGIJTNXYQASQUMPMKWIZ9DMINF9CYG'
        b'ZMILHRJOAJQEOVVALRYQGPPTUOD9OIOHIGRGGJMVKCZUNK9QVBLBTMXJRYWNVY9MZN'
        b'HZLVJUNZEYICXLLMQSYFPRZQAYJCX9KKFFTPZFCUKRV9EMRHPGPQ9ZNBYXXZXL9LGZ'
        b'LJZPKVA9HHZEBIIPJYIEBQJO9LIOPTWSUMOLHG9AEJWGANZBFG9IDPFTDDBFYBDPDJ'
        b'GQEPDFYAYADLQOODRXA9CBUDYXEJWITAKRMBACRDXDCPGGPDCQRHACTJIIVD9XTLVR'
        b'NNTWCMDTZXXVUOHBOYWCGIFCNEH9AG9KD99LOGNAYIMCRLVTCCCRFWHKVE99AOWOGU'
        b'UXNITAEYHJHVVOKOVFXVOSXBTGVIVNDJAZITQKXXXLLIMOAKOMZIIXEEYXGNQVYFQQ'
        b'TJJEDXSDPFA9JZNMWGADIUNVFOURDTGPHCKPEHJMIKZFISIYFOVWWYZYKFRXRH9TCJ'
        b'GOLCHPYDVFIPRCYLUINFQRBXJ9EDFRJPGPERLTJNTNJMDYHVLYSY9ZALOOEGBNLJQV'
        b'PAOPVMQPZ9QSUYW9FQFQYKFOYYYPQHIFEWPVWURDY9DWBBVOADLLEFXNLGIFYQXKY9'
        b'CDDYQXV9EP9MMJW9SUBHL9BGCXYF9LMGUIJDEDJOTSRZZGPKOWHOGFI9QJRNIN9FVB'
        b'WVFOAJGTYJBOWEPFZDHGKYVGB9IHHCHZODFWRJQSSYDV9IURZBCJALECSLQCXYHAUC'
        b'GJZNB99QZFZ9FCPFFRXJCUYGWMKRGICUDAUYBKIPPXRQOVBYPKALVCRBQATSBOONFP'
        b'EMTIYNOSEUC9JLBFMMQTVTBBXVJV9FIOXQDSJCZVTERZHSMTK9JKUYUHRJWOXXTZDD'
        b'XMJSUTPFQJHKDBMOUWSZRHUJAQOTPXINF9WDGRG99OSMKXGY9WCD9AJVXEDNLWSBWB'
        b'IHO9ODBQSQL9TUMJZASGNTWUQGNNIWSBNGXRSCLYCGRNTEYTGJAIIXX9YNGHLDZ9QV'
        b'BCQNMVIQ9EXTSCQAJJROGKIHJJTKLOABYQDHMXJFEKFN9L9FSYAMVOFTXIOSEOBBUA'
        b'UYSHWINHCJRQTIXWWPZBGKIGJC9TDKFAKZTIFBSDFSXVJCCICIBEAIGNUARDZGGBBH'
        b'RLMWQ9RSJB9TPLSDEP'
      ),

      hash_ = Hash(
        b'DDSHZCQGA9XFGHPFY9NEG9AGO9HEJEZHPEZDK9FA'
        b'BBUEI9PBJIBELHTBTCBAWGIAYCQ9ACLDTHFCVFWGI'
      ),
    )

    self.assertEqual(
      next(generator),

      TryteString(
        b'IIMLJOGFKELWFPWQZCMKWMUBCXYLMPETDI9CELEEBVVPSCRGXFLTMKWWD9SPYJGYXF'
        b'OUCKCMGBGTIJUXGCFOENDWEGCXOXZK9BNFGAUTSLGPIWCIWFQAGQTVECGNXLFCUTFQ'
        b'UJRUQKPJEDUFZTABRPTDXCIE99ZSGZCZYBGEGYJ9FGJJAUPJBQUA9KVSAWIQDES9OR'
        b'TIEWZJPFAUOHCDRPTCPKLWXIQADMKWUMUQPLIAOUWHENILVOWHOJEDNTCYYSIL9SLJ'
        b'QVRSFYQTHJ9FDISYRCKRPONEVSPRCEMCGNPK9LMANIRTOYJOVGLGENCWABUFIJ9HFI'
        b'URAXKZKTNULKABTUDTNVQEJRMVUXGOOWAUMSQCUBAOLEFO9PTVPCIFMFJGYAPNFIDW'
        b'T9TASFHSPX9ZHCMDPXSINJTMHGQNVOBBYCCI9CNGWRNRDZTPSWNJFRCIFQP9DGPAAK'
        b'XGIRMUJKYOJMXJZGDIXYHAIMQQWTUMBJMOPE9YJYGFQBBGHSGKADRELYQPCBALYVIN'
        b'CXXKMLZZFYYJYXHKMTKLQMVIKF9WXXWMYZSOE9HJANLNVQVAHQKEUKOPNNEQHRSREA'
        b'FOWSEKTWOAA9ACCUUYGZZXXLDOCEVQFFOXQAVFTMBLSUVYOOYBIGC9AFTSBBMWZKNL'
        b'U9WUTHBFDUUJJGAUT9VKGGCDNMUQTTZSHGEUJBX9SMBIG9XFSSDZOLYHLH9IAHZLQK'
        b'E9IKUSLYCDZMDPVEVDUUEEFHXCGZHTEDIO9GHQWIXKGDTSBQWASTQGHWKUHJVUSVDA'
        b'GZUYIOJBMSFFWFEMMSCZSVEVHUHPYF9JWGSDKTNORRFEGYJTDJFRYHLUFFHMYIURYE'
        b'FJXPXP9BTHGPGI9LSWDYZBNNPP9PRFMNGHFOEDXAGXCNTNUDQCCGKHIWGICNJETSUZ'
        b'TRDMEAOSGQHFVQERRRCGGRICYFAZUEHE9DWFICGESVXYTGFPESVPFMZQFBJ9CCOPCS'
        b'BQBTFFNKWTPHDDCUDSSUWYEQTDOXYXOYM9CBEIONNL9LXNLPTSTUGRBCWZVLSOYOIX'
        b'YJYSFSFHBYQXSMIJUCNDUMHLBVUFOXALSNFRFRJAZHOZ9HJESVBHIQGCNDLFVWOMTQ'
        b'MLCORHIZOJAMY9RMHOKKXFNGVFNFAGVKPIPTNGCQZPCOPBLZ9VGKHKSHOPYRVDECWG'
        b'DGVJWDEUKOXLUR9XZKCPTXBKSNIQVLUCBVZQYCRSRDHFFZVXCRTXLDKASADFICFV9C'
        b'ZEUJBXRBJRJCYIQAPRBPOJXDUGOANQGAOYGFB9SMHZMTYFYZBBQDKGTLYBKYZCXCDS'
        b'XZOWPRGJLFFNFWCFKCBCLMKJD9SUZFMDIRGRDGYTMLUTQQHUIWGWPEBCKBYDMPKS9V'
        b'ASYWPE9WPMZLKSMOFQBPULYKIVBTHTUUNGGRPOOMSSSNUCP9ASWMGONSIANVRPEYLF'
        b'OPJNSBXW9ZMTDGRJRIWRNNSXQXSR9TPAOLL9PUSCFYPMFYKZRCLCQL9UUQPTOWEZHX'
        b'ABZETWGPJEV9WLZIOVBJTVSOPGX9ONJVKLFFNZDKKSWSFATOGDWMATGVC9HNYNMKEX'
        b'RASPVWGLDQSHKRJGVGB99LCTNGATKUCWCHQNCYRZAX9KBBQZEBXCMEOZKSUXCGVQBL'
        b'HPYXYZOSOOPATETTKYP9KRSEZY99GYHAQYTGLKYMGNQWTYJJ9CVNFVQWMXDZCLHBMT'
        b'LUGNNFYFZATBGOSJJ9TWFHGRXYFRSCZAFLIDXPHZMOQHCPEDUDOLWIZXMKNGVEOHRZ'
        b'NZ9AUUMUGNYASBVKZMEUWQBADUZDOFRKARFQTXYPJWCHWETNWET9JNYRHMZAVGPBMN'
        b'SDKVQ9NKPTSGX9NFRERSOAUIPYLPMSATQKYGCZ9PNRNOEUMZFTHTLMIGIEMRSEBKGF'
        b'EXEBIAQPPEBQCVNSGOKCJSPOOP9RUQZHYHTZJH9ZJSEHZKZBNBRKAIFTJBOZPUTACH'
        b'XZHRIVIN9DZLZJDUAGOPWZDYJAWBKOKAVRPBIQWSTBIPCBADWNCSGUYYYNGDGPMJGK'
        b'MUQ9QCPVPJIOQAZOXMIP9POREGZIDEXJOWXHYTOILXMQBXRGBBNAOIJLRANGPHFWKI'
        b'UFYPFUBYQLTBBWCNOQWETHFHAYNJYGIOCSNYWN9OVEPQARCFRZISNSQNNGXZ9WBEZU'
        b'WSCVNWKOR'
      ),
    )

    self.assertEqual(
      next(generator),

      TryteString(
        b'QMPEPOWGTOPFHJYWAGNCFUJOXQMQPKCFI9XHFINAX9QDRPTYPFXXQKMDDBWGFOLPJE'
        b'KGLLDWNJJNIPYHXFRCEXTVBWZXCTXHCLXRWVFUEGYPBFWAASUQVGOHDYCWZAQVALEZ'
        b'GQEGJJYDFSKSRGJAJ9NMMXQIEEUVYBATTYJIKZIG9QZFARFCMKKVZ99GRYAAGGJONT'
        b'KUMPSUDCKRTBQZP9EAGGYBTQFSDMHXZVPANPKJVWTEXHMJJDDBNTWJDXVIPTSIZI9L'
        b'HJMIAEWOQVWBPOCFGLCAIVUZVKJSRMGZ9FLSABTTQDHZ9YRFWGCLINRCBSXVPGRWZM'
        b'UWDQAVFKAFBYPZSIJQPELNICVUJBKJQZCZAHNMNXTNEZTLSZA9QGHXUTOLCMRBWKGK'
        b'ABEWKMPEKCBGWFGIUWEIZQOTUNBGIMKXBUUBRNFOYVNJZKCDLABAEQDARGAIXCVQGI'
        b'CBCGEKO9CRPMHXWYLTJRTCNLCAKQWALTTFJKDFEOPBVXMJOSUQULYX9ZVCKTHERGEP'
        b'JNHHIUCVIZCTYSLXHOXMCPPNUKEYVDECBXUKVVJCGDMRIVGGZSAGLXSYDAKXVEHVPV'
        b'VVLGUNACNOVJGOFGZTNILHJTRGSHCLBPJPPRYZMRLKC9EWINJRREXTRZHHGLNNS99F'
        b'PMRIBJWNQJOAWBLZLPQMUGPPMQEZUUCKTTDENOORZIWRNVGDBFXOCUSL9YGNZDQKYI'
        b'YNUIE9SUBKYPDWVDHXXJQKZSHPPCEYJVITNPCHICTSVIBZXJPP9IRQWHOJUBJOABAS'
        b'SHZRGIMVUPHZXRBQYNLNLVZACFHUXFTGQVJSVRMA9FEMQJDZKCTCSDAMNXHYV9CSHF'
        b'YMGTCZSMNGMWMCPNIDBFKRCGIEPECPINVVZCOHPXIEHEJAIHFKUBZEID9SCGWAMNFV'
        b'QZZEBRNMCRGDEJNQWTAZZNXCARRNCRHXWHTGNOOQQNGQQPEBMBSMKGWSUHEPSERDQH'
        b'KVETGCWXOAIEIIIMU9GLYYNLROOTTLHMVWXZKJFSNEQAXMWJNSSUBNUZAHCMMFPZV9'
        b'IDAEJORNDLLJEWZRTUCVHJYWJZZZZSVWZAMKJQFAV9UUPP9RFPKBQIEQGALVUWHWKN'
        b'EVXHMNOMKMMWWZLDJOWY9PZLZZE9FOUQYGGQMUDOSZGF9FVFDOVFDVLAHBSYWKNHVG'
        b'WKJECMDTSPYQ99QNZSODIFAOZB9FMLWALBSMILWNHGRGYPAHSMPPMPZTMHMEJWHESL'
        b'MU9NZTWRAATVGSJQMS99WYCXPURWOSSZISEJZKW9AJGQRYHBXRUJLGDGKJUOJAR9KH'
        b'KSGASZNKVOKEPOKPUCKITRTZFPYIZCBACIFYK9K9C9P9HTGSKGNMMDRCINSJVCOJET'
        b'VLARADRJAOICANTEJOZAFV9VMWMBNBNPKXDUPJELBV9DDONZHLQYEDADLOAMICIUNB'
        b'YJHQAAUBBUUYEXITTBIEDZFKLRNGVATFGLDRSRMSXXSBU9WOQSLTDQMIPZQYPAKRJB'
        b'XANOTMDYR9DTDYZPUZH9NNY9SKHHXQKEOHQEOLACAAC9SRMOMNKTFVL9UVGIO9KQ9L'
        b'Z9XEWQGDNHPRSVHHCULTAKJSLSXXIXCKSRYYNKELSWMACWRJXVZCNYCLIM9MUFXEEW'
        b'MLTHNXHZWMIPVDRCIPRXDDKDNVSOELISCCSHCGDFEEBTRTOMMDRKACRKHLADGXNRFC'
        b'AOYVACYGTYBFMEKLRNFSXAOYSTXISSNUM9XPCHQDR9KRNROSNGRUEVZRROZOLSRAWI'
        b'VMOTPEDECEPJUDXAWETHKOMJIOXEZZYVHBDNYVDI9CLRXZ9HRUYGIXBEFVZJLQWC9R'
        b'ZWDIEPVDJXQWCPZSTAGWSWOSCMKLMNUBZNPZH9BMRXRWMZ9ZLVTOIDQMPGGCQGICFH'
        b'RYHTJVX9XXTMZLLV9LNDQLFIXSRXFAJVDIHCINQHVCQEYCIEWB9IHHZTWRN9YLEIHG'
        b'XKIICRTABEMSLCXNOCZKLBQPHFWLJWP9TFIAGZIWEGDJLEPYSGHNTOFJWRWIHGKMZR'
        b'FQENAASELQKWNLJYPGPKBKUHDS9ZBWVMQWBNMBULNGYMUSEBAAHYEBFCSIUAIVWYGU'
        b'D9TRNFSB9YBYCF9CLDRGODKVTJVIETTQHLVOJEP9YKJIM9LILRKCFXMQNSFKCF9YOR'
        b'YSYSZFD9I'
      ),
    )

    # A generator can only generate one signature fragment per fragment
    # in its private key.
    with self.assertRaises(StopIteration):
      next(generator)
