# coding=utf-8
from __future__ import absolute_import, division, print_function, \
  unicode_literals

from typing import Iterable, List, Optional
from unittest import TestCase

import filters as f
from filters.test import BaseFilterTestCase
from mock import Mock, patch

from iota import Address, BadApiResponse, Iota, ProposedTransaction, Tag, \
  TryteString
from iota.adapter import MockAdapter
from iota.commands.extended.prepare_transfer import PrepareTransferCommand
from iota.crypto.addresses import AddressGenerator
from iota.crypto.types import Seed
from iota.filters import GeneratedAddress, Trytes
from six import PY2, binary_type, text_type


class PrepareTransferRequestFilterTestCase(BaseFilterTestCase):
  filter_type = PrepareTransferCommand(MockAdapter()).get_request_filter
  skip_value_check = True

  # noinspection SpellCheckingInspection
  def setUp(self):
    super(PrepareTransferRequestFilterTestCase, self).setUp()

    # Define some tryte sequences that we can reuse between tests.
    self.trytes1 = (
      b'TESTVALUEONE9DONTUSEINPRODUCTION99999JBW'
      b'GEC99GBXFFBCHAEJHLC9DX9EEPAI9ICVCKBX9FFII'
    )

    self.trytes2 = (
      b'TESTVALUETWO9DONTUSEINPRODUCTION99999THZ'
      b'BODYHZM99IR9KOXLZXVUOJM9LQKCQJBWMTY999999'
    )

    self.trytes3 = (
      b'TESTVALUETHREE9DONTUSEINPRODUCTIONG99999'
      b'GTQ9CSNUFPYW9MBQ9LFQJSORCF9LGTY9BWQFY9999'
    )

    self.trytes4 = (
      b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999ZQ'
      b'HOGCBZCOTZVZRFBEHQKHENBIZWDTUQXTOVWEXRIK9'
    )

    self.transfer1 =\
      ProposedTransaction(
        address =
          Address(
            b'TESTVALUEFIVE9DONTUSEINPRODUCTION99999MG'
            b'AAAHJDZ9BBG9U9R9XEOHCBVCLCWCCCCBQCQGG9WHK'
          ),

        value = 42,
      )

    self.transfer2 =\
      ProposedTransaction(
        address =
          Address(
            b'TESTVALUESIX9DONTUSEINPRODUCTION99999GGT'
            b'FODSHHELBDERDCDRBCINDCGQEI9NAWDJBC9TGPFME'
          ),

        value = 86,
      )

  def test_pass_happy_path(self):
    """
    Request is valid.
    """
    request = {
      'change_address': Address(self.trytes1),
      'seed':           Seed(self.trytes2),
      'transfers':      [self.transfer1, self.transfer2],

      'inputs': [
        Address(self.trytes3, key_index=3),
        Address(self.trytes4, key_index=4),
      ],

    }

    filter_ = self._filter(request)

    self.assertFilterPasses(filter_)
    self.assertDictEqual(filter_.cleaned_data, request)

  def test_pass_compatible_types(self):
    """
    Request contains values that can be converted to the expected
    types.
    """
    filter_ = self._filter({
      # Any TrytesCompatible value works here.
      'change_address': binary_type(self.trytes1),
      'seed':           bytearray(self.trytes2),

      # These have to be :py:class:`Address` instances, so that we can
      # set ``key_index``.
      'inputs': [
        Address(self.trytes3, key_index=3),
        Address(self.trytes4, key_index=4),
      ],

      # These still have to have the correct type, however.
      'transfers': [self.transfer1, self.transfer2],
    })

    self.assertFilterPasses(filter_)
    self.assertDictEqual(
      filter_.cleaned_data,

      {
        'change_address': Address(self.trytes1),
        'seed':           Seed(self.trytes2),
        'transfers':      [self.transfer1, self.transfer2],

        'inputs': [
          Address(self.trytes3),
          Address(self.trytes4),
        ],
      },
    )

  def test_pass_optional_parameters_omitted(self):
    """
    Request omits optional parameters.
    """
    filter_ = self._filter({
      'seed':       Seed(self.trytes1),
      'transfers':  [self.transfer1],
    })

    self.assertFilterPasses(filter_)
    self.assertDictEqual(
      filter_.cleaned_data,

      {
        'seed':       Seed(self.trytes1),
        'transfers':  [self.transfer1],

        # These parameters are set to their default values.
        'change_address': None,
        'inputs':         None,
      },
    )

  def test_fail_empty(self):
    """
    Request is empty.
    """
    self.assertFilterErrors(
      {},

      {
        'seed':       [f.FilterMapper.CODE_MISSING_KEY],
        'transfers':  [f.FilterMapper.CODE_MISSING_KEY],
      },
    )

  def test_fail_unexpected_parameters(self):
    """
    Request contains unexpected parameters.
    """
    self.assertFilterErrors(
      {
        'seed': Seed(self.trytes1),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],

        # You guys give up? Or are you thirsty for more?
        'foo': 'bar',
      },

      {
        'foo': [f.FilterMapper.CODE_EXTRA_KEY],
      },
    )

  def test_fail_seed_null(self):
    """
    ``seed`` is null.
    """
    self.assertFilterErrors(
      {
        'seed': None,

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'seed': [f.Required.CODE_EMPTY],
      },
    )

  def test_fail_seed_wrong_type(self):
    """
    ``seed`` is not a TrytesCompatible value.
    """
    self.assertFilterErrors(
      {
        'seed': text_type(self.trytes1, 'ascii'),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'seed': [f.Type.CODE_WRONG_TYPE],
      },
    )

  def test_fail_seed_not_trytes(self):
    """
    ``seed`` contains invalid characters.
    """
    self.assertFilterErrors(
      {
        'seed': b'not valid; must contain only uppercase and "9"',

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'seed': [Trytes.CODE_NOT_TRYTES],
      },
    )

  def test_fail_transfers_wrong_type(self):
    """
    ``transfers`` is not an array.
    """
    self.assertFilterErrors(
      {
        # It's gotta be an array, even if there's only one transaction.
        'transfers':
          ProposedTransaction(address=Address(self.trytes2), value=42),

        'seed': Seed(self.trytes1),
      },

      {
        'transfers': [f.Type.CODE_WRONG_TYPE],
      },
    )

  def test_fail_transfers_empty(self):
    """
    ``transfers`` is an array, but it is empty.
    """
    self.assertFilterErrors(
      {
        'transfers': [],

        'seed': Seed(self.trytes1),
      },

      {
        'transfers': [f.Required.CODE_EMPTY],
      },
    )

  def test_fail_transfers_contents_invalid(self):
    """
    ``transfers`` is a non-empty array, but it contains invalid values.
    """
    self.assertFilterErrors(
      {
        'transfers': [
          None,

          # This value is valid; just adding it to make sure the filter
          # doesn't cheat!
          ProposedTransaction(address=Address(self.trytes2), value=42),

          {'address': Address(self.trytes2), 'value': 42},
        ],

        'seed': Seed(self.trytes1),
      },

      {
        'transfers.0': [f.Required.CODE_EMPTY],
        'transfers.2': [f.Type.CODE_WRONG_TYPE],
      },
    )

  def test_fail_change_address_wrong_type(self):
    """
    ``change_address`` is not a TrytesCompatible value.
    """
    self.assertFilterErrors(
      {
        'change_address': text_type(self.trytes3, 'ascii'),

        'seed': Seed(self.trytes1),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'change_address': [f.Type.CODE_WRONG_TYPE],
      },
    )

  def test_fail_change_address_not_trytes(self):
    """
    ``change_address`` contains invalid characters.
    """
    self.assertFilterErrors(
      {
        'change_address': b'not valid; must contain only uppercase and "9"',

        'seed': Seed(self.trytes1),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'change_address': [Trytes.CODE_NOT_TRYTES],
      },
    )

  def test_fail_inputs_wrong_type(self):
    """
    ``inputs`` is not an array.
    """
    self.assertFilterErrors(
      {
        # Must be an array, even if there's only one input.
        'inputs': Address(self.trytes3),

        'seed': Seed(self.trytes1),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'inputs': [f.Type.CODE_WRONG_TYPE],
      },
    )

  def test_fail_inputs_contents_invalid(self):
    """
    ``inputs`` is a non-empty array, but it contains invalid values.
    """
    self.assertFilterErrors(
      {
        'inputs': [
          None,
          binary_type(self.trytes1),

          # This is actually valid; I just added it to make sure the
          #   filter isn't cheating!
          Address(self.trytes1, key_index=1),

          # Inputs must have ``key_index`` set, so that we can generate
          # the correct private key to sign them.
          Address(b'', key_index=None),
        ],

        'seed': Seed(self.trytes1),

        'transfers': [
          ProposedTransaction(address=Address(self.trytes2), value=42),
        ],
      },

      {
        'inputs.0': [f.Required.CODE_EMPTY],
        'inputs.1': [f.Type.CODE_WRONG_TYPE],
        'inputs.3': [GeneratedAddress.CODE_NO_KEY_INDEX],
      },
    )


# noinspection SpellCheckingInspection
class PrepareTransferCommandTestCase(TestCase):
  def setUp(self):
    super(PrepareTransferCommandTestCase, self).setUp()

    self.adapter = MockAdapter()
    self.command = PrepareTransferCommand(self.adapter)

  def run(self, result=None):
    # Ensure that all tranactions use a predictable timestamp.
    self.timestamp = 1482938294

    def get_current_timestamp():
      return self.timestamp

    with patch(
        target  = 'iota.transaction.get_current_timestamp',
        new     = get_current_timestamp,
    ):
      return super(PrepareTransferCommandTestCase, self).run(result)

  def test_wireup(self):
    """
    Verify that the command is wired up correctly.
    """
    self.assertIsInstance(
      Iota(self.adapter).prepareTransfer,
      PrepareTransferCommand,
    )

  def test_pass_inputs_not_needed(self):
    """
    Preparing a bundle that does not transfer any IOTAs.
    """
    response = self.command(
      seed = Seed.random(),

      transfers = [
        ProposedTransaction(
          tag   = Tag(b'PYOTA9UNIT9TESTS9'),
          value = 0,

          address = Address(
            b'TESTVALUE9DONTUSEINPRODUCTION99999KJUPKN'
            b'RMTHKVJYWNBKBGCKOQWBTKBOBJIZZYQITTFJZKLOI'
          ),
        ),

        ProposedTransaction(
          value = 0,
          address = Address(
            b'TESTVALUE9DONTUSEINPRODUCTION99999YMSWGX'
            b'VNDMLXPT9HMVAOWUUZMLSJZFWGKDVGXPSQAWAEBJN'
          ),
        ),
      ],
    )

    self.assertDictEqual(
      response,

      {
        # Transactions that don't require signatures aren't too
        # interesting.  Things will get more exciting in subsequent
        # tests.
        'trytes': [
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUE9DONTUSEINPRODUCTION99999YMSWGXVNDMLXPT9HMVA'
            b'OWUUZMLSJZFWGKDVGXPSQAWAEBJN999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999A999999999EBBXLEONGGJMRUPZAO'
            b'HRAIOIEXDSZGQCXRWQMZNDUEQYYKDOSPHOI9KXZTCSBEUBW9WBHILISLYOZWIG99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUE9DONTUSEINPRODUCTION99999KJUPKNRMTHKVJYWNBKB'
            b'GCKOQWBTKBOBJIZZYQITTFJZKLOI999999999999999999999999999PYOTA9UNI'
            b'T9TESTS99999999999NYBKIVD99999999999A999999999EBBXLEONGGJMRUPZAO'
            b'HRAIOIEXDSZGQCXRWQMZNDUEQYYKDOSPHOI9KXZTCSBEUBW9WBHILISLYOZWIG99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_pass_inputs_explicit_no_change(self):
    """
    Preparing a bundle with specified inputs, no change address needed.
    """
    self.adapter.seed_response('getBalances', {
      'balances': [13, 29],
      'duration': '1',

      'milestone':
        'TESTVALUE9DONTUSEINPRODUCTION99999ZNIUXU'
        'FIVFBBYQHFYZYIEEWZL9VPMMKIIYTEZRRHXJXKIKF',
    })

    mock_signature_fragment_generator = MockSignatureFragmentGenerator([
      TryteString(
        b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWGHM'
        b'NUXBJTOBKZFDNJEMAOPPLR9OOQJCDVO9XSCYQJQVTXQDYWQEBIXKDZAFWINAHJELJT'
        b'DPVMUEWSVCJA9ONDYBNANWCGLBQMEMTBFDMWLCMQHGJLGYDQGIMLSNQHBGSVTDZSGN'
        b'QAL9OHRAPDKYSVBTNYRUUBNEEAINJMOVOHOWXUAIEDAIQDESQFCKJELHAVODSMXMKE'
        b'HTDKCDIWWISXSAHQE9TJTLJZGXIABHU9CUACMLVSSYV9UJREPWFVYWWXPYYJRP9DOE'
        b'KNDMBSBKKHIFMPXZXIJERXRZVBVDBYNZBBCCOSEDOLDGSNQK99HIYSWNYYEBLRT9MA'
        b'DLXLLZJOSZCFWAVZY9XUPNZUVOSKBMKXXJNRKDBOSGUGME9QNBMHIWXWXPEEUVQAQV'
        b'UXDJGMJOBXG9VJBWPRQRCCQSNBEHTLGOKJVYEPQOJO9QIZLYAVLCKVXKEKRGBSZJAC'
        b'9KTSSNMDQGKCLPZDJAQ9PBQMLUONVVFAWTMREGFXJMRRGL9MKNPOZGOYRPDCYEJCYJ'
        b'UN9HYNSNHXARMRJVXBUHOP9K9BIIEYGSHBUESKTAOQOEANEAIHYHVGSVNPXWRBTJAM'
        b'KMWEQOSYEWXLSRYVOSTMPOGYNPDNFLOICXVHYBDHSXVRKVWNVZOZQDOITZWICSYEW9'
        b'RGCPPUJYVIYVTSZILYENYUYUGDSGWVYWRMZJNCTTPVWDWXAPVZQQKI9CGEQPBFPCLG'
        b'DDEGBUUTISNCMJXQCTUNKQTLCATNOIRPMEUQBQTHHQYRGDLZEUZBALNQDXJYZBVXDP'
        b'LVOVVAUCQSCGRTUJRBBNRV9ORETTGFIXBBBVOPFHPKGPKVBYFTZMWUVZYVWWSDKQVO'
        b'NMPLLQTV9IZUWLUWZNLCVJNPMG9CMXQG9D9WYCANBRMYV9DU9FMJT9JHT9RWCGLHFC'
        b'ODXJVFQBLTKJWVNVGSUHNWLHNPLZDSWDMDVQTLVCSVFJJTIQZFAPCXJWDAXWJKJVOK'
        b'HALCQQTIXABPFXPUFK9IKXYUGMPXNSQCJDVETOVEX9LXYLXWRW9PFEYJCUJHLUB9NX'
        b'TUGLIQMDGPDPSJTWDYEWXQAICLN9BTGNBJWLVAXZGNCYXGHBMRUVVYTJGH9XDGSZHQ'
        b'DYKFGMOWORSFDFBLJHBRTXRSEBALCJIJTQJYDZZKWZGVAPFVKVEOXGYRLMBSPFHUIJ'
        b'ZZFMFVOTLPUWSYZCWFZMAALHRGSYSXSMOHWARYZZVIAKXAHGY9SROWPVFACXXLQEXX'
        b'OJCKXRRZHBZXJIBWQMMZTRDFYQBSBBZQQXGCAAECMQINHJRBSGOYPCGWKPWCHBKOJT'
        b'IGDASZFGONTUGDSOOLEMGOEBFCZZJZSCGXPHXHB9WGMMFVUTCHDBSAMYTECQZWGCXA'
        b'WTCTIBZHQVUAIBPZHBBTZAERYU9XAMKBHCHGZISSPOWJIRZTAXDHMAYBPXOXWDIUDH'
        b'NBTFJNVHHJO9AWAEC9UPRRFJLNGKTXJXFDGODDOPMGLALRIJBVIFLQTYQPKCKCRBYP'
        b'BYGUUFJGJFVCOURNKCGNTQNNKHDDPIVZHCJSLDUYHVPAX9YJOFTTFSKFHTOOQQRCPY'
        b'ZKTDVCUZGBOBZKLVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VZZGUNUCKOYLYXOV'
        b'HMGULWGSRCGXZLJVNIMZBLFOJJKOTUREMBXYOZXDUP9ROUVYOSJBGGFZMIFTKHJHHJ'
        b'GZJNOYQWFZAHLJWWDDFQQAMEGJUEUSIWOHKFJWRXRSJWYPGIGZGMFNAIDGDOUUQUVH'
        b'JZQPJMLCGKGADXAXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVG'
        b'XBBPBEBDVGZDBWMDMLPXYJBBRNOMKGR9TSVUXSRYXQTCTYLFQORMIGDKBJLNLCQXAC'
        b'VCBJGVWRJNYPCKOAILPLMWBYKDLDXLIZMZFWDXUWDEGDUURQGMJNUGJXDXYJGKOTQB'
        b'GCHATROPKEN9YTXDUOCMXPGHPDANTJFRRVEVBFVCNTWNMMOVAVKBNSJIWWBVHBMCSU'
        b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHVIT'
        b'GYROZZNQP'
      ),

      TryteString(
        b'SWHZKSNCOQXPCGRTYJPUGKLBNEJFXASKY9XAUROGDAO9QQLIVRZQDJDTPLNTBGUUFG'
        b'ELJPSGUMGPIUNCCTQEFU9UZIJJYJXCYWRADRHHXKBEDG9HTCHJHXUJRKMIUFOSKDGM'
        b'I9QPCYQSWDCUYKQQEONJEKYWELG9MSNBHRHILGSSKMRCQJBOGNYBKEMTOFEUBEOUBD'
        b'9ULP9PHWYKXEQNDMUR9BGDGPEUFRFRGNGFJFPYQXABSDALTYKL9SM9VVQCOHY9AS99'
        b'EYWSHUNEQVVNLS9CNPEVMPOKMWQYFPGTNJBZCDWYPFWSBKZOYXNNVMPODEHMHNIYZC'
        b'HIEDEAB9TLFOWVHF99GVRWUZWSN9IQOKWIXERKRQETZS9ZJJSQRLLPQXEWNMFVVBWO'
        b'IK9MBYCEGUJ9HJRIIMBVJNGXMGPGDLOLYWFVQNOKTFZRBJSSBJTETGAIUGZOYQOFTV'
        b'BKAQY9SSJWJXXYAUVUQWPXVCISFSDSHDQCPVMG9GVDAO9GIMDHZWJOKSUEUFHBGSCZ'
        b'KNBTZWJXSFYNJSBSEL9UMZBAZRGYCEHOSJBMKMPMNXKEVTMUDEFWBIKOXUSBNPTNEC'
        b'GVLYSOGUDJDPHYFADXRAOLQXJSJDANINJEOMCFAWWITREGCDF9OZ9ZKHPJZJNMOVGX'
        b'9OKQBSGVZYWKNOPVJEOZEI9BPE9GCUEQVAHSBBRBGQTEXVZCSL9ECOWPOWZCVSCBOU'
        b'SNQMTJIEKHXL9NCPRMLRNKQEHYJCLRHGZKFNBJIPKSKPRFTSKFJULTBTXFDQHWUYOS'
        b'DQBHPAINVEPKCCHJDTZOJIGJZOF9AEQDBKCZSZMIWUUVHVGAFKALGITVQQKBAHKCIF'
        b'SVMVZ9UDQABVIANTBUQOFBIXQBWB9KKQOVJZNVBEDAZKUTRNKGJQWMTEKV9KGCIBRD'
        b'CBAPKSTMCZGUV9HTAABQDKGQBCRFNXBMZRTHF9MO9GAGQDYDVLOFMDE9QQZYR9GDSB'
        b'LUVVMKMCZIMDPNCVLGDKBACWQJRWOQNKBTSDJFKQMKTVKXVNAHRHZALJGVAMXWJYRA'
        b'KTEJFXAHBQGSYWWQVECQYPXVFWILNFZKGGRIFCJBSIZRDJXRJHSURPCZKOWKLFRUMV'
        b'ENEGMNKUAOGVUECDSGAZNQZKBJDJPVBXLOTID9QLMFNGIWKAAIQTJJROSZBXPQRXAU'
        b'CV99OGCEOTQCJ9II9ASZL9XGNSVUXVKPXYOJMF9PX9GSLEROR9FXVQ9MLEMEW9IWNW'
        b'BNVAYXZ9ZETTDSMLGZAKHE9IUJBFUHXW9KWCNZOZCCTFGBGWSDAQGGSPSQHOMUVJML'
        b'WBDAKYQZMWPQLLYAGUMOVMVLFD9TO9OUBTVUHHUNSFSATSEGBFVGDZUBMTWWFDPSQV'
        b'CUFRVKHYYPDWRPNSKXRFTVEIBVZNGUZRQCPXVKBPKQDDLEBWIEBIPTEJIYFHBXCUVC'
        b'CKTKEJAYRZCKAXLMELINWUZHG9JFBSBAKHIXMWHUWUFHFNLXNO9GKINYKRTCNN99PH'
        b'PHO9MJAGUYZAPNSPWUZ99E9BEADKETLOALWNANYMHSLLQSBS9YTYVJKTVWFUVS9MFO'
        b'WCHLEUUFUWTYGLZXFDUXVABTVFXFPUEPIUEIAVSZSSZQJTHNGKBJXADRHVTIBERILM'
        b'CCGWUITYQHGEEGWIZZOII9B9EVVVFJNEYEWH9ZVOJGHKVPYKDEZZSPBAOBQGGWPWXT'
        b'CKSLSHJQYCDHAYIQ9QVSQFPXZDBYSJJKSNTRXORHLPVOYVMIGALRPXYWQWSJPPFTJC'
        b'YXAATLBFNSGVXKFJXHYTILNOQZROUCDUTWOMGYBVTWPKJY9RVKKWQQMVCHJEUBELJD'
        b'KJPGYLXUXATNOIJHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHV'
        b'HE9EEBQYSNWECSPAJHGLTEUCXALBRVGXFENUCOONSUFZLHTLVQNPDZDIVDQHWVLDED'
        b'PFQLJZWF9GFZMPZXFVEQECLUZBBFVSAPEXJLKKOMXEPHZAKP9WYTGQOML9FQSBMSFL'
        b'OGRLFQKUCUWFX9DNAOZSSKBUV9IBVIRNUWYBKJVKLJ9PPNLGJATKDCAGVFIVPXRABH'
        b'ZVZACJIG9WOKKLFCRDSMTWSCYHOZEEXRIMPQBXVXQAYKZIADSM9GUBICGKGQYNHKVY'
        b'OZFRVCHNM'
      ),

      TryteString(
        b'KJVG9EKTMPWE9PKWGGJJPDISCX9CJXGWUUPOLKKBVUWUYNBACOOF9LEQGNM9YYGNXJ'
        b'EMOBGSDCPLP9CQIFBCLENUCJGCCSWYU9WVFTRZZCPCZXEGMDDPSYDTYUIMVFSLGHTA'
        b'ZWJRHY9ZQMFGROIIUIQVDSOTUIRMSROYDPTWRURMZAILGBWIVADPYHPTSFGKAPPMVT'
        b'THGLYXZHPFUO9HBAJIOUJOOABAQSOQUKLSVQGSDIHEHGTQSPM9EDHLEQSFFAAQXR9M'
        b'UREVQ9MEGXNXMNJVWXYEIRYAOFHDFZAKNVWKTVIHKXWVT9VOZRUPUKXASIFAZQVZSW'
        b'HBQU9RGVLJMRVQCMCYSQEIMIAKOHNAKQLTIMEHMZMGAKCIPGHQTWORBLVKISGPKIIM'
        b'AMQWMZUNTKJSQZAZNYEGORGNRTKCLNRSOQJRBUCPSDLKLGGRBACIULLZBFBUNQXACK'
        b'L9WFEKKAHGLBBRNNEXZPPH9UZESFFKVBOPROFHQOKYAVTJDDVAUGUAURHLESIEIITD'
        b'VVRCTRKOGUPERJHNJMXTLNVMWVDZITSPEHRYJKEZVTZSJEYTOQEGNJRMCJLYYKPGDF'
        b'UFQHGWRDGEWBXYOGEZ9IXRWJAQLKHPROWIEVI9ILNOXTPOSRLETMNEQ9P9WLXCUZNM'
        b'GFK9EYHABBCSEZSGMNJZOEEGRVNU9ASSOOLCXXZKZPFWU9EEUUQRACVGZPL9MQINGL'
        b'YPUTUPTLPKWPHRFFBRHZQWIVOXPGAKCQQPRKPPZUHOJISYASMRYMCMJZNR9D9OQANU'
        b'XGJXSUSZQFWDJUTNCDKAUAFYKJNVAMBLTPPRPIJRRKQMCIHHGKPQPUQHWJNIEPDLRA'
        b'YSJXVSJVKAGBAJCMGQSCZFTEJSG9LUWZGFBGQUHFUHWDHND9WJBPOQQXDEATOBGXDG'
        b'M9BKSDCOEZ9IENZPPDUPMKCUKYBIBTBMJPJLDNSOPEKHVGQKLGUISUFSYMHR9I9LRP'
        b'LCXJTDHHEXKQEVIFOUGKJEILQIHFG9FWOUBXRHCRHLOYAXTFQUWKJBSX9GNPCWXUQJ'
        b'RHDBOBRZPQAPMKCIZGULPZDYLLBMAFJZXGIRVAAVUUCSNGDGJQJTAPV9QXYIABIHBX'
        b'ILKQLGDGXQUVADQGDFKKDKMU9WKBEEY9TAVRYQDQFKPDMLMUAEGBHVJPSIZOEQGCSY'
        b'NJCICXROXHPZFUXASQJXZEHQTEUKFIYQIGJWORKAIQUFROYGMIDFAJOUFAYYWMXUGJ'
        b'FPSRTGEUWWLOXEUTKZCZQHWFUNHTMZVIJ9VYOLBTAIFB9EN9NFVAABVFIBIWXLJSUO'
        b'YELOQSIPK99AXSXCPECWOXFUVDIANVO9PKZUESMFWIEVWLEHLCVKDXEROLNEMYRRCJ'
        b'DPAYVTYAYSL9AFZH9GXHXZORXZEQTUJEDJGCYCQAENYZRKDJSK9TOCKKCXOSSTOAIO'
        b'9UVAKQJBVOS9RUQIESCIJYRWYRUPMIJEHR9EGZ9YMHQXALUUDMCFYFOMLIGORMMBCD'
        b'JMFCNNELGPXHICRNRKONBKACHLLSABUNHQ9TU9OSSTQXGWBLRRTSKZORXILALQYRXD'
        b'DMXPPUTEGTVCHSOVYZEEJMRRECGBMXBORUTIQUNMJDXBSZSYYA9UOTFWMQOHURUFSU'
        b'ESLMILBBKGHTTFTZONNQIMJSLILKAQJRDTNVK9PHAMNKZXRHSOPGKKLJBRDYAC9BRU'
        b'RJWUIJLUWXNQOSVVLFEBROMJCGVYZWIPOYFQRBUUNJLIGPVDLADFLZJGZBLEBEQEUD'
        b'UZOIFFZLZRXCPQVMIARFLZRIOFFEHVFJZXFQFLCJSEXRPUKGMWBMGXEHIEZKOKGH9J'
        b'XAUXACEBLXKLZT9P9NJGXRZWZJAZCNKR9CHRRNCFOLBCSZXKXOIGZHZSTDKTHOWZTE'
        b'XWOIZLPEGPKTBRENSCOYBQJSZQ9XPNRORQOWMPGBXKSSLDIUVEAJODEUZJKZE9MBVT'
        b'QXWFXXXOG9QGDWMNZEWVWVDZWIFKSWZIDNBYEJP9VBKQNOJEXVPZQTHVUCSK9QCMEP'
        b'US9Y9FQPWEACAEBIQSVPJEL9ZBSETINIYMSPIXLADSHTDYRAYUTMXDCABIUUETMNLU'
        b'RELTPAGEDNMQZALFWOPAI9WUFOSUTOFUUWFAFRFVYOPITBVSG9IBVNJCOITYMTCCIJ'
        b'IZWVPYGQE'
      ),

      TryteString(
        b'GWLDXDNSEIQCQJKVVFEWPWR99OKSHTVIJCNFEGSUM9DUQRO9ZJUWOOGP9XLABZFDXN'
        b'GOXZLWETWXTTBT9KIGB9VOMMTKNJTUUFGJIYZIMHEAEJTNTIIOLLO9VWCYX9JA9RML'
        b'SB9COUYKMRZQWJXMIFXCETZWRDXHBBOYLYLURXBELK9YLIXXGHSP9TNNASKDGFVJQV'
        b'99CMXRM9VHASOBYBTWIMAJLBRUPZQLDCKOFAPHG9DKVVEFHTZAGNC9KH9K9HIFNLUI'
        b'NQFTQTSALBNV9HRWXDGDEBBKIMQCDWVTMPDIVCXHGKDFPAKTSYYJIROENCJOZXVBNL'
        b'UIUJHHAXZ9PTMNFGRRCNHQUVEESVSYNSIQXDRKKBMWJOQSMIK9FPHTNAJUYTQ9BLOG'
        b'9GZPXHACSPIFCDX9LIVQDISFAVZWQUXP9BROHMGBHFTVWEWCZRPTAMTXXLVLZBT9BM'
        b'OSJXAIGYUXICBUGQDOJRMYFWYGLT9UBTKGZZPNDIPNVIHQIBXFQACGYPWTKJSRHVQL'
        b'VJAJWFGNFLAJYOADR9XNOAYOLHKEUGWSOCXYJVHWLRRBE9XYLQDYJXYMURFPXTMNHE'
        b'EXJGVY9ADSJICXGWOUKYWVMXMWSJQVPKTUQUSCHTREWZNTXBDUJWDVTMXPABBHGYOC'
        b'UNFIFTUQTRAVTCFAQNNAAXBCRILNVZGGKEKIUOXBVMXLFNCSHFMH9PYR9DBXFKWIBT'
        b'AZLQTMVEGCZLWESPAHZLVTCGUUJXEAPCEYBXGGARHGDODWULDHHMMKEIYRFFEMQTTG'
        b'SGWTOGBZYEULWWFINFHGYEDHHXAJASMQCLBKWYXSBIWZLMEZVXUWP999OROQYLOFVA'
        b'ZGJIGHMTGJSZNGXFWMMUCGGQXB9ASA9UCVZLVYZG9XBIF9HUAB9HBYERWFJ9IEDMAY'
        b'ZSIFDHOX9HRQSDEGWUAODHRNVBQWTBK9JFZBNKBATUXBZOIEHPTFPQXSBGHGOEVMUT'
        b'RPSTRFOKHWEUPUSEROZEBSMTSTZSHFTQ9UXYTMDVLAPXSHZGYLPVDGTCGHOQSWJJED'
        b'ARRUPCYFHJOSPVSTNEERBJOERGU9TTOW9GSVZEODZOEQZATYADJ9NURBJNTPBYECGG'
        b'WP9SVOZAWXT9RLKBKL9TBBWXSDOSXRJHPKMLIPWKXSM9MPNQWEYLDPRLTUUNEFHXUF'
        b'CLLJLZIRGUMEJCTIHC9VPS9YPENZPBYKTSBPXIPZHNYZYDPOYRIFEZWOFDYMZTUOMZ'
        b'ZHLSLZMTDIMTTXMHHTDLIVRSIDSWJBCDCKEYZPTLLZP9IMNJSRXICEGTPZXVXAVIBG'
        b'JMMOUNPUKXHIANUPGJANUHTG9ZPZCBFRMLHYOPFAKGRZSZJARBEEPQZ9TKJRQLXEG9'
        b'IOHETGXCMKT9XZUBPMIQWXRRRFF9POXJBXW9NPUIOYNET9CTUWJB9RQDHVIAFLKILV'
        b'BDLOYZAKIRHAUXE9ORNAPVXRTUY9CNXAPFPNUADXHDQWGRCVBZMUASLOPAYHLNGNUV'
        b'VTDQCSOSTOOILZFXBXUPILJVVDUIRBWQUYNOJX99BTZNYQZGTENKGEKKADMPDWQB9I'
        b'CWBWFHKAPRNDGGWOUXDTJKMASYOPYNYPTOEN9EDLXVVUMELPGG9ZLAJXQFTIEA9HRJ'
        b'QCJLRUSLBGIWRWRXMTSAYVNHNJCYDSYNBPH9XEI9NFEDANKTZ9RWSCMPV9XVBTBZVD'
        b'O9HABGD9VDOIXFMWBCHERKTDPDQFQSVNZLZRPHVZTFTL9LRAIMXLMTEZFAKK9CMYVP'
        b'RTGBXGIMHUUVWCHDUUEZMZFMDSUQRVVPHZDUTOTLPSKQEHWNLOXKGGJKHHUNQIJXUT'
        b'NYMZIL9UOEKECBSTCRVTVKUWETWPECLAXJWUNXXNRDBR99KJSWCHJBTMK9TSLLKWUC'
        b'MMWNABUZLKLCJXHPUWVLIEIHYTZRPTZJTUMDDVEFCDRQYHPBF9WVMATUIQXGWTGAHQ'
        b'STNRVZZIPBRPIUOZLXRGEWSUVDXIQPAONF9QPFYIMUEMDXOMFPKKJNGRBNMKXNJUF9'
        b'IQIHPEBHSLWQWXJZNEBKCQUSRWOEGMWFZYGHFUUHDBBOBKSTXT9HGOORUQMFBFBICA'
        b'HBQNOBVDCZVGZGASCINUGVEMM9LLPWTNWWVKWYIYDIJEKAVBEFPAVMFWEOYMTOHLZV'
        b'PRMIINUJT'
      ),
    ])

    with patch(
        'iota.transaction.ProposedBundle._create_signature_fragment_generator',
        Mock(return_value=mock_signature_fragment_generator),
    ):
      response = self.command(
        seed = Seed(
          b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
          b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
        ),

        transfers = [
          ProposedTransaction(
            value = 42,
            address = Address(
              b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
              b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
            ),
          ),
        ],

        inputs = [
          Address(
            trytes =
              b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
              b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

            #
            # Normally, you would use an AddressGenerator to create
            # new addresses, so ``key_index`` would be populated
            # automatically.
            #
            # But, AddressGenerator runs a bit slowly, so to speed up
            # test execution, we will use hard-coded values.
            #
            key_index = 4,
          ),

          Address(
            trytes =
              b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJ'
              b'RBOSBIMNTGDYKUDYYFJFGZOHORYSQPCWJRKHIOVIY',

            key_index = 5,
          ),
        ],
      )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # Ipnut #2, Part 2 of 2
          TryteString(
            b'GWLDXDNSEIQCQJKVVFEWPWR99OKSHTVIJCNFEGSUM9DUQRO9ZJUWOOGP9XLABZFD'
            b'XNGOXZLWETWXTTBT9KIGB9VOMMTKNJTUUFGJIYZIMHEAEJTNTIIOLLO9VWCYX9JA'
            b'9RMLSB9COUYKMRZQWJXMIFXCETZWRDXHBBOYLYLURXBELK9YLIXXGHSP9TNNASKD'
            b'GFVJQV99CMXRM9VHASOBYBTWIMAJLBRUPZQLDCKOFAPHG9DKVVEFHTZAGNC9KH9K'
            b'9HIFNLUINQFTQTSALBNV9HRWXDGDEBBKIMQCDWVTMPDIVCXHGKDFPAKTSYYJIROE'
            b'NCJOZXVBNLUIUJHHAXZ9PTMNFGRRCNHQUVEESVSYNSIQXDRKKBMWJOQSMIK9FPHT'
            b'NAJUYTQ9BLOG9GZPXHACSPIFCDX9LIVQDISFAVZWQUXP9BROHMGBHFTVWEWCZRPT'
            b'AMTXXLVLZBT9BMOSJXAIGYUXICBUGQDOJRMYFWYGLT9UBTKGZZPNDIPNVIHQIBXF'
            b'QACGYPWTKJSRHVQLVJAJWFGNFLAJYOADR9XNOAYOLHKEUGWSOCXYJVHWLRRBE9XY'
            b'LQDYJXYMURFPXTMNHEEXJGVY9ADSJICXGWOUKYWVMXMWSJQVPKTUQUSCHTREWZNT'
            b'XBDUJWDVTMXPABBHGYOCUNFIFTUQTRAVTCFAQNNAAXBCRILNVZGGKEKIUOXBVMXL'
            b'FNCSHFMH9PYR9DBXFKWIBTAZLQTMVEGCZLWESPAHZLVTCGUUJXEAPCEYBXGGARHG'
            b'DODWULDHHMMKEIYRFFEMQTTGSGWTOGBZYEULWWFINFHGYEDHHXAJASMQCLBKWYXS'
            b'BIWZLMEZVXUWP999OROQYLOFVAZGJIGHMTGJSZNGXFWMMUCGGQXB9ASA9UCVZLVY'
            b'ZG9XBIF9HUAB9HBYERWFJ9IEDMAYZSIFDHOX9HRQSDEGWUAODHRNVBQWTBK9JFZB'
            b'NKBATUXBZOIEHPTFPQXSBGHGOEVMUTRPSTRFOKHWEUPUSEROZEBSMTSTZSHFTQ9U'
            b'XYTMDVLAPXSHZGYLPVDGTCGHOQSWJJEDARRUPCYFHJOSPVSTNEERBJOERGU9TTOW'
            b'9GSVZEODZOEQZATYADJ9NURBJNTPBYECGGWP9SVOZAWXT9RLKBKL9TBBWXSDOSXR'
            b'JHPKMLIPWKXSM9MPNQWEYLDPRLTUUNEFHXUFCLLJLZIRGUMEJCTIHC9VPS9YPENZ'
            b'PBYKTSBPXIPZHNYZYDPOYRIFEZWOFDYMZTUOMZZHLSLZMTDIMTTXMHHTDLIVRSID'
            b'SWJBCDCKEYZPTLLZP9IMNJSRXICEGTPZXVXAVIBGJMMOUNPUKXHIANUPGJANUHTG'
            b'9ZPZCBFRMLHYOPFAKGRZSZJARBEEPQZ9TKJRQLXEG9IOHETGXCMKT9XZUBPMIQWX'
            b'RRRFF9POXJBXW9NPUIOYNET9CTUWJB9RQDHVIAFLKILVBDLOYZAKIRHAUXE9ORNA'
            b'PVXRTUY9CNXAPFPNUADXHDQWGRCVBZMUASLOPAYHLNGNUVVTDQCSOSTOOILZFXBX'
            b'UPILJVVDUIRBWQUYNOJX99BTZNYQZGTENKGEKKADMPDWQB9ICWBWFHKAPRNDGGWO'
            b'UXDTJKMASYOPYNYPTOEN9EDLXVVUMELPGG9ZLAJXQFTIEA9HRJQCJLRUSLBGIWRW'
            b'RXMTSAYVNHNJCYDSYNBPH9XEI9NFEDANKTZ9RWSCMPV9XVBTBZVDO9HABGD9VDOI'
            b'XFMWBCHERKTDPDQFQSVNZLZRPHVZTFTL9LRAIMXLMTEZFAKK9CMYVPRTGBXGIMHU'
            b'UVWCHDUUEZMZFMDSUQRVVPHZDUTOTLPSKQEHWNLOXKGGJKHHUNQIJXUTNYMZIL9U'
            b'OEKECBSTCRVTVKUWETWPECLAXJWUNXXNRDBR99KJSWCHJBTMK9TSLLKWUCMMWNAB'
            b'UZLKLCJXHPUWVLIEIHYTZRPTZJTUMDDVEFCDRQYHPBF9WVMATUIQXGWTGAHQSTNR'
            b'VZZIPBRPIUOZLXRGEWSUVDXIQPAONF9QPFYIMUEMDXOMFPKKJNGRBNMKXNJUF9IQ'
            b'IHPEBHSLWQWXJZNEBKCQUSRWOEGMWFZYGHFUUHDBBOBKSTXT9HGOORUQMFBFBICA'
            b'HBQNOBVDCZVGZGASCINUGVEMM9LLPWTNWWVKWYIYDIJEKAVBEFPAVMFWEOYMTOHL'
            b'ZVPRMIINUJTTESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIY999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99D99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #2, Part 1 of 2
          TryteString(
            b'KJVG9EKTMPWE9PKWGGJJPDISCX9CJXGWUUPOLKKBVUWUYNBACOOF9LEQGNM9YYGN'
            b'XJEMOBGSDCPLP9CQIFBCLENUCJGCCSWYU9WVFTRZZCPCZXEGMDDPSYDTYUIMVFSL'
            b'GHTAZWJRHY9ZQMFGROIIUIQVDSOTUIRMSROYDPTWRURMZAILGBWIVADPYHPTSFGK'
            b'APPMVTTHGLYXZHPFUO9HBAJIOUJOOABAQSOQUKLSVQGSDIHEHGTQSPM9EDHLEQSF'
            b'FAAQXR9MUREVQ9MEGXNXMNJVWXYEIRYAOFHDFZAKNVWKTVIHKXWVT9VOZRUPUKXA'
            b'SIFAZQVZSWHBQU9RGVLJMRVQCMCYSQEIMIAKOHNAKQLTIMEHMZMGAKCIPGHQTWOR'
            b'BLVKISGPKIIMAMQWMZUNTKJSQZAZNYEGORGNRTKCLNRSOQJRBUCPSDLKLGGRBACI'
            b'ULLZBFBUNQXACKL9WFEKKAHGLBBRNNEXZPPH9UZESFFKVBOPROFHQOKYAVTJDDVA'
            b'UGUAURHLESIEIITDVVRCTRKOGUPERJHNJMXTLNVMWVDZITSPEHRYJKEZVTZSJEYT'
            b'OQEGNJRMCJLYYKPGDFUFQHGWRDGEWBXYOGEZ9IXRWJAQLKHPROWIEVI9ILNOXTPO'
            b'SRLETMNEQ9P9WLXCUZNMGFK9EYHABBCSEZSGMNJZOEEGRVNU9ASSOOLCXXZKZPFW'
            b'U9EEUUQRACVGZPL9MQINGLYPUTUPTLPKWPHRFFBRHZQWIVOXPGAKCQQPRKPPZUHO'
            b'JISYASMRYMCMJZNR9D9OQANUXGJXSUSZQFWDJUTNCDKAUAFYKJNVAMBLTPPRPIJR'
            b'RKQMCIHHGKPQPUQHWJNIEPDLRAYSJXVSJVKAGBAJCMGQSCZFTEJSG9LUWZGFBGQU'
            b'HFUHWDHND9WJBPOQQXDEATOBGXDGM9BKSDCOEZ9IENZPPDUPMKCUKYBIBTBMJPJL'
            b'DNSOPEKHVGQKLGUISUFSYMHR9I9LRPLCXJTDHHEXKQEVIFOUGKJEILQIHFG9FWOU'
            b'BXRHCRHLOYAXTFQUWKJBSX9GNPCWXUQJRHDBOBRZPQAPMKCIZGULPZDYLLBMAFJZ'
            b'XGIRVAAVUUCSNGDGJQJTAPV9QXYIABIHBXILKQLGDGXQUVADQGDFKKDKMU9WKBEE'
            b'Y9TAVRYQDQFKPDMLMUAEGBHVJPSIZOEQGCSYNJCICXROXHPZFUXASQJXZEHQTEUK'
            b'FIYQIGJWORKAIQUFROYGMIDFAJOUFAYYWMXUGJFPSRTGEUWWLOXEUTKZCZQHWFUN'
            b'HTMZVIJ9VYOLBTAIFB9EN9NFVAABVFIBIWXLJSUOYELOQSIPK99AXSXCPECWOXFU'
            b'VDIANVO9PKZUESMFWIEVWLEHLCVKDXEROLNEMYRRCJDPAYVTYAYSL9AFZH9GXHXZ'
            b'ORXZEQTUJEDJGCYCQAENYZRKDJSK9TOCKKCXOSSTOAIO9UVAKQJBVOS9RUQIESCI'
            b'JYRWYRUPMIJEHR9EGZ9YMHQXALUUDMCFYFOMLIGORMMBCDJMFCNNELGPXHICRNRK'
            b'ONBKACHLLSABUNHQ9TU9OSSTQXGWBLRRTSKZORXILALQYRXDDMXPPUTEGTVCHSOV'
            b'YZEEJMRRECGBMXBORUTIQUNMJDXBSZSYYA9UOTFWMQOHURUFSUESLMILBBKGHTTF'
            b'TZONNQIMJSLILKAQJRDTNVK9PHAMNKZXRHSOPGKKLJBRDYAC9BRURJWUIJLUWXNQ'
            b'OSVVLFEBROMJCGVYZWIPOYFQRBUUNJLIGPVDLADFLZJGZBLEBEQEUDUZOIFFZLZR'
            b'XCPQVMIARFLZRIOFFEHVFJZXFQFLCJSEXRPUKGMWBMGXEHIEZKOKGH9JXAUXACEB'
            b'LXKLZT9P9NJGXRZWZJAZCNKR9CHRRNCFOLBCSZXKXOIGZHZSTDKTHOWZTEXWOIZL'
            b'PEGPKTBRENSCOYBQJSZQ9XPNRORQOWMPGBXKSSLDIUVEAJODEUZJKZE9MBVTQXWF'
            b'XXXOG9QGDWMNZEWVWVDZWIFKSWZIDNBYEJP9VBKQNOJEXVPZQTHVUCSK9QCMEPUS'
            b'9Y9FQPWEACAEBIQSVPJEL9ZBSETINIYMSPIXLADSHTDYRAYUTMXDCABIUUETMNLU'
            b'RELTPAGEDNMQZALFWOPAI9WUFOSUTOFUUWFAFRFVYOPITBVSG9IBVNJCOITYMTCC'
            b'IJIZWVPYGQETESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIYYZ9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99C99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 2 of 2
          TryteString(
            b'SWHZKSNCOQXPCGRTYJPUGKLBNEJFXASKY9XAUROGDAO9QQLIVRZQDJDTPLNTBGUU'
            b'FGELJPSGUMGPIUNCCTQEFU9UZIJJYJXCYWRADRHHXKBEDG9HTCHJHXUJRKMIUFOS'
            b'KDGMI9QPCYQSWDCUYKQQEONJEKYWELG9MSNBHRHILGSSKMRCQJBOGNYBKEMTOFEU'
            b'BEOUBD9ULP9PHWYKXEQNDMUR9BGDGPEUFRFRGNGFJFPYQXABSDALTYKL9SM9VVQC'
            b'OHY9AS99EYWSHUNEQVVNLS9CNPEVMPOKMWQYFPGTNJBZCDWYPFWSBKZOYXNNVMPO'
            b'DEHMHNIYZCHIEDEAB9TLFOWVHF99GVRWUZWSN9IQOKWIXERKRQETZS9ZJJSQRLLP'
            b'QXEWNMFVVBWOIK9MBYCEGUJ9HJRIIMBVJNGXMGPGDLOLYWFVQNOKTFZRBJSSBJTE'
            b'TGAIUGZOYQOFTVBKAQY9SSJWJXXYAUVUQWPXVCISFSDSHDQCPVMG9GVDAO9GIMDH'
            b'ZWJOKSUEUFHBGSCZKNBTZWJXSFYNJSBSEL9UMZBAZRGYCEHOSJBMKMPMNXKEVTMU'
            b'DEFWBIKOXUSBNPTNECGVLYSOGUDJDPHYFADXRAOLQXJSJDANINJEOMCFAWWITREG'
            b'CDF9OZ9ZKHPJZJNMOVGX9OKQBSGVZYWKNOPVJEOZEI9BPE9GCUEQVAHSBBRBGQTE'
            b'XVZCSL9ECOWPOWZCVSCBOUSNQMTJIEKHXL9NCPRMLRNKQEHYJCLRHGZKFNBJIPKS'
            b'KPRFTSKFJULTBTXFDQHWUYOSDQBHPAINVEPKCCHJDTZOJIGJZOF9AEQDBKCZSZMI'
            b'WUUVHVGAFKALGITVQQKBAHKCIFSVMVZ9UDQABVIANTBUQOFBIXQBWB9KKQOVJZNV'
            b'BEDAZKUTRNKGJQWMTEKV9KGCIBRDCBAPKSTMCZGUV9HTAABQDKGQBCRFNXBMZRTH'
            b'F9MO9GAGQDYDVLOFMDE9QQZYR9GDSBLUVVMKMCZIMDPNCVLGDKBACWQJRWOQNKBT'
            b'SDJFKQMKTVKXVNAHRHZALJGVAMXWJYRAKTEJFXAHBQGSYWWQVECQYPXVFWILNFZK'
            b'GGRIFCJBSIZRDJXRJHSURPCZKOWKLFRUMVENEGMNKUAOGVUECDSGAZNQZKBJDJPV'
            b'BXLOTID9QLMFNGIWKAAIQTJJROSZBXPQRXAUCV99OGCEOTQCJ9II9ASZL9XGNSVU'
            b'XVKPXYOJMF9PX9GSLEROR9FXVQ9MLEMEW9IWNWBNVAYXZ9ZETTDSMLGZAKHE9IUJ'
            b'BFUHXW9KWCNZOZCCTFGBGWSDAQGGSPSQHOMUVJMLWBDAKYQZMWPQLLYAGUMOVMVL'
            b'FD9TO9OUBTVUHHUNSFSATSEGBFVGDZUBMTWWFDPSQVCUFRVKHYYPDWRPNSKXRFTV'
            b'EIBVZNGUZRQCPXVKBPKQDDLEBWIEBIPTEJIYFHBXCUVCCKTKEJAYRZCKAXLMELIN'
            b'WUZHG9JFBSBAKHIXMWHUWUFHFNLXNO9GKINYKRTCNN99PHPHO9MJAGUYZAPNSPWU'
            b'Z99E9BEADKETLOALWNANYMHSLLQSBS9YTYVJKTVWFUVS9MFOWCHLEUUFUWTYGLZX'
            b'FDUXVABTVFXFPUEPIUEIAVSZSSZQJTHNGKBJXADRHVTIBERILMCCGWUITYQHGEEG'
            b'WIZZOII9B9EVVVFJNEYEWH9ZVOJGHKVPYKDEZZSPBAOBQGGWPWXTCKSLSHJQYCDH'
            b'AYIQ9QVSQFPXZDBYSJJKSNTRXORHLPVOYVMIGALRPXYWQWSJPPFTJCYXAATLBFNS'
            b'GVXKFJXHYTILNOQZROUCDUTWOMGYBVTWPKJY9RVKKWQQMVCHJEUBELJDKJPGYLXU'
            b'XATNOIJHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHVHE9EEB'
            b'QYSNWECSPAJHGLTEUCXALBRVGXFENUCOONSUFZLHTLVQNPDZDIVDQHWVLDEDPFQL'
            b'JZWF9GFZMPZXFVEQECLUZBBFVSAPEXJLKKOMXEPHZAKP9WYTGQOML9FQSBMSFLOG'
            b'RLFQKUCUWFX9DNAOZSSKBUV9IBVIRNUWYBKJVKLJ9PPNLGJATKDCAGVFIVPXRABH'
            b'ZVZACJIG9WOKKLFCRDSMTWSCYHOZEEXRIMPQBXVXQAYKZIADSM9GUBICGKGQYNHK'
            b'VYOZFRVCHNMTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVW999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99B99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 1 of 2
          TryteString(
            b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWG'
            b'HMNUXBJTOBKZFDNJEMAOPPLR9OOQJCDVO9XSCYQJQVTXQDYWQEBIXKDZAFWINAHJ'
            b'ELJTDPVMUEWSVCJA9ONDYBNANWCGLBQMEMTBFDMWLCMQHGJLGYDQGIMLSNQHBGSV'
            b'TDZSGNQAL9OHRAPDKYSVBTNYRUUBNEEAINJMOVOHOWXUAIEDAIQDESQFCKJELHAV'
            b'ODSMXMKEHTDKCDIWWISXSAHQE9TJTLJZGXIABHU9CUACMLVSSYV9UJREPWFVYWWX'
            b'PYYJRP9DOEKNDMBSBKKHIFMPXZXIJERXRZVBVDBYNZBBCCOSEDOLDGSNQK99HIYS'
            b'WNYYEBLRT9MADLXLLZJOSZCFWAVZY9XUPNZUVOSKBMKXXJNRKDBOSGUGME9QNBMH'
            b'IWXWXPEEUVQAQVUXDJGMJOBXG9VJBWPRQRCCQSNBEHTLGOKJVYEPQOJO9QIZLYAV'
            b'LCKVXKEKRGBSZJAC9KTSSNMDQGKCLPZDJAQ9PBQMLUONVVFAWTMREGFXJMRRGL9M'
            b'KNPOZGOYRPDCYEJCYJUN9HYNSNHXARMRJVXBUHOP9K9BIIEYGSHBUESKTAOQOEAN'
            b'EAIHYHVGSVNPXWRBTJAMKMWEQOSYEWXLSRYVOSTMPOGYNPDNFLOICXVHYBDHSXVR'
            b'KVWNVZOZQDOITZWICSYEW9RGCPPUJYVIYVTSZILYENYUYUGDSGWVYWRMZJNCTTPV'
            b'WDWXAPVZQQKI9CGEQPBFPCLGDDEGBUUTISNCMJXQCTUNKQTLCATNOIRPMEUQBQTH'
            b'HQYRGDLZEUZBALNQDXJYZBVXDPLVOVVAUCQSCGRTUJRBBNRV9ORETTGFIXBBBVOP'
            b'FHPKGPKVBYFTZMWUVZYVWWSDKQVONMPLLQTV9IZUWLUWZNLCVJNPMG9CMXQG9D9W'
            b'YCANBRMYV9DU9FMJT9JHT9RWCGLHFCODXJVFQBLTKJWVNVGSUHNWLHNPLZDSWDMD'
            b'VQTLVCSVFJJTIQZFAPCXJWDAXWJKJVOKHALCQQTIXABPFXPUFK9IKXYUGMPXNSQC'
            b'JDVETOVEX9LXYLXWRW9PFEYJCUJHLUB9NXTUGLIQMDGPDPSJTWDYEWXQAICLN9BT'
            b'GNBJWLVAXZGNCYXGHBMRUVVYTJGH9XDGSZHQDYKFGMOWORSFDFBLJHBRTXRSEBAL'
            b'CJIJTQJYDZZKWZGVAPFVKVEOXGYRLMBSPFHUIJZZFMFVOTLPUWSYZCWFZMAALHRG'
            b'SYSXSMOHWARYZZVIAKXAHGY9SROWPVFACXXLQEXXOJCKXRRZHBZXJIBWQMMZTRDF'
            b'YQBSBBZQQXGCAAECMQINHJRBSGOYPCGWKPWCHBKOJTIGDASZFGONTUGDSOOLEMGO'
            b'EBFCZZJZSCGXPHXHB9WGMMFVUTCHDBSAMYTECQZWGCXAWTCTIBZHQVUAIBPZHBBT'
            b'ZAERYU9XAMKBHCHGZISSPOWJIRZTAXDHMAYBPXOXWDIUDHNBTFJNVHHJO9AWAEC9'
            b'UPRRFJLNGKTXJXFDGODDOPMGLALRIJBVIFLQTYQPKCKCRBYPBYGUUFJGJFVCOURN'
            b'KCGNTQNNKHDDPIVZHCJSLDUYHVPAX9YJOFTTFSKFHTOOQQRCPYZKTDVCUZGBOBZK'
            b'LVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VZZGUNUCKOYLYXOVHMGULWGSRCGX'
            b'ZLJVNIMZBLFOJJKOTUREMBXYOZXDUP9ROUVYOSJBGGFZMIFTKHJHHJGZJNOYQWFZ'
            b'AHLJWWDDFQQAMEGJUEUSIWOHKFJWRXRSJWYPGIGZGMFNAIDGDOUUQUVHJZQPJMLC'
            b'GKGADXAXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVGXBBPBE'
            b'BDVGZDBWMDMLPXYJBBRNOMKGR9TSVUXSRYXQTCTYLFQORMIGDKBJLNLCQXACVCBJ'
            b'GVWRJNYPCKOAILPLMWBYKDLDXLIZMZFWDXUWDEGDUURQGMJNUGJXDXYJGKOTQBGC'
            b'HATROPKEN9YTXDUOCMXPGHPDANTJFRRVEVBFVCNTWNMMOVAVKBNSJIWWBVHBMCSU'
            b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHV'
            b'ITGYROZZNQPTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVWN99999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Spend transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUETWO9DONTUSEINPRODUCTION99999XYYNXZLKBYNFPXA9'
            b'RUGZVEGVPLLFJEM9ZZOUINE9ONOWOB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99999999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_pass_inputs_explicit_with_change(self):
    """
    Preparing a bundle with specified inputs, change address needed.
    """
    self.adapter.seed_response('getBalances', {
      'balances': [86],
      'duration': '1',

      'milestone':
        'TESTVALUE9DONTUSEINPRODUCTION99999ZNIUXU'
        'FIVFBBYQHFYZYIEEWZL9VPMMKIIYTEZRRHXJXKIKF',
    })

    mock_signature_fragment_generator = MockSignatureFragmentGenerator([
      TryteString(
        b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWGHM'
        b'NUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHLYUHQ'
        b'P99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGBPNNFZD'
        b'WRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9KFEEXXVN'
        b'EXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFCYSWBUJYVGE'
        b'UXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMMKQVDYAZ9SVFI'
        b'UUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9OIZWLNINYRLGJQC'
        b'UBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNENNWXLF9ZVHZIDWBLF'
        b'KVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPWNOWQDMHPKOJTYYECKN'
        b'GCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJDTPJDOASLVRXFNJJHXQTK'
        b'MKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHFXPUIEEA9HPHJTCJJWZPUHK'
        b'AAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLFNTVDMYPRCYPQR9MNBBAMGOJX'
        b'PRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHFXTYBCMLRVHWNQDSQUIHHTAUTZC'
        b'JFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBUBUIO9TRXQLAYHDDSXGJ9NB9FKMUU'
        b'US9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXHGHYCV9VDCXHGTCOOUEXIITVKHXCSUS'
        b'OIRTMEAKMTYZCMAWURNX9JOVDICICKHXQYBXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMF'
        b'YQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLWRMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9C'
        b'CMCO9TCKLKSJEAMFKQMXEYETISVEYDOSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFC'
        b'SULGRHPONWJDVWT9UELEKEPQEAFKDLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYK'
        b'ADLUCRNEPAIWYSX9LT9QWQRKU9WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPD'
        b'QMMOOOCKYVEZJ9ZJQRJHNCKRFDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSX'
        b'DTOEEKLF9WWZGLSGIOQZWCJJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVM'
        b'XJGCSQSGYFLGYDZUH9EHUDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCU'
        b'Y9FFBTPSTJFEFROCEXFFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKN'
        b'OGCD9OPQ9WDMKRPIUNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPY'
        b'ZKTDVCUZGBOBZKLVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJH'
        b'UBWYXGECBPQOPOHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLR'
        b'JYTXTBANBZWFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQ'
        b'FLUXOFUNVDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVG'
        b'XBBPBEBDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICN'
        b'C9BT9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSK'
        b'NHJHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
        b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHVIT'
        b'GYROZZNQP'
      ),

      TryteString(
        b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKXFI'
        b'NPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPTYFFM'
        b'9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSCKCYPIE'
        b'AFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFLKOIRUEJU'
        b'PZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTLTKWYXHVAFU'
        b'VVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWLOFNXBDCT9JLA'
        b'9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFDGNJII9LSW9IQKV'
        b'99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKKXCKXLRGDT9NXIGQV'
        b'ZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEHKHVATNVWX9ZGTISUKP'
        b'KTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZHUDDUFJVQPBYNJREQ99U'
        b'HOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXUFYILOPOCJFVVEJEJN9ULGX'
        b'IABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLOFNA9KNERURWJPROBBXEWICDK'
        b'KCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXCGCLFETWOZYIUZVJVYVB9YGXSSD'
        b'XYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEHRIVMICUQPUUFRFCBF9NUUWSQBTVI'
        b'YFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUEVOFDMQLDESMQDPQUSOOKZJ9QLXTAFP'
        b'XXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFOEKQDEIWGXZPBXSOGGQFILUJTKDLWVKPV'
        b'ISU9QOATYVKJHLDLOKROZNFAGS9CICXXIUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTE'
        b'NYEWQWEHP9DDWOUJDWSTSOGYQPALFMKCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSU'
        b'Y9QTABAJOJAAMGVBCSLAAOBXZOJZLIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYC'
        b'HOVTLGDICIWTCIGNRHLBZBVSXMPBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCS'
        b'SCVPNBXEDCMHKFVDMHJTSP9JI9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFY'
        b'GVSNOYAEC9NWTCVEJBSXLYWTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQ'
        b'GVXENKIGW9XZTPBAIMZLHWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVC'
        b'FAXJKJNUC9NQHPEXWIOHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXA'
        b'JGFPHKAQDCRTKQBXPZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9U'
        b'SZRWUPZAASWDBXOVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQ'
        b'OA9DOY9RXZNWBFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFM'
        b'GRPGSCFRZINEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYND'
        b'TURLBPXGNQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHV'
        b'HE9EEBQYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGY'
        b'VPNSDOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLL'
        b'ASNQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
        b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZKW'
        b'FQNAJTO99'
      ),
    ])

    with patch(
        'iota.transaction.ProposedBundle._create_signature_fragment_generator',
        Mock(return_value=mock_signature_fragment_generator),
    ):
      response = self.command(
        seed = Seed(
          b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
          b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
        ),

        transfers = [
          ProposedTransaction(
            value = 42,
            address = Address(
              b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
              b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
            ),
          ),
        ],

        inputs = [
          Address(
            trytes =
              b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
              b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

            key_index = 4,
          ),
        ],

        change_address =
          Address(
            b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJ'
            b'RBOSBIMNTGDYKUDYYFJFGZOHORYSQPCWJRKHIOVIY',
          ),
      )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # Change transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIYQB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99C99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 2 of 2
          TryteString(
            b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKX'
            b'FINPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPT'
            b'YFFM9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSC'
            b'KCYPIEAFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFL'
            b'KOIRUEJUPZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTL'
            b'TKWYXHVAFUVVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWL'
            b'OFNXBDCT9JLA9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFD'
            b'GNJII9LSW9IQKV99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKK'
            b'XCKXLRGDT9NXIGQVZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEH'
            b'KHVATNVWX9ZGTISUKPKTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZ'
            b'HUDDUFJVQPBYNJREQ99UHOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXU'
            b'FYILOPOCJFVVEJEJN9ULGXIABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLO'
            b'FNA9KNERURWJPROBBXEWICDKKCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXC'
            b'GCLFETWOZYIUZVJVYVB9YGXSSDXYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEH'
            b'RIVMICUQPUUFRFCBF9NUUWSQBTVIYFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUE'
            b'VOFDMQLDESMQDPQUSOOKZJ9QLXTAFPXXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFO'
            b'EKQDEIWGXZPBXSOGGQFILUJTKDLWVKPVISU9QOATYVKJHLDLOKROZNFAGS9CICXX'
            b'IUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTENYEWQWEHP9DDWOUJDWSTSOGYQPALFM'
            b'KCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSUY9QTABAJOJAAMGVBCSLAAOBXZOJZ'
            b'LIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYCHOVTLGDICIWTCIGNRHLBZBVSXM'
            b'PBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCSSCVPNBXEDCMHKFVDMHJTSP9J'
            b'I9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFYGVSNOYAEC9NWTCVEJBSXLY'
            b'WTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQGVXENKIGW9XZTPBAIMZL'
            b'HWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVCFAXJKJNUC9NQHPEXWI'
            b'OHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXAJGFPHKAQDCRTKQBX'
            b'PZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9USZRWUPZAASWDBX'
            b'OVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQOA9DOY9RXZNW'
            b'BFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFMGRPGSCFRZI'
            b'NEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYNDTURLBPXG'
            b'NQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHVHE9EEB'
            b'QYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGYVPNS'
            b'DOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLLAS'
            b'NQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
            b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZ'
            b'KWFQNAJTO99TESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVW999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99B99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 1 of 2
          TryteString(
            b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWG'
            b'HMNUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHL'
            b'YUHQP99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGB'
            b'PNNFZDWRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9'
            b'KFEEXXVNEXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFC'
            b'YSWBUJYVGEUXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMM'
            b'KQVDYAZ9SVFIUUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9O'
            b'IZWLNINYRLGJQCUBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNEN'
            b'NWXLF9ZVHZIDWBLFKVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPW'
            b'NOWQDMHPKOJTYYECKNGCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJD'
            b'TPJDOASLVRXFNJJHXQTKMKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHF'
            b'XPUIEEA9HPHJTCJJWZPUHKAAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLF'
            b'NTVDMYPRCYPQR9MNBBAMGOJXPRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHF'
            b'XTYBCMLRVHWNQDSQUIHHTAUTZCJFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBU'
            b'BUIO9TRXQLAYHDDSXGJ9NB9FKMUUUS9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXH'
            b'GHYCV9VDCXHGTCOOUEXIITVKHXCSUSOIRTMEAKMTYZCMAWURNX9JOVDICICKHXQY'
            b'BXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMFYQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLW'
            b'RMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9CCMCO9TCKLKSJEAMFKQMXEYETISVEYD'
            b'OSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFCSULGRHPONWJDVWT9UELEKEPQEAFK'
            b'DLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYKADLUCRNEPAIWYSX9LT9QWQRKU9'
            b'WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPDQMMOOOCKYVEZJ9ZJQRJHNCKR'
            b'FDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSXDTOEEKLF9WWZGLSGIOQZWC'
            b'JJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVMXJGCSQSGYFLGYDZUH9EH'
            b'UDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCUY9FFBTPSTJFEFROCEX'
            b'FFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKNOGCD9OPQ9WDMKRPI'
            b'UNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPYZKTDVCUZGBOBZK'
            b'LVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJHUBWYXGECBPQO'
            b'POHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLRJYTXTBANBZ'
            b'WFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQFLUXOFUN'
            b'VDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVGXBBPBE'
            b'BDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICNC9BT'
            b'9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSKNH'
            b'JHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
            b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHV'
            b'ITGYROZZNQPTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVWVX9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Spend transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUETWO9DONTUSEINPRODUCTION99999XYYNXZLKBYNFPXA9'
            b'RUGZVEGVPLLFJEM9ZZOUINE9ONOWOB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99999999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_fail_inputs_explicit_insufficient(self):
    """
    Specified inputs are not sufficient to cover spend amount.
    """
    self.adapter.seed_response('getBalances', {
      'balances': [30],
      'duration': '1',

      'milestone':
        'TESTVALUE9DONTUSEINPRODUCTION99999ZNIUXU'
        'FIVFBBYQHFYZYIEEWZL9VPMMKIIYTEZRRHXJXKIKF',
    })

    with self.assertRaises(BadApiResponse):
      self.command(
        seed = Seed(
          b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
          b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
        ),

        transfers = [
          ProposedTransaction(
            value = 42,
            address = Address(
              b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
              b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
            ),
          ),
        ],

        inputs = [
          Address(
            trytes =
              b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
              b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

            key_index = 4,
          ),
        ],
      )

  def test_pass_inputs_implicit_no_change(self):
    """
    Preparing a bundle that finds inputs to use automatically, no
    change address needed.
    """
    # To keep the unit test focused, we will mock the ``getInputs``
    #   command that ``prepareTransfer`` calls internally.
    #
    #   References:
    #     - :py:class:`iota.commands.extended.prepare_transfer.PrepareTransferCommand`
    #     - :py:class:`iota.commands.extended.get_inputs.GetInputsCommand`
    mock_get_inputs = Mock(return_value={
      'inputs': [
        {
          'address':
            Address(
              trytes =
                b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
                b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

              balance   = 13,
              key_index = 4,
            ),

          'balance':  13,
          'keyIndex': 4,
        },

        {
          'address':
            Address(
              trytes =
                b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJ'
                b'RBOSBIMNTGDYKUDYYFJFGZOHORYSQPCWJRKHIOVIY',

              balance   = 29,
              key_index = 5,
            ),

          'balance': 29,
          'keyIndex': 5,
        },
      ],

      'totalBalance': 42,
    })

    mock_signature_fragment_generator = MockSignatureFragmentGenerator([
      TryteString(
        b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWGHM'
        b'NUXBJTOBKZFDNJEMAOPPLR9OOQJCDVO9XSCYQJQVTXQDYWQEBIXKDZAFWINAHJELJT'
        b'DPVMUEWSVCJA9ONDYBNANWCGLBQMEMTBFDMWLCMQHGJLGYDQGIMLSNQHBGSVTDZSGN'
        b'QAL9OHRAPDKYSVBTNYRUUBNEEAINJMOVOHOWXUAIEDAIQDESQFCKJELHAVODSMXMKE'
        b'HTDKCDIWWISXSAHQE9TJTLJZGXIABHU9CUACMLVSSYV9UJREPWFVYWWXPYYJRP9DOE'
        b'KNDMBSBKKHIFMPXZXIJERXRZVBVDBYNZBBCCOSEDOLDGSNQK99HIYSWNYYEBLRT9MA'
        b'DLXLLZJOSZCFWAVZY9XUPNZUVOSKBMKXXJNRKDBOSGUGME9QNBMHIWXWXPEEUVQAQV'
        b'UXDJGMJOBXG9VJBWPRQRCCQSNBEHTLGOKJVYEPQOJO9QIZLYAVLCKVXKEKRGBSZJAC'
        b'9KTSSNMDQGKCLPZDJAQ9PBQMLUONVVFAWTMREGFXJMRRGL9MKNPOZGOYRPDCYEJCYJ'
        b'UN9HYNSNHXARMRJVXBUHOP9K9BIIEYGSHBUESKTAOQOEANEAIHYHVGSVNPXWRBTJAM'
        b'KMWEQOSYEWXLSRYVOSTMPOGYNPDNFLOICXVHYBDHSXVRKVWNVZOZQDOITZWICSYEW9'
        b'RGCPPUJYVIYVTSZILYENYUYUGDSGWVYWRMZJNCTTPVWDWXAPVZQQKI9CGEQPBFPCLG'
        b'DDEGBUUTISNCMJXQCTUNKQTLCATNOIRPMEUQBQTHHQYRGDLZEUZBALNQDXJYZBVXDP'
        b'LVOVVAUCQSCGRTUJRBBNRV9ORETTGFIXBBBVOPFHPKGPKVBYFTZMWUVZYVWWSDKQVO'
        b'NMPLLQTV9IZUWLUWZNLCVJNPMG9CMXQG9D9WYCANBRMYV9DU9FMJT9JHT9RWCGLHFC'
        b'ODXJVFQBLTKJWVNVGSUHNWLHNPLZDSWDMDVQTLVCSVFJJTIQZFAPCXJWDAXWJKJVOK'
        b'HALCQQTIXABPFXPUFK9IKXYUGMPXNSQCJDVETOVEX9LXYLXWRW9PFEYJCUJHLUB9NX'
        b'TUGLIQMDGPDPSJTWDYEWXQAICLN9BTGNBJWLVAXZGNCYXGHBMRUVVYTJGH9XDGSZHQ'
        b'DYKFGMOWORSFDFBLJHBRTXRSEBALCJIJTQJYDZZKWZGVAPFVKVEOXGYRLMBSPFHUIJ'
        b'ZZFMFVOTLPUWSYZCWFZMAALHRGSYSXSMOHWARYZZVIAKXAHGY9SROWPVFACXXLQEXX'
        b'OJCKXRRZHBZXJIBWQMMZTRDFYQBSBBZQQXGCAAECMQINHJRBSGOYPCGWKPWCHBKOJT'
        b'IGDASZFGONTUGDSOOLEMGOEBFCZZJZSCGXPHXHB9WGMMFVUTCHDBSAMYTECQZWGCXA'
        b'WTCTIBZHQVUAIBPZHBBTZAERYU9XAMKBHCHGZISSPOWJIRZTAXDHMAYBPXOXWDIUDH'
        b'NBTFJNVHHJO9AWAEC9UPRRFJLNGKTXJXFDGODDOPMGLALRIJBVIFLQTYQPKCKCRBYP'
        b'BYGUUFJGJFVCOURNKCGNTQNNKHDDPIVZHCJSLDUYHVPAX9YJOFTTFSKFHTOOQQRCPY'
        b'ZKTDVCUZGBOBZKLVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VZZGUNUCKOYLYXOV'
        b'HMGULWGSRCGXZLJVNIMZBLFOJJKOTUREMBXYOZXDUP9ROUVYOSJBGGFZMIFTKHJHHJ'
        b'GZJNOYQWFZAHLJWWDDFQQAMEGJUEUSIWOHKFJWRXRSJWYPGIGZGMFNAIDGDOUUQUVH'
        b'JZQPJMLCGKGADXAXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVG'
        b'XBBPBEBDVGZDBWMDMLPXYJBBRNOMKGR9TSVUXSRYXQTCTYLFQORMIGDKBJLNLCQXAC'
        b'VCBJGVWRJNYPCKOAILPLMWBYKDLDXLIZMZFWDXUWDEGDUURQGMJNUGJXDXYJGKOTQB'
        b'GCHATROPKEN9YTXDUOCMXPGHPDANTJFRRVEVBFVCNTWNMMOVAVKBNSJIWWBVHBMCSU'
        b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHVIT'
        b'GYROZZNQP'
      ),

      TryteString(
        b'SWHZKSNCOQXPCGRTYJPUGKLBNEJFXASKY9XAUROGDAO9QQLIVRZQDJDTPLNTBGUUFG'
        b'ELJPSGUMGPIUNCCTQEFU9UZIJJYJXCYWRADRHHXKBEDG9HTCHJHXUJRKMIUFOSKDGM'
        b'I9QPCYQSWDCUYKQQEONJEKYWELG9MSNBHRHILGSSKMRCQJBOGNYBKEMTOFEUBEOUBD'
        b'9ULP9PHWYKXEQNDMUR9BGDGPEUFRFRGNGFJFPYQXABSDALTYKL9SM9VVQCOHY9AS99'
        b'EYWSHUNEQVVNLS9CNPEVMPOKMWQYFPGTNJBZCDWYPFWSBKZOYXNNVMPODEHMHNIYZC'
        b'HIEDEAB9TLFOWVHF99GVRWUZWSN9IQOKWIXERKRQETZS9ZJJSQRLLPQXEWNMFVVBWO'
        b'IK9MBYCEGUJ9HJRIIMBVJNGXMGPGDLOLYWFVQNOKTFZRBJSSBJTETGAIUGZOYQOFTV'
        b'BKAQY9SSJWJXXYAUVUQWPXVCISFSDSHDQCPVMG9GVDAO9GIMDHZWJOKSUEUFHBGSCZ'
        b'KNBTZWJXSFYNJSBSEL9UMZBAZRGYCEHOSJBMKMPMNXKEVTMUDEFWBIKOXUSBNPTNEC'
        b'GVLYSOGUDJDPHYFADXRAOLQXJSJDANINJEOMCFAWWITREGCDF9OZ9ZKHPJZJNMOVGX'
        b'9OKQBSGVZYWKNOPVJEOZEI9BPE9GCUEQVAHSBBRBGQTEXVZCSL9ECOWPOWZCVSCBOU'
        b'SNQMTJIEKHXL9NCPRMLRNKQEHYJCLRHGZKFNBJIPKSKPRFTSKFJULTBTXFDQHWUYOS'
        b'DQBHPAINVEPKCCHJDTZOJIGJZOF9AEQDBKCZSZMIWUUVHVGAFKALGITVQQKBAHKCIF'
        b'SVMVZ9UDQABVIANTBUQOFBIXQBWB9KKQOVJZNVBEDAZKUTRNKGJQWMTEKV9KGCIBRD'
        b'CBAPKSTMCZGUV9HTAABQDKGQBCRFNXBMZRTHF9MO9GAGQDYDVLOFMDE9QQZYR9GDSB'
        b'LUVVMKMCZIMDPNCVLGDKBACWQJRWOQNKBTSDJFKQMKTVKXVNAHRHZALJGVAMXWJYRA'
        b'KTEJFXAHBQGSYWWQVECQYPXVFWILNFZKGGRIFCJBSIZRDJXRJHSURPCZKOWKLFRUMV'
        b'ENEGMNKUAOGVUECDSGAZNQZKBJDJPVBXLOTID9QLMFNGIWKAAIQTJJROSZBXPQRXAU'
        b'CV99OGCEOTQCJ9II9ASZL9XGNSVUXVKPXYOJMF9PX9GSLEROR9FXVQ9MLEMEW9IWNW'
        b'BNVAYXZ9ZETTDSMLGZAKHE9IUJBFUHXW9KWCNZOZCCTFGBGWSDAQGGSPSQHOMUVJML'
        b'WBDAKYQZMWPQLLYAGUMOVMVLFD9TO9OUBTVUHHUNSFSATSEGBFVGDZUBMTWWFDPSQV'
        b'CUFRVKHYYPDWRPNSKXRFTVEIBVZNGUZRQCPXVKBPKQDDLEBWIEBIPTEJIYFHBXCUVC'
        b'CKTKEJAYRZCKAXLMELINWUZHG9JFBSBAKHIXMWHUWUFHFNLXNO9GKINYKRTCNN99PH'
        b'PHO9MJAGUYZAPNSPWUZ99E9BEADKETLOALWNANYMHSLLQSBS9YTYVJKTVWFUVS9MFO'
        b'WCHLEUUFUWTYGLZXFDUXVABTVFXFPUEPIUEIAVSZSSZQJTHNGKBJXADRHVTIBERILM'
        b'CCGWUITYQHGEEGWIZZOII9B9EVVVFJNEYEWH9ZVOJGHKVPYKDEZZSPBAOBQGGWPWXT'
        b'CKSLSHJQYCDHAYIQ9QVSQFPXZDBYSJJKSNTRXORHLPVOYVMIGALRPXYWQWSJPPFTJC'
        b'YXAATLBFNSGVXKFJXHYTILNOQZROUCDUTWOMGYBVTWPKJY9RVKKWQQMVCHJEUBELJD'
        b'KJPGYLXUXATNOIJHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHV'
        b'HE9EEBQYSNWECSPAJHGLTEUCXALBRVGXFENUCOONSUFZLHTLVQNPDZDIVDQHWVLDED'
        b'PFQLJZWF9GFZMPZXFVEQECLUZBBFVSAPEXJLKKOMXEPHZAKP9WYTGQOML9FQSBMSFL'
        b'OGRLFQKUCUWFX9DNAOZSSKBUV9IBVIRNUWYBKJVKLJ9PPNLGJATKDCAGVFIVPXRABH'
        b'ZVZACJIG9WOKKLFCRDSMTWSCYHOZEEXRIMPQBXVXQAYKZIADSM9GUBICGKGQYNHKVY'
        b'OZFRVCHNM'
      ),

      TryteString(
        b'KJVG9EKTMPWE9PKWGGJJPDISCX9CJXGWUUPOLKKBVUWUYNBACOOF9LEQGNM9YYGNXJ'
        b'EMOBGSDCPLP9CQIFBCLENUCJGCCSWYU9WVFTRZZCPCZXEGMDDPSYDTYUIMVFSLGHTA'
        b'ZWJRHY9ZQMFGROIIUIQVDSOTUIRMSROYDPTWRURMZAILGBWIVADPYHPTSFGKAPPMVT'
        b'THGLYXZHPFUO9HBAJIOUJOOABAQSOQUKLSVQGSDIHEHGTQSPM9EDHLEQSFFAAQXR9M'
        b'UREVQ9MEGXNXMNJVWXYEIRYAOFHDFZAKNVWKTVIHKXWVT9VOZRUPUKXASIFAZQVZSW'
        b'HBQU9RGVLJMRVQCMCYSQEIMIAKOHNAKQLTIMEHMZMGAKCIPGHQTWORBLVKISGPKIIM'
        b'AMQWMZUNTKJSQZAZNYEGORGNRTKCLNRSOQJRBUCPSDLKLGGRBACIULLZBFBUNQXACK'
        b'L9WFEKKAHGLBBRNNEXZPPH9UZESFFKVBOPROFHQOKYAVTJDDVAUGUAURHLESIEIITD'
        b'VVRCTRKOGUPERJHNJMXTLNVMWVDZITSPEHRYJKEZVTZSJEYTOQEGNJRMCJLYYKPGDF'
        b'UFQHGWRDGEWBXYOGEZ9IXRWJAQLKHPROWIEVI9ILNOXTPOSRLETMNEQ9P9WLXCUZNM'
        b'GFK9EYHABBCSEZSGMNJZOEEGRVNU9ASSOOLCXXZKZPFWU9EEUUQRACVGZPL9MQINGL'
        b'YPUTUPTLPKWPHRFFBRHZQWIVOXPGAKCQQPRKPPZUHOJISYASMRYMCMJZNR9D9OQANU'
        b'XGJXSUSZQFWDJUTNCDKAUAFYKJNVAMBLTPPRPIJRRKQMCIHHGKPQPUQHWJNIEPDLRA'
        b'YSJXVSJVKAGBAJCMGQSCZFTEJSG9LUWZGFBGQUHFUHWDHND9WJBPOQQXDEATOBGXDG'
        b'M9BKSDCOEZ9IENZPPDUPMKCUKYBIBTBMJPJLDNSOPEKHVGQKLGUISUFSYMHR9I9LRP'
        b'LCXJTDHHEXKQEVIFOUGKJEILQIHFG9FWOUBXRHCRHLOYAXTFQUWKJBSX9GNPCWXUQJ'
        b'RHDBOBRZPQAPMKCIZGULPZDYLLBMAFJZXGIRVAAVUUCSNGDGJQJTAPV9QXYIABIHBX'
        b'ILKQLGDGXQUVADQGDFKKDKMU9WKBEEY9TAVRYQDQFKPDMLMUAEGBHVJPSIZOEQGCSY'
        b'NJCICXROXHPZFUXASQJXZEHQTEUKFIYQIGJWORKAIQUFROYGMIDFAJOUFAYYWMXUGJ'
        b'FPSRTGEUWWLOXEUTKZCZQHWFUNHTMZVIJ9VYOLBTAIFB9EN9NFVAABVFIBIWXLJSUO'
        b'YELOQSIPK99AXSXCPECWOXFUVDIANVO9PKZUESMFWIEVWLEHLCVKDXEROLNEMYRRCJ'
        b'DPAYVTYAYSL9AFZH9GXHXZORXZEQTUJEDJGCYCQAENYZRKDJSK9TOCKKCXOSSTOAIO'
        b'9UVAKQJBVOS9RUQIESCIJYRWYRUPMIJEHR9EGZ9YMHQXALUUDMCFYFOMLIGORMMBCD'
        b'JMFCNNELGPXHICRNRKONBKACHLLSABUNHQ9TU9OSSTQXGWBLRRTSKZORXILALQYRXD'
        b'DMXPPUTEGTVCHSOVYZEEJMRRECGBMXBORUTIQUNMJDXBSZSYYA9UOTFWMQOHURUFSU'
        b'ESLMILBBKGHTTFTZONNQIMJSLILKAQJRDTNVK9PHAMNKZXRHSOPGKKLJBRDYAC9BRU'
        b'RJWUIJLUWXNQOSVVLFEBROMJCGVYZWIPOYFQRBUUNJLIGPVDLADFLZJGZBLEBEQEUD'
        b'UZOIFFZLZRXCPQVMIARFLZRIOFFEHVFJZXFQFLCJSEXRPUKGMWBMGXEHIEZKOKGH9J'
        b'XAUXACEBLXKLZT9P9NJGXRZWZJAZCNKR9CHRRNCFOLBCSZXKXOIGZHZSTDKTHOWZTE'
        b'XWOIZLPEGPKTBRENSCOYBQJSZQ9XPNRORQOWMPGBXKSSLDIUVEAJODEUZJKZE9MBVT'
        b'QXWFXXXOG9QGDWMNZEWVWVDZWIFKSWZIDNBYEJP9VBKQNOJEXVPZQTHVUCSK9QCMEP'
        b'US9Y9FQPWEACAEBIQSVPJEL9ZBSETINIYMSPIXLADSHTDYRAYUTMXDCABIUUETMNLU'
        b'RELTPAGEDNMQZALFWOPAI9WUFOSUTOFUUWFAFRFVYOPITBVSG9IBVNJCOITYMTCCIJ'
        b'IZWVPYGQE'
      ),

      TryteString(
        b'GWLDXDNSEIQCQJKVVFEWPWR99OKSHTVIJCNFEGSUM9DUQRO9ZJUWOOGP9XLABZFDXN'
        b'GOXZLWETWXTTBT9KIGB9VOMMTKNJTUUFGJIYZIMHEAEJTNTIIOLLO9VWCYX9JA9RML'
        b'SB9COUYKMRZQWJXMIFXCETZWRDXHBBOYLYLURXBELK9YLIXXGHSP9TNNASKDGFVJQV'
        b'99CMXRM9VHASOBYBTWIMAJLBRUPZQLDCKOFAPHG9DKVVEFHTZAGNC9KH9K9HIFNLUI'
        b'NQFTQTSALBNV9HRWXDGDEBBKIMQCDWVTMPDIVCXHGKDFPAKTSYYJIROENCJOZXVBNL'
        b'UIUJHHAXZ9PTMNFGRRCNHQUVEESVSYNSIQXDRKKBMWJOQSMIK9FPHTNAJUYTQ9BLOG'
        b'9GZPXHACSPIFCDX9LIVQDISFAVZWQUXP9BROHMGBHFTVWEWCZRPTAMTXXLVLZBT9BM'
        b'OSJXAIGYUXICBUGQDOJRMYFWYGLT9UBTKGZZPNDIPNVIHQIBXFQACGYPWTKJSRHVQL'
        b'VJAJWFGNFLAJYOADR9XNOAYOLHKEUGWSOCXYJVHWLRRBE9XYLQDYJXYMURFPXTMNHE'
        b'EXJGVY9ADSJICXGWOUKYWVMXMWSJQVPKTUQUSCHTREWZNTXBDUJWDVTMXPABBHGYOC'
        b'UNFIFTUQTRAVTCFAQNNAAXBCRILNVZGGKEKIUOXBVMXLFNCSHFMH9PYR9DBXFKWIBT'
        b'AZLQTMVEGCZLWESPAHZLVTCGUUJXEAPCEYBXGGARHGDODWULDHHMMKEIYRFFEMQTTG'
        b'SGWTOGBZYEULWWFINFHGYEDHHXAJASMQCLBKWYXSBIWZLMEZVXUWP999OROQYLOFVA'
        b'ZGJIGHMTGJSZNGXFWMMUCGGQXB9ASA9UCVZLVYZG9XBIF9HUAB9HBYERWFJ9IEDMAY'
        b'ZSIFDHOX9HRQSDEGWUAODHRNVBQWTBK9JFZBNKBATUXBZOIEHPTFPQXSBGHGOEVMUT'
        b'RPSTRFOKHWEUPUSEROZEBSMTSTZSHFTQ9UXYTMDVLAPXSHZGYLPVDGTCGHOQSWJJED'
        b'ARRUPCYFHJOSPVSTNEERBJOERGU9TTOW9GSVZEODZOEQZATYADJ9NURBJNTPBYECGG'
        b'WP9SVOZAWXT9RLKBKL9TBBWXSDOSXRJHPKMLIPWKXSM9MPNQWEYLDPRLTUUNEFHXUF'
        b'CLLJLZIRGUMEJCTIHC9VPS9YPENZPBYKTSBPXIPZHNYZYDPOYRIFEZWOFDYMZTUOMZ'
        b'ZHLSLZMTDIMTTXMHHTDLIVRSIDSWJBCDCKEYZPTLLZP9IMNJSRXICEGTPZXVXAVIBG'
        b'JMMOUNPUKXHIANUPGJANUHTG9ZPZCBFRMLHYOPFAKGRZSZJARBEEPQZ9TKJRQLXEG9'
        b'IOHETGXCMKT9XZUBPMIQWXRRRFF9POXJBXW9NPUIOYNET9CTUWJB9RQDHVIAFLKILV'
        b'BDLOYZAKIRHAUXE9ORNAPVXRTUY9CNXAPFPNUADXHDQWGRCVBZMUASLOPAYHLNGNUV'
        b'VTDQCSOSTOOILZFXBXUPILJVVDUIRBWQUYNOJX99BTZNYQZGTENKGEKKADMPDWQB9I'
        b'CWBWFHKAPRNDGGWOUXDTJKMASYOPYNYPTOEN9EDLXVVUMELPGG9ZLAJXQFTIEA9HRJ'
        b'QCJLRUSLBGIWRWRXMTSAYVNHNJCYDSYNBPH9XEI9NFEDANKTZ9RWSCMPV9XVBTBZVD'
        b'O9HABGD9VDOIXFMWBCHERKTDPDQFQSVNZLZRPHVZTFTL9LRAIMXLMTEZFAKK9CMYVP'
        b'RTGBXGIMHUUVWCHDUUEZMZFMDSUQRVVPHZDUTOTLPSKQEHWNLOXKGGJKHHUNQIJXUT'
        b'NYMZIL9UOEKECBSTCRVTVKUWETWPECLAXJWUNXXNRDBR99KJSWCHJBTMK9TSLLKWUC'
        b'MMWNABUZLKLCJXHPUWVLIEIHYTZRPTZJTUMDDVEFCDRQYHPBF9WVMATUIQXGWTGAHQ'
        b'STNRVZZIPBRPIUOZLXRGEWSUVDXIQPAONF9QPFYIMUEMDXOMFPKKJNGRBNMKXNJUF9'
        b'IQIHPEBHSLWQWXJZNEBKCQUSRWOEGMWFZYGHFUUHDBBOBKSTXT9HGOORUQMFBFBICA'
        b'HBQNOBVDCZVGZGASCINUGVEMM9LLPWTNWWVKWYIYDIJEKAVBEFPAVMFWEOYMTOHLZV'
        b'PRMIINUJT'
      ),
    ])

    with patch(
        'iota.transaction.ProposedBundle._create_signature_fragment_generator',
        Mock(return_value=mock_signature_fragment_generator),
    ):
      with patch(
          'iota.commands.extended.get_inputs.GetInputsCommand._execute',
          mock_get_inputs,
      ):
        response = self.command(
          seed = Seed(
            b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
            b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
          ),

          transfers = [
            ProposedTransaction(
              value = 42,
              address = Address(
                b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
                b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
              ),
            ),
          ],
        )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # Ipnut #2, Part 2 of 2
          TryteString(
            b'GWLDXDNSEIQCQJKVVFEWPWR99OKSHTVIJCNFEGSUM9DUQRO9ZJUWOOGP9XLABZFD'
            b'XNGOXZLWETWXTTBT9KIGB9VOMMTKNJTUUFGJIYZIMHEAEJTNTIIOLLO9VWCYX9JA'
            b'9RMLSB9COUYKMRZQWJXMIFXCETZWRDXHBBOYLYLURXBELK9YLIXXGHSP9TNNASKD'
            b'GFVJQV99CMXRM9VHASOBYBTWIMAJLBRUPZQLDCKOFAPHG9DKVVEFHTZAGNC9KH9K'
            b'9HIFNLUINQFTQTSALBNV9HRWXDGDEBBKIMQCDWVTMPDIVCXHGKDFPAKTSYYJIROE'
            b'NCJOZXVBNLUIUJHHAXZ9PTMNFGRRCNHQUVEESVSYNSIQXDRKKBMWJOQSMIK9FPHT'
            b'NAJUYTQ9BLOG9GZPXHACSPIFCDX9LIVQDISFAVZWQUXP9BROHMGBHFTVWEWCZRPT'
            b'AMTXXLVLZBT9BMOSJXAIGYUXICBUGQDOJRMYFWYGLT9UBTKGZZPNDIPNVIHQIBXF'
            b'QACGYPWTKJSRHVQLVJAJWFGNFLAJYOADR9XNOAYOLHKEUGWSOCXYJVHWLRRBE9XY'
            b'LQDYJXYMURFPXTMNHEEXJGVY9ADSJICXGWOUKYWVMXMWSJQVPKTUQUSCHTREWZNT'
            b'XBDUJWDVTMXPABBHGYOCUNFIFTUQTRAVTCFAQNNAAXBCRILNVZGGKEKIUOXBVMXL'
            b'FNCSHFMH9PYR9DBXFKWIBTAZLQTMVEGCZLWESPAHZLVTCGUUJXEAPCEYBXGGARHG'
            b'DODWULDHHMMKEIYRFFEMQTTGSGWTOGBZYEULWWFINFHGYEDHHXAJASMQCLBKWYXS'
            b'BIWZLMEZVXUWP999OROQYLOFVAZGJIGHMTGJSZNGXFWMMUCGGQXB9ASA9UCVZLVY'
            b'ZG9XBIF9HUAB9HBYERWFJ9IEDMAYZSIFDHOX9HRQSDEGWUAODHRNVBQWTBK9JFZB'
            b'NKBATUXBZOIEHPTFPQXSBGHGOEVMUTRPSTRFOKHWEUPUSEROZEBSMTSTZSHFTQ9U'
            b'XYTMDVLAPXSHZGYLPVDGTCGHOQSWJJEDARRUPCYFHJOSPVSTNEERBJOERGU9TTOW'
            b'9GSVZEODZOEQZATYADJ9NURBJNTPBYECGGWP9SVOZAWXT9RLKBKL9TBBWXSDOSXR'
            b'JHPKMLIPWKXSM9MPNQWEYLDPRLTUUNEFHXUFCLLJLZIRGUMEJCTIHC9VPS9YPENZ'
            b'PBYKTSBPXIPZHNYZYDPOYRIFEZWOFDYMZTUOMZZHLSLZMTDIMTTXMHHTDLIVRSID'
            b'SWJBCDCKEYZPTLLZP9IMNJSRXICEGTPZXVXAVIBGJMMOUNPUKXHIANUPGJANUHTG'
            b'9ZPZCBFRMLHYOPFAKGRZSZJARBEEPQZ9TKJRQLXEG9IOHETGXCMKT9XZUBPMIQWX'
            b'RRRFF9POXJBXW9NPUIOYNET9CTUWJB9RQDHVIAFLKILVBDLOYZAKIRHAUXE9ORNA'
            b'PVXRTUY9CNXAPFPNUADXHDQWGRCVBZMUASLOPAYHLNGNUVVTDQCSOSTOOILZFXBX'
            b'UPILJVVDUIRBWQUYNOJX99BTZNYQZGTENKGEKKADMPDWQB9ICWBWFHKAPRNDGGWO'
            b'UXDTJKMASYOPYNYPTOEN9EDLXVVUMELPGG9ZLAJXQFTIEA9HRJQCJLRUSLBGIWRW'
            b'RXMTSAYVNHNJCYDSYNBPH9XEI9NFEDANKTZ9RWSCMPV9XVBTBZVDO9HABGD9VDOI'
            b'XFMWBCHERKTDPDQFQSVNZLZRPHVZTFTL9LRAIMXLMTEZFAKK9CMYVPRTGBXGIMHU'
            b'UVWCHDUUEZMZFMDSUQRVVPHZDUTOTLPSKQEHWNLOXKGGJKHHUNQIJXUTNYMZIL9U'
            b'OEKECBSTCRVTVKUWETWPECLAXJWUNXXNRDBR99KJSWCHJBTMK9TSLLKWUCMMWNAB'
            b'UZLKLCJXHPUWVLIEIHYTZRPTZJTUMDDVEFCDRQYHPBF9WVMATUIQXGWTGAHQSTNR'
            b'VZZIPBRPIUOZLXRGEWSUVDXIQPAONF9QPFYIMUEMDXOMFPKKJNGRBNMKXNJUF9IQ'
            b'IHPEBHSLWQWXJZNEBKCQUSRWOEGMWFZYGHFUUHDBBOBKSTXT9HGOORUQMFBFBICA'
            b'HBQNOBVDCZVGZGASCINUGVEMM9LLPWTNWWVKWYIYDIJEKAVBEFPAVMFWEOYMTOHL'
            b'ZVPRMIINUJTTESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIY999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99D99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #2, Part 1 of 2
          TryteString(
            b'KJVG9EKTMPWE9PKWGGJJPDISCX9CJXGWUUPOLKKBVUWUYNBACOOF9LEQGNM9YYGN'
            b'XJEMOBGSDCPLP9CQIFBCLENUCJGCCSWYU9WVFTRZZCPCZXEGMDDPSYDTYUIMVFSL'
            b'GHTAZWJRHY9ZQMFGROIIUIQVDSOTUIRMSROYDPTWRURMZAILGBWIVADPYHPTSFGK'
            b'APPMVTTHGLYXZHPFUO9HBAJIOUJOOABAQSOQUKLSVQGSDIHEHGTQSPM9EDHLEQSF'
            b'FAAQXR9MUREVQ9MEGXNXMNJVWXYEIRYAOFHDFZAKNVWKTVIHKXWVT9VOZRUPUKXA'
            b'SIFAZQVZSWHBQU9RGVLJMRVQCMCYSQEIMIAKOHNAKQLTIMEHMZMGAKCIPGHQTWOR'
            b'BLVKISGPKIIMAMQWMZUNTKJSQZAZNYEGORGNRTKCLNRSOQJRBUCPSDLKLGGRBACI'
            b'ULLZBFBUNQXACKL9WFEKKAHGLBBRNNEXZPPH9UZESFFKVBOPROFHQOKYAVTJDDVA'
            b'UGUAURHLESIEIITDVVRCTRKOGUPERJHNJMXTLNVMWVDZITSPEHRYJKEZVTZSJEYT'
            b'OQEGNJRMCJLYYKPGDFUFQHGWRDGEWBXYOGEZ9IXRWJAQLKHPROWIEVI9ILNOXTPO'
            b'SRLETMNEQ9P9WLXCUZNMGFK9EYHABBCSEZSGMNJZOEEGRVNU9ASSOOLCXXZKZPFW'
            b'U9EEUUQRACVGZPL9MQINGLYPUTUPTLPKWPHRFFBRHZQWIVOXPGAKCQQPRKPPZUHO'
            b'JISYASMRYMCMJZNR9D9OQANUXGJXSUSZQFWDJUTNCDKAUAFYKJNVAMBLTPPRPIJR'
            b'RKQMCIHHGKPQPUQHWJNIEPDLRAYSJXVSJVKAGBAJCMGQSCZFTEJSG9LUWZGFBGQU'
            b'HFUHWDHND9WJBPOQQXDEATOBGXDGM9BKSDCOEZ9IENZPPDUPMKCUKYBIBTBMJPJL'
            b'DNSOPEKHVGQKLGUISUFSYMHR9I9LRPLCXJTDHHEXKQEVIFOUGKJEILQIHFG9FWOU'
            b'BXRHCRHLOYAXTFQUWKJBSX9GNPCWXUQJRHDBOBRZPQAPMKCIZGULPZDYLLBMAFJZ'
            b'XGIRVAAVUUCSNGDGJQJTAPV9QXYIABIHBXILKQLGDGXQUVADQGDFKKDKMU9WKBEE'
            b'Y9TAVRYQDQFKPDMLMUAEGBHVJPSIZOEQGCSYNJCICXROXHPZFUXASQJXZEHQTEUK'
            b'FIYQIGJWORKAIQUFROYGMIDFAJOUFAYYWMXUGJFPSRTGEUWWLOXEUTKZCZQHWFUN'
            b'HTMZVIJ9VYOLBTAIFB9EN9NFVAABVFIBIWXLJSUOYELOQSIPK99AXSXCPECWOXFU'
            b'VDIANVO9PKZUESMFWIEVWLEHLCVKDXEROLNEMYRRCJDPAYVTYAYSL9AFZH9GXHXZ'
            b'ORXZEQTUJEDJGCYCQAENYZRKDJSK9TOCKKCXOSSTOAIO9UVAKQJBVOS9RUQIESCI'
            b'JYRWYRUPMIJEHR9EGZ9YMHQXALUUDMCFYFOMLIGORMMBCDJMFCNNELGPXHICRNRK'
            b'ONBKACHLLSABUNHQ9TU9OSSTQXGWBLRRTSKZORXILALQYRXDDMXPPUTEGTVCHSOV'
            b'YZEEJMRRECGBMXBORUTIQUNMJDXBSZSYYA9UOTFWMQOHURUFSUESLMILBBKGHTTF'
            b'TZONNQIMJSLILKAQJRDTNVK9PHAMNKZXRHSOPGKKLJBRDYAC9BRURJWUIJLUWXNQ'
            b'OSVVLFEBROMJCGVYZWIPOYFQRBUUNJLIGPVDLADFLZJGZBLEBEQEUDUZOIFFZLZR'
            b'XCPQVMIARFLZRIOFFEHVFJZXFQFLCJSEXRPUKGMWBMGXEHIEZKOKGH9JXAUXACEB'
            b'LXKLZT9P9NJGXRZWZJAZCNKR9CHRRNCFOLBCSZXKXOIGZHZSTDKTHOWZTEXWOIZL'
            b'PEGPKTBRENSCOYBQJSZQ9XPNRORQOWMPGBXKSSLDIUVEAJODEUZJKZE9MBVTQXWF'
            b'XXXOG9QGDWMNZEWVWVDZWIFKSWZIDNBYEJP9VBKQNOJEXVPZQTHVUCSK9QCMEPUS'
            b'9Y9FQPWEACAEBIQSVPJEL9ZBSETINIYMSPIXLADSHTDYRAYUTMXDCABIUUETMNLU'
            b'RELTPAGEDNMQZALFWOPAI9WUFOSUTOFUUWFAFRFVYOPITBVSG9IBVNJCOITYMTCC'
            b'IJIZWVPYGQETESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIYYZ9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99C99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 2 of 2
          TryteString(
            b'SWHZKSNCOQXPCGRTYJPUGKLBNEJFXASKY9XAUROGDAO9QQLIVRZQDJDTPLNTBGUU'
            b'FGELJPSGUMGPIUNCCTQEFU9UZIJJYJXCYWRADRHHXKBEDG9HTCHJHXUJRKMIUFOS'
            b'KDGMI9QPCYQSWDCUYKQQEONJEKYWELG9MSNBHRHILGSSKMRCQJBOGNYBKEMTOFEU'
            b'BEOUBD9ULP9PHWYKXEQNDMUR9BGDGPEUFRFRGNGFJFPYQXABSDALTYKL9SM9VVQC'
            b'OHY9AS99EYWSHUNEQVVNLS9CNPEVMPOKMWQYFPGTNJBZCDWYPFWSBKZOYXNNVMPO'
            b'DEHMHNIYZCHIEDEAB9TLFOWVHF99GVRWUZWSN9IQOKWIXERKRQETZS9ZJJSQRLLP'
            b'QXEWNMFVVBWOIK9MBYCEGUJ9HJRIIMBVJNGXMGPGDLOLYWFVQNOKTFZRBJSSBJTE'
            b'TGAIUGZOYQOFTVBKAQY9SSJWJXXYAUVUQWPXVCISFSDSHDQCPVMG9GVDAO9GIMDH'
            b'ZWJOKSUEUFHBGSCZKNBTZWJXSFYNJSBSEL9UMZBAZRGYCEHOSJBMKMPMNXKEVTMU'
            b'DEFWBIKOXUSBNPTNECGVLYSOGUDJDPHYFADXRAOLQXJSJDANINJEOMCFAWWITREG'
            b'CDF9OZ9ZKHPJZJNMOVGX9OKQBSGVZYWKNOPVJEOZEI9BPE9GCUEQVAHSBBRBGQTE'
            b'XVZCSL9ECOWPOWZCVSCBOUSNQMTJIEKHXL9NCPRMLRNKQEHYJCLRHGZKFNBJIPKS'
            b'KPRFTSKFJULTBTXFDQHWUYOSDQBHPAINVEPKCCHJDTZOJIGJZOF9AEQDBKCZSZMI'
            b'WUUVHVGAFKALGITVQQKBAHKCIFSVMVZ9UDQABVIANTBUQOFBIXQBWB9KKQOVJZNV'
            b'BEDAZKUTRNKGJQWMTEKV9KGCIBRDCBAPKSTMCZGUV9HTAABQDKGQBCRFNXBMZRTH'
            b'F9MO9GAGQDYDVLOFMDE9QQZYR9GDSBLUVVMKMCZIMDPNCVLGDKBACWQJRWOQNKBT'
            b'SDJFKQMKTVKXVNAHRHZALJGVAMXWJYRAKTEJFXAHBQGSYWWQVECQYPXVFWILNFZK'
            b'GGRIFCJBSIZRDJXRJHSURPCZKOWKLFRUMVENEGMNKUAOGVUECDSGAZNQZKBJDJPV'
            b'BXLOTID9QLMFNGIWKAAIQTJJROSZBXPQRXAUCV99OGCEOTQCJ9II9ASZL9XGNSVU'
            b'XVKPXYOJMF9PX9GSLEROR9FXVQ9MLEMEW9IWNWBNVAYXZ9ZETTDSMLGZAKHE9IUJ'
            b'BFUHXW9KWCNZOZCCTFGBGWSDAQGGSPSQHOMUVJMLWBDAKYQZMWPQLLYAGUMOVMVL'
            b'FD9TO9OUBTVUHHUNSFSATSEGBFVGDZUBMTWWFDPSQVCUFRVKHYYPDWRPNSKXRFTV'
            b'EIBVZNGUZRQCPXVKBPKQDDLEBWIEBIPTEJIYFHBXCUVCCKTKEJAYRZCKAXLMELIN'
            b'WUZHG9JFBSBAKHIXMWHUWUFHFNLXNO9GKINYKRTCNN99PHPHO9MJAGUYZAPNSPWU'
            b'Z99E9BEADKETLOALWNANYMHSLLQSBS9YTYVJKTVWFUVS9MFOWCHLEUUFUWTYGLZX'
            b'FDUXVABTVFXFPUEPIUEIAVSZSSZQJTHNGKBJXADRHVTIBERILMCCGWUITYQHGEEG'
            b'WIZZOII9B9EVVVFJNEYEWH9ZVOJGHKVPYKDEZZSPBAOBQGGWPWXTCKSLSHJQYCDH'
            b'AYIQ9QVSQFPXZDBYSJJKSNTRXORHLPVOYVMIGALRPXYWQWSJPPFTJCYXAATLBFNS'
            b'GVXKFJXHYTILNOQZROUCDUTWOMGYBVTWPKJY9RVKKWQQMVCHJEUBELJDKJPGYLXU'
            b'XATNOIJHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHVHE9EEB'
            b'QYSNWECSPAJHGLTEUCXALBRVGXFENUCOONSUFZLHTLVQNPDZDIVDQHWVLDEDPFQL'
            b'JZWF9GFZMPZXFVEQECLUZBBFVSAPEXJLKKOMXEPHZAKP9WYTGQOML9FQSBMSFLOG'
            b'RLFQKUCUWFX9DNAOZSSKBUV9IBVIRNUWYBKJVKLJ9PPNLGJATKDCAGVFIVPXRABH'
            b'ZVZACJIG9WOKKLFCRDSMTWSCYHOZEEXRIMPQBXVXQAYKZIADSM9GUBICGKGQYNHK'
            b'VYOZFRVCHNMTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVW999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99B99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 1 of 2
          TryteString(
            b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWG'
            b'HMNUXBJTOBKZFDNJEMAOPPLR9OOQJCDVO9XSCYQJQVTXQDYWQEBIXKDZAFWINAHJ'
            b'ELJTDPVMUEWSVCJA9ONDYBNANWCGLBQMEMTBFDMWLCMQHGJLGYDQGIMLSNQHBGSV'
            b'TDZSGNQAL9OHRAPDKYSVBTNYRUUBNEEAINJMOVOHOWXUAIEDAIQDESQFCKJELHAV'
            b'ODSMXMKEHTDKCDIWWISXSAHQE9TJTLJZGXIABHU9CUACMLVSSYV9UJREPWFVYWWX'
            b'PYYJRP9DOEKNDMBSBKKHIFMPXZXIJERXRZVBVDBYNZBBCCOSEDOLDGSNQK99HIYS'
            b'WNYYEBLRT9MADLXLLZJOSZCFWAVZY9XUPNZUVOSKBMKXXJNRKDBOSGUGME9QNBMH'
            b'IWXWXPEEUVQAQVUXDJGMJOBXG9VJBWPRQRCCQSNBEHTLGOKJVYEPQOJO9QIZLYAV'
            b'LCKVXKEKRGBSZJAC9KTSSNMDQGKCLPZDJAQ9PBQMLUONVVFAWTMREGFXJMRRGL9M'
            b'KNPOZGOYRPDCYEJCYJUN9HYNSNHXARMRJVXBUHOP9K9BIIEYGSHBUESKTAOQOEAN'
            b'EAIHYHVGSVNPXWRBTJAMKMWEQOSYEWXLSRYVOSTMPOGYNPDNFLOICXVHYBDHSXVR'
            b'KVWNVZOZQDOITZWICSYEW9RGCPPUJYVIYVTSZILYENYUYUGDSGWVYWRMZJNCTTPV'
            b'WDWXAPVZQQKI9CGEQPBFPCLGDDEGBUUTISNCMJXQCTUNKQTLCATNOIRPMEUQBQTH'
            b'HQYRGDLZEUZBALNQDXJYZBVXDPLVOVVAUCQSCGRTUJRBBNRV9ORETTGFIXBBBVOP'
            b'FHPKGPKVBYFTZMWUVZYVWWSDKQVONMPLLQTV9IZUWLUWZNLCVJNPMG9CMXQG9D9W'
            b'YCANBRMYV9DU9FMJT9JHT9RWCGLHFCODXJVFQBLTKJWVNVGSUHNWLHNPLZDSWDMD'
            b'VQTLVCSVFJJTIQZFAPCXJWDAXWJKJVOKHALCQQTIXABPFXPUFK9IKXYUGMPXNSQC'
            b'JDVETOVEX9LXYLXWRW9PFEYJCUJHLUB9NXTUGLIQMDGPDPSJTWDYEWXQAICLN9BT'
            b'GNBJWLVAXZGNCYXGHBMRUVVYTJGH9XDGSZHQDYKFGMOWORSFDFBLJHBRTXRSEBAL'
            b'CJIJTQJYDZZKWZGVAPFVKVEOXGYRLMBSPFHUIJZZFMFVOTLPUWSYZCWFZMAALHRG'
            b'SYSXSMOHWARYZZVIAKXAHGY9SROWPVFACXXLQEXXOJCKXRRZHBZXJIBWQMMZTRDF'
            b'YQBSBBZQQXGCAAECMQINHJRBSGOYPCGWKPWCHBKOJTIGDASZFGONTUGDSOOLEMGO'
            b'EBFCZZJZSCGXPHXHB9WGMMFVUTCHDBSAMYTECQZWGCXAWTCTIBZHQVUAIBPZHBBT'
            b'ZAERYU9XAMKBHCHGZISSPOWJIRZTAXDHMAYBPXOXWDIUDHNBTFJNVHHJO9AWAEC9'
            b'UPRRFJLNGKTXJXFDGODDOPMGLALRIJBVIFLQTYQPKCKCRBYPBYGUUFJGJFVCOURN'
            b'KCGNTQNNKHDDPIVZHCJSLDUYHVPAX9YJOFTTFSKFHTOOQQRCPYZKTDVCUZGBOBZK'
            b'LVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VZZGUNUCKOYLYXOVHMGULWGSRCGX'
            b'ZLJVNIMZBLFOJJKOTUREMBXYOZXDUP9ROUVYOSJBGGFZMIFTKHJHHJGZJNOYQWFZ'
            b'AHLJWWDDFQQAMEGJUEUSIWOHKFJWRXRSJWYPGIGZGMFNAIDGDOUUQUVHJZQPJMLC'
            b'GKGADXAXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVGXBBPBE'
            b'BDVGZDBWMDMLPXYJBBRNOMKGR9TSVUXSRYXQTCTYLFQORMIGDKBJLNLCQXACVCBJ'
            b'GVWRJNYPCKOAILPLMWBYKDLDXLIZMZFWDXUWDEGDUURQGMJNUGJXDXYJGKOTQBGC'
            b'HATROPKEN9YTXDUOCMXPGHPDANTJFRRVEVBFVCNTWNMMOVAVKBNSJIWWBVHBMCSU'
            b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHV'
            b'ITGYROZZNQPTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVWN99999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Spend transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUETWO9DONTUSEINPRODUCTION99999XYYNXZLKBYNFPXA9'
            b'RUGZVEGVPLLFJEM9ZZOUINE9ONOWOB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99999999999D99999999PNTRTNQJVPM9LE9XJLX'
            b'YPUNOHQTOPTXDKJRPBLBCRIJPGPANCHVKGTPBRGHOVTLHVFPJKFRMZJWTUDNYC99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_pass_inputs_implicit_with_change(self):
    """
    Preparing a bundle that finds inputs to use automatically, change
    address needed.
    """
    # To keep the unit test focused, we will mock the ``getInputs``
    #   command that ``prepareTransfer`` calls internally.
    #
    #   References:
    #     - :py:class:`iota.commands.extended.prepare_transfer.PrepareTransferCommand`
    #     - :py:class:`iota.commands.extended.get_inputs.GetInputsCommand`
    mock_get_inputs = Mock(return_value={
      'inputs': [
        {
          'address':
            Address(
              trytes =
                b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
                b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

              balance   = 86,
              key_index = 4,
            ),

          'balance':  86,
          'keyIndex': 4,
        },
      ],

      'totalBalance': 86,
    })

    mock_signature_fragment_generator = MockSignatureFragmentGenerator([
      TryteString(
        b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWGHM'
        b'NUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHLYUHQ'
        b'P99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGBPNNFZD'
        b'WRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9KFEEXXVN'
        b'EXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFCYSWBUJYVGE'
        b'UXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMMKQVDYAZ9SVFI'
        b'UUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9OIZWLNINYRLGJQC'
        b'UBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNENNWXLF9ZVHZIDWBLF'
        b'KVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPWNOWQDMHPKOJTYYECKN'
        b'GCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJDTPJDOASLVRXFNJJHXQTK'
        b'MKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHFXPUIEEA9HPHJTCJJWZPUHK'
        b'AAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLFNTVDMYPRCYPQR9MNBBAMGOJX'
        b'PRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHFXTYBCMLRVHWNQDSQUIHHTAUTZC'
        b'JFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBUBUIO9TRXQLAYHDDSXGJ9NB9FKMUU'
        b'US9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXHGHYCV9VDCXHGTCOOUEXIITVKHXCSUS'
        b'OIRTMEAKMTYZCMAWURNX9JOVDICICKHXQYBXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMF'
        b'YQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLWRMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9C'
        b'CMCO9TCKLKSJEAMFKQMXEYETISVEYDOSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFC'
        b'SULGRHPONWJDVWT9UELEKEPQEAFKDLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYK'
        b'ADLUCRNEPAIWYSX9LT9QWQRKU9WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPD'
        b'QMMOOOCKYVEZJ9ZJQRJHNCKRFDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSX'
        b'DTOEEKLF9WWZGLSGIOQZWCJJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVM'
        b'XJGCSQSGYFLGYDZUH9EHUDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCU'
        b'Y9FFBTPSTJFEFROCEXFFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKN'
        b'OGCD9OPQ9WDMKRPIUNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPY'
        b'ZKTDVCUZGBOBZKLVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJH'
        b'UBWYXGECBPQOPOHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLR'
        b'JYTXTBANBZWFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQ'
        b'FLUXOFUNVDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVG'
        b'XBBPBEBDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICN'
        b'C9BT9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSK'
        b'NHJHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
        b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHVIT'
        b'GYROZZNQP'
      ),

      TryteString(
        b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKXFI'
        b'NPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPTYFFM'
        b'9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSCKCYPIE'
        b'AFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFLKOIRUEJU'
        b'PZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTLTKWYXHVAFU'
        b'VVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWLOFNXBDCT9JLA'
        b'9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFDGNJII9LSW9IQKV'
        b'99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKKXCKXLRGDT9NXIGQV'
        b'ZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEHKHVATNVWX9ZGTISUKP'
        b'KTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZHUDDUFJVQPBYNJREQ99U'
        b'HOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXUFYILOPOCJFVVEJEJN9ULGX'
        b'IABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLOFNA9KNERURWJPROBBXEWICDK'
        b'KCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXCGCLFETWOZYIUZVJVYVB9YGXSSD'
        b'XYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEHRIVMICUQPUUFRFCBF9NUUWSQBTVI'
        b'YFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUEVOFDMQLDESMQDPQUSOOKZJ9QLXTAFP'
        b'XXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFOEKQDEIWGXZPBXSOGGQFILUJTKDLWVKPV'
        b'ISU9QOATYVKJHLDLOKROZNFAGS9CICXXIUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTE'
        b'NYEWQWEHP9DDWOUJDWSTSOGYQPALFMKCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSU'
        b'Y9QTABAJOJAAMGVBCSLAAOBXZOJZLIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYC'
        b'HOVTLGDICIWTCIGNRHLBZBVSXMPBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCS'
        b'SCVPNBXEDCMHKFVDMHJTSP9JI9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFY'
        b'GVSNOYAEC9NWTCVEJBSXLYWTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQ'
        b'GVXENKIGW9XZTPBAIMZLHWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVC'
        b'FAXJKJNUC9NQHPEXWIOHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXA'
        b'JGFPHKAQDCRTKQBXPZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9U'
        b'SZRWUPZAASWDBXOVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQ'
        b'OA9DOY9RXZNWBFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFM'
        b'GRPGSCFRZINEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYND'
        b'TURLBPXGNQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHV'
        b'HE9EEBQYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGY'
        b'VPNSDOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLL'
        b'ASNQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
        b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZKW'
        b'FQNAJTO99'
      ),
    ])

    with patch(
        'iota.transaction.ProposedBundle._create_signature_fragment_generator',
        Mock(return_value=mock_signature_fragment_generator),
    ):
      with patch(
          'iota.commands.extended.get_inputs.GetInputsCommand._execute',
          mock_get_inputs,
      ):
        response = self.command(
          seed = Seed(
            b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
            b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
          ),

          transfers = [
            ProposedTransaction(
              value = 42,
              address = Address(
                b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
                b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
              ),
            ),
          ],

          change_address =
            Address(
              b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJ'
              b'RBOSBIMNTGDYKUDYYFJFGZOHORYSQPCWJRKHIOVIY',
            ),
        )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # Change transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIYQB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99C99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 2 of 2
          TryteString(
            b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKX'
            b'FINPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPT'
            b'YFFM9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSC'
            b'KCYPIEAFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFL'
            b'KOIRUEJUPZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTL'
            b'TKWYXHVAFUVVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWL'
            b'OFNXBDCT9JLA9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFD'
            b'GNJII9LSW9IQKV99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKK'
            b'XCKXLRGDT9NXIGQVZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEH'
            b'KHVATNVWX9ZGTISUKPKTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZ'
            b'HUDDUFJVQPBYNJREQ99UHOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXU'
            b'FYILOPOCJFVVEJEJN9ULGXIABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLO'
            b'FNA9KNERURWJPROBBXEWICDKKCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXC'
            b'GCLFETWOZYIUZVJVYVB9YGXSSDXYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEH'
            b'RIVMICUQPUUFRFCBF9NUUWSQBTVIYFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUE'
            b'VOFDMQLDESMQDPQUSOOKZJ9QLXTAFPXXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFO'
            b'EKQDEIWGXZPBXSOGGQFILUJTKDLWVKPVISU9QOATYVKJHLDLOKROZNFAGS9CICXX'
            b'IUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTENYEWQWEHP9DDWOUJDWSTSOGYQPALFM'
            b'KCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSUY9QTABAJOJAAMGVBCSLAAOBXZOJZ'
            b'LIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYCHOVTLGDICIWTCIGNRHLBZBVSXM'
            b'PBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCSSCVPNBXEDCMHKFVDMHJTSP9J'
            b'I9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFYGVSNOYAEC9NWTCVEJBSXLY'
            b'WTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQGVXENKIGW9XZTPBAIMZL'
            b'HWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVCFAXJKJNUC9NQHPEXWI'
            b'OHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXAJGFPHKAQDCRTKQBX'
            b'PZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9USZRWUPZAASWDBX'
            b'OVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQOA9DOY9RXZNW'
            b'BFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFMGRPGSCFRZI'
            b'NEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYNDTURLBPXG'
            b'NQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHVHE9EEB'
            b'QYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGYVPNS'
            b'DOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLLAS'
            b'NQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
            b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZ'
            b'KWFQNAJTO99TESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVW999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99B99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 1 of 2
          TryteString(
            b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWG'
            b'HMNUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHL'
            b'YUHQP99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGB'
            b'PNNFZDWRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9'
            b'KFEEXXVNEXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFC'
            b'YSWBUJYVGEUXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMM'
            b'KQVDYAZ9SVFIUUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9O'
            b'IZWLNINYRLGJQCUBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNEN'
            b'NWXLF9ZVHZIDWBLFKVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPW'
            b'NOWQDMHPKOJTYYECKNGCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJD'
            b'TPJDOASLVRXFNJJHXQTKMKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHF'
            b'XPUIEEA9HPHJTCJJWZPUHKAAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLF'
            b'NTVDMYPRCYPQR9MNBBAMGOJXPRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHF'
            b'XTYBCMLRVHWNQDSQUIHHTAUTZCJFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBU'
            b'BUIO9TRXQLAYHDDSXGJ9NB9FKMUUUS9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXH'
            b'GHYCV9VDCXHGTCOOUEXIITVKHXCSUSOIRTMEAKMTYZCMAWURNX9JOVDICICKHXQY'
            b'BXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMFYQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLW'
            b'RMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9CCMCO9TCKLKSJEAMFKQMXEYETISVEYD'
            b'OSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFCSULGRHPONWJDVWT9UELEKEPQEAFK'
            b'DLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYKADLUCRNEPAIWYSX9LT9QWQRKU9'
            b'WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPDQMMOOOCKYVEZJ9ZJQRJHNCKR'
            b'FDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSXDTOEEKLF9WWZGLSGIOQZWC'
            b'JJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVMXJGCSQSGYFLGYDZUH9EH'
            b'UDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCUY9FFBTPSTJFEFROCEX'
            b'FFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKNOGCD9OPQ9WDMKRPI'
            b'UNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPYZKTDVCUZGBOBZK'
            b'LVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJHUBWYXGECBPQO'
            b'POHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLRJYTXTBANBZ'
            b'WFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQFLUXOFUN'
            b'VDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVGXBBPBE'
            b'BDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICNC9BT'
            b'9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSKNH'
            b'JHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
            b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHV'
            b'ITGYROZZNQPTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVWVX9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Spend transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUETWO9DONTUSEINPRODUCTION99999XYYNXZLKBYNFPXA9'
            b'RUGZVEGVPLLFJEM9ZZOUINE9ONOWOB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99999999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_fail_inputs_implicit_insufficient(self):
    """
    Account's total balance is not enough to cover spend amount.
    """
    # To keep the unit test focused, we will mock the ``getInputs``
    #   command that ``prepareTransfer`` calls internally.
    #
    #   References:
    #     - :py:class:`iota.commands.extended.prepare_transfer.PrepareTransferCommand`
    #     - :py:class:`iota.commands.extended.get_inputs.GetInputsCommand`
    mock_get_inputs = Mock(side_effect=BadApiResponse)

    with patch(
        'iota.commands.extended.get_inputs.GetInputsCommand._execute',
        mock_get_inputs,
    ):
      with self.assertRaises(BadApiResponse):
        self.command(
          seed = Seed(
            b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
            b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
          ),

          transfers = [
            ProposedTransaction(
              value = 42,
              address = Address(
                b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
                b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
              ),
            ),
          ],
        )

  def test_pass_change_address_auto_generated(self):
    """
    Preparing a bundle with an auto-generated change address.
    """
    # To keep the unit test focused, we will mock the ``getNewAddresses``
    #   command that ``prepareTransfer`` calls internally.
    #
    #   References:
    #     - :py:class:`iota.commands.extended.prepare_transfer.PrepareTransferCommand`
    #     - :py:class:`iota.commands.extended.get_new_addresses.GetNewAddressesCommand`
    mock_get_new_addresses_command = Mock(return_value=[
      Address(
        trytes =
          b'TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJ'
          b'RBOSBIMNTGDYKUDYYFJFGZOHORYSQPCWJRKHIOVIY',

        key_index = 5,
      ),
    ])

    self.adapter.seed_response('getBalances', {
      'balances': [86],
      'duration': '1',

      'milestone':
        'TESTVALUE9DONTUSEINPRODUCTION99999ZNIUXU'
        'FIVFBBYQHFYZYIEEWZL9VPMMKIIYTEZRRHXJXKIKF',
    })

    mock_signature_fragment_generator = MockSignatureFragmentGenerator([
      TryteString(
        b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWGHM'
        b'NUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHLYUHQ'
        b'P99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGBPNNFZD'
        b'WRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9KFEEXXVN'
        b'EXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFCYSWBUJYVGE'
        b'UXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMMKQVDYAZ9SVFI'
        b'UUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9OIZWLNINYRLGJQC'
        b'UBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNENNWXLF9ZVHZIDWBLF'
        b'KVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPWNOWQDMHPKOJTYYECKN'
        b'GCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJDTPJDOASLVRXFNJJHXQTK'
        b'MKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHFXPUIEEA9HPHJTCJJWZPUHK'
        b'AAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLFNTVDMYPRCYPQR9MNBBAMGOJX'
        b'PRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHFXTYBCMLRVHWNQDSQUIHHTAUTZC'
        b'JFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBUBUIO9TRXQLAYHDDSXGJ9NB9FKMUU'
        b'US9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXHGHYCV9VDCXHGTCOOUEXIITVKHXCSUS'
        b'OIRTMEAKMTYZCMAWURNX9JOVDICICKHXQYBXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMF'
        b'YQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLWRMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9C'
        b'CMCO9TCKLKSJEAMFKQMXEYETISVEYDOSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFC'
        b'SULGRHPONWJDVWT9UELEKEPQEAFKDLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYK'
        b'ADLUCRNEPAIWYSX9LT9QWQRKU9WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPD'
        b'QMMOOOCKYVEZJ9ZJQRJHNCKRFDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSX'
        b'DTOEEKLF9WWZGLSGIOQZWCJJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVM'
        b'XJGCSQSGYFLGYDZUH9EHUDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCU'
        b'Y9FFBTPSTJFEFROCEXFFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKN'
        b'OGCD9OPQ9WDMKRPIUNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPY'
        b'ZKTDVCUZGBOBZKLVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJH'
        b'UBWYXGECBPQOPOHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLR'
        b'JYTXTBANBZWFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQ'
        b'FLUXOFUNVDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVG'
        b'XBBPBEBDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICN'
        b'C9BT9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSK'
        b'NHJHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
        b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHVIT'
        b'GYROZZNQP'
      ),

      TryteString(
        b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKXFI'
        b'NPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPTYFFM'
        b'9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSCKCYPIE'
        b'AFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFLKOIRUEJU'
        b'PZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTLTKWYXHVAFU'
        b'VVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWLOFNXBDCT9JLA'
        b'9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFDGNJII9LSW9IQKV'
        b'99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKKXCKXLRGDT9NXIGQV'
        b'ZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEHKHVATNVWX9ZGTISUKP'
        b'KTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZHUDDUFJVQPBYNJREQ99U'
        b'HOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXUFYILOPOCJFVVEJEJN9ULGX'
        b'IABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLOFNA9KNERURWJPROBBXEWICDK'
        b'KCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXCGCLFETWOZYIUZVJVYVB9YGXSSD'
        b'XYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEHRIVMICUQPUUFRFCBF9NUUWSQBTVI'
        b'YFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUEVOFDMQLDESMQDPQUSOOKZJ9QLXTAFP'
        b'XXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFOEKQDEIWGXZPBXSOGGQFILUJTKDLWVKPV'
        b'ISU9QOATYVKJHLDLOKROZNFAGS9CICXXIUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTE'
        b'NYEWQWEHP9DDWOUJDWSTSOGYQPALFMKCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSU'
        b'Y9QTABAJOJAAMGVBCSLAAOBXZOJZLIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYC'
        b'HOVTLGDICIWTCIGNRHLBZBVSXMPBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCS'
        b'SCVPNBXEDCMHKFVDMHJTSP9JI9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFY'
        b'GVSNOYAEC9NWTCVEJBSXLYWTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQ'
        b'GVXENKIGW9XZTPBAIMZLHWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVC'
        b'FAXJKJNUC9NQHPEXWIOHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXA'
        b'JGFPHKAQDCRTKQBXPZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9U'
        b'SZRWUPZAASWDBXOVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQ'
        b'OA9DOY9RXZNWBFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFM'
        b'GRPGSCFRZINEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYND'
        b'TURLBPXGNQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHV'
        b'HE9EEBQYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGY'
        b'VPNSDOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLL'
        b'ASNQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
        b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZKW'
        b'FQNAJTO99'
      ),
    ])

    with patch(
        'iota.transaction.ProposedBundle._create_signature_fragment_generator',
        Mock(return_value=mock_signature_fragment_generator),
    ):
      with patch(
        'iota.commands.extended.get_new_addresses.GetNewAddressesCommand._execute',
        mock_get_new_addresses_command,
      ):
        response = self.command(
          seed = Seed(
            b'TESTVALUEONE9DONTUSEINPRODUCTION99999C9V'
            b'C9RHFCQAIGSFICL9HIY9ZEUATFVHFGAEUHSECGQAK'
          ),

          transfers = [
            ProposedTransaction(
              value = 42,
              address = Address(
                b'TESTVALUETWO9DONTUSEINPRODUCTION99999XYY'
                b'NXZLKBYNFPXA9RUGZVEGVPLLFJEM9ZZOUINE9ONOW'
              ),
            ),
          ],

          inputs = [
            Address(
              trytes =
                b'TESTVALUETHREE9DONTUSEINPRODUCTION99999N'
                b'UMQE9RGHNRRSKKAOSD9WEYBHIUM9LWUWKEFSQOCVW',

              key_index = 4,
            ),
          ],
        )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # Change transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUEFOUR9DONTUSEINPRODUCTION99999WJRBOSBIMNTGDYK'
            b'UDYYFJFGZOHORYSQPCWJRKHIOVIYQB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99C99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 2 of 2
          TryteString(
            b'ZOJNUMZOBEHLYDSDAVZKXHF9MAHAJICBMJTZZHTQTCACVQAUSSCFUMGCSJTONNKX'
            b'FINPOAXQIKSJ9GUV9GXM9KYDCDWUHULIJMSKMOLDZBYE9FTGFMKLODKHFF9YUCPT'
            b'YFFM9EDCJDCKRFLZUHGGYNYFJLBFWXCIUF9HMGUQKPUCJ9OQ99FXHSUSRRBEUSSC'
            b'KCYPIEAFZJQNXEUYWLEXKZWLRINBEGAZTJMYTUEQTTORMIIQASISHSHZDQJXANFL'
            b'KOIRUEJUPZZHUJFWHEXFIZ9OU99SQLDDNLARDFPGYSCMXQCMGPRB9QLM99QUBLTL'
            b'TKWYXHVAFUVVAMHEYCCNVEITSPVQWMSEIZJSLPWNGWISKWQNXCNRNOIGRGUHGYWL'
            b'OFNXBDCT9JLA9CEKW9BFGOESKGOQLJBTLUMOICBEZDHCR9SZCJUZVXIEAVITFJFD'
            b'GNJII9LSW9IQKV99UJWWAACGIRPCZUENXGILUXCMJIGW9REUNA99MWSANWL9KVKK'
            b'XCKXLRGDT9NXIGQVZWG9NBQPOQKEEET9ZUSENFPGFDNNHGBITCPASGHOPBNYKKEH'
            b'KHVATNVWX9ZGTISUKPKTMWMPCGVVJSGMRJWNFICSFUAVAHIZWA9PDOIXFJGWCPTZ'
            b'HUDDUFJVQPBYNJREQ99UHOESTT9FELDMVK9VHZYPRVOWEW9NXTCYDCIMT9UIWGXU'
            b'FYILOPOCJFVVEJEJN9ULGXIABFJWWRKAD9NHZBULMWUKESZLCPRQVVKWOHEWSTLO'
            b'FNA9KNERURWJPROBBXEWICDKKCQXWYMJUCQLWEUPFXRSNMIJWQUEJUNIKDYJILXC'
            b'GCLFETWOZYIUZVJVYVB9YGXSSDXYXSJXTOQZ9CCCAKMCNNKQCYEDGSGTBICCOGEH'
            b'RIVMICUQPUUFRFCBF9NUUWSQBTVIYFVWAASTQJZFDDWWUUIHPKTIIVAGGIEQCZUE'
            b'VOFDMQLDESMQDPQUSOOKZJ9QLXTAFPXXILFHFUIFJTKSEHXXZBPTZUGLYUZNORFO'
            b'EKQDEIWGXZPBXSOGGQFILUJTKDLWVKPVISU9QOATYVKJHLDLOKROZNFAGS9CICXX'
            b'IUQQVLLRPPPDYJVSCW9OWIHKADCVSKPWTENYEWQWEHP9DDWOUJDWSTSOGYQPALFM'
            b'KCTUGLSXHNYETTMYTS999SYQVQSPHQPKRJSUY9QTABAJOJAAMGVBCSLAAOBXZOJZ'
            b'LIFXUYOVXBKHPFVTKKGSIHUXMBDTMGNVL9NXYCHOVTLGDICIWTCIGNRHLBZBVSXM'
            b'PBFAWIXPCDJWNDUFHUVLBSPBWICZNYIUJPRRTOCSSCVPNBXEDCMHKFVDMHJTSP9J'
            b'I9BXTD9ZILEEOCBMHCQRRDNL9EUKJGJ9MPQGQU9ZFYGVSNOYAEC9NWTCVEJBSXLY'
            b'WTUPMXNAAWXSBIAJYSGYHGLYOMAHFTYMICZRDZTQXHAQGVXENKIGW9XZTPBAIMZL'
            b'HWAJCGY9ZDNQOTGDRCTXSJCEJVTTMVRYYKWAFYSV9WVEVCFAXJKJNUC9NQHPEXWI'
            b'OHOJQEXJNLEW9GLO9AJCJXIEXDONOGKXFJ9OXXXETUEHLBXAJGFPHKAQDCRTKQBX'
            b'PZYQZBQODTVIBUTSAEXMBFBMTAXOQZCOHWEWRJEKNKHZXXSO9USZRWUPZAASWDBX'
            b'OVAEGSAGYDIOZWSSEAIQVRWFDSOXSRRRQHRCWDJWZXXJOGPZRLKQOA9DOY9RXZNW'
            b'BFJTKUOVRRQNSDUOFGCUQNHOBMJSFQZXVBPHHBRRIXZNLXAH9P9EFMGRPGSCFRZI'
            b'NEPOQPXPKHTSRJWARXRGJGYMTPUKQISLV9GUC9VTJLOISKGUZCTZEYNDTURLBPXG'
            b'NQLVXHAHUVNGIHVMZOHLEUBDTRFXFXXVRYBRUF9ULNMSZZOZBYDJUWTMHVHE9EEB'
            b'QYSNWECSPAJHGLTEUCXALBRVTKMWSWCBPUMZFVSEEFIHBAGJVVQV9QLFEGGYVPNS'
            b'DOBZEQGLEFLCQVPDJA9MQDRHYNVZVNTYNJ9GJCXKED9NEWTD9RVMNA9HOHUBLLAS'
            b'NQSDLDZKOMFOEGBJZPYVYZCVHYFEGSVEHSWV9WAGMEQIUDZQZUACWYQLTD9LHBVK'
            b'KNXXXDWQUWRJKTCDP9CEJOHLLPTWCIKKHHIFAFFDVMFZR9A9LYVMTQAPAXAVPJOZ'
            b'KWFQNAJTO99TESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVW999999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99B99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Input #1, Part 1 of 2
          TryteString(
            b'OGTAZHXTC9FFCADHPLXKNQPKBWWOJGDCEKSHUPGLOFGXRNDRUWGKN9TYYKWVEWWG'
            b'HMNUXBJTOBKZFDNJEZUKCKWGUHVSU9ZJYAVSQSOFDCOIEP9LCXYLTEFMCYUJAAHL'
            b'YUHQP99S9XRWHXHRPZCWHDMIDYW9OQAWUPTFMBTJGDCWRVNVRDPIWISVYNUDWUGB'
            b'PNNFZDWRVZ9FGAVSEWFXRXGGLXJTPJTJLC9JYHMFBKYAUJRAMHQHKUUZHRWZIVC9'
            b'KFEEXXVNEXJRYUSFV9PEPFUDCNRRTSCZXSTUEGJKDV9UCYNZSBRDYGOKFGYKWVFC'
            b'YSWBUJYVGEUXWTDGPWTWURH9RKEZRFCUUBFBPKSFONMDXWGYKWAUWVUOQVBIGQMM'
            b'KQVDYAZ9SVFIUUNMHOJGRQVXZGIIPKVNNBKABGKZLRNFK9KSIHTCGYPVCWYGDS9O'
            b'IZWLNINYRLGJQCUBWYMAVDWFAURLALQPMRMFRAZCMCPOWM99SGBVEZPAFAXHXNEN'
            b'NWXLF9ZVHZIDWBLFKVWKBUYNBXOXTVPDWAGZXIOMDAEKNMRFGZVIGIFOSHGMPIPW'
            b'NOWQDMHPKOJTYYECKNGCDDTJVALGPZSX9IH9LEGQSDACLBWKNXUW9BAZSHAISUJD'
            b'TPJDOASLVRXFNJJHXQTKMKZUZIMJFPOKHEQXSCJQH9JPRNZHDVVZKWTHWWFNFMHF'
            b'XPUIEEA9HPHJTCJJWZPUHKAAWJQQSAIF9HRETYYPXAZ9YCFJRCXTGCOLJQA9HDLF'
            b'NTVDMYPRCYPQR9MNBBAMGOJXPRFCUSIIZN9VROZDPMOKZBCILKGB9EPCXOYWLPHF'
            b'XTYBCMLRVHWNQDSQUIHHTAUTZCJFQ9CO9GTONKYKMDBSREZC9SUBHYK9JDOBYDBU'
            b'BUIO9TRXQLAYHDDSXGJ9NB9FKMUUUS9GANWVMQLIHX9MPJGLTAOMCZTQYDYVOWXH'
            b'GHYCV9VDCXHGTCOOUEXIITVKHXCSUSOIRTMEAKMTYZCMAWURNX9JOVDICICKHXQY'
            b'BXKWTXWXBZVZWRIDC9YCZVSKYIKJYYMFYQRTWBNJHWXRL9JFSZAXJYYTGDYLTHLW'
            b'RMBUEG9QTGNRPVTBGRYFPEJQSIWTLPGV9CCMCO9TCKLKSJEAMFKQMXEYETISVEYD'
            b'OSCRZ99RFDPUQPHMQ9NVRUBXITDGFZCYQNFCSULGRHPONWJDVWT9UELEKEPQEAFK'
            b'DLDNYPABC9GUASVFJBFZF9Z9CHIUNLJWHKGDYKADLUCRNEPAIWYSX9LT9QWQRKU9'
            b'WEVDPKSTSA9PPEVNTBNLN9ZOPETINXGKA9DCOHPDQMMOOOCKYVEZJ9ZJQRJHNCKR'
            b'FDRPHUVPGVGQYKZBLOILZTPIX9MIBKTXOJKVAYRLSXDTOEEKLF9WWZGLSGIOQZWC'
            b'JJHSBTXYWRDYVEQTCNUENYWDRLZZIVTGCXEAJDRY9OVMXJGCSQSGYFLGYDZUH9EH'
            b'UDQTCXLSDPMNDYQRZYRXYXKY9GIYOSIDQPXXHKJKDQLSCUY9FFBTPSTJFEFROCEX'
            b'FFYTFYHQROAVTYKQOCOQQWBN9RKJ9JJEURKTVOECYRITTYKNOGCD9OPQ9WDMKRPI'
            b'UNRAVUSLFMC9WZWHSESGLDUYHVPAX9YJOFTTFSKFHTOOQQRCPYZKTDVCUZGBOBZK'
            b'LVBVBCWTUS9XOBJADZYN9TMLGCKXEXFEQFQ9VDFKWVEWV9WGXPJHUBWYXGECBPQO'
            b'POHG9YCVXDWOXTEAOFBCEEAV9JCHUVLIRIMHXMUSZPOMMRBF9PLVLRJYTXTBANBZ'
            b'WFQWGNGFGXFOZ9YGMQSZFEJHLFZTTVHRLJPATA9TYCM9LSEWMNEUDNWQFLUXOFUN'
            b'VDKSNIIXCXVUYZZOKVYNNQDZVUQEQFWVF9EIQELSWDJXGMQRVUGGVBMRVGXBBPBE'
            b'BDVGZDBWMDMLPXYJBBRNOMKGPMCG9FTSLMRADFVPUTTEIOUCBLPRYZHGOICNC9BT'
            b'9WHJJJPDOSOMLD9EKRGKYUHUMMCAVHGYWOVQXFLTCXAAUDYKGKGKOYHLDCCQSKNH'
            b'JHPSXTJVTW9QPFOQ9FDZIDDKIVF9CDYGU9ABRESMDLIBONAQWFVGCNOTEDHBMCSU'
            b'H9GKYZPBX9QJELYYMSGDFU9EVTROODXVUAELBUKKXCDYNMHYBVAVUYGABCRIYOHV'
            b'ITGYROZZNQPTESTVALUETHREE9DONTUSEINPRODUCTION99999NUMQE9RGHNRRSK'
            b'KAOSD9WEYBHIUM9LWUWKEFSQOCVWVX9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99A99999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          # Spend transaction, Part 1 of 1
          TryteString(
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUETWO9DONTUSEINPRODUCTION99999XYYNXZLKBYNFPXA9'
            b'RUGZVEGVPLLFJEM9ZZOUINE9ONOWOB9999999999999999999999999999999999'
            b'999999999999999999NYBKIVD99999999999C99999999VEUNVMI9BSZTFZMGEZJ'
            b'CPMPOTRTUR9PSISHCXAESJQU9CEYAGXVHBAXAFRWHQNAFHGNID9BAOMKSJJDEO99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_pass_message_short(self):
    """
    Adding a message to a transaction.
    """
    response = self.command(
      seed = Seed.random(),

      transfers = [
        ProposedTransaction(
          tag   = Tag(b'PYOTA9UNIT9TESTS9'),
          value = 0,

          address = Address(
            b'TESTVALUE9DONTUSEINPRODUCTION99999YMSWGX'
            b'VNDMLXPT9HMVAOWUUZMLSJZFWGKDVGXPSQAWAEBJN'
          ),

          message = TryteString.from_string('สวัสดีชาวโลก!'),
        ),
      ],
    )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          TryteString(
            # Note that the tryte sequence starts with the transaction
            # message.
            b'HHVFHFHHVFEFHHVFOFHHVFHFHHVFMEHHVFSFHHVFCEHHVFPFHHVFEFHHWFVDHHVF'
            b'CFHHVFUDFA999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUE9DONTUSEINPRODUCTION99999YMSWGXVNDMLXPT9HMVA'
            b'OWUUZMLSJZFWGKDVGXPSQAWAEBJN999999999999999999999999999PYOTA9UNI'
            b'T9TESTS99999999999NYBKIVD99999999999999999999D9XYVJTKVWN9RUQAPIO'
            b'JUXXTOQTWNMOKRKLUURUGERIIZLUURHPQWZMSYROAKYLZJEKSAMLRCVWEDINFK99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )

  def test_pass_message_long(self):
    """
    The message is too long to fit into a single transaction.
    """
    response = self.command(
      seed = Seed.random(),

      transfers = [
        ProposedTransaction(
          tag   = Tag(b'PYOTA9UNIT9TESTS9'),
          value = 0,

          address = Address(
            b'TESTVALUE9DONTUSEINPRODUCTION99999YMSWGX'
            b'VNDMLXPT9HMVAOWUUZMLSJZFWGKDVGXPSQAWAEBJN'
          ),

          message = TryteString.from_string(
            'Вы не можете справиться правду! Сын, мы живем в мире, который '
            'имеет стены. И эти стены должны быть охраняют люди с оружием. '
            'Кто будет это делать? Вы? Вы, лейтенант Weinberg? У меня есть '
            'большая ответственность, чем вы можете понять. Ты плачешь '
            'Сантьяго и прокляни морских пехотинцев. У вас есть такой роскоши. '
            'У вас есть роскошь, не зная, что я знаю: что смерть Сантьяго, в '
            'то время как трагический, вероятно, спас жизнь. И мое '
            'существование, в то время как гротеск и непонятными для вас, '
            'спасает жизни ... Вы не хотите знать правду. Потому что в '
            'глубине души, в тех местах, вы не говорите о на вечеринках, вы '
            'хотите меня на этой стене. Вы должны меня на этой стене. Мы '
            'используем такие слова, как честь, код, верность ... мы '
            'используем эти слова в качестве основы к жизни провел, защищая '
            'что-то. Вы можете использовать им, как пуанта. У меня нет ни '
            'времени, ни желания, чтобы объясниться с человеком, который '
            'поднимается и спит под одеялом самой свободы я обеспечиваю, то '
            'ставит под сомнение то, каким образом я предоставить ему! Я бы '
            'предпочел, чтобы вы просто сказал спасибо и пошел на своем пути. '
            'В противном случае, я предлагаю вам подобрать оружие и встать '
            'пост. В любом случае, я не наплевать, что вы думаете, что вы '
            'имеете право!'
          ),
        ),
      ],
    )

    self.assertDictEqual(
      response,

      {
        'trytes': [
          # The message is so long that it has to be split across three
          # separate transactions!
          TryteString(
            b'EASGBGTGTDSGNFSGPFSGAGFA9999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'99999999999TESTVALUE9DONTUSEINPRODUCTION99999YMSWGXVNDMLXPT9HMVA'
            b'OWUUZMLSJZFWGKDVGXPSQAWAEBJN999999999999999999999999999PYOTA9UNI'
            b'T9TESTS99999999999NYBKIVD99B99999999B99999999YJVDLFI9FFXKNVTUKHO'
            b'PTZUWZPOTRTHNZ9YZDXFRVBAUGO9APIQQWFSCLGFQMLMVCEPCTBFAVMIIXHUPG99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          TryteString(
            b'FSG9GTGHEEASG9GSGNFEATGFETGVDSGAGSGWFEATGUDTGVDSGSFSG9GSGSFSAEAS'
            b'GKETGDEEASGRFSGAGSGYFSGTFSG9GTGDEEASGZFSGSFSG9GTGHEEASG9GSGNFEAT'
            b'GFETGVDSGAGSGWFEATGUDTGVDSGSFSG9GSGSFSAEASGUETGDEEASGVFTGUDSGBGS'
            b'GAGSGYFTGEESGUFTGWDSGSFSGZFEATGVDSGNFSGXFSGVFSGSFEATGUDSGYFSGAGS'
            b'GPFSGNFQAEASGXFSGNFSGXFEATG9ESGSFTGUDTGVDTGEEQAEASGXFSGAGSGRFQAE'
            b'ASGPFSGSFTGTDSG9GSGAGTGUDTGVDTGEEEASASASAEASGZFTGDEEASGVFTGUDSGB'
            b'GSGAGSGYFTGEESGUFTGWDSGSFSGZFEATGFETGVDSGVFEATGUDSGYFSGAGSGPFSGN'
            b'FEASGPFEASGXFSGNFTG9ESGSFTGUDTGVDSGPFSGSFEASGAGTGUDSG9GSGAGSGPFT'
            b'GDEEASGXFEASGTFSGVFSGUFSG9GSGVFEASGBGTGTDSGAGSGPFSGSFSGYFQAEASGU'
            b'FSGNFTGBESGVFTGBESGNFTGHEEATG9ETGVDSGAGRATGVDSGAGSAEASGKETGDEEAS'
            b'GZFSGAGSGTFSGSFTGVDSGSFEASGVFTGUDSGBGSGAGSGYFTGEESGUFSGAGSGPFSGN'
            b'FTGVDTGEEEASGVFSGZFQAEASGXFSGNFSGXFEASGBGTGWDSGNFSG9GTGVDSGNFSAE'
            b'ASGAFEASGZFSGSFSG9GTGHEEASG9GSGSFTGVDEASG9GSGVFEASGPFTGTDSGSFSGZ'
            b'FSGSFSG9GSGVFQAEASG9GSGVFEASGTFSGSFSGYFSGNFSG9GSGVFTGHEQAEATG9ET'
            b'GVDSGAGSGOFTGDEEASGAGSGOFTGCETGHETGUDSG9GSGVFTGVDTGEETGUDTGHEEAT'
            b'GUDEATG9ESGSFSGYFSGAGSGPFSGSFSGXFSGAGSGZFQAEASGXFSGAGTGVDSGAGTGT'
            b'DTGDESGWFEASGBGSGAGSGRFSG9GSGVFSGZFSGNFSGSFTGVDTGUDTGHEEASGVFEAT'
            b'GUDSGBGSGVFTGVDEASGBGSGAGSGRFEASGAGSGRFSGSFTGHESGYFSGAGSGZFEATGU'
            b'DSGNFSGZFSGAGSGWFEATGUDSGPFSGAGSGOFSGAGSGRFTGDEEATGHEEASGAGSGOFS'
            b'GSFTGUDSGBGSGSFTG9ESGVFSGPFSGNFTGGEQAEATGVDSGAGEATGUDTGVDSGNFSGP'
            b'FSGVFTGVDEASGBGSGAGSGRFEATGUDSGAGSGZFSG9GSGSFSG9GSGVFSGSFEATGVDS'
            b'GAGQAEASGXFSGNFSGXFSGVFSGZFEASGAGSGOFTGTDSGNFSGUFSGAGSGZFEATGHEE'
            b'ASGBGTGTDSGSFSGRFSGAGTGUDTGVDSGNFSGPFSGVFTGVDTGEEEASGSFSGZFTGWDF'
            b'AEASGMFEASGOFTGDEEASGBGTGTDSGSFSGRFSGBGSGAGTG9ESGSFSGYFQAEATG9ET'
            b'GVDSGAGSGOFTGDEEASGPFTGDEEASGBGTGTDSGAGTGUDTGVDSGAGEATGUDSGXFSGN'
            b'FSGUFSGNFSGYFEATGUDSGBGSGNFTGUDSGVFSGOFSGAGEASGVFEASGBGSGAGTGAES'
            b'GSFSGYFEASG9GSGNFEATGUDSGPFSGAGSGSFSGZFEASGBGTGWDTGVDSGVFSAEASGK'
            b'EEASGBGTGTDSGAGTGVDSGVFSGPFSG9GSGAGSGZFEATGUDSGYFTGWDTG9ESGNFSGS'
            b'FQAEATGHEEASGBGTGTDSGSFSGRFSGYFSGNFSGQFSGNFTGGEEASGPFSGNFSGZFEAS'
            b'GBGSGAGSGRFSGAGSGOFTGTDSGNFTGVDTGEEEASGAGTGTDTGWDSGTFSGVFSGSFEAS'
            b'GVFEASGPFTGUDTGVDSGNFTGVDTGEEEASGBGSGAGTGUDTGVDSAEASGKEEASGYFTGG'
            b'ESGOFSGAGSGZFEATGUDSGYFTGWDTG9ESGNFSGSFQAEATGHEEASG9GSGSFEASG9GS'
            b'GNFSGBGSGYFSGSFSGPFSGNFTGVDTGEEQAEATG9ETGVDSGAGEASGPFTGDEEASGRFT'
            b'GWDSGZFSGNFSGSFTGVDSGSFQAEATG9ETGVDSGAGEASGPFTGDEEASGVFSGZFSGSFS'
            b'GSFTGVDSGSFTESTVALUE9DONTUSEINPRODUCTION99999YMSWGXVNDMLXPT9HMVA'
            b'OWUUZMLSJZFWGKDVGXPSQAWAEBJN999999999999999999999999999PYOTA9UNI'
            b'T9TESTS99999999999NYBKIVD99A99999999B99999999YJVDLFI9FFXKNVTUKHO'
            b'PTZUWZPOTRTHNZ9YZDXFRVBAUGO9APIQQWFSCLGFQMLMVCEPCTBFAVMIIXHUPG99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),

          TryteString(
            b'SGKETGDEEASG9GSGSFEASGZFSGAGSGTFSGSFTGVDSGSFEATGUDSGBGTGTDSGNFSG'
            b'PFSGVFTGVDTGEETGUDTGHEEASGBGTGTDSGNFSGPFSGRFTGWDFAEASGZETGDESG9G'
            b'QAEASGZFTGDEEASGTFSGVFSGPFSGSFSGZFEASGPFEASGZFSGVFTGTDSGSFQAEASG'
            b'XFSGAGTGVDSGAGTGTDTGDESGWFEASGVFSGZFSGSFSGSFTGVDEATGUDTGVDSGSFSG'
            b'9GTGDESAEASGQEEATGFETGVDSGVFEATGUDTGVDSGSFSG9GTGDEEASGRFSGAGSGYF'
            b'SGTFSG9GTGDEEASGOFTGDETGVDTGEEEASGAGTGYDTGTDSGNFSG9GTGHETGGETGVD'
            b'EASGYFTGGESGRFSGVFEATGUDEASGAGTGTDTGWDSGTFSGVFSGSFSGZFSAEASGSETG'
            b'VDSGAGEASGOFTGWDSGRFSGSFTGVDEATGFETGVDSGAGEASGRFSGSFSGYFSGNFTGVD'
            b'TGEEIBEASGKETGDEIBEASGKETGDEQAEASGYFSGSFSGWFTGVDSGSFSG9GSGNFSG9G'
            b'TGVDEAFCTCXCBDQCTCFDVCIBEASGAFEASGZFSGSFSG9GTGHEEASGSFTGUDTGVDTG'
            b'EEEASGOFSGAGSGYFTGEETGAESGNFTGHEEASGAGTGVDSGPFSGSFTGVDTGUDTGVDSG'
            b'PFSGSFSG9GSG9GSGAGTGUDTGVDTGEEQAEATG9ESGSFSGZFEASGPFTGDEEASGZFSG'
            b'AGSGTFSGSFTGVDSGSFEASGBGSGAGSG9GTGHETGVDTGEESAEASG9FTGDEEASGBGSG'
            b'YFSGNFTG9ESGSFTGAETGEEEASGZESGNFSG9GTGVDTGEETGHESGQFSGAGEASGVFEA'
            b'SGBGTGTDSGAGSGXFSGYFTGHESG9GSGVFEASGZFSGAGTGTDTGUDSGXFSGVFTGYDEA'
            b'SGBGSGSFTGYDSGAGTGVDSGVFSG9GTGZDSGSFSGPFSAEASGAFEASGPFSGNFTGUDEA'
            b'SGSFTGUDTGVDTGEEEATGVDSGNFSGXFSGAGSGWFEATGTDSGAGTGUDSGXFSGAGTGAE'
            b'SGVFSAEASGAFEASGPFSGNFTGUDEASGSFTGUDTGVDTGEEEATGTDSGAGTGUDSGXFSG'
            b'AGTGAETGEEQAEASG9GSGSFEASGUFSG9GSGNFTGHEQAEATG9ETGVDSGAGEATGHEEA'
            b'SGUFSG9GSGNFTGGEDBEATG9ETGVDSGAGEATGUDSGZFSGSFTGTDTGVDTGEEEASGZE'
            b'SGNFSG9GTGVDTGEETGHESGQFSGAGQAEASGPFEATGVDSGAGEASGPFTGTDSGSFSGZF'
            b'TGHEEASGXFSGNFSGXFEATGVDTGTDSGNFSGQFSGVFTG9ESGSFTGUDSGXFSGVFSGWF'
            b'QAEASGPFSGSFTGTDSGAGTGHETGVDSG9GSGAGQAEATGUDSGBGSGNFTGUDEASGTFSG'
            b'VFSGUFSG9GTGEESAEASGQEEASGZFSGAGSGSFEATGUDTGWDTGBESGSFTGUDTGVDSG'
            b'PFSGAGSGPFSGNFSG9GSGVFSGSFQAEASGPFEATGVDSGAGEASGPFTGTDSGSFSGZFTG'
            b'HEEASGXFSGNFSGXFEASGQFTGTDSGAGTGVDSGSFTGUDSGXFEASGVFEASG9GSGSFSG'
            b'BGSGAGSG9GTGHETGVDSG9GTGDESGZFSGVFEASGRFSGYFTGHEEASGPFSGNFTGUDQA'
            b'EATGUDSGBGSGNFTGUDSGNFSGSFTGVDEASGTFSGVFSGUFSG9GSGVFEASASASAEASG'
            b'KETGDEEASG9GSGSFEATGYDSGAGTGVDSGVFTGVDSGSFEASGUFSG9GSGNFTGVDTGEE'
            b'EASGBGTGTDSGNFSGPFSGRFTGWDSAEASGXESGAGTGVDSGAGSGZFTGWDEATG9ETGVD'
            b'SGAGEASGPFEASGQFSGYFTGWDSGOFSGVFSG9GSGSFEASGRFTGWDTGAESGVFQAEASG'
            b'PFEATGVDSGSFTGYDEASGZFSGSFTGUDTGVDSGNFTGYDQAEASGPFTGDEEASG9GSGSF'
            b'EASGQFSGAGSGPFSGAGTGTDSGVFTGVDSGSFEASGAGEASG9GSGNFEASGPFSGSFTG9E'
            b'SGSFTGTDSGVFSG9GSGXFSGNFTGYDQAEASGPFTGDEEATGYDSGAGTGVDSGVFTGVDSG'
            b'SFEASGZFSGSTESTVALUE9DONTUSEINPRODUCTION99999YMSWGXVNDMLXPT9HMVA'
            b'OWUUZMLSJZFWGKDVGXPSQAWAEBJN999999999999999999999999999PYOTA9UNI'
            b'T9TESTS99999999999NYBKIVD99999999999B99999999YJVDLFI9FFXKNVTUKHO'
            b'PTZUWZPOTRTHNZ9YZDXFRVBAUGO9APIQQWFSCLGFQMLMVCEPCTBFAVMIIXHUPG99'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999999999999999999'
            b'9999999999999999999999999999999999999999999999999'
          ),
        ],
      },
    )


class MockSignatureFragmentGenerator(object):
  """
  Mocks the behavior of ``SignatureFragmentGenerator`` to speed up unit
  tests.

  Note that ``SignatureFragmentGenerator`` already has its own test
  case, so this approach does not compromise test integrity.

  References:
    - :py:class:`iota.crypto.signing.SignatureFragmentGenerator`
    - :py:meth:`iota.transaction.ProposedBundle.sign_inputs`
  """
  def __init__(
      self,
      fragments = None,
      length    = AddressGenerator.DIGEST_ITERATIONS
  ):
    # type: (Optional[Iterable[TryteString]], int) -> None
    """
    :param fragments:
      Provide fragments to seed (shortcut for calling :py:meth:`seed`).

    :param length:
      Length that the generator will report to the bundle, used to
      ensure that it iterates the correct number of times.
    """
    super(MockSignatureFragmentGenerator, self).__init__()

    self.fragments  = list(fragments or []) # type: List[TryteString]
    self.length     = length

  def __iter__(self):
    return self

  def __len__(self):
    return self.length

  def seed(self, fragment):
    # type: (TryteString) -> None
    self.fragments.append(fragment)

  # noinspection PyUnusedLocal
  def __next__(self):
    # type: (TryteString) -> TryteString
    return self.fragments.pop(0)

  if PY2:
    next = __next__
