# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tail_recursive']

package_data = \
{'': ['*']}

install_requires = \
['pytest-timeout>=1.4.2,<2.0.0']

setup_kwargs = {
    'name': 'tail-recursive',
    'version': '2.0.0',
    'description': 'Tail recursion with a simple decorator api.',
    'long_description': '![tests](https://github.com/0scarB/tail-recursive/workflows/Tests/badge.svg)\n\n## Installation\n\n`pip install tail-recursive`\n\n## Basic Usage\n\nUse the `tail_recursive` decorator to simply define tail recursive functions.\n\nIf you are encountering **maximum recursion depth errors** or **out-of-memory crashes** tail recursion can be a helpful strategy.\n\n### Example\n\n```python\nfrom tail_recursive import tail_recursive\n\n\n# Pick a larger value if n is below your system\'s recursion limit.\nx = 5000\n\n\ndef factorial_without_tail_recursion(n):\n    if n <= 1:\n        return n\n    return n * factorial_without_tail_recursion(n - 1)\n\n\ntry:\n    # This will exceed the maximum recursion depth.\n    factorial_without_tail_recursion(x)\nexcept RecursionError:\n    pass\n\n\n@tail_recursive\ndef factorial(n):\n    if n <= 1:\n        return n\n    # It is important that you return the return value of the `tail_call`\n    # method for tail recursion to take effect!\n    # Note tail calls work with dunder methods, such as __mul__ and __rmul__,\n    # by default.\n    return n * factorial.tail_call(n - 1)\n\n\n# Implementation with tail recursion succeeds because the function is\n# called sequentially under the hood.\nfactorial(x)\n```\n\n### How it works\n\nWhen a function (in this case `factorial`) is decorated by `@tail_recursive`, it returns\nan object implementing the `tail_call` method. This object also overrides the `__call__`\nmethod, meaning that it can be called just like the original function (e.g. `factorial(x)`).\n\nDecorated functions test whether they return a call to `tail_call(...)`. If this is the case\nthen the return value is pushed on a call stack implemented as a list. `tail_call` returns\nan object storing the function it was called on (e.g. `factorial`) and the (keyword)\narguments (e.g. `n - 1`) it was called with. If the arguments contain a nested call to `tail_call` then this\ncall is also pushed onto the call stack. On the other hand if `tail_call` is passed no nested\n`tail_call`s then the function that it stores is called with the stored (keyword) arguments. The\nreturn value of this lazy call then (a) replaces the argument it was passed as or (b)\nreturns another `tail_call` which is pushed to the stack or (c) is the final return value of\nthe call to the decorated function (e.g. `factorial(x)`).\n\nBut how can `factorial.tail_call(n - 1)` be multiplied by `n`? Well, the object returned by\n`tail_call` overrides most dunder methods, such as `__mul__` and `__rmul__`, pushing the\nequivalent of `tail_recursive(int.__rmul__).tail_call(n, factorial.tail_call(n - 1)` to the\ncall stack.\n\nThe call stack for `factorial(3)` would looks something like this.\n\n1. Because `factorial(3)` is called, `<lazy_call_obj>(func=factorial, args=(3,), kwargs={})`\n   is **pushed** on the stack.\n\n```python\n[\n    <lazy_call_obj>(func=factorial, args=(3,), kwargs={}),\n]\n```\n\n2. Because `<lazy_call_obj>(func=factorial, args=(3,), kwargs={})` contains no nested arguments,\n   it is **popped** off the stack. It is then lazily evaluated, returning another `<lazy_call_obj>`, which is **pushed** to the stack.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args(<lazy_call_obj>(func=factorial, args=(2,), kwargs={}), 3), kwargs={}),\n]\n```\n\n3. The lazy call to `__rmul__` has a nested call as an argument. Consequentially, this\n   argument is **pushed** on the call stack.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(2,), kwargs={}), 3), kwargs={}),\n    <lazy_call_obj>(func=factorial, args=(2,), kwargs={}),\n]\n```\n\n4. As in step _2_ the lazy call to `factorial(2)` is **pop** off the stack and its return\n   value is **pushed** on.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(2,), kwargs={}), 3), kwargs={}),\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(1,), kwargs={}), 2), kwargs={}),\n]\n```\n\n5. Similarly to step _3_, because the lazy call to `__rmul__` has a nested call as an\n   argument, this argument is **pushed** on the stack.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(2,), kwargs={}), 3), kwargs={}),\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(1,), kwargs={}), 2), kwargs={}),\n    <lazy_call_obj>(func=factorial, args=(1,), kwargs={}),\n]\n```\n\n6. `<lazy_call_obj>(func=int.__rmul__, args=(1,), kwargs={})` has no nested lazy calls\n   as arguments, so it is **popped** off the stack and its return value replaces\n   the argument of `__rmul__` that it was originally passed as.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args=(<lazy_call_obj>(func=factorial, args=(2,), kwargs={}), 3), kwargs={}),\n    <lazy_call_obj>(func=int.__rmul__, args=(1, 2), kwargs={}),\n]\n```\n\n7. The same process as in _6_ is repeated, where\n   `<lazy_call_obj>(func=int.__rmul__, args=(2, 1), kwargs={})` is **popped** off the\n   stack and its return value replaces the second argument of the lazy call to\n   `int.__rmul__(3, ...)`.\n\n```python\n[\n    <lazy_call_obj>(func=int.__rmul__, args=(2, 3), kwargs={}),\n]\n```\n\n8. Finally, because the lazy call to `__rmul__` no longer has any nested calls as\n   arguments, it can be **popped** off the stack. Furthermore, it was not passed\n   as an argument of a previous call on the stack and, for that reason, is returned\n   from our decorated function (i.e. `factorial(3) = int.__rmul__(2, 3) = 6`).\n\n```python\n[]\n```\n\n## Features\n\n### Nested Tail Calls\n\n(only works for `feature_set="full"|FeatureSet.FULL`)\n\nAs mentioned above nested tail calls are sequentially evaluated by creating a call stack.\n\n```python\n...\n\n@tail_recursive\ndef mul(a, b):\n    return a * b\n\n@tail_recursive\ndef factorial(n):\n    if n == 1:\n        return n\n    return mul.tail_call(n, factorial.tail_call(n - 1))\n\n...\n```\n\nNested calls, however, comes a performance cost and can be disabled as follows.\n\n```python\n@tail_recursive(feature_set="base")\ndef factorial(n, accumulator=1):\n    if n == 1:\n        return accumulator\n    return factorial.tail_call(n - 1, n * accumulator)\n```\n\nor\n\n```python\nfrom tail_recursive import tail_recursive, FeatureSet\n\n...\n\n@tail_recursive(nested_call_mode=FeatureSet.BASE)\ndef factorial(n, accumulator=1):\n    ...\n```\n\nSimilarly, use `feature_set="full"` or `feature_set=FeatureSet.FULL`\nto explicitly enable this feature.\n\n### Dunder Method Overrides\n\n(only works for `feature_set="full"|FeatureSet.FULL`)\n\n`n * factorial.tail_call(n - 1)` shows that numeric operations\ncan be done on tail calls and so long as the result of the expression\nis returned by the function. These expression will ultimately\nevaluate to the same value that they would have if `tail_call` had been omitted.\nThis is also true for comparison and bitwise\noperations, attribute and index access (i.e. `<func>.tail_call(...)[...]`)\nand much more functionality provided by dunder methods.\n\nThat being said, attribute assignment (i.e. `<func>.tail_call(...).<attr> = val`)\nand the functionality provided by the following dunder methods are not currently\nsupported with `tail_call`.\n\n- `__del__`\n- `__getattribute__`\n- `__setattr__`\n- `__get__`\n- `__set__`\n- `__delete__`\n- `__set_name__`\n- `__init_subclass__`\n- `__prepare__`\n\nNote that also `__init__` and `__new__` cannot be called directly on a tail call\n(e.g. `<func>.tail_call(...).__init__(...)`) and are instead implicitly lazily evaluated\nwith the arguments passed to `tail_call` while popping off/unwinding the tail call stack.\n\nFuthermore, dunder methods added after 3.8 and in standard library or third-party packages/modules may also not be supported.\n\nAnother important note is that dunder attributes will currently not be lazily evaluated.\ne.g.\n\n- `__doc__`\n- `__name__`\n- `__qualname__`\n- `__module__`\n- `__defaults__`\n- `__defaults__`\n- `__code__`\n- `__globals__`\n- `__dict__`\n- `__closure__`\n- `__annotations__`\n- `__kwdefaults__`\n\nFinally, since `__repr__` and `__str__` are overridden use\n`<func>.tail_call(...)._to_string()` to pretty print tail calls.\n\n## Usage with other Decorators\n\nEspecially in recursive algorithms it can significantly increase performance\nto use memoization. In this use case it is best to place the decorator enabling\nmemoization after `@tail_recursive`. e.g.\n\n```python\nimport functools\n\n@tail_recursive(feature_set="full")\n@functools.lru_cache\ndef fibonacci(n):\n    if n <= 1:\n        return n\n    return fibonacci.tail_call(n - 1) + fibonacci.tail_call(n - 2)\n```\n\nFor properties place the `@property` decorator before `@tail_recursive`.\n\n## Current Limitations\n\n### Return Values\n\nCurrently tail calls that are returned as item/member in a tuple or other\ndata structures are not evaluated.\n\nThe following will not evaluate the tail call.\n\n```python\nfrom tail_recursive import tail_recursive\n\n@tail_recursive\ndef func(...):\n    ...\n    return return_val1, func.tail_call(...)\n```\n\nA workaround is to use factory functions.\n\n```python\nfrom tail_recursive import tail_recursive\n\n@tail_recursive\ndef tuple_factory(*args):\n    return tuple(args)\n\n@tail_recursive\ndef func(...):\n    ...\n    return tuple_factory.tail_call(\n        return_val1,\n        func.tail_call(...)\n    )\n```\n\nOr pass the container object\'s type directly to `tail_recursive`.\n\n```python\nfrom tail_recursive import tail_recursive\n\n@tail_recursive\ndef func(...):\n    ...\n    return tail_recursive(tuple).tail_call((\n        return_val1,\n        func.tail_call(...)\n    ))\n```\n\n### Method Decorators\n\nCurrently, when calling `tail_call` on a decorated method, you need to explicitly pass\nself (the current objects instance) as the first argument. e.g.\n\n```python\nclass MathStuff:\n\n    @tail_recursive(feature_set="full")\n    def fibonacci(self, n):\n        if n <= 1:\n            return n\n        return self.fibonacci.tail_call(self, n - 1) + self.fibonacci.tail_call(self, n - 2)\n                                        ^^^^                                    ^^^^\n```\n\n## Other Packages\n\nCheck out [tco](https://github.com/baruchel/tco) for an alternative api with extra functionality.\n',
    'author': '0scarB',
    'author_email': 'oscarb@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/0scarB/tail-recursive',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
