import asyncio
import serial
import time

import numpy as np

from . import callbacks
from . import parsers
from . import boards


def read_all(ser, read_buffer=b"", **args):
    """Read all available bytes from the serial port
    and append to the read buffer.

    Parameters
    ----------
    ser : serial.Serial() instance
        The device we are reading from.
    read_buffer : bytes, default b''
        Previous read buffer that is appended to.

    Returns
    -------
    output : bytes
        Bytes object that contains read_buffer + read.

    Notes
    -----
    .. `**args` appears, but is never used. This is for
       compatibility with `read_all_newlines()` as a
       drop-in replacement for this function.
    """
    # Set timeout to None to make sure we read all bytes
    previous_timeout = ser.timeout
    ser.timeout = None

    in_waiting = ser.in_waiting
    read = ser.read(size=in_waiting)

    # Reset to previous timeout
    ser.timeout = previous_timeout

    return read_buffer + read


def read_all_newlines(ser, read_buffer=b"", n_reads=1):
    """Read data in until encountering newlines.

    Parameters
    ----------
    ser : serial.Serial() instance
        The device we are reading from.
    n_reads : int
        The number of reads up to newlines
    read_buffer : bytes, default b''
        Previous read buffer that is appended to.

    Returns
    -------
    output : bytes
        Bytes object that contains read_buffer + read.

    Notes
    -----
    .. This is a drop-in replacement for read_all(). It is not currently
       used within the dashboard app, but can be useful if there are
       issues reading a stream of data without newlines that is
       sometimes encountered on Windows machines.
    """
    raw = read_buffer
    for _ in range(n_reads):
        raw += ser.read_until()

    return raw


async def daq_stream(
    plotter, monitor, serial_connection, n_reads_per_chunk=1, reader=read_all
):
    """Obtain streaming data

    Parameters
    ----------
    plotter : SerialPlotter instance
        Plotter displaying parsed data. plotter.data is updated in this
        coroutine.
    monitor : SerialMonitor instance
        Monitor displaying data coming from the serial connection.
        monitor.data is updated in this coroutine.
    serial_connection : SerialConnection instance
        Details about the serial connection
    n_reads_per_chunk : int, default 1
        This parameter is only used if `reader` is `read_all_newlines`.
        `n_read_per_chunk` lines are read in.
    reader : function, default read_all
        Either `read_all` or `read_all_newlines`. `read_all_newlines` is
        only necessary on some windows machines that have problems
        reading in data that does not end with a newline.
    """
    # Receive data
    read_buffer = [b""]
    while True:
        # Read in chunk` of data
        raw = reader(serial_connection.ser, read_buffer=b"", n_reads=n_reads_per_chunk)

        if monitor.streaming:
            monitor.data += raw.decode()

        if plotter.streaming:
            # Parse it, passing if it is gibberish or otherwise corrupted
            try:
                data, n_reads, read_buffer[0] = parsers.parse_read(
                    read_buffer[0] + raw, sep=plotter.delimiter
                )

                # Proceed if we actually read in data
                if len(data) > 0:
                    # If this is our first data, add them into plot_data
                    if len(plotter.data) == 0:
                        plotter.data = data
                    else:
                        plotter.data += data
            except:
                pass

        # Sleep 80% of the time before we need to start reading chunks
        await asyncio.sleep(
            0.8 * n_reads_per_chunk * serial_connection.daq_delay / 1000
        )


async def port_search(serial_connection):
    """Search for ports and update dictionary of ports.

    Parameters
    ----------
    serial_connection : SerialConnection instance
        The `available_ports` and `reverse_available_ports` attributes
        are updated by this coroutine.
    """
    while True:
        serial_connection.portsearch(on_change=True)

        # Sleep before searching again
        await asyncio.sleep(serial_connection.port_search_delay / 1000.0)


def device_name(port):
    """Generate a name for a device at a given port.

    Parameters
    ----------
    port : serial.tools.list_ports_common.ListPortInfo instance
        Port in question. The `port` instance is usually generated by
        calling `serial.tools.list_ports.comports()`.

    Returns
    -------
    output : str
        A name for the port. If there is an HWID record known to the
        serial-dashboard package, the name of the device is drawn from
        that; otherwise, if available, the name of the manufacturer is
        used. If none of those are available, then `port.device`
        appended with two spaces is returned.
    """
    device = port.device + "  "

    # If there is an HWID record, use it, otherwise use manufacturer
    try:
        i = port.hwid.find("VID:PID=") + 8
        j = port.hwid.find(" ", i)
        vid_pid = port.hwid[i:j]
        if vid_pid in boards.vid_pid_boards:
            device += boards.vid_pid_boards[vid_pid]
        else:
            device += vid_pid
    except:
        if port_name.manufacturer is not None:
            device += port.manufacturer

    return device


def handshake_board(ser, sleep_time=1):
    """Connect to board by closing, then opening connection, discarding
    anything that may be in the input buffer.

    Parameters
    ----------
    ser : serial.Serial() instance
        The device we are reading from.
    sleep_time : int, default 1
        Time in seconds to wait after opening the serial connection
        before clearing the input buffer.
    """
    # Close and reopen
    ser.close()
    ser.open()

    # Chill out while everything gets set
    time.sleep(sleep_time)

    # Set a long timeout to complete handshake
    timeout = ser.timeout
    ser.timeout = 2

    # Read and discard everything that may be in the input buffer
    _ = ser.read_all()

    # Reset the timeout
    ser.timeout = timeout
