"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleMetadata = exports.PackName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const cdk_nag_1 = require("cdk-nag");
/**
 * Pack name
 *
 */
exports.PackName = "AwsPrototyping";
/**
 * Rule metadata
 *
 */
exports.RuleMetadata = [
    {
        info: "The REST API stage is not associated with AWS WAFv2 web ACL.",
        explanation: "AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAssociatedWithWAF,
    },
    {
        info: "The API does not implement authorization.",
        explanation: "In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.\n\nExample threat: An actor with a network path to an API gateway stage end-point can interact with the API method in question without authorization, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAuthorization,
    },
    {
        info: "The API Gateway stage does not does not have caching and encryption enabled for all methods.",
        explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWCacheEnabledAndEncrypted,
    },
    {
        info: "The REST API does not have request validation enabled.",
        explanation: "The API should have basic request validation enabled. If the API is integrated with a custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWRequestValidation,
    },
    {
        info: "The API Gateway REST API stage is not configured with SSL certificates.",
        explanation: "Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWSSLEnabled,
    },
    {
        info: "The GraphQL API does not have request level logging enabled.",
        explanation: "It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.appsync.AppSyncGraphQLRequestLogging,
    },
    {
        info: "The Athena workgroup does not encrypt query results.",
        explanation: "Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.athena.AthenaWorkgroupEncryptedQueryResults,
    },
    {
        info: "The Auto Scaling group (which is associated with a load balancer) does not utilize ELB health checks.",
        explanation: "The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingGroupELBHealthCheckRequired,
    },
    {
        info: "The Auto Scaling launch configuration does not have public IP addresses disabled.",
        explanation: "If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingLaunchConfigPublicIpDisabled,
    },
    {
        info: "The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.",
        explanation: "SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.\n\nExample threat: A global internet-based actor can discover Cloud9 EC2 instances that have public IP addresses and that are exposing SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloud9.Cloud9InstanceNoIngressSystemsManager,
    },
    {
        info: "The CloudFront distribution may require Geo restrictions.",
        explanation: "Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.\n\nExample threat: A global internet-based actor can discover prototype web assets that are exposed via CloudFront distributions, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionGeoRestrictions,
    },
    {
        info: "The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the SslSupportMethod are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified MinimumProtocolVersion",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
    },
    {
        info: "The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionNoOutdatedSSL,
    },
    {
        info: "The CloudFront distribution does not use an origin access identity an S3 origin.",
        explanation: "Origin access identities help with security by restricting any direct access to objects through S3 URLs.\n\nExample threat: A global internet-based actor can bypass the CloudFront distribution and associated controls (e.g. geo blocking), which may lead to direct access to static assets hosted on the S3 origin possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 origin for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionS3OriginAccessIdentity,
    },
    {
        info: "The CloudFront distribution may require integration with AWS WAF.",
        explanation: "The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionWAFIntegration,
    },
    {
        info: "The trail does not have encryption enabled.",
        explanation: "Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudtrail.CloudTrailEncryptionEnabled,
    },
    {
        info: "The CloudWatch Log Group is not encrypted with an AWS KMS key.",
        explanation: "To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudwatch.CloudWatchLogGroupEncrypted,
    },
    {
        info: "The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.",
        explanation: "Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.\n\nExample threat: An actor who can view the CodeBuild environment variables can obtain the AWS Access Key and Secret Access Key, which may lead to the actor being able to do anything the AWS keys are authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectEnvVarAwsCred,
    },
    {
        info: "The CodeBuild project does not use an AWS KMS key for encryption.",
        explanation: "Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectKMSEncryptedArtifacts,
    },
    {
        info: "The CodeBuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.",
        explanation: "Explaining differences/edits to Docker images helps operators better understand system dependencies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectManagedImages,
    },
    {
        info: "The CodeBuild project has privileged mode enabled.",
        explanation: "Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectPrivilegedModeDisabled,
    },
    {
        info: "The CodeBuild project which utilizes either a GitHub or BitBucket repository does not utilize OAUTH.",
        explanation: "OAUTH is the most secure method of authenticating your Codebuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectSourceRepoUrl,
    },
    {
        info: "The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.",
        explanation: "Advanced security features enable the system to detect and act upon malicious sign-in attempts.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAdvancedSecurityModeEnforced,
    },
    {
        info: "The API Gateway method does not use a Cognito user pool authorizer.",
        explanation: "API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAPIGWAuthorizer,
    },
    {
        info: "The Cognito user pool is not configured to require MFA.",
        explanation: "Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolMFA,
    },
    {
        info: "The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.",
        explanation: "Applications do not warrant unauthenticated guest access in many cases. Metadata explaining the use case allows for transparency to operators.\n\nExample threat: A global internet-based actor who has discovered a prototype endpoint with Cognito unauthenticated logins can does not need to provide credentials to interact with the endpoint, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolNoUnauthenticatedLogins,
    },
    {
        info: "The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.",
        explanation: "Strong password policies increase system security by encouraging users to create reliable and secure passwords.\n\nExample threat: An actor who has discovered a prototype endpoint with Cognito authenticated logins can perform a dictionary or brute force attack to authenticate as an authorized Cognito user, which may lead to the actor being able to do anything the associated Cognito user is authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolStrongPasswordPolicy,
    },
    {
        info: "The DMS replication instance is public.",
        explanation: "DMS replication instances can contain sensitive information and access control is required for such accounts.\n\nExample threat: A global internet-based actor can discover DMS instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.dms.DMSReplicationNotPublic,
    },
    {
        info: "The Document DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterEncryptionAtRest,
    },
    {
        info: "The Document DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterNonDefaultPort,
    },
    {
        info: "The Document DB cluster does not have the username and password stored in Secrets Manager.",
        explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.\n\nExample threat: An actor who can view the DocumentDB configuration can obtain the username and password for the DocumentDB cluster, which may lead to the actor being able to access anything the associated DocumentDB user is authorised to do possibly impacting the confidentiality, integrity and availability of the data assets hosted on the DocumentDB cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.documentdb.DocumentDBCredentialsInSecretsManager,
    },
    {
        info: "The DAX cluster does not have server-side encryption enabled.",
        explanation: "Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.dynamodb.DAXEncrypted,
    },
    {
        info: "The EBS volume has encryption disabled.",
        explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2EBSVolumeEncrypted,
    },
    {
        info: "The EC2 instance is associated with a public IP address.",
        explanation: "Amazon EC2 instances can contain sensitive information and access control is required for such resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceNoPublicIp,
    },
    {
        info: "The EC2 instance does not have an instance profile attached.",
        explanation: "EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceProfileAttached,
    },
    {
        info: "The EC2 instance is not within a VPC.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2InstancesInVPC,
    },
    {
        info: "The EC2 instance is not part of an ASG and has Termination Protection disabled.",
        explanation: "Termination Protection safety feature should be enabled in order to protect the instances from being accidentally terminated.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceTerminationProtection,
    },
    {
        info: "The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).",
        explanation: "Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.\n\nExample threat: A global internet-based actor can discover exposed services (e.g. Telnet, SSH, RDS, MySQL) using their common port numbers, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedCommonPorts,
    },
    {
        info: "The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.",
        explanation: "Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public EC2 instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedInbound,
    },
    {
        info: "The Security Group allows unrestricted SSH access.",
        explanation: "Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address to SSH or move laterally to non-public EC2 instances via SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedSSH,
    },
    {
        info: "The Security Group does not have a description.",
        explanation: "Descriptions help simplify operations and remove any opportunities for operator errors.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2SecurityGroupDescription,
    },
    {
        info: "The ECR Repository allows open access.",
        explanation: "Removing * principals in an ECR Repository helps protect against unauthorized access.\n\nExample threat: A global internet-based actor who has discovered a ECR repository can access the container images hosted within the repository, which may lead to information disclosure that aids in the intrusion activities being successful against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ecr.ECROpenAccess,
    },
    {
        info: "The ECS Cluster has CloudWatch Container Insights disabled.",
        explanation: "CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSClusterCloudWatchContainerInsights,
    },
    {
        info: "One or more containers in the ECS Task Definition do not have container logging enabled.",
        explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionContainerLogging,
    },
    {
        info: "The ECS Task Definition includes a container definition that directly specifies environment variables.",
        explanation: "Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionNoEnvironmentVariables,
    },
    {
        info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty.",
        explanation: "If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionUserForHostMode,
    },
    {
        info: "The EFS does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.efs.EFSEncrypted,
    },
    {
        info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
        explanation: "EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format LogExport::<log> for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.eks.EKSClusterControlPlaneLogs,
    },
    {
        info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
        explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).\n\nExample threat: A global internet-based actor who has discovered a EKS cluster Kubernetes API server endpoint can perform reconnaissance and intrusion activities against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eks.EKSClusterNoEndpointPublicAccess,
    },
    {
        info: "The ElastiCache cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover the ElastiCache cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterInVPC,
    },
    {
        info: "The ElastiCache cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterNonDefaultPort,
    },
    {
        info: "The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.",
        explanation: "Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterEncryption,
    },
    {
        info: "The ElastiCache Redis cluster does not use Redis AUTH for user authentication.",
        explanation: "Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterRedisAuth,
    },
    {
        info: "The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.",
        explanation: "Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkEC2InstanceLogsToS3,
    },
    {
        info: "The Elastic Beanstalk environment does not have managed updates enabled.",
        explanation: "Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.\n\nExample threat: An actor with a network path to the Elastic Beanstalk environment can attempt to take advantage of a known vulnerability in a platform component used by Elastic Beanstalk, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkManagedUpdatesEnabled,
    },
    {
        info: "The Elastic Beanstalk environment is not configured to use a specific VPC.",
        explanation: "Use a non-default VPC in order to separate your environment from default resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkVPCSpecified,
    },
    {
        info: "The ALB does not have invalid HTTP header dropping enabled.",
        explanation: "Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist in these headers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpDropInvalidHeaderEnabled,
    },
    {
        info: "The ALB's HTTP listeners are not configured to redirect to HTTPS.",
        explanation: "To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpToHttpsRedirection,
    },
    {
        info: "The ALB is not associated with AWS WAFv2 web ACL.",
        explanation: "A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBWAFEnabled,
    },
    {
        info: "The CLB does not have connection draining enabled.",
        explanation: "With Connection Draining feature enabled, if an EC2 backend instance fails health checks the CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.CLBConnectionDraining,
    },
    {
        info: "The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).",
        explanation: "Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBACMCertificateRequired,
    },
    {
        info: "The CLB does not restrict its listeners to only the SSL and HTTPS protocols.",
        explanation: "Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBTlsHttpsListenersOnly,
    },
    {
        info: "The ALB, NLB, or GLB listener do not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).",
        explanation: "Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBv2ACMCertificateRequired,
    },
    {
        info: "The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.",
        explanation: "SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRAuthEC2KeyPairOrKerberos,
    },
    {
        info: "The EMR cluster does not use a security configuration with encryption in transit enabled and configured.",
        explanation: "EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMREncryptionInTransit,
    },
    {
        info: "The EMR cluster does not have Kerberos enabled.",
        explanation: "The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRKerberosEnabled,
    },
    {
        info: "The EMR cluster does not use a security configuration with local disk encryption enabled.",
        explanation: "Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRLocalDiskEncryption,
    },
    {
        info: "The event bus policy allows for open access.",
        explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.\n\nExample threat: A global internet-based actor who has discovered the Event Bridge event bus (e.g. Endpoint ID) can put arbitrary events onto the bus, which may lead to which could be processed by the prototype possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype',
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eventbridge.EventBusOpenAccess,
    },
    {
        info: "The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.",
        explanation: "Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueEncryptedCloudWatchLogs,
    },
    {
        info: "The Glue job does not have use a security configuration with job bookmark encryption enabled.",
        explanation: "Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueJobBookmarkEncrypted,
    },
    {
        info: "The IAM user, role, or group uses AWS managed policies.",
        explanation: "An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Policy::<policy> for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoManagedPolicies,
    },
    {
        info: "The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.",
        explanation: "Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Action::<action> for policy actions and Resource::<resource> for resources. Example: appliesTo: ['Action::s3:*'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoWildcardPermissions,
    },
    {
        info: "The IAM policy grants admin access - meaning the policy allows a principal to perform unlimited actions on any service",
        explanation: "AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.\n\nExample threat: A global internet-based actor who has successfully obtained valid keys or a session associated of the IAM Principal associated with the IAM policy can perform unlimited AWS actions on any AWS service which are exposed via the AWS API/Management Console/CLI, which may lead to broad and unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithAdminAccess,
    },
    {
        info: "The IAM policy grants full access - meaning the policy allows unlimited actions for a given AWS service",
        explanation: "Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithFullAccess,
    },
    {
        info: "The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.",
        explanation: "Enabling encryption allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataFirehoseSSE,
    },
    {
        info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
        explanation: "Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamDefaultKeyWhenSSE,
    },
    {
        info: "The Kinesis Data Stream does not have server-side encryption enabled.",
        explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamSSE,
    },
    {
        info: "The KMS Symmetric key does not have automatic key rotation enabled.",
        explanation: "KMS key rotation allow a system to set a rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kms.KMSBackingKeyRotationEnabled,
    },
    {
        info: "The Lambda function permission grants public access",
        explanation: "Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.\n\nExample threat: A global internet-based actor who has discovered the Lambda function name or ARN can invoke/delete/modify the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionPublicAccessProhibited,
    },
    {
        info: "The Lambda Function URL allows for public, unauthenticated access.",
        explanation: "AWS Lambda Function URLs allow you to invoke your function via a HTTPS end-point, setting the authentication to NONE allows anyone on the internet to invoke your function.\n\nExample threat: A global internet-based actor who has discovered the Lambda Function URL can invoke the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionUrlAuth,
    },
    {
        info: "The Lambda function is not VPC enabled.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaInsideVPC,
    },
    {
        info: "The non-container Lambda function is not configured to use the latest runtime version.",
        explanation: "Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaLatestVersion,
    },
    {
        info: "The MediaStore container does not define a metric policy to send metrics to CloudWatch.",
        explanation: "Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreCloudWatchMetricPolicy,
    },
    {
        info: "The MediaStore container does not define a CORS policy.",
        explanation: "Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerCORSPolicy,
    },
    {
        info: "The MediaStore container does not define a container policy.",
        explanation: "Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerHasContainerPolicy,
    },
    {
        info: "The MediaStore container does not define a lifecycle policy.",
        explanation: "Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerLifecyclePolicy,
    },
    {
        info: "The MediaStore container does not require requests to use SSL.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerSSLRequestsOnly,
    },
    {
        info: "The MSK cluster uses plaintext communication between brokers.",
        explanation: "TLS communication secures data-in-transit by encrypting the connection between brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKBrokerToBrokerTLS,
    },
    {
        info: "The MSK cluster uses plaintext communication between clients and brokers.",
        explanation: "TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKClientToBrokerTLS,
    },
    {
        info: "The Neptune DB instance does have Auto Minor Version Upgrade enabled.",
        explanation: "The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.\n\nExample threat: An actor with a network path to the Neptune cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by Neptune, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterAutomaticMinorVersionUpgrade,
    },
    {
        info: "The Neptune DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterEncryptionAtRest,
    },
    {
        info: "The Neptune DB cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterIAMAuth,
    },
    {
        info: "The OpenSearch Service domain does not only grant access via allowlisted IP addresses.",
        explanation: "Using allowlisted IP addresses helps protect the domain against unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchAllowlistedIPs,
    },
    {
        info: "The OpenSearch Service domain does not use dedicated master nodes.",
        explanation: "Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchDedicatedMasterNode,
    },
    {
        info: "The OpenSearch Service domain does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchEncryptedAtRest,
    },
    {
        info: "The OpenSearch Service domain is not provisioned inside a VPC.",
        explanation: "Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.\n\nExample threat: A global internet-based actor can discover the OpenSearch Service domain that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchInVPCOnly,
    },
    {
        info: "The OpenSearch Service domain does not have node-to-node encryption enabled.",
        explanation: "Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNodeToNodeEncryption,
    },
    {
        info: "The OpenSearch Service domain does not allow for unsigned requests or anonymous access.",
        explanation: "Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.\n\nExample threat: An actor with a network path to the OpenSearch Service domain can directly access the domain without authentication, which may lead to allowing access to data hosted within the domain possibly impacting the confidentiality, integrity and availability of the data assets hosted on the OpenSearch Service domain for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNoUnsignedOrAnonymousAccess,
    },
    {
        info: "The Quicksight data sources connection is not configured to use SSL.",
        explanation: "SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.quicksight.QuicksightSSLConnections,
    },
    {
        info: "The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.AuroraMySQLPostgresIAMAuth,
    },
    {
        info: "The RDS DB instance does not have automatic minor version upgrades enabled.",
        explanation: "Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.\n\nExample threat: An actor with a network path to the RDS cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by RDS, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSAutomaticMinorVersionUpgradeEnabled,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.",
        explanation: "Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSInstanceDeletionProtectionEnabled,
    },
    {
        info: "The RDS DB instance allows public access.",
        explanation: "Amazon RDS database instances can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the RDS DB instance endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSInstancePublicAccess,
    },
    {
        info: "The RDS instance or Aurora DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSNonDefaultPort,
    },
    {
        info: "The RDS DB Security Group allows for 0.0.0.0/0 inbound access.",
        explanation: "RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.\n\nExample threat: A global internet-based actor can discover RDS DB instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public RDS DB instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSRestrictedInbound,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSStorageEncrypted,
    },
    {
        info: "The Redshift cluster does not have audit logging enabled.",
        explanation: "Audit logging helps operators troubleshoot issues and ensure security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterAuditLogging,
    },
    {
        info: "The Redshift cluster does not have encryption or audit logging enabled.",
        explanation: "To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterConfiguration,
    },
    {
        info: "The Redshift cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterEncryptionAtRest,
    },
    {
        info: "The Redshift cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover a RedShift cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterInVPC,
    },
    {
        info: "The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured.",
        explanation: "Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterMaintenanceSettings,
    },
    {
        info: "The Redshift cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultPort,
    },
    {
        info: 'The Redshift cluster uses the default "awsuser" username.',
        explanation: 'Using a custom user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultUsername,
    },
    {
        info: "The Redshift cluster allows public access.",
        explanation: "Amazon Redshift clusters can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the Redshift cluster endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterPublicAccess,
    },
    {
        info: "The Redshift cluster does not have version upgrade enabled.",
        explanation: "Version Upgrade must enabled on the cluster in order to automatically receive upgrades during the maintenance window.\n\nExample threat: An actor with a network path to the Redshift cluster can attempt to take advantage of a known vulnerability in a component exposed by Redshift, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterVersionUpgrade,
    },
    {
        info: "The Redshift cluster does not have enhanced VPC routing enabled.",
        explanation: "Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftEnhancedVPCRoutingEnabled,
    },
    {
        info: "The Redshift cluster does not require TLS/SSL encryption.",
        explanation: "Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftRequireTlsSSL,
    },
    {
        info: "The S3 bucket does not prohibit public access through bucket level settings.",
        explanation: "Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read or write can read or write data to or from the S3 bucket, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketLevelPublicAccessProhibited,
    },
    {
        info: "The S3 Bucket does not have server access logs enabled.",
        explanation: "Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketLoggingEnabled,
    },
    {
        info: "The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read can read data from the S3 bucket, which may lead to possibly impacting the confidentiality of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicReadProhibited,
    },
    {
        info: "The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public write can write data to, or overwrite data within the S3 bucket, which may lead to possibly impacting the integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicWriteProhibited,
    },
    {
        info: "The S3 Bucket does not have default server-side encryption enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketServerSideEncryptionEnabled,
    },
    {
        info: "The S3 Bucket or bucket policy does not require requests to use SSL/TLS.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketSSLRequestsOnly,
    },
    {
        info: "The S3 Bucket is not encrypted with a KMS Key by default.",
        explanation: "Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3DefaultEncryptionKMS,
    },
    {
        info: "The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.",
        explanation: "An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.\n\nExample threat: A global internet-based actor who has discovered a S3 hosted website can discover prototype web assets that are hosted on the website, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3WebBucketOAIAccess,
    },
    {
        info: "The SageMaker resource endpoint is not encrypted with a KMS key.",
        explanation: "Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerEndpointConfigurationKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance does not have an encrypted storage volume.",
        explanation: "Encrypting storage volumes helps protect SageMaker data-at-rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInstanceKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance is not provisioned inside a VPC.",
        explanation: "Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInVPC,
    },
    {
        info: "The SageMaker notebook does not disable direct internet access.",
        explanation: "By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookNoDirectInternetAccess,
    },
    {
        info: "The secret does not have automatic rotation scheduled.",
        explanation: "Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerRotationEnabled,
    },
    {
        info: "The secret is not encrypted with a KMS Customer managed key.",
        explanation: "To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerUsingKMSKey,
    },
    {
        info: "The SNS topic does not have KMS encryption enabled.",
        explanation: "To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS). Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSEncryptedKMS,
    },
    {
        info: "The SNS Topic does not require publishers to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns:Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSTopicSSLPublishOnly,
    },
    {
        info: "The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk-nag rule suppression indicating it is a DLQ.",
        explanation: "Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueDLQ,
    },
    {
        info: "The SQS Queue does not have server-side encryption enabled.",
        explanation: "Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSE,
    },
    {
        info: "The SQS queue does not require requests to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSLRequestsOnly,
    },
    {
        info: "The Timestream database does not use a Customer Managed KMS Key for at rest encryption.",
        explanation: "All Timestream tables in a database are encrypted at rest by default using an AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an periodic basis.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.timestream.TimestreamDatabaseCustomerManagedKey,
    },
    {
        info: "The VPC's default security group allows inbound or outbound traffic.",
        explanation: "When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) that have public IP addresses and allow ingress to all internet address or move laterally to non-public VPC-attached instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCDefaultSecurityGroupClosed,
    },
    {
        info: "A Network ACL or Network ACL entry has been implemented.",
        explanation: "Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoNACLs,
    },
    {
        info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').",
        explanation: "Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoUnrestrictedRouteToIGW,
    },
    {
        info: "The subnet auto-assigns public IP addresses.",
        explanation: "Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) within the subnet in question that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCSubnetAutoAssignPublicIpDisabled,
    },
];
//# sourceMappingURL=data:application/json;base64,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