"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function toCfnTags(tags) {
    return Object.keys(tags).map(key => {
        return { Key: key, Value: tags[key] };
    });
}
function hasTags(expectedTags) {
    return (props) => {
        try {
            const tags = props.Tags;
            const actualTags = tags.filter((tag) => {
                for (const expectedTag of expectedTags) {
                    if (assert_1.isSuperObject(expectedTag, tag)) {
                        return true;
                    }
                    else {
                        continue;
                    }
                }
                // no values in array so expecting empty
                return false;
            });
            return actualTags.length === expectedTags.length;
        }
        catch (e) {
            // tslint:disable:no-console
            console.error('Tags are incorrect');
            console.error('found tags ', props.Tags);
            console.error('expected tags ', expectedTags);
            // tslint:enable:no-console
            throw e;
        }
    };
}
module.exports = {
    "When creating a VPC": {
        "with the default CIDR range": {
            "vpc.vpcId returns a token to the VPC ID"(test) {
                const stack = getTestStack();
                const vpc = new lib_1.Vpc(stack, 'TheVPC');
                test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
                test.done();
            },
            "it uses the correct network range"(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                    CidrBlock: lib_1.Vpc.DEFAULT_CIDR_RANGE,
                    EnableDnsHostnames: true,
                    EnableDnsSupport: true,
                    InstanceTenancy: lib_1.DefaultInstanceTenancy.DEFAULT,
                }));
                test.done();
            },
            'the Name tag is defaulted to path'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                test.done();
            },
        },
        "with all of the properties set, it successfully sets the correct VPC properties"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: "192.168.0.0/16",
                enableDnsHostnames: false,
                enableDnsSupport: false,
                defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
                EnableDnsHostnames: false,
                EnableDnsSupport: false,
                InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            }));
            test.done();
        },
        "contains the correct number of subnets"(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const zones = stack.availabilityZones.length;
            test.equal(vpc.publicSubnets.length, zones);
            test.equal(vpc.privateSubnets.length, zones);
            test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
            test.done();
        },
        "with only isolated subnets, the VPC should not contain an IGW or NAT Gateways"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::InternetGateway"));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                MapPublicIpOnLaunch: false
            }));
            test.done();
        },
        "with no private subnets, the VPC should have an IGW but no NAT Gateways"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            test.done();
        },
        "with private subnets and custom networkAcl."(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                        name: 'private',
                    }
                ]
            });
            const nacl1 = new lib_1.NetworkAcl(stack, 'myNACL1', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PRIVATE },
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSEgress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.EGRESS,
                cidr: lib_1.AclCidr.ipv4('10.0.0.0/16'),
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSIngress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.INGRESS,
                cidr: lib_1.AclCidr.anyIpv4(),
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAcl', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAclEntry', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SubnetNetworkAclAssociation', 3));
            test.done();
        },
        "with no subnets defined, the VPC should have an IGW, and a NAT Gateway per AZ"(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {});
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", zones));
            test.done();
        },
        "with subnets and reserved subnets defined, VPC subnet count should not contain reserved subnets "(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            test.done();
        },
        "with reserved subnets, any other subnets should not have cidrBlock from within reserved space"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 24,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    }
                ],
                maxAzs: 3
            });
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 3; i < 6; i++) {
                assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 6; i < 9; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            test.done();
        },
        "with custom subnets, the VPC should have the right number of subnets, an IGW, and a NAT Gateway per AZ"(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", zones));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        "with custom subents and natGateways = 2 there should be only two NATGW"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                natGateways: 2,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::InternetGateway", 1));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 2));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        "with enableDnsHostnames enabled but enableDnsSupport disabled, should throw an Error"(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'TheVPC', {
                enableDnsHostnames: true,
                enableDnsSupport: false
            }));
            test.done();
        },
        "with public subnets MapPublicIpOnLaunch is true"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 1,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    }
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 1));
            assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::NatGateway"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                MapPublicIpOnLaunch: true
            }));
            test.done();
        },
        "maxAZs defaults to 3 if unset"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 6));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i * 32}.0/19`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        "with maxAZs set to 2"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', { maxAzs: 2 });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 4));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 4));
            for (let i = 0; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::Subnet", {
                    CidrBlock: `10.0.${i * 64}.0/18`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        "with natGateway set to 1"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 1,
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Subnet", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::Route", 6));
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::EC2::Route", {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway subnets defined'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'egress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'egress'
                },
            });
            assert_1.expect(stack).to(assert_1.countResources("AWS::EC2::NatGateway", 3));
            for (let i = 1; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                        Key: 'Name',
                        Value: `VPC/egressSubnet${i}`,
                    }, {
                        Key: 'aws-cdk:subnet-name',
                        Value: 'egress',
                    }])));
            }
            test.done();
        },
        'natGateways = 0 requires there to be no PRIVATE subnets'(test) {
            const stack = getTestStack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'VPC', {
                    natGateways: 0,
                });
            }, /make sure you don't configure any PRIVATE subnets/);
            test.done();
        },
        'with mis-matched nat and subnet configs it throws'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'notthere',
                },
            }));
            test.done();
        },
        'with a vpn gateway'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                vpnGateway: true,
                vpnGatewayAsn: 65000
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGateway', {
                AmazonSideAsn: 65000,
                Type: 'ipsec.1'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCGatewayAttachment', {
                VpcId: {
                    Ref: 'VPCB9E5F0B4'
                },
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on private and isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'Private' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.PRIVATE
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'fails when specifying vpnConnections with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnConnections: {
                    VpnConnection: {
                        asn: 65000,
                        ip: '192.0.2.1'
                    }
                }
            }), /`vpnConnections`.+`vpnGateway`.+false/);
            test.done();
        },
        'fails when specifying vpnGatewayAsn with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnGatewayAsn: 65000,
            }), /`vpnGatewayAsn`.+`vpnGateway`.+false/);
            test.done();
        },
        'Subnets have a defaultChild'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            test.ok(vpc.publicSubnets[0].node.defaultChild instanceof lib_1.CfnSubnet);
            test.done();
        },
        'CIDR cannot be a Token'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'Vpc', {
                    cidr: core_1.Lazy.stringValue({ produce: () => 'abc' })
                });
            }, /property must be a concrete CIDR string/);
            test.done();
        },
    },
    'NAT instances': {
        'Can configure NAT instances instead of NAT gateways'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1'
                    })
                })
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 3));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                ImageId: "ami-1",
                InstanceType: "q86.mega",
                SourceDestCheck: false,
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Route', {
                RouteTableId: { Ref: "TheVPCPrivateSubnet1RouteTableF6513BC2" },
                DestinationCidrBlock: "0.0.0.0/0",
                InstanceId: { Ref: "TheVPCPublicSubnet1NatInstanceCC514192" }
            }));
            test.done();
        },
        'natGateways controls amount of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1'
                    })
                }),
                natGateways: 1
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 1));
            test.done();
        },
    },
    'Network ACL association': {
        'by default uses default ACL reference'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { "Fn::GetAtt": ["TheVPCPublicSubnet1Subnet770D4FF2", "NetworkAclAssociationId"] }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
        'if ACL is replaced new ACL reference is returned'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            // WHEN
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            new lib_1.NetworkAcl(stack, 'ACL', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PUBLIC }
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { Ref: "ACLDBD1BB49" }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
    },
    "When creating a VPC with a custom CIDR range": {
        "vpc.vpcCidrBlock is the correct network range"(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
                CidrBlock: '192.168.0.0/16'
            }));
            test.done();
        }
    },
    'When tagging': {
        'VPC propagated tags will be on subnet, IGW, routetables, NATGW'(test) {
            const stack = getTestStack();
            const tags = {
                VpcType: 'Good',
            };
            const noPropTags = {
                BusinessUnit: 'Marketing',
            };
            const allTags = Object.assign(Object.assign({}, tags), noPropTags);
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            // overwrite to set propagate
            vpc.node.applyAspect(new core_1.Tag('BusinessUnit', 'Marketing', { includeResourceTypes: [lib_1.CfnVPC.CFN_RESOURCE_TYPE_NAME] }));
            vpc.node.applyAspect(new core_1.Tag('VpcType', 'Good'));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", hasTags(toCfnTags(allTags))));
            const taggables = ['Subnet', 'InternetGateway', 'NatGateway', 'RouteTable'];
            const propTags = toCfnTags(tags);
            const noProp = toCfnTags(noPropTags);
            for (const resource of taggables) {
                assert_1.expect(stack).to(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(propTags)));
                assert_1.expect(stack).notTo(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(noProp)));
            }
            test.done();
        },
        'Subnet Name will propagate to route tables and NATGW'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            for (const subnet of vpc.publicSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::NatGateway', hasTags([tag])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            for (const subnet of vpc.privateSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            test.done();
        },
        'Tags can be added after the Vpc is created with `vpc.tags.setTag(...)`'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const tag = { Key: 'Late', Value: 'Adder' };
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            vpc.node.applyAspect(new core_1.Tag(tag.Key, tag.Value));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            test.done();
        },
    },
    'subnet selection': {
        'selecting default subnets returns the private ones'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select public subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PUBLIC });
            // THEN
            test.deepEqual(subnetIds, vpc.publicSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select isolated subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.ISOLATED });
            // THEN
            test.deepEqual(subnetIds, vpc.isolatedSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select subnets by name'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetGroupName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'subnetName is an alias for subnetGroupName (backwards compat)'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'selecting default subnets in a VPC with only isolated subnets returns the isolateds'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                isolatedSubnetIds: ['iso-1', 'iso-2', 'iso-3'],
                isolatedSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['iso-1', 'iso-2', 'iso-3']);
            test.done();
        },
        'selecting default subnets in a VPC with only public subnets returns the publics'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['pub-1', 'pub-2', 'pub-3']);
            test.done();
        },
        'selecting subnets by name fails if the name is unknown'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            test.throws(() => {
                vpc.selectSubnets({ subnetGroupName: 'Toot' });
            }, /There are no subnet groups with name 'Toot' in this VPC. Available names: Public,Private/);
            test.done();
        },
        'select subnets with az restriction'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
                maxAzs: 1,
                subnetConfiguration: [
                    { name: 'lb', subnetType: lib_1.SubnetType.PUBLIC },
                    { name: 'app', subnetType: lib_1.SubnetType.PRIVATE },
                    { name: 'db', subnetType: lib_1.SubnetType.PRIVATE },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ onePerAz: true });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], vpc.privateSubnets[0].subnetId);
            test.done();
        },
        'select explicitly defined subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            const subnet = new lib_1.PrivateSubnet(stack, 'Subnet', {
                availabilityZone: vpc.availabilityZones[0],
                cidrBlock: '10.0.0.0/28',
                vpcId: vpc.vpcId
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnets: [subnet] });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], subnet.subnetId);
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC52cGMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnZwYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQW9IO0FBQ3BILHdDQUE0RDtBQUU1RCxnQ0FDcUg7QUFnOEJySCxTQUFTLFlBQVk7SUFDbkIsT0FBTyxJQUFJLFlBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0FBQ3RHLENBQUM7QUFFRCxTQUFTLFNBQVMsQ0FBQyxJQUFTO0lBQzFCLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUUsR0FBRyxDQUFDLEVBQUU7UUFDbEMsT0FBTyxFQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBQyxDQUFDO0lBQ3RDLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMsT0FBTyxDQUFDLFlBQWlEO0lBQ2hFLE9BQU8sQ0FBQyxLQUFVLEVBQUUsRUFBRTtRQUNwQixJQUFJO1lBQ0YsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztZQUN4QixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFFLENBQUMsR0FBaUMsRUFBRSxFQUFFO2dCQUNwRSxLQUFLLE1BQU0sV0FBVyxJQUFJLFlBQVksRUFBRTtvQkFDdEMsSUFBSSxzQkFBYSxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUMsRUFBRTt3QkFDbkMsT0FBTyxJQUFJLENBQUM7cUJBQ2I7eUJBQU07d0JBQ0wsU0FBUztxQkFDVjtpQkFDRjtnQkFDRCx3Q0FBd0M7Z0JBQ3hDLE9BQU8sS0FBSyxDQUFDO1lBQ2YsQ0FBQyxDQUFDLENBQUM7WUFDSCxPQUFPLFVBQVUsQ0FBQyxNQUFNLEtBQUssWUFBWSxDQUFDLE1BQU0sQ0FBQztTQUNsRDtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsNEJBQTRCO1lBQzVCLE9BQU8sQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQztZQUNwQyxPQUFPLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDekMsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxZQUFZLENBQUMsQ0FBQztZQUM5QywyQkFBMkI7WUFDM0IsTUFBTSxDQUFDLENBQUM7U0FDVDtJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFqK0JELGlCQUFTO0lBQ1AscUJBQXFCLEVBQUU7UUFDckIsNkJBQTZCLEVBQUU7WUFFN0IseUNBQXlDLENBQUMsSUFBVTtnQkFDbEQsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7Z0JBQzdCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztnQkFDckMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFDLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFFLENBQUM7Z0JBQ3BFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO2dCQUM1QyxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztnQkFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO2dCQUN6QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZUFBZSxFQUFFO29CQUM3QyxTQUFTLEVBQUUsU0FBRyxDQUFDLGtCQUFrQjtvQkFDakMsa0JBQWtCLEVBQUUsSUFBSTtvQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtvQkFDdEIsZUFBZSxFQUFFLDRCQUFzQixDQUFDLE9BQU87aUJBQ2hELENBQUMsQ0FBQyxDQUFDO2dCQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFDRCxtQ0FBbUMsQ0FBQyxJQUFVO2dCQUM1QyxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztnQkFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO2dCQUN6QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUNkLHFCQUFZLENBQUMsZUFBZSxFQUMxQixPQUFPLENBQUUsQ0FBRSxFQUFDLEdBQUcsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBQyxDQUFFLENBQUMsQ0FBQyxDQUNoRCxDQUFDO2dCQUNGLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQ2QscUJBQVksQ0FBQywyQkFBMkIsRUFDdEMsT0FBTyxDQUFFLENBQUUsRUFBQyxHQUFHLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUMsQ0FBRSxDQUFDLENBQUMsQ0FDaEQsQ0FBQztnQkFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBRUY7UUFFRCxpRkFBaUYsQ0FBQyxJQUFVO1lBQzFGLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3ZCLElBQUksRUFBRSxnQkFBZ0I7Z0JBQ3RCLGtCQUFrQixFQUFFLEtBQUs7Z0JBQ3pCLGdCQUFnQixFQUFFLEtBQUs7Z0JBQ3ZCLHNCQUFzQixFQUFFLDRCQUFzQixDQUFDLFNBQVM7YUFDekQsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGVBQWUsRUFBRTtnQkFDN0MsU0FBUyxFQUFFLGdCQUFnQjtnQkFDM0Isa0JBQWtCLEVBQUUsS0FBSztnQkFDekIsZ0JBQWdCLEVBQUUsS0FBSztnQkFDdkIsZUFBZSxFQUFFLDRCQUFzQixDQUFDLFNBQVM7YUFDbEQsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsd0NBQXdDLENBQUMsSUFBVTtZQUNqRCxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDckMsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQztZQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQzVDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDN0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7WUFDcEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELCtFQUErRSxDQUFDLElBQVU7WUFDeEYsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDdkIsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVE7d0JBQy9CLElBQUksRUFBRSxVQUFVO3FCQUNqQjtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUNILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMscUJBQVksQ0FBQywyQkFBMkIsQ0FBQyxDQUFDLENBQUM7WUFDL0QsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQztZQUMxRCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELG1CQUFtQixFQUFFLEtBQUs7YUFDM0IsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQseUVBQXlFLENBQUMsSUFBVTtZQUNsRixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUN2QixtQkFBbUIsRUFBRTtvQkFDbkI7d0JBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTTt3QkFDN0IsSUFBSSxFQUFFLFFBQVE7cUJBQ2Y7b0JBQ0Q7d0JBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUTt3QkFDL0IsSUFBSSxFQUFFLFVBQVU7cUJBQ2pCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQztZQUMxRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsNkNBQTZDLENBQUMsSUFBVTtZQUN0RCxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNuQyxtQkFBbUIsRUFBRTtvQkFDbkI7d0JBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTTt3QkFDN0IsSUFBSSxFQUFFLFFBQVE7cUJBQ2Y7b0JBQ0Q7d0JBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTzt3QkFDOUIsSUFBSSxFQUFFLFNBQVM7cUJBQ2hCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsTUFBTSxLQUFLLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7Z0JBQzdDLEdBQUc7Z0JBQ0gsZUFBZSxFQUFFLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTyxFQUFFO2FBQ3BELENBQUMsQ0FBQztZQUVILElBQUkscUJBQWUsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQzNDLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixVQUFVLEVBQUUsR0FBRztnQkFDZixPQUFPLEVBQUUsZ0JBQVUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUMvQixTQUFTLEVBQUUsc0JBQWdCLENBQUMsTUFBTTtnQkFDbEMsSUFBSSxFQUFFLGFBQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO2FBQ2xDLENBQUMsQ0FBQztZQUVILElBQUkscUJBQWUsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7Z0JBQzVDLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixVQUFVLEVBQUUsR0FBRztnQkFDZixPQUFPLEVBQUUsZ0JBQVUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUMvQixTQUFTLEVBQUUsc0JBQWdCLENBQUMsT0FBTztnQkFDbkMsSUFBSSxFQUFFLGFBQU8sQ0FBQyxPQUFPLEVBQUU7YUFDeEIsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLHNCQUFzQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDNUQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLHVDQUF1QyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0UsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELCtFQUErRSxDQUFDLElBQVU7WUFDeEYsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQztZQUM3QyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUcsQ0FBQyxDQUFDO1lBQzlCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQywyQkFBMkIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pFLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ2hFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxrR0FBa0csQ0FBQyxJQUFVO1lBQzNHLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3ZCLElBQUksRUFBRSxhQUFhO2dCQUNuQixtQkFBbUIsRUFBRTtvQkFDbkI7d0JBQ0UsUUFBUSxFQUFFLEVBQUU7d0JBQ1osVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTTt3QkFDN0IsSUFBSSxFQUFFLFFBQVE7cUJBQ2Y7b0JBQ0Q7d0JBQ0UsUUFBUSxFQUFFLEVBQUU7d0JBQ1osSUFBSSxFQUFFLFVBQVU7d0JBQ2hCLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU87d0JBQzlCLFFBQVEsRUFBRSxJQUFJO3FCQUNmO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxLQUFLO3dCQUNYLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVE7cUJBQ2hDO2lCQUNGO2dCQUNELE1BQU0sRUFBRSxDQUFDO2FBQ1YsQ0FBQyxDQUFDO1lBQ0gsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELCtGQUErRixDQUFDLElBQVU7WUFDeEcsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDdkIsSUFBSSxFQUFFLGFBQWE7Z0JBQ25CLG1CQUFtQixFQUFFO29CQUNuQjt3QkFDRSxRQUFRLEVBQUUsRUFBRTt3QkFDWixJQUFJLEVBQUUsU0FBUzt3QkFDZixVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxNQUFNO3FCQUM5QjtvQkFDRDt3QkFDRSxRQUFRLEVBQUUsRUFBRTt3QkFDWixJQUFJLEVBQUUsVUFBVTt3QkFDaEIsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTzt3QkFDOUIsUUFBUSxFQUFFLElBQUk7cUJBQ2Y7b0JBQ0Q7d0JBQ0UsUUFBUSxFQUFFLEVBQUU7d0JBQ1osSUFBSSxFQUFFLEtBQUs7d0JBQ1gsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTztxQkFDL0I7aUJBQ0Y7Z0JBQ0QsTUFBTSxFQUFFLENBQUM7YUFDVixDQUFDLENBQUM7WUFDSCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUMxQixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7b0JBQ2hELFNBQVMsRUFBRSxRQUFRLENBQUMsT0FBTztpQkFDNUIsQ0FBQyxDQUFDLENBQUM7YUFDTDtZQUNELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUU7Z0JBQzFCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtvQkFDbkQsU0FBUyxFQUFFLFFBQVEsQ0FBQyxPQUFPO2lCQUM1QixDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDMUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUNoRCxTQUFTLEVBQUUsUUFBUSxDQUFDLE9BQU87aUJBQzVCLENBQUMsQ0FBQyxDQUFDO2FBQ0w7WUFDRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsd0dBQXdHLENBQUMsSUFBVTtZQUNqSCxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDO1lBQzdDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3ZCLElBQUksRUFBRSxhQUFhO2dCQUNuQixtQkFBbUIsRUFBRTtvQkFDbkI7d0JBQ0UsUUFBUSxFQUFFLEVBQUU7d0JBQ1osSUFBSSxFQUFFLFNBQVM7d0JBQ2YsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTTtxQkFDOUI7b0JBQ0Q7d0JBQ0UsUUFBUSxFQUFFLEVBQUU7d0JBQ1osSUFBSSxFQUFFLGFBQWE7d0JBQ25CLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU87cUJBQy9CO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxLQUFLO3dCQUNYLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVE7cUJBQ2hDO2lCQUNGO2dCQUNELE1BQU0sRUFBRSxDQUFDO2FBQ1YsQ0FBQyxDQUFDO1lBQ0gsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLHNCQUFzQixFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDaEUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEQsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDMUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUNoRCxTQUFTLEVBQUUsUUFBUSxDQUFDLE9BQU87aUJBQzVCLENBQUMsQ0FBQyxDQUFDO2FBQ0w7WUFDRCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUMxQixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7b0JBQ2hELFNBQVMsRUFBRSxVQUFVLENBQUMsR0FBRyxFQUFFLEtBQUs7aUJBQ2pDLENBQUMsQ0FBQyxDQUFDO2FBQ0w7WUFDRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsd0VBQXdFLENBQUMsSUFBVTtZQUNqRixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUN2QixJQUFJLEVBQUUsYUFBYTtnQkFDbkIsV0FBVyxFQUFFLENBQUM7Z0JBQ2QsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU07cUJBQzlCO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxhQUFhO3dCQUNuQixVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxPQUFPO3FCQUMvQjtvQkFDRDt3QkFDRSxRQUFRLEVBQUUsRUFBRTt3QkFDWixJQUFJLEVBQUUsS0FBSzt3QkFDWCxVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxRQUFRO3FCQUNoQztpQkFDRjtnQkFDRCxNQUFNLEVBQUUsQ0FBQzthQUNWLENBQUMsQ0FBQztZQUNILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQywyQkFBMkIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pFLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxzQkFBc0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzVELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3hELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUU7Z0JBQzFCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtvQkFDaEQsU0FBUyxFQUFFLFFBQVEsQ0FBQyxPQUFPO2lCQUM1QixDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDMUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUNoRCxTQUFTLEVBQUUsVUFBVSxDQUFDLEdBQUcsRUFBRSxLQUFLO2lCQUNqQyxDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHNGQUFzRixDQUFDLElBQVU7WUFDL0YsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUN6QyxrQkFBa0IsRUFBRSxJQUFJO2dCQUN4QixnQkFBZ0IsRUFBRSxLQUFLO2FBQ3hCLENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELGlEQUFpRCxDQUFDLElBQVU7WUFDMUQsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDcEIsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU07cUJBQzlCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQztZQUMxRCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELG1CQUFtQixFQUFFLElBQUk7YUFDMUIsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsK0JBQStCLENBQUMsSUFBVTtZQUN4QyxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDdEIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdkQsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDMUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUNoRCxTQUFTLEVBQUUsUUFBUSxDQUFDLEdBQUcsRUFBRSxPQUFPO2lCQUNqQyxDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRTtnQkFDbkQsb0JBQW9CLEVBQUUsV0FBVztnQkFDakMsWUFBWSxFQUFFLEVBQUc7YUFDbEIsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsc0JBQXNCLENBQUMsSUFBVTtZQUMvQixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDckMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdkQsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtnQkFDMUIsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUNoRCxTQUFTLEVBQUUsUUFBUSxDQUFDLEdBQUcsRUFBRSxPQUFPO2lCQUNqQyxDQUFDLENBQUMsQ0FBQzthQUNMO1lBQ0QsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRTtnQkFDbkQsb0JBQW9CLEVBQUUsV0FBVztnQkFDakMsWUFBWSxFQUFFLEVBQUc7YUFDbEIsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsMEJBQTBCLENBQUMsSUFBVTtZQUNuQyxNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUNwQixXQUFXLEVBQUUsQ0FBQzthQUNmLENBQUMsQ0FBQztZQUNILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3hELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3ZELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxzQkFBc0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzVELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsaUJBQWlCLEVBQUU7Z0JBQ25ELG9CQUFvQixFQUFFLFdBQVc7Z0JBQ2pDLFlBQVksRUFBRSxFQUFHO2FBQ2xCLENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELGlDQUFpQyxDQUFDLElBQVU7WUFDMUMsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDcEIsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU07cUJBQzlCO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxRQUFRO3dCQUNkLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU07cUJBQzlCO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU87cUJBQy9CO2lCQUNGO2dCQUNELGlCQUFpQixFQUFFO29CQUNqQixlQUFlLEVBQUUsUUFBUTtpQkFDMUI7YUFDRixDQUFDLENBQUM7WUFDSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsc0JBQXNCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1RCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUMxQixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUUsT0FBTyxDQUFDLENBQUM7d0JBQ3pELEdBQUcsRUFBRSxNQUFNO3dCQUNYLEtBQUssRUFBRSxtQkFBbUIsQ0FBQyxFQUFFO3FCQUM5QixFQUFFO3dCQUNDLEdBQUcsRUFBRSxxQkFBcUI7d0JBQzFCLEtBQUssRUFBRSxRQUFRO3FCQUNsQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDUDtZQUNELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx5REFBeUQsQ0FBQyxJQUFVO1lBQ2xFLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7b0JBQ3BCLFdBQVcsRUFBRSxDQUFDO2lCQUNmLENBQUMsQ0FBQztZQUNMLENBQUMsRUFBRSxtREFBbUQsQ0FBQyxDQUFDO1lBQ3hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUVkLENBQUM7UUFFRCxtREFBbUQsQ0FBQyxJQUFVO1lBQzVELE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDdEMsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU07cUJBQzlCO29CQUNEO3dCQUNFLFFBQVEsRUFBRSxFQUFFO3dCQUNaLElBQUksRUFBRSxTQUFTO3dCQUNmLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU87cUJBQy9CO2lCQUNGO2dCQUNELGlCQUFpQixFQUFFO29CQUNqQixlQUFlLEVBQUUsVUFBVTtpQkFDNUI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxvQkFBb0IsQ0FBQyxJQUFVO1lBQzdCLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ3BCLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixhQUFhLEVBQUUsS0FBSzthQUNyQixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUU7Z0JBQ3BELGFBQWEsRUFBRSxLQUFLO2dCQUNwQixJQUFJLEVBQUUsU0FBUzthQUNoQixDQUFDLENBQUMsQ0FBQztZQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDOUQsS0FBSyxFQUFFO29CQUNMLEdBQUcsRUFBRSxhQUFhO2lCQUNuQjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osR0FBRyxFQUFFLHVCQUF1QjtpQkFDN0I7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxzQ0FBc0MsRUFBRTtnQkFDcEUsYUFBYSxFQUFFO29CQUNiO3dCQUNFLEdBQUcsRUFBRSxxQ0FBcUM7cUJBQzNDO29CQUNEO3dCQUNFLEdBQUcsRUFBRSxxQ0FBcUM7cUJBQzNDO29CQUNEO3dCQUNFLEdBQUcsRUFBRSxxQ0FBcUM7cUJBQzNDO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixHQUFHLEVBQUUsdUJBQXVCO2lCQUM3QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELDhEQUE4RCxDQUFDLElBQVU7WUFDdkUsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDcEIsbUJBQW1CLEVBQUU7b0JBQ25CLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUU7b0JBQ2pELEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUU7aUJBQ3REO2dCQUNELFVBQVUsRUFBRSxJQUFJO2dCQUNoQixtQkFBbUIsRUFBRTtvQkFDbkI7d0JBQ0UsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUTtxQkFDaEM7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0NBQXNDLEVBQUU7Z0JBQ3BFLGFBQWEsRUFBRTtvQkFDYjt3QkFDRSxHQUFHLEVBQUUsc0NBQXNDO3FCQUM1QztvQkFDRDt3QkFDRSxHQUFHLEVBQUUsc0NBQXNDO3FCQUM1QztvQkFDRDt3QkFDRSxHQUFHLEVBQUUsc0NBQXNDO3FCQUM1QztpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osR0FBRyxFQUFFLHVCQUF1QjtpQkFDN0I7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCwwRUFBMEUsQ0FBQyxJQUFVO1lBQ25GLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ3BCLG1CQUFtQixFQUFFO29CQUNuQixFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFO29CQUNqRCxFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFO29CQUNuRCxFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFO2lCQUN0RDtnQkFDRCxVQUFVLEVBQUUsSUFBSTtnQkFDaEIsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU87cUJBQy9CO29CQUNEO3dCQUNFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVE7cUJBQ2hDO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNDQUFzQyxFQUFFO2dCQUNwRSxhQUFhLEVBQUU7b0JBQ2I7d0JBQ0UsR0FBRyxFQUFFLHFDQUFxQztxQkFDM0M7b0JBQ0Q7d0JBQ0UsR0FBRyxFQUFFLHFDQUFxQztxQkFDM0M7b0JBQ0Q7d0JBQ0UsR0FBRyxFQUFFLHFDQUFxQztxQkFDM0M7b0JBQ0Q7d0JBQ0UsR0FBRyxFQUFFLHNDQUFzQztxQkFDNUM7b0JBQ0Q7d0JBQ0UsR0FBRyxFQUFFLHNDQUFzQztxQkFDNUM7b0JBQ0Q7d0JBQ0UsR0FBRyxFQUFFLHNDQUFzQztxQkFDNUM7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLEdBQUcsRUFBRSx1QkFBdUI7aUJBQzdCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsbUVBQW1FLENBQUMsSUFBVTtZQUM1RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7Z0JBQzdDLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixjQUFjLEVBQUU7b0JBQ2QsYUFBYSxFQUFFO3dCQUNiLEdBQUcsRUFBRSxLQUFLO3dCQUNWLEVBQUUsRUFBRSxXQUFXO3FCQUNoQjtpQkFDRjthQUNGLENBQUMsRUFBRSx1Q0FBdUMsQ0FBQyxDQUFDO1lBRTdDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxrRUFBa0UsQ0FBQyxJQUFVO1lBQzNFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtnQkFDN0MsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLGFBQWEsRUFBRSxLQUFLO2FBQ3JCLENBQUMsRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1lBRTVDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2QkFBNkIsQ0FBQyxJQUFVO1lBQ3RDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV6QyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksWUFBWSxlQUFTLENBQUMsQ0FBQztZQUVyRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsd0JBQXdCLENBQUMsSUFBVTtZQUNqQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7b0JBQ3BCLElBQUksRUFBRSxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLEtBQUssRUFBRSxDQUFDO2lCQUNqRCxDQUFDLENBQUM7WUFDTCxDQUFDLEVBQUUseUNBQXlDLENBQUMsQ0FBQztZQUU5QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxlQUFlLEVBQUU7UUFDZixxREFBcUQsQ0FBQyxJQUFVO1lBQzlELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUU3QixPQUFPO1lBQ1AsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDdkIsa0JBQWtCLEVBQUUsaUJBQVcsQ0FBQyxRQUFRLENBQUM7b0JBQ3ZDLFlBQVksRUFBRSxJQUFJLGtCQUFZLENBQUMsVUFBVSxDQUFDO29CQUMxQyxZQUFZLEVBQUUsSUFBSSx1QkFBaUIsQ0FBQzt3QkFDbEMsV0FBVyxFQUFFLE9BQU87cUJBQ3JCLENBQUM7aUJBQ0gsQ0FBQzthQUNILENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMxRCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ2xELE9BQU8sRUFBRSxPQUFPO2dCQUNoQixZQUFZLEVBQUUsVUFBVTtnQkFDeEIsZUFBZSxFQUFFLEtBQUs7YUFDdkIsQ0FBQyxDQUFDLENBQUM7WUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9DLFlBQVksRUFBRSxFQUFFLEdBQUcsRUFBRSx3Q0FBd0MsRUFBRTtnQkFDL0Qsb0JBQW9CLEVBQUUsV0FBVztnQkFDakMsVUFBVSxFQUFFLEVBQUUsR0FBRyxFQUFFLHdDQUF3QyxFQUFFO2FBQzlELENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDhDQUE4QyxDQUFDLElBQVU7WUFDdkQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBRTdCLE9BQU87WUFDUCxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUN2QixrQkFBa0IsRUFBRSxpQkFBVyxDQUFDLFFBQVEsQ0FBQztvQkFDdkMsWUFBWSxFQUFFLElBQUksa0JBQVksQ0FBQyxVQUFVLENBQUM7b0JBQzFDLFlBQVksRUFBRSxJQUFJLHVCQUFpQixDQUFDO3dCQUNsQyxXQUFXLEVBQUUsT0FBTztxQkFDckIsQ0FBQztpQkFDSCxDQUFDO2dCQUNGLFdBQVcsRUFBRSxDQUFDO2FBQ2YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxvQkFBb0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTFELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELHlCQUF5QixFQUFFO1FBQ3pCLHVDQUF1QyxDQUFDLElBQVU7WUFDaEQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBRTdCLE9BQU87WUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztZQUNqRSxJQUFJLGdCQUFTLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDN0IsS0FBSyxFQUFHLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFZLENBQUMsNkJBQTZCO2FBQ3RFLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7Z0JBQ3BCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUU7d0JBQ04sS0FBSyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsbUNBQW1DLEVBQUUseUJBQXlCLENBQUUsRUFBRTtxQkFDNUY7aUJBQ0Y7YUFDRixFQUFFLG1CQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7WUFFeEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGtEQUFrRCxDQUFDLElBQVU7WUFDM0QsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1lBRWpFLE9BQU87WUFDUCxJQUFJLGdCQUFTLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDN0IsS0FBSyxFQUFHLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFZLENBQUMsNkJBQTZCO2FBQ3RFLENBQUMsQ0FBQztZQUNILElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUMzQixHQUFHO2dCQUNILGVBQWUsRUFBRSxFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU0sRUFBRTthQUNuRCxDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixPQUFPLEVBQUU7b0JBQ1AsTUFBTSxFQUFFO3dCQUNOLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUM7cUJBQzdCO2lCQUNGO2FBQ0YsRUFBRSxtQkFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBRXhCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELDhDQUE4QyxFQUFFO1FBQzlDLCtDQUErQyxDQUFDLElBQVU7WUFDeEQsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7WUFDckQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGVBQWUsRUFBRTtnQkFDN0MsU0FBUyxFQUFFLGdCQUFnQjthQUM1QixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUNELGNBQWMsRUFBRTtRQUNkLGdFQUFnRSxDQUFDLElBQVU7WUFDekUsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxJQUFJLEdBQUk7Z0JBQ1osT0FBTyxFQUFFLE1BQU07YUFDaEIsQ0FBQztZQUNGLE1BQU0sVUFBVSxHQUFHO2dCQUNqQixZQUFZLEVBQUUsV0FBVzthQUMxQixDQUFDO1lBQ0YsTUFBTSxPQUFPLG1DQUFRLElBQUksR0FBSyxVQUFVLENBQUMsQ0FBQztZQUUxQyxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDckMsNkJBQTZCO1lBQzdCLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksVUFBRyxDQUFDLGNBQWMsRUFBRSxXQUFXLEVBQUUsRUFBQyxvQkFBb0IsRUFBRSxDQUFDLFlBQU0sQ0FBQyxzQkFBc0IsQ0FBQyxFQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3BILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksVUFBRyxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1lBQ2pELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM3RSxNQUFNLFNBQVMsR0FBRyxDQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxZQUFZLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDNUUsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pDLE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUNyQyxLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTtnQkFDaEMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGFBQWEsUUFBUSxFQUFFLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDM0UsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLGFBQWEsUUFBUSxFQUFFLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUM3RTtZQUNELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxzREFBc0QsQ0FBQyxJQUFVO1lBQy9ELE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUNyQyxLQUFLLE1BQU0sTUFBTSxJQUFJLEdBQUcsQ0FBQyxhQUFhLEVBQUU7Z0JBQ3RDLE1BQU0sR0FBRyxHQUFHLEVBQUMsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUMsQ0FBQztnQkFDbkQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2RSxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDeEU7WUFDRCxLQUFLLE1BQU0sTUFBTSxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUU7Z0JBQ3ZDLE1BQU0sR0FBRyxHQUFHLEVBQUMsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUMsQ0FBQztnQkFDbkQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3hFO1lBQ0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHdFQUF3RSxDQUFDLElBQVU7WUFDakYsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFFN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3JDLE1BQU0sR0FBRyxHQUFHLEVBQUMsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFDLENBQUM7WUFDMUMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxxQkFBWSxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNuRSxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFVBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ2xELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxlQUFlLEVBQUUsT0FBTyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsa0JBQWtCLEVBQUU7UUFDbEIsb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRWxDLE9BQU87WUFDUCxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBRTFDLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ25FLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwyQkFBMkIsQ0FBQyxJQUFVO1lBQ3BDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFbEMsT0FBTztZQUNQLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztZQUUzRSxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUVsRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNkJBQTZCLENBQUMsSUFBVTtZQUN0QyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDaEMsbUJBQW1CLEVBQUU7b0JBQ25CLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUU7b0JBQ2pELEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUU7aUJBQ3REO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztZQUU3RSxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsR0FBRyxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUVwRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNEJBQTRCLENBQUMsSUFBVTtZQUNyQyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDaEMsbUJBQW1CLEVBQUU7b0JBQ25CLEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUU7b0JBQ2pELEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7b0JBQ3hELEVBQUUsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUU7aUJBQzNEO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsZUFBZSxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7WUFFN0UsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDbkUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELCtEQUErRCxDQUFDLElBQVU7WUFDeEUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLFlBQVksRUFBRSxDQUFDO1lBQzdCLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ2hDLG1CQUFtQixFQUFFO29CQUNuQixFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFO29CQUNqRCxFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsY0FBYyxFQUFFO29CQUN4RCxFQUFFLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFO2lCQUMzRDthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLFVBQVUsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1lBRXhFLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ25FLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxxRkFBcUYsQ0FBQyxJQUFVO1lBQzlGLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sR0FBRyxHQUFHLFNBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUM5QyxLQUFLLEVBQUUsVUFBVTtnQkFDakIsaUJBQWlCLEVBQUUsQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLFNBQVMsQ0FBQztnQkFDcEQsaUJBQWlCLEVBQUUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQztnQkFDOUMsMkJBQTJCLEVBQUUsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQzthQUN0RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBRXBDLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDL0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGlGQUFpRixDQUFDLElBQVU7WUFDMUYsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxHQUFHLEdBQUcsU0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQzlDLEtBQUssRUFBRSxVQUFVO2dCQUNqQixpQkFBaUIsRUFBRSxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsU0FBUyxDQUFDO2dCQUNwRCxlQUFlLEVBQUUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQztnQkFDNUMseUJBQXlCLEVBQUUsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQzthQUNwRCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBRXBDLE9BQU87WUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDL0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHdEQUF3RCxDQUFDLElBQVU7WUFDakUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRWxDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRSxlQUFlLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztZQUNqRCxDQUFDLEVBQUUsMEZBQTBGLENBQUMsQ0FBQztZQUUvRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsb0NBQW9DLENBQUMsSUFBVTtZQUM3QyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsWUFBWSxFQUFFLENBQUM7WUFDN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtnQkFDdkMsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsbUJBQW1CLEVBQUU7b0JBQ25CLEVBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxNQUFNLEVBQUU7b0JBQzVDLEVBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxPQUFPLEVBQUU7b0JBQzlDLEVBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxPQUFPLEVBQUU7aUJBQzlDO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7WUFFNUQsT0FBTztZQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzdELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxZQUFZLEVBQUUsQ0FBQztZQUM3QixNQUFNLEdBQUcsR0FBRyxTQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDOUMsS0FBSyxFQUFFLFVBQVU7Z0JBQ2pCLGlCQUFpQixFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLENBQUM7Z0JBQ3BELGVBQWUsRUFBRSxDQUFDLE9BQU8sRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDO2dCQUM1Qyx5QkFBeUIsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsTUFBTSxDQUFDO2FBQ3BELENBQUMsQ0FBQztZQUNILE1BQU0sTUFBTSxHQUFHLElBQUksbUJBQWEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoRCxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO2dCQUMxQyxTQUFTLEVBQUUsYUFBYTtnQkFDeEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUUvRCxPQUFPO1lBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM5QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY291bnRSZXNvdXJjZXMsIGV4cGVjdCwgaGF2ZVJlc291cmNlLCBoYXZlUmVzb3VyY2VMaWtlLCBpc1N1cGVyT2JqZWN0LCBNYXRjaFN0eWxlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IENmbk91dHB1dCwgTGF6eSwgU3RhY2ssIFRhZyB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IEFjbENpZHIsIEFjbFRyYWZmaWMsIENmblN1Ym5ldCwgQ2ZuVlBDLCBEZWZhdWx0SW5zdGFuY2VUZW5hbmN5LCBHZW5lcmljTGludXhJbWFnZSwgSW5zdGFuY2VUeXBlLFxuICBOYXRQcm92aWRlciwgTmV0d29ya0FjbCwgTmV0d29ya0FjbEVudHJ5LCBQcml2YXRlU3VibmV0LCBTdWJuZXQsIFN1Ym5ldFR5cGUsIFRyYWZmaWNEaXJlY3Rpb24sIFZwYyB9IGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgXCJXaGVuIGNyZWF0aW5nIGEgVlBDXCI6IHtcbiAgICBcIndpdGggdGhlIGRlZmF1bHQgQ0lEUiByYW5nZVwiOiB7XG5cbiAgICAgIFwidnBjLnZwY0lkIHJldHVybnMgYSB0b2tlbiB0byB0aGUgVlBDIElEXCIodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJyk7XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUodnBjLnZwY0lkKSwge1JlZjogJ1RoZVZQQzkyNjM2QUIwJyB9ICk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgXCJpdCB1c2VzIHRoZSBjb3JyZWN0IG5ldHdvcmsgcmFuZ2VcIih0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICAgIG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnKTtcbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUEMnLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBWcGMuREVGQVVMVF9DSURSX1JBTkdFLFxuICAgICAgICAgIEVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgICAgICBFbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgICAgICAgIEluc3RhbmNlVGVuYW5jeTogRGVmYXVsdEluc3RhbmNlVGVuYW5jeS5ERUZBVUxULFxuICAgICAgICB9KSk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICAgICd0aGUgTmFtZSB0YWcgaXMgZGVmYXVsdGVkIHRvIHBhdGgnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgICAgbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycpO1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKFxuICAgICAgICAgIGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQQycsXG4gICAgICAgICAgICBoYXNUYWdzKCBbIHtLZXk6ICdOYW1lJywgVmFsdWU6ICdUaGVWUEMnfSBdKSlcbiAgICAgICAgKTtcbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhcbiAgICAgICAgICBoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXknLFxuICAgICAgICAgICAgaGFzVGFncyggWyB7S2V5OiAnTmFtZScsIFZhbHVlOiAnVGhlVlBDJ30gXSkpXG4gICAgICAgICk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgIH0sXG5cbiAgICBcIndpdGggYWxsIG9mIHRoZSBwcm9wZXJ0aWVzIHNldCwgaXQgc3VjY2Vzc2Z1bGx5IHNldHMgdGhlIGNvcnJlY3QgVlBDIHByb3BlcnRpZXNcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycsIHtcbiAgICAgICAgY2lkcjogXCIxOTIuMTY4LjAuMC8xNlwiLFxuICAgICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IGZhbHNlLFxuICAgICAgICBlbmFibGVEbnNTdXBwb3J0OiBmYWxzZSxcbiAgICAgICAgZGVmYXVsdEluc3RhbmNlVGVuYW5jeTogRGVmYXVsdEluc3RhbmNlVGVuYW5jeS5ERURJQ0FURUQsXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUEMnLCB7XG4gICAgICAgIENpZHJCbG9jazogJzE5Mi4xNjguMC4wLzE2JyxcbiAgICAgICAgRW5hYmxlRG5zSG9zdG5hbWVzOiBmYWxzZSxcbiAgICAgICAgRW5hYmxlRG5zU3VwcG9ydDogZmFsc2UsXG4gICAgICAgIEluc3RhbmNlVGVuYW5jeTogRGVmYXVsdEluc3RhbmNlVGVuYW5jeS5ERURJQ0FURUQsXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJjb250YWlucyB0aGUgY29ycmVjdCBudW1iZXIgb2Ygc3VibmV0c1wiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJyk7XG4gICAgICBjb25zdCB6b25lcyA9IHN0YWNrLmF2YWlsYWJpbGl0eVpvbmVzLmxlbmd0aDtcbiAgICAgIHRlc3QuZXF1YWwodnBjLnB1YmxpY1N1Ym5ldHMubGVuZ3RoLCB6b25lcyk7XG4gICAgICB0ZXN0LmVxdWFsKHZwYy5wcml2YXRlU3VibmV0cy5sZW5ndGgsIHpvbmVzKTtcbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUodnBjLnZwY0lkKSwgeyBSZWY6ICdUaGVWUEM5MjYzNkFCMCcgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJ3aXRoIG9ubHkgaXNvbGF0ZWQgc3VibmV0cywgdGhlIFZQQyBzaG91bGQgbm90IGNvbnRhaW4gYW4gSUdXIG9yIE5BVCBHYXRld2F5c1wiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5JU09MQVRFRCxcbiAgICAgICAgICAgIG5hbWU6ICdJc29sYXRlZCcsXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9KTtcbiAgICAgIGV4cGVjdChzdGFjaykubm90VG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OkludGVybmV0R2F0ZXdheVwiKSk7XG4gICAgICBleHBlY3Qoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgIE1hcFB1YmxpY0lwT25MYXVuY2g6IGZhbHNlXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJ3aXRoIG5vIHByaXZhdGUgc3VibmV0cywgdGhlIFZQQyBzaG91bGQgaGF2ZSBhbiBJR1cgYnV0IG5vIE5BVCBHYXRld2F5c1wiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgICAgICBuYW1lOiAnUHVibGljJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAgICAgICAgICBuYW1lOiAnSXNvbGF0ZWQnLFxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKCdBV1M6OkVDMjo6SW50ZXJuZXRHYXRld2F5JywgMSkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoXCJBV1M6OkVDMjo6TmF0R2F0ZXdheVwiKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwid2l0aCBwcml2YXRlIHN1Ym5ldHMgYW5kIGN1c3RvbSBuZXR3b3JrQWNsLlwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgICAgICBuYW1lOiAnUHVibGljJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgICAgIG5hbWU6ICdwcml2YXRlJyxcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBuYWNsMSA9IG5ldyBOZXR3b3JrQWNsKHN0YWNrLCAnbXlOQUNMMScsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICBzdWJuZXRTZWxlY3Rpb246IHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFIH0sXG4gICAgICB9KTtcblxuICAgICAgbmV3IE5ldHdvcmtBY2xFbnRyeShzdGFjaywgJ0FsbG93RE5TRWdyZXNzJywge1xuICAgICAgICBuZXR3b3JrQWNsOiBuYWNsMSxcbiAgICAgICAgcnVsZU51bWJlcjogMTAwLFxuICAgICAgICB0cmFmZmljOiBBY2xUcmFmZmljLnVkcFBvcnQoNTMpLFxuICAgICAgICBkaXJlY3Rpb246IFRyYWZmaWNEaXJlY3Rpb24uRUdSRVNTLFxuICAgICAgICBjaWRyOiBBY2xDaWRyLmlwdjQoJzEwLjAuMC4wLzE2JyksXG4gICAgICB9KTtcblxuICAgICAgbmV3IE5ldHdvcmtBY2xFbnRyeShzdGFjaywgJ0FsbG93RE5TSW5ncmVzcycsIHtcbiAgICAgICAgbmV0d29ya0FjbDogbmFjbDEsXG4gICAgICAgIHJ1bGVOdW1iZXI6IDEwMCxcbiAgICAgICAgdHJhZmZpYzogQWNsVHJhZmZpYy51ZHBQb3J0KDUzKSxcbiAgICAgICAgZGlyZWN0aW9uOiBUcmFmZmljRGlyZWN0aW9uLklOR1JFU1MsXG4gICAgICAgIGNpZHI6IEFjbENpZHIuYW55SXB2NCgpLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6RUMyOjpOZXR3b3JrQWNsJywgMSkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcygnQVdTOjpFQzI6Ok5ldHdvcmtBY2xFbnRyeScsIDIpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6RUMyOjpTdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb24nLCAzKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJ3aXRoIG5vIHN1Ym5ldHMgZGVmaW5lZCwgdGhlIFZQQyBzaG91bGQgaGF2ZSBhbiBJR1csIGFuZCBhIE5BVCBHYXRld2F5IHBlciBBWlwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB6b25lcyA9IHN0YWNrLmF2YWlsYWJpbGl0eVpvbmVzLmxlbmd0aDtcbiAgICAgIG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnLCB7IH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXlcIiwgMSkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIsIHpvbmVzKSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJ3aXRoIHN1Ym5ldHMgYW5kIHJlc2VydmVkIHN1Ym5ldHMgZGVmaW5lZCwgVlBDIHN1Ym5ldCBjb3VudCBzaG91bGQgbm90IGNvbnRhaW4gcmVzZXJ2ZWQgc3VibmV0cyBcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycsIHtcbiAgICAgICAgY2lkcjogJzEwLjAuMC4wLzE2JyxcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgICAgICAgbmFtZTogJ1B1YmxpYycsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjQsXG4gICAgICAgICAgICBuYW1lOiAncmVzZXJ2ZWQnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgICAgICAgICAgcmVzZXJ2ZWQ6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjgsXG4gICAgICAgICAgICBuYW1lOiAncmRzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBtYXhBenM6IDNcbiAgICAgIH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpTdWJuZXRcIiwgNikpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcIndpdGggcmVzZXJ2ZWQgc3VibmV0cywgYW55IG90aGVyIHN1Ym5ldHMgc2hvdWxkIG5vdCBoYXZlIGNpZHJCbG9jayBmcm9tIHdpdGhpbiByZXNlcnZlZCBzcGFjZVwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBjaWRyOiAnMTAuMC4wLjAvMTYnLFxuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgICAgICAgICAgbmFtZTogJ2luZ3Jlc3MnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjQsXG4gICAgICAgICAgICBuYW1lOiAncmVzZXJ2ZWQnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgICAgICAgICAgcmVzZXJ2ZWQ6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjQsXG4gICAgICAgICAgICBuYW1lOiAncmRzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIG1heEF6czogM1xuICAgICAgfSk7XG4gICAgICBmb3IgKGxldCBpID0gMDsgaSA8IDM7IGkrKykge1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpTdWJuZXRcIiwge1xuICAgICAgICAgIENpZHJCbG9jazogYDEwLjAuJHtpfS4wLzI0YFxuICAgICAgICB9KSk7XG4gICAgICB9XG4gICAgICBmb3IgKGxldCBpID0gMzsgaSA8IDY7IGkrKykge1xuICAgICAgICBleHBlY3Qoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpTdWJuZXRcIiwge1xuICAgICAgICAgIENpZHJCbG9jazogYDEwLjAuJHtpfS4wLzI0YFxuICAgICAgICB9KSk7XG4gICAgICB9XG4gICAgICBmb3IgKGxldCBpID0gNjsgaSA8IDk7IGkrKykge1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpTdWJuZXRcIiwge1xuICAgICAgICAgIENpZHJCbG9jazogYDEwLjAuJHtpfS4wLzI0YFxuICAgICAgICB9KSk7XG4gICAgICB9XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwid2l0aCBjdXN0b20gc3VibmV0cywgdGhlIFZQQyBzaG91bGQgaGF2ZSB0aGUgcmlnaHQgbnVtYmVyIG9mIHN1Ym5ldHMsIGFuIElHVywgYW5kIGEgTkFUIEdhdGV3YXkgcGVyIEFaXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgIGNvbnN0IHpvbmVzID0gc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXMubGVuZ3RoO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycsIHtcbiAgICAgICAgY2lkcjogJzEwLjAuMC4wLzIxJyxcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIG5hbWU6ICdpbmdyZXNzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgICAgICAgICAgbmFtZTogJ2FwcGxpY2F0aW9uJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyOCxcbiAgICAgICAgICAgIG5hbWU6ICdyZHMnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5JU09MQVRFRCxcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIG1heEF6czogM1xuICAgICAgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKFwiQVdTOjpFQzI6OkludGVybmV0R2F0ZXdheVwiLCAxKSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKFwiQVdTOjpFQzI6Ok5hdEdhdGV3YXlcIiwgem9uZXMpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6U3VibmV0XCIsIDkpKTtcbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgNjsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBgMTAuMC4ke2l9LjAvMjRgXG4gICAgICAgIH0pKTtcbiAgICAgIH1cbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgMzsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBgMTAuMC42LiR7aSAqIDE2fS8yOGBcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcIndpdGggY3VzdG9tIHN1YmVudHMgYW5kIG5hdEdhdGV3YXlzID0gMiB0aGVyZSBzaG91bGQgYmUgb25seSB0d28gTkFUR1dcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycsIHtcbiAgICAgICAgY2lkcjogJzEwLjAuMC4wLzIxJyxcbiAgICAgICAgbmF0R2F0ZXdheXM6IDIsXG4gICAgICAgIHN1Ym5ldENvbmZpZ3VyYXRpb246IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjQsXG4gICAgICAgICAgICBuYW1lOiAnaW5ncmVzcycsXG4gICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIG5hbWU6ICdhcHBsaWNhdGlvbicsXG4gICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBjaWRyTWFzazogMjgsXG4gICAgICAgICAgICBuYW1lOiAncmRzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBtYXhBenM6IDNcbiAgICAgIH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXlcIiwgMSkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIsIDIpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6U3VibmV0XCIsIDkpKTtcbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgNjsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBgMTAuMC4ke2l9LjAvMjRgXG4gICAgICAgIH0pKTtcbiAgICAgIH1cbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgMzsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBgMTAuMC42LiR7aSAqIDE2fS8yOGBcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcIndpdGggZW5hYmxlRG5zSG9zdG5hbWVzIGVuYWJsZWQgYnV0IGVuYWJsZURuc1N1cHBvcnQgZGlzYWJsZWQsIHNob3VsZCB0aHJvdyBhbiBFcnJvclwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IGZhbHNlXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwid2l0aCBwdWJsaWMgc3VibmV0cyBNYXBQdWJsaWNJcE9uTGF1bmNoIGlzIHRydWVcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgbWF4QXpzOiAxLFxuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgICAgICAgICAgbmFtZTogJ2luZ3Jlc3MnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCAxKSk7XG4gICAgICBleHBlY3Qoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgIE1hcFB1YmxpY0lwT25MYXVuY2g6IHRydWVcbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICBcIm1heEFacyBkZWZhdWx0cyB0byAzIGlmIHVuc2V0XCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgIG5ldyBWcGMoc3RhY2ssICdWUEMnKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6U3VibmV0XCIsIDYpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6Um91dGVcIiwgNikpO1xuICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCA2OyBpKyspIHtcbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OkVDMjo6U3VibmV0XCIsIHtcbiAgICAgICAgICBDaWRyQmxvY2s6IGAxMC4wLiR7aSAqIDMyfS4wLzE5YFxuICAgICAgICB9KSk7XG4gICAgICB9XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDMjo6Um91dGVcIiwge1xuICAgICAgICBEZXN0aW5hdGlvbkNpZHJCbG9jazogJzAuMC4wLjAvMCcsXG4gICAgICAgIE5hdEdhdGV3YXlJZDogeyB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgXCJ3aXRoIG1heEFacyBzZXQgdG8gMlwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVlBDJywgeyBtYXhBenM6IDIgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCA0KSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKFwiQVdTOjpFQzI6OlJvdXRlXCIsIDQpKTtcbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgNDsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpFQzI6OlN1Ym5ldFwiLCB7XG4gICAgICAgICAgQ2lkckJsb2NrOiBgMTAuMC4ke2kgKiA2NH0uMC8xOGBcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQzI6OlJvdXRlXCIsIHtcbiAgICAgICAgRGVzdGluYXRpb25DaWRyQmxvY2s6ICcwLjAuMC4wLzAnLFxuICAgICAgICBOYXRHYXRld2F5SWQ6IHsgfSxcbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgXCJ3aXRoIG5hdEdhdGV3YXkgc2V0IHRvIDFcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgbmF0R2F0ZXdheXM6IDEsXG4gICAgICB9KTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6U3VibmV0XCIsIDYpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoXCJBV1M6OkVDMjo6Um91dGVcIiwgNikpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIsIDEpKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUMyOjpSb3V0ZVwiLCB7XG4gICAgICAgIERlc3RpbmF0aW9uQ2lkckJsb2NrOiAnMC4wLjAuMC8wJyxcbiAgICAgICAgTmF0R2F0ZXdheUlkOiB7IH0sXG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICd3aXRoIG5hdEdhdGV3YXkgc3VibmV0cyBkZWZpbmVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIG5hbWU6ICdpbmdyZXNzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgICAgICAgICAgbmFtZTogJ2VncmVzcycsXG4gICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIG5hbWU6ICdwcml2YXRlJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBuYXRHYXRld2F5U3VibmV0czoge1xuICAgICAgICAgIHN1Ym5ldEdyb3VwTmFtZTogJ2VncmVzcydcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcyhcIkFXUzo6RUMyOjpOYXRHYXRld2F5XCIsIDMpKTtcbiAgICAgIGZvciAobGV0IGkgPSAxOyBpIDwgNDsgaSsrKSB7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U3VibmV0JywgaGFzVGFncyhbe1xuICAgICAgICAgIEtleTogJ05hbWUnLFxuICAgICAgICAgIFZhbHVlOiBgVlBDL2VncmVzc1N1Ym5ldCR7aX1gLFxuICAgICAgICB9LCB7XG4gICAgICAgICAgICBLZXk6ICdhd3MtY2RrOnN1Ym5ldC1uYW1lJyxcbiAgICAgICAgICAgIFZhbHVlOiAnZWdyZXNzJyxcbiAgICAgICAgfV0pKSk7XG4gICAgICB9XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ25hdEdhdGV3YXlzID0gMCByZXF1aXJlcyB0aGVyZSB0byBiZSBubyBQUklWQVRFIHN1Ym5ldHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIG5ldyBWcGMoc3RhY2ssICdWUEMnLCB7XG4gICAgICAgICAgbmF0R2F0ZXdheXM6IDAsXG4gICAgICAgIH0pO1xuICAgICAgfSwgL21ha2Ugc3VyZSB5b3UgZG9uJ3QgY29uZmlndXJlIGFueSBQUklWQVRFIHN1Ym5ldHMvKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuXG4gICAgfSxcblxuICAgICd3aXRoIG1pcy1tYXRjaGVkIG5hdCBhbmQgc3VibmV0IGNvbmZpZ3MgaXQgdGhyb3dzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNpZHJNYXNrOiAyNCxcbiAgICAgICAgICAgIG5hbWU6ICdpbmdyZXNzJyxcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgICAgICAgICAgbmFtZTogJ3ByaXZhdGUnLFxuICAgICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIG5hdEdhdGV3YXlTdWJuZXRzOiB7XG4gICAgICAgICAgc3VibmV0R3JvdXBOYW1lOiAnbm90dGhlcmUnLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnd2l0aCBhIHZwbiBnYXRld2F5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgdnBuR2F0ZXdheTogdHJ1ZSxcbiAgICAgICAgdnBuR2F0ZXdheUFzbjogNjUwMDBcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQTkdhdGV3YXknLCB7XG4gICAgICAgIEFtYXpvblNpZGVBc246IDY1MDAwLFxuICAgICAgICBUeXBlOiAnaXBzZWMuMSdcbiAgICAgIH0pKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUENHYXRld2F5QXR0YWNobWVudCcsIHtcbiAgICAgICAgVnBjSWQ6IHtcbiAgICAgICAgICBSZWY6ICdWUENCOUU1RjBCNCdcbiAgICAgICAgfSxcbiAgICAgICAgVnBuR2F0ZXdheUlkOiB7XG4gICAgICAgICAgUmVmOiAnVlBDVnBuR2F0ZXdheUI1QUJBRTY4J1xuICAgICAgICB9XG4gICAgICB9KSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6VlBOR2F0ZXdheVJvdXRlUHJvcGFnYXRpb24nLCB7XG4gICAgICAgIFJvdXRlVGFibGVJZHM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdWUENQcml2YXRlU3VibmV0MVJvdXRlVGFibGVCRThBNjAyNydcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZQQ1ByaXZhdGVTdWJuZXQyUm91dGVUYWJsZTBBMTlFMTBFJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVlBDUHJpdmF0ZVN1Ym5ldDNSb3V0ZVRhYmxlMTkyMTg2RjgnXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWcG5HYXRld2F5SWQ6IHtcbiAgICAgICAgICBSZWY6ICdWUENWcG5HYXRld2F5QjVBQkFFNjgnXG4gICAgICAgIH1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnd2l0aCBhIHZwbiBnYXRld2F5IGFuZCByb3V0ZSBwcm9wYWdhdGlvbiBvbiBpc29sYXRlZCBzdWJuZXRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsIG5hbWU6ICdQdWJsaWMnIH0sXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLklTT0xBVEVELCBuYW1lOiAnSXNvbGF0ZWQnIH0sXG4gICAgICAgIF0sXG4gICAgICAgIHZwbkdhdGV3YXk6IHRydWUsXG4gICAgICAgIHZwblJvdXRlUHJvcGFnYXRpb246IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLklTT0xBVEVEXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUE5HYXRld2F5Um91dGVQcm9wYWdhdGlvbicsIHtcbiAgICAgICAgUm91dGVUYWJsZUlkczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZQQ0lzb2xhdGVkU3VibmV0MVJvdXRlVGFibGVFQjE1NjIxMCdcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZQQ0lzb2xhdGVkU3VibmV0MlJvdXRlVGFibGU5QjRGNzhEQydcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZQQ0lzb2xhdGVkU3VibmV0M1JvdXRlVGFibGVDQjZBMUZEQSdcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFZwbkdhdGV3YXlJZDoge1xuICAgICAgICAgIFJlZjogJ1ZQQ1ZwbkdhdGV3YXlCNUFCQUU2OCdcbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICd3aXRoIGEgdnBuIGdhdGV3YXkgYW5kIHJvdXRlIHByb3BhZ2F0aW9uIG9uIHByaXZhdGUgYW5kIGlzb2xhdGVkIHN1Ym5ldHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQywgbmFtZTogJ1B1YmxpYycgfSxcbiAgICAgICAgICB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSwgbmFtZTogJ1ByaXZhdGUnIH0sXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLklTT0xBVEVELCBuYW1lOiAnSXNvbGF0ZWQnIH0sXG4gICAgICAgIF0sXG4gICAgICAgIHZwbkdhdGV3YXk6IHRydWUsXG4gICAgICAgIHZwblJvdXRlUHJvcGFnYXRpb246IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURURcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQTkdhdGV3YXlSb3V0ZVByb3BhZ2F0aW9uJywge1xuICAgICAgICBSb3V0ZVRhYmxlSWRzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVlBDUHJpdmF0ZVN1Ym5ldDFSb3V0ZVRhYmxlQkU4QTYwMjcnXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdWUENQcml2YXRlU3VibmV0MlJvdXRlVGFibGUwQTE5RTEwRSdcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ1ZQQ1ByaXZhdGVTdWJuZXQzUm91dGVUYWJsZTE5MjE4NkY4J1xuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVlBDSXNvbGF0ZWRTdWJuZXQxUm91dGVUYWJsZUVCMTU2MjEwJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVlBDSXNvbGF0ZWRTdWJuZXQyUm91dGVUYWJsZTlCNEY3OERDJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnVlBDSXNvbGF0ZWRTdWJuZXQzUm91dGVUYWJsZUNCNkExRkRBJ1xuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVnBuR2F0ZXdheUlkOiB7XG4gICAgICAgICAgUmVmOiAnVlBDVnBuR2F0ZXdheUI1QUJBRTY4J1xuICAgICAgICB9XG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2ZhaWxzIHdoZW4gc3BlY2lmeWluZyB2cG5Db25uZWN0aW9ucyB3aXRoIHZwbkdhdGV3YXkgc2V0IHRvIGZhbHNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnLCB7XG4gICAgICAgIHZwbkdhdGV3YXk6IGZhbHNlLFxuICAgICAgICB2cG5Db25uZWN0aW9uczoge1xuICAgICAgICAgIFZwbkNvbm5lY3Rpb246IHtcbiAgICAgICAgICAgIGFzbjogNjUwMDAsXG4gICAgICAgICAgICBpcDogJzE5Mi4wLjIuMSdcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0pLCAvYHZwbkNvbm5lY3Rpb25zYC4rYHZwbkdhdGV3YXlgLitmYWxzZS8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdmYWlscyB3aGVuIHNwZWNpZnlpbmcgdnBuR2F0ZXdheUFzbiB3aXRoIHZwbkdhdGV3YXkgc2V0IHRvIGZhbHNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnLCB7XG4gICAgICAgIHZwbkdhdGV3YXk6IGZhbHNlLFxuICAgICAgICB2cG5HYXRld2F5QXNuOiA2NTAwMCxcbiAgICAgIH0pLCAvYHZwbkdhdGV3YXlBc25gLitgdnBuR2F0ZXdheWAuK2ZhbHNlLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnU3VibmV0cyBoYXZlIGEgZGVmYXVsdENoaWxkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnKTtcblxuICAgICAgdGVzdC5vayh2cGMucHVibGljU3VibmV0c1swXS5ub2RlLmRlZmF1bHRDaGlsZCBpbnN0YW5jZW9mIENmblN1Ym5ldCk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnQ0lEUiBjYW5ub3QgYmUgYSBUb2tlbicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgbmV3IFZwYyhzdGFjaywgJ1ZwYycsIHtcbiAgICAgICAgICBjaWRyOiBMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gJ2FiYycgfSlcbiAgICAgICAgfSk7XG4gICAgICB9LCAvcHJvcGVydHkgbXVzdCBiZSBhIGNvbmNyZXRlIENJRFIgc3RyaW5nLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgJ05BVCBpbnN0YW5jZXMnOiB7XG4gICAgJ0NhbiBjb25maWd1cmUgTkFUIGluc3RhbmNlcyBpbnN0ZWFkIG9mIE5BVCBnYXRld2F5cycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnLCB7XG4gICAgICAgIG5hdEdhdGV3YXlQcm92aWRlcjogTmF0UHJvdmlkZXIuaW5zdGFuY2Uoe1xuICAgICAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgncTg2Lm1lZ2EnKSxcbiAgICAgICAgICBtYWNoaW5lSW1hZ2U6IG5ldyBHZW5lcmljTGludXhJbWFnZSh7XG4gICAgICAgICAgICAndXMtZWFzdC0xJzogJ2FtaS0xJ1xuICAgICAgICAgIH0pXG4gICAgICAgIH0pXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcygnQVdTOjpFQzI6Okluc3RhbmNlJywgMykpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICAgICAgSW1hZ2VJZDogXCJhbWktMVwiLFxuICAgICAgICBJbnN0YW5jZVR5cGU6IFwicTg2Lm1lZ2FcIixcbiAgICAgICAgU291cmNlRGVzdENoZWNrOiBmYWxzZSxcbiAgICAgIH0pKTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6Um91dGUnLCB7XG4gICAgICAgIFJvdXRlVGFibGVJZDogeyBSZWY6IFwiVGhlVlBDUHJpdmF0ZVN1Ym5ldDFSb3V0ZVRhYmxlRjY1MTNCQzJcIiB9LFxuICAgICAgICBEZXN0aW5hdGlvbkNpZHJCbG9jazogXCIwLjAuMC4wLzBcIixcbiAgICAgICAgSW5zdGFuY2VJZDogeyBSZWY6IFwiVGhlVlBDUHVibGljU3VibmV0MU5hdEluc3RhbmNlQ0M1MTQxOTJcIiB9XG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnbmF0R2F0ZXdheXMgY29udHJvbHMgYW1vdW50IG9mIE5BVCBpbnN0YW5jZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywge1xuICAgICAgICBuYXRHYXRld2F5UHJvdmlkZXI6IE5hdFByb3ZpZGVyLmluc3RhbmNlKHtcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3E4Ni5tZWdhJyksXG4gICAgICAgICAgbWFjaGluZUltYWdlOiBuZXcgR2VuZXJpY0xpbnV4SW1hZ2Uoe1xuICAgICAgICAgICAgJ3VzLWVhc3QtMSc6ICdhbWktMSdcbiAgICAgICAgICB9KVxuICAgICAgICB9KSxcbiAgICAgICAgbmF0R2F0ZXdheXM6IDFcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKCdBV1M6OkVDMjo6SW5zdGFuY2UnLCAxKSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgJ05ldHdvcmsgQUNMIGFzc29jaWF0aW9uJzoge1xuICAgICdieSBkZWZhdWx0IHVzZXMgZGVmYXVsdCBBQ0wgcmVmZXJlbmNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycsIHsgY2lkcjogJzE5Mi4xNjguMC4wLzE2JyB9KTtcbiAgICAgIG5ldyBDZm5PdXRwdXQoc3RhY2ssICdPdXRwdXQnLCB7XG4gICAgICAgIHZhbHVlOiAodnBjLnB1YmxpY1N1Ym5ldHNbMF0gYXMgU3VibmV0KS5zdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb25JZFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICAgIE91dHB1dHM6IHtcbiAgICAgICAgICBPdXRwdXQ6IHtcbiAgICAgICAgICAgIFZhbHVlOiB7IFwiRm46OkdldEF0dFwiOiBbIFwiVGhlVlBDUHVibGljU3VibmV0MVN1Ym5ldDc3MEQ0RkYyXCIsIFwiTmV0d29ya0FjbEFzc29jaWF0aW9uSWRcIiBdIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0sIE1hdGNoU3R5bGUuU1VQRVJTRVQpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2lmIEFDTCBpcyByZXBsYWNlZCBuZXcgQUNMIHJlZmVyZW5jZSBpcyByZXR1cm5lZCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVGhlVlBDJywgeyBjaWRyOiAnMTkyLjE2OC4wLjAvMTYnIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgQ2ZuT3V0cHV0KHN0YWNrLCAnT3V0cHV0Jywge1xuICAgICAgICB2YWx1ZTogKHZwYy5wdWJsaWNTdWJuZXRzWzBdIGFzIFN1Ym5ldCkuc3VibmV0TmV0d29ya0FjbEFzc29jaWF0aW9uSWRcbiAgICAgIH0pO1xuICAgICAgbmV3IE5ldHdvcmtBY2woc3RhY2ssICdBQ0wnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgc3VibmV0U2VsZWN0aW9uOiB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDIH1cbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgICBPdXRwdXRzOiB7XG4gICAgICAgICAgT3V0cHV0OiB7XG4gICAgICAgICAgICBWYWx1ZTogeyBSZWY6IFwiQUNMREJEMUJCNDlcIn1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0sIE1hdGNoU3R5bGUuU1VQRVJTRVQpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gIFwiV2hlbiBjcmVhdGluZyBhIFZQQyB3aXRoIGEgY3VzdG9tIENJRFIgcmFuZ2VcIjoge1xuICAgIFwidnBjLnZwY0NpZHJCbG9jayBpcyB0aGUgY29ycmVjdCBuZXR3b3JrIHJhbmdlXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgIG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnLCB7IGNpZHI6ICcxOTIuMTY4LjAuMC8xNicgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpWUENcIiwge1xuICAgICAgICBDaWRyQmxvY2s6ICcxOTIuMTY4LjAuMC8xNidcbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH1cbiAgfSxcbiAgJ1doZW4gdGFnZ2luZyc6IHtcbiAgICAnVlBDIHByb3BhZ2F0ZWQgdGFncyB3aWxsIGJlIG9uIHN1Ym5ldCwgSUdXLCByb3V0ZXRhYmxlcywgTkFUR1cnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB0YWdzID0gIHtcbiAgICAgICAgVnBjVHlwZTogJ0dvb2QnLFxuICAgICAgfTtcbiAgICAgIGNvbnN0IG5vUHJvcFRhZ3MgPSB7XG4gICAgICAgIEJ1c2luZXNzVW5pdDogJ01hcmtldGluZycsXG4gICAgICB9O1xuICAgICAgY29uc3QgYWxsVGFncyAgPSB7Li4udGFncywgLi4ubm9Qcm9wVGFnc307XG5cbiAgICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnKTtcbiAgICAgIC8vIG92ZXJ3cml0ZSB0byBzZXQgcHJvcGFnYXRlXG4gICAgICB2cGMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdCdXNpbmVzc1VuaXQnLCAnTWFya2V0aW5nJywge2luY2x1ZGVSZXNvdXJjZVR5cGVzOiBbQ2ZuVlBDLkNGTl9SRVNPVVJDRV9UWVBFX05BTUVdfSkpO1xuICAgICAgdnBjLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnVnBjVHlwZScsICdHb29kJykpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OkVDMjo6VlBDXCIsIGhhc1RhZ3ModG9DZm5UYWdzKGFsbFRhZ3MpKSkpO1xuICAgICAgY29uc3QgdGFnZ2FibGVzID0gWydTdWJuZXQnLCAnSW50ZXJuZXRHYXRld2F5JywgJ05hdEdhdGV3YXknLCAnUm91dGVUYWJsZSddO1xuICAgICAgY29uc3QgcHJvcFRhZ3MgPSB0b0NmblRhZ3ModGFncyk7XG4gICAgICBjb25zdCBub1Byb3AgPSB0b0NmblRhZ3Mobm9Qcm9wVGFncyk7XG4gICAgICBmb3IgKGNvbnN0IHJlc291cmNlIG9mIHRhZ2dhYmxlcykge1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShgQVdTOjpFQzI6OiR7cmVzb3VyY2V9YCwgaGFzVGFncyhwcm9wVGFncykpKTtcbiAgICAgICAgZXhwZWN0KHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoYEFXUzo6RUMyOjoke3Jlc291cmNlfWAsIGhhc1RhZ3Mobm9Qcm9wKSkpO1xuICAgICAgfVxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnU3VibmV0IE5hbWUgd2lsbCBwcm9wYWdhdGUgdG8gcm91dGUgdGFibGVzIGFuZCBOQVRHVycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdUaGVWUEMnKTtcbiAgICAgIGZvciAoY29uc3Qgc3VibmV0IG9mIHZwYy5wdWJsaWNTdWJuZXRzKSB7XG4gICAgICAgIGNvbnN0IHRhZyA9IHtLZXk6ICdOYW1lJywgVmFsdWU6IHN1Ym5ldC5ub2RlLnBhdGh9O1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6Ok5hdEdhdGV3YXknLCBoYXNUYWdzKFt0YWddKSkpO1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlJvdXRlVGFibGUnLCBoYXNUYWdzKFt0YWddKSkpO1xuICAgICAgfVxuICAgICAgZm9yIChjb25zdCBzdWJuZXQgb2YgdnBjLnByaXZhdGVTdWJuZXRzKSB7XG4gICAgICAgIGNvbnN0IHRhZyA9IHtLZXk6ICdOYW1lJywgVmFsdWU6IHN1Ym5ldC5ub2RlLnBhdGh9O1xuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlJvdXRlVGFibGUnLCBoYXNUYWdzKFt0YWddKSkpO1xuICAgICAgfVxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnVGFncyBjYW4gYmUgYWRkZWQgYWZ0ZXIgdGhlIFZwYyBpcyBjcmVhdGVkIHdpdGggYHZwYy50YWdzLnNldFRhZyguLi4pYCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcblxuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1RoZVZQQycpO1xuICAgICAgY29uc3QgdGFnID0ge0tleTogJ0xhdGUnLCBWYWx1ZTogJ0FkZGVyJ307XG4gICAgICBleHBlY3Qoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQQycsIGhhc1RhZ3MoW3RhZ10pKSk7XG4gICAgICB2cGMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKHRhZy5LZXksIHRhZy5WYWx1ZSkpO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUEMnLCBoYXNUYWdzKFt0YWddKSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAnc3VibmV0IHNlbGVjdGlvbic6IHtcbiAgICAnc2VsZWN0aW5nIGRlZmF1bHQgc3VibmV0cyByZXR1cm5zIHRoZSBwcml2YXRlIG9uZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCB7IHN1Ym5ldElkcyB9ID0gdnBjLnNlbGVjdFN1Ym5ldHMoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3VibmV0SWRzLCB2cGMucHJpdmF0ZVN1Ym5ldHMubWFwKHMgPT4gcy5zdWJuZXRJZCkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdjYW4gc2VsZWN0IHB1YmxpYyBzdWJuZXRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBnZXRUZXN0U3RhY2soKTtcbiAgICAgIGNvbnN0IHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWUEMnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgeyBzdWJuZXRJZHMgfSA9IHZwYy5zZWxlY3RTdWJuZXRzKHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcywgdnBjLnB1YmxpY1N1Ym5ldHMubWFwKHMgPT4gcy5zdWJuZXRJZCkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBzZWxlY3QgaXNvbGF0ZWQgc3VibmV0cycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQywgbmFtZTogJ1B1YmxpYycgfSxcbiAgICAgICAgICB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsIG5hbWU6ICdJc29sYXRlZCcgfSxcbiAgICAgICAgXVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHsgc3VibmV0SWRzIH0gPSB2cGMuc2VsZWN0U3VibmV0cyh7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcywgdnBjLmlzb2xhdGVkU3VibmV0cy5tYXAocyA9PiBzLnN1Ym5ldElkKSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIHNlbGVjdCBzdWJuZXRzIGJ5IG5hbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsIG5hbWU6ICdCbGFCbGEnIH0sXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsIG5hbWU6ICdEb250VGFsa1RvTWUnIH0sXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLklTT0xBVEVELCBuYW1lOiAnRG9udFRhbGtBdEFsbCcgfSxcbiAgICAgICAgXVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHsgc3VibmV0SWRzIH0gPSB2cGMuc2VsZWN0U3VibmV0cyh7IHN1Ym5ldEdyb3VwTmFtZTogJ0RvbnRUYWxrVG9NZScgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcywgdnBjLnByaXZhdGVTdWJuZXRzLm1hcChzID0+IHMuc3VibmV0SWQpKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnc3VibmV0TmFtZSBpcyBhbiBhbGlhcyBmb3Igc3VibmV0R3JvdXBOYW1lIChiYWNrd2FyZHMgY29tcGF0KScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gZ2V0VGVzdFN0YWNrKCk7XG4gICAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJywge1xuICAgICAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICAgICAgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQywgbmFtZTogJ0JsYUJsYScgfSxcbiAgICAgICAgICB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSwgbmFtZTogJ0RvbnRUYWxrVG9NZScgfSxcbiAgICAgICAgICB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsIG5hbWU6ICdEb250VGFsa0F0QWxsJyB9LFxuICAgICAgICBdXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgeyBzdWJuZXRJZHMgfSA9IHZwYy5zZWxlY3RTdWJuZXRzKHsgc3VibmV0TmFtZTogJ0RvbnRUYWxrVG9NZScgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcywgdnBjLnByaXZhdGVTdWJuZXRzLm1hcChzID0+IHMuc3VibmV0SWQpKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnc2VsZWN0aW5nIGRlZmF1bHQgc3VibmV0cyBpbiBhIFZQQyB3aXRoIG9ubHkgaXNvbGF0ZWQgc3VibmV0cyByZXR1cm5zIHRoZSBpc29sYXRlZHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gVnBjLmZyb21WcGNBdHRyaWJ1dGVzKHN0YWNrLCAnVlBDJywge1xuICAgICAgICB2cGNJZDogJ3ZwYy0xMjM0JyxcbiAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZXM6IFsnZHVtbXkxYScsICdkdW1teTFiJywgJ2R1bW15MWMnXSxcbiAgICAgICAgaXNvbGF0ZWRTdWJuZXRJZHM6IFsnaXNvLTEnLCAnaXNvLTInLCAnaXNvLTMnXSxcbiAgICAgICAgaXNvbGF0ZWRTdWJuZXRSb3V0ZVRhYmxlSWRzOiBbJ3J0LTEnLCAncnQtMicsICdydC0zJ10sXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3Qgc3VibmV0cyA9IHZwYy5zZWxlY3RTdWJuZXRzKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldHMuc3VibmV0SWRzLCBbJ2lzby0xJywgJ2lzby0yJywgJ2lzby0zJ10pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdzZWxlY3RpbmcgZGVmYXVsdCBzdWJuZXRzIGluIGEgVlBDIHdpdGggb25seSBwdWJsaWMgc3VibmV0cyByZXR1cm5zIHRoZSBwdWJsaWNzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIGNvbnN0IHZwYyA9IFZwYy5mcm9tVnBjQXR0cmlidXRlcyhzdGFjaywgJ1ZQQycsIHtcbiAgICAgICAgdnBjSWQ6ICd2cGMtMTIzNCcsXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBbJ2R1bW15MWEnLCAnZHVtbXkxYicsICdkdW1teTFjJ10sXG4gICAgICAgIHB1YmxpY1N1Ym5ldElkczogWydwdWItMScsICdwdWItMicsICdwdWItMyddLFxuICAgICAgICBwdWJsaWNTdWJuZXRSb3V0ZVRhYmxlSWRzOiBbJ3J0LTEnLCAncnQtMicsICdydC0zJ10sXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3Qgc3VibmV0cyA9IHZwYy5zZWxlY3RTdWJuZXRzKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldHMuc3VibmV0SWRzLCBbJ3B1Yi0xJywgJ3B1Yi0yJywgJ3B1Yi0zJ10pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdzZWxlY3Rpbmcgc3VibmV0cyBieSBuYW1lIGZhaWxzIGlmIHRoZSBuYW1lIGlzIHVua25vd24nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xuXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIHZwYy5zZWxlY3RTdWJuZXRzKHsgc3VibmV0R3JvdXBOYW1lOiAnVG9vdCcgfSk7XG4gICAgICB9LCAvVGhlcmUgYXJlIG5vIHN1Ym5ldCBncm91cHMgd2l0aCBuYW1lICdUb290JyBpbiB0aGlzIFZQQy4gQXZhaWxhYmxlIG5hbWVzOiBQdWJsaWMsUHJpdmF0ZS8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3NlbGVjdCBzdWJuZXRzIHdpdGggYXogcmVzdHJpY3Rpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZwY05ldHdvcmsnLCB7XG4gICAgICAgIG1heEF6czogMSxcbiAgICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICAgIHtuYW1lOiAnbGInLCBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyB9LFxuICAgICAgICAgIHtuYW1lOiAnYXBwJywgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFIH0sXG4gICAgICAgICAge25hbWU6ICdkYicsIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSB9LFxuICAgICAgICBdXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgeyBzdWJuZXRJZHMgfSA9IHZwYy5zZWxlY3RTdWJuZXRzKHsgb25lUGVyQXo6IHRydWUgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcy5sZW5ndGgsIDEpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3VibmV0SWRzWzBdLCB2cGMucHJpdmF0ZVN1Ym5ldHNbMF0uc3VibmV0SWQpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdzZWxlY3QgZXhwbGljaXRseSBkZWZpbmVkIHN1Ym5ldHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IGdldFRlc3RTdGFjaygpO1xuICAgICAgY29uc3QgdnBjID0gVnBjLmZyb21WcGNBdHRyaWJ1dGVzKHN0YWNrLCAnVlBDJywge1xuICAgICAgICB2cGNJZDogJ3ZwYy0xMjM0JyxcbiAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZXM6IFsnZHVtbXkxYScsICdkdW1teTFiJywgJ2R1bW15MWMnXSxcbiAgICAgICAgcHVibGljU3VibmV0SWRzOiBbJ3B1Yi0xJywgJ3B1Yi0yJywgJ3B1Yi0zJ10sXG4gICAgICAgIHB1YmxpY1N1Ym5ldFJvdXRlVGFibGVJZHM6IFsncnQtMScsICdydC0yJywgJ3J0LTMnXSxcbiAgICAgIH0pO1xuICAgICAgY29uc3Qgc3VibmV0ID0gbmV3IFByaXZhdGVTdWJuZXQoc3RhY2ssICdTdWJuZXQnLCB7XG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHZwYy5hdmFpbGFiaWxpdHlab25lc1swXSxcbiAgICAgICAgY2lkckJsb2NrOiAnMTAuMC4wLjAvMjgnLFxuICAgICAgICB2cGNJZDogdnBjLnZwY0lkXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgeyBzdWJuZXRJZHMgfSA9IHZwYy5zZWxlY3RTdWJuZXRzKHsgc3VibmV0czogW3N1Ym5ldF0gfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN1Ym5ldElkcy5sZW5ndGgsIDEpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3VibmV0SWRzWzBdLCBzdWJuZXQuc3VibmV0SWQpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuICB9LFxufTtcblxuZnVuY3Rpb24gZ2V0VGVzdFN0YWNrKCk6IFN0YWNrIHtcbiAgcmV0dXJuIG5ldyBTdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJywgcmVnaW9uOiAndXMtZWFzdC0xJyB9IH0pO1xufVxuXG5mdW5jdGlvbiB0b0NmblRhZ3ModGFnczogYW55KTogQXJyYXk8e0tleTogc3RyaW5nLCBWYWx1ZTogc3RyaW5nfT4ge1xuICByZXR1cm4gT2JqZWN0LmtleXModGFncykubWFwKCBrZXkgPT4ge1xuICAgIHJldHVybiB7S2V5OiBrZXksIFZhbHVlOiB0YWdzW2tleV19O1xuICB9KTtcbn1cblxuZnVuY3Rpb24gaGFzVGFncyhleHBlY3RlZFRhZ3M6IEFycmF5PHtLZXk6IHN0cmluZywgVmFsdWU6IHN0cmluZ30+KTogKHByb3BzOiBhbnkpID0+IGJvb2xlYW4ge1xuICByZXR1cm4gKHByb3BzOiBhbnkpID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgdGFncyA9IHByb3BzLlRhZ3M7XG4gICAgICBjb25zdCBhY3R1YWxUYWdzID0gdGFncy5maWx0ZXIoICh0YWc6IHtLZXk6IHN0cmluZywgVmFsdWU6IHN0cmluZ30pID0+IHtcbiAgICAgICAgZm9yIChjb25zdCBleHBlY3RlZFRhZyBvZiBleHBlY3RlZFRhZ3MpIHtcbiAgICAgICAgICBpZiAoaXNTdXBlck9iamVjdChleHBlY3RlZFRhZywgdGFnKSkge1xuICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICAvLyBubyB2YWx1ZXMgaW4gYXJyYXkgc28gZXhwZWN0aW5nIGVtcHR5XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIGFjdHVhbFRhZ3MubGVuZ3RoID09PSBleHBlY3RlZFRhZ3MubGVuZ3RoO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlOm5vLWNvbnNvbGVcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ1RhZ3MgYXJlIGluY29ycmVjdCcpO1xuICAgICAgY29uc29sZS5lcnJvcignZm91bmQgdGFncyAnLCBwcm9wcy5UYWdzKTtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ2V4cGVjdGVkIHRhZ3MgJywgZXhwZWN0ZWRUYWdzKTtcbiAgICAgIC8vIHRzbGludDplbmFibGU6bm8tY29uc29sZVxuICAgICAgdGhyb3cgZTtcbiAgICB9XG4gIH07XG59XG4iXX0=