"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'instance is created correctly'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
        }));
        test.done();
    },
    'instance is created with source/dest check switched off'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
            sourceDestCheck: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
            SourceDestCheck: false,
        }));
        test.done();
    },
    'instance is grantable'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const param = new aws_ssm_1.StringParameter(stack, 'Param', { stringValue: 'Foobar' });
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        });
        // WHEN
        param.grantRead(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            "ssm:DescribeParameters",
                            "ssm:GetParameters",
                            "ssm:GetParameter",
                            "ssm:GetParameterHistory"
                        ],
                        Effect: "Allow",
                        Resource: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":ssm:",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ":",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":parameter/",
                                    {
                                        Ref: "Param165332EC"
                                    }
                                ]
                            ]
                        }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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