"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpn_1 = require("./vpn");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC).
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. Internet traffic will
     * be routed via a NAT Gateway.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            availabilityZones: subnets.map(s => s.availabilityZone),
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s))
        };
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, Object.assign({ vpc: this }, options));
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, Object.assign({ vpc: this }, options));
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, Object.assign({ vpc: this }, options));
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        else if (selection.subnetGroupName !== undefined) { // Select by name
            return this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else {
            const type = selection.subnetType || SubnetType.PRIVATE;
            return this.selectSubnetObjectsByType(type, !!selection.onePerAz);
        }
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType, onePerAz) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        let subnets = allSubnets[subnetType];
        if (onePerAz && subnets.length > 0) {
            // Restrict to at most one subnet group
            subnets = subnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === util_1.subnetGroupNameFromConstructId(subnets[0]));
        }
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error(`Please use only 'subnetGroupName' ('subnetName' is deprecated and has the same behavior)`);
            }
            placement = Object.assign(Object.assign({}, placement), { subnetGroupName: placement.subnetName });
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            if (this.privateSubnets.length > 0) {
                return { subnetType: SubnetType.PRIVATE, onePerAz: placement.onePerAz };
            }
            if (this.isolatedSubnets.length > 0) {
                return { subnetType: SubnetType.ISOLATED, onePerAz: placement.onePerAz };
            }
            return { subnetType: SubnetType.PUBLIC, onePerAz: placement.onePerAz };
        }
        return placement;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * import { SubnetType, Vpc } from '@aws-cdk/aws-ec2'
 *
 * const vpc = new Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error(`'cidr' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)`);
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        const enableDnsHostnames = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        const enableDnsSupport = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames,
            enableDnsSupport,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs !== undefined ? props.maxAzs : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, Vpc.DEFAULT_SUBNETS);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            const vpnGateway = new ec2_generated_1.CfnVPNGateway(this, 'VpnGateway', {
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1
            });
            const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
                vpcId: this.vpcId,
                vpnGatewayId: vpnGateway.ref,
            });
            this.vpnGatewayId = vpnGateway.ref;
            // Propagate routes on route tables associated with the right subnets
            const vpnRoutePropagation = props.vpnRoutePropagation || [{ subnetType: SubnetType.PRIVATE }];
            const routeTableIds = util_1.allRouteTableIds(...vpnRoutePropagation.map(s => this.selectSubnets(s)));
            const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
                routeTableIds,
                vpnGatewayId: this.vpnGatewayId
            });
            // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
            // until it has successfully attached to the VPC.
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
            routePropagation.node.addDependency(attachment);
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
    }
    /**
     * Import an exported VPC
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * See the package-level documentation of this package for constraints
     * on importing existing VPCs.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error(`All arguments to Vpc.fromLookup() must be concrete (no Tokens)`);
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaing = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaing);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            subnet.node.applyAspect(new core_1.Tag(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes }));
            subnet.node.applyAspect(new core_1.Tag(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes }));
        });
    }
}
exports.Vpc = Vpc;
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    }
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZone = props.availabilityZone;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.stringValue({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, `DefaultRoute`, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explic it ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatwayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error(`Cannot specify both 'destinationCidrBlock' and 'destinationIpv6CidrBlock'`);
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationCidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, `NATGateway`, {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, `EIP`, {
                domain: 'vpc'
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
}
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.availabilityZones = props.availabilityZones;
        this.vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // tslint:disable:max-line-length
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        // tslint:enable:max-line-length
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = [];
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            }
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            const ref = core_1.Token.isUnresolved(attrs.subnetId)
                ? `at '${scope.node.path}/${id}'`
                : `'${attrs.subnetId}'`;
            // tslint:disable-next-line: max-line-length
            scope.node.addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this.availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId
        };
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // tslint:disable-next-line:max-line-length
        throw new Error(`If you do not want NAT gateways (natGateways=0), make sure you don't configure any PRIVATE subnets in 'subnetConfiguration' (make them PUBLIC or ISOLATED instead)`);
    }
    if (count > 0 && !hasPublicSubnets) {
        // tslint:disable-next-line:max-line-length
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy-1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                },
                {
                    availabilityZone: 'dummy-1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy-1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                },
                {
                    availabilityZone: 'dummy-1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidnBjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQ3FGO0FBQ3JGLHlDQUEwQztBQUMxQyxtREFFeUk7QUFDekksK0JBQW9DO0FBQ3BDLCtDQUFxRjtBQUNyRixpREFBZ0Q7QUFDaEQsaUNBQTJIO0FBQzNILGlEQUFnSztBQUVoSywrQkFBK0U7QUFFL0UsTUFBTSxpQkFBaUIsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7QUFzR25FOztHQUVHO0FBQ0gsSUFBWSxVQWlDWDtBQWpDRCxXQUFZLFVBQVU7SUFDcEI7Ozs7O09BS0c7SUFDSCxtQ0FBcUIsQ0FBQTtJQUVyQjs7Ozs7Ozs7OztPQVVHO0lBQ0gsaUNBQW1CLENBQUE7SUFFbkI7Ozs7Ozs7OztPQVNHO0lBQ0gsK0JBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQWpDVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQWlDckI7QUFvR0Q7O0dBRUc7QUFDSCxNQUFlLE9BQVEsU0FBUSxlQUFRO0lBQXZDOztRQXFDRTs7OztXQUlHO1FBQ2dCLG9CQUFlLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztRQUU3RDs7V0FFRztRQUNPLCtCQUEwQixHQUFZLEtBQUssQ0FBQztJQW9JeEQsQ0FBQztJQWxJQzs7T0FFRztJQUNJLGFBQWEsQ0FBQyxZQUE2QixFQUFFO1FBQ2xELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNwRCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFekMsT0FBTztZQUNMLFNBQVMsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztZQUN2QyxpQkFBaUIsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDO1lBQ3ZELCtCQUErQixFQUFFLEdBQUcsQ0FBQyxJQUFJLG1CQUFtQixFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsK0JBQStCLENBQUMsQ0FBQyxDQUFDO1lBQ3BJLE9BQU87WUFDUCxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDMUMsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLGdCQUFnQixDQUFDLEVBQVUsRUFBRSxPQUE2QjtRQUMvRCxPQUFPLElBQUksbUJBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxrQkFDL0IsR0FBRyxFQUFFLElBQUksSUFDTixPQUFPLEVBQ1YsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLG9CQUFvQixDQUFDLEVBQVUsRUFBRSxPQUFvQztRQUMxRSxPQUFPLElBQUksbUNBQW9CLENBQUMsSUFBSSxFQUFFLEVBQUUsa0JBQ3RDLEdBQUcsRUFBRSxJQUFJLElBQ04sT0FBTyxFQUNWLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsT0FBa0M7UUFDdEUsT0FBTyxJQUFJLGlDQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLGtCQUNwQyxHQUFHLEVBQUUsSUFBSSxJQUNOLE9BQU8sRUFDVixDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ08sbUJBQW1CLENBQUMsWUFBNkIsRUFBRTtRQUMzRCxTQUFTLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRW5ELElBQUksU0FBUyxDQUFDLE9BQU8sS0FBSyxTQUFTLEVBQUU7WUFDbkMsT0FBTyxTQUFTLENBQUMsT0FBTyxDQUFDO1NBRTFCO2FBQU0sSUFBSSxTQUFTLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRSxFQUFFLGlCQUFpQjtZQUNyRSxPQUFPLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLENBQUM7U0FFbEU7YUFBTTtZQUNMLE1BQU0sSUFBSSxHQUFHLFNBQVMsQ0FBQyxVQUFVLElBQUksVUFBVSxDQUFDLE9BQU8sQ0FBQztZQUN4RCxPQUFPLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUNuRTtJQUNILENBQUM7SUFFTyx5QkFBeUIsQ0FBQyxTQUFpQjtRQUNqRCxNQUFNLFVBQVUsR0FBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLGFBQWEsRUFBRSxHQUFHLElBQUksQ0FBQyxjQUFjLEVBQUUsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDN0YsTUFBTSxPQUFPLEdBQUcsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLHFDQUE4QixDQUFDLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBRXhGLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsMEJBQTBCLEVBQUU7WUFDNUQsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLHFDQUE4QixDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2xGLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLFNBQVMsbUNBQW1DLEtBQUssRUFBRSxDQUFDLENBQUM7U0FDL0c7UUFFRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRU8seUJBQXlCLENBQUMsVUFBc0IsRUFBRSxRQUFpQjtRQUN6RSxNQUFNLFVBQVUsR0FBRztZQUNqQixDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRSxJQUFJLENBQUMsZUFBZTtZQUMzQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsRUFBRSxJQUFJLENBQUMsY0FBYztZQUN6QyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRSxJQUFJLENBQUMsYUFBYTtTQUN4QyxDQUFDO1FBRUYsSUFBSSxPQUFPLEdBQUcsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXJDLElBQUksUUFBUSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2xDLHVDQUF1QztZQUN2QyxPQUFPLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLHFDQUE4QixDQUFDLENBQUMsQ0FBQyxLQUFLLHFDQUE4QixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDakg7UUFFRCwwRUFBMEU7UUFDMUUsa0JBQWtCO1FBRWxCLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsMEJBQTBCLEVBQUU7WUFDNUQsTUFBTSxjQUFjLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDMUgsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsVUFBVSxpREFBaUQsY0FBYyxFQUFFLENBQUMsQ0FBQztTQUMvRztRQUVELE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLHNCQUFzQixDQUFDLFNBQTBCO1FBQ3ZELElBQUksU0FBUyxDQUFDLFVBQVUsS0FBSyxTQUFTLEVBQUU7WUFDdEMsSUFBSSxTQUFTLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtnQkFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQywwRkFBMEYsQ0FBQyxDQUFDO2FBQzdHO1lBQ0QsU0FBUyxtQ0FBTyxTQUFTLEtBQUUsZUFBZSxFQUFFLFNBQVMsQ0FBQyxVQUFVLEdBQUUsQ0FBQztTQUNwRTtRQUVELE1BQU0sbUJBQW1CLEdBQWlDLENBQUMsU0FBUyxFQUFFLFlBQVksRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3ZHLE1BQU0sa0JBQWtCLEdBQUcsbUJBQW1CLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBQzNGLElBQUksa0JBQWtCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLGdCQUFnQixrQkFBa0Isd0NBQXdDLENBQUMsQ0FBQztTQUM3RjtRQUVELElBQUksU0FBUyxDQUFDLFVBQVUsS0FBSyxTQUFTLElBQUksU0FBUyxDQUFDLGVBQWUsS0FBSyxTQUFTLElBQUksU0FBUyxDQUFDLE9BQU8sS0FBSyxTQUFTLEVBQUU7WUFDcEgsa0VBQWtFO1lBQ2xFLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLE9BQU8sRUFBRSxRQUFRLEVBQUUsU0FBUyxDQUFDLFFBQVEsRUFBRSxDQUFDO2FBQUU7WUFDaEgsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsUUFBUSxFQUFFLFFBQVEsRUFBRSxTQUFTLENBQUMsUUFBUSxFQUFFLENBQUM7YUFBRTtZQUNsSCxPQUFPLEVBQUUsVUFBVSxFQUFFLFVBQVUsQ0FBQyxNQUFNLEVBQUUsUUFBUSxFQUFFLFNBQVMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUN4RTtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Q0FDRjtBQXNHRDs7R0FFRztBQUNILE1BQU0sUUFBUSxHQUFXLE1BQU0sQ0FBQztBQWtMaEM7O0dBRUc7QUFDSCxJQUFZLHNCQVVYO0FBVkQsV0FBWSxzQkFBc0I7SUFDaEM7O09BRUc7SUFDSCw2Q0FBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILGlEQUF1QixDQUFBO0FBQ3pCLENBQUMsRUFWVyxzQkFBc0IsR0FBdEIsOEJBQXNCLEtBQXRCLDhCQUFzQixRQVVqQztBQStDRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwQkc7QUFDSCxNQUFhLEdBQUksU0FBUSxPQUFPO0lBZ0s5Qjs7Ozs7T0FLRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBa0IsRUFBRTtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBbkRuQjs7V0FFRztRQUNhLGtCQUFhLEdBQWMsRUFBRSxDQUFDO1FBRTlDOztXQUVHO1FBQ2EsbUJBQWMsR0FBYyxFQUFFLENBQUM7UUFFL0M7O1dBRUc7UUFDYSxvQkFBZSxHQUFjLEVBQUUsQ0FBQztRQXdCaEQ7O1dBRUc7UUFDSyx3QkFBbUIsR0FBMEIsRUFBRSxDQUFDO1FBRXZDLHFDQUFnQyxHQUFHLElBQUkseUJBQWtCLEVBQUUsQ0FBQztRQVczRSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTdCLDBEQUEwRDtRQUMxRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUN2RCxNQUFNLElBQUksS0FBSyxDQUFDLHlGQUF5RixDQUFDLENBQUM7U0FDNUc7UUFFRCxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUNsRSxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyw2R0FBNkcsQ0FBQyxDQUFDO1NBQ2hJO1FBRUQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLDZCQUFjLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFcEQsTUFBTSxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztRQUM5RixNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQ3hGLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxTQUFTLENBQUM7UUFDbEUsSUFBSSxDQUFDLCtCQUErQixHQUFHLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQztRQUU3RSw2Q0FBNkM7UUFDN0MsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHNCQUFNLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzQyxTQUFTO1lBQ1Qsa0JBQWtCO1lBQ2xCLGdCQUFnQjtZQUNoQixlQUFlO1NBQ2hCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLHFCQUFxQixDQUFDO1FBQ2hFLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLHlCQUF5QixDQUFDO1FBQ3hFLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUM7UUFDaEQsSUFBSSxDQUFDLHVCQUF1QixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsd0JBQXdCLENBQUM7UUFDdEUsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUM7UUFFMUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxVQUFHLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUV6RCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBRWpELE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRWpFLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFL0IsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUUsR0FBRyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBRXZGLE1BQU0sbUJBQW1CLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsVUFBVSxFQUFFLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUN6RixNQUFNLGVBQWUsR0FBRyx3QkFBd0IsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFN0gsK0RBQStEO1FBQy9ELElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUVyQixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsTUFBTSxDQUNuRCxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLFVBQVUsS0FBSyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBRXBFLHdEQUF3RDtRQUN4RCxJQUFJLGFBQWEsRUFBRTtZQUNqQixNQUFNLEdBQUcsR0FBRyxJQUFJLGtDQUFrQixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsRUFDL0MsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUMvQyxNQUFNLEdBQUcsR0FBRyxJQUFJLHVDQUF1QixDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7Z0JBQ3JELGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxHQUFHO2dCQUMxQixLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHO2FBQ3pCLENBQUMsQ0FBQztZQUVGLElBQUksQ0FBQyxhQUFnQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRTtnQkFDNUQsWUFBWSxDQUFDLHVCQUF1QixDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFDckQsQ0FBQyxDQUFDLENBQUM7WUFFSCxxQ0FBcUM7WUFDckMsSUFBSSxlQUFlLEdBQUcsQ0FBQyxFQUFFO2dCQUN2QixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksaUJBQVcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztnQkFDbkUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxlQUFlLEVBQUUsbUJBQW1CLENBQUMsQ0FBQzthQUN4RTtTQUNGO1FBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEtBQUssQ0FBQyxVQUFVLEtBQUssS0FBSyxFQUFFO1lBQy9FLE1BQU0sSUFBSSxLQUFLLENBQUMsdUZBQXVGLENBQUMsQ0FBQztTQUMxRztRQUVELElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDbkUsTUFBTSxVQUFVLEdBQUcsSUFBSSw2QkFBYSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7Z0JBQ3ZELGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtnQkFDbEMsSUFBSSxFQUFFLHVCQUFpQixDQUFDLE9BQU87YUFDaEMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxVQUFVLEdBQUcsSUFBSSx1Q0FBdUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUMvRCxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7Z0JBQ2pCLFlBQVksRUFBRSxVQUFVLENBQUMsR0FBRzthQUM3QixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsWUFBWSxHQUFHLFVBQVUsQ0FBQyxHQUFHLENBQUM7WUFFbkMscUVBQXFFO1lBQ3JFLE1BQU0sbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDOUYsTUFBTSxhQUFhLEdBQUcsdUJBQWdCLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMvRixNQUFNLGdCQUFnQixHQUFHLElBQUksNkNBQTZCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO2dCQUNuRixhQUFhO2dCQUNiLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTthQUNoQyxDQUFDLENBQUM7WUFFSCwrRUFBK0U7WUFDL0UsaURBQWlEO1lBQ2pELGdIQUFnSDtZQUNoSCxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBRWhELE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDO1lBQ2xELEtBQUssTUFBTSxDQUFDLFlBQVksRUFBRSxVQUFVLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUN2RSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxFQUFFLFVBQVUsQ0FBQyxDQUFDO2FBQ2pEO1NBQ0Y7UUFFRCwyRUFBMkU7UUFDM0UsNkVBQTZFO1FBQzdFLHVFQUF1RTtRQUN2RSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxFQUFFLENBQUM7WUFDdEQsS0FBSyxNQUFNLENBQUMsVUFBVSxFQUFFLFFBQVEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsRUFBRTtnQkFDckUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFVBQVUsRUFBRSxRQUFRLENBQUMsQ0FBQzthQUMvQztTQUNGO0lBQ0gsQ0FBQztJQXhRRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUNoRixPQUFPLElBQUksV0FBVyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxPQUF5QjtRQUM5RSxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQztlQUNoQyxZQUFLLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUM7ZUFDbkMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDO2VBQzFELE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQzdELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUVELE1BQU0sTUFBTSxHQUE0QixhQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXBFLHlDQUF5QztRQUN6QyxJQUFJLE9BQU8sQ0FBQyxLQUFLLEVBQUU7WUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQztTQUFFO1FBQ3hELElBQUksT0FBTyxDQUFDLE9BQU8sRUFBRTtZQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDO1NBQUU7UUFDOUQsSUFBSSxPQUFPLENBQUMsU0FBUyxLQUFLLFNBQVMsRUFBRTtZQUNuQyxNQUFNLENBQUMsU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1NBQ3pEO1FBRUQsTUFBTSxVQUFVLEdBQTZCLHNCQUFlLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRTtZQUMzRSxRQUFRLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDNUIsS0FBSyxFQUFFO2dCQUNMLE1BQU07Z0JBQ04sdUJBQXVCLEVBQUUsSUFBSTtnQkFDN0Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjthQUN0QjtZQUMxQixVQUFVLEVBQUUsU0FBUztTQUN0QixDQUFDLENBQUMsS0FBSyxDQUFDO1FBRVQsT0FBTyxJQUFJLFdBQVcsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLFVBQVUsSUFBSSxlQUFlLEVBQUUsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBRTNGOztXQUVHO1FBQ0gsU0FBUyxhQUFhLENBQUMsSUFBNEM7WUFDakUsTUFBTSxNQUFNLEdBQStCLEVBQUUsQ0FBQztZQUM5QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLEVBQUU7Z0JBQ3RELE1BQU0sQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDO2FBQy9CO1lBQ0QsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUE4TUQ7Ozs7T0FJRztJQUNJLGFBQWEsQ0FBQyxFQUFVLEVBQUUsT0FBMkI7UUFDMUQsT0FBTyxJQUFJLGlDQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdEMsT0FBTyxFQUFFLDJDQUE0QixDQUFDLEVBQUU7WUFDeEMsR0FBRyxFQUFFLElBQUk7WUFDVCxPQUFPO1NBQ1IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsT0FBMkI7UUFDaEUsT0FBTyxJQUFJLGlDQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdEMsT0FBTyxFQUFFLDJDQUE0QixDQUFDLFFBQVE7WUFDOUMsR0FBRyxFQUFFLElBQUk7WUFDVCxPQUFPO1NBQ1IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQixDQUFDLFFBQXFCLEVBQUUsUUFBZ0IsRUFBRSxTQUEwQjtRQUMzRixNQUFNLFVBQVUsR0FBbUIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBbUIsQ0FBQztRQUN6RixLQUFLLE1BQU0sR0FBRyxJQUFJLFVBQVUsRUFBRTtZQUM1QixJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO2dCQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixTQUFTLCtCQUErQixHQUFHLEVBQUUsQ0FBQyxDQUFDO2FBQ3ZGO1NBQ0Y7UUFFRCxRQUFRLENBQUMsWUFBWSxDQUFDO1lBQ3BCLEdBQUcsRUFBRSxJQUFJO1lBQ1QsVUFBVSxFQUFFLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQztZQUN6QyxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWlDO1NBQ3ZELENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7O09BR0c7SUFDSyxhQUFhO1FBQ25CLE1BQU0scUJBQXFCLEdBQTBCLEVBQUUsQ0FBQztRQUV4RCxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QyxJQUFJLE1BQU0sQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFO2dCQUNqQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ25DLFNBQVM7YUFDVjtZQUNELElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsTUFBTSxjQUFjLEdBQUcscUJBQXFCLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUM7UUFDcEYsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLHVCQUF1QixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3ZGLEtBQUssTUFBTSxNQUFNLElBQUkscUJBQXFCLEVBQUU7WUFDMUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3hEO0lBQ0gsQ0FBQztJQUVPLHFCQUFxQixDQUFDLFlBQWlDLEVBQUUsUUFBZ0I7UUFDL0UsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsRUFBRTtZQUM3QyxJQUFJLFlBQVksQ0FBQyxRQUFRLEtBQUssSUFBSSxFQUFFO2dCQUNsQywrRUFBK0U7Z0JBQy9FLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUN4QyxPQUFPO2FBQ1I7WUFFRCxNQUFNLElBQUksR0FBRyxlQUFRLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztZQUNoRCxNQUFNLFdBQVcsR0FBZ0I7Z0JBQy9CLGdCQUFnQixFQUFFLElBQUk7Z0JBQ3RCLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztnQkFDakIsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztnQkFDbEQsbUJBQW1CLEVBQUUsQ0FBQyxZQUFZLENBQUMsVUFBVSxLQUFLLFVBQVUsQ0FBQyxNQUFNLENBQUM7YUFDckUsQ0FBQztZQUVGLElBQUksTUFBYyxDQUFDO1lBQ25CLFFBQVEsWUFBWSxDQUFDLFVBQVUsRUFBRTtnQkFDL0IsS0FBSyxVQUFVLENBQUMsTUFBTTtvQkFDcEIsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDL0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7b0JBQ3RDLE1BQU0sR0FBRyxZQUFZLENBQUM7b0JBQ3RCLE1BQU07Z0JBQ1IsS0FBSyxVQUFVLENBQUMsT0FBTztvQkFDckIsTUFBTSxhQUFhLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDakUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7b0JBQ3hDLE1BQU0sR0FBRyxhQUFhLENBQUM7b0JBQ3ZCLE1BQU07Z0JBQ1IsS0FBSyxVQUFVLENBQUMsUUFBUTtvQkFDdEIsTUFBTSxjQUFjLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDbEUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7b0JBQzFDLE1BQU0sR0FBRyxjQUFjLENBQUM7b0JBQ3hCLE1BQU07Z0JBQ1I7b0JBQ0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsWUFBWSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7YUFDM0U7WUFFRCx1RUFBdUU7WUFDdkUsTUFBTSxvQkFBb0IsR0FBRyxDQUFDLHlCQUFTLENBQUMsc0JBQXNCLENBQUMsQ0FBQztZQUNoRSxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFVBQUcsQ0FBQyxjQUFjLEVBQUUsWUFBWSxDQUFDLElBQUksRUFBRSxFQUFDLG9CQUFvQixFQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzVGLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksVUFBRyxDQUFDLGNBQWMsRUFBRSxrQkFBa0IsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBQyxvQkFBb0IsRUFBQyxDQUFDLENBQUMsQ0FBQztRQUN4SCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBM1lILGtCQTRZQztBQTNZQzs7OztHQUlHO0FBQ29CLHNCQUFrQixHQUFXLGFBQWEsQ0FBQztBQUVsRTs7OztHQUlHO0FBQ29CLG1CQUFlLEdBQTBCO0lBQzlEO1FBQ0UsVUFBVSxFQUFFLFVBQVUsQ0FBQyxNQUFNO1FBQzdCLElBQUksRUFBRSx3QkFBaUIsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDO0tBQzNDO0lBQ0Q7UUFDRSxVQUFVLEVBQUUsVUFBVSxDQUFDLE9BQU87UUFDOUIsSUFBSSxFQUFFLHdCQUFpQixDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUM7S0FDNUM7Q0FDRixDQUFDO0FBd1hKLE1BQU0sY0FBYyxHQUFHLHFCQUFxQixDQUFDO0FBQzdDLE1BQU0sY0FBYyxHQUFHLHFCQUFxQixDQUFDO0FBRTdDLFNBQVMsa0JBQWtCLENBQUMsSUFBZ0I7SUFDMUMsUUFBUSxJQUFJLEVBQUU7UUFDWixLQUFLLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLFFBQVEsQ0FBQztRQUN4QyxLQUFLLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLFNBQVMsQ0FBQztRQUMxQyxLQUFLLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLFVBQVUsQ0FBQztLQUM3QztBQUNILENBQUM7QUE4QkQ7Ozs7R0FJRztBQUNILE1BQWEsTUFBTyxTQUFRLGVBQVE7SUF3RGxDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBa0I7UUFDMUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQWpCbkI7O1dBRUc7UUFDYSx1QkFBa0IsR0FBa0IsRUFBRSxDQUFDO1FBU3RDLHFDQUFnQyxHQUFHLElBQUkseUJBQWtCLEVBQUUsQ0FBQztRQU8zRSxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRWhFLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksVUFBRyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFFekQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUMvQyxNQUFNLE1BQU0sR0FBRyxJQUFJLHlCQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUMzQyxLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7WUFDbEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtTQUMvQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUM7UUFDM0IsSUFBSSxDQUFDLFdBQVcsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxNQUFNLENBQUMsb0JBQW9CLENBQUM7UUFDMUQsSUFBSSxDQUFDLG9CQUFvQixHQUFHLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQztRQUV0RCx5RUFBeUU7UUFDekUsc0VBQXNFO1FBQ3RFLElBQUksQ0FBQyxXQUFXLEdBQUcsd0JBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ2hHLElBQUksQ0FBQyw2QkFBNkIsR0FBRyxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUN4RyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxNQUFNLENBQUM7UUFFaEMsTUFBTSxLQUFLLEdBQUcsSUFBSSw2QkFBYSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDbEQsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1NBQ25CLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxVQUFVLEdBQUcsRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRTlDLG1FQUFtRTtRQUNuRSxJQUFJLDhDQUE4QixDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUNoRSxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsWUFBWSxFQUFFLEtBQUssQ0FBQyxHQUFHO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywrQkFBK0IsR0FBRyxJQUFJLENBQUMsZ0NBQWdDLENBQUM7SUFDL0UsQ0FBQztJQTNGTSxNQUFNLENBQUMsV0FBVyxDQUFDLENBQU07UUFDOUIsT0FBTyxpQkFBaUIsSUFBSSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVNLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUN0RixPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQXVGRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUIsQ0FBQyxTQUFpQixFQUFFLGlCQUE4QjtRQUM5RSxNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMvQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFZO1lBQzFDLG9CQUFvQixFQUFFLFdBQVc7WUFDakMsU0FBUztTQUNWLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFFNUMsNERBQTREO1FBQzVELG9DQUFvQztRQUNwQyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDO0lBQzFCLENBQUM7SUFFRDs7O09BR0c7SUFDSSxrQkFBa0IsQ0FBQyxZQUFvQjtRQUM1QyxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsRUFBRTtZQUM1QixVQUFVLEVBQUUsVUFBVSxDQUFDLFdBQVc7WUFDbEMsUUFBUSxFQUFFLFlBQVk7WUFDdEIsMkJBQTJCLEVBQUUsSUFBSTtTQUNsQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRLENBQUMsRUFBVSxFQUFFLE9BQXdCO1FBQ2xELElBQUksT0FBTyxDQUFDLG9CQUFvQixJQUFJLE9BQU8sQ0FBQyx3QkFBd0IsRUFBRTtZQUNwRSxNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7U0FDOUY7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFRLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNuQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFZO1lBQzFDLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2hJLHdCQUF3QixFQUFFLE9BQU8sQ0FBQyxvQkFBb0I7WUFDdEQsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsRUFBRSxPQUFPLENBQUMsUUFBUTtTQUM3RCxDQUFDLENBQUM7UUFFSCxJQUFJLE9BQU8sQ0FBQywyQkFBMkIsRUFBRTtZQUN2QyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ2xEO0lBQ0gsQ0FBQztJQUVNLG1CQUFtQixDQUFDLEVBQVUsRUFBRSxVQUF1QjtRQUM1RCxJQUFJLENBQUMsV0FBVyxHQUFHLFVBQVUsQ0FBQztRQUU5QixNQUFNLEtBQUssR0FBRyxnQkFBUyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDcEUsTUFBTSxLQUFLLEdBQUcsZ0JBQVMsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDO1FBQ3BFLElBQUkseUNBQTJCLENBQUMsS0FBSyxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUMvRCxVQUFVO1lBQ1YsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUEzS0Qsd0JBMktDO0FBMkNEOztHQUVHO0FBQ0gsSUFBWSxVQThCWDtBQTlCRCxXQUFZLFVBQVU7SUFDcEI7O09BRUc7SUFDSCx3RUFBMEQsQ0FBQTtJQUUxRDs7T0FFRztJQUNILGlDQUFtQixDQUFBO0lBRW5COztPQUVHO0lBQ0gsbUNBQXFCLENBQUE7SUFFckI7O09BRUc7SUFDSCx3Q0FBMEIsQ0FBQTtJQUUxQjs7T0FFRztJQUNILG9EQUFzQyxDQUFBO0lBRXRDOztPQUVHO0lBQ0gsNkRBQStDLENBQUE7QUFDakQsQ0FBQyxFQTlCVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQThCckI7QUFFRCxTQUFTLG9CQUFvQixDQUFDLFVBQXNCO0lBQ2xELE9BQU8sQ0FBQztRQUNOLENBQUMsVUFBVSxDQUFDLDRCQUE0QixDQUFDLEVBQUUsNkJBQTZCO1FBQ3hFLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxFQUFFLFdBQVc7UUFDakMsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLEVBQUUsWUFBWTtRQUNuQyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjO1FBQ3hDLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsb0JBQW9CO1FBQ3BELENBQUMsVUFBVSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsd0JBQXdCO0tBQzlELENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUNqQixDQUFDO0FBV0Q7O0dBRUc7QUFDSCxNQUFhLFlBQWEsU0FBUSxNQUFNO0lBRS9CLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNsRyxPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxhQUFhO1FBQ2xCLDZDQUE2QztRQUM3QyxNQUFNLEdBQUcsR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUNoRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsWUFBWSxFQUFFLElBQUksc0JBQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUNwQyxNQUFNLEVBQUUsS0FBSzthQUNkLENBQUMsQ0FBQyxnQkFBZ0I7U0FDcEIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUF6QkQsb0NBeUJDO0FBV0Q7O0dBRUc7QUFDSCxNQUFhLGFBQWMsU0FBUSxNQUFNO0lBRWhDLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUNwRyxPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUIsQ0FBQztDQUNGO0FBVEQsc0NBU0M7QUFFRCxTQUFTLFdBQVcsQ0FBSSxLQUFvQixFQUFFLFlBQWU7SUFDM0QsT0FBTyxLQUFLLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztBQUNwRCxDQUFDO0FBRUQsTUFBTSxXQUFZLFNBQVEsT0FBTztJQVMvQixZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CLEVBQUUsWUFBcUI7UUFDbkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUhILG9DQUErQixHQUFnQixJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFLdEYsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUM7UUFDakQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBQ3ZDLElBQUksQ0FBQywwQkFBMEIsR0FBRyxZQUFZLENBQUM7UUFFL0MsaUNBQWlDO1FBQ2pDLE1BQU0sR0FBRyxHQUFHLElBQUksd0JBQWlCLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRSxLQUFLLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLHlCQUF5QixFQUFFLFVBQVUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixFQUFFLGlCQUFpQixFQUFFLG1CQUFtQixFQUFFLDJCQUEyQixDQUFDLENBQUM7UUFDbk8sTUFBTSxJQUFJLEdBQUcsSUFBSSx3QkFBaUIsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQywwQkFBMEIsRUFBRSxVQUFVLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxrQkFBa0IsRUFBRSxvQkFBb0IsRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO1FBQzNPLE1BQU0sR0FBRyxHQUFHLElBQUksd0JBQWlCLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsMkJBQTJCLEVBQUUsVUFBVSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCLEVBQUUsbUJBQW1CLEVBQUUscUJBQXFCLEVBQUUsNkJBQTZCLENBQUMsQ0FBQztRQUNqUCxnQ0FBZ0M7UUFFaEMsSUFBSSxDQUFDLGFBQWEsR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsZUFBZSxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUMsQ0FBQztDQUNGO0FBRUQsTUFBTSxXQUFZLFNBQVEsT0FBTztJQVMvQixZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCLEVBQUUsWUFBcUI7UUFDOUYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVBILG9DQUErQixHQUFnQixJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFTdEYsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsMEJBQTBCLEdBQUcsWUFBWSxDQUFDO1FBRS9DLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO1FBQzlDLE1BQU0saUJBQWlCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBUyxPQUFPLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQyxFQUFFO1lBQ3ZGLE9BQU8sV0FBVyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNwRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDTCxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLGlCQUFpQixHQUFHLGlCQUFpQixDQUFDO1FBRTNDLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDOUYsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNoRyxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3BHLENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxZQUFvQyxFQUFFLGVBQXlDO1FBQzFHLE9BQU8sT0FBTyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxLQUFLLGVBQWUsQ0FBQyxFQUNuRixXQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxXQUFpQztRQUM1RCxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBVyxDQUFDO1FBQ2pDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxXQUFXLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUNuRCxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxHQUFHLFdBQVcsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFO2dCQUM5RSxnQkFBZ0IsRUFBRSxTQUFTLENBQUMsZ0JBQWdCO2dCQUM1QyxRQUFRLEVBQUUsU0FBUyxDQUFDLFFBQVE7Z0JBQzVCLFlBQVksRUFBRSxTQUFTLENBQUMsWUFBWTthQUNyQyxDQUFDLENBQUMsQ0FBQztTQUNMO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUFFRCxTQUFTLE9BQU8sQ0FBTyxFQUFPLEVBQUUsRUFBaUI7SUFDL0MsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQUssQ0FBQztJQUMzQixLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUNsQixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDcEI7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxNQUFNLG1CQUFtQjtJQUd2QjtRQUZpQixnQkFBVyxHQUFHLElBQUksS0FBSyxFQUFlLENBQUM7UUFHdEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLHNCQUFlLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRTtZQUM5QixJQUFJLGVBQWU7Z0JBQ2pCLE1BQU0sR0FBRyxHQUFHLEVBQUUsQ0FBQztnQkFDZixLQUFLLE1BQU0sR0FBRyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7b0JBQ2xDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxzQkFBZSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQztpQkFDdkQ7Z0JBQ0QsT0FBTyxHQUFHLENBQUM7WUFDYixDQUFDO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksR0FBRyxDQUFDLEdBQWdCO1FBQ3pCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQzdCLENBQUM7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsU0FBUyxHQUFHLENBQUksQ0FBSSxFQUFFLEVBQWtCO0lBQ3RDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNOLE9BQU8sQ0FBQyxDQUFDO0FBQ1gsQ0FBQztBQUVELE1BQU0sY0FBZSxTQUFRLGVBQVE7SUFNbkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBTkgsb0NBQStCLEdBQWdCLElBQUkseUJBQWtCLEVBQUUsQ0FBQztRQVF0RixJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN2QixNQUFNLEdBQUcsR0FBRyxZQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUM7Z0JBQzVDLENBQUMsQ0FBQyxPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxJQUFJLEVBQUUsR0FBRztnQkFDakMsQ0FBQyxDQUFDLElBQUksS0FBSyxDQUFDLFFBQVEsR0FBRyxDQUFDO1lBQzFCLDRDQUE0QztZQUM1QyxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyw4Q0FBOEMsR0FBRyxxSUFBcUksQ0FBQyxDQUFDO1NBQy9NO1FBRUQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUMvQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLFVBQVUsR0FBRztZQUNoQiw2SEFBNkg7WUFDN0gsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFhO1NBQ2xDLENBQUM7SUFDSixDQUFDO0lBRU0sbUJBQW1CLENBQUMsRUFBVSxFQUFFLFVBQXVCO1FBQzVELE1BQU0sS0FBSyxHQUFHLGdCQUFTLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUNwRSxNQUFNLEtBQUssR0FBRyxnQkFBUyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUM7UUFDcEUsSUFBSSx5Q0FBMkIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQy9ELFVBQVU7WUFDVixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQUVEOzs7Ozs7Ozs7O0dBVUc7QUFDSCxTQUFTLHdCQUF3QixDQUFDLGNBQWtDLEVBQUUsWUFBbUMsRUFBRSxPQUFlO0lBQ3hILE1BQU0saUJBQWlCLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLEtBQUssVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3RGLE1BQU0sZ0JBQWdCLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLEtBQUssVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBRXBGLE1BQU0sS0FBSyxHQUFHLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRW5ILElBQUksS0FBSyxLQUFLLENBQUMsSUFBSSxpQkFBaUIsRUFBRTtRQUNwQywyQ0FBMkM7UUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxvS0FBb0ssQ0FBQyxDQUFDO0tBQ3ZMO0lBRUQsSUFBSSxLQUFLLEdBQUcsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7UUFDbEMsMkNBQTJDO1FBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUlBQXVJLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQ3pMO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILE1BQU0sZUFBZSxHQUE2QjtJQUNoRCxpQkFBaUIsRUFBRSxFQUFFO0lBQ3JCLGlCQUFpQixFQUFFLFNBQVM7SUFDNUIsbUJBQW1CLEVBQUUsU0FBUztJQUM5QiwyQkFBMkIsRUFBRSxTQUFTO0lBQ3RDLGdCQUFnQixFQUFFLFNBQVM7SUFDM0Isa0JBQWtCLEVBQUUsU0FBUztJQUM3QiwwQkFBMEIsRUFBRSxTQUFTO0lBQ3JDLGVBQWUsRUFBRSxTQUFTO0lBQzFCLGlCQUFpQixFQUFFLFNBQVM7SUFDNUIseUJBQXlCLEVBQUUsU0FBUztJQUNwQyxZQUFZLEVBQUU7UUFDWjtZQUNFLElBQUksRUFBRSxRQUFRO1lBQ2QsSUFBSSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNO1lBQ3JDLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxnQkFBZ0IsRUFBRSxVQUFVO29CQUM1QixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7aUJBQzNCO2dCQUNEO29CQUNFLGdCQUFnQixFQUFFLFVBQVU7b0JBQzVCLFFBQVEsRUFBRSxTQUFTO29CQUNuQixZQUFZLEVBQUUsWUFBWTtpQkFDM0I7YUFDRjtTQUNGO1FBQ0Q7WUFDRSxJQUFJLEVBQUUsU0FBUztZQUNmLElBQUksRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsT0FBTztZQUN0QyxPQUFPLEVBQUU7Z0JBQ1A7b0JBQ0UsZ0JBQWdCLEVBQUUsVUFBVTtvQkFDNUIsUUFBUSxFQUFFLFNBQVM7b0JBQ25CLFlBQVksRUFBRSxZQUFZO2lCQUMzQjtnQkFDRDtvQkFDRSxnQkFBZ0IsRUFBRSxVQUFVO29CQUM1QixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7aUJBQzNCO2FBQ0Y7U0FDRjtLQUNGO0lBQ0QsS0FBSyxFQUFFLFdBQVc7Q0FDbkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbmNyZXRlRGVwZW5kYWJsZSwgQ29uc3RydWN0LCBDb250ZXh0UHJvdmlkZXIsIERlcGVuZGFibGVUcmFpdCwgSUNvbnN0cnVjdCxcbiAgICBJRGVwZW5kYWJsZSwgSVJlc291cmNlLCBMYXp5LCBSZXNvdXJjZSwgU3RhY2ssIFRhZywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCBjeGFwaSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2N4LWFwaScpO1xuaW1wb3J0IHtcbiAgQ2ZuRUlQLCBDZm5JbnRlcm5ldEdhdGV3YXksIENmbk5hdEdhdGV3YXksIENmblJvdXRlLCBDZm5Sb3V0ZVRhYmxlLCBDZm5TdWJuZXQsXG4gIENmblN1Ym5ldFJvdXRlVGFibGVBc3NvY2lhdGlvbiwgQ2ZuVlBDLCBDZm5WUENHYXRld2F5QXR0YWNobWVudCwgQ2ZuVlBOR2F0ZXdheSwgQ2ZuVlBOR2F0ZXdheVJvdXRlUHJvcGFnYXRpb24gfSBmcm9tICcuL2VjMi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgTmF0UHJvdmlkZXIgfSBmcm9tICcuL25hdCc7XG5pbXBvcnQgeyBJTmV0d29ya0FjbCwgTmV0d29ya0FjbCwgU3VibmV0TmV0d29ya0FjbEFzc29jaWF0aW9uIH0gZnJvbSAnLi9uZXR3b3JrLWFjbCc7XG5pbXBvcnQgeyBOZXR3b3JrQnVpbGRlciB9IGZyb20gJy4vbmV0d29yay11dGlsJztcbmltcG9ydCB7IGFsbFJvdXRlVGFibGVJZHMsIGRlZmF1bHRTdWJuZXROYW1lLCBJbXBvcnRTdWJuZXRHcm91cCwgc3VibmV0R3JvdXBOYW1lRnJvbUNvbnN0cnVjdElkLCBzdWJuZXRJZCAgfSBmcm9tICcuL3V0aWwnO1xuaW1wb3J0IHsgR2F0ZXdheVZwY0VuZHBvaW50LCBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLCBHYXRld2F5VnBjRW5kcG9pbnRPcHRpb25zLCBJbnRlcmZhY2VWcGNFbmRwb2ludCwgSW50ZXJmYWNlVnBjRW5kcG9pbnRPcHRpb25zIH0gZnJvbSAnLi92cGMtZW5kcG9pbnQnO1xuaW1wb3J0IHsgVnBjTG9va3VwT3B0aW9ucyB9IGZyb20gJy4vdnBjLWxvb2t1cCc7XG5pbXBvcnQgeyBWcG5Db25uZWN0aW9uLCBWcG5Db25uZWN0aW9uT3B0aW9ucywgVnBuQ29ubmVjdGlvblR5cGUgfSBmcm9tICcuL3Zwbic7XG5cbmNvbnN0IFZQQ19TVUJORVRfU1lNQk9MID0gU3ltYm9sLmZvcignQGF3cy1jZGsvYXdzLWVjMi5WcGNTdWJuZXQnKTtcblxuZXhwb3J0IGludGVyZmFjZSBJU3VibmV0IGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIFRoZSBBdmFpbGFiaWxpdHkgWm9uZSB0aGUgc3VibmV0IGlzIGxvY2F0ZWQgaW5cbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldElkIGZvciB0aGlzIHBhcnRpY3VsYXIgc3VibmV0XG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlcGVuZGFibGUgdGhhdCBjYW4gYmUgZGVwZW5kZWQgdXBvbiB0byBmb3JjZSBpbnRlcm5ldCBjb25uZWN0aXZpdHkgZXN0YWJsaXNoZWQgb24gdGhlIFZQQ1xuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGU7XG5cbiAgLyoqXG4gICAqIFRoZSByb3V0ZSB0YWJsZSBmb3IgdGhpcyBzdWJuZXRcbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlVGFibGU6IElSb3V0ZVRhYmxlO1xuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgYSBOZXR3b3JrIEFDTCB3aXRoIHRoaXMgc3VibmV0XG4gICAqXG4gICAqIEBwYXJhbSBhY2wgVGhlIE5ldHdvcmsgQUNMIHRvIGFzc29jaWF0ZVxuICAgKi9cbiAgYXNzb2NpYXRlTmV0d29ya0FjbChpZDogc3RyaW5nLCBhY2w6IElOZXR3b3JrQWNsKTogdm9pZDtcbn1cblxuLyoqXG4gKiBBbiBhYnN0cmFjdCByb3V0ZSB0YWJsZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSb3V0ZVRhYmxlIHtcbiAgLyoqXG4gICAqIFJvdXRlIHRhYmxlIElEXG4gICAqL1xuICByZWFkb25seSByb3V0ZVRhYmxlSWQ6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJVnBjIGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIElkZW50aWZpZXIgZm9yIHRoaXMgVlBDXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHZwY0lkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IHByaXZhdGVTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgaXNvbGF0ZWQgc3VibmV0cyBpbiB0aGlzIFZQQ1xuICAgKi9cbiAgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIEFacyBmb3IgdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogSWRlbnRpZmllciBmb3IgdGhlIFZQTiBnYXRld2F5XG4gICAqL1xuICByZWFkb25seSB2cG5HYXRld2F5SWQ/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBEZXBlbmRhYmxlIHRoYXQgY2FuIGJlIGRlcGVuZGVkIHVwb24gdG8gZm9yY2UgaW50ZXJuZXQgY29ubmVjdGl2aXR5IGVzdGFibGlzaGVkIG9uIHRoZSBWUENcbiAgICovXG4gIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlO1xuXG4gIC8qKlxuICAgKiBSZXR1cm4gaW5mb3JtYXRpb24gb24gdGhlIHN1Ym5ldHMgYXBwcm9wcmlhdGUgZm9yIHRoZSBnaXZlbiBzZWxlY3Rpb24gc3RyYXRlZ3lcbiAgICpcbiAgICogUmVxdWlyZXMgdGhhdCBhdCBsZWFzdCBvbmUgc3VibmV0IGlzIG1hdGNoZWQsIHRocm93cyBhIGRlc2NyaXB0aXZlXG4gICAqIGVycm9yIG1lc3NhZ2Ugb3RoZXJ3aXNlLlxuICAgKi9cbiAgc2VsZWN0U3VibmV0cyhzZWxlY3Rpb24/OiBTdWJuZXRTZWxlY3Rpb24pOiBTZWxlY3RlZFN1Ym5ldHM7XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgVlBOIGNvbm5lY3Rpb24gdG8gdGhpcyBWUENcbiAgICovXG4gIGFkZFZwbkNvbm5lY3Rpb24oaWQ6IHN0cmluZywgb3B0aW9uczogVnBuQ29ubmVjdGlvbk9wdGlvbnMpOiBWcG5Db25uZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IGdhdGV3YXkgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICovXG4gIGFkZEdhdGV3YXlFbmRwb2ludChpZDogc3RyaW5nLCBvcHRpb25zOiBHYXRld2F5VnBjRW5kcG9pbnRPcHRpb25zKTogR2F0ZXdheVZwY0VuZHBvaW50XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgaW50ZXJmYWNlIGVuZHBvaW50IHRvIHRoaXMgVlBDXG4gICAqL1xuICBhZGRJbnRlcmZhY2VFbmRwb2ludChpZDogc3RyaW5nLCBvcHRpb25zOiBJbnRlcmZhY2VWcGNFbmRwb2ludE9wdGlvbnMpOiBJbnRlcmZhY2VWcGNFbmRwb2ludFxufVxuXG4vKipcbiAqIFRoZSB0eXBlIG9mIFN1Ym5ldFxuICovXG5leHBvcnQgZW51bSBTdWJuZXRUeXBlIHtcbiAgLyoqXG4gICAqIElzb2xhdGVkIFN1Ym5ldHMgZG8gbm90IHJvdXRlIHRyYWZmaWMgdG8gdGhlIEludGVybmV0IChpbiB0aGlzIFZQQykuXG4gICAqXG4gICAqIFRoaXMgY2FuIGJlIGdvb2QgZm9yIHN1Ym5ldHMgd2l0aCBSRFMgb3IgRWxhc3RpY2FjaGUgaW5zdGFuY2VzLFxuICAgKiBvciB3aGljaCByb3V0ZSBJbnRlcm5ldCB0cmFmZmljIHRocm91Z2ggYSBwZWVyIFZQQy5cbiAgICovXG4gIElTT0xBVEVEID0gJ0lzb2xhdGVkJyxcblxuICAvKipcbiAgICogU3VibmV0IHRoYXQgcm91dGVzIHRvIHRoZSBpbnRlcm5ldCwgYnV0IG5vdCB2aWNlIHZlcnNhLlxuICAgKlxuICAgKiBJbnN0YW5jZXMgaW4gYSBwcml2YXRlIHN1Ym5ldCBjYW4gY29ubmVjdCB0byB0aGUgSW50ZXJuZXQsIGJ1dCB3aWxsIG5vdFxuICAgKiBhbGxvdyBjb25uZWN0aW9ucyB0byBiZSBpbml0aWF0ZWQgZnJvbSB0aGUgSW50ZXJuZXQuIEludGVybmV0IHRyYWZmaWMgd2lsbFxuICAgKiBiZSByb3V0ZWQgdmlhIGEgTkFUIEdhdGV3YXkuXG4gICAqXG4gICAqIE5vcm1hbGx5IGEgUHJpdmF0ZSBzdWJuZXQgd2lsbCB1c2UgYSBOQVQgZ2F0ZXdheSBpbiB0aGUgc2FtZSBBWiwgYnV0XG4gICAqIGlmIGBuYXRHYXRld2F5c2AgaXMgdXNlZCB0byByZWR1Y2UgdGhlIG51bWJlciBvZiBOQVQgZ2F0ZXdheXMsIGEgTkFUXG4gICAqIGdhdGV3YXkgZnJvbSBhbm90aGVyIEFaIHdpbGwgYmUgdXNlZCBpbnN0ZWFkLlxuICAgKi9cbiAgUFJJVkFURSA9ICdQcml2YXRlJyxcblxuICAvKipcbiAgICogU3VibmV0IGNvbm5lY3RlZCB0byB0aGUgSW50ZXJuZXRcbiAgICpcbiAgICogSW5zdGFuY2VzIGluIGEgUHVibGljIHN1Ym5ldCBjYW4gY29ubmVjdCB0byB0aGUgSW50ZXJuZXQgYW5kIGNhbiBiZVxuICAgKiBjb25uZWN0ZWQgdG8gZnJvbSB0aGUgSW50ZXJuZXQgYXMgbG9uZyBhcyB0aGV5IGFyZSBsYXVuY2hlZCB3aXRoIHB1YmxpY1xuICAgKiBJUHMgKGNvbnRyb2xsZWQgb24gdGhlIEF1dG9TY2FsaW5nR3JvdXAgb3Igb3RoZXIgY29uc3RydWN0cyB0aGF0IGxhdW5jaFxuICAgKiBpbnN0YW5jZXMpLlxuICAgKlxuICAgKiBQdWJsaWMgc3VibmV0cyByb3V0ZSBvdXRib3VuZCB0cmFmZmljIHZpYSBhbiBJbnRlcm5ldCBHYXRld2F5LlxuICAgKi9cbiAgUFVCTElDID0gJ1B1YmxpYydcbn1cblxuLyoqXG4gKiBDdXN0b21pemUgc3VibmV0cyB0aGF0IGFyZSBzZWxlY3RlZCBmb3IgcGxhY2VtZW50IG9mIEVOSXNcbiAqXG4gKiBDb25zdHJ1Y3RzIHRoYXQgYWxsb3cgY3VzdG9taXphdGlvbiBvZiBWUEMgcGxhY2VtZW50IHVzZSBwYXJhbWV0ZXJzIG9mIHRoaXNcbiAqIHR5cGUgdG8gcHJvdmlkZSBwbGFjZW1lbnQgc2V0dGluZ3MuXG4gKlxuICogQnkgZGVmYXVsdCwgdGhlIGluc3RhbmNlcyBhcmUgcGxhY2VkIGluIHRoZSBwcml2YXRlIHN1Ym5ldHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3VibmV0U2VsZWN0aW9uIHtcbiAgLyoqXG4gICAqIFNlbGVjdCBhbGwgc3VibmV0cyBvZiB0aGUgZ2l2ZW4gdHlwZVxuICAgKlxuICAgKiBBdCBtb3N0IG9uZSBvZiBgc3VibmV0VHlwZWAgYW5kIGBzdWJuZXRHcm91cE5hbWVgIGNhbiBiZSBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgU3VibmV0VHlwZS5QUklWQVRFIChvciBJU09MQVRFRCBvciBQVUJMSUMgaWYgdGhlcmUgYXJlIG5vIFBSSVZBVEUgc3VibmV0cylcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldFR5cGU/OiBTdWJuZXRUeXBlO1xuXG4gIC8qKlxuICAgKiBTZWxlY3QgdGhlIHN1Ym5ldCBncm91cCB3aXRoIHRoZSBnaXZlbiBuYW1lXG4gICAqXG4gICAqIFNlbGVjdCB0aGUgc3VibmV0IGdyb3VwIHdpdGggdGhlIGdpdmVuIG5hbWUuIFRoaXMgb25seSBuZWVkc1xuICAgKiB0byBiZSB1c2VkIGlmIHlvdSBoYXZlIG11bHRpcGxlIHN1Ym5ldCBncm91cHMgb2YgdGhlIHNhbWUgdHlwZVxuICAgKiBhbmQgeW91IG5lZWQgdG8gZGlzdGluZ3Vpc2ggYmV0d2VlbiB0aGVtLiBPdGhlcndpc2UsIHByZWZlclxuICAgKiBgc3VibmV0VHlwZWAuXG4gICAqXG4gICAqIFRoaXMgZmllbGQgZG9lcyBub3Qgc2VsZWN0IGluZGl2aWR1YWwgc3VibmV0cywgaXQgc2VsZWN0cyBhbGwgc3VibmV0cyB0aGF0XG4gICAqIHNoYXJlIHRoZSBnaXZlbiBzdWJuZXQgZ3JvdXAgbmFtZS4gVGhpcyBpcyB0aGUgbmFtZSBzdXBwbGllZCBpblxuICAgKiBgc3VibmV0Q29uZmlndXJhdGlvbmAuXG4gICAqXG4gICAqIEF0IG1vc3Qgb25lIG9mIGBzdWJuZXRUeXBlYCBhbmQgYHN1Ym5ldEdyb3VwTmFtZWAgY2FuIGJlIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFNlbGVjdGlvbiBieSB0eXBlIGluc3RlYWQgb2YgYnkgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0R3JvdXBOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbGlhcyBmb3IgYHN1Ym5ldEdyb3VwTmFtZWBcbiAgICpcbiAgICogU2VsZWN0IHRoZSBzdWJuZXQgZ3JvdXAgd2l0aCB0aGUgZ2l2ZW4gbmFtZS4gVGhpcyBvbmx5IG5lZWRzXG4gICAqIHRvIGJlIHVzZWQgaWYgeW91IGhhdmUgbXVsdGlwbGUgc3VibmV0IGdyb3VwcyBvZiB0aGUgc2FtZSB0eXBlXG4gICAqIGFuZCB5b3UgbmVlZCB0byBkaXN0aW5ndWlzaCBiZXR3ZWVuIHRoZW0uXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIFVzZSBgc3VibmV0R3JvdXBOYW1lYCBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSBzdWJuZXROYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJZiB0cnVlLCByZXR1cm4gYXQgbW9zdCBvbmUgc3VibmV0IHBlciBBWlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgb25lUGVyQXo/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFeHBsaWNpdGx5IHNlbGVjdCBpbmRpdmlkdWFsIHN1Ym5ldHNcbiAgICpcbiAgICogVXNlIHRoaXMgaWYgeW91IGRvbid0IHdhbnQgdG8gYXV0b21hdGljYWxseSB1c2UgYWxsIHN1Ym5ldHMgaW5cbiAgICogYSBncm91cCwgYnV0IGhhdmUgYSBuZWVkIHRvIGNvbnRyb2wgc2VsZWN0aW9uIGRvd24gdG9cbiAgICogaW5kaXZpZHVhbCBzdWJuZXRzLlxuICAgKlxuICAgKiBDYW5ub3QgYmUgc3BlY2lmaWVkIHRvZ2V0aGVyIHdpdGggYHN1Ym5ldFR5cGVgIG9yIGBzdWJuZXRHcm91cE5hbWVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFVzZSBhbGwgc3VibmV0cyBpbiBhIHNlbGVjdGVkIGdyb3VwIChhbGwgcHJpdmF0ZSBzdWJuZXRzIGJ5IGRlZmF1bHQpXG4gICAqL1xuICByZWFkb25seSBzdWJuZXRzPzogSVN1Ym5ldFtdXG59XG5cbi8qKlxuICogUmVzdWx0IG9mIHNlbGVjdGluZyBhIHN1YnNldCBvZiBzdWJuZXRzIGZyb20gYSBWUENcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZWxlY3RlZFN1Ym5ldHMge1xuICAvKipcbiAgICogVGhlIHN1Ym5ldCBJRHNcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldElkczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSByZXNwZWN0aXZlIEFacyBvZiBlYWNoIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZXM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBEZXBlbmRlbmN5IHJlcHJlc2VudGluZyBpbnRlcm5ldCBjb25uZWN0aXZpdHkgZm9yIHRoZXNlIHN1Ym5ldHNcbiAgICovXG4gIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlO1xuXG4gIC8qKlxuICAgKiBTZWxlY3RlZCBzdWJuZXQgb2JqZWN0c1xuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0czogSVN1Ym5ldFtdO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIGFueSBvZiB0aGUgZ2l2ZW4gc3VibmV0cyBhcmUgZnJvbSB0aGUgVlBDJ3MgcHVibGljIHN1Ym5ldHMuXG4gICAqL1xuICByZWFkb25seSBoYXNQdWJsaWM6IGJvb2xlYW47XG59XG5cbi8qKlxuICogQSBuZXcgb3IgaW1wb3J0ZWQgVlBDXG4gKi9cbmFic3RyYWN0IGNsYXNzIFZwY0Jhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElWcGMge1xuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVyIGZvciB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHZwY0lkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwdWJsaWNTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHJpdmF0ZSBzdWJuZXRzIGluIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldHM6IElTdWJuZXRbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBpc29sYXRlZCBzdWJuZXRzIGluIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIEFacyBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXIgZm9yIHRoZSBWUE4gZ2F0ZXdheVxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHZwbkdhdGV3YXlJZD86IHN0cmluZztcblxuICAvKipcbiAgICogRGVwZW5kZW5jaWVzIGZvciBpbnRlcm5ldCBjb25uZWN0aXZpdHlcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkOiBJRGVwZW5kYWJsZTtcblxuICAvKipcbiAgICogRGVwZW5kZW5jaWVzIGZvciBOQVQgY29ubmVjdGl2aXR5XG4gICAqXG4gICAqIEBkZXByZWNhdGVkIC0gVGhpcyB2YWx1ZSBpcyBubyBsb25nZXIgdXNlZC5cbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBuYXREZXBlbmRlbmNpZXMgPSBuZXcgQXJyYXk8SUNvbnN0cnVjdD4oKTtcblxuICAvKipcbiAgICogSWYgdGhpcyBpcyBzZXQgdG8gdHJ1ZSwgZG9uJ3QgZXJyb3Igb3V0IG9uIHRyeWluZyB0byBzZWxlY3Qgc3VibmV0c1xuICAgKi9cbiAgcHJvdGVjdGVkIGluY29tcGxldGVTdWJuZXREZWZpbml0aW9uOiBib29sZWFuID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgSURzIG9mIHNlbGVjdGVkIHN1Ym5ldHNcbiAgICovXG4gIHB1YmxpYyBzZWxlY3RTdWJuZXRzKHNlbGVjdGlvbjogU3VibmV0U2VsZWN0aW9uID0ge30pOiBTZWxlY3RlZFN1Ym5ldHMge1xuICAgIGNvbnN0IHN1Ym5ldHMgPSB0aGlzLnNlbGVjdFN1Ym5ldE9iamVjdHMoc2VsZWN0aW9uKTtcbiAgICBjb25zdCBwdWJzID0gbmV3IFNldCh0aGlzLnB1YmxpY1N1Ym5ldHMpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHN1Ym5ldElkczogc3VibmV0cy5tYXAocyA9PiBzLnN1Ym5ldElkKSxcbiAgICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBzdWJuZXRzLm1hcChzID0+IHMuYXZhaWxhYmlsaXR5Wm9uZSksXG4gICAgICBpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkOiB0YXAobmV3IENvbXBvc2l0ZURlcGVuZGFibGUoKSwgZCA9PiBzdWJuZXRzLmZvckVhY2gocyA9PiBkLmFkZChzLmludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQpKSksXG4gICAgICBzdWJuZXRzLFxuICAgICAgaGFzUHVibGljOiBzdWJuZXRzLnNvbWUocyA9PiBwdWJzLmhhcyhzKSlcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgVlBOIGNvbm5lY3Rpb24gdG8gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhZGRWcG5Db25uZWN0aW9uKGlkOiBzdHJpbmcsIG9wdGlvbnM6IFZwbkNvbm5lY3Rpb25PcHRpb25zKTogVnBuQ29ubmVjdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWcG5Db25uZWN0aW9uKHRoaXMsIGlkLCB7XG4gICAgICB2cGM6IHRoaXMsXG4gICAgICAuLi5vcHRpb25zXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG5ldyBpbnRlcmZhY2UgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhZGRJbnRlcmZhY2VFbmRwb2ludChpZDogc3RyaW5nLCBvcHRpb25zOiBJbnRlcmZhY2VWcGNFbmRwb2ludE9wdGlvbnMpOiBJbnRlcmZhY2VWcGNFbmRwb2ludCB7XG4gICAgcmV0dXJuIG5ldyBJbnRlcmZhY2VWcGNFbmRwb2ludCh0aGlzLCBpZCwge1xuICAgICAgdnBjOiB0aGlzLFxuICAgICAgLi4ub3B0aW9uc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgZ2F0ZXdheSBlbmRwb2ludCB0byB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIGFkZEdhdGV3YXlFbmRwb2ludChpZDogc3RyaW5nLCBvcHRpb25zOiBHYXRld2F5VnBjRW5kcG9pbnRPcHRpb25zKTogR2F0ZXdheVZwY0VuZHBvaW50IHtcbiAgICByZXR1cm4gbmV3IEdhdGV3YXlWcGNFbmRwb2ludCh0aGlzLCBpZCwge1xuICAgICAgdnBjOiB0aGlzLFxuICAgICAgLi4ub3B0aW9uc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgc3VibmV0cyBhcHByb3ByaWF0ZSBmb3IgdGhlIHBsYWNlbWVudCBzdHJhdGVneVxuICAgKi9cbiAgcHJvdGVjdGVkIHNlbGVjdFN1Ym5ldE9iamVjdHMoc2VsZWN0aW9uOiBTdWJuZXRTZWxlY3Rpb24gPSB7fSk6IElTdWJuZXRbXSB7XG4gICAgc2VsZWN0aW9uID0gdGhpcy5yZWlmeVNlbGVjdGlvbkRlZmF1bHRzKHNlbGVjdGlvbik7XG5cbiAgICBpZiAoc2VsZWN0aW9uLnN1Ym5ldHMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHNlbGVjdGlvbi5zdWJuZXRzO1xuXG4gICAgfSBlbHNlIGlmIChzZWxlY3Rpb24uc3VibmV0R3JvdXBOYW1lICE9PSB1bmRlZmluZWQpIHsgLy8gU2VsZWN0IGJ5IG5hbWVcbiAgICAgIHJldHVybiB0aGlzLnNlbGVjdFN1Ym5ldE9iamVjdHNCeU5hbWUoc2VsZWN0aW9uLnN1Ym5ldEdyb3VwTmFtZSk7XG5cbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgdHlwZSA9IHNlbGVjdGlvbi5zdWJuZXRUeXBlIHx8IFN1Ym5ldFR5cGUuUFJJVkFURTtcbiAgICAgIHJldHVybiB0aGlzLnNlbGVjdFN1Ym5ldE9iamVjdHNCeVR5cGUodHlwZSwgISFzZWxlY3Rpb24ub25lUGVyQXopO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc2VsZWN0U3VibmV0T2JqZWN0c0J5TmFtZShncm91cE5hbWU6IHN0cmluZykge1xuICAgIGNvbnN0IGFsbFN1Ym5ldHMgPSAgWy4uLnRoaXMucHVibGljU3VibmV0cywgLi4udGhpcy5wcml2YXRlU3VibmV0cywgLi4udGhpcy5pc29sYXRlZFN1Ym5ldHNdO1xuICAgIGNvbnN0IHN1Ym5ldHMgPSBhbGxTdWJuZXRzLmZpbHRlcihzID0+IHN1Ym5ldEdyb3VwTmFtZUZyb21Db25zdHJ1Y3RJZChzKSA9PT0gZ3JvdXBOYW1lKTtcblxuICAgIGlmIChzdWJuZXRzLmxlbmd0aCA9PT0gMCAmJiAhdGhpcy5pbmNvbXBsZXRlU3VibmV0RGVmaW5pdGlvbikge1xuICAgICAgY29uc3QgbmFtZXMgPSBBcnJheS5mcm9tKG5ldyBTZXQoYWxsU3VibmV0cy5tYXAoc3VibmV0R3JvdXBOYW1lRnJvbUNvbnN0cnVjdElkKSkpO1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGVyZSBhcmUgbm8gc3VibmV0IGdyb3VwcyB3aXRoIG5hbWUgJyR7Z3JvdXBOYW1lfScgaW4gdGhpcyBWUEMuIEF2YWlsYWJsZSBuYW1lczogJHtuYW1lc31gKTtcbiAgICB9XG5cbiAgICByZXR1cm4gc3VibmV0cztcbiAgfVxuXG4gIHByaXZhdGUgc2VsZWN0U3VibmV0T2JqZWN0c0J5VHlwZShzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLCBvbmVQZXJBejogYm9vbGVhbikge1xuICAgIGNvbnN0IGFsbFN1Ym5ldHMgPSB7XG4gICAgICBbU3VibmV0VHlwZS5JU09MQVRFRF06IHRoaXMuaXNvbGF0ZWRTdWJuZXRzLFxuICAgICAgW1N1Ym5ldFR5cGUuUFJJVkFURV06IHRoaXMucHJpdmF0ZVN1Ym5ldHMsXG4gICAgICBbU3VibmV0VHlwZS5QVUJMSUNdOiB0aGlzLnB1YmxpY1N1Ym5ldHMsXG4gICAgfTtcblxuICAgIGxldCBzdWJuZXRzID0gYWxsU3VibmV0c1tzdWJuZXRUeXBlXTtcblxuICAgIGlmIChvbmVQZXJBeiAmJiBzdWJuZXRzLmxlbmd0aCA+IDApIHtcbiAgICAgIC8vIFJlc3RyaWN0IHRvIGF0IG1vc3Qgb25lIHN1Ym5ldCBncm91cFxuICAgICAgc3VibmV0cyA9IHN1Ym5ldHMuZmlsdGVyKHMgPT4gc3VibmV0R3JvdXBOYW1lRnJvbUNvbnN0cnVjdElkKHMpID09PSBzdWJuZXRHcm91cE5hbWVGcm9tQ29uc3RydWN0SWQoc3VibmV0c1swXSkpO1xuICAgIH1cblxuICAgIC8vIEZvcmNlIG1lcmdlIGNvbmZsaWN0IGhlcmUgd2l0aCBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvcHVsbC80MDg5XG4gICAgLy8gc2VlIEltcG9ydGVkVnBjXG5cbiAgICBpZiAoc3VibmV0cy5sZW5ndGggPT09IDAgJiYgIXRoaXMuaW5jb21wbGV0ZVN1Ym5ldERlZmluaXRpb24pIHtcbiAgICAgIGNvbnN0IGF2YWlsYWJsZVR5cGVzID0gT2JqZWN0LmVudHJpZXMoYWxsU3VibmV0cykuZmlsdGVyKChbXywgc3Vic10pID0+IHN1YnMubGVuZ3RoID4gMCkubWFwKChbdHlwZU5hbWUsIF9dKSA9PiB0eXBlTmFtZSk7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZXJlIGFyZSBubyAnJHtzdWJuZXRUeXBlfScgc3VibmV0IGdyb3VwcyBpbiB0aGlzIFZQQy4gQXZhaWxhYmxlIHR5cGVzOiAke2F2YWlsYWJsZVR5cGVzfWApO1xuICAgIH1cblxuICAgIHJldHVybiBzdWJuZXRzO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoZSBmaWVsZHMgaW4gYSBTdWJuZXRTZWxlY3Rpb24gb2JqZWN0LCBhbmQgcmVpZnkgZGVmYXVsdHMgaWYgbmVjZXNzYXJ5XG4gICAqXG4gICAqIEluIGNhc2Ugb2YgZGVmYXVsdCBzZWxlY3Rpb24sIHNlbGVjdCB0aGUgZmlyc3QgdHlwZSBvZiBQUklWQVRFLCBJU09MQVRFRCxcbiAgICogUFVCTElDIChpbiB0aGF0IG9yZGVyKSB0aGF0IGhhcyBhbnkgc3VibmV0cy5cbiAgICovXG4gIHByaXZhdGUgcmVpZnlTZWxlY3Rpb25EZWZhdWx0cyhwbGFjZW1lbnQ6IFN1Ym5ldFNlbGVjdGlvbik6IFN1Ym5ldFNlbGVjdGlvbiB7XG4gICAgaWYgKHBsYWNlbWVudC5zdWJuZXROYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGlmIChwbGFjZW1lbnQuc3VibmV0R3JvdXBOYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBQbGVhc2UgdXNlIG9ubHkgJ3N1Ym5ldEdyb3VwTmFtZScgKCdzdWJuZXROYW1lJyBpcyBkZXByZWNhdGVkIGFuZCBoYXMgdGhlIHNhbWUgYmVoYXZpb3IpYCk7XG4gICAgICB9XG4gICAgICBwbGFjZW1lbnQgPSB7Li4ucGxhY2VtZW50LCBzdWJuZXRHcm91cE5hbWU6IHBsYWNlbWVudC5zdWJuZXROYW1lIH07XG4gICAgfVxuXG4gICAgY29uc3QgZXhjbHVzaXZlU2VsZWN0aW9uczogQXJyYXk8a2V5b2YgU3VibmV0U2VsZWN0aW9uPiA9IFsnc3VibmV0cycsICdzdWJuZXRUeXBlJywgJ3N1Ym5ldEdyb3VwTmFtZSddO1xuICAgIGNvbnN0IHByb3ZpZGVkU2VsZWN0aW9ucyA9IGV4Y2x1c2l2ZVNlbGVjdGlvbnMuZmlsdGVyKGtleSA9PiBwbGFjZW1lbnRba2V5XSAhPT0gdW5kZWZpbmVkKTtcbiAgICBpZiAocHJvdmlkZWRTZWxlY3Rpb25zLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgT25seSBvbmUgb2YgJyR7cHJvdmlkZWRTZWxlY3Rpb25zfScgY2FuIGJlIHN1cHBsaWVkIHRvIHN1Ym5ldCBzZWxlY3Rpb24uYCk7XG4gICAgfVxuXG4gICAgaWYgKHBsYWNlbWVudC5zdWJuZXRUeXBlID09PSB1bmRlZmluZWQgJiYgcGxhY2VtZW50LnN1Ym5ldEdyb3VwTmFtZSA9PT0gdW5kZWZpbmVkICYmIHBsYWNlbWVudC5zdWJuZXRzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIFJldHVybiBkZWZhdWx0IHN1Ym5ldCB0eXBlIGJhc2VkIG9uIHN1Ym5ldHMgdGhhdCBhY3R1YWxseSBleGlzdFxuICAgICAgaWYgKHRoaXMucHJpdmF0ZVN1Ym5ldHMubGVuZ3RoID4gMCkgeyByZXR1cm4geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsIG9uZVBlckF6OiBwbGFjZW1lbnQub25lUGVyQXogfTsgfVxuICAgICAgaWYgKHRoaXMuaXNvbGF0ZWRTdWJuZXRzLmxlbmd0aCA+IDApIHsgcmV0dXJuIHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5JU09MQVRFRCwgb25lUGVyQXo6IHBsYWNlbWVudC5vbmVQZXJBeiB9OyB9XG4gICAgICByZXR1cm4geyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQywgb25lUGVyQXo6IHBsYWNlbWVudC5vbmVQZXJBeiB9O1xuICAgIH1cblxuICAgIHJldHVybiBwbGFjZW1lbnQ7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRoYXQgcmVmZXJlbmNlIGFuIGV4dGVybmFsIFZwY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFZwY0F0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVlBDJ3MgaWRlbnRpZmllclxuICAgKi9cbiAgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogTGlzdCBvZiBhdmFpbGFiaWxpdHkgem9uZXMgZm9yIHRoZSBzdWJuZXRzIGluIHRoaXMgVlBDLlxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZXM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIHB1YmxpYyBzdWJuZXQgSURzXG4gICAqXG4gICAqIE11c3QgYmUgdW5kZWZpbmVkIG9yIG1hdGNoIHRoZSBhdmFpbGFiaWxpdHkgem9uZXMgaW4gbGVuZ3RoIGFuZCBvcmRlci5cbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldElkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIG5hbWVzIGZvciB0aGUgcHVibGljIHN1Ym5ldHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgaGF2ZSBhIG5hbWUgZm9yIGV2ZXJ5IHB1YmxpYyBzdWJuZXQgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBwdWJsaWNTdWJuZXROYW1lcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIElEcyBvZiByb3V0aW5nIHRhYmxlcyBmb3IgdGhlIHB1YmxpYyBzdWJuZXRzLlxuICAgKlxuICAgKiBNdXN0IGJlIHVuZGVmaW5lZCBvciBoYXZlIGEgbmFtZSBmb3IgZXZlcnkgcHVibGljIHN1Ym5ldCBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldFJvdXRlVGFibGVJZHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldCBJRHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgbWF0Y2ggdGhlIGF2YWlsYWJpbGl0eSB6b25lcyBpbiBsZW5ndGggYW5kIG9yZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldElkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIG5hbWVzIGZvciB0aGUgcHJpdmF0ZSBzdWJuZXRzXG4gICAqXG4gICAqIE11c3QgYmUgdW5kZWZpbmVkIG9yIGhhdmUgYSBuYW1lIGZvciBldmVyeSBwcml2YXRlIHN1Ym5ldCBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHByaXZhdGVTdWJuZXROYW1lcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIElEcyBvZiByb3V0aW5nIHRhYmxlcyBmb3IgdGhlIHByaXZhdGUgc3VibmV0cy5cbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgaGF2ZSBhIG5hbWUgZm9yIGV2ZXJ5IHByaXZhdGUgc3VibmV0IGdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldFJvdXRlVGFibGVJZHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBpc29sYXRlZCBzdWJuZXQgSURzXG4gICAqXG4gICAqIE11c3QgYmUgdW5kZWZpbmVkIG9yIG1hdGNoIHRoZSBhdmFpbGFiaWxpdHkgem9uZXMgaW4gbGVuZ3RoIGFuZCBvcmRlci5cbiAgICovXG4gIHJlYWRvbmx5IGlzb2xhdGVkU3VibmV0SWRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgbmFtZXMgZm9yIHRoZSBpc29sYXRlZCBzdWJuZXRzXG4gICAqXG4gICAqIE11c3QgYmUgdW5kZWZpbmVkIG9yIGhhdmUgYSBuYW1lIGZvciBldmVyeSBpc29sYXRlZCBzdWJuZXQgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBpc29sYXRlZFN1Ym5ldE5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgSURzIG9mIHJvdXRpbmcgdGFibGVzIGZvciB0aGUgaXNvbGF0ZWQgc3VibmV0cy5cbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgaGF2ZSBhIG5hbWUgZm9yIGV2ZXJ5IGlzb2xhdGVkIHN1Ym5ldCBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IGlzb2xhdGVkU3VibmV0Um91dGVUYWJsZUlkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBWUE4gZ2F0ZXdheSdzIGlkZW50aWZpZXJcbiAgICovXG4gIHJlYWRvbmx5IHZwbkdhdGV3YXlJZD86IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBTdWJuZXRBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBBdmFpbGFiaWxpdHkgWm9uZSB0aGUgc3VibmV0IGlzIGxvY2F0ZWQgaW5cbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldElkIGZvciB0aGlzIHBhcnRpY3VsYXIgc3VibmV0XG4gICAqL1xuICByZWFkb25seSBzdWJuZXRJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIHJvdXRlIHRhYmxlIGZvciB0aGlzIHBhcnRpY3VsYXIgc3VibmV0XG4gICAqL1xuICByZWFkb25seSByb3V0ZVRhYmxlSWQ/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogTmFtZSB0YWcgY29uc3RhbnRcbiAqL1xuY29uc3QgTkFNRV9UQUc6IHN0cmluZyA9ICdOYW1lJztcblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBWcGNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWcGNQcm9wcyB7XG5cbiAgLyoqXG4gICAqIFRoZSBDSURSIHJhbmdlIHRvIHVzZSBmb3IgdGhlIFZQQywgZS5nLiAnMTAuMC4wLjAvMTYnLlxuICAgKlxuICAgKiBTaG91bGQgYmUgYSBtaW5pbXVtIG9mIC8yOCBhbmQgbWF4aW11bSBzaXplIG9mIC8xNi4gVGhlIHJhbmdlIHdpbGwgYmVcbiAgICogc3BsaXQgYWNyb3NzIGFsbCBzdWJuZXRzIHBlciBBdmFpbGFiaWxpdHkgWm9uZS5cbiAgICpcbiAgICogQGRlZmF1bHQgVnBjLkRFRkFVTFRfQ0lEUl9SQU5HRVxuICAgKi9cbiAgcmVhZG9ubHkgY2lkcj86IHN0cmluZztcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgVlBDIGdldCBwdWJsaWMgRE5TIGhvc3RuYW1lcy5cbiAgICpcbiAgICogSWYgdGhpcyBhdHRyaWJ1dGUgaXMgdHJ1ZSwgaW5zdGFuY2VzIGluIHRoZSBWUEMgZ2V0IHB1YmxpYyBETlMgaG9zdG5hbWVzLFxuICAgKiBidXQgb25seSBpZiB0aGUgZW5hYmxlRG5zU3VwcG9ydCBhdHRyaWJ1dGUgaXMgYWxzbyBzZXQgdG8gdHJ1ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlRG5zSG9zdG5hbWVzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIEROUyByZXNvbHV0aW9uIGlzIHN1cHBvcnRlZCBmb3IgdGhlIFZQQy5cbiAgICpcbiAgICogSWYgdGhpcyBhdHRyaWJ1dGUgaXMgZmFsc2UsIHRoZSBBbWF6b24tcHJvdmlkZWQgRE5TIHNlcnZlciBpbiB0aGUgVlBDIHRoYXRcbiAgICogcmVzb2x2ZXMgcHVibGljIEROUyBob3N0bmFtZXMgdG8gSVAgYWRkcmVzc2VzIGlzIG5vdCBlbmFibGVkLiBJZiB0aGlzXG4gICAqIGF0dHJpYnV0ZSBpcyB0cnVlLCBxdWVyaWVzIHRvIHRoZSBBbWF6b24gcHJvdmlkZWQgRE5TIHNlcnZlciBhdCB0aGVcbiAgICogMTY5LjI1NC4xNjkuMjUzIElQIGFkZHJlc3MsIG9yIHRoZSByZXNlcnZlZCBJUCBhZGRyZXNzIGF0IHRoZSBiYXNlIG9mIHRoZVxuICAgKiBWUEMgSVB2NCBuZXR3b3JrIHJhbmdlIHBsdXMgdHdvIHdpbGwgc3VjY2VlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlRG5zU3VwcG9ydD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHRlbmFuY3kgb2YgaW5zdGFuY2VzIGxhdW5jaGVkIGludG8gdGhlIFZQQy5cbiAgICpcbiAgICogQnkgc2V0dGluZyB0aGlzIHRvIGRlZGljYXRlZCB0ZW5hbmN5LCBpbnN0YW5jZXMgd2lsbCBiZSBsYXVuY2hlZCBvblxuICAgKiBoYXJkd2FyZSBkZWRpY2F0ZWQgdG8gYSBzaW5nbGUgQVdTIGN1c3RvbWVyLCB1bmxlc3Mgc3BlY2lmaWNhbGx5IHNwZWNpZmllZFxuICAgKiBhdCBpbnN0YW5jZSBsYXVuY2ggdGltZS4gUGxlYXNlIG5vdGUsIG5vdCBhbGwgaW5zdGFuY2UgdHlwZXMgYXJlIHVzYWJsZVxuICAgKiB3aXRoIERlZGljYXRlZCB0ZW5hbmN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBEZWZhdWx0SW5zdGFuY2VUZW5hbmN5LkRlZmF1bHQgKHNoYXJlZCkgdGVuYW5jeVxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdEluc3RhbmNlVGVuYW5jeT86IERlZmF1bHRJbnN0YW5jZVRlbmFuY3k7XG5cbiAgLyoqXG4gICAqIERlZmluZSB0aGUgbWF4aW11bSBudW1iZXIgb2YgQVpzIHRvIHVzZSBpbiB0aGlzIHJlZ2lvblxuICAgKlxuICAgKiBJZiB0aGUgcmVnaW9uIGhhcyBtb3JlIEFacyB0aGFuIHlvdSB3YW50IHRvIHVzZSAoZm9yIGV4YW1wbGUsIGJlY2F1c2Ugb2ZcbiAgICogRUlQIGxpbWl0cyksIHBpY2sgYSBsb3dlciBudW1iZXIgaGVyZS4gVGhlIEFacyB3aWxsIGJlIHNvcnRlZCBhbmQgcGlja2VkXG4gICAqIGZyb20gdGhlIHN0YXJ0IG9mIHRoZSBsaXN0LlxuICAgKlxuICAgKiBJZiB5b3UgcGljayBhIGhpZ2hlciBudW1iZXIgdGhhbiB0aGUgbnVtYmVyIG9mIEFacyBpbiB0aGUgcmVnaW9uLCBhbGwgQVpzXG4gICAqIGluIHRoZSByZWdpb24gd2lsbCBiZSBzZWxlY3RlZC4gVG8gdXNlIFwiYWxsIEFac1wiIGF2YWlsYWJsZSB0byB5b3VyXG4gICAqIGFjY291bnQsIHVzZSBhIGhpZ2ggbnVtYmVyIChzdWNoIGFzIDk5KS5cbiAgICpcbiAgICogQmUgYXdhcmUgdGhhdCBlbnZpcm9ubWVudC1hZ25vc3RpYyBzdGFja3Mgd2lsbCBiZSBjcmVhdGVkIHdpdGggYWNjZXNzIHRvXG4gICAqIG9ubHkgMiBBWnMsIHNvIHRvIHVzZSBtb3JlIHRoYW4gMiBBWnMsIGJlIHN1cmUgdG8gc3BlY2lmeSB0aGUgYWNjb3VudCBhbmRcbiAgICogcmVnaW9uIG9uIHlvdXIgc3RhY2suXG4gICAqXG4gICAqIEBkZWZhdWx0IDNcbiAgICovXG4gIHJlYWRvbmx5IG1heEF6cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBOQVQgR2F0ZXdheXMvSW5zdGFuY2VzIHRvIGNyZWF0ZS5cbiAgICpcbiAgICogVGhlIHR5cGUgb2YgTkFUIGdhdGV3YXkgb3IgaW5zdGFuY2Ugd2lsbCBiZSBkZXRlcm1pbmVkIGJ5IHRoZVxuICAgKiBgbmF0R2F0ZXdheVByb3ZpZGVyYCBwYXJhbWV0ZXIuXG4gICAqXG4gICAqIFlvdSBjYW4gc2V0IHRoaXMgbnVtYmVyIGxvd2VyIHRoYW4gdGhlIG51bWJlciBvZiBBdmFpbGFiaWxpdHkgWm9uZXMgaW4geW91clxuICAgKiBWUEMgaW4gb3JkZXIgdG8gc2F2ZSBvbiBOQVQgY29zdC4gQmUgYXdhcmUgeW91IG1heSBiZSBjaGFyZ2VkIGZvclxuICAgKiBjcm9zcy1BWiBkYXRhIHRyYWZmaWMgaW5zdGVhZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBPbmUgTkFUIGdhdGV3YXkvaW5zdGFuY2UgcGVyIEF2YWlsYWJpbGl0eSBab25lXG4gICAqL1xuICByZWFkb25seSBuYXRHYXRld2F5cz86IG51bWJlcjtcblxuICAvKipcbiAgICogQ29uZmlndXJlcyB0aGUgc3VibmV0cyB3aGljaCB3aWxsIGhhdmUgTkFUIEdhdGV3YXlzL0luc3RhbmNlc1xuICAgKlxuICAgKiBZb3UgY2FuIHBpY2sgYSBzcGVjaWZpYyBncm91cCBvZiBzdWJuZXRzIGJ5IHNwZWNpZnlpbmcgdGhlIGdyb3VwIG5hbWU7XG4gICAqIHRoZSBwaWNrZWQgc3VibmV0cyBtdXN0IGJlIHB1YmxpYyBzdWJuZXRzLlxuICAgKlxuICAgKiBPbmx5IG5lY2Vzc2FyeSBpZiB5b3UgaGF2ZSBtb3JlIHRoYW4gb25lIHB1YmxpYyBzdWJuZXQgZ3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQWxsIHB1YmxpYyBzdWJuZXRzLlxuICAgKi9cbiAgcmVhZG9ubHkgbmF0R2F0ZXdheVN1Ym5ldHM/OiBTdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIFdoYXQgdHlwZSBvZiBOQVQgcHJvdmlkZXIgdG8gdXNlXG4gICAqXG4gICAqIFNlbGVjdCBiZXR3ZWVuIE5BVCBnYXRld2F5cyBvciBOQVQgaW5zdGFuY2VzLiBOQVQgZ2F0ZXdheXNcbiAgICogbWF5IG5vdCBiZSBhdmFpbGFibGUgaW4gYWxsIEFXUyByZWdpb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCBOYXRQcm92aWRlci5nYXRld2F5KClcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgcmVhZG9ubHkgbmF0R2F0ZXdheVByb3ZpZGVyPzogTmF0UHJvdmlkZXI7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZSB0aGUgc3VibmV0cyB0byBidWlsZCBmb3IgZWFjaCBBWlxuICAgKlxuICAgKiBFYWNoIGVudHJ5IGluIHRoaXMgbGlzdCBjb25maWd1cmVzIGEgU3VibmV0IEdyb3VwOyBlYWNoIGdyb3VwIHdpbGwgY29udGFpbiBhXG4gICAqIHN1Ym5ldCBmb3IgZWFjaCBBdmFpbGFiaWxpdHkgWm9uZS5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIGlmIHlvdSB3YW50IDEgcHVibGljIHN1Ym5ldCwgMSBwcml2YXRlIHN1Ym5ldCwgYW5kIDEgaXNvbGF0ZWRcbiAgICogc3VibmV0IGluIGVhY2ggQVogcHJvdmlkZSB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiBgYGB0c1xuICAgKiBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAqICAgIHtcbiAgICogICAgICBjaWRyTWFzazogMjQsXG4gICAqICAgICAgbmFtZTogJ2luZ3Jlc3MnLFxuICAgKiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDLFxuICAgKiAgICB9LFxuICAgKiAgICB7XG4gICAqICAgICAgY2lkck1hc2s6IDI0LFxuICAgKiAgICAgIG5hbWU6ICdhcHBsaWNhdGlvbicsXG4gICAqICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgKiAgICB9LFxuICAgKiAgICB7XG4gICAqICAgICAgY2lkck1hc2s6IDI4LFxuICAgKiAgICAgIG5hbWU6ICdyZHMnLFxuICAgKiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAqICAgIH1cbiAgICogXVxuICAgKiBgYGBcbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgVlBDIENJRFIgd2lsbCBiZSBldmVubHkgZGl2aWRlZCBiZXR3ZWVuIDEgcHVibGljIGFuZCAxXG4gICAqIHByaXZhdGUgc3VibmV0IHBlciBBWi5cbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldENvbmZpZ3VyYXRpb24/OiBTdWJuZXRDb25maWd1cmF0aW9uW107XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIGEgVlBOIGdhdGV3YXkgc2hvdWxkIGJlIGNyZWF0ZWQgYW5kIGF0dGFjaGVkIHRvIHRoaXMgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgd2hlbiB2cG5HYXRld2F5QXNuIG9yIHZwbkNvbm5lY3Rpb25zIGlzIHNwZWNpZmllZC5cbiAgICovXG4gIHJlYWRvbmx5IHZwbkdhdGV3YXk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJpdmF0ZSBBdXRvbm9tb3VzIFN5c3RlbSBOdW1iZXIgKEFTTikgZm9yIHRoZSBWUE4gZ2F0ZXdheS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbWF6b24gZGVmYXVsdCBBU04uXG4gICAqL1xuICByZWFkb25seSB2cG5HYXRld2F5QXNuPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBWUE4gY29ubmVjdGlvbnMgdG8gdGhpcyBWUEMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY29ubmVjdGlvbnMuXG4gICAqL1xuICByZWFkb25seSB2cG5Db25uZWN0aW9ucz86IHsgW2lkOiBzdHJpbmddOiBWcG5Db25uZWN0aW9uT3B0aW9ucyB9XG5cbiAgLyoqXG4gICAqIFdoZXJlIHRvIHByb3BhZ2F0ZSBWUE4gcm91dGVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE9uIHRoZSByb3V0ZSB0YWJsZXMgYXNzb2NpYXRlZCB3aXRoIHByaXZhdGUgc3VibmV0cy5cbiAgICovXG4gIHJlYWRvbmx5IHZwblJvdXRlUHJvcGFnYXRpb24/OiBTdWJuZXRTZWxlY3Rpb25bXVxuXG4gIC8qKlxuICAgKiBHYXRld2F5IGVuZHBvaW50cyB0byBhZGQgdG8gdGhpcyBWUEMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IGdhdGV3YXlFbmRwb2ludHM/OiB7IFtpZDogc3RyaW5nXTogR2F0ZXdheVZwY0VuZHBvaW50T3B0aW9ucyB9XG59XG5cbi8qKlxuICogVGhlIGRlZmF1bHQgdGVuYW5jeSBvZiBpbnN0YW5jZXMgbGF1bmNoZWQgaW50byB0aGUgVlBDLlxuICovXG5leHBvcnQgZW51bSBEZWZhdWx0SW5zdGFuY2VUZW5hbmN5IHtcbiAgLyoqXG4gICAqIEluc3RhbmNlcyBjYW4gYmUgbGF1bmNoZWQgd2l0aCBhbnkgdGVuYW5jeS5cbiAgICovXG4gIERFRkFVTFQgPSAnZGVmYXVsdCcsXG5cbiAgLyoqXG4gICAqIEFueSBpbnN0YW5jZSBsYXVuY2hlZCBpbnRvIHRoZSBWUEMgYXV0b21hdGljYWxseSBoYXMgZGVkaWNhdGVkIHRlbmFuY3ksIHVubGVzcyB5b3UgbGF1bmNoIGl0IHdpdGggdGhlIGRlZmF1bHQgdGVuYW5jeS5cbiAgICovXG4gIERFRElDQVRFRCA9ICdkZWRpY2F0ZWQnXG59XG5cbi8qKlxuICogU3BlY2lmeSBjb25maWd1cmF0aW9uIHBhcmFtZXRlcnMgZm9yIGEgc2luZ2xlIHN1Ym5ldCBncm91cCBpbiBhIFZQQy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdWJuZXRDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgbGVhZGluZyAxIGJpdHMgaW4gdGhlIHJvdXRpbmcgbWFzay5cbiAgICpcbiAgICogVGhlIG51bWJlciBvZiBhdmFpbGFibGUgSVAgYWRkcmVzc2VzIGluIGVhY2ggc3VibmV0IG9mIHRoaXMgZ3JvdXBcbiAgICogd2lsbCBiZSBlcXVhbCB0byBgMl4oMzIgLSBjaWRyTWFzaykgLSAyYC5cbiAgICpcbiAgICogVmFsaWQgdmFsdWVzIGFyZSBgMTYtLTI4YC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBdmFpbGFibGUgSVAgc3BhY2UgaXMgZXZlbmx5IGRpdmlkZWQgYWNyb3NzIHN1Ym5ldHMuXG4gICAqL1xuICByZWFkb25seSBjaWRyTWFzaz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgU3VibmV0IHRvIGNvbmZpZ3VyZS5cbiAgICpcbiAgICogVGhlIFN1Ym5ldCB0eXBlIHdpbGwgY29udHJvbCB0aGUgYWJpbGl0eSB0byByb3V0ZSBhbmQgY29ubmVjdCB0byB0aGVcbiAgICogSW50ZXJuZXQuXG4gICAqL1xuICByZWFkb25seSBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlO1xuXG4gIC8qKlxuICAgKiBMb2dpY2FsIG5hbWUgZm9yIHRoZSBzdWJuZXQgZ3JvdXAuXG4gICAqXG4gICAqIFRoaXMgbmFtZSBjYW4gYmUgdXNlZCB3aGVuIHNlbGVjdGluZyBWUEMgc3VibmV0cyB0byBkaXN0aW5ndWlzaFxuICAgKiBiZXR3ZWVuIGRpZmZlcmVudCBzdWJuZXQgZ3JvdXBzIG9mIHRoZSBzYW1lIHR5cGUuXG4gICAqL1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAvKipcbiAgKiBDb250cm9scyBpZiBzdWJuZXQgSVAgc3BhY2UgbmVlZHMgdG8gYmUgcmVzZXJ2ZWQuXG4gICpcbiAgKiBXaGVuIHRydWUsIHRoZSBJUCBzcGFjZSBmb3IgdGhlIHN1Ym5ldCBpcyByZXNlcnZlZCBidXQgbm8gYWN0dWFsXG4gICogcmVzb3VyY2VzIGFyZSBwcm92aXNpb25lZC4gVGhpcyBzcGFjZSBpcyBvbmx5IGRlcGVuZGVudCBvbiB0aGVcbiAgKiBudW1iZXIgb2YgYXZhaWxpYmlsaXR5IHpvbmVzIGFuZCBvbiBgY2lkck1hc2tgIC0gYWxsIG90aGVyIHN1Ym5ldFxuICAqIHByb3BlcnRpZXMgYXJlIGlnbm9yZWQuXG4gICpcbiAgKiBAZGVmYXVsdCBmYWxzZVxuICAqL1xuICByZWFkb25seSByZXNlcnZlZD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogRGVmaW5lIGFuIEFXUyBWaXJ0dWFsIFByaXZhdGUgQ2xvdWRcbiAqXG4gKiBTZWUgdGhlIHBhY2thZ2UtbGV2ZWwgZG9jdW1lbnRhdGlvbiBvZiB0aGlzIHBhY2thZ2UgZm9yIGFuIG92ZXJ2aWV3XG4gKiBvZiB0aGUgdmFyaW91cyBkaW1lbnNpb25zIGluIHdoaWNoIHlvdSBjYW4gY29uZmlndXJlIHlvdXIgVlBDLlxuICpcbiAqIEZvciBleGFtcGxlOlxuICpcbiAqIGBgYHRzXG4gKiBpbXBvcnQgeyBTdWJuZXRUeXBlLCBWcGMgfSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJ1xuICpcbiAqIGNvbnN0IHZwYyA9IG5ldyBWcGModGhpcywgJ1RoZVZQQycsIHtcbiAqICAgY2lkcjogXCIxMC4wLjAuMC8xNlwiXG4gKiB9KVxuICpcbiAqIC8vIEl0ZXJhdGUgdGhlIHByaXZhdGUgc3VibmV0c1xuICogY29uc3Qgc2VsZWN0aW9uID0gdnBjLnNlbGVjdFN1Ym5ldHMoe1xuICogICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVcbiAqIH0pO1xuICpcbiAqIGZvciAoY29uc3Qgc3VibmV0IG9mIHNlbGVjdGlvbi5zdWJuZXRzKSB7XG4gKiAgIC8vIC4uLlxuICogfVxuICogYGBgXG4gKlxuICogQHJlc291cmNlIEFXUzo6RUMyOjpWUENcbiAqL1xuZXhwb3J0IGNsYXNzIFZwYyBleHRlbmRzIFZwY0Jhc2Uge1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgQ0lEUiByYW5nZSB1c2VkIHdoZW4gY3JlYXRpbmcgVlBDcy5cbiAgICogVGhpcyBjYW4gYmUgb3ZlcnJpZGRlbiB1c2luZyBWcGNQcm9wcyB3aGVuIGNyZWF0aW5nIGEgVlBDTmV0d29yayByZXNvdXJjZS5cbiAgICogZS5nLiBuZXcgVnBjUmVzb3VyY2UodGhpcywgeyBjaWRyOiAnMTkyLjE2OC4wLjAuLzE2JyB9KVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZBVUxUX0NJRFJfUkFOR0U6IHN0cmluZyA9ICcxMC4wLjAuMC8xNic7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHN1Ym5ldCBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIDEgUHVibGljIGFuZCAxIFByaXZhdGUgc3VibmV0IHBlciBBWiBldmVubHkgc3BsaXRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgREVGQVVMVF9TVUJORVRTOiBTdWJuZXRDb25maWd1cmF0aW9uW10gPSBbXG4gICAge1xuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICBuYW1lOiBkZWZhdWx0U3VibmV0TmFtZShTdWJuZXRUeXBlLlBVQkxJQyksXG4gICAgfSxcbiAgICB7XG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICBuYW1lOiBkZWZhdWx0U3VibmV0TmFtZShTdWJuZXRUeXBlLlBSSVZBVEUpLFxuICAgIH1cbiAgXTtcblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4cG9ydGVkIFZQQ1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVnBjQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogVnBjQXR0cmlidXRlcyk6IElWcGMge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRWcGMoc2NvcGUsIGlkLCBhdHRycywgZmFsc2UpO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBWUEMgZnJvbSBieSBxdWVyeWluZyB0aGUgQVdTIGVudmlyb25tZW50IHRoaXMgc3RhY2sgaXMgZGVwbG95ZWQgdG8uXG4gICAqXG4gICAqIFRoaXMgZnVuY3Rpb24gb25seSBuZWVkcyB0byBiZSB1c2VkIHRvIHVzZSBWUENzIG5vdCBkZWZpbmVkIGluIHlvdXIgQ0RLXG4gICAqIGFwcGxpY2F0aW9uLiBJZiB5b3UgYXJlIGxvb2tpbmcgdG8gc2hhcmUgYSBWUEMgYmV0d2VlbiBzdGFja3MsIHlvdSBjYW5cbiAgICogcGFzcyB0aGUgYFZwY2Agb2JqZWN0IGJldHdlZW4gc3RhY2tzIGFuZCB1c2UgaXQgYXMgbm9ybWFsLlxuICAgKlxuICAgKiBTZWUgdGhlIHBhY2thZ2UtbGV2ZWwgZG9jdW1lbnRhdGlvbiBvZiB0aGlzIHBhY2thZ2UgZm9yIGNvbnN0cmFpbnRzXG4gICAqIG9uIGltcG9ydGluZyBleGlzdGluZyBWUENzLlxuICAgKlxuICAgKiBDYWxsaW5nIHRoaXMgbWV0aG9kIHdpbGwgbGVhZCB0byBhIGxvb2t1cCB3aGVuIHRoZSBDREsgQ0xJIGlzIGV4ZWN1dGVkLlxuICAgKiBZb3UgY2FuIHRoZXJlZm9yZSBub3QgdXNlIGFueSB2YWx1ZXMgdGhhdCB3aWxsIG9ubHkgYmUgYXZhaWxhYmxlIGF0XG4gICAqIENsb3VkRm9ybWF0aW9uIGV4ZWN1dGlvbiB0aW1lIChpLmUuLCBUb2tlbnMpLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tTG9va3VwKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG9wdGlvbnM6IFZwY0xvb2t1cE9wdGlvbnMpOiBJVnBjIHtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKG9wdGlvbnMudnBjSWQpXG4gICAgICB8fCBUb2tlbi5pc1VucmVzb2x2ZWQob3B0aW9ucy52cGNOYW1lKVxuICAgICAgfHwgT2JqZWN0LnZhbHVlcyhvcHRpb25zLnRhZ3MgfHwge30pLnNvbWUoVG9rZW4uaXNVbnJlc29sdmVkKVxuICAgICAgfHwgT2JqZWN0LmtleXMob3B0aW9ucy50YWdzIHx8IHt9KS5zb21lKFRva2VuLmlzVW5yZXNvbHZlZCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQWxsIGFyZ3VtZW50cyB0byBWcGMuZnJvbUxvb2t1cCgpIG11c3QgYmUgY29uY3JldGUgKG5vIFRva2VucylgKTtcbiAgICB9XG5cbiAgICBjb25zdCBmaWx0ZXI6IHtba2V5OiBzdHJpbmddOiBzdHJpbmd9ID0gbWFrZVRhZ0ZpbHRlcihvcHRpb25zLnRhZ3MpO1xuXG4gICAgLy8gV2UgZ2l2ZSBzcGVjaWFsIHRyZWF0bWVudCB0byBzb21lIHRhZ3NcbiAgICBpZiAob3B0aW9ucy52cGNJZCkgeyBmaWx0ZXJbJ3ZwYy1pZCddID0gb3B0aW9ucy52cGNJZDsgfVxuICAgIGlmIChvcHRpb25zLnZwY05hbWUpIHsgZmlsdGVyWyd0YWc6TmFtZSddID0gb3B0aW9ucy52cGNOYW1lOyB9XG4gICAgaWYgKG9wdGlvbnMuaXNEZWZhdWx0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGZpbHRlci5pc0RlZmF1bHQgPSBvcHRpb25zLmlzRGVmYXVsdCA/ICd0cnVlJyA6ICdmYWxzZSc7XG4gICAgfVxuXG4gICAgY29uc3QgYXR0cmlidXRlczogY3hhcGkuVnBjQ29udGV4dFJlc3BvbnNlID0gQ29udGV4dFByb3ZpZGVyLmdldFZhbHVlKHNjb3BlLCB7XG4gICAgICBwcm92aWRlcjogY3hhcGkuVlBDX1BST1ZJREVSLFxuICAgICAgcHJvcHM6IHtcbiAgICAgICAgZmlsdGVyLFxuICAgICAgICByZXR1cm5Bc3ltbWV0cmljU3VibmV0czogdHJ1ZSxcbiAgICAgICAgc3VibmV0R3JvdXBOYW1lVGFnOiBvcHRpb25zLnN1Ym5ldEdyb3VwTmFtZVRhZyxcbiAgICAgIH0gYXMgY3hhcGkuVnBjQ29udGV4dFF1ZXJ5LFxuICAgICAgZHVtbXlWYWx1ZTogdW5kZWZpbmVkLFxuICAgIH0pLnZhbHVlO1xuXG4gICAgcmV0dXJuIG5ldyBMb29rZWRVcFZwYyhzY29wZSwgaWQsIGF0dHJpYnV0ZXMgfHwgRFVNTVlfVlBDX1BST1BTLCBhdHRyaWJ1dGVzID09PSB1bmRlZmluZWQpO1xuXG4gICAgLyoqXG4gICAgICogUHJlZml4ZXMgYWxsIGtleXMgaW4gdGhlIGFyZ3VtZW50IHdpdGggYHRhZzpgLmBcbiAgICAgKi9cbiAgICBmdW5jdGlvbiBtYWtlVGFnRmlsdGVyKHRhZ3M6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9IHwgdW5kZWZpbmVkKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0ge1xuICAgICAgY29uc3QgcmVzdWx0OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgICAgZm9yIChjb25zdCBbbmFtZSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHRhZ3MgfHwge30pKSB7XG4gICAgICAgIHJlc3VsdFtgdGFnOiR7bmFtZX1gXSA9IHZhbHVlO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHJlc3VsdDtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSWRlbnRpZmllciBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjQ2lkckJsb2NrOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNEZWZhdWx0TmV0d29ya0FjbDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjQ2lkckJsb2NrQXNzb2NpYXRpb25zOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwY0RlZmF1bHRTZWN1cml0eUdyb3VwOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNJcHY2Q2lkckJsb2Nrczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaWNTdWJuZXRzOiBJU3VibmV0W10gPSBbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwcml2YXRlU3VibmV0czogSVN1Ym5ldFtdID0gW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgaXNvbGF0ZWQgc3VibmV0cyBpbiB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlzb2xhdGVkU3VibmV0czogSVN1Ym5ldFtdID0gW107XG5cbiAgLyoqXG4gICAqIEFacyBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXIgZm9yIHRoZSBWUE4gZ2F0ZXdheVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwbkdhdGV3YXlJZD86IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGU7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2U6IENmblZQQztcblxuICAvKipcbiAgICogVGhlIE5ldHdvcmtCdWlsZGVyXG4gICAqL1xuICBwcml2YXRlIG5ldHdvcmtCdWlsZGVyOiBOZXR3b3JrQnVpbGRlcjtcblxuICAvKipcbiAgICogU3VibmV0IGNvbmZpZ3VyYXRpb25zIGZvciB0aGlzIFZQQ1xuICAgKi9cbiAgcHJpdmF0ZSBzdWJuZXRDb25maWd1cmF0aW9uOiBTdWJuZXRDb25maWd1cmF0aW9uW10gPSBbXTtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9pbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkID0gbmV3IENvbmNyZXRlRGVwZW5kYWJsZSgpO1xuXG4gIC8qKlxuICAgKiBWcGMgY3JlYXRlcyBhIFZQQyB0aGF0IHNwYW5zIGEgd2hvbGUgcmVnaW9uLlxuICAgKiBJdCB3aWxsIGF1dG9tYXRpY2FsbHkgZGl2aWRlIHRoZSBwcm92aWRlZCBWUEMgQ0lEUiByYW5nZSwgYW5kIGNyZWF0ZSBwdWJsaWMgYW5kIHByaXZhdGUgc3VibmV0cyBwZXIgQXZhaWxhYmlsaXR5IFpvbmUuXG4gICAqIE5ldHdvcmsgcm91dGluZyBmb3IgdGhlIHB1YmxpYyBzdWJuZXRzIHdpbGwgYmUgY29uZmlndXJlZCB0byBhbGxvdyBvdXRib3VuZCBhY2Nlc3MgZGlyZWN0bHkgdmlhIGFuIEludGVybmV0IEdhdGV3YXkuXG4gICAqIE5ldHdvcmsgcm91dGluZyBmb3IgdGhlIHByaXZhdGUgc3VibmV0cyB3aWxsIGJlIGNvbmZpZ3VyZWQgdG8gYWxsb3cgb3V0Ym91bmQgYWNjZXNzIHZpYSBhIHNldCBvZiByZXNpbGllbnQgTkFUIEdhdGV3YXlzIChvbmUgcGVyIEFaKS5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWcGNQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICAvLyBDYW4ndCBoYXZlIGVuYWJsZWREbnNIb3N0bmFtZXMgd2l0aG91dCBlbmFibGVEbnNTdXBwb3J0XG4gICAgaWYgKHByb3BzLmVuYWJsZURuc0hvc3RuYW1lcyAmJiAhcHJvcHMuZW5hYmxlRG5zU3VwcG9ydCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUbyB1c2UgRE5TIEhvc3RuYW1lcywgRE5TIFN1cHBvcnQgbXVzdCBiZSBlbmFibGVkLCBob3dldmVyLCBpdCB3YXMgZXhwbGljaXRseSBkaXNhYmxlZC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBjaWRyQmxvY2sgPSBpZlVuZGVmaW5lZChwcm9wcy5jaWRyLCBWcGMuREVGQVVMVF9DSURSX1JBTkdFKTtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGNpZHJCbG9jaykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJ2NpZHInIHByb3BlcnR5IG11c3QgYmUgYSBjb25jcmV0ZSBDSURSIHN0cmluZywgZ290IGEgVG9rZW4gKHdlIG5lZWQgdG8gcGFyc2UgaXQgZm9yIGF1dG9tYXRpYyBzdWJkaXZpc2lvbilgKTtcbiAgICB9XG5cbiAgICB0aGlzLm5ldHdvcmtCdWlsZGVyID0gbmV3IE5ldHdvcmtCdWlsZGVyKGNpZHJCbG9jayk7XG5cbiAgICBjb25zdCBlbmFibGVEbnNIb3N0bmFtZXMgPSBwcm9wcy5lbmFibGVEbnNIb3N0bmFtZXMgPT0gbnVsbCA/IHRydWUgOiBwcm9wcy5lbmFibGVEbnNIb3N0bmFtZXM7XG4gICAgY29uc3QgZW5hYmxlRG5zU3VwcG9ydCA9IHByb3BzLmVuYWJsZURuc1N1cHBvcnQgPT0gbnVsbCA/IHRydWUgOiBwcm9wcy5lbmFibGVEbnNTdXBwb3J0O1xuICAgIGNvbnN0IGluc3RhbmNlVGVuYW5jeSA9IHByb3BzLmRlZmF1bHRJbnN0YW5jZVRlbmFuY3kgfHwgJ2RlZmF1bHQnO1xuICAgIHRoaXMuaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCA9IHRoaXMuX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ7XG5cbiAgICAvLyBEZWZpbmUgYSBWUEMgdXNpbmcgdGhlIHByb3ZpZGVkIENJRFIgcmFuZ2VcbiAgICB0aGlzLnJlc291cmNlID0gbmV3IENmblZQQyh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBjaWRyQmxvY2ssXG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXMsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0LFxuICAgICAgaW5zdGFuY2VUZW5hbmN5LFxuICAgIH0pO1xuXG4gICAgdGhpcy52cGNEZWZhdWx0TmV0d29ya0FjbCA9IHRoaXMucmVzb3VyY2UuYXR0ckRlZmF1bHROZXR3b3JrQWNsO1xuICAgIHRoaXMudnBjQ2lkckJsb2NrQXNzb2NpYXRpb25zID0gdGhpcy5yZXNvdXJjZS5hdHRyQ2lkckJsb2NrQXNzb2NpYXRpb25zO1xuICAgIHRoaXMudnBjQ2lkckJsb2NrID0gdGhpcy5yZXNvdXJjZS5hdHRyQ2lkckJsb2NrO1xuICAgIHRoaXMudnBjRGVmYXVsdFNlY3VyaXR5R3JvdXAgPSB0aGlzLnJlc291cmNlLmF0dHJEZWZhdWx0U2VjdXJpdHlHcm91cDtcbiAgICB0aGlzLnZwY0lwdjZDaWRyQmxvY2tzID0gdGhpcy5yZXNvdXJjZS5hdHRySXB2NkNpZHJCbG9ja3M7XG5cbiAgICB0aGlzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZyhOQU1FX1RBRywgdGhpcy5ub2RlLnBhdGgpKTtcblxuICAgIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMgPSBzdGFjay5hdmFpbGFiaWxpdHlab25lcztcblxuICAgIGNvbnN0IG1heEFacyA9IHByb3BzLm1heEF6cyAhPT0gdW5kZWZpbmVkID8gcHJvcHMubWF4QXpzIDogMztcbiAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzID0gdGhpcy5hdmFpbGFiaWxpdHlab25lcy5zbGljZSgwLCBtYXhBWnMpO1xuXG4gICAgdGhpcy52cGNJZCA9IHRoaXMucmVzb3VyY2UucmVmO1xuXG4gICAgdGhpcy5zdWJuZXRDb25maWd1cmF0aW9uID0gaWZVbmRlZmluZWQocHJvcHMuc3VibmV0Q29uZmlndXJhdGlvbiwgVnBjLkRFRkFVTFRfU1VCTkVUUyk7XG5cbiAgICBjb25zdCBuYXRHYXRld2F5UGxhY2VtZW50ID0gcHJvcHMubmF0R2F0ZXdheVN1Ym5ldHMgfHwgeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyB9O1xuICAgIGNvbnN0IG5hdEdhdGV3YXlDb3VudCA9IGRldGVybWluZU5hdEdhdGV3YXlDb3VudChwcm9wcy5uYXRHYXRld2F5cywgdGhpcy5zdWJuZXRDb25maWd1cmF0aW9uLCB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzLmxlbmd0aCk7XG5cbiAgICAvLyBzdWJuZXRDb25maWd1cmF0aW9uIG11c3QgYmUgc2V0IGJlZm9yZSBjYWxsaW5nIGNyZWF0ZVN1Ym5ldHNcbiAgICB0aGlzLmNyZWF0ZVN1Ym5ldHMoKTtcblxuICAgIGNvbnN0IGFsbG93T3V0Ym91bmQgPSB0aGlzLnN1Ym5ldENvbmZpZ3VyYXRpb24uZmlsdGVyKFxuICAgICAgc3VibmV0ID0+IChzdWJuZXQuc3VibmV0VHlwZSAhPT0gU3VibmV0VHlwZS5JU09MQVRFRCkpLmxlbmd0aCA+IDA7XG5cbiAgICAvLyBDcmVhdGUgYW4gSW50ZXJuZXQgR2F0ZXdheSBhbmQgYXR0YWNoIGl0IGlmIG5lY2Vzc2FyeVxuICAgIGlmIChhbGxvd091dGJvdW5kKSB7XG4gICAgICBjb25zdCBpZ3cgPSBuZXcgQ2ZuSW50ZXJuZXRHYXRld2F5KHRoaXMsICdJR1cnLCB7XG4gICAgICB9KTtcbiAgICAgIHRoaXMuX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQuYWRkKGlndyk7XG4gICAgICBjb25zdCBhdHQgPSBuZXcgQ2ZuVlBDR2F0ZXdheUF0dGFjaG1lbnQodGhpcywgJ1ZQQ0dXJywge1xuICAgICAgICBpbnRlcm5ldEdhdGV3YXlJZDogaWd3LnJlZixcbiAgICAgICAgdnBjSWQ6IHRoaXMucmVzb3VyY2UucmVmXG4gICAgICB9KTtcblxuICAgICAgKHRoaXMucHVibGljU3VibmV0cyBhcyBQdWJsaWNTdWJuZXRbXSkuZm9yRWFjaChwdWJsaWNTdWJuZXQgPT4ge1xuICAgICAgICBwdWJsaWNTdWJuZXQuYWRkRGVmYXVsdEludGVybmV0Um91dGUoaWd3LnJlZiwgYXR0KTtcbiAgICAgIH0pO1xuXG4gICAgICAvLyBpZiBnYXRld2F5cyBhcmUgbmVlZGVkIGNyZWF0ZSB0aGVtXG4gICAgICBpZiAobmF0R2F0ZXdheUNvdW50ID4gMCkge1xuICAgICAgICBjb25zdCBwcm92aWRlciA9IHByb3BzLm5hdEdhdGV3YXlQcm92aWRlciB8fCBOYXRQcm92aWRlci5nYXRld2F5KCk7XG4gICAgICAgIHRoaXMuY3JlYXRlTmF0R2F0ZXdheXMocHJvdmlkZXIsIG5hdEdhdGV3YXlDb3VudCwgbmF0R2F0ZXdheVBsYWNlbWVudCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKChwcm9wcy52cG5Db25uZWN0aW9ucyB8fCBwcm9wcy52cG5HYXRld2F5QXNuKSAmJiBwcm9wcy52cG5HYXRld2F5ID09PSBmYWxzZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc3BlY2lmeSBgdnBuQ29ubmVjdGlvbnNgIG9yIGB2cG5HYXRld2F5QXNuYCB3aGVuIGB2cG5HYXRld2F5YCBpcyBzZXQgdG8gZmFsc2UuJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnZwbkdhdGV3YXkgfHwgcHJvcHMudnBuQ29ubmVjdGlvbnMgfHwgcHJvcHMudnBuR2F0ZXdheUFzbikge1xuICAgICAgY29uc3QgdnBuR2F0ZXdheSA9IG5ldyBDZm5WUE5HYXRld2F5KHRoaXMsICdWcG5HYXRld2F5Jywge1xuICAgICAgICBhbWF6b25TaWRlQXNuOiBwcm9wcy52cG5HYXRld2F5QXNuLFxuICAgICAgICB0eXBlOiBWcG5Db25uZWN0aW9uVHlwZS5JUFNFQ18xXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgYXR0YWNobWVudCA9IG5ldyBDZm5WUENHYXRld2F5QXR0YWNobWVudCh0aGlzLCAnVlBDVlBOR1cnLCB7XG4gICAgICAgIHZwY0lkOiB0aGlzLnZwY0lkLFxuICAgICAgICB2cG5HYXRld2F5SWQ6IHZwbkdhdGV3YXkucmVmLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMudnBuR2F0ZXdheUlkID0gdnBuR2F0ZXdheS5yZWY7XG5cbiAgICAgIC8vIFByb3BhZ2F0ZSByb3V0ZXMgb24gcm91dGUgdGFibGVzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmlnaHQgc3VibmV0c1xuICAgICAgY29uc3QgdnBuUm91dGVQcm9wYWdhdGlvbiA9IHByb3BzLnZwblJvdXRlUHJvcGFnYXRpb24gfHwgW3sgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFIH1dO1xuICAgICAgY29uc3Qgcm91dGVUYWJsZUlkcyA9IGFsbFJvdXRlVGFibGVJZHMoLi4udnBuUm91dGVQcm9wYWdhdGlvbi5tYXAocyA9PiB0aGlzLnNlbGVjdFN1Ym5ldHMocykpKTtcbiAgICAgIGNvbnN0IHJvdXRlUHJvcGFnYXRpb24gPSBuZXcgQ2ZuVlBOR2F0ZXdheVJvdXRlUHJvcGFnYXRpb24odGhpcywgJ1JvdXRlUHJvcGFnYXRpb24nLCB7XG4gICAgICAgIHJvdXRlVGFibGVJZHMsXG4gICAgICAgIHZwbkdhdGV3YXlJZDogdGhpcy52cG5HYXRld2F5SWRcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUaGUgQVdTOjpFQzI6OlZQTkdhdGV3YXlSb3V0ZVByb3BhZ2F0aW9uIHJlc291cmNlIGNhbm5vdCB1c2UgdGhlIFZQTiBnYXRld2F5XG4gICAgICAvLyB1bnRpbCBpdCBoYXMgc3VjY2Vzc2Z1bGx5IGF0dGFjaGVkIHRvIHRoZSBWUEMuXG4gICAgICAvLyBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWVjMi12cG4tZ2F0ZXdheXJvdXRlcHJvcC5odG1sXG4gICAgICByb3V0ZVByb3BhZ2F0aW9uLm5vZGUuYWRkRGVwZW5kZW5jeShhdHRhY2htZW50KTtcblxuICAgICAgY29uc3QgdnBuQ29ubmVjdGlvbnMgPSBwcm9wcy52cG5Db25uZWN0aW9ucyB8fCB7fTtcbiAgICAgIGZvciAoY29uc3QgW2Nvbm5lY3Rpb25JZCwgY29ubmVjdGlvbl0gb2YgT2JqZWN0LmVudHJpZXModnBuQ29ubmVjdGlvbnMpKSB7XG4gICAgICAgIHRoaXMuYWRkVnBuQ29ubmVjdGlvbihjb25uZWN0aW9uSWQsIGNvbm5lY3Rpb24pO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIEFsbG93IGNyZWF0aW9uIG9mIGdhdGV3YXkgZW5kcG9pbnRzIG9uIFZQQyBpbnN0YW50aWF0aW9uIGFzIHRob3NlIGNhbiBiZVxuICAgIC8vIGltbWVkaWF0ZWx5IGZ1bmN0aW9uYWwgd2l0aG91dCBmdXJ0aGVyIGNvbmZpZ3VyYXRpb24uIFRoaXMgaXMgbm90IHRoZSBjYXNlXG4gICAgLy8gZm9yIGludGVyZmFjZSBlbmRwb2ludHMgd2hlcmUgdGhlIHNlY3VyaXR5IGdyb3VwIG11c3QgYmUgY29uZmlndXJlZC5cbiAgICBpZiAocHJvcHMuZ2F0ZXdheUVuZHBvaW50cykge1xuICAgICAgY29uc3QgZ2F0ZXdheUVuZHBvaW50cyA9IHByb3BzLmdhdGV3YXlFbmRwb2ludHMgfHwge307XG4gICAgICBmb3IgKGNvbnN0IFtlbmRwb2ludElkLCBlbmRwb2ludF0gb2YgT2JqZWN0LmVudHJpZXMoZ2F0ZXdheUVuZHBvaW50cykpIHtcbiAgICAgICAgdGhpcy5hZGRHYXRld2F5RW5kcG9pbnQoZW5kcG9pbnRJZCwgZW5kcG9pbnQpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IFMzIGdhdGV3YXkgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBhZGRHYXRld2F5RW5kcG9pbnQoKWAgaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZFMzRW5kcG9pbnQoaWQ6IHN0cmluZywgc3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbltdKTogR2F0ZXdheVZwY0VuZHBvaW50IHtcbiAgICByZXR1cm4gbmV3IEdhdGV3YXlWcGNFbmRwb2ludCh0aGlzLCBpZCwge1xuICAgICAgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5TMyxcbiAgICAgIHZwYzogdGhpcyxcbiAgICAgIHN1Ym5ldHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IER5bmFtb0RCIGdhdGV3YXkgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBhZGRHYXRld2F5RW5kcG9pbnQoKWAgaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZER5bmFtb0RiRW5kcG9pbnQoaWQ6IHN0cmluZywgc3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbltdKTogR2F0ZXdheVZwY0VuZHBvaW50IHtcbiAgICByZXR1cm4gbmV3IEdhdGV3YXlWcGNFbmRwb2ludCh0aGlzLCBpZCwge1xuICAgICAgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5EWU5BTU9EQixcbiAgICAgIHZwYzogdGhpcyxcbiAgICAgIHN1Ym5ldHNcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlTmF0R2F0ZXdheXMocHJvdmlkZXI6IE5hdFByb3ZpZGVyLCBuYXRDb3VudDogbnVtYmVyLCBwbGFjZW1lbnQ6IFN1Ym5ldFNlbGVjdGlvbik6IHZvaWQge1xuICAgIGNvbnN0IG5hdFN1Ym5ldHM6IFB1YmxpY1N1Ym5ldFtdID0gdGhpcy5zZWxlY3RTdWJuZXRPYmplY3RzKHBsYWNlbWVudCkgYXMgUHVibGljU3VibmV0W107XG4gICAgZm9yIChjb25zdCBzdWIgb2YgbmF0U3VibmV0cykge1xuICAgICAgaWYgKHRoaXMucHVibGljU3VibmV0cy5pbmRleE9mKHN1YikgPT09IC0xKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgbmF0R2F0ZXdheVBsYWNlbWVudCAke3BsYWNlbWVudH0gY29udGFpbnMgbm9uIHB1YmxpYyBzdWJuZXQgJHtzdWJ9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcHJvdmlkZXIuY29uZmlndXJlTmF0KHtcbiAgICAgIHZwYzogdGhpcyxcbiAgICAgIG5hdFN1Ym5ldHM6IG5hdFN1Ym5ldHMuc2xpY2UoMCwgbmF0Q291bnQpLFxuICAgICAgcHJpdmF0ZVN1Ym5ldHM6IHRoaXMucHJpdmF0ZVN1Ym5ldHMgYXMgUHJpdmF0ZVN1Ym5ldFtdXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogY3JlYXRlU3VibmV0cyBjcmVhdGVzIHRoZSBzdWJuZXRzIHNwZWNpZmllZCBieSB0aGUgc3VibmV0IGNvbmZpZ3VyYXRpb25cbiAgICogYXJyYXkgb3IgY3JlYXRlcyB0aGUgYERFRkFVTFRfU1VCTkVUU2AgY29uZmlndXJhdGlvblxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVTdWJuZXRzKCkge1xuICAgIGNvbnN0IHJlbWFpbmluZ1NwYWNlU3VibmV0czogU3VibmV0Q29uZmlndXJhdGlvbltdID0gW107XG5cbiAgICBmb3IgKGNvbnN0IHN1Ym5ldCBvZiB0aGlzLnN1Ym5ldENvbmZpZ3VyYXRpb24pIHtcbiAgICAgIGlmIChzdWJuZXQuY2lkck1hc2sgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICByZW1haW5pbmdTcGFjZVN1Ym5ldHMucHVzaChzdWJuZXQpO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIHRoaXMuY3JlYXRlU3VibmV0UmVzb3VyY2VzKHN1Ym5ldCwgc3VibmV0LmNpZHJNYXNrKTtcbiAgICB9XG5cbiAgICBjb25zdCB0b3RhbFJlbWFpbmluZyA9IHJlbWFpbmluZ1NwYWNlU3VibmV0cy5sZW5ndGggKiB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzLmxlbmd0aDtcbiAgICBjb25zdCBjaWRyTWFza0ZvclJlbWFpbmcgPSB0aGlzLm5ldHdvcmtCdWlsZGVyLm1hc2tGb3JSZW1haW5pbmdTdWJuZXRzKHRvdGFsUmVtYWluaW5nKTtcbiAgICBmb3IgKGNvbnN0IHN1Ym5ldCBvZiByZW1haW5pbmdTcGFjZVN1Ym5ldHMpIHtcbiAgICAgIHRoaXMuY3JlYXRlU3VibmV0UmVzb3VyY2VzKHN1Ym5ldCwgY2lkck1hc2tGb3JSZW1haW5nKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVN1Ym5ldFJlc291cmNlcyhzdWJuZXRDb25maWc6IFN1Ym5ldENvbmZpZ3VyYXRpb24sIGNpZHJNYXNrOiBudW1iZXIpIHtcbiAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzLmZvckVhY2goKHpvbmUsIGluZGV4KSA9PiB7XG4gICAgICBpZiAoc3VibmV0Q29uZmlnLnJlc2VydmVkID09PSB0cnVlKSB7XG4gICAgICAgIC8vIEZvciByZXNlcnZlZCBzdWJuZXRzLCBqdXN0IGFsbG9jYXRlIGlwIHNwYWNlIGJ1dCBkbyBub3QgY3JlYXRlIGFueSByZXNvdXJjZXNcbiAgICAgICAgdGhpcy5uZXR3b3JrQnVpbGRlci5hZGRTdWJuZXQoY2lkck1hc2spO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IG5hbWUgPSBzdWJuZXRJZChzdWJuZXRDb25maWcubmFtZSwgaW5kZXgpO1xuICAgICAgY29uc3Qgc3VibmV0UHJvcHM6IFN1Ym5ldFByb3BzID0ge1xuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiB6b25lLFxuICAgICAgICB2cGNJZDogdGhpcy52cGNJZCxcbiAgICAgICAgY2lkckJsb2NrOiB0aGlzLm5ldHdvcmtCdWlsZGVyLmFkZFN1Ym5ldChjaWRyTWFzayksXG4gICAgICAgIG1hcFB1YmxpY0lwT25MYXVuY2g6IChzdWJuZXRDb25maWcuc3VibmV0VHlwZSA9PT0gU3VibmV0VHlwZS5QVUJMSUMpLFxuICAgICAgfTtcblxuICAgICAgbGV0IHN1Ym5ldDogU3VibmV0O1xuICAgICAgc3dpdGNoIChzdWJuZXRDb25maWcuc3VibmV0VHlwZSkge1xuICAgICAgICBjYXNlIFN1Ym5ldFR5cGUuUFVCTElDOlxuICAgICAgICAgIGNvbnN0IHB1YmxpY1N1Ym5ldCA9IG5ldyBQdWJsaWNTdWJuZXQodGhpcywgbmFtZSwgc3VibmV0UHJvcHMpO1xuICAgICAgICAgIHRoaXMucHVibGljU3VibmV0cy5wdXNoKHB1YmxpY1N1Ym5ldCk7XG4gICAgICAgICAgc3VibmV0ID0gcHVibGljU3VibmV0O1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICBjYXNlIFN1Ym5ldFR5cGUuUFJJVkFURTpcbiAgICAgICAgICBjb25zdCBwcml2YXRlU3VibmV0ID0gbmV3IFByaXZhdGVTdWJuZXQodGhpcywgbmFtZSwgc3VibmV0UHJvcHMpO1xuICAgICAgICAgIHRoaXMucHJpdmF0ZVN1Ym5ldHMucHVzaChwcml2YXRlU3VibmV0KTtcbiAgICAgICAgICBzdWJuZXQgPSBwcml2YXRlU3VibmV0O1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICBjYXNlIFN1Ym5ldFR5cGUuSVNPTEFURUQ6XG4gICAgICAgICAgY29uc3QgaXNvbGF0ZWRTdWJuZXQgPSBuZXcgUHJpdmF0ZVN1Ym5ldCh0aGlzLCBuYW1lLCBzdWJuZXRQcm9wcyk7XG4gICAgICAgICAgdGhpcy5pc29sYXRlZFN1Ym5ldHMucHVzaChpc29sYXRlZFN1Ym5ldCk7XG4gICAgICAgICAgc3VibmV0ID0gaXNvbGF0ZWRTdWJuZXQ7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnJlY29nbml6ZWQgc3VibmV0IHR5cGU6ICR7c3VibmV0Q29uZmlnLnN1Ym5ldFR5cGV9YCk7XG4gICAgICB9XG5cbiAgICAgIC8vIFRoZXNlIHZhbHVlcyB3aWxsIGJlIHVzZWQgdG8gcmVjb3ZlciB0aGUgY29uZmlnIHVwb24gcHJvdmlkZXIgaW1wb3J0XG4gICAgICBjb25zdCBpbmNsdWRlUmVzb3VyY2VUeXBlcyA9IFtDZm5TdWJuZXQuQ0ZOX1JFU09VUkNFX1RZUEVfTkFNRV07XG4gICAgICBzdWJuZXQubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKFNVQk5FVE5BTUVfVEFHLCBzdWJuZXRDb25maWcubmFtZSwge2luY2x1ZGVSZXNvdXJjZVR5cGVzfSkpO1xuICAgICAgc3VibmV0Lm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZyhTVUJORVRUWVBFX1RBRywgc3VibmV0VHlwZVRhZ1ZhbHVlKHN1Ym5ldENvbmZpZy5zdWJuZXRUeXBlKSwge2luY2x1ZGVSZXNvdXJjZVR5cGVzfSkpO1xuICAgIH0pO1xuICB9XG59XG5cbmNvbnN0IFNVQk5FVFRZUEVfVEFHID0gJ2F3cy1jZGs6c3VibmV0LXR5cGUnO1xuY29uc3QgU1VCTkVUTkFNRV9UQUcgPSAnYXdzLWNkazpzdWJuZXQtbmFtZSc7XG5cbmZ1bmN0aW9uIHN1Ym5ldFR5cGVUYWdWYWx1ZSh0eXBlOiBTdWJuZXRUeXBlKSB7XG4gIHN3aXRjaCAodHlwZSkge1xuICAgIGNhc2UgU3VibmV0VHlwZS5QVUJMSUM6IHJldHVybiAnUHVibGljJztcbiAgICBjYXNlIFN1Ym5ldFR5cGUuUFJJVkFURTogcmV0dXJuICdQcml2YXRlJztcbiAgICBjYXNlIFN1Ym5ldFR5cGUuSVNPTEFURUQ6IHJldHVybiAnSXNvbGF0ZWQnO1xuICB9XG59XG5cbi8qKlxuICogU3BlY2lmeSBjb25maWd1cmF0aW9uIHBhcmFtZXRlcnMgZm9yIGEgVlBDIHN1Ym5ldFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN1Ym5ldFByb3BzIHtcblxuICAvKipcbiAgICogVGhlIGF2YWlsYWJpbGl0eSB6b25lIGZvciB0aGUgc3VibmV0XG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgd2hpY2ggdGhpcyBzdWJuZXQgaXMgcGFydCBvZlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIENJRFIgbm90YXRpb24gZm9yIHRoaXMgc3VibmV0XG4gICAqL1xuICByZWFkb25seSBjaWRyQmxvY2s6IHN0cmluZztcblxuICAvKipcbiAgICogQ29udHJvbHMgaWYgYSBwdWJsaWMgSVAgaXMgYXNzb2NpYXRlZCB0byBhbiBpbnN0YW5jZSBhdCBsYXVuY2hcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZSBpbiBTdWJuZXQuUHVibGljLCBmYWxzZSBpbiBTdWJuZXQuUHJpdmF0ZSBvciBTdWJuZXQuSXNvbGF0ZWQuXG4gICAqL1xuICByZWFkb25seSBtYXBQdWJsaWNJcE9uTGF1bmNoPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgbmV3IFZQQyBzdWJuZXQgcmVzb3VyY2VcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpFQzI6OlN1Ym5ldFxuICovXG5leHBvcnQgY2xhc3MgU3VibmV0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJU3VibmV0IHtcblxuICBwdWJsaWMgc3RhdGljIGlzVnBjU3VibmV0KHg6IGFueSk6IHggaXMgU3VibmV0IHtcbiAgICByZXR1cm4gVlBDX1NVQk5FVF9TWU1CT0wgaW4geDtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVN1Ym5ldEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFN1Ym5ldEF0dHJpYnV0ZXMpOiBJU3VibmV0IHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkU3VibmV0KHNjb3BlLCBpZCwgYXR0cnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBdmFpbGFiaWxpdHkgWm9uZSB0aGUgc3VibmV0IGlzIGxvY2F0ZWQgaW5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzdWJuZXRJZCBmb3IgdGhpcyBwYXJ0aWN1bGFyIHN1Ym5ldFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN1Ym5ldElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdWJuZXRWcGNJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0QXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0SXB2NkNpZHJCbG9ja3M6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0TmV0d29ya0FjbEFzc29jaWF0aW9uSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUGFydHMgb2YgdGhpcyBWUEMgc3VibmV0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwZW5kZW5jeUVsZW1lbnRzOiBJRGVwZW5kYWJsZVtdID0gW107XG5cbiAgLyoqXG4gICAqIFRoZSByb3V0ZVRhYmxlSWQgYXR0YWNoZWQgdG8gdGhpcyBzdWJuZXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcm91dGVUYWJsZTogSVJvdXRlVGFibGU7XG5cbiAgcHVibGljIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG5cbiAgcHJpdmF0ZSBfbmV0d29ya0FjbDogSU5ldHdvcmtBY2w7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN1Ym5ldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eSh0aGlzLCBWUENfU1VCTkVUX1NZTUJPTCwgeyB2YWx1ZTogdHJ1ZSB9KTtcblxuICAgIHRoaXMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKE5BTUVfVEFHLCB0aGlzLm5vZGUucGF0aCkpO1xuXG4gICAgdGhpcy5hdmFpbGFiaWxpdHlab25lID0gcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZTtcbiAgICBjb25zdCBzdWJuZXQgPSBuZXcgQ2ZuU3VibmV0KHRoaXMsICdTdWJuZXQnLCB7XG4gICAgICB2cGNJZDogcHJvcHMudnBjSWQsXG4gICAgICBjaWRyQmxvY2s6IHByb3BzLmNpZHJCbG9jayxcbiAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHByb3BzLmF2YWlsYWJpbGl0eVpvbmUsXG4gICAgICBtYXBQdWJsaWNJcE9uTGF1bmNoOiBwcm9wcy5tYXBQdWJsaWNJcE9uTGF1bmNoLFxuICAgIH0pO1xuICAgIHRoaXMuc3VibmV0SWQgPSBzdWJuZXQucmVmO1xuICAgIHRoaXMuc3VibmV0VnBjSWQgPSBzdWJuZXQuYXR0clZwY0lkO1xuICAgIHRoaXMuc3VibmV0QXZhaWxhYmlsaXR5Wm9uZSA9IHN1Ym5ldC5hdHRyQXZhaWxhYmlsaXR5Wm9uZTtcbiAgICB0aGlzLnN1Ym5ldElwdjZDaWRyQmxvY2tzID0gc3VibmV0LmF0dHJJcHY2Q2lkckJsb2NrcztcblxuICAgIC8vIHN1Ym5ldC5hdHRyTmV0d29ya0FjbEFzc29jaWF0aW9uSWQgaXMgdGhlIGRlZmF1bHQgQUNMIGFmdGVyIHRoZSBzdWJuZXRcbiAgICAvLyB3YXMganVzdCBjcmVhdGVkLiBIb3dldmVyLCB0aGUgQUNMIGNhbiBiZSByZXBsYWNlZCBhdCBhIGxhdGVyIHRpbWUuXG4gICAgdGhpcy5fbmV0d29ya0FjbCA9IE5ldHdvcmtBY2wuZnJvbU5ldHdvcmtBY2xJZCh0aGlzLCAnQWNsJywgc3VibmV0LmF0dHJOZXR3b3JrQWNsQXNzb2NpYXRpb25JZCk7XG4gICAgdGhpcy5zdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb25JZCA9IExhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl9uZXR3b3JrQWNsLm5ldHdvcmtBY2xJZCB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gc3VibmV0O1xuXG4gICAgY29uc3QgdGFibGUgPSBuZXcgQ2ZuUm91dGVUYWJsZSh0aGlzLCAnUm91dGVUYWJsZScsIHtcbiAgICAgIHZwY0lkOiBwcm9wcy52cGNJZCxcbiAgICB9KTtcbiAgICB0aGlzLnJvdXRlVGFibGUgPSB7IHJvdXRlVGFibGVJZDogdGFibGUucmVmIH07XG5cbiAgICAvLyBBc3NvY2lhdGUgdGhlIHB1YmxpYyByb3V0ZSB0YWJsZSBmb3IgdGhpcyBzdWJuZXQsIHRvIHRoaXMgc3VibmV0XG4gICAgbmV3IENmblN1Ym5ldFJvdXRlVGFibGVBc3NvY2lhdGlvbih0aGlzLCAnUm91dGVUYWJsZUFzc29jaWF0aW9uJywge1xuICAgICAgc3VibmV0SWQ6IHRoaXMuc3VibmV0SWQsXG4gICAgICByb3V0ZVRhYmxlSWQ6IHRhYmxlLnJlZlxuICAgIH0pO1xuXG4gICAgdGhpcy5pbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkID0gdGhpcy5faW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBkZWZhdWx0IHJvdXRlIHRoYXQgcG9pbnRzIHRvIGEgcGFzc2VkIElHVywgd2l0aCBhIGRlcGVuZGVuY3lcbiAgICogb24gdGhlIElHVydzIGF0dGFjaG1lbnQgdG8gdGhlIFZQQy5cbiAgICpcbiAgICogQHBhcmFtIGdhdGV3YXlJZCB0aGUgbG9naWNhbCBJRCAocmVmKSBvZiB0aGUgZ2F0ZXdheSBhdHRhY2hlZCB0byB5b3VyIFZQQ1xuICAgKiBAcGFyYW0gZ2F0ZXdheUF0dGFjaG1lbnQgdGhlIGdhdGV3YXkgYXR0YWNobWVudCBjb25zdHJ1Y3QgdG8gYmUgYWRkZWQgYXMgYSBkZXBlbmRlbmN5XG4gICAqL1xuICBwdWJsaWMgYWRkRGVmYXVsdEludGVybmV0Um91dGUoZ2F0ZXdheUlkOiBzdHJpbmcsIGdhdGV3YXlBdHRhY2htZW50OiBJRGVwZW5kYWJsZSkge1xuICAgIGNvbnN0IHJvdXRlID0gbmV3IENmblJvdXRlKHRoaXMsIGBEZWZhdWx0Um91dGVgLCB7XG4gICAgICByb3V0ZVRhYmxlSWQ6IHRoaXMucm91dGVUYWJsZS5yb3V0ZVRhYmxlSWQsXG4gICAgICBkZXN0aW5hdGlvbkNpZHJCbG9jazogJzAuMC4wLjAvMCcsXG4gICAgICBnYXRld2F5SWRcbiAgICB9KTtcbiAgICByb3V0ZS5ub2RlLmFkZERlcGVuZGVuY3koZ2F0ZXdheUF0dGFjaG1lbnQpO1xuXG4gICAgLy8gU2luY2UgdGhlICdyb3V0ZScgZGVwZW5kcyBvbiB0aGUgZ2F0ZXdheSBhdHRhY2htZW50LCBqdXN0XG4gICAgLy8gZGVwZW5kaW5nIG9uIHRoZSByb3V0ZSBpcyBlbm91Z2guXG4gICAgdGhpcy5faW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZC5hZGQocm91dGUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE5ldHdvcmsgQUNMIGFzc29jaWF0ZWQgd2l0aCB0aGlzIFN1Ym5ldFxuICAgKlxuICAgKiBVcG9uIGNyZWF0aW9uLCB0aGlzIGlzIHRoZSBkZWZhdWx0IEFDTCB3aGljaCBhbGxvd3MgYWxsIHRyYWZmaWMsIGV4Y2VwdFxuICAgKiBleHBsaWNpdCBERU5ZIGVudHJpZXMgdGhhdCB5b3UgYWRkLlxuICAgKlxuICAgKiBZb3UgY2FuIHJlcGxhY2UgaXQgd2l0aCBhIGN1c3RvbSBBQ0wgd2hpY2ggZGVuaWVzIGFsbCB0cmFmZmljIGV4Y2VwdFxuICAgKiB0aGUgZXhwbGljIGl0IEFMTE9XIGVudHJpZXMgdGhhdCB5b3UgYWRkIGJ5IGNyZWF0aW5nIGEgYE5ldHdvcmtBY2xgXG4gICAqIG9iamVjdCBhbmQgY2FsbGluZyBgYXNzb2NpYXRlTmV0d29ya0FjbCgpYC5cbiAgICovXG4gIHB1YmxpYyBnZXQgbmV0d29ya0FjbCgpOiBJTmV0d29ya0FjbCB7XG4gICAgcmV0dXJuIHRoaXMuX25ldHdvcmtBY2w7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBlbnRyeSB0byB0aGlzIHN1Ym5ldHMgcm91dGUgdGFibGUgdGhhdCBwb2ludHMgdG8gdGhlIHBhc3NlZCBOQVRHYXR3YXlJZFxuICAgKiBAcGFyYW0gbmF0R2F0ZXdheUlkIFRoZSBJRCBvZiB0aGUgTkFUIGdhdGV3YXlcbiAgICovXG4gIHB1YmxpYyBhZGREZWZhdWx0TmF0Um91dGUobmF0R2F0ZXdheUlkOiBzdHJpbmcpIHtcbiAgICB0aGlzLmFkZFJvdXRlKCdEZWZhdWx0Um91dGUnLCB7XG4gICAgICByb3V0ZXJUeXBlOiBSb3V0ZXJUeXBlLk5BVF9HQVRFV0FZLFxuICAgICAgcm91dGVySWQ6IG5hdEdhdGV3YXlJZCxcbiAgICAgIGVuYWJsZXNJbnRlcm5ldENvbm5lY3Rpdml0eTogdHJ1ZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFuIGVudHJ5IHRvIHRoaXMgc3VibmV0cyByb3V0ZSB0YWJsZVxuICAgKi9cbiAgcHVibGljIGFkZFJvdXRlKGlkOiBzdHJpbmcsIG9wdGlvbnM6IEFkZFJvdXRlT3B0aW9ucykge1xuICAgIGlmIChvcHRpb25zLmRlc3RpbmF0aW9uQ2lkckJsb2NrICYmIG9wdGlvbnMuZGVzdGluYXRpb25JcHY2Q2lkckJsb2NrKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBzcGVjaWZ5IGJvdGggJ2Rlc3RpbmF0aW9uQ2lkckJsb2NrJyBhbmQgJ2Rlc3RpbmF0aW9uSXB2NkNpZHJCbG9jaydgKTtcbiAgICB9XG5cbiAgICBjb25zdCByb3V0ZSA9IG5ldyBDZm5Sb3V0ZSh0aGlzLCBpZCwge1xuICAgICAgcm91dGVUYWJsZUlkOiB0aGlzLnJvdXRlVGFibGUucm91dGVUYWJsZUlkLFxuICAgICAgZGVzdGluYXRpb25DaWRyQmxvY2s6IG9wdGlvbnMuZGVzdGluYXRpb25DaWRyQmxvY2sgfHwgKG9wdGlvbnMuZGVzdGluYXRpb25JcHY2Q2lkckJsb2NrID09PSB1bmRlZmluZWQgPyAnMC4wLjAuMC8wJyA6IHVuZGVmaW5lZCksXG4gICAgICBkZXN0aW5hdGlvbklwdjZDaWRyQmxvY2s6IG9wdGlvbnMuZGVzdGluYXRpb25DaWRyQmxvY2ssXG4gICAgICBbcm91dGVyVHlwZVRvUHJvcE5hbWUob3B0aW9ucy5yb3V0ZXJUeXBlKV06IG9wdGlvbnMucm91dGVySWQsXG4gICAgfSk7XG5cbiAgICBpZiAob3B0aW9ucy5lbmFibGVzSW50ZXJuZXRDb25uZWN0aXZpdHkpIHtcbiAgICAgIHRoaXMuX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQuYWRkKHJvdXRlKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYXNzb2NpYXRlTmV0d29ya0FjbChpZDogc3RyaW5nLCBuZXR3b3JrQWNsOiBJTmV0d29ya0FjbCkge1xuICAgIHRoaXMuX25ldHdvcmtBY2wgPSBuZXR3b3JrQWNsO1xuXG4gICAgY29uc3Qgc2NvcGUgPSBDb25zdHJ1Y3QuaXNDb25zdHJ1Y3QobmV0d29ya0FjbCkgPyBuZXR3b3JrQWNsIDogdGhpcztcbiAgICBjb25zdCBvdGhlciA9IENvbnN0cnVjdC5pc0NvbnN0cnVjdChuZXR3b3JrQWNsKSA/IHRoaXMgOiBuZXR3b3JrQWNsO1xuICAgIG5ldyBTdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb24oc2NvcGUsIGlkICsgb3RoZXIubm9kZS51bmlxdWVJZCwge1xuICAgICAgbmV0d29ya0FjbCxcbiAgICAgIHN1Ym5ldDogdGhpcyxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGFkZGluZyBhIG5ldyByb3V0ZSB0byBhIHN1Ym5ldFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZFJvdXRlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJUHY0IHJhbmdlIHRoaXMgcm91dGUgYXBwbGllcyB0b1xuICAgKlxuICAgKiBAZGVmYXVsdCAnMC4wLjAuMC8wJ1xuICAgKi9cbiAgcmVhZG9ubHkgZGVzdGluYXRpb25DaWRyQmxvY2s/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElQdjYgcmFuZ2UgdGhpcyByb3V0ZSBhcHBsaWVzIHRvXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVXNlcyBJUHY2XG4gICAqL1xuICByZWFkb25seSBkZXN0aW5hdGlvbklwdjZDaWRyQmxvY2s/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoYXQgdHlwZSBvZiByb3V0ZXIgdG8gcm91dGUgdGhpcyB0cmFmZmljIHRvXG4gICAqL1xuICByZWFkb25seSByb3V0ZXJUeXBlOiBSb3V0ZXJUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIHJvdXRlclxuICAgKlxuICAgKiBDYW4gYmUgYW4gaW5zdGFuY2UgSUQsIGdhdGV3YXkgSUQsIGV0YywgZGVwZW5kaW5nIG9uIHRoZSByb3V0ZXIgdHlwZS5cbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlcklkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyByb3V0ZSB3aWxsIGVuYWJsZSBpbnRlcm5ldCBjb25uZWN0aXZpdHlcbiAgICpcbiAgICogSWYgdHJ1ZSwgdGhpcyByb3V0ZSB3aWxsIGJlIGFkZGVkIGJlZm9yZSBhbnkgQVdTIHJlc291cmNlcyB0aGF0IGRlcGVuZFxuICAgKiBvbiBpbnRlcm5ldCBjb25uZWN0aXZpdHkgaW4gdGhlIFZQQyB3aWxsIGJlIGNyZWF0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVzSW50ZXJuZXRDb25uZWN0aXZpdHk/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFR5cGUgb2Ygcm91dGVyIHVzZWQgaW4gcm91dGVcbiAqL1xuZXhwb3J0IGVudW0gUm91dGVyVHlwZSB7XG4gIC8qKlxuICAgKiBFZ3Jlc3Mtb25seSBJbnRlcm5ldCBHYXRld2F5XG4gICAqL1xuICBFR1JFU1NfT05MWV9JTlRFUk5FVF9HQVRFV0FZID0gJ0VncmVzc09ubHlJbnRlcm5ldEdhdGV3YXknLFxuXG4gIC8qKlxuICAgKiBJbnRlcm5ldCBHYXRld2F5XG4gICAqL1xuICBHQVRFV0FZID0gJ0dhdGV3YXknLFxuXG4gIC8qKlxuICAgKiBJbnN0YW5jZVxuICAgKi9cbiAgSU5TVEFOQ0UgPSAnSW5zdGFuY2UnLFxuXG4gIC8qKlxuICAgKiBOQVQgR2F0ZXdheVxuICAgKi9cbiAgTkFUX0dBVEVXQVkgPSAnTmF0R2F0ZXdheScsXG5cbiAgLyoqXG4gICAqIE5ldHdvcmsgSW50ZXJmYWNlXG4gICAqL1xuICBORVRXT1JLX0lOVEVSRkFDRSA9ICdOZXR3b3JrSW50ZXJmYWNlJyxcblxuICAvKipcbiAgICogVlBDIHBlZXJpbmcgY29ubmVjdGlvblxuICAgKi9cbiAgVlBDX1BFRVJJTkdfQ09OTkVDVElPTiA9ICdWcGNQZWVyaW5nQ29ubmVjdGlvbicsXG59XG5cbmZ1bmN0aW9uIHJvdXRlclR5cGVUb1Byb3BOYW1lKHJvdXRlclR5cGU6IFJvdXRlclR5cGUpIHtcbiAgcmV0dXJuICh7XG4gICAgW1JvdXRlclR5cGUuRUdSRVNTX09OTFlfSU5URVJORVRfR0FURVdBWV06ICdlZ3Jlc3NPbmx5SW50ZXJuZXRHYXRld2F5SWQnLFxuICAgIFtSb3V0ZXJUeXBlLkdBVEVXQVldOiAnZ2F0ZXdheUlkJyxcbiAgICBbUm91dGVyVHlwZS5JTlNUQU5DRV06ICdpbnN0YW5jZUlkJyxcbiAgICBbUm91dGVyVHlwZS5OQVRfR0FURVdBWV06ICduYXRHYXRld2F5SWQnLFxuICAgIFtSb3V0ZXJUeXBlLk5FVFdPUktfSU5URVJGQUNFXTogJ25ldHdvcmtJbnRlcmZhY2VJZCcsXG4gICAgW1JvdXRlclR5cGUuVlBDX1BFRVJJTkdfQ09OTkVDVElPTl06ICd2cGNQZWVyaW5nQ29ubmVjdGlvbklkJyxcbiAgfSlbcm91dGVyVHlwZV07XG59XG5cbi8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTpuby1lbXB0eS1pbnRlcmZhY2VcbmV4cG9ydCBpbnRlcmZhY2UgUHVibGljU3VibmV0UHJvcHMgZXh0ZW5kcyBTdWJuZXRQcm9wcyB7XG5cbn1cblxuZXhwb3J0IGludGVyZmFjZSBJUHVibGljU3VibmV0IGV4dGVuZHMgSVN1Ym5ldCB7IH1cblxuZXhwb3J0IGludGVyZmFjZSBQdWJsaWNTdWJuZXRBdHRyaWJ1dGVzIGV4dGVuZHMgU3VibmV0QXR0cmlidXRlcyB7IH1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgcHVibGljIFZQQyBzdWJuZXQgcmVzb3VyY2VcbiAqL1xuZXhwb3J0IGNsYXNzIFB1YmxpY1N1Ym5ldCBleHRlbmRzIFN1Ym5ldCBpbXBsZW1lbnRzIElQdWJsaWNTdWJuZXQge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVB1YmxpY1N1Ym5ldEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFB1YmxpY1N1Ym5ldEF0dHJpYnV0ZXMpOiBJUHVibGljU3VibmV0IHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkU3VibmV0KHNjb3BlLCBpZCwgYXR0cnMpO1xuICB9XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFB1YmxpY1N1Ym5ldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBtYW5hZ2VkIE5BVCBnYXRld2F5IGF0dGFjaGVkIHRvIHRoaXMgcHVibGljIHN1Ym5ldC5cbiAgICogQWxzbyBhZGRzIHRoZSBFSVAgZm9yIHRoZSBtYW5hZ2VkIE5BVC5cbiAgICogQHJldHVybnMgQSByZWYgdG8gdGhlIHRoZSBOQVQgR2F0ZXdheSBJRFxuICAgKi9cbiAgcHVibGljIGFkZE5hdEdhdGV3YXkoKSB7XG4gICAgLy8gQ3JlYXRlIGEgTkFUIEdhdGV3YXkgaW4gdGhpcyBwdWJsaWMgc3VibmV0XG4gICAgY29uc3Qgbmd3ID0gbmV3IENmbk5hdEdhdGV3YXkodGhpcywgYE5BVEdhdGV3YXlgLCB7XG4gICAgICBzdWJuZXRJZDogdGhpcy5zdWJuZXRJZCxcbiAgICAgIGFsbG9jYXRpb25JZDogbmV3IENmbkVJUCh0aGlzLCBgRUlQYCwge1xuICAgICAgICBkb21haW46ICd2cGMnXG4gICAgICB9KS5hdHRyQWxsb2NhdGlvbklkLFxuICAgIH0pO1xuICAgIHJldHVybiBuZ3c7XG4gIH1cbn1cblxuLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm5vLWVtcHR5LWludGVyZmFjZVxuZXhwb3J0IGludGVyZmFjZSBQcml2YXRlU3VibmV0UHJvcHMgZXh0ZW5kcyBTdWJuZXRQcm9wcyB7XG5cbn1cblxuZXhwb3J0IGludGVyZmFjZSBJUHJpdmF0ZVN1Ym5ldCBleHRlbmRzIElTdWJuZXQgeyB9XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJpdmF0ZVN1Ym5ldEF0dHJpYnV0ZXMgZXh0ZW5kcyBTdWJuZXRBdHRyaWJ1dGVzIHsgfVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBwcml2YXRlIFZQQyBzdWJuZXQgcmVzb3VyY2VcbiAqL1xuZXhwb3J0IGNsYXNzIFByaXZhdGVTdWJuZXQgZXh0ZW5kcyBTdWJuZXQgaW1wbGVtZW50cyBJUHJpdmF0ZVN1Ym5ldCB7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tUHJpdmF0ZVN1Ym5ldEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFByaXZhdGVTdWJuZXRBdHRyaWJ1dGVzKTogSVByaXZhdGVTdWJuZXQge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRTdWJuZXQoc2NvcGUsIGlkLCBhdHRycyk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUHJpdmF0ZVN1Ym5ldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gIH1cbn1cblxuZnVuY3Rpb24gaWZVbmRlZmluZWQ8VD4odmFsdWU6IFQgfCB1bmRlZmluZWQsIGRlZmF1bHRWYWx1ZTogVCk6IFQge1xuICByZXR1cm4gdmFsdWUgIT09IHVuZGVmaW5lZCA/IHZhbHVlIDogZGVmYXVsdFZhbHVlO1xufVxuXG5jbGFzcyBJbXBvcnRlZFZwYyBleHRlbmRzIFZwY0Jhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcbiAgcHVibGljIHJlYWRvbmx5IHByaXZhdGVTdWJuZXRzOiBJU3VibmV0W107XG4gIHB1YmxpYyByZWFkb25seSBpc29sYXRlZFN1Ym5ldHM6IElTdWJuZXRbXTtcbiAgcHVibGljIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IHZwbkdhdGV3YXlJZD86IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlID0gbmV3IENvbmNyZXRlRGVwZW5kYWJsZSgpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWcGNBdHRyaWJ1dGVzLCBpc0luY29tcGxldGU6IGJvb2xlYW4pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52cGNJZCA9IHByb3BzLnZwY0lkO1xuICAgIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMgPSBwcm9wcy5hdmFpbGFiaWxpdHlab25lcztcbiAgICB0aGlzLnZwbkdhdGV3YXlJZCA9IHByb3BzLnZwbkdhdGV3YXlJZDtcbiAgICB0aGlzLmluY29tcGxldGVTdWJuZXREZWZpbml0aW9uID0gaXNJbmNvbXBsZXRlO1xuXG4gICAgLy8gdHNsaW50OmRpc2FibGU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgY29uc3QgcHViID0gbmV3IEltcG9ydFN1Ym5ldEdyb3VwKHByb3BzLnB1YmxpY1N1Ym5ldElkcywgcHJvcHMucHVibGljU3VibmV0TmFtZXMsIHByb3BzLnB1YmxpY1N1Ym5ldFJvdXRlVGFibGVJZHMsIFN1Ym5ldFR5cGUuUFVCTElDLCB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzLCAncHVibGljU3VibmV0SWRzJywgJ3B1YmxpY1N1Ym5ldE5hbWVzJywgJ3B1YmxpY1N1Ym5ldFJvdXRlVGFibGVJZHMnKTtcbiAgICBjb25zdCBwcml2ID0gbmV3IEltcG9ydFN1Ym5ldEdyb3VwKHByb3BzLnByaXZhdGVTdWJuZXRJZHMsIHByb3BzLnByaXZhdGVTdWJuZXROYW1lcywgcHJvcHMucHJpdmF0ZVN1Ym5ldFJvdXRlVGFibGVJZHMsIFN1Ym5ldFR5cGUuUFJJVkFURSwgdGhpcy5hdmFpbGFiaWxpdHlab25lcywgJ3ByaXZhdGVTdWJuZXRJZHMnLCAncHJpdmF0ZVN1Ym5ldE5hbWVzJywgJ3ByaXZhdGVTdWJuZXRSb3V0ZVRhYmxlSWRzJyk7XG4gICAgY29uc3QgaXNvID0gbmV3IEltcG9ydFN1Ym5ldEdyb3VwKHByb3BzLmlzb2xhdGVkU3VibmV0SWRzLCBwcm9wcy5pc29sYXRlZFN1Ym5ldE5hbWVzLCBwcm9wcy5pc29sYXRlZFN1Ym5ldFJvdXRlVGFibGVJZHMsIFN1Ym5ldFR5cGUuSVNPTEFURUQsIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMsICdpc29sYXRlZFN1Ym5ldElkcycsICdpc29sYXRlZFN1Ym5ldE5hbWVzJywgJ2lzb2xhdGVkU3VibmV0Um91dGVUYWJsZUlkcycpO1xuICAgIC8vIHRzbGludDplbmFibGU6bWF4LWxpbmUtbGVuZ3RoXG5cbiAgICB0aGlzLnB1YmxpY1N1Ym5ldHMgPSBwdWIuaW1wb3J0KHRoaXMpO1xuICAgIHRoaXMucHJpdmF0ZVN1Ym5ldHMgPSBwcml2LmltcG9ydCh0aGlzKTtcbiAgICB0aGlzLmlzb2xhdGVkU3VibmV0cyA9IGlzby5pbXBvcnQodGhpcyk7XG4gIH1cbn1cblxuY2xhc3MgTG9va2VkVXBWcGMgZXh0ZW5kcyBWcGNCYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHZwY0lkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSB2cG5HYXRld2F5SWQ/OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkOiBJRGVwZW5kYWJsZSA9IG5ldyBDb25jcmV0ZURlcGVuZGFibGUoKTtcbiAgcHVibGljIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcbiAgcHVibGljIHJlYWRvbmx5IHByaXZhdGVTdWJuZXRzOiBJU3VibmV0W107XG4gIHB1YmxpYyByZWFkb25seSBpc29sYXRlZFN1Ym5ldHM6IElTdWJuZXRbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogY3hhcGkuVnBjQ29udGV4dFJlc3BvbnNlLCBpc0luY29tcGxldGU6IGJvb2xlYW4pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52cGNJZCA9IHByb3BzLnZwY0lkO1xuICAgIHRoaXMudnBuR2F0ZXdheUlkID0gcHJvcHMudnBuR2F0ZXdheUlkO1xuICAgIHRoaXMuaW5jb21wbGV0ZVN1Ym5ldERlZmluaXRpb24gPSBpc0luY29tcGxldGU7XG5cbiAgICBjb25zdCBzdWJuZXRHcm91cHMgPSBwcm9wcy5zdWJuZXRHcm91cHMgfHwgW107XG4gICAgY29uc3QgYXZhaWxhYmlsaXR5Wm9uZXMgPSBBcnJheS5mcm9tKG5ldyBTZXQ8c3RyaW5nPihmbGF0TWFwKHN1Ym5ldEdyb3Vwcywgc3VibmV0R3JvdXAgPT4ge1xuICAgICAgcmV0dXJuIHN1Ym5ldEdyb3VwLnN1Ym5ldHMubWFwKHN1Ym5ldCA9PiBzdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSk7XG4gICAgfSkpKTtcbiAgICBhdmFpbGFiaWxpdHlab25lcy5zb3J0KChhejEsIGF6MikgPT4gYXoxLmxvY2FsZUNvbXBhcmUoYXoyKSk7XG4gICAgdGhpcy5hdmFpbGFiaWxpdHlab25lcyA9IGF2YWlsYWJpbGl0eVpvbmVzO1xuXG4gICAgdGhpcy5wdWJsaWNTdWJuZXRzID0gdGhpcy5leHRyYWN0U3VibmV0c09mVHlwZShzdWJuZXRHcm91cHMsIGN4YXBpLlZwY1N1Ym5ldEdyb3VwVHlwZS5QVUJMSUMpO1xuICAgIHRoaXMucHJpdmF0ZVN1Ym5ldHMgPSB0aGlzLmV4dHJhY3RTdWJuZXRzT2ZUeXBlKHN1Ym5ldEdyb3VwcywgY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlLlBSSVZBVEUpO1xuICAgIHRoaXMuaXNvbGF0ZWRTdWJuZXRzID0gdGhpcy5leHRyYWN0U3VibmV0c09mVHlwZShzdWJuZXRHcm91cHMsIGN4YXBpLlZwY1N1Ym5ldEdyb3VwVHlwZS5JU09MQVRFRCk7XG4gIH1cblxuICBwcml2YXRlIGV4dHJhY3RTdWJuZXRzT2ZUeXBlKHN1Ym5ldEdyb3VwczogY3hhcGkuVnBjU3VibmV0R3JvdXBbXSwgc3VibmV0R3JvdXBUeXBlOiBjeGFwaS5WcGNTdWJuZXRHcm91cFR5cGUpOiBJU3VibmV0W10ge1xuICAgIHJldHVybiBmbGF0TWFwKHN1Ym5ldEdyb3Vwcy5maWx0ZXIoc3VibmV0R3JvdXAgPT4gc3VibmV0R3JvdXAudHlwZSA9PT0gc3VibmV0R3JvdXBUeXBlKSxcbiAgICAgICAgc3VibmV0R3JvdXAgPT4gdGhpcy5zdWJuZXRHcm91cFRvU3VibmV0cyhzdWJuZXRHcm91cCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBzdWJuZXRHcm91cFRvU3VibmV0cyhzdWJuZXRHcm91cDogY3hhcGkuVnBjU3VibmV0R3JvdXApOiBJU3VibmV0W10ge1xuICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxJU3VibmV0PigpO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgc3VibmV0R3JvdXAuc3VibmV0cy5sZW5ndGg7IGkrKykge1xuICAgICAgY29uc3QgdnBjU3VibmV0ID0gc3VibmV0R3JvdXAuc3VibmV0c1tpXTtcbiAgICAgIHJldC5wdXNoKFN1Ym5ldC5mcm9tU3VibmV0QXR0cmlidXRlcyh0aGlzLCBgJHtzdWJuZXRHcm91cC5uYW1lfVN1Ym5ldCR7aSArIDF9YCwge1xuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiB2cGNTdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSxcbiAgICAgICAgc3VibmV0SWQ6IHZwY1N1Ym5ldC5zdWJuZXRJZCxcbiAgICAgICAgcm91dGVUYWJsZUlkOiB2cGNTdWJuZXQucm91dGVUYWJsZUlkLFxuICAgICAgfSkpO1xuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG59XG5cbmZ1bmN0aW9uIGZsYXRNYXA8VCwgVT4oeHM6IFRbXSwgZm46ICh4OiBUKSA9PiBVW10pOiBVW10ge1xuICBjb25zdCByZXQgPSBuZXcgQXJyYXk8VT4oKTtcbiAgZm9yIChjb25zdCB4IG9mIHhzKSB7XG4gICAgcmV0LnB1c2goLi4uZm4oeCkpO1xuICB9XG4gIHJldHVybiByZXQ7XG59XG5cbmNsYXNzIENvbXBvc2l0ZURlcGVuZGFibGUgaW1wbGVtZW50cyBJRGVwZW5kYWJsZSB7XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVwZW5kYWJsZXMgPSBuZXcgQXJyYXk8SURlcGVuZGFibGU+KCk7XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgRGVwZW5kYWJsZVRyYWl0LmltcGxlbWVudCh0aGlzLCB7XG4gICAgICBnZXQgZGVwZW5kZW5jeVJvb3RzKCkge1xuICAgICAgICBjb25zdCByZXQgPSBbXTtcbiAgICAgICAgZm9yIChjb25zdCBkZXAgb2Ygc2VsZi5kZXBlbmRhYmxlcykge1xuICAgICAgICAgIHJldC5wdXNoKC4uLkRlcGVuZGFibGVUcmFpdC5nZXQoZGVwKS5kZXBlbmRlbmN5Um9vdHMpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgY29uc3RydWN0IHRvIHRoZSBkZXBlbmRlbmN5IHJvb3RzXG4gICAqL1xuICBwdWJsaWMgYWRkKGRlcDogSURlcGVuZGFibGUpIHtcbiAgICB0aGlzLmRlcGVuZGFibGVzLnB1c2goZGVwKTtcbiAgfVxufVxuXG4vKipcbiAqIEludm9rZSBhIGZ1bmN0aW9uIG9uIGEgdmFsdWUgKGZvciBpdHMgc2lkZSBlZmZlY3QpIGFuZCByZXR1cm4gdGhlIHZhbHVlXG4gKi9cbmZ1bmN0aW9uIHRhcDxUPih4OiBULCBmbjogKHg6IFQpID0+IHZvaWQpOiBUIHtcbiAgZm4oeCk7XG4gIHJldHVybiB4O1xufVxuXG5jbGFzcyBJbXBvcnRlZFN1Ym5ldCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVN1Ym5ldCwgSVB1YmxpY1N1Ym5ldCwgSVByaXZhdGVTdWJuZXQge1xuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGUgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBzdWJuZXRJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcm91dGVUYWJsZTogSVJvdXRlVGFibGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFN1Ym5ldEF0dHJpYnV0ZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKCFhdHRycy5yb3V0ZVRhYmxlSWQpIHtcbiAgICAgIGNvbnN0IHJlZiA9IFRva2VuLmlzVW5yZXNvbHZlZChhdHRycy5zdWJuZXRJZClcbiAgICAgICAgPyBgYXQgJyR7c2NvcGUubm9kZS5wYXRofS8ke2lkfSdgXG4gICAgICAgIDogYCcke2F0dHJzLnN1Ym5ldElkfSdgO1xuICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOiBtYXgtbGluZS1sZW5ndGhcbiAgICAgIHNjb3BlLm5vZGUuYWRkV2FybmluZyhgTm8gcm91dGVUYWJsZUlkIHdhcyBwcm92aWRlZCB0byB0aGUgc3VibmV0ICR7cmVmfS4gQXR0ZW1wdGluZyB0byByZWFkIGl0cyAucm91dGVUYWJsZS5yb3V0ZVRhYmxlSWQgd2lsbCByZXR1cm4gbnVsbC91bmRlZmluZWQuIChNb3JlIGluZm86IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzMxNzEpYCk7XG4gICAgfVxuXG4gICAgdGhpcy5hdmFpbGFiaWxpdHlab25lID0gYXR0cnMuYXZhaWxhYmlsaXR5Wm9uZTtcbiAgICB0aGlzLnN1Ym5ldElkID0gYXR0cnMuc3VibmV0SWQ7XG4gICAgdGhpcy5yb3V0ZVRhYmxlID0ge1xuICAgICAgLy8gRm9yY2luZyByb3V0ZVRhYmxlSWQgdG8gcHJldGVuZCBub24tbnVsbCB0byBtYWludGFpbiBiYWNrd2FyZHMtY29tcGF0aWJpbGl0eS4gU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzMxNzFcbiAgICAgIHJvdXRlVGFibGVJZDogYXR0cnMucm91dGVUYWJsZUlkIVxuICAgIH07XG4gIH1cblxuICBwdWJsaWMgYXNzb2NpYXRlTmV0d29ya0FjbChpZDogc3RyaW5nLCBuZXR3b3JrQWNsOiBJTmV0d29ya0FjbCk6IHZvaWQge1xuICAgIGNvbnN0IHNjb3BlID0gQ29uc3RydWN0LmlzQ29uc3RydWN0KG5ldHdvcmtBY2wpID8gbmV0d29ya0FjbCA6IHRoaXM7XG4gICAgY29uc3Qgb3RoZXIgPSBDb25zdHJ1Y3QuaXNDb25zdHJ1Y3QobmV0d29ya0FjbCkgPyB0aGlzIDogbmV0d29ya0FjbDtcbiAgICBuZXcgU3VibmV0TmV0d29ya0FjbEFzc29jaWF0aW9uKHNjb3BlLCBpZCArIG90aGVyLm5vZGUudW5pcXVlSWQsIHtcbiAgICAgIG5ldHdvcmtBY2wsXG4gICAgICBzdWJuZXQ6IHRoaXMsXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBEZXRlcm1pbmUgKGFuZCB2YWxpZGF0ZSkgdGhlIE5BVCBnYXRld2F5IGNvdW50IHcuci50LiB0aGUgcmVzdCBvZiB0aGUgc3VibmV0IGNvbmZpZ3VyYXRpb25cbiAqXG4gKiBXZSBoYXZlIHRoZSBmb2xsb3dpbmcgcmVxdWlyZW1lbnRzOlxuICpcbiAqIC0gTmF0R2F0ZXdheUNvdW50ID0gMCA9PT4gdGhlcmUgYXJlIG5vIHByaXZhdGUgc3VibmV0c1xuICogLSBOYXRHYXRld2F5Q291bnQgPiAwID09PiB0aGVyZSBtdXN0IGJlIHB1YmxpYyBzdWJuZXRzXG4gKlxuICogRG8gd2Ugd2FudCB0byByZXF1aXJlIHRoYXQgdGhlcmUgYXJlIHByaXZhdGUgc3VibmV0cyBpZiB0aGVyZSBhcmUgTmF0R2F0ZXdheXM/XG4gKiBUaGV5IHNlZW0gcG9pbnRsZXNzIGJ1dCBJIHNlZSBubyByZWFzb24gdG8gcHJldmVudCBpdC5cbiAqL1xuZnVuY3Rpb24gZGV0ZXJtaW5lTmF0R2F0ZXdheUNvdW50KHJlcXVlc3RlZENvdW50OiBudW1iZXIgfCB1bmRlZmluZWQsIHN1Ym5ldENvbmZpZzogU3VibmV0Q29uZmlndXJhdGlvbltdLCBhekNvdW50OiBudW1iZXIpIHtcbiAgY29uc3QgaGFzUHJpdmF0ZVN1Ym5ldHMgPSBzdWJuZXRDb25maWcuc29tZShjID0+IGMuc3VibmV0VHlwZSA9PT0gU3VibmV0VHlwZS5QUklWQVRFKTtcbiAgY29uc3QgaGFzUHVibGljU3VibmV0cyA9IHN1Ym5ldENvbmZpZy5zb21lKGMgPT4gYy5zdWJuZXRUeXBlID09PSBTdWJuZXRUeXBlLlBVQkxJQyk7XG5cbiAgY29uc3QgY291bnQgPSByZXF1ZXN0ZWRDb3VudCAhPT0gdW5kZWZpbmVkID8gTWF0aC5taW4ocmVxdWVzdGVkQ291bnQsIGF6Q291bnQpIDogKGhhc1ByaXZhdGVTdWJuZXRzID8gYXpDb3VudCA6IDApO1xuXG4gIGlmIChjb3VudCA9PT0gMCAmJiBoYXNQcml2YXRlU3VibmV0cykge1xuICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICB0aHJvdyBuZXcgRXJyb3IoYElmIHlvdSBkbyBub3Qgd2FudCBOQVQgZ2F0ZXdheXMgKG5hdEdhdGV3YXlzPTApLCBtYWtlIHN1cmUgeW91IGRvbid0IGNvbmZpZ3VyZSBhbnkgUFJJVkFURSBzdWJuZXRzIGluICdzdWJuZXRDb25maWd1cmF0aW9uJyAobWFrZSB0aGVtIFBVQkxJQyBvciBJU09MQVRFRCBpbnN0ZWFkKWApO1xuICB9XG5cbiAgaWYgKGNvdW50ID4gMCAmJiAhaGFzUHVibGljU3VibmV0cykge1xuICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICB0aHJvdyBuZXcgRXJyb3IoYElmIHlvdSBjb25maWd1cmUgUFJJVkFURSBzdWJuZXRzIGluICdzdWJuZXRDb25maWd1cmF0aW9uJywgeW91IG11c3QgYWxzbyBjb25maWd1cmUgUFVCTElDIHN1Ym5ldHMgdG8gcHV0IHRoZSBOQVQgZ2F0ZXdheXMgaW50byAoZ290ICR7SlNPTi5zdHJpbmdpZnkoc3VibmV0Q29uZmlnKX0uYCk7XG4gIH1cblxuICByZXR1cm4gY291bnQ7XG59XG5cbi8qKlxuICogVGhlcmUgYXJlIHJldHVybmVkIHdoZW4gdGhlIHByb3ZpZGVyIGhhcyBub3Qgc3VwcGxpZWQgcHJvcHMgeWV0XG4gKlxuICogSXQncyBvbmx5IHVzZWQgZm9yIHRlc3RpbmcgYW5kIG9uIHRoZSBmaXJzdCBydW4tdGhyb3VnaC5cbiAqL1xuY29uc3QgRFVNTVlfVlBDX1BST1BTOiBjeGFwaS5WcGNDb250ZXh0UmVzcG9uc2UgPSB7XG4gIGF2YWlsYWJpbGl0eVpvbmVzOiBbXSxcbiAgaXNvbGF0ZWRTdWJuZXRJZHM6IHVuZGVmaW5lZCxcbiAgaXNvbGF0ZWRTdWJuZXROYW1lczogdW5kZWZpbmVkLFxuICBpc29sYXRlZFN1Ym5ldFJvdXRlVGFibGVJZHM6IHVuZGVmaW5lZCxcbiAgcHJpdmF0ZVN1Ym5ldElkczogdW5kZWZpbmVkLFxuICBwcml2YXRlU3VibmV0TmFtZXM6IHVuZGVmaW5lZCxcbiAgcHJpdmF0ZVN1Ym5ldFJvdXRlVGFibGVJZHM6IHVuZGVmaW5lZCxcbiAgcHVibGljU3VibmV0SWRzOiB1bmRlZmluZWQsXG4gIHB1YmxpY1N1Ym5ldE5hbWVzOiB1bmRlZmluZWQsXG4gIHB1YmxpY1N1Ym5ldFJvdXRlVGFibGVJZHM6IHVuZGVmaW5lZCxcbiAgc3VibmV0R3JvdXBzOiBbXG4gICAge1xuICAgICAgbmFtZTogJ1B1YmxpYycsXG4gICAgICB0eXBlOiBjeGFwaS5WcGNTdWJuZXRHcm91cFR5cGUuUFVCTElDLFxuICAgICAgc3VibmV0czogW1xuICAgICAgICB7XG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2R1bW15LTFhJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3MtMTIzNDUnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi0xMjM0NXMnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2R1bW15LTFiJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3MtNjc4OTAnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi02Nzg5MHMnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9LFxuICAgIHtcbiAgICAgIG5hbWU6ICdQcml2YXRlJyxcbiAgICAgIHR5cGU6IGN4YXBpLlZwY1N1Ym5ldEdyb3VwVHlwZS5QUklWQVRFLFxuICAgICAgc3VibmV0czogW1xuICAgICAgICB7XG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2R1bW15LTFhJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3AtMTIzNDUnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi0xMjM0NXAnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2R1bW15LTFiJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3AtNjc4OTAnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi01Nzg5MHAnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9LFxuICBdLFxuICB2cGNJZDogJ3ZwYy0xMjM0NScsXG59O1xuIl19