"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
const vpc_1 = require("./vpc");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets || [{ subnetType: vpc_1.SubnetType.PRIVATE }];
        const routeTableIds = util_1.allRouteTableIds(...subnets.map(s => props.vpc.selectSubnets(s)));
        if (routeTableIds.length === 0) {
            throw new Error(`Can't add a gateway endpoint to VPC; route table IDs are not available`);
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuil-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups
        });
        const subnets = props.vpc.selectSubnets(Object.assign(Object.assign({}, props.subnets), { onePerAz: true }));
        const subnetIds = subnets.subnetIds;
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: props.privateDnsEnabled !== undefined ? props.privateDnsEnabled : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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