import iam = require('@aws-cdk/aws-iam');
import { Construct, IResource, Resource } from '@aws-cdk/core';
import { Connections, IConnectable } from './connections';
import { ISecurityGroup } from './security-group';
import { IVpc, SubnetSelection } from './vpc';
/**
 * A VPC endpoint.
 */
export interface IVpcEndpoint extends IResource {
    /**
     * The VPC endpoint identifier.
     * @attribute
     */
    readonly vpcEndpointId: string;
}
export declare abstract class VpcEndpoint extends Resource implements IVpcEndpoint {
    abstract readonly vpcEndpointId: string;
    protected policyDocument?: iam.PolicyDocument;
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement: iam.PolicyStatement): void;
}
/**
 * A gateway VPC endpoint.
 */
export interface IGatewayVpcEndpoint extends IVpcEndpoint {
}
/**
 * The type of VPC endpoint.
 */
export declare enum VpcEndpointType {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    INTERFACE = "Interface",
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    GATEWAY = "Gateway"
}
/**
 * A service for a gateway VPC endpoint.
 */
export interface IGatewayVpcEndpointService {
    /**
     * The name of the service.
     */
    readonly name: string;
}
/**
 * An AWS service for a gateway VPC endpoint.
 */
export declare class GatewayVpcEndpointAwsService implements IGatewayVpcEndpointService {
    static readonly DYNAMODB: GatewayVpcEndpointAwsService;
    static readonly S3: GatewayVpcEndpointAwsService;
    /**
     * The name of the service.
     */
    readonly name: string;
    constructor(name: string, prefix?: string);
}
/**
 * Options to add a gateway endpoint to a VPC.
 */
export interface GatewayVpcEndpointOptions {
    /**
     * The service to use for this gateway VPC endpoint.
     */
    readonly service: IGatewayVpcEndpointService;
    /**
     * Where to add endpoint routing.
     *
     * @default private subnets
     */
    readonly subnets?: SubnetSelection[];
}
/**
 * Construction properties for a GatewayVpcEndpoint.
 */
export interface GatewayVpcEndpointProps extends GatewayVpcEndpointOptions {
    /**
     * The VPC network in which the gateway endpoint will be used.
     */
    readonly vpc: IVpc;
}
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
export declare class GatewayVpcEndpoint extends VpcEndpoint implements IGatewayVpcEndpoint {
    static fromGatewayVpcEndpointId(scope: Construct, id: string, gatewayVpcEndpointId: string): IGatewayVpcEndpoint;
    /**
     * The gateway VPC endpoint identifier.
     */
    readonly vpcEndpointId: string;
    /**
     * The date and time the gateway VPC endpoint was created.
     * @attribute
     */
    readonly vpcEndpointCreationTimestamp: string;
    /**
     * @attribute
     */
    readonly vpcEndpointNetworkInterfaceIds: string[];
    /**
     * @attribute
     */
    readonly vpcEndpointDnsEntries: string[];
    constructor(scope: Construct, id: string, props: GatewayVpcEndpointProps);
}
/**
 * A service for an interface VPC endpoint.
 */
export interface IInterfaceVpcEndpointService {
    /**
     * The name of the service.
     */
    readonly name: string;
    /**
     * The port of the service.
     */
    readonly port: number;
}
/**
 * An AWS service for an interface VPC endpoint.
 */
export declare class InterfaceVpcEndpointAwsService implements IInterfaceVpcEndpointService {
    static readonly SAGEMAKER_NOTEBOOK: InterfaceVpcEndpointAwsService;
    static readonly CLOUDFORMATION: InterfaceVpcEndpointAwsService;
    static readonly CLOUDTRAIL: InterfaceVpcEndpointAwsService;
    static readonly CODEBUILD: InterfaceVpcEndpointAwsService;
    static readonly CODEBUILD_FIPS: InterfaceVpcEndpointAwsService;
    static readonly CODECOMMIT: InterfaceVpcEndpointAwsService;
    static readonly CODECOMMIT_FIPS: InterfaceVpcEndpointAwsService;
    static readonly CODEPIPELINE: InterfaceVpcEndpointAwsService;
    static readonly CONFIG: InterfaceVpcEndpointAwsService;
    static readonly EC2: InterfaceVpcEndpointAwsService;
    static readonly EC2_MESSAGES: InterfaceVpcEndpointAwsService;
    static readonly ECR: InterfaceVpcEndpointAwsService;
    static readonly ECR_DOCKER: InterfaceVpcEndpointAwsService;
    static readonly ECS: InterfaceVpcEndpointAwsService;
    static readonly ECS_AGENT: InterfaceVpcEndpointAwsService;
    static readonly ECS_TELEMETRY: InterfaceVpcEndpointAwsService;
    static readonly ELASTIC_INFERENCE_RUNTIME: InterfaceVpcEndpointAwsService;
    static readonly ELASTIC_LOAD_BALANCING: InterfaceVpcEndpointAwsService;
    static readonly CLOUDWATCH_EVENTS: InterfaceVpcEndpointAwsService;
    static readonly APIGATEWAY: InterfaceVpcEndpointAwsService;
    static readonly CODECOMMIT_GIT: InterfaceVpcEndpointAwsService;
    static readonly CODECOMMIT_GIT_FIPS: InterfaceVpcEndpointAwsService;
    static readonly KINESIS_STREAMS: InterfaceVpcEndpointAwsService;
    static readonly KMS: InterfaceVpcEndpointAwsService;
    static readonly CLOUDWATCH_LOGS: InterfaceVpcEndpointAwsService;
    static readonly CLOUDWATCH: InterfaceVpcEndpointAwsService;
    static readonly SAGEMAKER_API: InterfaceVpcEndpointAwsService;
    static readonly SAGEMAKER_RUNTIME: InterfaceVpcEndpointAwsService;
    static readonly SAGEMAKER_RUNTIME_FIPS: InterfaceVpcEndpointAwsService;
    static readonly SECRETS_MANAGER: InterfaceVpcEndpointAwsService;
    static readonly SERVICE_CATALOG: InterfaceVpcEndpointAwsService;
    static readonly SNS: InterfaceVpcEndpointAwsService;
    static readonly SQS: InterfaceVpcEndpointAwsService;
    static readonly SSM: InterfaceVpcEndpointAwsService;
    static readonly SSM_MESSAGES: InterfaceVpcEndpointAwsService;
    static readonly STS: InterfaceVpcEndpointAwsService;
    static readonly TRANSFER: InterfaceVpcEndpointAwsService;
    static readonly STORAGE_GATEWAY: InterfaceVpcEndpointAwsService;
    /**
     * The name of the service.
     */
    readonly name: string;
    /**
     * The port of the service.
     */
    readonly port: number;
    constructor(name: string, prefix?: string, port?: number);
}
/**
 * Options to add an interface endpoint to a VPC.
 */
export interface InterfaceVpcEndpointOptions {
    /**
     * The service to use for this interface VPC endpoint.
     */
    readonly service: IInterfaceVpcEndpointService;
    /**
     * Whether to associate a private hosted zone with the specified VPC. This
     * allows you to make requests to the service using its default DNS hostname.
     *
     * @default true
     */
    readonly privateDnsEnabled?: boolean;
    /**
     * The subnets in which to create an endpoint network interface. At most one
     * per availability zone.
     *
     * @default - private subnets
     */
    readonly subnets?: SubnetSelection;
    /**
     * The security groups to associate with this interface VPC endpoint.
     *
     * @default - a new security group is created
     */
    readonly securityGroups?: ISecurityGroup[];
}
/**
 * Construction properties for an InterfaceVpcEndpoint.
 */
export interface InterfaceVpcEndpointProps extends InterfaceVpcEndpointOptions {
    /**
     * The VPC network in which the interface endpoint will be used.
     */
    readonly vpc: IVpc;
}
/**
 * An interface VPC endpoint.
 */
export interface IInterfaceVpcEndpoint extends IVpcEndpoint, IConnectable {
}
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
export declare class InterfaceVpcEndpoint extends VpcEndpoint implements IInterfaceVpcEndpoint {
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope: Construct, id: string, attrs: InterfaceVpcEndpointAttributes): IInterfaceVpcEndpoint;
    /**
     * The interface VPC endpoint identifier.
     */
    readonly vpcEndpointId: string;
    /**
     * The date and time the interface VPC endpoint was created.
     * @attribute
     */
    readonly vpcEndpointCreationTimestamp: string;
    /**
     * The DNS entries for the interface VPC endpoint.
     * @attribute
     */
    readonly vpcEndpointDnsEntries: string[];
    /**
     * One or more network interfaces for the interface VPC endpoint.
     * @attribute
     */
    readonly vpcEndpointNetworkInterfaceIds: string[];
    /**
     * The identifier of the first security group associated with this interface
     * VPC endpoint.
     *
     * @deprecated use the `connections` object
     */
    readonly securityGroupId: string;
    /**
     * Access to network connections.
     */
    readonly connections: Connections;
    constructor(scope: Construct, id: string, props: InterfaceVpcEndpointProps);
}
/**
 * Construction properties for an ImportedInterfaceVpcEndpoint.
 */
export interface InterfaceVpcEndpointAttributes {
    /**
     * The interface VPC endpoint identifier.
     */
    readonly vpcEndpointId: string;
    /**
     * The identifier of the security group associated with the interface VPC endpoint.
     *
     * @deprecated use `securityGroups` instead
     */
    readonly securityGroupId?: string;
    /**
     * The security groups associated with the interface VPC endpoint.
     *
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * The port of the service of the interface VPC endpoint.
     */
    readonly port: number;
}
