"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            // tslint:disable-next-line: max-line-length
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(...ssns) {
    const ret = new Set();
    for (const ssn of ssns) {
        for (const subnet of ssn.subnets) {
            if (subnet.routeTable && subnet.routeTable.routeTableId) {
                ret.add(subnet.routeTable.routeTableId);
            }
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
//# sourceMappingURL=data:application/json;base64,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