"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    get uniqueId() {
        return this.node.uniqueId;
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, Object.assign(Object.assign(Object.assign({ groupId: this.securityGroupId }, peer.toIngressRuleConfig()), connection.toRuleJson()), { description }));
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, Object.assign(Object.assign(Object.assign({ groupId: this.securityGroupId }, peer.toEgressRuleConfig()), connection.toRuleJson()), { description }));
        }
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${peer.uniqueId}:${connection}`.replace('/', '_')];
    }
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * This class has an additional optimization over imported security groups that it can also create
 * inline ingress and egress rule (which saves on the total number of resources inside
 * the template).
 */
class SecurityGroup extends SecurityGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.anyValue({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.anyValue({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Import an existing security group into this app.
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                super(...arguments);
                this.securityGroupId = securityGroupId;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                super(...arguments);
                this.securityGroupId = securityGroupId;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule(Object.assign(Object.assign(Object.assign({}, peer.toIngressRuleConfig()), connection.toRuleJson()), { description }));
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = Object.assign(Object.assign(Object.assign({}, peer.toEgressRuleConfig()), connection.toRuleJson()), { description });
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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