"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * Protocol for use in Connection Rules
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["TCP"] = "tcp";
    Protocol["UDP"] = "udp";
    Protocol["ICMP"] = "icmp";
    Protocol["ICMPV6"] = "58";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? `{IndirectPort}` : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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