/**
 * Protocol for use in Connection Rules
 */
export declare enum Protocol {
    ALL = "-1",
    TCP = "tcp",
    UDP = "udp",
    ICMP = "icmp",
    ICMPV6 = "58"
}
/**
 * Properties to create a port range
 */
export interface PortProps {
    /**
     * The protocol for the range
     */
    readonly protocol: Protocol;
    /**
     * The starting port for the range
     *
     * @default - Not included in the rule
     */
    readonly fromPort?: number;
    /**
     * The ending port for the range
     *
     * @default - Not included in the rule
     */
    readonly toPort?: number;
    /**
     * String representation for this object
     */
    readonly stringRepresentation: string;
}
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
export declare class Port {
    private readonly props;
    /**
     * A single TCP port
     */
    static tcp(port: number): Port;
    /**
     * A TCP port range
     */
    static tcpRange(startPort: number, endPort: number): Port;
    /**
     * Any TCP traffic
     */
    static allTcp(): Port;
    /**
     * A single UDP port
     */
    static udp(port: number): Port;
    /**
     * A UDP port range
     */
    static udpRange(startPort: number, endPort: number): Port;
    /**
     * Any UDP traffic
     */
    static allUdp(): Port;
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type: number, code: number): Port;
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type: number): Port;
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing(): Port;
    /**
     * All ICMP traffic
     */
    static allIcmp(): Port;
    /**
     * All traffic
     */
    static allTraffic(): Port;
    /**
     * Whether the rule containing this port range can be inlined into a securitygroup or not.
     */
    readonly canInlineRule: boolean;
    constructor(props: PortProps);
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson(): any;
    toString(): string;
}
