"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway() {
        return new NatGateway();
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        return new NatInstance(props);
    }
}
exports.NatProvider = NatProvider;
class NatGateway extends NatProvider {
    configureNat(options) {
        // Create the NAT gateways
        const gatewayIds = new PrefSet();
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            gatewayIds.add(sub.availabilityZone, gateway.ref);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            sub.addRoute('DefaultRoute', {
                routerType: vpc_1.RouterType.NAT_GATEWAY,
                routerId: gatewayIds.pick(sub.availabilityZone),
                enablesInternetConnectivity: true,
            });
        }
    }
}
class NatInstance extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
    }
    configureNat(options) {
        // Create the NAT instances. They can share a security group and a Role.
        const instances = new PrefSet();
        const machineImage = this.props.machineImage || new NatInstanceImage();
        const sg = new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
        });
        sg.connections.allowFromAnyIpv4(port_1.Port.allTcp());
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: sg,
                role,
                keyName: this.props.keyName
            });
            // NAT instance routes all traffic, both ways
            instances.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            sub.addRoute('DefaultRoute', {
                routerType: vpc_1.RouterType.INSTANCE,
                routerId: instances.pick(sub.availabilityZone).instanceId,
                enablesInternetConnectivity: true,
            });
        }
    }
}
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push(value);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length];
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
//# sourceMappingURL=data:application/json;base64,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