"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
/**
 * Select the latest version of the indicated Windows version
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * Selects the latest version of Amazon Linux
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            'x86_64',
            this.storage
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        return {
            imageId: value,
            osType: this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX,
            userData: this.props.userData
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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