"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const peer_1 = require("./peer");
/**
 * Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 */
class Connections {
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * Add a security group to the list of security groups managed by this object
     */
    addSecurityGroup(...securityGroups) {
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * Allow connections to the peer on the given port
     */
    allowTo(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow connections from the peer on the given port
     */
    allowFrom(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow hosts inside the security group to connect to each other on the given port
     */
    allowInternally(portRange, description) {
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * Allow to all IPv4 ranges
     */
    allowToAnyIpv4(portRange, description) {
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow from any IPv4 ranges
     */
    allowFromAnyIpv4(portRange, description) {
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortFrom(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * Allow hosts inside the security group to connect to each other
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * Allow default connections from all IPv4 ranges
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * Allow connections to the security group on their default port
     */
    allowToDefaultPort(other, description) {
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call alloToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortTo(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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