"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericSyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
class TextractGenericSyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 300 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 300 : props.lambdaMemory;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        this.textractSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_sync/')),
            memorySize: lambdaMemory,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractSyncCallTask = new tasks.LambdaInvoke(this, 'TextractSyncCallTask', { lambdaFunction: this.textractSyncCallFunction });
        textractSyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Analyze*',
                'textract:Detect*',
            ],
            resources: ['*'],
        }));
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        this.textractSyncLambdaLogGroup = this.textractSyncCallFunction.logGroup;
        const workflow_chain = sfn.Chain.start(textractSyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure', 'states:SendTaskSuccess',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractConstructGenericSync';
        // OPERATIONAL
        const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Duration',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                .whereString('message', '=', 'textract_sync_generic_call_duration_in_ms:'),
            metricValue: '$durationMs',
        });
        const asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
        const syncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_processed:'),
            metricValue: '$pages',
        });
        const syncNumberPagesMetric = syncNumberPagesMetricFilter.metric({ statistic: 'sum' });
        const syncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_send_to_process:'),
            metricValue: '$pages',
        });
        const syncNumberPagesSendMetric = syncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
        // TODO: expose all filters as properties of Construct, so users can build their own metrics
        const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsFinished',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_processed:'),
            metricValue: '1',
        });
        const asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
        const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsStarted',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                .whereString('message', '=', 'textract_async_generic_job_started'),
            metricValue: '1',
        });
        const asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
        const syncTimedOutMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-TimedOutFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'DurationSeconds',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message1', 'durationS', 'message2')
                .whereString('message1', '=', 'Task timed out after').whereString('message2', '=', 'seconds'),
            metricValue: '$durationS',
        });
        const syncTimedOutMetric = syncTimedOutMetricFilter.metric({ statistic: 'sum' });
        // OPERATIONAL STOP
        // CALCUATED OPERATIONAL METRICS
        const pagesPerSecond = new cloudwatch.MathExpression({
            expression: 'pages / (duration / 1000)',
            usingMetrics: {
                pages: syncNumberPagesMetric,
                duration: asyncDurationMetric,
            },
        });
        const openJobs = new cloudwatch.MathExpression({
            expression: 'startedJobs - finishedJobs',
            usingMetrics: {
                startedJobs: asyncJobStartedMetric,
                finishedJobs: asyncJobFinshedMetric,
            },
        });
        // CALCUATED OPERATIONAL METRICS STOP
        const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Errors',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
            metricValue: '1',
        });
        const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'LimitExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
            metricValue: '1',
        });
        const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ThrottlingException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
            metricValue: '1',
        });
        const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ProvisionedThroughputExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
            metricValue: '1',
        });
        const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'InternalServerError',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
            metricValue: '1',
        });
        const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
        const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
        const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
        const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
        const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
        const textractStartDocumentTextThrottle = new cloudwatch.Metric({
            namespace: 'AWS/Textract',
            metricName: 'ThrottledCount',
            dimensionsMap: {
                Operation: 'StartDocumentTextDetection',
            },
            statistic: 'sum',
        });
        // DASHBOARD LAMBDA
        const dashboardWidth = 24;
        // const widgetStandardHeight=9;
        // DASHBOARD
        new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
            end: 'end',
            periodOverride: cloudwatch.PeriodOverride.AUTO,
            start: 'start',
            widgets: [
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesProcessed', left: [syncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesSendToProcess', left: [syncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [asyncJobFinshedMetric], right: [asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractSyncCallFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Errors', left: [this.textractSyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Invocations', left: [this.textractSyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Throttles', left: [this.textractSyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-TimedOut', left: [syncTimedOutMetric], width: Math.floor(dashboardWidth / 4) }),
                ],
            ],
        });
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericSyncSfnTask = TextractGenericSyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericSyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericSyncSfnTask", version: "0.0.3" };
TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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