c -------------------- rocdrift.for ---------------------------
c
c  ROCSAT-1 Vertical Drift Model
c  Quiet time equatorial F-region vertical plasma drift model
c  derived from ROCSAT-1 observations
c
c  AUTHOR:
c
c    B. G. Fejer
c    Center for Atmospheric and Space Sciences
c    Utah State University
c    Logan, Utah
c    bela.fejer@usu.edu
c
c  Reference:
c   Fejer, B. G., J. W. Jensen, and S.-Y. Su (2008),
c   Quiet time equatorial F region vertical plasma drift model derived
c   from ROCSAT-1 observations, J. Geophys. Res., 113, A05304
c
c  May, 2008
c
c  Includes subroutines vfjmodelroc, vfjmodelrocinit, fjlin22dex,
c  fjlocate, vfjmodelrocstart 
c ----------------------------------------------------------------
c Input arguments:
c        f107 - daily solar flux
c        idoy - day of the year [integer]
c        ttl  - local time [hours]
c        gglon - geographic longitude [degrees]
c
c Output argument:
c        viv - vertical plasma drift [m/s]
c
c How to use it:
c    1) initialize drift arrays (fill in missing values)
c        CALL vfjmodelrocstart()
c    2) initialize data array for solar activity and day of year
c        CALL vfjmodelrocinit( f107, idoy )
c    3) retrieve vertical drift parsing hour and geographic longitude
c        CALL vfjmodelroc(ttl,gglon,viv)
c
c 2020.10 04/30/22 added subroutine arguments to incop. into IRI 
c 2020.10 04/30/22 deleted blanks in main data statement 
c 2020.10 04/30/22 included fjrocdata.h in vfjmodelrocstart  
c
c ----------------------------------------------------------------

	  subroutine vfjmodelroc(fjm,ttl,gglon,jseas,jsfl,viv)
c-------------------------------------------------------------
c  Inputs: ttl - local time [hour]
c          gglon - geographic longitude [degrees]
c  Output: viv - vertical plasma drift [m/s]
c-------------------------------------------------------------c
      dimension glon(25),tl(59),aa(59,25),fjm(59,25,4,11)

      data glon / -180.,-165.,-150.,-135.,-120.,-105., -90., -75., -60.,
     &             -45., -30., -15.,   0.,  15.,  30.,  45.,  60.,  75.,
     &              90., 105., 120., 135., 150., 165., 180.  /
      data tl /0.00,0.50,1.00,1.50,2.00,2.50,3.00,3.50,4.00,
     &    4.50,5.00,5.50,6.00,6.50,7.00,7.50,8.00,8.50,
     &    9.00,  9.50, 10.00, 10.50, 11.00, 11.50, 12.00, 12.50, 13.00,
     &    13.50, 14.00, 14.50, 15.00, 15.50, 16.00, 16.50, 17.00, 17.25,
     &    17.50, 17.75, 18.00, 18.25, 18.50, 18.75, 19.00, 19.25, 19.50,
     &    19.75, 20.00, 20.25, 20.50, 20.75, 21.00, 21.25, 21.50, 21.75,
     &        22.00, 22.50, 23.00, 23.50, 24.00 /

	  if( xgglon.gt.180. )  xgglon= xgglon - 360.
	  xtl= ttl
	  if( xtl.lt.0.  )  xtl= xtl+24.
	  if( xtl.gt.24. )  xtl= xtl-24.
      do 777 j=1,59
        do 777 jj=1,25
777       aa(j,jj)=fjm(j,jj,jseas,jsfl)
	  call fjlin22dex(viv,xtl,xgglon,tl,glon,aa,59,25)
	  return
	  end subroutine vfjmodelroc
c
c
      subroutine vfjmodelrocinit(f107,idoy,jseas,jsfl)
c-------------------------------------------------------------
c  initialize data array for solar activity and day of year
c-------------------------------------------------------------
      real seas(4),sfl(11)
      
      data seas /   59., 120., 243., 304. /
      data sfl /   100., 110., 120., 130., 140.,
     &             150., 160., 170., 180., 190., 200.  /

      doy=idoy
c	  if(0.lt.doy .and. doy.le.seas(1)) then
	  	jseas= 1
      if(doy.gt.seas(1).and.doy.le.seas(2)) then
	    jseas= 2
	  else if(doy.gt.seas(2).and.doy.le.seas(3)) then
	    jseas= 3
	  else if(doy.gt.seas(3).and.doy.le.seas(4)) then
	    jseas= 4
c	  else
c	    jseas= 1
	  endif

345   MSOL=11
	  if(f107.lt.sfl(2)) then
	    jsfl= 1
	  else if(f107.ge.sfl(MSOL)) then
	    jsfl= MSOL
	  else
	    do 10 i= 1,MSOL-1
	      if(sfl(i).le.f107.and.f107.lt.sfl(i+1)) then
	        jsfl=i
	        return
	        endif
10	      continue
	  endif
	return
	end subroutine vfjmodelrocinit
c
c
	subroutine fjlin22dex( z, x, y, xd, yd, d, nx, ny )
c-------------------------------------------------------------
c linear interpolation in two dimensions  z = d(x,y)
c-------------------------------------------------------------
	dimension d(nx,ny), xd(nx), yd(ny)

	call fjlocate( kx, x, xd, nx )
      kx= max( 1, min( kx, nx-1 ) )
      hx= xd(kx+1)-xd(kx)
      ax= (xd(kx+1)-x)/hx
      bx= (x-xd(kx))/hx
	call fjlocate( ky, y, yd, ny )
      ky= max( 1, min( ky, ny-1 ) )
      hy= yd(ky+1)-yd(ky)
      ay= (yd(ky+1)-y)/hy
      by= (y-yd(ky))/hy

      z=      ax*ay*d(kx,ky)   + bx*ay*d(kx+1,ky)
     &	    + ax*by*d(kx,ky+1) + bx*by*d(kx+1,ky+1)
      return
      end subroutine fjlin22dex
c
c
      subroutine fjlocate( kl, x, xa, na )
c-------------------------------------------------------------
c  locates nearest item in array xa to item x
c   (items xa(kl) and xa(kl+1) bracket x if x is in the range of xa)
c-------------------------------------------------------------
      dimension xa(na)

      if( na.le.1 )  then
	kl= 0
	return
	endif

      klo=1
      if( xa(2).gt.xa(1) )  then
	if( x.lt.xa(1) )  then
	  kl= 0
	  return
	  else if( x.ge.xa(na) )  then
	  kl= na
	  return
	  else if( xa(klo).le.x .and. x.lt.xa(klo+1) )  then
	  kl= klo
	  return
	  endif
      klo=1
      khi=na
1     if (khi-klo.gt.1) then
        k=(khi+klo)/2
        if(xa(k).gt.x)  then
          khi=k
        else
          klo=k
        endif
      goto 1
      endif
      kl= klo

      else
c      xa is in descending order
	if( x.gt.xa(1) )  then
	  kl= 0
	  return
	  else if( x.lt.xa(na) )  then
	  kl= na
	  return
	  else if( xa(klo).ge.x .and. x.gt.xa(klo+1) )  then
	  kl= klo
	  return
	  endif
      klo=1
      khi=na
2     if (khi-klo.gt.1) then
        k=(khi+klo)/2
        if(xa(k).ge.x)  then
          klo=k
        else
          khi=k
        endif
      goto 2
      endif
      kl= klo
      endif
      return
      end subroutine fjlocate
c
c
	   subroutine vfjmodelrocstart(vzm)
c-------------------------------------------------------------
c  initialize drift arrays and fill in missing values.
c-------------------------------------------------------------
c
       dimension fjrocvz(59,25,4,11),vzm(59,25,4,11)
       
       data (fjrocvz(it, 1,1, 1), it=1,59) /
     &-10.831,-10.619,-10.484,-10.820,-11.596,-11.695,
     &-12.857,-14.141,-15.304,-16.786,-15.644,-9.367,
     &0.228,7.737,13.589,17.890,22.218,26.370,
     &30.203,32.486,31.662,29.929,27.615,24.653,
     &21.553,17.056,12.885,9.634,5.922,2.121,
     &-0.119,-0.906,-1.301,-1.790,-2.275,-999.000,
     &-2.775,-999.000,-1.926,-999.000,-0.199,-999.000,
     &-0.536,-999.000,-1.197,-999.000,-3.424,-999.000,
     &-6.975,-999.000,-10.075,-999.000,-11.090,-999.000,
     &-13.300,-14.334,-13.718,-11.925,-10.831
     &/
       data (fjrocvz(it, 2,1, 1), it=1,59) /
     &-12.409,-11.658,-11.520,-11.678,-13.053,-13.824,
     &-14.213,-14.971,-15.591,-16.757,-13.216,-3.747,
     &5.009,11.971,17.585,23.506,28.632,31.844,
     &35.431,37.582,37.012,35.135,31.940,28.276,
     &24.128,18.811,13.842,9.841,5.826,1.719,
     &-1.609,-3.774,-5.392,-5.645,-5.949,-999.000,
     &-5.342,-999.000,-4.343,-999.000,-5.762,-999.000,
     &-3.662,-999.000,-4.637,-999.000,-6.687,-999.000,
     &-9.918,-999.000,-12.942,-999.000,-13.920,-999.000,
     &-14.933,-15.564,-15.293,-13.928,-12.409
     &/
       data (fjrocvz(it, 3,1, 1), it=1,59) /
     &-13.449,-13.025,-13.000,-13.496,-14.567,-14.323,
     &-13.414,-13.438,-13.170,-12.753,-7.299,1.002,
     &7.736,13.336,18.675,23.982,27.467,30.223,
     &33.712,36.533,36.295,34.004,30.405,26.946,
     &23.902,20.010,15.910,11.732,7.658,3.111,
     &-0.516,-2.652,-4.466,-4.298,-4.411,-999.000,
     &-3.621,-999.000,-3.428,-999.000,-5.164,-2.321,
     &-0.009,-0.596,-3.187,-3.683,-6.241,-999.000,
     &-8.565,-999.000,-10.969,-999.000,-11.444,-999.000,
     &-14.262,-15.492,-15.968,-14.728,-13.449
     &/
       data (fjrocvz(it, 4,1, 1), it=1,59) /
     &-13.211,-14.023,-14.128,-13.333,-12.502,-12.319,
     &-12.194,-12.321,-10.931,-8.681,-1.877,5.975,
     &10.391,14.186,19.374,23.470,26.611,29.320,
     &32.730,34.777,33.472,30.234,26.440,23.596,
     &21.038,17.080,13.128,10.136,7.944,4.712,
     &1.483,-1.189,-3.225,-2.990,-1.638,-999.000,
     &-1.497,-999.000,-0.823,-999.000,-3.317,-0.222,
     &2.294,0.352,-3.242,-999.000,-4.554,-999.000,
     &-5.239,-999.000,-7.820,-999.000,-11.009,-999.000,
     &-13.830,-14.871,-14.707,-13.109,-13.211
     &/
       data (fjrocvz(it, 5,1, 1), it=1,59) /
     &-9.870,-10.412,-10.391,-9.301,-8.487,-9.183,
     &-10.865,-11.847,-10.566,-8.452,-1.315,7.139,
     &10.722,13.620,18.479,22.835,25.482,28.191,
     &31.473,33.594,32.896,30.060,27.095,24.390,
     &21.218,16.489,12.310,9.934,8.535,6.530,
     &4.235,1.274,-1.261,-1.959,-1.214,-999.000,
     &-0.492,-999.000,1.129,-999.000,0.000,3.050,
     &3.134,-1.056,-2.351,-999.000,-3.036,-999.000,
     &-3.575,-999.000,-6.553,-999.000,-11.207,-999.000,
     &-13.165,-13.418,-12.396,-10.236,-9.870
     &/
       data (fjrocvz(it, 6,1, 1), it=1,59) /
     &-8.675,-8.968,-9.477,-10.010,-9.978,-10.570,
     &-12.332,-13.492,-12.505,-10.088,-4.158,3.069,
     &6.113,8.160,11.507,15.046,17.554,21.355,
     &25.735,29.286,29.803,28.294,26.302,24.149,
     &21.368,17.539,14.235,11.587,9.822,8.385,
     &7.363,5.576,2.344,0.341,0.283,-999.000,
     &0.845,-999.000,2.167,-999.000,2.940,-999.000,
     &1.038,-999.000,-0.210,-999.000,0.022,-999.000,
     &-0.737,-999.000,-3.257,-999.000,-7.418,-999.000,
     &-10.458,-12.229,-12.036,-9.688,-8.675
     &/
       data (fjrocvz(it, 7,1, 1), it=1,59) /
     &-12.587,-12.634,-13.334,-13.913,-13.843,-13.861,
     &-14.017,-13.804,-12.646,-11.791,-8.691,-5.622,
     &-3.253,0.907,5.589,8.953,11.886,15.687,
     &20.833,24.500,25.687,24.632,22.581,20.597,
     &18.841,16.160,13.052,10.231,8.269,7.426,
     &7.727,6.982,4.475,2.051,1.127,-999.000,
     &0.474,-999.000,1.787,-999.000,4.147,5.796,
     &9.435,9.206,4.462,-999.000,3.015,-999.000,
     &1.200,-999.000,-1.632,-999.000,-6.110,-999.000,
     &-9.402,-12.816,-14.237,-13.434,-12.587
     &/
       data (fjrocvz(it, 8,1, 1), it=1,59) /
     &-16.596,-15.351,-14.757,-14.290,-13.635,-13.500,
     &-13.929,-14.551,-14.881,-15.638,-13.549,-8.987,
     &-5.114,-2.221,0.400,2.904,5.672,10.187,
     &15.810,20.194,22.373,21.848,19.871,17.780,
     &15.779,13.232,10.673,8.111,6.558,5.850,
     &6.876,7.106,5.827,3.608,2.517,-999.000,
     &1.964,-999.000,3.958,-999.000,9.219,15.955,
     &18.853,19.863,15.280,8.412,6.167,-999.000,
     &-1.718,-999.000,-6.805,-999.000,-10.976,-999.000,
     &-13.821,-17.326,-18.896,-17.828,-16.596
     &/
       data (fjrocvz(it, 9,1, 1), it=1,59) /
     &-17.685,-15.865,-14.682,-13.760,-12.701,-12.940,
     &-14.100,-15.359,-16.012,-16.896,-15.245,-11.669,
     &-8.462,-4.477,-0.073,2.943,5.080,8.582,
     &13.685,18.233,20.608,20.948,20.047,18.667,
     &16.576,14.557,12.926,11.311,9.756,8.286,
     &8.274,7.895,6.593,4.852,4.390,3.192,
     &2.380,1.755,4.195,8.769,13.894,22.659,
     &28.896,26.623,15.203,3.457,-3.333,-8.053,
     &-9.491,-999.000,-13.846,-999.000,-16.879,-999.000,
     &-18.875,-21.098,-21.438,-19.624,-17.685
     &/
       data (fjrocvz(it,10,1, 1), it=1,59) /
     &-15.924,-14.203,-13.088,-12.216,-12.088,-13.009,
     &-14.031,-14.750,-15.725,-18.357,-17.710,-13.508,
     &-8.343,-2.038,3.045,6.810,9.534,12.205,
     &15.901,19.299,21.906,23.633,24.165,23.406,
     &21.285,19.182,17.624,16.204,14.276,12.279,
     &11.255,10.282,8.570,6.378,4.811,-999.000,
     &2.097,1.807,7.921,16.631,26.322,33.790,
     &29.296,19.508,7.509,-5.545,-9.610,-12.661,
     &-14.487,-999.000,-17.388,-999.000,-20.119,-999.000,
     &-21.422,-21.616,-20.624,-18.400,-15.924
     &/
       data (fjrocvz(it,11,1, 1), it=1,59) /
     &-15.626,-14.251,-13.803,-14.198,-15.484,-16.612,
     &-17.734,-18.361,-18.779,-20.137,-18.346,-11.669,
     &-4.691,2.189,7.778,11.728,14.100,15.378,
     &17.952,21.883,25.739,28.153,28.624,27.260,
     &25.129,22.753,20.893,18.476,15.813,13.516,
     &11.621,10.171,8.703,7.297,6.762,5.056,
     &4.296,6.554,11.109,18.694,23.543,25.568,
     &15.309,4.070,-3.008,-11.473,-16.481,-999.000,
     &-17.547,-999.000,-19.384,-999.000,-21.004,-999.000,
     &-21.428,-20.984,-19.727,-17.564,-15.626
     &/
       data (fjrocvz(it,12,1, 1), it=1,59) /
     &-18.226,-17.692,-17.977,-19.283,-20.601,-20.059,
     &-19.900,-20.202,-20.341,-20.677,-18.878,-13.984,
     &-6.284,1.173,10.059,14.244,16.383,17.976,
     &20.433,24.163,26.917,28.639,29.086,28.475,
     &27.087,24.619,21.813,18.508,14.818,11.876,
     &10.037,8.541,7.433,7.451,9.165,-999.000,
     &9.296,7.720,10.086,15.856,22.955,22.638,
     &13.893,6.503,0.516,-999.000,-12.215,-999.000,
     &-19.759,-999.000,-22.054,-999.000,-22.474,-999.000,
     &-21.929,-21.301,-20.580,-19.274,-18.226
     &/
       data (fjrocvz(it,13,1, 1), it=1,59) /
     &-20.195,-20.775,-21.183,-22.070,-22.587,-20.761,
     &-20.188,-20.820,-20.852,-21.017,-19.413,-14.597,
     &-4.810,1.448,8.695,14.060,16.451,18.274,
     &20.839,23.996,26.047,27.089,27.010,26.110,
     &24.191,21.531,19.229,16.122,11.630,8.798,
     &7.995,7.542,6.805,6.418,6.451,-999.000,
     &6.903,-999.000,8.042,12.264,18.651,16.793,
     &11.687,-999.000,3.844,-999.000,-8.612,-999.000,
     &-17.955,-999.000,-22.157,-999.000,-22.858,-999.000,
     &-22.328,-21.462,-20.484,-19.929,-20.195
     &/
       data (fjrocvz(it,14,1, 1), it=1,59) /
     &-20.062,-21.264,-21.926,-22.048,-21.446,-19.556,
     &-20.256,-21.670,-21.410,-20.854,-17.437,-13.226,
     &-5.958,-1.623,5.636,11.584,14.086,15.854,
     &18.029,21.014,23.684,24.789,24.341,22.711,
     &20.724,18.517,16.338,12.892,8.704,5.915,
     &5.233,4.934,3.682,1.780,1.276,-999.000,
     &2.439,1.730,5.995,8.546,11.194,12.553,
     &11.393,6.101,3.418,-999.000,-5.744,-999.000,
     &-14.957,-999.000,-20.864,-999.000,-22.656,-999.000,
     &-22.419,-21.440,-19.883,-19.451,-20.062
     &/
       data (fjrocvz(it,15,1, 1), it=1,59) /
     &-20.230,-20.950,-22.117,-22.753,-22.231,-20.219,
     &-20.431,-21.981,-23.491,-24.912,-22.167,-16.348,
     &-7.974,-3.446,2.333,7.679,10.782,13.245,
     &15.388,16.981,18.221,18.810,17.956,16.291,
     &14.734,13.235,11.705,9.603,6.268,3.274,
     &2.240,1.704,0.266,-0.234,0.143,-999.000,
     &1.603,-999.000,5.523,9.312,12.732,15.094,
     &14.266,8.570,5.017,-999.000,-3.198,-999.000,
     &-12.123,-999.000,-19.546,-999.000,-22.302,-999.000,
     &-22.301,-21.572,-20.610,-20.260,-20.230
     &/
       data (fjrocvz(it,16,1, 1), it=1,59) /
     &-22.344,-22.290,-22.862,-23.570,-23.089,-20.898,
     &-20.559,-21.235,-22.899,-28.444,-29.528,-24.280,
     &-9.581,-2.109,2.063,6.463,9.866,11.877,
     &13.998,15.719,17.069,17.019,16.054,14.463,
     &12.795,11.066,9.378,6.935,3.478,1.127,
     &0.793,1.088,0.925,1.270,2.207,-999.000,
     &4.482,-999.000,9.467,12.936,14.572,14.819,
     &13.003,-999.000,4.500,-999.000,-2.459,-999.000,
     &-11.751,-999.000,-19.252,-999.000,-22.354,-999.000,
     &-23.154,-22.843,-22.533,-22.506,-22.344
     &/
       data (fjrocvz(it,17,1, 1), it=1,59) /
     &-23.821,-23.684,-23.894,-24.662,-25.290,-23.844,
     &-23.477,-21.811,-20.507,-25.258,-28.299,-23.874,
     &-9.071,2.341,5.905,8.747,11.587,13.254,
     &15.252,17.072,17.987,18.453,18.085,16.656,
     &14.822,12.616,10.586,7.912,4.139,1.722,
     &1.260,1.743,1.491,1.153,2.348,-999.000,
     &3.667,6.138,8.425,10.554,11.383,13.262,
     &10.917,-999.000,-1.364,-999.000,-8.990,-999.000,
     &-17.336,-999.000,-23.249,-999.000,-25.736,-999.000,
     &-25.474,-24.101,-23.335,-23.591,-23.821
     &/
       data (fjrocvz(it,18,1, 1), it=1,59) /
     &-24.415,-24.250,-24.455,-25.569,-27.307,-25.950,
     &-25.727,-24.688,-21.638,-21.837,-21.071,-16.259,
     &-3.775,7.188,11.272,13.755,16.501,17.990,
     &19.726,20.241,20.188,20.605,20.105,18.359,
     &15.759,13.165,10.773,8.423,5.195,3.181,
     &2.255,2.298,2.371,2.776,3.626,-999.000,
     &4.989,-999.000,7.208,10.773,12.614,12.038,
     &7.005,-0.564,-6.597,-999.000,-14.675,-999.000,
     &-22.095,-999.000,-27.571,-999.000,-29.008,-999.000,
     &-27.426,-25.236,-24.121,-24.299,-24.415
     &/
       data (fjrocvz(it,19,1, 1), it=1,59) /
     &-25.157,-24.132,-23.263,-23.520,-24.145,-21.329,
     &-19.596,-18.856,-16.650,-17.344,-16.507,-12.075,
     &-0.901,9.423,15.130,18.650,21.037,22.760,
     &24.600,25.102,24.793,24.643,24.254,22.233,
     &19.508,16.955,13.800,9.732,5.884,4.803,
     &4.591,4.360,4.111,3.422,3.511,-999.000,
     &4.797,-999.000,6.554,9.590,8.634,5.353,
     &2.217,-999.000,-7.408,-999.000,-15.393,-999.000,
     &-23.001,-999.000,-27.908,-999.000,-29.108,-999.000,
     &-27.628,-25.714,-24.657,-25.050,-25.157
     &/
       data (fjrocvz(it,20,1, 1), it=1,59) /
     &-23.987,-23.028,-22.716,-22.749,-23.186,-20.447,
     &-18.692,-17.496,-15.690,-15.578,-16.971,-13.745,
     &-3.016,8.587,14.782,17.897,21.334,24.009,
     &26.528,27.607,27.043,26.898,26.580,25.219,
     &23.707,21.999,19.394,15.177,10.075,8.170,
     &7.664,6.958,5.413,4.061,3.868,-999.000,
     &3.076,-999.000,2.391,5.169,8.906,6.378,
     &-0.283,-999.000,-8.468,-999.000,-17.174,-999.000,
     &-24.527,-999.000,-29.322,-999.000,-30.739,-999.000,
     &-30.071,-28.325,-26.461,-25.182,-23.987
     &/
       data (fjrocvz(it,21,1, 1), it=1,59) /
     &-25.411,-25.017,-25.152,-25.810,-26.634,-23.673,
     &-21.069,-20.180,-19.972,-21.379,-23.366,-20.494,
     &-8.890,4.054,10.988,14.818,19.226,21.216,
     &23.948,25.934,26.022,25.634,24.887,24.186,
     &23.586,22.132,19.647,15.353,9.939,7.601,
     &7.039,6.522,5.146,3.382,2.346,-999.000,
     &3.069,-999.000,4.421,6.945,8.711,5.027,
     &-2.584,-999.000,-11.312,-999.000,-18.076,-999.000,
     &-24.350,-999.000,-29.176,-999.000,-30.939,-999.000,
     &-30.878,-29.991,-28.686,-26.967,-25.411
     &/
       data (fjrocvz(it,22,1, 1), it=1,59) /
     &-23.650,-23.625,-24.140,-25.456,-26.708,-24.738,
     &-22.823,-22.599,-23.034,-24.095,-25.375,-22.576,
     &-11.470,1.315,8.586,12.701,16.627,19.207,
     &22.264,25.091,25.092,24.561,24.523,24.299,
     &23.237,21.117,17.695,12.515,6.879,4.315,
     &3.947,3.877,2.725,1.423,1.170,-999.000,
     &1.741,-999.000,1.251,-999.000,2.555,-999.000,
     &-0.908,-999.000,-11.021,-999.000,-15.768,-999.000,
     &-20.857,-999.000,-25.436,-999.000,-27.663,-999.000,
     &-27.864,-26.893,-25.393,-24.358,-23.650
     &/
       data (fjrocvz(it,23,1, 1), it=1,59) /
     &-17.272,-17.797,-18.326,-19.304,-20.801,-19.713,
     &-19.094,-19.319,-19.040,-19.132,-18.856,-14.886,
     &-6.101,2.591,8.493,11.481,14.564,16.561,
     &19.628,22.612,23.748,25.574,26.496,25.717,
     &23.564,20.745,17.751,13.765,8.878,5.738,
     &3.769,2.697,1.794,1.159,1.200,-999.000,
     &2.015,-999.000,5.265,-999.000,6.744,-999.000,
     &4.581,-999.000,0.272,-999.000,-8.774,-999.000,
     &-14.527,-999.000,-19.349,-999.000,-21.494,-999.000,
     &-21.807,-20.858,-19.205,-17.670,-17.272
     &/
       data (fjrocvz(it,24,1, 1), it=1,59) /
     &-12.391,-12.617,-12.822,-13.696,-14.537,-13.080,
     &-12.751,-13.527,-13.635,-15.016,-14.691,-10.064,
     &-1.579,5.374,10.248,13.904,16.586,18.731,
     &21.722,24.493,25.429,25.967,25.339,23.659,
     &21.279,18.444,15.699,12.857,8.719,5.160,
     &2.670,1.812,1.628,1.014,0.294,-999.000,
     &1.018,-999.000,2.762,-999.000,4.933,-999.000,
     &4.486,-999.000,2.137,-999.000,-1.968,-999.000,
     &-7.171,-999.000,-11.518,-999.000,-13.536,-999.000,
     &-14.693,-15.110,-14.235,-12.987,-12.391
     &/
       data (fjrocvz(it,25,1, 1), it=1,59) /
     &-10.831,-10.619,-10.484,-10.820,-11.596,-11.695,
     &-12.857,-14.141,-15.304,-16.786,-15.644,-9.367,
     &0.228,7.737,13.589,17.890,22.218,26.370,
     &30.203,32.486,31.662,29.929,27.615,24.653,
     &21.553,17.056,12.885,9.634,5.922,2.121,
     &-0.119,-0.906,-1.301,-1.790,-2.275,-999.000,
     &-2.775,-999.000,-1.926,-999.000,-0.199,-999.000,
     &-0.536,-999.000,-1.197,-999.000,-3.424,-999.000,
     &-6.975,-999.000,-10.075,-999.000,-11.090,-999.000,
     &-13.300,-14.334,-13.718,-11.925,-10.831
     &/
       data (fjrocvz(it, 1,2, 1), it=1,59) /
     &-13.535,-12.797,-11.568,-11.006,-12.202,-13.208,
     &-14.090,-15.243,-16.093,-15.702,-12.733,-6.409,
     &-7.625,-2.935,2.360,8.520,14.206,23.106,
     &30.108,32.937,32.599,30.712,27.919,23.461,
     &18.659,13.544,7.728,3.354,1.647,2.375,
     &2.742,2.706,2.503,3.419,5.294,-999.000,
     &7.512,-999.000,10.081,10.084,7.779,2.029,
     &-3.415,-999.000,-2.205,-999.000,-6.035,-999.000,
     &-10.103,-999.000,-11.923,-999.000,-15.399,-999.000,
     &-16.544,-16.400,-15.959,-14.886,-13.535
     &/
       data (fjrocvz(it, 2,2, 1), it=1,59) /
     &-11.871,-11.260,-10.687,-10.511,-10.135,-9.384,
     &-9.758,-11.699,-13.099,-12.194,-6.712,1.189,
     &7.001,9.902,14.046,16.587,20.331,25.544,
     &30.949,34.575,35.243,33.624,30.468,25.813,
     &20.852,16.500,12.264,8.094,5.060,4.487,
     &5.295,5.843,6.015,6.749,7.534,-999.000,
     &8.578,-999.000,11.915,11.859,11.336,8.858,
     &3.457,-999.000,-2.119,-999.000,-6.793,-999.000,
     &-7.540,-999.000,-11.258,-999.000,-15.249,-999.000,
     &-15.806,-16.013,-14.917,-12.878,-11.871
     &/
       data (fjrocvz(it, 3,2, 1), it=1,59) /
     &-11.963,-10.776,-10.170,-9.294,-8.535,-8.285,
     &-8.035,-8.343,-8.862,-8.358,-3.789,3.040,
     &9.049,12.054,14.325,17.287,20.680,23.711,
     &26.966,29.480,30.541,28.994,25.624,21.353,
     &17.153,13.745,10.860,8.247,5.659,4.851,
     &5.639,6.467,7.221,7.360,8.315,-999.000,
     &10.376,-999.000,12.268,15.210,16.614,10.692,
     &1.173,-2.207,-2.957,-7.002,-9.046,-8.555,
     &-8.725,-999.000,-14.886,-999.000,-17.306,-999.000,
     &-17.877,-16.912,-15.539,-13.657,-11.963
     &/
       data (fjrocvz(it, 4,2, 1), it=1,59) /
     &-13.235,-12.125,-10.638,-9.280,-9.101,-8.609,
     &-7.762,-6.215,-5.028,-4.436,-1.358,4.347,
     &10.324,12.488,13.598,16.081,19.347,23.018,
     &26.211,27.925,27.056,24.249,20.855,17.023,
     &13.089,9.738,7.313,5.703,4.527,4.283,
     &4.901,6.064,6.845,8.059,10.434,-999.000,
     &11.916,-999.000,14.676,17.840,17.141,10.594,
     &4.110,2.546,-2.734,-10.875,-11.599,-999.000,
     &-10.734,-999.000,-16.247,-999.000,-19.101,-999.000,
     &-18.668,-16.744,-15.554,-14.099,-13.235
     &/
       data (fjrocvz(it, 5,2, 1), it=1,59) /
     &-14.193,-13.080,-10.971,-9.487,-9.615,-9.333,
     &-8.694,-6.980,-5.252,-4.425,-1.454,4.735,
     &10.690,12.985,15.301,17.062,19.998,24.177,
     &27.953,29.909,28.981,26.160,22.244,18.001,
     &13.723,10.720,8.768,7.657,6.401,5.803,
     &6.286,7.486,8.851,9.887,10.810,-999.000,
     &14.044,-999.000,17.536,18.903,19.182,16.982,
     &11.544,5.167,-1.878,-7.527,-10.054,-12.934,
     &-15.548,-20.091,-22.339,-22.112,-20.804,-999.000,
     &-19.364,-17.290,-15.757,-14.693,-14.193
     &/
       data (fjrocvz(it, 6,2, 1), it=1,59) /
     &-15.069,-14.291,-12.916,-11.659,-11.474,-9.999,
     &-8.123,-6.560,-4.494,-2.876,-0.876,0.299,
     &3.294,7.998,13.839,18.191,20.657,24.682,
     &28.491,30.416,30.584,28.736,25.784,21.843,
     &18.149,15.568,13.430,11.536,9.755,8.414,
     &8.114,9.442,11.136,11.160,11.964,-999.000,
     &15.543,-999.000,19.471,22.303,25.530,25.072,
     &21.215,12.874,4.330,-3.832,-8.508,-14.895,
     &-17.924,-999.000,-22.172,-999.000,-22.373,-999.000,
     &-20.880,-17.935,-16.424,-15.817,-15.069
     &/
       data (fjrocvz(it, 7,2, 1), it=1,59) /
     &-16.268,-15.609,-14.506,-13.481,-12.662,-9.838,
     &-7.477,-5.703,-2.941,-2.195,-0.229,1.920,
     &6.752,8.291,11.517,14.203,16.616,21.321,
     &25.955,29.333,29.737,29.187,27.616,25.030,
     &22.154,19.460,17.111,15.037,12.708,10.506,
     &9.701,10.350,11.574,11.719,11.753,-999.000,
     &14.160,-999.000,17.559,22.228,26.826,30.957,
     &26.249,14.347,4.180,-2.539,-8.490,-16.897,
     &-21.792,-23.800,-24.326,-24.238,-23.500,-999.000,
     &-21.848,-19.922,-18.543,-16.946,-16.268
     &/
       data (fjrocvz(it, 8,2, 1), it=1,59) /
     &-15.624,-13.868,-12.759,-12.112,-11.033,-9.425,
     &-8.207,-6.954,-4.487,-6.932,-9.073,-11.132,
     &-6.503,-0.374,4.647,9.535,11.570,14.735,
     &20.331,24.590,26.220,26.372,25.738,24.334,
     &22.389,20.498,18.189,15.963,13.288,10.821,
     &9.725,10.023,11.145,11.253,11.132,-999.000,
     &13.588,-999.000,17.235,20.031,24.074,24.703,
     &19.648,13.200,5.214,-2.183,-8.250,-15.212,
     &-19.371,-20.402,-18.937,-999.000,-19.801,-999.000,
     &-19.636,-19.541,-18.665,-17.015,-15.624
     &/
       data (fjrocvz(it, 9,2, 1), it=1,59) /
     &-12.394,-9.876,-7.776,-6.331,-5.832,-6.101,
     &-7.353,-8.482,-8.932,-11.173,-14.147,-16.690,
     &-12.249,-3.315,4.171,8.865,9.912,13.140,
     &17.996,21.416,22.677,23.142,22.760,21.206,
     &18.795,16.482,14.577,12.587,10.248,8.886,
     &8.766,9.296,10.332,10.496,11.014,-999.000,
     &12.427,-999.000,15.875,19.889,22.447,25.644,
     &26.624,21.359,10.853,2.143,-3.103,-999.000,
     &-7.134,-999.000,-10.943,-999.000,-14.109,-999.000,
     &-15.020,-15.360,-15.836,-14.689,-12.394
     &/
       data (fjrocvz(it,10,2, 1), it=1,59) /
     &-11.647,-9.213,-6.797,-5.351,-4.085,-5.139,
     &-8.376,-10.968,-12.050,-13.101,-14.095,-14.448,
     &-11.148,-1.321,6.229,10.859,12.293,16.198,
     &21.238,24.327,24.497,23.636,22.085,20.229,
     &17.700,14.824,12.428,10.294,8.671,8.048,
     &8.069,8.356,8.239,8.164,9.278,-999.000,
     &10.680,-999.000,16.786,21.594,26.466,29.411,
     &29.451,20.808,12.217,-999.000,5.784,-999.000,
     &-0.413,-999.000,-6.028,-999.000,-7.694,-999.000,
     &-9.296,-10.871,-12.460,-12.390,-11.647
     &/
       data (fjrocvz(it,11,2, 1), it=1,59) /
     &-13.664,-12.981,-12.664,-12.406,-11.789,-12.555,
     &-13.457,-15.068,-16.654,-18.408,-17.835,-14.082,
     &-9.233,-0.085,7.046,12.657,16.437,19.282,
     &23.426,26.836,27.765,27.084,25.759,23.715,
     &21.339,18.441,15.188,12.776,10.589,9.567,
     &8.872,8.052,7.443,7.407,9.138,-999.000,
     &12.390,-999.000,17.477,22.594,29.306,29.488,
     &24.510,17.105,9.729,-999.000,1.318,-999.000,
     &-5.182,-999.000,-8.410,-999.000,-9.174,-999.000,
     &-11.640,-12.019,-13.487,-14.472,-13.664
     &/
       data (fjrocvz(it,12,2, 1), it=1,59) /
     &-16.408,-17.342,-17.322,-17.208,-17.470,-17.151,
     &-16.886,-18.608,-21.464,-22.570,-22.767,-19.010,
     &-14.485,-5.509,3.958,9.609,13.999,18.330,
     &24.641,29.207,30.541,29.720,28.256,26.629,
     &24.380,20.901,17.328,14.376,12.122,11.286,
     &10.293,8.996,7.993,7.788,9.742,-999.000,
     &13.124,-999.000,18.410,23.432,29.971,32.142,
     &27.192,14.981,6.229,0.205,-4.576,-999.000,
     &-6.999,-999.000,-8.440,-999.000,-12.298,-999.000,
     &-14.600,-15.057,-15.791,-16.243,-16.408
     &/
       data (fjrocvz(it,13,2, 1), it=1,59) /
     &-17.503,-17.150,-16.994,-17.075,-16.814,-17.514,
     &-19.048,-20.371,-22.039,-21.517,-20.706,-18.306,
     &-14.306,-6.369,2.260,8.809,13.383,18.295,
     &27.063,32.394,32.580,31.323,29.900,28.331,
     &25.728,21.832,18.306,15.728,14.095,13.657,
     &12.828,11.440,9.860,9.446,10.987,-999.000,
     &13.484,-999.000,21.005,25.070,31.310,34.894,
     &27.999,15.543,2.911,-2.816,-4.039,-5.596,
     &-6.249,-999.000,-9.211,-999.000,-12.709,-999.000,
     &-15.447,-17.802,-18.042,-17.502,-17.503
     &/
       data (fjrocvz(it,14,2, 1), it=1,59) /
     &-16.639,-16.262,-15.959,-15.406,-15.413,-17.013,
     &-19.724,-23.301,-26.586,-25.195,-21.465,-16.563,
     &-11.358,-8.868,-2.224,5.335,10.719,16.455,
     &25.406,30.663,31.664,30.689,29.126,27.263,
     &24.271,21.042,18.299,16.396,15.152,14.512,
     &13.334,11.874,10.169,9.829,11.633,-999.000,
     &15.113,-999.000,22.395,26.310,28.823,29.375,
     &23.838,12.962,6.532,-999.000,-0.617,-999.000,
     &-8.811,-999.000,-12.793,-999.000,-15.894,-999.000,
     &-19.025,-19.943,-19.181,-17.601,-16.639
     &/
       data (fjrocvz(it,15,2, 1), it=1,59) /
     &-14.873,-14.199,-13.463,-12.909,-12.731,-13.909,
     &-17.571,-23.491,-27.368,-27.977,-24.133,-16.318,
     &-10.681,-7.071,-2.297,4.687,10.048,15.367,
     &22.640,27.486,28.881,27.733,25.560,22.620,
     &19.340,16.653,14.243,12.483,11.158,10.057,
     &9.007,7.852,7.032,7.456,9.807,-999.000,
     &12.588,-999.000,18.384,20.786,21.981,23.553,
     &19.213,9.267,3.846,-999.000,-4.527,-999.000,
     &-11.546,-999.000,-16.582,-999.000,-20.298,-999.000,
     &-22.496,-22.713,-20.318,-16.712,-14.873
     &/
       data (fjrocvz(it,16,2, 1), it=1,59) /
     &-14.997,-13.385,-13.364,-13.311,-13.510,-14.874,
     &-18.203,-22.502,-25.340,-28.120,-26.961,-19.248,
     &-9.674,-6.331,-1.477,4.244,8.748,14.143,
     &21.443,26.471,28.159,27.320,25.526,21.816,
     &17.124,12.849,9.849,7.902,6.661,5.885,
     &4.895,3.627,2.625,3.064,5.046,-999.000,
     &7.455,-999.000,12.538,14.241,15.801,16.790,
     &12.121,2.354,0.622,-999.000,-7.595,-999.000,
     &-15.116,-999.000,-20.082,-999.000,-23.142,-999.000,
     &-24.595,-23.936,-21.284,-17.759,-14.997
     &/
       data (fjrocvz(it,17,2, 1), it=1,59) /
     &-17.622,-16.523,-15.598,-14.916,-15.205,-15.707,
     &-18.165,-22.073,-24.529,-26.213,-24.507,-16.386,
     &-7.628,-3.692,1.904,6.522,10.008,15.610,
     &23.323,28.468,30.663,30.402,28.774,25.397,
     &20.114,14.949,10.739,7.646,6.032,5.025,
     &3.752,2.078,0.151,-0.561,1.191,-999.000,
     &4.404,-999.000,10.239,11.127,12.811,10.685,
     &7.059,0.519,-4.929,-11.500,-16.999,-20.462,
     &-21.983,-999.000,-22.334,-999.000,-24.130,-999.000,
     &-24.521,-22.971,-21.068,-19.211,-17.622
     &/
       data (fjrocvz(it,18,2, 1), it=1,59) /
     &-18.478,-16.669,-15.274,-14.748,-14.333,-13.939,
     &-16.873,-20.782,-22.203,-23.840,-21.328,-14.050,
     &-5.216,-1.161,2.848,7.551,11.498,18.348,
     &25.303,30.732,34.037,33.823,31.128,26.695,
     &21.935,17.451,12.894,8.819,6.277,4.529,
     &3.051,1.272,-0.335,-0.249,2.076,-999.000,
     &4.422,6.192,9.424,10.640,9.798,7.381,
     &0.714,-3.247,-9.155,-999.000,-18.221,-999.000,
     &-24.352,-999.000,-26.248,-999.000,-25.968,-999.000,
     &-24.263,-23.194,-21.111,-19.633,-18.478
     &/
       data (fjrocvz(it,19,2, 1), it=1,59) /
     &-17.201,-15.226,-14.875,-14.251,-13.623,-12.719,
     &-14.535,-16.351,-16.821,-18.176,-15.303,-8.099,
     &0.314,2.577,6.511,11.407,15.247,21.431,
     &28.950,34.636,37.087,36.641,33.610,29.127,
     &23.992,18.914,14.237,10.082,7.867,6.737,
     &5.119,3.146,1.518,1.237,3.250,-999.000,
     &6.444,-999.000,11.180,10.136,8.412,1.633,
     &-0.941,-5.380,-11.045,-19.278,-23.530,-26.552,
     &-26.994,-999.000,-27.004,-999.000,-25.297,-999.000,
     &-23.189,-22.299,-20.548,-18.972,-17.201
     &/
       data (fjrocvz(it,20,2, 1), it=1,59) /
     &-19.803,-17.755,-15.825,-13.879,-12.291,-11.036,
     &-12.368,-14.651,-14.664,-15.071,-12.377,-3.004,
     &5.329,5.649,9.242,12.869,16.138,22.537,
     &29.769,36.233,38.936,38.887,36.513,32.598,
     &27.933,23.081,18.171,13.962,11.157,9.372,
     &7.453,5.316,3.329,3.180,5.656,-999.000,
     &8.421,-999.000,11.671,10.369,11.034,8.845,
     &3.677,-4.698,-9.782,-15.049,-19.001,-23.546,
     &-25.884,-999.000,-25.564,-999.000,-25.524,-999.000,
     &-24.221,-23.574,-22.457,-21.160,-19.803
     &/
       data (fjrocvz(it,21,2, 1), it=1,59) /
     &-20.408,-17.910,-15.976,-14.741,-14.011,-12.402,
     &-12.822,-14.780,-15.982,-16.183,-13.142,-5.536,
     &0.391,3.862,7.988,12.561,16.998,22.112,
     &27.494,33.673,36.252,36.741,34.496,30.397,
     &25.970,21.625,17.288,12.514,8.843,6.588,
     &4.653,3.239,2.147,1.782,3.208,-999.000,
     &6.235,-999.000,11.409,12.834,14.264,13.219,
     &4.479,-999.000,-4.880,-999.000,-15.583,-999.000,
     &-22.108,-999.000,-23.985,-999.000,-24.259,-999.000,
     &-23.548,-23.143,-22.486,-21.713,-20.408
     &/
       data (fjrocvz(it,22,2, 1), it=1,59) /
     &-19.286,-18.524,-17.535,-16.898,-17.283,-16.554,
     &-16.527,-18.459,-21.661,-22.366,-19.254,-12.255,
     &-3.935,-1.113,4.187,8.865,13.356,18.709,
     &25.138,30.800,32.921,32.329,29.753,25.242,
     &19.708,14.511,10.371,6.696,3.556,1.705,
     &0.661,0.767,1.018,0.698,1.812,-999.000,
     &6.581,-999.000,11.395,9.333,10.167,9.755,
     &4.296,-999.000,-7.221,-999.000,-16.190,-999.000,
     &-22.946,-999.000,-23.281,-999.000,-22.866,-999.000,
     &-22.639,-22.002,-20.940,-20.110,-19.286
     &/
       data (fjrocvz(it,23,2, 1), it=1,59) /
     &-16.775,-16.799,-15.578,-13.945,-14.101,-15.181,
     &-17.026,-19.316,-21.584,-22.553,-20.118,-14.026,
     &-8.926,-5.770,-0.250,5.057,10.367,15.853,
     &24.003,29.633,31.287,29.551,26.509,21.917,
     &16.428,10.659,6.446,3.228,0.723,-0.487,
     &-1.151,-1.457,-0.636,0.450,2.434,-999.000,
     &4.398,-999.000,8.293,8.752,8.021,7.821,
     &5.398,-1.150,-6.133,-999.000,-14.241,-999.000,
     &-20.214,-999.000,-22.097,-999.000,-21.927,-999.000,
     &-21.556,-20.443,-18.409,-16.973,-16.775
     &/
       data (fjrocvz(it,24,2, 1), it=1,59) /
     &-14.366,-13.418,-12.011,-10.492,-11.334,-13.602,
     &-15.811,-17.276,-18.854,-19.895,-18.304,-13.599,
     &-7.312,-1.688,1.942,5.423,10.419,18.576,
     &26.759,30.377,30.975,29.142,25.725,20.889,
     &15.769,10.089,5.314,1.874,0.237,0.541,
     &0.539,-0.090,0.399,1.439,3.006,-999.000,
     &4.873,-999.000,8.522,8.058,7.932,6.141,
     &1.520,-999.000,-1.657,-999.000,-7.303,-999.000,
     &-12.344,-999.000,-15.961,-999.000,-16.998,-999.000,
     &-18.043,-17.617,-16.153,-15.293,-14.366
     &/
       data (fjrocvz(it,25,2, 1), it=1,59) /
     &-13.535,-12.797,-11.568,-11.006,-12.202,-13.208,
     &-14.090,-15.243,-16.093,-15.702,-12.733,-6.409,
     &-7.625,-2.935,2.360,8.520,14.206,23.106,
     &30.108,32.937,32.599,30.712,27.919,23.461,
     &18.659,13.544,7.728,3.354,1.647,2.375,
     &2.742,2.706,2.503,3.419,5.294,-999.000,
     &7.512,-999.000,10.081,10.084,7.779,2.029,
     &-3.415,-999.000,-2.205,-999.000,-6.035,-999.000,
     &-10.103,-999.000,-11.923,-999.000,-15.399,-999.000,
     &-16.544,-16.400,-15.959,-14.886,-13.535
     &/
       data (fjrocvz(it, 1,3, 1), it=1,59) /
     &-22.055,-23.644,-23.612,-22.829,-21.756,-19.871,
     &-19.486,-22.095,-27.665,-30.245,-23.834,-10.763,
     &-0.258,3.948,6.717,8.858,8.964,7.383,
     &7.413,9.728,12.455,13.655,13.160,10.756,
     &7.633,6.526,7.334,8.703,9.121,7.950,
     &7.200,7.012,6.616,5.316,4.992,-999.000,
     &4.142,-999.000,4.303,-999.000,8.658,11.280,
     &11.091,9.415,2.257,-999.000,0.086,-999.000,
     &-6.257,-999.000,-9.348,-999.000,-12.346,-999.000,
     &-16.083,-18.841,-20.614,-21.132,-22.055
     &/
       data (fjrocvz(it, 2,3, 1), it=1,59) /
     &-19.660,-20.311,-19.919,-18.837,-17.739,-16.826,
     &-17.879,-20.648,-24.269,-26.249,-18.634,-3.594,
     &8.627,11.476,12.484,13.267,12.947,12.000,
     &12.464,13.651,14.338,14.864,13.864,11.795,
     &9.969,9.610,10.384,11.371,12.302,11.942,
     &11.265,11.013,10.857,10.006,9.095,-999.000,
     &6.964,-999.000,7.934,-999.000,10.560,13.558,
     &12.759,10.789,3.272,-999.000,-0.013,-999.000,
     &-6.385,-999.000,-10.971,-999.000,-14.239,-999.000,
     &-17.417,-19.740,-20.541,-19.701,-19.660
     &/
       data (fjrocvz(it, 3,3, 1), it=1,59) /
     &-18.446,-17.211,-16.611,-15.831,-15.196,-14.465,
     &-15.244,-16.683,-19.651,-24.280,-17.662,0.961,
     &17.379,18.680,17.029,15.506,13.727,12.778,
     &13.256,13.475,12.052,10.517,8.837,7.802,
     &7.128,7.182,8.167,9.432,10.922,12.186,
     &12.197,11.901,11.862,11.103,10.304,-999.000,
     &9.713,-999.000,9.556,-999.000,9.365,10.490,
     &9.181,9.817,4.743,-999.000,-2.030,-999.000,
     &-8.724,-999.000,-14.269,-999.000,-17.702,-999.000,
     &-20.859,-22.985,-23.225,-20.782,-18.446
     &/
       data (fjrocvz(it, 4,3, 1), it=1,59) /
     &-20.822,-18.680,-18.096,-18.057,-17.386,-15.789,
     &-14.692,-15.028,-17.826,-21.207,-14.169,5.497,
     &19.684,21.176,19.633,18.459,16.400,13.698,
     &11.724,10.189,8.544,7.367,6.917,7.046,
     &6.632,6.999,7.778,9.165,10.823,11.844,
     &12.335,12.515,12.284,11.120,11.418,-999.000,
     &11.165,-999.000,9.428,-999.000,6.563,9.506,
     &9.636,6.027,3.326,-999.000,-5.074,-999.000,
     &-11.990,-999.000,-17.168,-999.000,-20.970,-999.000,
     &-23.356,-25.096,-25.469,-23.352,-20.822
     &/
       data (fjrocvz(it, 5,3, 1), it=1,59) /
     &-23.676,-22.242,-21.412,-19.636,-17.357,-15.211,
     &-14.050,-14.266,-15.153,-16.073,-10.268,3.856,
     &18.162,22.178,21.678,20.632,18.418,15.923,
     &13.920,12.485,11.823,11.548,11.209,10.168,
     &8.629,7.611,7.237,7.838,8.588,9.308,
     &10.230,10.968,11.065,10.475,10.075,-999.000,
     &9.755,-999.000,9.091,7.444,6.417,2.263,
     &4.660,6.307,0.851,-999.000,-7.019,-999.000,
     &-14.140,-999.000,-19.257,-999.000,-22.853,-999.000,
     &-25.695,-27.514,-27.267,-25.256,-23.676
     &/
       data (fjrocvz(it, 6,3, 1), it=1,59) /
     &-26.406,-26.089,-25.793,-23.403,-19.251,-15.353,
     &-14.148,-14.432,-13.470,-12.048,-7.780,2.398,
     &16.123,20.067,19.535,18.583,17.571,16.522,
     &15.792,16.513,17.181,17.040,15.910,14.347,
     &12.365,10.373,9.021,8.120,8.132,8.398,
     &8.973,9.432,10.003,9.848,9.663,-999.000,
     &10.795,-999.000,9.973,9.555,5.953,3.456,
     &-0.045,-999.000,-3.010,-999.000,-10.191,-999.000,
     &-17.647,-999.000,-22.861,-999.000,-26.441,-999.000,
     &-29.360,-30.778,-29.541,-27.574,-26.406
     &/
       data (fjrocvz(it, 7,3, 1), it=1,59) /
     &-29.670,-27.879,-27.342,-26.307,-23.374,-20.040,
     &-18.464,-17.259,-15.230,-11.899,-7.561,-0.320,
     &7.496,11.439,12.563,13.295,13.539,13.607,
     &14.529,16.224,17.856,18.733,18.768,18.270,
     &17.239,16.217,14.540,12.543,11.244,10.477,
     &10.107,9.619,9.389,9.062,9.037,-999.000,
     &8.718,-999.000,7.493,-999.000,1.513,-999.000,
     &-1.929,-999.000,-5.324,-999.000,-11.667,-999.000,
     &-19.175,-999.000,-24.839,-999.000,-28.723,-999.000,
     &-31.481,-33.039,-32.683,-31.387,-29.670
     &/
       data (fjrocvz(it, 8,3, 1), it=1,59) /
     &-25.205,-24.134,-24.977,-25.465,-24.148,-19.880,
     &-14.939,-11.720,-9.913,-8.337,-5.311,-1.462,
     &1.250,2.994,4.205,5.692,7.044,7.993,
     &9.299,10.947,13.560,16.514,18.152,18.439,
     &18.047,16.866,15.049,12.629,10.416,9.674,
     &9.072,8.218,7.086,5.978,5.080,-999.000,
     &4.418,-999.000,3.056,-999.000,0.431,-999.000,
     &-2.077,-999.000,-5.706,-999.000,-10.312,-999.000,
     &-15.894,-999.000,-20.519,-999.000,-24.751,-999.000,
     &-27.758,-28.693,-26.967,-25.783,-25.205
     &/
       data (fjrocvz(it, 9,3, 1), it=1,59) /
     &-17.784,-17.960,-18.178,-18.466,-18.038,-14.370,
     &-9.915,-7.376,-6.257,-7.094,-6.226,-3.815,
     &-3.921,-0.518,3.895,7.987,11.598,12.002,
     &12.741,13.414,13.827,13.007,14.052,14.881,
     &15.378,14.694,12.629,9.774,7.461,6.078,
     &4.731,3.275,2.037,1.829,1.784,-999.000,
     &0.499,-999.000,-0.788,-999.000,-0.665,-999.000,
     &-0.014,-999.000,-0.433,-999.000,-3.476,-999.000,
     &-7.240,-999.000,-10.303,-999.000,-12.408,-999.000,
     &-14.322,-15.552,-14.968,-16.019,-17.784
     &/
       data (fjrocvz(it,10,3, 1), it=1,59) /
     &-14.735,-15.127,-14.268,-13.184,-13.572,-13.927,
     &-13.202,-11.732,-10.625,-10.970,-10.988,-12.308,
     &-14.451,-6.437,2.057,7.714,13.370,14.224,
     &15.176,15.786,16.267,16.021,15.664,15.358,
     &14.962,14.475,12.656,10.153,7.526,5.031,
     &2.608,-0.116,-2.271,-2.955,-3.788,-999.000,
     &-5.454,-999.000,-5.430,-999.000,-4.731,-999.000,
     &-1.456,-999.000,0.591,-999.000,1.223,-999.000,
     &0.060,-999.000,-1.739,-999.000,-3.572,-999.000,
     &-5.151,-8.053,-10.057,-12.394,-14.735
     &/
       data (fjrocvz(it,11,3, 1), it=1,59) /
     &-19.140,-20.641,-20.918,-20.223,-19.955,-18.830,
     &-17.113,-16.567,-19.036,-17.914,-17.526,-11.672,
     &-15.208,-8.894,-2.605,3.872,10.367,12.438,
     &14.581,16.299,19.068,20.703,19.955,18.974,
     &17.990,16.858,15.515,13.869,11.262,8.050,
     &4.771,1.832,-0.366,-2.368,-4.248,-999.000,
     &-5.610,-999.000,-5.703,-999.000,-5.425,-999.000,
     &-0.323,3.638,4.826,3.119,1.756,-999.000,
     &1.027,-999.000,0.607,-999.000,-0.854,-999.000,
     &-3.057,-9.111,-13.442,-16.429,-19.140
     &/
       data (fjrocvz(it,12,3, 1), it=1,59) /
     &-20.791,-22.754,-25.106,-26.841,-26.601,-24.617,
     &-24.513,-28.712,-33.558,-31.457,-25.084,-15.525,
     &-14.806,-11.684,-8.292,-2.245,3.763,7.643,
     &10.956,15.271,19.784,21.543,21.540,20.920,
     &19.938,18.493,16.974,15.948,13.861,10.356,
     &7.256,5.388,3.434,1.688,-0.405,-999.000,
     &-2.538,-999.000,-1.527,-999.000,2.338,5.626,
     &7.775,9.761,8.313,-999.000,5.436,-999.000,
     &2.614,-999.000,1.933,-999.000,0.254,-999.000,
     &-2.760,-9.127,-14.660,-18.043,-20.791
     &/
       data (fjrocvz(it,13,3, 1), it=1,59) /
     &-18.782,-20.491,-21.195,-22.235,-24.307,-26.166,
     &-28.616,-31.473,-34.040,-32.606,-25.715,-18.109,
     &-14.527,-13.730,-10.761,-7.444,-4.447,2.654,
     &10.910,16.984,21.391,23.488,23.681,22.449,
     &21.113,19.955,18.525,17.071,15.205,12.326,
     &10.029,7.525,5.318,4.111,2.425,-999.000,
     &1.866,-999.000,4.195,-999.000,6.141,9.083,
     &13.053,15.014,11.065,-999.000,8.075,-999.000,
     &4.718,-999.000,1.785,-999.000,0.092,-999.000,
     &-2.008,-6.493,-11.206,-15.338,-18.782
     &/
       data (fjrocvz(it,14,3, 1), it=1,59) /
     &-18.362,-20.657,-20.761,-21.700,-24.733,-28.662,
     &-31.044,-33.320,-36.811,-36.127,-27.649,-17.130,
     &-12.732,-12.542,-11.132,-8.656,-6.286,0.535,
     &9.932,15.988,21.194,23.688,23.950,22.322,
     &21.120,20.511,19.731,18.651,17.613,15.292,
     &12.893,10.246,8.134,6.589,5.822,-999.000,
     &5.563,-999.000,7.394,-999.000,10.000,11.892,
     &17.061,17.809,15.589,11.306,8.853,-999.000,
     &4.873,-999.000,2.855,-999.000,0.494,-999.000,
     &-1.957,-5.883,-9.855,-14.500,-18.362
     &/
       data (fjrocvz(it,15,3, 1), it=1,59) /
     &-17.969,-20.075,-20.571,-21.217,-23.673,-27.482,
     &-30.478,-33.991,-38.327,-39.728,-31.889,-20.534,
     &-14.382,-13.757,-13.089,-10.820,-7.386,-0.740,
     &7.386,13.451,18.392,20.808,20.334,18.377,
     &16.836,15.755,15.032,13.943,12.589,10.945,
     &9.537,8.032,5.896,4.781,4.389,-999.000,
     &3.430,-999.000,5.166,-999.000,8.189,11.410,
     &13.810,11.846,8.460,6.345,4.845,-999.000,
     &1.479,-999.000,-0.770,-999.000,-2.233,-999.000,
     &-4.531,-7.997,-11.428,-14.725,-17.969
     &/
       data (fjrocvz(it,16,3, 1), it=1,59) /
     &-16.229,-17.972,-18.209,-18.729,-21.683,-26.221,
     &-30.743,-34.657,-40.104,-42.983,-38.700,-26.202,
     &-16.613,-13.579,-12.670,-10.316,-6.331,-0.313,
     &6.811,12.342,16.396,17.706,16.452,14.751,
     &13.171,11.446,9.509,6.940,4.583,2.839,
     &2.054,0.924,-0.758,-2.025,-2.359,-999.000,
     &-2.218,-999.000,-0.877,-999.000,1.603,4.745,
     &5.567,0.903,-0.680,-999.000,-4.325,-999.000,
     &-6.699,-999.000,-8.566,-999.000,-8.912,-999.000,
     &-9.180,-10.856,-12.240,-14.068,-16.229
     &/
       data (fjrocvz(it,17,3, 1), it=1,59) /
     &-17.341,-18.479,-18.124,-17.995,-20.170,-23.712,
     &-28.127,-34.755,-42.471,-45.484,-40.377,-25.377,
     &-14.134,-11.473,-9.524,-6.263,-2.110,2.946,
     &9.217,14.726,18.303,19.370,18.389,16.060,
     &13.636,11.427,9.080,5.988,2.956,0.449,
     &-1.702,-3.109,-4.446,-5.738,-6.521,-999.000,
     &-7.774,-999.000,-7.097,-999.000,-4.906,0.213,
     &1.780,-1.439,-6.118,-999.000,-10.063,-999.000,
     &-13.032,-999.000,-14.021,-999.000,-14.237,-999.000,
     &-14.086,-15.153,-15.778,-16.304,-17.341
     &/
       data (fjrocvz(it,18,3, 1), it=1,59) /
     &-24.876,-24.257,-22.646,-20.836,-20.414,-22.111,
     &-24.872,-31.796,-41.891,-44.844,-38.020,-16.560,
     &-8.181,-7.371,-5.581,-2.267,1.756,7.622,
     &14.097,19.208,21.920,22.903,22.363,20.476,
     &17.600,14.455,11.842,8.661,5.697,2.529,
     &-0.237,-2.239,-4.172,-5.919,-7.697,-999.000,
     &-9.336,-999.000,-8.245,-999.000,-6.988,-3.987,
     &-5.001,-7.781,-10.491,-999.000,-15.306,-999.000,
     &-18.299,-999.000,-18.864,-999.000,-18.910,-999.000,
     &-18.782,-20.349,-21.866,-23.314,-24.876
     &/
       data (fjrocvz(it,19,3, 1), it=1,59) /
     &-27.725,-27.259,-25.450,-23.339,-22.517,-24.481,
     &-25.802,-28.554,-35.005,-36.739,-31.144,-14.682,
     &-6.559,-4.951,-3.356,-0.493,3.939,10.261,
     &16.829,21.795,25.183,25.930,25.232,23.684,
     &21.452,18.702,15.594,12.501,9.617,6.320,
     &3.813,1.994,0.115,-2.544,-4.670,-999.000,
     &-5.355,-999.000,-5.032,-999.000,-4.580,-3.219,
     &-5.135,-7.622,-11.595,-999.000,-15.219,-999.000,
     &-17.102,-999.000,-19.519,-999.000,-20.196,-999.000,
     &-20.126,-21.507,-22.953,-25.542,-27.725
     &/
       data (fjrocvz(it,20,3, 1), it=1,59) /
     &-25.518,-25.756,-24.507,-22.707,-22.737,-25.487,
     &-28.180,-30.705,-32.428,-30.386,-24.098,-14.527,
     &-7.453,-4.201,-2.581,-0.585,3.119,9.922,
     &16.954,23.410,27.369,28.045,26.816,24.702,
     &22.709,20.746,19.229,17.843,15.514,12.270,
     &9.296,6.986,4.284,1.804,0.238,-999.000,
     &-0.919,-999.000,-0.409,-999.000,0.637,2.416,
     &1.125,-1.388,-5.737,-999.000,-9.338,-999.000,
     &-10.867,-999.000,-13.571,-999.000,-15.104,-999.000,
     &-17.045,-18.883,-20.335,-22.810,-25.518
     &/
       data (fjrocvz(it,21,3, 1), it=1,59) /
     &-22.301,-23.627,-22.999,-22.035,-23.217,-26.880,
     &-29.867,-31.894,-35.129,-37.027,-33.751,-23.475,
     &-12.312,-7.227,-4.993,-3.044,0.018,5.351,
     &11.855,17.676,21.476,22.438,21.919,21.071,
     &19.974,18.692,17.931,16.588,14.705,11.865,
     &9.515,7.299,5.252,3.993,3.070,-999.000,
     &2.351,-999.000,1.937,-999.000,1.144,1.860,
     &3.029,2.602,-0.380,-999.000,-3.925,-999.000,
     &-4.282,-999.000,-7.821,-999.000,-10.175,-999.000,
     &-12.554,-14.807,-16.455,-19.185,-22.301
     &/
       data (fjrocvz(it,22,3, 1), it=1,59) /
     &-19.078,-19.476,-18.621,-18.886,-21.686,-25.033,
     &-27.598,-31.247,-37.340,-40.934,-38.348,-27.977,
     &-16.362,-8.735,-6.613,-4.800,-2.277,1.207,
     &5.843,10.175,13.542,14.310,13.871,13.736,
     &13.553,12.791,11.892,10.397,8.425,6.459,
     &4.839,3.770,2.995,2.851,3.676,-999.000,
     &3.577,-999.000,2.780,-999.000,3.000,3.018,
     &6.550,7.264,1.991,-999.000,-2.235,-999.000,
     &-3.825,-999.000,-5.871,-999.000,-8.233,-999.000,
     &-10.514,-13.233,-15.458,-17.717,-19.078
     &/
       data (fjrocvz(it,23,3, 1), it=1,59) /
     &-20.304,-20.431,-19.219,-18.877,-21.815,-25.320,
     &-28.637,-31.774,-35.276,-36.741,-32.839,-22.997,
     &-11.912,-6.264,-5.417,-4.154,-1.933,-0.019,
     &2.977,7.212,11.021,12.655,11.760,10.022,
     &8.383,6.845,5.892,4.916,3.453,1.948,
     &1.522,1.260,0.720,0.612,1.685,-999.000,
     &1.360,-999.000,2.197,-999.000,4.493,7.691,
     &10.258,10.673,5.394,-999.000,-0.230,-999.000,
     &-3.626,-999.000,-5.890,-999.000,-8.613,-999.000,
     &-11.705,-14.770,-17.183,-19.122,-20.304
     &/
       data (fjrocvz(it,24,3, 1), it=1,59) /
     &-22.701,-23.964,-22.932,-21.429,-21.895,-23.268,
     &-25.198,-27.558,-31.618,-32.995,-28.660,-18.036,
     &-7.811,-2.876,-0.976,0.710,2.308,2.719,
     &4.432,7.456,10.550,11.684,11.106,9.628,
     &6.964,5.064,4.164,4.821,4.900,4.203,
     &4.101,3.804,3.312,2.407,2.280,-999.000,
     &2.741,-999.000,4.091,-999.000,7.169,10.642,
     &10.272,8.641,7.042,-999.000,-0.084,-999.000,
     &-5.709,-999.000,-7.328,-999.000,-10.772,-999.000,
     &-14.958,-18.519,-20.611,-21.634,-22.701
     &/
       data (fjrocvz(it,25,3, 1), it=1,59) /
     &-22.055,-23.644,-23.612,-22.829,-21.756,-19.871,
     &-19.486,-22.095,-27.665,-30.245,-23.834,-10.763,
     &-0.258,3.948,6.717,8.858,8.964,7.383,
     &7.413,9.728,12.455,13.655,13.160,10.756,
     &7.633,6.526,7.334,8.703,9.121,7.950,
     &7.200,7.012,6.616,5.316,4.992,-999.000,
     &4.142,-999.000,4.303,-999.000,8.658,11.280,
     &11.091,9.415,2.257,-999.000,0.086,-999.000,
     &-6.257,-999.000,-9.348,-999.000,-12.346,-999.000,
     &-16.083,-18.841,-20.614,-21.132,-22.055
     &/
       data (fjrocvz(it, 1,4, 1), it=1,59) /
     &-13.535,-12.797,-11.568,-11.006,-12.202,-13.208,
     &-14.090,-15.243,-16.093,-15.702,-12.733,-6.409,
     &-7.625,-2.935,2.360,8.520,14.206,23.106,
     &30.108,32.937,32.599,30.712,27.919,23.461,
     &18.659,13.544,7.728,3.354,1.647,2.375,
     &2.742,2.706,2.503,3.419,5.294,-999.000,
     &7.512,-999.000,10.081,10.084,7.779,2.029,
     &-3.415,-999.000,-2.205,-999.000,-6.035,-999.000,
     &-10.103,-999.000,-11.923,-999.000,-15.399,-999.000,
     &-16.544,-16.400,-15.959,-14.886,-13.535
     &/
       data (fjrocvz(it, 2,4, 1), it=1,59) /
     &-11.871,-11.260,-10.687,-10.511,-10.135,-9.384,
     &-9.758,-11.699,-13.099,-12.194,-6.712,1.189,
     &7.001,9.902,14.046,16.587,20.331,25.544,
     &30.949,34.575,35.243,33.624,30.468,25.813,
     &20.852,16.500,12.264,8.094,5.060,4.487,
     &5.295,5.843,6.015,6.749,7.534,-999.000,
     &8.578,-999.000,11.915,11.859,11.336,8.858,
     &3.457,-999.000,-2.119,-999.000,-6.793,-999.000,
     &-7.540,-999.000,-11.258,-999.000,-15.249,-999.000,
     &-15.806,-16.013,-14.917,-12.878,-11.871
     &/
       data (fjrocvz(it, 3,4, 1), it=1,59) /
     &-11.963,-10.776,-10.170,-9.294,-8.535,-8.285,
     &-8.035,-8.343,-8.862,-8.358,-3.789,3.040,
     &9.049,12.054,14.325,17.287,20.680,23.711,
     &26.966,29.480,30.541,28.994,25.624,21.353,
     &17.153,13.745,10.860,8.247,5.659,4.851,
     &5.639,6.467,7.221,7.360,8.315,-999.000,
     &10.376,-999.000,12.268,15.210,16.614,10.692,
     &1.173,-2.207,-2.957,-7.002,-9.046,-8.555,
     &-8.725,-999.000,-14.886,-999.000,-17.306,-999.000,
     &-17.877,-16.912,-15.539,-13.657,-11.963
     &/
       data (fjrocvz(it, 4,4, 1), it=1,59) /
     &-13.235,-12.125,-10.638,-9.280,-9.101,-8.609,
     &-7.762,-6.215,-5.028,-4.436,-1.358,4.347,
     &10.324,12.488,13.598,16.081,19.347,23.018,
     &26.211,27.925,27.056,24.249,20.855,17.023,
     &13.089,9.738,7.313,5.703,4.527,4.283,
     &4.901,6.064,6.845,8.059,10.434,-999.000,
     &11.916,-999.000,14.676,17.840,17.141,10.594,
     &4.110,2.546,-2.734,-10.875,-11.599,-999.000,
     &-10.734,-999.000,-16.247,-999.000,-19.101,-999.000,
     &-18.668,-16.744,-15.554,-14.099,-13.235
     &/
       data (fjrocvz(it, 5,4, 1), it=1,59) /
     &-14.193,-13.080,-10.971,-9.487,-9.615,-9.333,
     &-8.694,-6.980,-5.252,-4.425,-1.454,4.735,
     &10.690,12.985,15.301,17.062,19.998,24.177,
     &27.953,29.909,28.981,26.160,22.244,18.001,
     &13.723,10.720,8.768,7.657,6.401,5.803,
     &6.286,7.486,8.851,9.887,10.810,-999.000,
     &14.044,-999.000,17.536,18.903,19.182,16.982,
     &11.544,5.167,-1.878,-7.527,-10.054,-12.934,
     &-15.548,-20.091,-22.339,-22.112,-20.804,-999.000,
     &-19.364,-17.290,-15.757,-14.693,-14.193
     &/
       data (fjrocvz(it, 6,4, 1), it=1,59) /
     &-15.069,-14.291,-12.916,-11.659,-11.474,-9.999,
     &-8.123,-6.560,-4.494,-2.876,-0.876,0.299,
     &3.294,7.998,13.839,18.191,20.657,24.682,
     &28.491,30.416,30.584,28.736,25.784,21.843,
     &18.149,15.568,13.430,11.536,9.755,8.414,
     &8.114,9.442,11.136,11.160,11.964,-999.000,
     &15.543,-999.000,19.471,22.303,25.530,25.072,
     &21.215,12.874,4.330,-3.832,-8.508,-14.895,
     &-17.924,-999.000,-22.172,-999.000,-22.373,-999.000,
     &-20.880,-17.935,-16.424,-15.817,-15.069
     &/
       data (fjrocvz(it, 7,4, 1), it=1,59) /
     &-16.268,-15.609,-14.506,-13.481,-12.662,-9.838,
     &-7.477,-5.703,-2.941,-2.195,-0.229,1.920,
     &6.752,8.291,11.517,14.203,16.616,21.321,
     &25.955,29.333,29.737,29.187,27.616,25.030,
     &22.154,19.460,17.111,15.037,12.708,10.506,
     &9.701,10.350,11.574,11.719,11.753,-999.000,
     &14.160,-999.000,17.559,22.228,26.826,30.957,
     &26.249,14.347,4.180,-2.539,-8.490,-16.897,
     &-21.792,-23.800,-24.326,-24.238,-23.500,-999.000,
     &-21.848,-19.922,-18.543,-16.946,-16.268
     &/
       data (fjrocvz(it, 8,4, 1), it=1,59) /
     &-15.624,-13.868,-12.759,-12.112,-11.033,-9.425,
     &-8.207,-6.954,-4.487,-6.932,-9.073,-11.132,
     &-6.503,-0.374,4.647,9.535,11.570,14.735,
     &20.331,24.590,26.220,26.372,25.738,24.334,
     &22.389,20.498,18.189,15.963,13.288,10.821,
     &9.725,10.023,11.145,11.253,11.132,-999.000,
     &13.588,-999.000,17.235,20.031,24.074,24.703,
     &19.648,13.200,5.214,-2.183,-8.250,-15.212,
     &-19.371,-20.402,-18.937,-999.000,-19.801,-999.000,
     &-19.636,-19.541,-18.665,-17.015,-15.624
     &/
       data (fjrocvz(it, 9,4, 1), it=1,59) /
     &-12.394,-9.876,-7.776,-6.331,-5.832,-6.101,
     &-7.353,-8.482,-8.932,-11.173,-14.147,-16.690,
     &-12.249,-3.315,4.171,8.865,9.912,13.140,
     &17.996,21.416,22.677,23.142,22.760,21.206,
     &18.795,16.482,14.577,12.587,10.248,8.886,
     &8.766,9.296,10.332,10.496,11.014,-999.000,
     &12.427,-999.000,15.875,19.889,22.447,25.644,
     &26.624,21.359,10.853,2.143,-3.103,-999.000,
     &-7.134,-999.000,-10.943,-999.000,-14.109,-999.000,
     &-15.020,-15.360,-15.836,-14.689,-12.394
     &/
       data (fjrocvz(it,10,4, 1), it=1,59) /
     &-11.647,-9.213,-6.797,-5.351,-4.085,-5.139,
     &-8.376,-10.968,-12.050,-13.101,-14.095,-14.448,
     &-11.148,-1.321,6.229,10.859,12.293,16.198,
     &21.238,24.327,24.497,23.636,22.085,20.229,
     &17.700,14.824,12.428,10.294,8.671,8.048,
     &8.069,8.356,8.239,8.164,9.278,-999.000,
     &10.680,-999.000,16.786,21.594,26.466,29.411,
     &29.451,20.808,12.217,-999.000,5.784,-999.000,
     &-0.413,-999.000,-6.028,-999.000,-7.694,-999.000,
     &-9.296,-10.871,-12.460,-12.390,-11.647
     &/
       data (fjrocvz(it,11,4, 1), it=1,59) /
     &-13.664,-12.981,-12.664,-12.406,-11.789,-12.555,
     &-13.457,-15.068,-16.654,-18.408,-17.835,-14.082,
     &-9.233,-0.085,7.046,12.657,16.437,19.282,
     &23.426,26.836,27.765,27.084,25.759,23.715,
     &21.339,18.441,15.188,12.776,10.589,9.567,
     &8.872,8.052,7.443,7.407,9.138,-999.000,
     &12.390,-999.000,17.477,22.594,29.306,29.488,
     &24.510,17.105,9.729,-999.000,1.318,-999.000,
     &-5.182,-999.000,-8.410,-999.000,-9.174,-999.000,
     &-11.640,-12.019,-13.487,-14.472,-13.664
     &/
       data (fjrocvz(it,12,4, 1), it=1,59) /
     &-16.408,-17.342,-17.322,-17.208,-17.470,-17.151,
     &-16.886,-18.608,-21.464,-22.570,-22.767,-19.010,
     &-14.485,-5.509,3.958,9.609,13.999,18.330,
     &24.641,29.207,30.541,29.720,28.256,26.629,
     &24.380,20.901,17.328,14.376,12.122,11.286,
     &10.293,8.996,7.993,7.788,9.742,-999.000,
     &13.124,-999.000,18.410,23.432,29.971,32.142,
     &27.192,14.981,6.229,0.205,-4.576,-999.000,
     &-6.999,-999.000,-8.440,-999.000,-12.298,-999.000,
     &-14.600,-15.057,-15.791,-16.243,-16.408
     &/
       data (fjrocvz(it,13,4, 1), it=1,59) /
     &-17.503,-17.150,-16.994,-17.075,-16.814,-17.514,
     &-19.048,-20.371,-22.039,-21.517,-20.706,-18.306,
     &-14.306,-6.369,2.260,8.809,13.383,18.295,
     &27.063,32.394,32.580,31.323,29.900,28.331,
     &25.728,21.832,18.306,15.728,14.095,13.657,
     &12.828,11.440,9.860,9.446,10.987,-999.000,
     &13.484,-999.000,21.005,25.070,31.310,34.894,
     &27.999,15.543,2.911,-2.816,-4.039,-5.596,
     &-6.249,-999.000,-9.211,-999.000,-12.709,-999.000,
     &-15.447,-17.802,-18.042,-17.502,-17.503
     &/
       data (fjrocvz(it,14,4, 1), it=1,59) /
     &-16.639,-16.262,-15.959,-15.406,-15.413,-17.013,
     &-19.724,-23.301,-26.586,-25.195,-21.465,-16.563,
     &-11.358,-8.868,-2.224,5.335,10.719,16.455,
     &25.406,30.663,31.664,30.689,29.126,27.263,
     &24.271,21.042,18.299,16.396,15.152,14.512,
     &13.334,11.874,10.169,9.829,11.633,-999.000,
     &15.113,-999.000,22.395,26.310,28.823,29.375,
     &23.838,12.962,6.532,-999.000,-0.617,-999.000,
     &-8.811,-999.000,-12.793,-999.000,-15.894,-999.000,
     &-19.025,-19.943,-19.181,-17.601,-16.639
     &/
       data (fjrocvz(it,15,4, 1), it=1,59) /
     &-14.873,-14.199,-13.463,-12.909,-12.731,-13.909,
     &-17.571,-23.491,-27.368,-27.977,-24.133,-16.318,
     &-10.681,-7.071,-2.297,4.687,10.048,15.367,
     &22.640,27.486,28.881,27.733,25.560,22.620,
     &19.340,16.653,14.243,12.483,11.158,10.057,
     &9.007,7.852,7.032,7.456,9.807,-999.000,
     &12.588,-999.000,18.384,20.786,21.981,23.553,
     &19.213,9.267,3.846,-999.000,-4.527,-999.000,
     &-11.546,-999.000,-16.582,-999.000,-20.298,-999.000,
     &-22.496,-22.713,-20.318,-16.712,-14.873
     &/
       data (fjrocvz(it,16,4, 1), it=1,59) /
     &-14.997,-13.385,-13.364,-13.311,-13.510,-14.874,
     &-18.203,-22.502,-25.340,-28.120,-26.961,-19.248,
     &-9.674,-6.331,-1.477,4.244,8.748,14.143,
     &21.443,26.471,28.159,27.320,25.526,21.816,
     &17.124,12.849,9.849,7.902,6.661,5.885,
     &4.895,3.627,2.625,3.064,5.046,-999.000,
     &7.455,-999.000,12.538,14.241,15.801,16.790,
     &12.121,2.354,0.622,-999.000,-7.595,-999.000,
     &-15.116,-999.000,-20.082,-999.000,-23.142,-999.000,
     &-24.595,-23.936,-21.284,-17.759,-14.997
     &/
       data (fjrocvz(it,17,4, 1), it=1,59) /
     &-17.622,-16.523,-15.598,-14.916,-15.205,-15.707,
     &-18.165,-22.073,-24.529,-26.213,-24.507,-16.386,
     &-7.628,-3.692,1.904,6.522,10.008,15.610,
     &23.323,28.468,30.663,30.402,28.774,25.397,
     &20.114,14.949,10.739,7.646,6.032,5.025,
     &3.752,2.078,0.151,-0.561,1.191,-999.000,
     &4.404,-999.000,10.239,11.127,12.811,10.685,
     &7.059,0.519,-4.929,-11.500,-16.999,-20.462,
     &-21.983,-999.000,-22.334,-999.000,-24.130,-999.000,
     &-24.521,-22.971,-21.068,-19.211,-17.622
     &/
       data (fjrocvz(it,18,4, 1), it=1,59) /
     &-18.478,-16.669,-15.274,-14.748,-14.333,-13.939,
     &-16.873,-20.782,-22.203,-23.840,-21.328,-14.050,
     &-5.216,-1.161,2.848,7.551,11.498,18.348,
     &25.303,30.732,34.037,33.823,31.128,26.695,
     &21.935,17.451,12.894,8.819,6.277,4.529,
     &3.051,1.272,-0.335,-0.249,2.076,-999.000,
     &4.422,6.192,9.424,10.640,9.798,7.381,
     &0.714,-3.247,-9.155,-999.000,-18.221,-999.000,
     &-24.352,-999.000,-26.248,-999.000,-25.968,-999.000,
     &-24.263,-23.194,-21.111,-19.633,-18.478
     &/
       data (fjrocvz(it,19,4, 1), it=1,59) /
     &-17.201,-15.226,-14.875,-14.251,-13.623,-12.719,
     &-14.535,-16.351,-16.821,-18.176,-15.303,-8.099,
     &0.314,2.577,6.511,11.407,15.247,21.431,
     &28.950,34.636,37.087,36.641,33.610,29.127,
     &23.992,18.914,14.237,10.082,7.867,6.737,
     &5.119,3.146,1.518,1.237,3.250,-999.000,
     &6.444,-999.000,11.180,10.136,8.412,1.633,
     &-0.941,-5.380,-11.045,-19.278,-23.530,-26.552,
     &-26.994,-999.000,-27.004,-999.000,-25.297,-999.000,
     &-23.189,-22.299,-20.548,-18.972,-17.201
     &/
       data (fjrocvz(it,20,4, 1), it=1,59) /
     &-19.803,-17.755,-15.825,-13.879,-12.291,-11.036,
     &-12.368,-14.651,-14.664,-15.071,-12.377,-3.004,
     &5.329,5.649,9.242,12.869,16.138,22.537,
     &29.769,36.233,38.936,38.887,36.513,32.598,
     &27.933,23.081,18.171,13.962,11.157,9.372,
     &7.453,5.316,3.329,3.180,5.656,-999.000,
     &8.421,-999.000,11.671,10.369,11.034,8.845,
     &3.677,-4.698,-9.782,-15.049,-19.001,-23.546,
     &-25.884,-999.000,-25.564,-999.000,-25.524,-999.000,
     &-24.221,-23.574,-22.457,-21.160,-19.803
     &/
       data (fjrocvz(it,21,4, 1), it=1,59) /
     &-20.408,-17.910,-15.976,-14.741,-14.011,-12.402,
     &-12.822,-14.780,-15.982,-16.183,-13.142,-5.536,
     &0.391,3.862,7.988,12.561,16.998,22.112,
     &27.494,33.673,36.252,36.741,34.496,30.397,
     &25.970,21.625,17.288,12.514,8.843,6.588,
     &4.653,3.239,2.147,1.782,3.208,-999.000,
     &6.235,-999.000,11.409,12.834,14.264,13.219,
     &4.479,-999.000,-4.880,-999.000,-15.583,-999.000,
     &-22.108,-999.000,-23.985,-999.000,-24.259,-999.000,
     &-23.548,-23.143,-22.486,-21.713,-20.408
     &/
       data (fjrocvz(it,22,4, 1), it=1,59) /
     &-19.286,-18.524,-17.535,-16.898,-17.283,-16.554,
     &-16.527,-18.459,-21.661,-22.366,-19.254,-12.255,
     &-3.935,-1.113,4.187,8.865,13.356,18.709,
     &25.138,30.800,32.921,32.329,29.753,25.242,
     &19.708,14.511,10.371,6.696,3.556,1.705,
     &0.661,0.767,1.018,0.698,1.812,-999.000,
     &6.581,-999.000,11.395,9.333,10.167,9.755,
     &4.296,-999.000,-7.221,-999.000,-16.190,-999.000,
     &-22.946,-999.000,-23.281,-999.000,-22.866,-999.000,
     &-22.639,-22.002,-20.940,-20.110,-19.286
     &/
       data (fjrocvz(it,23,4, 1), it=1,59) /
     &-16.775,-16.799,-15.578,-13.945,-14.101,-15.181,
     &-17.026,-19.316,-21.584,-22.553,-20.118,-14.026,
     &-8.926,-5.770,-0.250,5.057,10.367,15.853,
     &24.003,29.633,31.287,29.551,26.509,21.917,
     &16.428,10.659,6.446,3.228,0.723,-0.487,
     &-1.151,-1.457,-0.636,0.450,2.434,-999.000,
     &4.398,-999.000,8.293,8.752,8.021,7.821,
     &5.398,-1.150,-6.133,-999.000,-14.241,-999.000,
     &-20.214,-999.000,-22.097,-999.000,-21.927,-999.000,
     &-21.556,-20.443,-18.409,-16.973,-16.775
     &/
       data (fjrocvz(it,24,4, 1), it=1,59) /
     &-14.366,-13.418,-12.011,-10.492,-11.334,-13.602,
     &-15.811,-17.276,-18.854,-19.895,-18.304,-13.599,
     &-7.312,-1.688,1.942,5.423,10.419,18.576,
     &26.759,30.377,30.975,29.142,25.725,20.889,
     &15.769,10.089,5.314,1.874,0.237,0.541,
     &0.539,-0.090,0.399,1.439,3.006,-999.000,
     &4.873,-999.000,8.522,8.058,7.932,6.141,
     &1.520,-999.000,-1.657,-999.000,-7.303,-999.000,
     &-12.344,-999.000,-15.961,-999.000,-16.998,-999.000,
     &-18.043,-17.617,-16.153,-15.293,-14.366
     &/
       data (fjrocvz(it,25,4, 1), it=1,59) /
     &-13.535,-12.797,-11.568,-11.006,-12.202,-13.208,
     &-14.090,-15.243,-16.093,-15.702,-12.733,-6.409,
     &-7.625,-2.935,2.360,8.520,14.206,23.106,
     &30.108,32.937,32.599,30.712,27.919,23.461,
     &18.659,13.544,7.728,3.354,1.647,2.375,
     &2.742,2.706,2.503,3.419,5.294,-999.000,
     &7.512,-999.000,10.081,10.084,7.779,2.029,
     &-3.415,-999.000,-2.205,-999.000,-6.035,-999.000,
     &-10.103,-999.000,-11.923,-999.000,-15.399,-999.000,
     &-16.544,-16.400,-15.959,-14.886,-13.535
     &/
       data (fjrocvz(it, 1,1, 2), it=1,59) /
     &-11.176,-10.890,-10.718,-11.017,-11.719,-12.019,
     &-13.282,-14.667,-16.031,-17.525,-16.390,-10.119,
     &-0.536,6.993,12.855,17.166,21.514,25.873,
     &29.811,32.197,31.581,29.841,27.525,24.560,
     &21.453,17.071,12.958,9.764,6.167,2.482,
     &0.299,-0.430,-0.710,-1.155,-1.596,-999.000,
     &-2.094,-999.000,-1.244,-999.000,0.605,-999.000,
     &0.140,-999.000,-0.899,-999.000,-3.724,-999.000,
     &-7.467,-999.000,-10.597,-999.000,-11.814,-999.000,
     &-13.955,-14.914,-14.223,-12.350,-11.176
     &/
       data (fjrocvz(it, 2,1, 2), it=1,59) /
     &-12.865,-12.037,-11.860,-11.980,-13.277,-14.189,
     &-14.648,-15.477,-16.237,-17.360,-13.798,-4.308,
     &4.492,11.380,16.957,22.842,27.894,31.317,
     &35.008,37.265,36.905,35.024,31.827,28.161,
     &24.008,18.831,13.931,10.000,6.125,2.194,
     &-1.047,-3.124,-4.566,-4.753,-4.991,-999.000,
     &-4.407,-999.000,-3.253,-999.000,-4.351,-999.000,
     &-2.757,-999.000,-4.148,-999.000,-6.832,-999.000,
     &-10.356,-999.000,-13.511,-999.000,-14.706,-999.000,
     &-15.712,-16.238,-15.863,-14.441,-12.865
     &/
       data (fjrocvz(it, 3,1, 2), it=1,59) /
     &-14.026,-13.557,-13.510,-13.984,-15.010,-14.809,
     &-13.922,-13.969,-13.744,-13.217,-7.708,0.648,
     &7.492,13.010,18.308,23.574,26.977,29.888,
     &33.455,36.353,36.271,33.964,30.356,26.889,
     &23.828,20.030,15.977,11.846,7.865,3.518,
     &-0.009,-2.045,-3.659,-3.438,-3.499,-999.000,
     &-2.695,-999.000,-2.409,-999.000,-3.430,-0.847,
     &1.206,0.438,-2.334,-3.185,-6.099,-999.000,
     &-9.026,-999.000,-11.678,-999.000,-12.560,-999.000,
     &-15.053,-16.138,-16.471,-15.268,-14.026
     &/
       data (fjrocvz(it, 4,1, 2), it=1,59) /
     &-13.814,-14.626,-14.731,-13.936,-13.105,-12.993,
     &-12.904,-13.066,-11.747,-9.342,-2.461,5.469,
     &10.040,13.850,19.045,23.149,26.305,29.135,
     &32.605,34.712,33.528,30.293,26.500,23.657,
     &21.102,17.197,13.271,10.305,8.167,5.121,
     &1.984,-0.595,-2.446,-2.145,-0.727,-999.000,
     &-0.482,-999.000,0.361,-999.000,-1.422,1.572,
     &3.987,1.802,-2.034,-999.000,-4.262,-999.000,
     &-5.978,-999.000,-8.824,-999.000,-11.912,-999.000,
     &-14.575,-15.451,-15.123,-13.618,-13.814
     &/
       data (fjrocvz(it, 5,1, 2), it=1,59) /
     &-10.520,-11.121,-11.129,-10.068,-9.313,-10.113,
     &-11.847,-12.881,-11.705,-9.384,-2.144,6.413,
     &10.202,13.239,18.167,22.592,25.377,28.134,
     &31.440,33.585,32.933,30.113,27.156,24.458,
     &21.302,16.646,12.502,10.163,8.836,7.005,
     &4.797,1.923,-0.438,-1.088,-0.294,-999.000,
     &0.659,-999.000,2.577,-999.000,2.032,5.309,
     &5.440,1.409,-0.660,-999.000,-2.738,-999.000,
     &-4.507,-999.000,-7.810,-999.000,-12.041,-999.000,
     &-13.953,-14.026,-12.824,-10.775,-10.520
     &/
       data (fjrocvz(it, 6,1, 2), it=1,59) /
     &-9.228,-9.606,-10.157,-10.732,-10.785,-11.471,
     &-13.280,-14.488,-13.595,-10.902,-4.834,2.531,
     &5.851,8.052,11.477,15.093,17.756,21.520,
     &25.882,29.415,29.895,28.378,26.382,24.225,
     &21.436,17.685,14.420,11.812,10.125,8.845,
     &7.900,6.191,3.114,1.184,1.200,-999.000,
     &2.006,-999.000,3.562,-999.000,4.904,-999.000,
     &3.599,-999.000,1.588,-999.000,0.520,-999.000,
     &-1.379,-999.000,-4.333,-999.000,-8.391,-999.000,
     &-11.328,-12.853,-12.415,-10.154,-9.228
     &/
       data (fjrocvz(it, 7,1, 2), it=1,59) /
     &-12.928,-13.111,-13.879,-14.527,-14.593,-14.713,
     &-14.920,-14.758,-13.703,-12.535,-9.279,-6.054,
     &-3.372,0.946,5.708,9.151,12.242,15.981,
     &21.096,24.732,25.858,24.770,22.702,20.701,
     &18.911,16.302,13.231,10.446,8.555,7.857,
     &8.230,7.557,5.194,2.905,2.116,-999.000,
     &1.772,-999.000,3.394,-999.000,6.250,8.095,
     &11.964,11.658,6.425,-999.000,3.718,-999.000,
     &0.997,-999.000,-2.341,-999.000,-6.855,-999.000,
     &-10.183,-13.291,-14.406,-13.688,-12.928
     &/
       data (fjrocvz(it, 8,1, 2), it=1,59) /
     &-16.827,-15.735,-15.218,-14.828,-14.325,-14.282,
     &-14.757,-15.425,-15.846,-16.323,-14.092,-9.390,
     &-5.236,-2.233,0.442,3.001,5.878,10.343,
     &15.940,20.299,22.427,21.879,19.890,17.786,
     &15.762,13.285,10.761,8.234,6.752,6.205,
     &7.312,7.623,6.505,4.495,3.612,-999.000,
     &3.470,-999.000,5.873,-999.000,11.539,18.449,
     &21.520,22.321,17.207,9.740,6.895,-999.000,
     &-1.662,-999.000,-7.161,-999.000,-11.451,-999.000,
     &-14.414,-17.655,-18.959,-17.975,-16.827
     &/
       data (fjrocvz(it, 9,1, 2), it=1,59) /
     &-17.934,-16.255,-15.143,-14.291,-13.372,-13.704,
     &-14.911,-16.216,-16.963,-17.646,-15.894,-12.218,
     &-8.809,-4.772,-0.343,2.699,4.887,8.385,
     &13.486,18.032,20.402,20.753,19.857,18.482,
     &16.402,14.463,12.872,11.297,9.822,8.535,
     &8.615,8.328,7.210,5.747,5.564,4.615,
     &4.053,3.854,6.721,11.545,16.920,25.620,
     &31.793,29.133,16.829,4.406,-2.823,-7.758,
     &-9.409,-999.000,-14.086,-999.000,-17.206,-999.000,
     &-19.289,-21.350,-21.528,-19.793,-17.934
     &/
       data (fjrocvz(it,10,1, 2), it=1,59) /
     &-16.295,-14.667,-13.599,-12.774,-12.739,-13.754,
     &-14.824,-15.590,-16.659,-19.099,-18.356,-14.058,
     &-8.700,-2.444,2.614,6.355,9.030,11.742,
     &15.458,18.877,21.524,23.292,23.843,23.104,
     &21.023,19.013,17.502,16.128,14.294,12.509,
     &11.591,10.723,9.223,7.365,6.130,-999.000,
     &4.148,4.590,10.936,19.721,29.487,36.655,
     &31.862,21.282,8.492,-4.937,-9.379,-12.550,
     &-14.496,-999.000,-17.638,-999.000,-20.395,-999.000,
     &-21.725,-21.851,-20.792,-18.669,-16.295
     &/
       data (fjrocvz(it,11,1, 2), it=1,59) /
     &-15.835,-14.480,-14.042,-14.447,-15.752,-17.025,
     &-18.219,-18.917,-19.479,-20.718,-18.867,-12.130,
     &-5.033,1.774,7.326,11.240,13.539,14.910,
     &17.530,21.508,25.456,27.922,28.419,27.082,
     &25.003,22.734,20.927,18.564,16.008,13.890,
     &12.084,10.724,9.435,8.322,8.080,6.786,
     &6.436,9.178,14.263,21.739,26.479,28.044,
     &17.327,5.318,-2.530,-11.143,-16.299,-999.000,
     &-17.595,-999.000,-19.663,-999.000,-21.261,-999.000,
     &-21.663,-21.149,-19.824,-17.717,-15.835
     &/
       data (fjrocvz(it,12,1, 2), it=1,59) /
     &-18.263,-17.724,-18.005,-19.309,-20.622,-20.341,
     &-20.313,-20.746,-21.147,-21.479,-19.678,-14.781,
     &-7.077,0.455,9.378,13.600,15.813,17.543,
     &20.069,23.867,26.758,28.554,29.038,28.464,
     &27.149,24.814,22.073,18.834,15.277,12.473,
     &10.704,9.278,8.309,8.538,10.463,-999.000,
     &11.296,10.167,13.080,18.939,25.310,24.255,
     &15.242,7.331,0.823,-999.000,-12.163,-999.000,
     &-19.847,-999.000,-22.282,-999.000,-22.648,-999.000,
     &-22.049,-21.364,-20.585,-19.295,-18.263
     &/
       data (fjrocvz(it,13,1, 2), it=1,59) /
     &-20.052,-20.601,-20.993,-21.864,-22.350,-20.831,
     &-20.411,-21.196,-21.535,-21.868,-20.348,-15.615,
     &-5.994,0.468,7.818,13.286,15.882,17.861,
     &20.503,23.737,25.943,27.075,27.041,26.185,
     &24.356,21.859,19.638,16.612,12.283,9.572,
     &8.829,8.436,7.820,7.521,7.642,-999.000,
     &8.593,-999.000,10.565,14.904,20.626,18.311,
     &12.749,-999.000,4.153,-999.000,-8.558,-999.000,
     &-17.979,-999.000,-22.259,-999.000,-22.957,-999.000,
     &-22.423,-21.536,-20.538,-19.884,-20.052
     &/
       data (fjrocvz(it,14,1, 2), it=1,59) /
     &-20.001,-21.201,-21.862,-21.984,-21.381,-19.791,
     &-20.642,-22.206,-22.245,-21.862,-18.531,-14.406,
     &-7.310,-2.648,4.774,10.886,13.715,15.636,
     &17.888,20.950,23.773,24.964,24.560,22.973,
     &21.072,19.013,16.908,13.536,9.496,6.823,
     &6.200,5.958,4.823,2.934,2.443,-999.000,
     &3.835,3.603,8.051,10.817,13.224,14.098,
     &12.457,6.746,3.642,-999.000,-5.898,-999.000,
     &-15.115,-999.000,-21.027,-999.000,-22.849,-999.000,
     &-22.641,-21.640,-20.061,-19.510,-20.001
     &/
       data (fjrocvz(it,15,1, 2), it=1,59) /
     &-20.221,-20.964,-22.142,-22.790,-22.290,-20.501,
     &-20.823,-22.485,-24.218,-25.696,-22.980,-17.191,
     &-8.874,-4.093,1.812,7.284,10.640,13.249,
     &15.464,17.129,18.514,19.177,18.360,16.732,
     &15.248,13.868,12.397,10.354,7.137,4.257,
     &3.280,2.800,1.476,0.960,1.322,-999.000,
     &2.934,-999.000,7.292,11.364,14.842,16.739,
     &15.231,9.188,5.288,-999.000,-3.466,-999.000,
     &-12.496,-999.000,-19.852,-999.000,-22.588,-999.000,
     &-22.567,-21.807,-20.813,-20.357,-20.221
     &/
       data (fjrocvz(it,16,1, 2), it=1,59) /
     &-22.299,-22.281,-22.871,-23.598,-23.154,-21.164,
     &-20.926,-21.701,-23.567,-28.993,-30.018,-24.710,
     &-9.892,-2.330,1.888,6.332,9.826,11.994,
     &14.194,15.993,17.501,17.526,16.600,15.047,
     &13.455,11.822,10.183,7.788,4.427,2.148,
     &1.850,2.181,2.090,2.413,3.328,-999.000,
     &5.775,-999.000,11.007,14.783,16.726,16.566,
     &14.046,-999.000,4.797,-999.000,-2.741,-999.000,
     &-12.144,-999.000,-19.584,-999.000,-22.615,-999.000,
     &-23.345,-22.974,-22.603,-22.518,-22.299
     &/
       data (fjrocvz(it,17,1, 2), it=1,59) /
     &-23.559,-23.420,-23.629,-24.395,-25.021,-23.870,
     &-23.650,-22.131,-21.123,-25.752,-28.733,-24.248,
     &-9.324,2.069,5.623,8.456,11.277,13.131,
     &15.222,17.136,18.238,18.827,18.522,17.155,
     &15.446,13.373,11.409,8.803,5.163,2.802,
     &2.369,2.879,2.683,2.360,3.571,-999.000,
     &5.074,7.637,10.016,12.418,13.520,14.969,
     &11.944,-999.000,-0.830,-999.000,-9.048,-999.000,
     &-17.537,-999.000,-23.421,-999.000,-25.851,-999.000,
     &-25.530,-24.089,-23.254,-23.420,-23.559
     &/
       data (fjrocvz(it,18,1, 2), it=1,59) /
     &-23.942,-23.761,-23.957,-25.062,-26.783,-25.814,
     &-25.786,-24.941,-22.279,-22.507,-21.756,-16.958,
     &-4.504,6.493,10.594,13.094,15.873,17.593,
     &19.444,20.074,20.251,20.833,20.416,18.752,
     &16.317,13.898,11.592,9.328,6.275,4.292,
     &3.382,3.441,3.546,3.929,4.755,-999.000,
     &6.339,-999.000,8.835,12.655,14.502,13.466,
     &7.971,0.138,-6.160,-999.000,-14.528,-999.000,
     &-21.965,-999.000,-27.458,-999.000,-28.905,-999.000,
     &-27.334,-25.111,-23.963,-23.984,-23.942
     &/
       data (fjrocvz(it,19,1, 2), it=1,59) /
     &-24.720,-23.671,-22.791,-23.035,-23.638,-21.320,
     &-19.836,-19.345,-17.638,-18.502,-17.751,-13.404,
     &-2.400,8.124,13.932,17.553,20.140,22.156,
     &24.141,24.790,24.773,24.816,24.524,22.601,
     &20.068,17.721,14.669,10.704,7.062,5.977,
     &5.763,5.531,5.277,4.554,4.608,-999.000,
     &6.022,-999.000,8.048,11.135,10.229,6.583,
     &3.083,-999.000,-6.986,-999.000,-15.176,-999.000,
     &-22.777,-999.000,-27.679,-999.000,-28.925,-999.000,
     &-27.491,-25.569,-24.504,-24.755,-24.720
     &/
       data (fjrocvz(it,20,1, 2), it=1,59) /
     &-23.566,-22.553,-22.215,-22.220,-22.603,-20.300,
     &-18.764,-17.786,-16.417,-16.631,-18.187,-15.124,
     &-4.721,7.036,13.309,16.501,20.094,23.095,
     &25.777,27.019,26.781,26.839,26.622,25.362,
     &24.053,22.572,20.079,15.976,11.100,9.226,
     &8.735,8.045,6.530,5.149,4.925,-999.000,
     &4.212,-999.000,3.806,6.471,10.096,7.342,
     &0.455,-999.000,-8.115,-999.000,-16.968,-999.000,
     &-24.296,-999.000,-29.067,-999.000,-30.515,-999.000,
     &-29.879,-28.119,-26.240,-24.861,-23.566
     &/
       data (fjrocvz(it,21,1, 2), it=1,59) /
     &-24.962,-24.467,-24.551,-25.159,-25.882,-23.287,
     &-20.867,-20.162,-20.320,-22.081,-24.245,-21.551,
     &-10.300,2.613,9.531,13.345,17.721,20.060,
     &22.966,25.126,25.563,25.377,24.731,24.130,
     &23.733,22.503,20.131,15.949,10.760,8.471,
     &7.933,7.441,6.114,4.351,3.316,-999.000,
     &4.039,-999.000,5.392,7.810,9.470,5.681,
     &-2.037,-999.000,-10.976,-999.000,-17.952,-999.000,
     &-24.203,-999.000,-29.007,-999.000,-30.746,-999.000,
     &-30.662,-29.718,-28.354,-26.576,-24.962
     &/
       data (fjrocvz(it,22,1, 2), it=1,59) /
     &-23.219,-23.071,-23.523,-24.778,-25.907,-24.199,
     &-22.414,-22.321,-23.017,-24.418,-25.867,-23.237,
     &-12.471,0.164,7.361,11.401,15.179,18.082,
     &21.300,24.288,24.611,24.245,24.289,24.147,
     &23.250,21.286,17.941,12.839,7.359,4.840,
     &4.496,4.449,3.342,2.035,1.777,-999.000,
     &2.342,-999.000,1.847,-999.000,2.966,-999.000,
     &-0.681,-999.000,-10.978,-999.000,-15.910,-999.000,
     &-20.933,-999.000,-25.448,-999.000,-27.610,-999.000,
     &-27.745,-26.696,-25.118,-24.005,-23.219
     &/
       data (fjrocvz(it,23,1, 2), it=1,59) /
     &-17.073,-17.516,-18.004,-18.940,-20.356,-19.492,
     &-18.985,-19.322,-19.267,-19.542,-19.358,-15.479,
     &-6.877,1.742,7.608,10.559,13.570,15.835,
     &19.036,22.155,23.560,25.459,26.418,25.676,
     &23.597,20.834,17.868,13.910,9.079,5.938,
     &3.969,2.896,1.991,1.345,1.375,-999.000,
     &2.178,-999.000,5.417,-999.000,6.787,-999.000,
     &4.515,-999.000,0.097,-999.000,-9.058,-999.000,
     &-14.749,-999.000,-19.508,-999.000,-21.591,-999.000,
     &-21.841,-20.834,-19.123,-17.529,-17.073
     &/
       data (fjrocvz(it,24,1, 2), it=1,59) /
     &-12.507,-12.668,-12.841,-13.683,-14.460,-13.268,
     &-13.072,-13.980,-14.353,-15.796,-15.501,-10.905,
     &-2.481,4.510,9.403,13.079,15.800,18.204,
     &21.324,24.225,25.420,25.984,25.369,23.702,
     &21.348,18.566,15.847,13.032,8.947,5.407,
     &2.927,2.078,1.913,1.324,0.629,-999.000,
     &1.357,-999.000,3.103,-999.000,5.198,-999.000,
     &4.646,-999.000,2.031,-999.000,-2.398,-999.000,
     &-7.680,-999.000,-11.943,-999.000,-13.929,-999.000,
     &-15.055,-15.423,-14.499,-13.177,-12.507
     &/
       data (fjrocvz(it,25,1, 2), it=1,59) /
     &-11.176,-10.890,-10.718,-11.017,-11.719,-12.019,
     &-13.282,-14.667,-16.031,-17.525,-16.390,-10.119,
     &-0.536,6.993,12.855,17.166,21.514,25.873,
     &29.811,32.197,31.581,29.841,27.525,24.560,
     &21.453,17.071,12.958,9.764,6.167,2.482,
     &0.299,-0.430,-0.710,-1.155,-1.596,-999.000,
     &-2.094,-999.000,-1.244,-999.000,0.605,-999.000,
     &0.140,-999.000,-0.899,-999.000,-3.724,-999.000,
     &-7.467,-999.000,-10.597,-999.000,-11.814,-999.000,
     &-13.955,-14.914,-14.223,-12.350,-11.176
     &/
       data (fjrocvz(it, 1,2, 2), it=1,59) /
     &-14.469,-13.657,-12.390,-11.791,-12.913,-13.905,
     &-14.781,-15.927,-16.763,-16.339,-13.353,-7.013,
     &-8.195,-3.436,1.893,8.087,13.842,22.742,
     &29.744,32.573,32.235,30.458,27.720,23.318,
     &18.627,13.775,8.090,3.848,2.403,3.287,
     &3.731,3.773,3.724,4.801,6.836,-999.000,
     &9.357,-999.000,12.228,12.889,11.117,5.567,
     &-0.315,-999.000,-1.798,-999.000,-6.742,-999.000,
     &-11.191,-999.000,-13.146,-999.000,-16.561,-999.000,
     &-17.644,-17.499,-17.058,-15.903,-14.469
     &/
       data (fjrocvz(it, 2,2, 2), it=1,59) /
     &-12.791,-12.125,-11.524,-11.319,-10.887,-10.180,
     &-10.575,-12.537,-13.980,-13.089,-7.613,0.281,
     &6.080,9.105,13.311,15.914,19.783,25.045,
     &30.475,34.126,34.844,33.348,30.253,25.658,
     &20.820,16.717,12.605,8.560,5.776,5.357,
     &6.242,6.867,7.193,8.100,9.057,-999.000,
     &10.357,-999.000,13.950,14.432,14.332,12.151,
     &6.635,-999.000,-1.698,-999.000,-7.885,-999.000,
     &-9.091,-999.000,-12.940,-999.000,-16.747,-999.000,
     &-17.121,-17.200,-15.976,-13.868,-12.791
     &/
       data (fjrocvz(it, 3,2, 2), it=1,59) /
     &-12.899,-11.690,-11.074,-10.187,-9.407,-9.214,
     &-8.993,-9.329,-9.906,-9.395,-4.823,2.010,
     &8.026,11.174,13.518,16.551,20.087,23.216,
     &26.520,29.083,30.240,28.848,25.555,21.362,
     &17.316,14.126,11.351,8.846,6.477,5.788,
     &6.635,7.524,8.397,8.693,9.805,-999.000,
     &12.088,-999.000,14.202,17.556,19.372,14.188,
     &4.712,-0.176,-2.484,-7.832,-10.582,-10.260,
     &-10.600,-999.000,-16.475,-999.000,-18.713,-999.000,
     &-19.102,-18.010,-16.511,-14.611,-12.899
     &/
       data (fjrocvz(it, 4,2, 2), it=1,59) /
     &-14.191,-13.050,-11.548,-10.175,-9.966,-9.584,
     &-8.791,-7.300,-6.222,-5.618,-2.534,3.178,
     &9.168,11.457,12.628,15.173,18.564,22.398,
     &25.672,27.468,26.762,24.150,20.854,17.119,
     &13.381,10.233,7.909,6.401,5.428,5.291,
     &5.963,7.180,8.068,9.429,11.952,-999.000,
     &13.582,-999.000,16.488,20.079,19.807,14.268,
     &8.004,4.737,-2.357,-11.705,-13.271,-999.000,
     &-12.670,-999.000,-17.862,-999.000,-20.529,-999.000,
     &-19.908,-17.882,-16.590,-15.095,-14.191
     &/
       data (fjrocvz(it, 5,2, 2), it=1,59) /
     &-15.076,-13.913,-11.779,-10.271,-10.349,-10.274,
     &-9.738,-8.127,-6.606,-5.776,-2.804,3.386,
     &9.344,11.828,14.238,16.093,19.218,23.593,
     &27.468,29.522,28.790,26.164,22.346,18.200,
     &14.116,11.265,9.390,8.355,7.250,6.729,
     &7.251,8.490,9.932,11.117,12.189,-999.000,
     &15.533,-999.000,19.137,20.971,21.725,20.327,
     &15.043,7.068,-1.715,-8.363,-11.889,-14.843,
     &-17.532,-21.808,-23.788,-23.478,-22.087,-999.000,
     &-20.482,-18.339,-16.738,-15.625,-15.076
     &/
       data (fjrocvz(it, 6,2, 2), it=1,59) /
     &-15.950,-15.140,-13.750,-12.478,-12.262,-11.061,
     &-9.322,-7.896,-6.103,-4.444,-2.423,-1.228,
     &1.809,6.799,12.783,17.278,20.031,24.236,
     &28.135,30.150,30.498,28.807,25.934,22.072,
     &18.535,16.038,13.943,12.091,10.395,9.098,
     &8.819,10.169,11.907,12.147,13.165,-999.000,
     &16.835,-999.000,20.853,24.324,27.979,27.911,
     &23.725,14.391,4.263,-4.781,-10.341,-16.687,
     &-19.677,-999.000,-23.580,-999.000,-23.610,-999.000,
     &-21.947,-18.953,-17.393,-16.742,-15.950
     &/
       data (fjrocvz(it, 7,2, 2), it=1,59) /
     &-17.274,-16.651,-15.567,-14.560,-13.777,-11.196,
     &-8.956,-7.304,-4.786,-3.983,-1.988,0.189,
     &5.078,6.977,10.383,13.248,16.021,20.906,
     &25.629,29.097,29.680,29.216,27.688,25.144,
     &22.354,19.695,17.363,15.307,13.014,10.839,
     &10.047,10.709,11.960,12.387,12.701,-999.000,
     &15.225,-999.000,18.864,24.090,29.080,33.195,
     &27.731,14.871,3.747,-3.648,-10.277,-18.638,
     &-23.488,-25.358,-25.748,-25.562,-24.726,-999.000,
     &-22.877,-20.914,-19.497,-17.927,-17.274
     &/
       data (fjrocvz(it, 8,2, 2), it=1,59) /
     &-16.748,-15.082,-14.017,-13.415,-12.425,-11.003,
     &-9.877,-8.717,-6.436,-8.780,-10.871,-12.879,
     &-8.150,-1.682,3.510,8.568,10.942,14.253,
     &19.921,24.253,26.028,26.219,25.604,24.219,
     &22.311,20.444,18.147,15.932,13.281,10.839,
     &9.755,10.065,11.212,11.589,11.736,-999.000,
     &14.333,-999.000,18.358,21.606,26.079,26.748,
     &20.804,13.261,4.672,-3.366,-9.739,-16.735,
     &-20.927,-21.897,-20.371,-999.000,-21.091,-999.000,
     &-20.782,-20.656,-19.749,-18.119,-16.748
     &/
       data (fjrocvz(it, 9,2, 2), it=1,59) /
     &-13.642,-11.240,-9.198,-7.812,-7.428,-7.770,
     &-9.058,-10.223,-10.746,-12.868,-15.782,-18.266,
     &-13.706,-4.526,3.084,7.901,9.195,12.521,
     &17.426,20.895,22.254,22.759,22.398,20.863,
     &18.493,16.228,14.348,12.382,10.092,8.768,
     &8.667,9.216,10.289,10.636,11.335,-999.000,
     &12.943,-999.000,16.940,21.353,24.472,27.665,
     &27.982,21.420,10.634,1.495,-4.182,-999.000,
     &-8.583,-999.000,-12.323,-999.000,-15.418,-999.000,
     &-16.258,-16.560,-16.998,-15.894,-13.642
     &/
       data (fjrocvz(it,10,2, 2), it=1,59) /
     &-13.146,-10.782,-8.401,-6.989,-5.792,-6.841,
     &-10.075,-12.665,-13.741,-14.670,-15.604,-15.897,
     &-12.475,-2.462,5.182,9.905,11.525,15.492,
     &20.563,23.683,23.913,23.137,21.628,19.814,
     &17.369,14.584,12.233,10.144,8.613,8.026,
     &8.065,8.370,8.289,8.333,9.567,-999.000,
     &11.386,-999.000,17.814,23.030,28.427,31.581,
     &31.217,21.913,12.660,-999.000,5.006,-999.000,
     &-1.803,-999.000,-7.590,-999.000,-9.283,-999.000,
     &-10.912,-12.395,-13.893,-13.856,-13.146
     &/
       data (fjrocvz(it,11,2, 2), it=1,59) /
     &-15.256,-14.592,-14.284,-14.036,-13.438,-14.142,
     &-15.013,-16.593,-18.118,-19.805,-19.199,-15.413,
     &-10.498,-1.263,5.911,11.567,15.434,18.386,
     &22.583,26.047,27.083,26.550,25.299,23.328,
     &21.100,18.346,15.165,12.825,10.783,9.784,
     &9.101,8.293,7.708,7.760,9.578,-999.000,
     &13.152,-999.000,18.561,24.009,31.052,31.323,
     &26.434,18.516,10.626,-999.000,0.985,-999.000,
     &-6.246,-999.000,-10.018,-999.000,-10.826,-999.000,
     &-13.336,-13.616,-14.984,-16.016,-15.256
     &/
       data (fjrocvz(it,12,2, 2), it=1,59) /
     &-17.805,-18.721,-18.692,-18.569,-18.812,-18.452,
     &-18.168,-19.869,-22.685,-23.855,-24.084,-20.359,
     &-15.900,-6.874,2.616,8.292,12.731,17.237,
     &23.635,28.288,29.796,29.168,27.801,26.272,
     &24.216,20.939,17.466,14.615,12.563,11.742,
     &10.757,9.468,8.480,8.383,10.447,-999.000,
     &14.038,-999.000,19.534,24.814,31.612,34.211,
     &29.053,16.113,6.631,0.093,-5.203,-999.000,
     &-8.063,-999.000,-10.004,-999.000,-13.864,-999.000,
     &-16.168,-16.535,-17.178,-17.635,-17.805
     &/
       data (fjrocvz(it,13,2, 2), it=1,59) /
     &-18.526,-18.150,-17.982,-18.053,-17.768,-18.440,
     &-19.959,-21.268,-22.907,-22.548,-21.820,-19.501,
     &-15.665,-7.736,0.888,7.433,11.998,17.114,
     &25.984,31.417,31.807,30.775,29.465,28.009,
     &25.632,21.957,18.542,16.075,14.664,14.264,
     &13.455,12.086,10.545,10.222,11.855,-999.000,
     &14.559,-999.000,22.288,26.748,33.383,37.173,
     &29.959,16.715,3.242,-3.176,-5.089,-6.714,
     &-7.435,-999.000,-10.539,-999.000,-13.939,-999.000,
     &-16.578,-18.871,-19.050,-18.517,-18.526
     &/
       data (fjrocvz(it,14,2, 2), it=1,59) /
     &-17.508,-17.119,-16.810,-16.251,-16.246,-17.827,
     &-20.529,-24.098,-27.364,-26.154,-22.514,-17.703,
     &-12.678,-10.112,-3.430,4.167,9.628,15.527,
     &24.558,29.897,31.061,30.291,28.830,27.070,
     &24.282,21.257,18.617,16.815,15.776,15.220,
     &14.085,12.666,11.046,10.792,12.682,-999.000,
     &16.384,-999.000,23.886,28.721,31.846,32.214,
     &25.999,14.369,6.830,-999.000,-1.960,-999.000,
     &-10.080,-999.000,-13.879,-999.000,-16.879,-999.000,
     &-19.910,-20.796,-20.003,-18.446,-17.508
     &/
       data (fjrocvz(it,15,2, 2), it=1,59) /
     &-15.742,-15.096,-14.375,-13.834,-13.685,-14.898,
     &-18.578,-24.516,-28.428,-29.139,-25.346,-17.581,
     &-12.046,-8.261,-3.399,3.672,9.209,14.656,
     &21.992,26.902,28.425,27.441,25.351,22.493,
     &19.377,16.860,14.534,12.859,11.702,10.725,
     &9.738,8.645,7.949,8.495,10.967,-999.000,
     &13.999,-999.000,20.046,23.459,25.456,27.106,
     &22.102,11.096,4.334,-999.000,-5.594,-999.000,
     &-12.611,-999.000,-17.536,-999.000,-21.145,-999.000,
     &-23.237,-23.461,-21.075,-17.525,-15.742
     &/
       data (fjrocvz(it,16,2, 2), it=1,59) /
     &-15.844,-14.300,-14.313,-14.293,-14.559,-16.022,
     &-19.401,-23.750,-26.689,-29.463,-28.301,-20.585,
     &-11.007,-7.377,-2.380,3.484,8.275,13.670,
     &20.969,25.997,27.684,26.997,25.279,21.645,
     &17.106,12.998,10.082,8.219,7.145,6.512,
     &5.592,4.395,3.536,4.063,6.134,-999.000,
     &8.882,-999.000,14.302,16.987,19.310,20.468,
     &15.151,4.110,0.952,-999.000,-8.431,-999.000,
     &-15.978,-999.000,-20.919,-999.000,-23.936,-999.000,
     &-25.346,-24.720,-22.101,-18.591,-15.844
     &/
       data (fjrocvz(it,17,2, 2), it=1,59) /
     &-18.535,-17.489,-16.590,-15.934,-16.275,-16.981,
     &-19.542,-23.552,-26.213,-27.855,-26.129,-17.987,
     &-9.187,-4.923,0.838,5.620,9.436,15.055,
     &22.777,27.931,30.144,30.056,28.515,25.224,
     &20.114,15.116,10.990,7.979,6.533,5.642,
     &4.429,2.813,1.003,0.399,2.259,-999.000,
     &5.850,-999.000,12.063,13.734,15.975,14.374,
     &10.153,2.214,-4.385,-11.649,-17.514,-21.052,
     &-22.647,-999.000,-23.146,-999.000,-24.969,-999.000,
     &-25.387,-23.907,-22.075,-20.171,-18.535
     &/
       data (fjrocvz(it,18,2, 2), it=1,59) /
     &-19.435,-17.678,-16.310,-15.810,-15.446,-15.300,
     &-18.357,-22.389,-24.057,-25.652,-23.119,-15.820,
     &-6.945,-2.558,1.617,6.485,10.764,17.702,
     &24.702,30.176,33.570,33.543,30.941,26.601,
     &22.028,17.700,13.220,9.223,6.838,5.177,
     &3.743,2.007,0.488,0.542,2.834,-999.000,
     &5.593,7.716,11.302,13.300,13.239,10.630,
     &3.770,-1.187,-8.091,-999.000,-18.513,-999.000,
     &-24.837,-999.000,-26.924,-999.000,-26.791,-999.000,
     &-25.231,-24.271,-22.297,-20.704,-19.435
     &/
       data (fjrocvz(it,19,2, 2), it=1,59) /
     &-18.092,-16.160,-15.830,-15.228,-14.642,-14.059,
     &-16.036,-18.012,-18.803,-20.175,-17.310,-10.115,
     &-1.719,0.830,4.909,9.948,14.075,20.539,
     &28.199,34.025,36.756,36.490,33.551,29.158,
     &24.203,19.290,14.695,10.622,8.573,7.508,
     &5.922,3.982,2.419,2.109,4.094,-999.000,
     &7.635,-999.000,13.013,13.082,12.000,5.732,
     &2.534,-2.848,-9.829,-18.555,-23.486,-26.658,
     &-27.251,-999.000,-27.562,-999.000,-26.077,-999.000,
     &-24.192,-23.396,-21.739,-20.013,-18.092
     &/
       data (fjrocvz(it,20,2, 2), it=1,59) /
     &-20.543,-18.570,-16.678,-14.769,-13.256,-12.286,
     &-13.760,-16.185,-16.482,-16.918,-14.238,-4.878,
     &3.426,3.974,7.681,11.422,14.919,21.604,
     &28.980,35.588,38.577,38.696,36.406,32.576,
     &28.079,23.428,18.619,14.509,11.905,10.200,
     &8.322,6.224,4.318,4.129,6.564,-999.000,
     &9.660,-999.000,13.535,13.315,14.719,12.944,
     &7.330,-2.166,-8.747,-14.544,-19.027,-23.707,
     &-26.181,-999.000,-26.130,-999.000,-26.254,-999.000,
     &-25.114,-24.518,-23.453,-22.028,-20.543
     &/
       data (fjrocvz(it,21,2, 2), it=1,59) /
     &-21.069,-18.578,-16.647,-15.416,-14.693,-13.341,
     &-13.890,-15.975,-17.435,-17.688,-14.673,-7.093,
     &-1.219,2.421,6.631,11.288,15.893,21.223,
     &26.712,32.999,35.793,36.456,34.298,30.287,
     &26.034,21.938,17.725,13.076,9.655,7.503,
     &5.620,4.257,3.270,2.992,4.506,-999.000,
     &7.760,-999.000,13.236,15.347,17.463,16.377,
     &7.594,-999.000,-3.953,-999.000,-15.513,-999.000,
     &-22.363,-999.000,-24.566,-999.000,-24.978,-999.000,
     &-24.406,-23.999,-23.340,-22.470,-21.069
     &/
       data (fjrocvz(it,22,2, 2), it=1,59) /
     &-19.957,-19.121,-18.095,-17.421,-17.732,-17.116,
     &-17.146,-19.134,-22.448,-23.174,-20.073,-13.084,
     &-4.784,-1.926,3.391,8.088,12.614,18.067,
     &24.547,30.260,32.481,32.064,29.575,25.152,
     &19.793,14.841,10.825,7.272,4.378,2.636,
     &1.646,1.806,2.166,2.027,3.321,-999.000,
     &8.158,-999.000,13.357,12.157,13.418,12.488,
     &6.666,-999.000,-6.369,-999.000,-16.209,-999.000,
     &-23.222,-999.000,-23.814,-999.000,-23.504,-999.000,
     &-23.381,-22.784,-21.761,-20.856,-19.957
     &/
       data (fjrocvz(it,23,2, 2), it=1,59) /
     &-17.562,-17.481,-16.208,-14.522,-14.574,-15.649,
     &-17.492,-19.780,-22.044,-22.973,-20.518,-14.405,
     &-9.265,-6.142,-0.639,4.652,9.928,15.435,
     &23.596,29.237,30.913,29.337,26.376,21.864,
     &16.536,11.018,6.931,3.839,1.585,0.493,
     &-0.112,-0.359,0.581,1.876,4.070,-999.000,
     &6.390,-999.000,10.652,11.724,11.499,10.993,
     &7.811,0.415,-5.427,-999.000,-14.412,-999.000,
     &-20.520,-999.000,-22.538,-999.000,-22.494,-999.000,
     &-22.251,-21.247,-19.323,-17.823,-17.562
     &/
       data (fjrocvz(it,24,2, 2), it=1,59) /
     &-15.246,-14.178,-12.711,-11.133,-11.854,-14.060,
     &-16.238,-17.671,-19.186,-20.189,-18.578,-13.855,
     &-7.529,-1.966,1.633,5.084,10.018,18.182,
     &26.368,29.989,30.594,28.899,25.551,20.784,
     &15.803,10.387,5.744,2.437,1.063,1.517,
     &1.589,1.034,1.672,2.912,4.678,-999.000,
     &6.841,-999.000,10.784,11.208,11.583,9.598,
     &4.269,-999.000,-1.205,-999.000,-7.744,-999.000,
     &-13.089,-999.000,-16.887,-999.000,-17.935,-999.000,
     &-18.991,-18.626,-17.224,-16.269,-15.246
     &/
       data (fjrocvz(it,25,2, 2), it=1,59) /
     &-14.469,-13.657,-12.390,-11.791,-12.913,-13.905,
     &-14.781,-15.927,-16.763,-16.339,-13.353,-7.013,
     &-8.195,-3.436,1.893,8.087,13.842,22.742,
     &29.744,32.573,32.235,30.458,27.720,23.318,
     &18.627,13.775,8.090,3.848,2.403,3.287,
     &3.731,3.773,3.724,4.801,6.836,-999.000,
     &9.357,-999.000,12.228,12.889,11.117,5.567,
     &-0.315,-999.000,-1.798,-999.000,-6.742,-999.000,
     &-11.191,-999.000,-13.146,-999.000,-16.561,-999.000,
     &-17.644,-17.499,-17.058,-15.903,-14.469
     &/
       data (fjrocvz(it, 1,3, 2), it=1,59) /
     &-22.044,-23.500,-23.401,-22.552,-21.346,-19.582,
     &-19.258,-21.928,-27.618,-30.081,-23.611,-10.481,
     &0.141,4.321,7.077,9.206,9.285,7.801,
     &7.879,10.242,13.066,14.213,13.691,11.261,
     &8.085,6.885,7.645,8.968,9.292,8.169,
     &7.442,7.278,6.930,5.773,5.592,-999.000,
     &5.325,-999.000,5.993,-999.000,10.631,13.367,
     &13.293,11.138,3.502,-999.000,-0.431,-999.000,
     &-6.894,-999.000,-9.963,-999.000,-12.704,-999.000,
     &-16.184,-18.872,-20.576,-21.108,-22.044
     &/
       data (fjrocvz(it, 2,3, 2), it=1,59) /
     &-19.733,-20.357,-19.952,-18.856,-17.731,-16.971,
     &-18.100,-20.946,-24.720,-26.555,-18.867,-3.755,
     &8.611,11.557,12.613,13.444,13.221,12.408,
     &12.939,14.194,15.015,15.528,14.522,12.445,
     &10.607,10.147,10.870,11.806,12.637,12.266,
     &11.583,11.325,11.158,10.500,9.782,-999.000,
     &8.161,-999.000,9.565,-999.000,12.519,15.476,
     &14.636,12.064,3.946,-999.000,-0.644,-999.000,
     &-7.121,-999.000,-11.527,-999.000,-14.569,-999.000,
     &-17.522,-19.779,-20.514,-19.724,-19.733
     &/
       data (fjrocvz(it, 3,3, 2), it=1,59) /
     &-18.566,-17.384,-16.811,-16.058,-15.478,-14.933,
     &-15.805,-17.338,-20.492,-24.956,-18.256,0.448,
     &17.031,18.491,16.919,15.475,13.854,13.094,
     &13.667,13.979,12.745,11.281,9.635,8.635,
     &8.031,8.011,8.958,10.186,11.602,12.808,
     &12.791,12.466,12.370,11.832,11.254,-999.000,
     &10.983,-999.000,11.070,-999.000,11.438,12.376,
     &10.881,10.825,5.058,-999.000,-2.541,-999.000,
     &-9.309,-999.000,-14.643,-999.000,-17.935,-999.000,
     &-20.950,-23.019,-23.201,-20.830,-18.566
     &/
       data (fjrocvz(it, 4,3, 2), it=1,59) /
     &-20.805,-18.711,-18.152,-18.137,-17.514,-16.091,
     &-15.081,-15.504,-18.477,-21.738,-14.641,5.085,
     &19.391,21.035,19.568,18.470,16.564,14.047,
     &12.164,10.722,9.261,8.158,7.745,7.910,
     &7.570,7.883,8.634,9.994,11.597,12.563,
     &13.027,13.179,12.894,12.011,12.592,-999.000,
     &12.430,-999.000,11.023,-999.000,9.239,11.754,
     &11.455,6.963,3.379,-999.000,-5.258,-999.000,
     &-12.289,-999.000,-17.441,-999.000,-21.164,-999.000,
     &-23.472,-25.108,-25.377,-23.298,-20.805
     &/
       data (fjrocvz(it, 5,3, 2), it=1,59) /
     &-23.424,-22.082,-21.298,-19.568,-17.380,-15.430,
     &-14.367,-14.682,-15.766,-16.594,-10.744,3.426,
     &17.823,22.001,21.581,20.615,18.563,16.222,
     &14.295,12.936,12.428,12.220,11.914,10.906,
     &9.434,8.405,8.026,8.622,9.362,10.090,
     &11.016,11.758,11.864,11.533,11.391,-999.000,
     &11.220,-999.000,10.945,10.146,9.632,5.682,
     &6.663,6.532,0.843,-999.000,-7.281,-999.000,
     &-14.364,-999.000,-19.442,-999.000,-22.948,-999.000,
     &-25.700,-27.425,-27.083,-25.038,-23.424
     &/
       data (fjrocvz(it, 6,3, 2), it=1,59) /
     &-26.025,-25.807,-25.561,-23.221,-19.169,-15.505,
     &-14.417,-14.819,-14.091,-12.646,-8.366,1.823,
     &15.571,19.678,19.227,18.356,17.506,16.612,
     &15.960,16.759,17.582,17.518,16.427,14.902,
     &12.997,11.034,9.697,8.810,8.851,9.142,
     &9.730,10.201,10.797,10.791,10.754,-999.000,
     &11.995,-999.000,11.520,11.589,8.475,5.403,
     &1.325,-999.000,-3.061,-999.000,-10.441,-999.000,
     &-17.824,-999.000,-22.964,-999.000,-26.459,-999.000,
     &-29.293,-30.636,-29.325,-27.275,-26.025
     &/
       data (fjrocvz(it, 7,3, 2), it=1,59) /
     &-29.192,-27.487,-26.993,-26.001,-23.154,-20.059,
     &-18.604,-17.519,-15.730,-12.444,-8.129,-0.909,
     &6.861,10.952,12.151,12.958,13.350,13.569,
     &14.567,16.338,18.122,19.090,19.171,18.718,
     &17.777,16.806,15.154,13.183,11.934,11.161,
     &10.787,10.297,10.061,9.763,9.768,-999.000,
     &9.556,-999.000,8.439,-999.000,2.834,-999.000,
     &-1.269,-999.000,-5.349,-999.000,-11.892,-999.000,
     &-19.291,-999.000,-24.846,-999.000,-28.665,-999.000,
     &-31.359,-32.836,-32.397,-31.006,-29.192
     &/
       data (fjrocvz(it, 8,3, 2), it=1,59) /
     &-25.022,-24.013,-24.886,-25.405,-24.150,-20.052,
     &-15.195,-12.061,-10.424,-8.870,-5.855,-2.017,
     &0.673,2.545,3.820,5.371,6.851,7.917,
     &9.281,10.987,13.716,16.771,18.460,18.797,
     &18.506,17.415,15.642,13.267,11.144,10.405,
     &9.804,8.953,7.824,6.644,5.672,-999.000,
     &4.989,-999.000,3.606,-999.000,0.916,-999.000,
     &-1.656,-999.000,-5.257,-999.000,-10.015,-999.000,
     &-15.710,-999.000,-20.448,-999.000,-24.772,-999.000,
     &-27.870,-28.851,-27.170,-25.793,-25.022
     &/
       data (fjrocvz(it, 9,3, 2), it=1,59) /
     &-17.853,-18.079,-18.323,-18.636,-18.259,-14.814,
     &-10.472,-8.044,-7.149,-7.922,-7.021,-4.577,
     &-4.618,-1.087,3.391,7.547,11.286,11.777,
     &12.560,13.276,13.776,13.053,14.147,15.024,
     &15.619,15.071,13.072,10.285,8.108,6.762,
     &5.434,3.996,2.796,2.501,2.368,-999.000,
     &1.135,-999.000,-0.100,-999.000,0.003,-999.000,
     &0.634,-999.000,0.238,-999.000,-2.963,-999.000,
     &-6.954,-999.000,-10.244,-999.000,-12.552,-999.000,
     &-14.668,-15.978,-15.474,-16.307,-17.853
     &/
       data (fjrocvz(it,10,3, 2), it=1,59) /
     &-14.865,-15.281,-14.434,-13.362,-13.774,-14.265,
     &-13.608,-12.205,-11.235,-11.501,-11.480,-12.759,
     &-14.824,-6.756,1.765,7.449,13.159,14.045,
     &15.012,15.639,16.151,16.000,15.691,15.433,
     &15.132,14.772,13.016,10.577,8.076,5.669,
     &3.291,0.610,-1.457,-2.205,-3.102,-999.000,
     &-4.653,-999.000,-4.513,-999.000,-3.649,-999.000,
     &-0.332,-999.000,1.680,-999.000,2.027,-999.000,
     &0.404,-999.000,-1.854,-999.000,-3.990,-999.000,
     &-5.871,-8.728,-10.686,-12.774,-14.865
     &/
       data (fjrocvz(it,11,3, 2), it=1,59) /
     &-19.101,-20.585,-20.854,-20.150,-19.865,-18.854,
     &-17.194,-16.705,-19.289,-18.128,-17.721,-11.849,
     &-15.347,-9.008,-2.706,3.783,10.302,12.397,
     &14.551,16.281,19.073,20.793,20.088,19.150,
     &18.251,17.199,15.896,14.290,11.763,8.647,
     &5.416,2.524,0.421,-1.606,-3.512,-999.000,
     &-4.771,-999.000,-4.762,-999.000,-4.103,-999.000,
     &1.408,5.144,6.108,4.182,2.600,-999.000,
     &1.315,-999.000,0.339,-999.000,-1.426,-999.000,
     &-3.933,-9.760,-13.865,-16.621,-19.101
     &/
       data (fjrocvz(it,12,3, 2), it=1,59) /
     &-20.635,-22.601,-24.954,-26.689,-26.452,-24.464,
     &-24.357,-28.554,-33.397,-31.275,-24.891,-15.321,
     &-14.581,-11.384,-7.955,-1.870,4.213,8.060,
     &11.356,15.654,20.135,21.921,21.931,21.325,
     &20.371,18.935,17.421,16.400,14.323,10.928,
     &7.883,6.069,4.225,2.506,0.439,-999.000,
     &-1.628,-999.000,-0.552,-999.000,3.932,7.545,
     &10.019,11.821,10.188,-999.000,6.571,-999.000,
     &3.061,-999.000,1.673,-999.000,-0.258,-999.000,
     &-3.526,-9.628,-14.896,-18.084,-20.635
     &/
       data (fjrocvz(it,13,3, 2), it=1,59) /
     &-18.804,-20.517,-21.223,-22.265,-24.341,-26.173,
     &-28.610,-31.454,-33.996,-32.481,-25.548,-17.902,
     &-14.238,-13.285,-10.238,-6.844,-3.692,3.320,
     &11.531,17.559,21.877,23.963,24.152,22.915,
     &21.569,20.381,18.935,17.466,15.569,12.815,
     &10.582,8.141,6.059,4.918,3.298,-999.000,
     &2.806,-999.000,5.200,-999.000,7.895,11.265,
     &15.627,17.620,13.378,-999.000,9.386,-999.000,
     &5.221,-999.000,1.793,-999.000,-0.361,-999.000,
     &-2.790,-7.028,-11.493,-15.493,-18.804
     &/
       data (fjrocvz(it,14,3, 2), it=1,59) /
     &-18.514,-20.768,-20.853,-21.771,-24.764,-28.648,
     &-31.006,-33.260,-36.706,-35.911,-27.378,-16.804,
     &-12.295,-11.931,-10.435,-7.872,-5.330,1.338,
     &10.657,16.636,21.688,24.127,24.362,22.706,
     &21.450,20.786,19.979,18.871,17.779,15.596,
     &13.266,10.688,8.715,7.231,6.524,-999.000,
     &6.396,-999.000,8.358,-999.000,11.656,14.074,
     &19.403,20.398,18.001,13.142,10.112,-999.000,
     &5.221,-999.000,2.627,-999.000,0.041,-999.000,
     &-2.604,-6.373,-10.186,-14.742,-18.514
     &/
       data (fjrocvz(it,15,3, 2), it=1,59) /
     &-18.135,-20.162,-20.619,-21.226,-23.604,-27.336,
     &-30.293,-33.768,-38.027,-39.300,-31.398,-19.979,
     &-13.698,-12.983,-12.269,-9.956,-6.432,0.032,
     &8.066,14.039,18.797,21.145,20.638,18.648,
     &17.040,15.913,15.168,14.056,12.657,11.135,
     &9.787,8.343,6.329,5.267,4.927,-999.000,
     &4.195,-999.000,6.158,-999.000,9.629,13.064,
     &15.680,14.064,10.591,7.939,5.887,-999.000,
     &1.679,-999.000,-0.942,-999.000,-2.591,-999.000,
     &-5.075,-8.401,-11.691,-14.939,-18.135
     &/
       data (fjrocvz(it,16,3, 2), it=1,59) /
     &-16.305,-17.928,-18.105,-18.565,-21.398,-25.875,
     &-30.366,-34.249,-39.635,-42.414,-38.082,-25.534,
     &-15.846,-12.820,-11.914,-9.565,-5.587,0.296,
     &7.352,12.816,16.734,18.001,16.726,15.004,
     &13.381,11.642,9.699,7.123,4.753,3.109,
     &2.373,1.292,-0.291,-1.553,-1.882,-999.000,
     &-1.492,-999.000,0.098,-999.000,2.852,6.121,
     &7.070,2.907,1.019,-999.000,-3.412,-999.000,
     &-6.460,-999.000,-8.558,-999.000,-9.035,-999.000,
     &-9.433,-11.093,-12.459,-14.216,-16.305
     &/
       data (fjrocvz(it,17,3, 2), it=1,59) /
     &-17.400,-18.451,-18.053,-17.880,-19.968,-23.451,
     &-27.837,-34.435,-42.093,-45.008,-39.852,-24.804,
     &-13.463,-10.827,-8.891,-5.642,-1.514,3.456,
     &9.685,15.151,18.643,19.695,18.705,16.369,
     &13.930,11.728,9.386,6.298,3.274,0.844,
     &-1.268,-2.637,-3.896,-5.184,-5.962,-999.000,
     &-6.982,-999.000,-6.073,-999.000,-3.428,1.662,
     &3.200,0.002,-4.656,-999.000,-9.043,-999.000,
     &-12.565,-999.000,-14.044,-999.000,-14.347,-999.000,
     &-14.282,-15.321,-15.919,-16.405,-17.400
     &/
       data (fjrocvz(it,18,3, 2), it=1,59) /
     &-24.904,-24.288,-22.679,-20.871,-20.451,-22.081,
     &-24.808,-31.699,-41.727,-44.564,-37.681,-16.163,
     &-7.669,-6.805,-4.988,-1.647,2.429,8.211,
     &14.644,19.713,22.341,23.311,22.764,20.870,
     &17.982,14.825,12.207,9.020,6.045,2.953,
     &0.225,-1.739,-3.596,-5.287,-7.010,-999.000,
     &-8.460,-999.000,-7.180,-999.000,-5.191,-2.256,
     &-3.338,-6.332,-9.256,-999.000,-14.420,-999.000,
     &-17.885,-999.000,-18.923,-999.000,-19.092,-999.000,
     &-19.087,-20.632,-22.129,-23.460,-24.904
     &/
       data (fjrocvz(it,19,3, 2), it=1,59) /
     &-27.504,-27.105,-25.329,-23.252,-22.496,-24.389,
     &-25.675,-28.392,-34.772,-36.364,-30.698,-14.164,
     &-5.898,-4.230,-2.605,0.288,4.781,10.971,
     &17.473,22.374,25.630,26.327,25.604,24.031,
     &21.748,18.976,15.858,12.753,9.848,6.636,
     &4.170,2.393,0.598,-1.971,-4.008,-999.000,
     &-4.540,-999.000,-4.064,-999.000,-2.737,-1.416,
     &-3.373,-6.131,-10.376,-999.000,-14.470,-999.000,
     &-16.967,-999.000,-19.497,-999.000,-20.283,-999.000,
     &-20.322,-21.721,-23.183,-25.546,-27.504
     &/
       data (fjrocvz(it,20,3, 2), it=1,59) /
     &-25.166,-25.450,-24.226,-22.449,-22.525,-25.235,
     &-27.907,-30.412,-32.094,-29.924,-23.572,-13.936,
     &-6.734,-3.413,-1.757,0.273,4.046,10.680,
     &17.627,23.998,27.787,28.368,27.093,24.932,
     &22.845,20.850,19.317,17.915,15.554,12.396,
     &9.465,7.198,4.583,2.206,0.743,-999.000,
     &-0.241,-999.000,0.441,-999.000,2.096,3.909,
     &2.653,-0.054,-4.597,-999.000,-8.788,-999.000,
     &-10.979,-999.000,-13.721,-999.000,-15.267,-999.000,
     &-17.221,-19.062,-20.516,-22.724,-25.166
     &/
       data (fjrocvz(it,21,3, 2), it=1,59) /
     &-21.913,-23.129,-22.446,-21.428,-22.500,-26.126,
     &-29.094,-31.103,-34.302,-36.151,-32.851,-22.551,
     &-11.340,-6.268,-4.040,-2.099,0.950,6.149,
     &12.587,18.340,22.007,22.873,22.305,21.408,
     &20.215,18.899,18.120,16.760,14.843,12.055,
     &9.731,7.540,5.545,4.332,3.454,-999.000,
     &2.881,-999.000,2.614,-999.000,2.229,3.105,
     &4.434,3.764,0.538,-999.000,-3.482,-999.000,
     &-4.673,-999.000,-8.191,-999.000,-10.553,-999.000,
     &-12.941,-15.138,-16.730,-19.128,-21.913
     &/
       data (fjrocvz(it,22,3, 2), it=1,59) /
     &-18.952,-19.144,-18.186,-18.348,-20.942,-24.252,
     &-26.798,-30.430,-36.485,-40.046,-37.444,-27.056,
     &-15.408,-7.826,-5.725,-3.934,-1.455,1.982,
     &6.594,10.903,14.222,14.919,14.444,14.273,
     &14.019,13.215,12.294,10.778,8.764,6.790,
     &5.167,4.095,3.313,3.207,4.070,-999.000,
     &4.110,-999.000,3.453,-999.000,4.021,4.278,
     &7.849,8.251,2.767,-999.000,-1.866,-999.000,
     &-4.097,-999.000,-6.429,-999.000,-8.792,-999.000,
     &-11.075,-13.680,-15.791,-17.821,-18.952
     &/
       data (fjrocvz(it,23,3, 2), it=1,59) /
     &-20.221,-20.089,-18.747,-18.276,-20.955,-24.448,
     &-27.759,-30.890,-34.380,-35.807,-31.884,-22.024,
     &-10.900,-5.387,-4.607,-3.412,-1.326,0.630,
     &3.648,7.904,11.755,13.346,12.430,10.671,
     &8.989,7.393,6.410,5.405,3.882,2.356,
     &1.920,1.646,1.085,1.044,2.185,-999.000,
     &2.088,-999.000,3.153,-999.000,5.837,9.173,
     &11.878,11.986,6.401,-999.000,-0.087,-999.000,
     &-3.943,-999.000,-6.436,-999.000,-9.132,-999.000,
     &-12.196,-15.123,-17.399,-19.188,-20.221
     &/
       data (fjrocvz(it,24,3, 2), it=1,59) /
     &-22.619,-23.630,-22.472,-20.843,-21.056,-22.445,
     &-24.384,-26.753,-30.829,-32.145,-27.780,-17.125,
     &-6.840,-2.066,-0.247,1.359,2.795,3.270,
     &5.015,8.070,11.228,12.300,11.691,10.183,
     &7.457,5.468,4.525,5.137,5.128,4.442,
     &4.347,4.056,3.576,2.799,2.800,-999.000,
     &3.657,-999.000,5.402,-999.000,8.908,12.532,
     &12.313,10.528,8.283,-999.000,-0.125,-999.000,
     &-6.007,-999.000,-7.882,-999.000,-11.166,-999.000,
     &-15.192,-18.627,-20.592,-21.583,-22.619
     &/
       data (fjrocvz(it,25,3, 2), it=1,59) /
     &-22.044,-23.500,-23.401,-22.552,-21.346,-19.582,
     &-19.258,-21.928,-27.618,-30.081,-23.611,-10.481,
     &0.141,4.321,7.077,9.206,9.285,7.801,
     &7.879,10.242,13.066,14.213,13.691,11.261,
     &8.085,6.885,7.645,8.968,9.292,8.169,
     &7.442,7.278,6.930,5.773,5.592,-999.000,
     &5.325,-999.000,5.993,-999.000,10.631,13.367,
     &13.293,11.138,3.502,-999.000,-0.431,-999.000,
     &-6.894,-999.000,-9.963,-999.000,-12.704,-999.000,
     &-16.184,-18.872,-20.576,-21.108,-22.044
     &/
       data (fjrocvz(it, 1,4, 2), it=1,59) /
     &-14.469,-13.657,-12.390,-11.791,-12.913,-13.905,
     &-14.781,-15.927,-16.763,-16.339,-13.353,-7.013,
     &-8.195,-3.436,1.893,8.087,13.842,22.742,
     &29.744,32.573,32.235,30.458,27.720,23.318,
     &18.627,13.775,8.090,3.848,2.403,3.287,
     &3.731,3.773,3.724,4.801,6.836,-999.000,
     &9.357,-999.000,12.228,12.889,11.117,5.567,
     &-0.315,-999.000,-1.798,-999.000,-6.742,-999.000,
     &-11.191,-999.000,-13.146,-999.000,-16.561,-999.000,
     &-17.644,-17.499,-17.058,-15.903,-14.469
     &/
       data (fjrocvz(it, 2,4, 2), it=1,59) /
     &-12.791,-12.125,-11.524,-11.319,-10.887,-10.180,
     &-10.575,-12.537,-13.980,-13.089,-7.613,0.281,
     &6.080,9.105,13.311,15.914,19.783,25.045,
     &30.475,34.126,34.844,33.348,30.253,25.658,
     &20.820,16.717,12.605,8.560,5.776,5.357,
     &6.242,6.867,7.193,8.100,9.057,-999.000,
     &10.357,-999.000,13.950,14.432,14.332,12.151,
     &6.635,-999.000,-1.698,-999.000,-7.885,-999.000,
     &-9.091,-999.000,-12.940,-999.000,-16.747,-999.000,
     &-17.121,-17.200,-15.976,-13.868,-12.791
     &/
       data (fjrocvz(it, 3,4, 2), it=1,59) /
     &-12.899,-11.690,-11.074,-10.187,-9.407,-9.214,
     &-8.993,-9.329,-9.906,-9.395,-4.823,2.010,
     &8.026,11.174,13.518,16.551,20.087,23.216,
     &26.520,29.083,30.240,28.848,25.555,21.362,
     &17.316,14.126,11.351,8.846,6.477,5.788,
     &6.635,7.524,8.397,8.693,9.805,-999.000,
     &12.088,-999.000,14.202,17.556,19.372,14.188,
     &4.712,-0.176,-2.484,-7.832,-10.582,-10.260,
     &-10.600,-999.000,-16.475,-999.000,-18.713,-999.000,
     &-19.102,-18.010,-16.511,-14.611,-12.899
     &/
       data (fjrocvz(it, 4,4, 2), it=1,59) /
     &-14.191,-13.050,-11.548,-10.175,-9.966,-9.584,
     &-8.791,-7.300,-6.222,-5.618,-2.534,3.178,
     &9.168,11.457,12.628,15.173,18.564,22.398,
     &25.672,27.468,26.762,24.150,20.854,17.119,
     &13.381,10.233,7.909,6.401,5.428,5.291,
     &5.963,7.180,8.068,9.429,11.952,-999.000,
     &13.582,-999.000,16.488,20.079,19.807,14.268,
     &8.004,4.737,-2.357,-11.705,-13.271,-999.000,
     &-12.670,-999.000,-17.862,-999.000,-20.529,-999.000,
     &-19.908,-17.882,-16.590,-15.095,-14.191
     &/
       data (fjrocvz(it, 5,4, 2), it=1,59) /
     &-15.076,-13.913,-11.779,-10.271,-10.349,-10.274,
     &-9.738,-8.127,-6.606,-5.776,-2.804,3.386,
     &9.344,11.828,14.238,16.093,19.218,23.593,
     &27.468,29.522,28.790,26.164,22.346,18.200,
     &14.116,11.265,9.390,8.355,7.250,6.729,
     &7.251,8.490,9.932,11.117,12.189,-999.000,
     &15.533,-999.000,19.137,20.971,21.725,20.327,
     &15.043,7.068,-1.715,-8.363,-11.889,-14.843,
     &-17.532,-21.808,-23.788,-23.478,-22.087,-999.000,
     &-20.482,-18.339,-16.738,-15.625,-15.076
     &/
       data (fjrocvz(it, 6,4, 2), it=1,59) /
     &-15.950,-15.140,-13.750,-12.478,-12.262,-11.061,
     &-9.322,-7.896,-6.103,-4.444,-2.423,-1.228,
     &1.809,6.799,12.783,17.278,20.031,24.236,
     &28.135,30.150,30.498,28.807,25.934,22.072,
     &18.535,16.038,13.943,12.091,10.395,9.098,
     &8.819,10.169,11.907,12.147,13.165,-999.000,
     &16.835,-999.000,20.853,24.324,27.979,27.911,
     &23.725,14.391,4.263,-4.781,-10.341,-16.687,
     &-19.677,-999.000,-23.580,-999.000,-23.610,-999.000,
     &-21.947,-18.953,-17.393,-16.742,-15.950
     &/
       data (fjrocvz(it, 7,4, 2), it=1,59) /
     &-17.274,-16.651,-15.567,-14.560,-13.777,-11.196,
     &-8.956,-7.304,-4.786,-3.983,-1.988,0.189,
     &5.078,6.977,10.383,13.248,16.021,20.906,
     &25.629,29.097,29.680,29.216,27.688,25.144,
     &22.354,19.695,17.363,15.307,13.014,10.839,
     &10.047,10.709,11.960,12.387,12.701,-999.000,
     &15.225,-999.000,18.864,24.090,29.080,33.195,
     &27.731,14.871,3.747,-3.648,-10.277,-18.638,
     &-23.488,-25.358,-25.748,-25.562,-24.726,-999.000,
     &-22.877,-20.914,-19.497,-17.927,-17.274
     &/
       data (fjrocvz(it, 8,4, 2), it=1,59) /
     &-16.748,-15.082,-14.017,-13.415,-12.425,-11.003,
     &-9.877,-8.717,-6.436,-8.780,-10.871,-12.879,
     &-8.150,-1.682,3.510,8.568,10.942,14.253,
     &19.921,24.253,26.028,26.219,25.604,24.219,
     &22.311,20.444,18.147,15.932,13.281,10.839,
     &9.755,10.065,11.212,11.589,11.736,-999.000,
     &14.333,-999.000,18.358,21.606,26.079,26.748,
     &20.804,13.261,4.672,-3.366,-9.739,-16.735,
     &-20.927,-21.897,-20.371,-999.000,-21.091,-999.000,
     &-20.782,-20.656,-19.749,-18.119,-16.748
     &/
       data (fjrocvz(it, 9,4, 2), it=1,59) /
     &-13.642,-11.240,-9.198,-7.812,-7.428,-7.770,
     &-9.058,-10.223,-10.746,-12.868,-15.782,-18.266,
     &-13.706,-4.526,3.084,7.901,9.195,12.521,
     &17.426,20.895,22.254,22.759,22.398,20.863,
     &18.493,16.228,14.348,12.382,10.092,8.768,
     &8.667,9.216,10.289,10.636,11.335,-999.000,
     &12.943,-999.000,16.940,21.353,24.472,27.665,
     &27.982,21.420,10.634,1.495,-4.182,-999.000,
     &-8.583,-999.000,-12.323,-999.000,-15.418,-999.000,
     &-16.258,-16.560,-16.998,-15.894,-13.642
     &/
       data (fjrocvz(it,10,4, 2), it=1,59) /
     &-13.146,-10.782,-8.401,-6.989,-5.792,-6.841,
     &-10.075,-12.665,-13.741,-14.670,-15.604,-15.897,
     &-12.475,-2.462,5.182,9.905,11.525,15.492,
     &20.563,23.683,23.913,23.137,21.628,19.814,
     &17.369,14.584,12.233,10.144,8.613,8.026,
     &8.065,8.370,8.289,8.333,9.567,-999.000,
     &11.386,-999.000,17.814,23.030,28.427,31.581,
     &31.217,21.913,12.660,-999.000,5.006,-999.000,
     &-1.803,-999.000,-7.590,-999.000,-9.283,-999.000,
     &-10.912,-12.395,-13.893,-13.856,-13.146
     &/
       data (fjrocvz(it,11,4, 2), it=1,59) /
     &-15.256,-14.592,-14.284,-14.036,-13.438,-14.142,
     &-15.013,-16.593,-18.118,-19.805,-19.199,-15.413,
     &-10.498,-1.263,5.911,11.567,15.434,18.386,
     &22.583,26.047,27.083,26.550,25.299,23.328,
     &21.100,18.346,15.165,12.825,10.783,9.784,
     &9.101,8.293,7.708,7.760,9.578,-999.000,
     &13.152,-999.000,18.561,24.009,31.052,31.323,
     &26.434,18.516,10.626,-999.000,0.985,-999.000,
     &-6.246,-999.000,-10.018,-999.000,-10.826,-999.000,
     &-13.336,-13.616,-14.984,-16.016,-15.256
     &/
       data (fjrocvz(it,12,4, 2), it=1,59) /
     &-17.805,-18.721,-18.692,-18.569,-18.812,-18.452,
     &-18.168,-19.869,-22.685,-23.855,-24.084,-20.359,
     &-15.900,-6.874,2.616,8.292,12.731,17.237,
     &23.635,28.288,29.796,29.168,27.801,26.272,
     &24.216,20.939,17.466,14.615,12.563,11.742,
     &10.757,9.468,8.480,8.383,10.447,-999.000,
     &14.038,-999.000,19.534,24.814,31.612,34.211,
     &29.053,16.113,6.631,0.093,-5.203,-999.000,
     &-8.063,-999.000,-10.004,-999.000,-13.864,-999.000,
     &-16.168,-16.535,-17.178,-17.635,-17.805
     &/
       data (fjrocvz(it,13,4, 2), it=1,59) /
     &-18.526,-18.150,-17.982,-18.053,-17.768,-18.440,
     &-19.959,-21.268,-22.907,-22.548,-21.820,-19.501,
     &-15.665,-7.736,0.888,7.433,11.998,17.114,
     &25.984,31.417,31.807,30.775,29.465,28.009,
     &25.632,21.957,18.542,16.075,14.664,14.264,
     &13.455,12.086,10.545,10.222,11.855,-999.000,
     &14.559,-999.000,22.288,26.748,33.383,37.173,
     &29.959,16.715,3.242,-3.176,-5.089,-6.714,
     &-7.435,-999.000,-10.539,-999.000,-13.939,-999.000,
     &-16.578,-18.871,-19.050,-18.517,-18.526
     &/
       data (fjrocvz(it,14,4, 2), it=1,59) /
     &-17.508,-17.119,-16.810,-16.251,-16.246,-17.827,
     &-20.529,-24.098,-27.364,-26.154,-22.514,-17.703,
     &-12.678,-10.112,-3.430,4.167,9.628,15.527,
     &24.558,29.897,31.061,30.291,28.830,27.070,
     &24.282,21.257,18.617,16.815,15.776,15.220,
     &14.085,12.666,11.046,10.792,12.682,-999.000,
     &16.384,-999.000,23.886,28.721,31.846,32.214,
     &25.999,14.369,6.830,-999.000,-1.960,-999.000,
     &-10.080,-999.000,-13.879,-999.000,-16.879,-999.000,
     &-19.910,-20.796,-20.003,-18.446,-17.508
     &/
       data (fjrocvz(it,15,4, 2), it=1,59) /
     &-15.742,-15.096,-14.375,-13.834,-13.685,-14.898,
     &-18.578,-24.516,-28.428,-29.139,-25.346,-17.581,
     &-12.046,-8.261,-3.399,3.672,9.209,14.656,
     &21.992,26.902,28.425,27.441,25.351,22.493,
     &19.377,16.860,14.534,12.859,11.702,10.725,
     &9.738,8.645,7.949,8.495,10.967,-999.000,
     &13.999,-999.000,20.046,23.459,25.456,27.106,
     &22.102,11.096,4.334,-999.000,-5.594,-999.000,
     &-12.611,-999.000,-17.536,-999.000,-21.145,-999.000,
     &-23.237,-23.461,-21.075,-17.525,-15.742
     &/
       data (fjrocvz(it,16,4, 2), it=1,59) /
     &-15.844,-14.300,-14.313,-14.293,-14.559,-16.022,
     &-19.401,-23.750,-26.689,-29.463,-28.301,-20.585,
     &-11.007,-7.377,-2.380,3.484,8.275,13.670,
     &20.969,25.997,27.684,26.997,25.279,21.645,
     &17.106,12.998,10.082,8.219,7.145,6.512,
     &5.592,4.395,3.536,4.063,6.134,-999.000,
     &8.882,-999.000,14.302,16.987,19.310,20.468,
     &15.151,4.110,0.952,-999.000,-8.431,-999.000,
     &-15.978,-999.000,-20.919,-999.000,-23.936,-999.000,
     &-25.346,-24.720,-22.101,-18.591,-15.844
     &/
       data (fjrocvz(it,17,4, 2), it=1,59) /
     &-18.535,-17.489,-16.590,-15.934,-16.275,-16.981,
     &-19.542,-23.552,-26.213,-27.855,-26.129,-17.987,
     &-9.187,-4.923,0.838,5.620,9.436,15.055,
     &22.777,27.931,30.144,30.056,28.515,25.224,
     &20.114,15.116,10.990,7.979,6.533,5.642,
     &4.429,2.813,1.003,0.399,2.259,-999.000,
     &5.850,-999.000,12.063,13.734,15.975,14.374,
     &10.153,2.214,-4.385,-11.649,-17.514,-21.052,
     &-22.647,-999.000,-23.146,-999.000,-24.969,-999.000,
     &-25.387,-23.907,-22.075,-20.171,-18.535
     &/
       data (fjrocvz(it,18,4, 2), it=1,59) /
     &-19.435,-17.678,-16.310,-15.810,-15.446,-15.300,
     &-18.357,-22.389,-24.057,-25.652,-23.119,-15.820,
     &-6.945,-2.558,1.617,6.485,10.764,17.702,
     &24.702,30.176,33.570,33.543,30.941,26.601,
     &22.028,17.700,13.220,9.223,6.838,5.177,
     &3.743,2.007,0.488,0.542,2.834,-999.000,
     &5.593,7.716,11.302,13.300,13.239,10.630,
     &3.770,-1.187,-8.091,-999.000,-18.513,-999.000,
     &-24.837,-999.000,-26.924,-999.000,-26.791,-999.000,
     &-25.231,-24.271,-22.297,-20.704,-19.435
     &/
       data (fjrocvz(it,19,4, 2), it=1,59) /
     &-18.092,-16.160,-15.830,-15.228,-14.642,-14.059,
     &-16.036,-18.012,-18.803,-20.175,-17.310,-10.115,
     &-1.719,0.830,4.909,9.948,14.075,20.539,
     &28.199,34.025,36.756,36.490,33.551,29.158,
     &24.203,19.290,14.695,10.622,8.573,7.508,
     &5.922,3.982,2.419,2.109,4.094,-999.000,
     &7.635,-999.000,13.013,13.082,12.000,5.732,
     &2.534,-2.848,-9.829,-18.555,-23.486,-26.658,
     &-27.251,-999.000,-27.562,-999.000,-26.077,-999.000,
     &-24.192,-23.396,-21.739,-20.013,-18.092
     &/
       data (fjrocvz(it,20,4, 2), it=1,59) /
     &-20.543,-18.570,-16.678,-14.769,-13.256,-12.286,
     &-13.760,-16.185,-16.482,-16.918,-14.238,-4.878,
     &3.426,3.974,7.681,11.422,14.919,21.604,
     &28.980,35.588,38.577,38.696,36.406,32.576,
     &28.079,23.428,18.619,14.509,11.905,10.200,
     &8.322,6.224,4.318,4.129,6.564,-999.000,
     &9.660,-999.000,13.535,13.315,14.719,12.944,
     &7.330,-2.166,-8.747,-14.544,-19.027,-23.707,
     &-26.181,-999.000,-26.130,-999.000,-26.254,-999.000,
     &-25.114,-24.518,-23.453,-22.028,-20.543
     &/
       data (fjrocvz(it,21,4, 2), it=1,59) /
     &-21.069,-18.578,-16.647,-15.416,-14.693,-13.341,
     &-13.890,-15.975,-17.435,-17.688,-14.673,-7.093,
     &-1.219,2.421,6.631,11.288,15.893,21.223,
     &26.712,32.999,35.793,36.456,34.298,30.287,
     &26.034,21.938,17.725,13.076,9.655,7.503,
     &5.620,4.257,3.270,2.992,4.506,-999.000,
     &7.760,-999.000,13.236,15.347,17.463,16.377,
     &7.594,-999.000,-3.953,-999.000,-15.513,-999.000,
     &-22.363,-999.000,-24.566,-999.000,-24.978,-999.000,
     &-24.406,-23.999,-23.340,-22.470,-21.069
     &/
       data (fjrocvz(it,22,4, 2), it=1,59) /
     &-19.957,-19.121,-18.095,-17.421,-17.732,-17.116,
     &-17.146,-19.134,-22.448,-23.174,-20.073,-13.084,
     &-4.784,-1.926,3.391,8.088,12.614,18.067,
     &24.547,30.260,32.481,32.064,29.575,25.152,
     &19.793,14.841,10.825,7.272,4.378,2.636,
     &1.646,1.806,2.166,2.027,3.321,-999.000,
     &8.158,-999.000,13.357,12.157,13.418,12.488,
     &6.666,-999.000,-6.369,-999.000,-16.209,-999.000,
     &-23.222,-999.000,-23.814,-999.000,-23.504,-999.000,
     &-23.381,-22.784,-21.761,-20.856,-19.957
     &/
       data (fjrocvz(it,23,4, 2), it=1,59) /
     &-17.562,-17.481,-16.208,-14.522,-14.574,-15.649,
     &-17.492,-19.780,-22.044,-22.973,-20.518,-14.405,
     &-9.265,-6.142,-0.639,4.652,9.928,15.435,
     &23.596,29.237,30.913,29.337,26.376,21.864,
     &16.536,11.018,6.931,3.839,1.585,0.493,
     &-0.112,-0.359,0.581,1.876,4.070,-999.000,
     &6.390,-999.000,10.652,11.724,11.499,10.993,
     &7.811,0.415,-5.427,-999.000,-14.412,-999.000,
     &-20.520,-999.000,-22.538,-999.000,-22.494,-999.000,
     &-22.251,-21.247,-19.323,-17.823,-17.562
     &/
       data (fjrocvz(it,24,4, 2), it=1,59) /
     &-15.246,-14.178,-12.711,-11.133,-11.854,-14.060,
     &-16.238,-17.671,-19.186,-20.189,-18.578,-13.855,
     &-7.529,-1.966,1.633,5.084,10.018,18.182,
     &26.368,29.989,30.594,28.899,25.551,20.784,
     &15.803,10.387,5.744,2.437,1.063,1.517,
     &1.589,1.034,1.672,2.912,4.678,-999.000,
     &6.841,-999.000,10.784,11.208,11.583,9.598,
     &4.269,-999.000,-1.205,-999.000,-7.744,-999.000,
     &-13.089,-999.000,-16.887,-999.000,-17.935,-999.000,
     &-18.991,-18.626,-17.224,-16.269,-15.246
     &/
       data (fjrocvz(it,25,4, 2), it=1,59) /
     &-14.469,-13.657,-12.390,-11.791,-12.913,-13.905,
     &-14.781,-15.927,-16.763,-16.339,-13.353,-7.013,
     &-8.195,-3.436,1.893,8.087,13.842,22.742,
     &29.744,32.573,32.235,30.458,27.720,23.318,
     &18.627,13.775,8.090,3.848,2.403,3.287,
     &3.731,3.773,3.724,4.801,6.836,-999.000,
     &9.357,-999.000,12.228,12.889,11.117,5.567,
     &-0.315,-999.000,-1.798,-999.000,-6.742,-999.000,
     &-11.191,-999.000,-13.146,-999.000,-16.561,-999.000,
     &-17.644,-17.499,-17.058,-15.903,-14.469
     &/
       data (fjrocvz(it, 1,1, 3), it=1,59) /
     &-11.521,-11.160,-10.952,-11.214,-11.842,-12.343,
     &-13.707,-15.192,-16.758,-18.265,-17.135,-10.871,
     &-1.300,6.249,12.120,16.441,20.810,25.377,
     &29.418,31.908,31.499,29.754,27.434,24.466,
     &21.354,17.086,13.030,9.894,6.413,2.842,
     &0.716,0.045,-0.119,-0.520,-0.917,-999.000,
     &-1.414,-999.000,-0.562,-999.000,1.409,-999.000,
     &0.817,-999.000,-0.602,-999.000,-4.023,-999.000,
     &-7.959,-999.000,-11.120,-999.000,-12.538,-999.000,
     &-14.610,-15.494,-14.728,-12.775,-11.521
     &/
       data (fjrocvz(it, 2,1, 3), it=1,59) /
     &-13.321,-12.416,-12.201,-12.281,-13.502,-14.554,
     &-15.084,-15.982,-16.884,-17.964,-14.380,-4.868,
     &3.974,10.789,16.330,22.178,27.157,30.789,
     &34.586,36.947,36.798,34.913,31.713,28.045,
     &23.889,18.851,14.021,10.160,6.424,2.669,
     &-0.484,-2.473,-3.740,-3.860,-4.033,-999.000,
     &-3.472,-999.000,-2.164,-999.000,-2.939,-999.000,
     &-1.852,-999.000,-3.659,-999.000,-6.977,-999.000,
     &-10.795,-999.000,-14.080,-999.000,-15.491,-999.000,
     &-16.490,-16.913,-16.433,-14.954,-13.321
     &/
       data (fjrocvz(it, 3,1, 3), it=1,59) /
     &-14.603,-14.089,-14.020,-14.471,-15.452,-15.296,
     &-14.431,-14.499,-14.318,-13.681,-8.117,0.294,
     &7.248,12.684,17.941,23.166,26.487,29.554,
     &33.198,36.174,36.247,33.923,30.308,26.833,
     &23.755,20.050,16.043,11.959,8.072,3.924,
     &0.498,-1.438,-2.852,-2.579,-2.587,-999.000,
     &-1.768,-999.000,-1.389,-999.000,-1.695,0.628,
     &2.420,1.472,-1.481,-2.687,-5.957,-999.000,
     &-9.488,-999.000,-12.387,-999.000,-13.677,-999.000,
     &-15.843,-16.785,-16.974,-15.808,-14.603
     &/
       data (fjrocvz(it, 4,1, 3), it=1,59) /
     &-14.417,-15.229,-15.335,-14.540,-13.709,-13.668,
     &-13.613,-13.811,-12.562,-10.003,-3.044,4.963,
     &9.688,13.514,18.717,22.828,25.999,28.949,
     &32.480,34.647,33.584,30.351,26.560,23.719,
     &21.166,17.314,13.415,10.475,8.390,5.529,
     &2.485,-0.001,-1.666,-1.300,0.184,-999.000,
     &0.532,-999.000,1.545,-999.000,0.473,3.365,
     &5.679,3.252,-0.826,-999.000,-3.970,-999.000,
     &-6.718,-999.000,-9.828,-999.000,-12.815,-999.000,
     &-15.321,-16.032,-15.538,-14.127,-14.417
     &/
       data (fjrocvz(it, 5,1, 3), it=1,59) /
     &-11.170,-11.829,-11.867,-10.835,-10.138,-11.043,
     &-12.828,-13.915,-12.843,-10.316,-2.973,5.687,
     &9.682,12.857,17.855,22.349,25.273,28.077,
     &31.406,33.575,32.971,30.166,27.217,24.527,
     &21.386,16.803,12.695,10.392,9.138,7.480,
     &5.359,2.572,0.385,-0.217,0.626,-999.000,
     &1.810,-999.000,4.025,-999.000,4.065,7.568,
     &7.747,3.875,1.031,-999.000,-2.440,-999.000,
     &-5.439,-999.000,-9.066,-999.000,-12.875,-999.000,
     &-14.740,-14.633,-13.252,-11.314,-11.170
     &/
       data (fjrocvz(it, 6,1, 3), it=1,59) /
     &-9.782,-10.244,-10.837,-11.455,-11.592,-12.372,
     &-14.228,-15.483,-14.684,-11.716,-5.510,1.993,
     &5.588,7.944,11.446,15.140,17.957,21.686,
     &26.029,29.544,29.987,28.462,26.462,24.300,
     &21.503,17.831,14.606,12.037,10.429,9.304,
     &8.437,6.805,3.884,2.027,2.116,-999.000,
     &3.167,-999.000,4.956,-999.000,6.867,-999.000,
     &6.161,-999.000,3.385,-999.000,1.017,-999.000,
     &-2.020,-999.000,-5.410,-999.000,-9.364,-999.000,
     &-12.198,-13.477,-12.793,-10.620,-9.782
     &/
       data (fjrocvz(it, 7,1, 3), it=1,59) /
     &-13.268,-13.588,-14.424,-15.141,-15.343,-15.565,
     &-15.823,-15.712,-14.759,-13.279,-9.866,-6.485,
     &-3.490,0.986,5.826,9.348,12.597,16.275,
     &21.360,24.965,26.029,24.907,22.823,20.805,
     &18.982,16.445,13.409,10.660,8.842,8.287,
     &8.733,8.132,5.913,3.759,3.106,-999.000,
     &3.071,-999.000,5.001,-999.000,8.352,10.393,
     &14.493,14.111,8.388,-999.000,4.420,-999.000,
     &0.794,-999.000,-3.050,-999.000,-7.601,-999.000,
     &-10.964,-13.766,-14.574,-13.943,-13.268
     &/
       data (fjrocvz(it, 8,1, 3), it=1,59) /
     &-17.058,-16.120,-15.679,-15.366,-15.016,-15.064,
     &-15.586,-16.299,-16.812,-17.007,-14.636,-9.793,
     &-5.358,-2.246,0.484,3.098,6.084,10.498,
     &16.071,20.404,22.482,21.909,19.908,17.793,
     &15.745,13.338,10.848,8.357,6.945,6.560,
     &7.748,8.139,7.183,5.382,4.708,-999.000,
     &4.975,-999.000,7.789,-999.000,13.859,20.943,
     &24.188,24.780,19.134,11.068,7.624,-999.000,
     &-1.606,-999.000,-7.518,-999.000,-11.926,-999.000,
     &-15.008,-17.983,-19.022,-18.122,-17.058
     &/
       data (fjrocvz(it, 9,1, 3), it=1,59) /
     &-18.183,-16.645,-15.603,-14.821,-14.044,-14.468,
     &-15.722,-17.073,-17.913,-18.395,-16.543,-12.766,
     &-9.157,-5.068,-0.613,2.455,4.695,8.189,
     &13.287,17.831,20.197,20.558,19.668,18.298,
     &16.228,14.369,12.818,11.283,9.888,8.785,
     &8.956,8.761,7.826,6.643,6.739,6.039,
     &5.725,5.953,9.247,14.321,19.946,28.582,
     &34.690,31.643,18.454,5.354,-2.313,-7.462,
     &-9.327,-999.000,-14.327,-999.000,-17.533,-999.000,
     &-19.704,-21.602,-21.618,-19.963,-18.183
     &/
       data (fjrocvz(it,10,1, 3), it=1,59) /
     &-16.666,-15.132,-14.111,-13.332,-13.391,-14.500,
     &-15.617,-16.429,-17.593,-19.841,-19.001,-14.607,
     &-9.057,-2.850,2.184,5.900,8.527,11.279,
     &15.016,18.455,21.143,22.950,23.522,22.802,
     &20.760,18.844,17.380,16.053,14.312,12.738,
     &11.926,11.165,9.875,8.351,7.450,-999.000,
     &6.198,7.372,13.951,22.811,32.652,39.520,
     &34.427,23.057,9.476,-4.330,-9.148,-12.439,
     &-14.506,-999.000,-17.888,-999.000,-20.671,-999.000,
     &-22.028,-22.086,-20.960,-18.939,-16.666
     &/
       data (fjrocvz(it,11,1, 3), it=1,59) /
     &-16.045,-14.709,-14.281,-14.696,-16.021,-17.437,
     &-18.703,-19.473,-20.179,-21.299,-19.388,-12.592,
     &-5.374,1.359,6.875,10.752,12.977,14.442,
     &17.108,21.132,25.173,27.692,28.215,26.903,
     &24.876,22.715,20.962,18.652,16.203,14.264,
     &12.548,11.277,10.167,9.348,9.399,8.515,
     &8.575,11.803,17.416,24.784,29.416,30.521,
     &19.344,6.566,-2.051,-10.812,-16.116,-999.000,
     &-17.643,-999.000,-19.942,-999.000,-21.518,-999.000,
     &-21.898,-21.315,-19.920,-17.869,-16.045
     &/
       data (fjrocvz(it,12,1, 3), it=1,59) /
     &-18.299,-17.755,-18.034,-19.335,-20.643,-20.624,
     &-20.727,-21.291,-21.953,-22.280,-20.477,-15.578,
     &-7.870,-0.264,8.697,12.956,15.243,17.110,
     &19.705,23.571,26.599,28.469,28.989,28.452,
     &27.211,25.008,22.333,19.160,15.735,13.070,
     &11.371,10.014,9.185,9.625,11.761,-999.000,
     &13.297,12.614,16.074,22.023,27.665,25.873,
     &16.591,8.159,1.130,-999.000,-12.112,-999.000,
     &-19.935,-999.000,-22.510,-999.000,-22.822,-999.000,
     &-22.169,-21.426,-20.590,-19.316,-18.299
     &/
       data (fjrocvz(it,13,1, 3), it=1,59) /
     &-19.909,-20.426,-20.803,-21.658,-22.112,-20.900,
     &-20.634,-21.573,-22.219,-22.718,-21.282,-16.633,
     &-7.179,-0.511,6.941,12.512,15.313,17.447,
     &20.167,23.479,25.840,27.062,27.072,26.261,
     &24.520,22.186,20.047,17.102,12.936,10.346,
     &9.664,9.331,8.836,8.625,8.833,-999.000,
     &10.283,-999.000,13.088,17.545,22.602,19.830,
     &13.810,-999.000,4.461,-999.000,-8.504,-999.000,
     &-18.003,-999.000,-22.360,-999.000,-23.055,-999.000,
     &-22.519,-21.611,-20.591,-19.839,-19.909
     &/
       data (fjrocvz(it,14,1, 3), it=1,59) /
     &-19.939,-21.138,-21.799,-21.919,-21.316,-20.027,
     &-21.027,-22.741,-23.081,-22.870,-19.625,-15.586,
     &-8.662,-3.673,3.913,10.188,13.344,15.419,
     &17.747,20.886,23.863,25.140,24.778,23.235,
     &21.420,19.509,17.478,14.180,10.288,7.731,
     &7.166,6.983,5.963,4.087,3.610,-999.000,
     &5.232,5.477,10.106,13.088,15.255,15.643,
     &13.522,7.390,3.866,-999.000,-6.051,-999.000,
     &-15.273,-999.000,-21.189,-999.000,-23.041,-999.000,
     &-22.864,-21.841,-20.240,-19.568,-19.939
     &/
       data (fjrocvz(it,15,1, 3), it=1,59) /
     &-20.213,-20.978,-22.167,-22.826,-22.349,-20.783,
     &-21.216,-22.989,-24.945,-26.481,-23.794,-18.034,
     &-9.775,-4.741,1.291,6.889,10.498,13.252,
     &15.540,17.278,18.808,19.544,18.764,17.172,
     &15.762,14.500,13.089,11.105,8.007,5.240,
     &4.320,3.896,2.685,2.155,2.501,-999.000,
     &4.264,-999.000,9.060,13.415,16.952,18.385,
     &16.195,9.805,5.559,-999.000,-3.733,-999.000,
     &-12.868,-999.000,-20.157,-999.000,-22.874,-999.000,
     &-22.834,-22.042,-21.016,-20.455,-20.213
     &/
       data (fjrocvz(it,16,1, 3), it=1,59) /
     &-22.253,-22.272,-22.881,-23.625,-23.218,-21.430,
     &-21.292,-22.168,-24.235,-29.543,-30.507,-25.140,
     &-10.203,-2.550,1.712,6.202,9.786,12.111,
     &14.390,16.268,17.933,18.034,17.145,15.631,
     &14.115,12.578,10.987,8.640,5.376,3.169,
     &2.907,3.274,3.255,3.556,4.449,-999.000,
     &7.067,-999.000,12.548,16.631,18.881,18.314,
     &15.090,-999.000,5.094,-999.000,-3.022,-999.000,
     &-12.536,-999.000,-19.917,-999.000,-22.877,-999.000,
     &-23.536,-23.104,-22.673,-22.530,-22.253
     &/
       data (fjrocvz(it,17,1, 3), it=1,59) /
     &-23.297,-23.155,-23.363,-24.129,-24.752,-23.896,
     &-23.823,-22.452,-21.738,-26.247,-29.167,-24.622,
     &-9.577,1.798,5.342,8.165,10.967,13.008,
     &15.192,17.200,18.489,19.202,18.959,17.654,
     &16.069,14.130,12.233,9.693,6.188,3.882,
     &3.477,4.015,3.875,3.568,4.794,-999.000,
     &6.481,9.135,11.607,14.282,15.657,16.675,
     &12.971,-999.000,-0.295,-999.000,-9.107,-999.000,
     &-17.738,-999.000,-23.593,-999.000,-25.965,-999.000,
     &-25.586,-24.077,-23.174,-23.249,-23.297
     &/
       data (fjrocvz(it,18,1, 3), it=1,59) /
     &-23.470,-23.271,-23.459,-24.555,-26.259,-25.679,
     &-25.844,-25.194,-22.921,-23.178,-22.441,-17.657,
     &-5.232,5.798,9.915,12.432,15.245,17.195,
     &19.161,19.906,20.314,21.061,20.727,19.146,
     &16.876,14.630,12.411,10.234,7.354,5.403,
     &4.509,4.584,4.721,5.081,5.885,-999.000,
     &7.688,-999.000,10.463,14.537,16.391,14.893,
     &8.937,0.839,-5.723,-999.000,-14.381,-999.000,
     &-21.835,-999.000,-27.346,-999.000,-28.803,-999.000,
     &-27.241,-24.986,-23.806,-23.669,-23.470
     &/
       data (fjrocvz(it,19,1, 3), it=1,59) /
     &-24.282,-23.210,-22.318,-22.551,-23.130,-21.311,
     &-20.076,-19.835,-18.626,-19.660,-18.994,-14.733,
     &-3.899,6.826,12.734,16.455,19.244,21.551,
     &23.683,24.477,24.753,24.990,24.795,22.968,
     &20.629,18.488,15.538,11.676,8.240,7.151,
     &6.935,6.701,6.444,5.686,5.705,-999.000,
     &7.248,-999.000,9.543,12.679,11.823,7.813,
     &3.949,-999.000,-6.564,-999.000,-14.959,-999.000,
     &-22.554,-999.000,-27.449,-999.000,-28.742,-999.000,
     &-27.355,-25.425,-24.352,-24.460,-24.282
     &/
       data (fjrocvz(it,20,1, 3), it=1,59) /
     &-23.145,-22.079,-21.713,-21.691,-22.021,-20.154,
     &-18.836,-18.076,-17.144,-17.684,-19.403,-16.503,
     &-6.426,5.486,11.836,15.106,18.854,22.181,
     &25.026,26.431,26.519,26.779,26.664,25.506,
     &24.400,23.144,20.765,16.775,12.126,10.282,
     &9.807,9.132,7.648,6.236,5.983,-999.000,
     &5.347,-999.000,5.221,7.774,11.285,8.306,
     &1.193,-999.000,-7.762,-999.000,-16.761,-999.000,
     &-24.066,-999.000,-28.812,-999.000,-30.292,-999.000,
     &-29.686,-27.912,-26.020,-24.541,-23.145
     &/
       data (fjrocvz(it,21,1, 3), it=1,59) /
     &-24.513,-23.917,-23.951,-24.508,-25.130,-22.902,
     &-20.665,-20.143,-20.669,-22.784,-25.124,-22.607,
     &-11.710,1.171,8.074,11.872,16.216,18.904,
     &21.983,24.318,25.103,25.119,24.574,24.075,
     &23.880,22.875,20.614,16.545,11.580,9.340,
     &8.828,8.359,7.082,5.320,4.285,-999.000,
     &5.009,-999.000,6.363,8.675,10.229,6.334,
     &-1.489,-999.000,-10.640,-999.000,-17.828,-999.000,
     &-24.056,-999.000,-28.837,-999.000,-30.554,-999.000,
     &-30.447,-29.444,-28.022,-26.185,-24.513
     &/
       data (fjrocvz(it,22,1, 3), it=1,59) /
     &-22.787,-22.516,-22.907,-24.101,-25.106,-23.659,
     &-22.005,-22.043,-23.000,-24.740,-26.359,-23.899,
     &-13.472,-0.986,6.136,10.102,13.730,16.956,
     &20.335,23.485,24.130,23.929,24.056,23.996,
     &23.264,21.455,18.188,13.163,7.839,5.366,
     &5.044,5.020,3.960,2.648,2.383,-999.000,
     &2.943,-999.000,2.442,-999.000,3.378,-999.000,
     &-0.454,-999.000,-10.936,-999.000,-16.052,-999.000,
     &-21.010,-999.000,-25.460,-999.000,-27.556,-999.000,
     &-27.627,-26.500,-24.843,-23.651,-22.787
     &/
       data (fjrocvz(it,23,1, 3), it=1,59) /
     &-16.874,-17.235,-17.682,-18.577,-19.911,-19.271,
     &-18.876,-19.324,-19.494,-19.952,-19.859,-16.072,
     &-7.653,0.893,6.723,9.638,12.576,15.110,
     &18.445,21.698,23.372,25.345,26.341,25.635,
     &23.630,20.923,17.985,14.055,9.280,6.138,
     &4.168,3.094,2.188,1.530,1.550,-999.000,
     &2.341,-999.000,5.569,-999.000,6.830,-999.000,
     &4.449,-999.000,-0.079,-999.000,-9.343,-999.000,
     &-14.971,-999.000,-19.668,-999.000,-21.689,-999.000,
     &-21.876,-20.811,-19.041,-17.388,-16.874
     &/
       data (fjrocvz(it,24,1, 3), it=1,59) /
     &-12.623,-12.719,-12.860,-13.670,-14.382,-13.456,
     &-13.393,-14.434,-15.072,-16.575,-16.312,-11.745,
     &-3.383,3.647,8.559,12.254,15.013,17.676,
     &20.926,23.957,25.411,26.002,25.399,23.745,
     &21.418,18.688,15.996,13.207,9.175,5.654,
     &3.183,2.343,2.198,1.634,0.965,-999.000,
     &1.695,-999.000,3.444,-999.000,5.462,-999.000,
     &4.806,-999.000,1.926,-999.000,-2.828,-999.000,
     &-8.189,-999.000,-12.367,-999.000,-14.322,-999.000,
     &-15.417,-15.736,-14.762,-13.366,-12.623
     &/
       data (fjrocvz(it,25,1, 3), it=1,59) /
     &-11.521,-11.160,-10.952,-11.214,-11.842,-12.343,
     &-13.707,-15.192,-16.758,-18.265,-17.135,-10.871,
     &-1.300,6.249,12.120,16.441,20.810,25.377,
     &29.418,31.908,31.499,29.754,27.434,24.466,
     &21.354,17.086,13.030,9.894,6.413,2.842,
     &0.716,0.045,-0.119,-0.520,-0.917,-999.000,
     &-1.414,-999.000,-0.562,-999.000,1.409,-999.000,
     &0.817,-999.000,-0.602,-999.000,-4.023,-999.000,
     &-7.959,-999.000,-11.120,-999.000,-12.538,-999.000,
     &-14.610,-15.494,-14.728,-12.775,-11.521
     &/
       data (fjrocvz(it, 1,2, 3), it=1,59) /
     &-15.403,-14.516,-13.213,-12.576,-13.623,-14.603,
     &-15.471,-16.611,-17.434,-16.976,-13.973,-7.617,
     &-8.765,-3.937,1.426,7.655,13.478,22.377,
     &29.380,32.209,31.871,30.205,27.522,23.175,
     &18.595,14.006,8.453,4.341,3.160,4.199,
     &4.720,4.840,4.946,6.183,8.379,-999.000,
     &11.202,-999.000,14.376,15.694,14.454,9.105,
     &2.785,-999.000,-1.390,-999.000,-7.449,-999.000,
     &-12.279,-999.000,-14.370,-999.000,-17.723,-999.000,
     &-18.745,-18.599,-18.156,-16.919,-15.403
     &/
       data (fjrocvz(it, 2,2, 3), it=1,59) /
     &-13.712,-12.989,-12.361,-12.128,-11.640,-10.975,
     &-11.391,-13.376,-14.861,-13.983,-8.514,-0.627,
     &5.158,8.308,12.576,15.241,19.235,24.547,
     &30.002,33.678,34.445,33.071,30.037,25.504,
     &20.787,16.934,12.947,9.026,6.491,6.227,
     &7.189,7.892,8.371,9.450,10.580,-999.000,
     &12.136,-999.000,15.985,17.004,17.329,15.443,
     &9.812,-999.000,-1.276,-999.000,-8.978,-999.000,
     &-10.641,-999.000,-14.622,-999.000,-18.245,-999.000,
     &-18.435,-18.386,-17.035,-14.857,-13.712
     &/
       data (fjrocvz(it, 3,2, 3), it=1,59) /
     &-13.835,-12.605,-11.978,-11.080,-10.278,-10.144,
     &-9.951,-10.316,-10.951,-10.432,-5.857,0.979,
     &7.002,10.295,12.710,15.816,19.495,22.721,
     &26.073,28.685,29.940,28.702,25.486,21.370,
     &17.479,14.507,11.841,9.446,7.295,6.725,
     &7.632,8.580,9.572,10.026,11.295,-999.000,
     &13.800,-999.000,16.137,19.902,22.131,17.684,
     &8.251,1.854,-2.010,-8.662,-12.117,-11.965,
     &-12.474,-999.000,-18.065,-999.000,-20.120,-999.000,
     &-20.327,-19.108,-17.483,-15.565,-13.835
     &/
       data (fjrocvz(it, 4,2, 3), it=1,59) /
     &-15.148,-13.976,-12.459,-11.070,-10.830,-10.559,
     &-9.821,-8.384,-7.417,-6.800,-3.709,2.009,
     &8.012,10.425,11.658,14.266,17.781,21.778,
     &25.134,27.011,26.468,24.051,20.853,17.216,
     &13.673,10.728,8.506,7.099,6.329,6.300,
     &7.025,8.295,9.292,10.800,13.470,-999.000,
     &15.247,-999.000,18.301,22.319,22.473,17.942,
     &11.899,6.928,-1.981,-12.535,-14.942,-999.000,
     &-14.606,-999.000,-19.477,-999.000,-21.957,-999.000,
     &-21.148,-19.021,-17.626,-16.091,-15.148
     &/
       data (fjrocvz(it, 5,2, 3), it=1,59) /
     &-15.959,-14.747,-12.588,-11.055,-11.083,-11.215,
     &-10.782,-9.274,-7.960,-7.128,-4.154,2.037,
     &7.998,10.670,13.175,15.125,18.437,23.010,
     &26.983,29.135,28.599,26.168,22.447,18.399,
     &14.510,11.811,10.011,9.052,8.099,7.656,
     &8.216,9.493,11.013,12.346,13.567,-999.000,
     &17.023,-999.000,20.738,23.039,24.268,23.672,
     &18.542,8.968,-1.553,-9.199,-13.724,-16.753,
     &-19.517,-23.525,-25.237,-24.844,-23.370,-999.000,
     &-21.599,-19.388,-17.719,-16.557,-15.959
     &/
       data (fjrocvz(it, 6,2, 3), it=1,59) /
     &-16.831,-15.990,-14.584,-13.296,-13.050,-12.122,
     &-10.520,-9.231,-7.713,-6.012,-3.971,-2.755,
     &0.323,5.600,11.727,16.366,19.405,23.790,
     &27.779,29.884,30.412,28.879,26.084,22.301,
     &18.921,16.509,14.455,12.646,11.035,9.782,
     &9.525,10.897,12.679,13.133,14.367,-999.000,
     &18.127,-999.000,22.236,26.346,30.428,30.750,
     &26.235,15.909,4.197,-5.730,-12.173,-18.480,
     &-21.430,-999.000,-24.988,-999.000,-24.847,-999.000,
     &-23.014,-19.971,-18.362,-17.666,-16.831
     &/
       data (fjrocvz(it, 7,2, 3), it=1,59) /
     &-18.281,-17.694,-16.627,-15.638,-14.891,-12.554,
     &-10.436,-8.905,-6.630,-5.771,-3.748,-1.541,
     &3.404,5.663,9.248,12.294,15.426,20.490,
     &25.303,28.861,29.623,29.244,27.759,25.259,
     &22.554,19.930,17.616,15.577,13.319,11.171,
     &10.393,11.069,12.347,13.055,13.650,-999.000,
     &16.289,-999.000,20.169,25.953,31.334,35.432,
     &29.213,15.396,3.315,-4.758,-12.063,-20.379,
     &-25.183,-26.917,-27.170,-26.886,-25.951,-999.000,
     &-23.906,-21.906,-20.451,-18.907,-18.281
     &/
       data (fjrocvz(it, 8,2, 3), it=1,59) /
     &-17.873,-16.295,-15.275,-14.717,-13.817,-12.580,
     &-11.547,-10.480,-8.384,-10.628,-12.669,-14.627,
     &-9.798,-2.989,2.372,7.600,10.315,13.771,
     &19.511,23.915,25.836,26.065,25.469,24.103,
     &22.234,20.390,18.104,15.901,13.273,10.856,
     &9.784,10.107,11.279,11.924,12.340,-999.000,
     &15.077,-999.000,19.482,23.180,28.083,28.793,
     &21.961,13.321,4.130,-4.550,-11.227,-18.257,
     &-22.484,-23.393,-21.806,-999.000,-22.381,-999.000,
     &-21.928,-21.771,-20.833,-19.223,-17.873
     &/
       data (fjrocvz(it, 9,2, 3), it=1,59) /
     &-14.891,-12.605,-10.621,-9.292,-9.025,-9.438,
     &-10.763,-11.964,-12.559,-14.563,-17.417,-19.842,
     &-15.164,-5.736,1.997,6.938,8.479,11.902,
     &16.857,20.374,21.831,22.377,22.035,20.520,
     &18.191,15.974,14.119,12.177,9.936,8.650,
     &8.568,9.136,10.247,10.775,11.657,-999.000,
     &13.458,-999.000,18.006,22.817,26.496,29.687,
     &29.340,21.481,10.416,0.846,-5.261,-999.000,
     &-10.033,-999.000,-13.702,-999.000,-16.726,-999.000,
     &-17.495,-17.760,-18.159,-17.098,-14.891
     &/
       data (fjrocvz(it,10,2, 3), it=1,59) /
     &-14.646,-12.351,-10.004,-8.627,-7.499,-8.543,
     &-11.774,-14.361,-15.432,-16.240,-17.113,-17.345,
     &-13.802,-3.602,4.134,8.951,10.758,14.786,
     &19.888,23.038,23.330,22.637,21.170,19.399,
     &17.037,14.343,12.038,9.995,8.554,8.004,
     &8.060,8.384,8.339,8.503,9.855,-999.000,
     &12.092,-999.000,18.841,24.466,30.388,33.751,
     &32.984,23.017,13.102,-999.000,4.227,-999.000,
     &-3.194,-999.000,-9.152,-999.000,-10.872,-999.000,
     &-12.528,-13.919,-15.325,-15.322,-14.646
     &/
       data (fjrocvz(it,11,2, 3), it=1,59) /
     &-16.848,-16.203,-15.904,-15.665,-15.087,-15.729,
     &-16.569,-18.119,-19.582,-21.202,-20.563,-16.744,
     &-11.763,-2.440,4.777,10.476,14.430,17.490,
     &21.741,25.259,26.402,26.016,24.838,22.941,
     &20.861,18.251,15.142,12.875,10.977,10.002,
     &9.330,8.534,7.972,8.112,10.018,-999.000,
     &13.914,-999.000,19.645,25.424,32.799,33.158,
     &28.358,19.927,11.523,-999.000,0.653,-999.000,
     &-7.310,-999.000,-11.625,-999.000,-12.477,-999.000,
     &-15.032,-15.212,-16.480,-17.561,-16.848
     &/
       data (fjrocvz(it,12,2, 3), it=1,59) /
     &-19.203,-20.101,-20.062,-19.929,-20.155,-19.754,
     &-19.449,-21.131,-23.905,-25.140,-25.402,-21.709,
     &-17.314,-8.240,1.275,6.976,11.463,16.143,
     &22.628,27.369,29.051,28.617,27.347,25.914,
     &24.052,20.977,17.604,14.855,13.004,12.198,
     &11.220,9.939,8.966,8.979,11.151,-999.000,
     &14.952,-999.000,20.657,26.196,33.252,36.279,
     &30.915,17.244,7.032,-0.020,-5.830,-999.000,
     &-9.127,-999.000,-11.567,-999.000,-15.431,-999.000,
     &-17.737,-18.013,-18.566,-19.028,-19.203
     &/
       data (fjrocvz(it,13,2, 3), it=1,59) /
     &-19.549,-19.150,-18.971,-19.030,-18.723,-19.365,
     &-20.870,-22.164,-23.775,-23.580,-22.933,-20.696,
     &-17.024,-9.104,-0.484,6.057,10.614,15.933,
     &24.905,30.440,31.033,30.227,29.030,27.687,
     &25.535,22.083,18.778,16.422,15.232,14.871,
     &14.082,12.732,11.230,10.999,12.722,-999.000,
     &15.634,-999.000,23.570,28.425,35.456,39.451,
     &31.920,17.886,3.573,-3.535,-6.138,-7.832,
     &-8.622,-999.000,-11.868,-999.000,-15.168,-999.000,
     &-17.708,-19.940,-20.057,-19.532,-19.549
     &/
       data (fjrocvz(it,14,2, 3), it=1,59) /
     &-18.377,-17.976,-17.661,-17.096,-17.079,-18.642,
     &-21.335,-24.894,-28.142,-27.113,-23.563,-18.842,
     &-13.999,-11.356,-4.636,3.000,8.536,14.598,
     &23.711,29.131,30.457,29.892,28.534,26.876,
     &24.293,21.473,18.934,17.235,16.400,15.929,
     &14.835,13.459,11.923,11.755,13.732,-999.000,
     &17.654,-999.000,25.378,31.132,34.869,35.053,
     &28.160,15.775,7.129,-999.000,-3.304,-999.000,
     &-11.349,-999.000,-14.965,-999.000,-17.864,-999.000,
     &-20.794,-21.649,-20.824,-19.291,-18.377
     &/
       data (fjrocvz(it,15,2, 3), it=1,59) /
     &-16.611,-15.993,-15.286,-14.760,-14.638,-15.888,
     &-19.585,-25.541,-29.488,-30.301,-26.559,-18.845,
     &-13.412,-9.451,-4.501,2.658,8.370,13.944,
     &21.345,26.319,27.969,27.150,25.142,22.366,
     &19.415,17.066,14.825,13.234,12.246,11.394,
     &10.469,9.438,8.867,9.533,12.126,-999.000,
     &15.410,-999.000,21.709,26.131,28.932,30.660,
     &24.991,12.925,4.822,-999.000,-6.661,-999.000,
     &-13.676,-999.000,-18.490,-999.000,-21.993,-999.000,
     &-23.977,-24.210,-21.831,-18.338,-16.611
     &/
       data (fjrocvz(it,16,2, 3), it=1,59) /
     &-16.692,-15.214,-15.261,-15.274,-15.607,-17.171,
     &-20.600,-24.998,-28.037,-30.806,-29.641,-21.923,
     &-12.339,-8.423,-3.282,2.725,7.802,13.196,
     &20.496,25.522,27.209,26.675,25.033,21.475,
     &17.088,13.148,10.316,8.536,7.630,7.138,
     &6.290,5.163,4.446,5.062,7.223,-999.000,
     &10.308,-999.000,16.067,19.733,22.819,24.146,
     &18.180,5.865,1.282,-999.000,-9.266,-999.000,
     &-16.840,-999.000,-21.755,-999.000,-24.730,-999.000,
     &-26.097,-25.504,-22.918,-19.423,-16.692
     &/
       data (fjrocvz(it,17,2, 3), it=1,59) /
     &-19.448,-18.454,-17.582,-16.952,-17.345,-18.256,
     &-20.919,-25.031,-27.896,-29.497,-27.750,-19.588,
     &-10.746,-6.153,-0.228,4.719,8.864,14.501,
     &22.232,27.395,29.626,29.711,28.256,25.051,
     &20.114,15.283,11.240,8.313,7.033,6.260,
     &5.105,3.548,1.856,1.359,3.327,-999.000,
     &7.296,-999.000,13.887,16.342,19.138,18.064,
     &13.247,3.909,-3.841,-11.798,-18.028,-21.641,
     &-23.311,-999.000,-23.959,-999.000,-25.809,-999.000,
     &-26.254,-24.844,-23.081,-21.131,-19.448
     &/
       data (fjrocvz(it,18,2, 3), it=1,59) /
     &-20.392,-18.687,-17.345,-16.871,-16.559,-16.660,
     &-19.841,-23.997,-25.912,-27.465,-24.911,-17.590,
     &-8.673,-3.955,0.385,5.419,10.030,17.057,
     &24.101,29.619,33.103,33.262,30.754,26.507,
     &22.121,17.948,13.547,9.628,7.398,5.825,
     &4.434,2.742,1.310,1.332,3.592,-999.000,
     &6.764,9.240,13.180,15.959,16.680,13.879,
     &6.826,0.873,-7.027,-999.000,-18.805,-999.000,
     &-25.321,-999.000,-27.600,-999.000,-27.613,-999.000,
     &-26.199,-25.348,-23.483,-21.776,-20.392
     &/
       data (fjrocvz(it,19,2, 3), it=1,59) /
     &-18.983,-17.094,-16.786,-16.205,-15.662,-15.399,
     &-17.537,-19.674,-20.785,-22.174,-19.318,-12.131,
     &-3.752,-0.916,3.306,8.488,12.903,19.647,
     &27.447,33.414,36.425,36.340,33.491,29.188,
     &24.414,19.666,15.154,11.163,9.279,8.278,
     &6.725,4.818,3.320,2.981,4.937,-999.000,
     &8.826,-999.000,14.846,16.027,15.588,9.832,
     &6.010,-0.316,-8.613,-17.831,-23.441,-26.764,
     &-27.507,-999.000,-28.119,-999.000,-26.858,-999.000,
     &-25.196,-24.493,-22.929,-21.054,-18.983
     &/
       data (fjrocvz(it,20,2, 3), it=1,59) /
     &-21.283,-19.385,-17.531,-15.659,-14.221,-13.536,
     &-15.152,-17.719,-18.300,-18.764,-16.099,-6.753,
     &1.523,2.299,6.120,9.975,13.700,20.672,
     &28.191,34.942,38.218,38.506,36.300,32.554,
     &28.225,23.775,19.066,15.057,12.653,11.028,
     &9.190,7.133,5.307,5.077,7.472,-999.000,
     &10.899,-999.000,15.400,16.260,18.404,17.044,
     &10.982,0.366,-7.713,-14.040,-19.053,-23.868,
     &-26.477,-999.000,-26.697,-999.000,-26.984,-999.000,
     &-26.008,-25.463,-24.448,-22.896,-21.283
     &/
       data (fjrocvz(it,21,2, 3), it=1,59) /
     &-21.730,-19.246,-17.319,-16.091,-15.376,-14.280,
     &-14.957,-17.171,-18.888,-19.193,-16.204,-8.651,
     &-2.828,0.979,5.274,10.015,14.789,20.333,
     &25.930,32.325,35.334,36.171,34.100,30.176,
     &26.097,22.251,18.163,13.638,10.466,8.418,
     &6.587,5.276,4.392,4.202,5.804,-999.000,
     &9.285,-999.000,15.063,17.860,20.662,19.534,
     &10.710,-999.000,-3.026,-999.000,-15.442,-999.000,
     &-22.618,-999.000,-25.147,-999.000,-25.698,-999.000,
     &-25.263,-24.855,-24.194,-23.228,-21.730
     &/
       data (fjrocvz(it,22,2, 3), it=1,59) /
     &-20.628,-19.718,-18.655,-17.944,-18.180,-17.678,
     &-17.764,-19.808,-23.236,-23.982,-20.891,-13.913,
     &-5.634,-2.740,2.595,7.310,11.872,17.426,
     &23.956,29.719,32.042,31.799,29.397,25.062,
     &19.878,15.172,11.278,7.849,5.200,3.567,
     &2.631,2.846,3.314,3.356,4.830,-999.000,
     &9.735,-999.000,15.320,14.981,16.668,15.221,
     &9.037,-999.000,-5.517,-999.000,-16.228,-999.000,
     &-23.499,-999.000,-24.348,-999.000,-24.142,-999.000,
     &-24.123,-23.565,-22.582,-21.602,-20.628
     &/
       data (fjrocvz(it,23,2, 3), it=1,59) /
     &-18.349,-18.163,-16.838,-15.099,-15.046,-16.117,
     &-17.958,-20.244,-22.504,-23.392,-20.917,-14.784,
     &-9.605,-6.515,-1.028,4.246,9.489,15.018,
     &23.189,28.840,30.538,29.123,26.242,21.811,
     &16.644,11.378,7.416,4.449,2.447,1.474,
     &0.928,0.740,1.798,3.303,5.706,-999.000,
     &8.382,-999.000,13.010,14.696,14.976,14.164,
     &10.223,1.980,-4.722,-999.000,-14.583,-999.000,
     &-20.826,-999.000,-22.979,-999.000,-23.062,-999.000,
     &-22.945,-22.050,-20.236,-18.673,-18.349
     &/
       data (fjrocvz(it,24,2, 3), it=1,59) /
     &-16.126,-14.938,-13.411,-11.773,-12.375,-14.518,
     &-16.664,-18.066,-19.518,-20.483,-18.853,-14.110,
     &-7.746,-2.245,1.325,4.745,9.618,17.787,
     &25.977,29.601,30.212,28.655,25.377,20.679,
     &15.837,10.685,6.175,2.999,1.890,2.492,
     &2.639,2.159,2.945,4.385,6.351,-999.000,
     &8.808,-999.000,13.046,14.359,15.234,13.055,
     &7.017,-999.000,-0.752,-999.000,-8.186,-999.000,
     &-13.835,-999.000,-17.814,-999.000,-18.872,-999.000,
     &-19.939,-19.635,-18.294,-17.244,-16.126
     &/
       data (fjrocvz(it,25,2, 3), it=1,59) /
     &-15.403,-14.516,-13.213,-12.576,-13.623,-14.603,
     &-15.471,-16.611,-17.434,-16.976,-13.973,-7.617,
     &-8.765,-3.937,1.426,7.655,13.478,22.377,
     &29.380,32.209,31.871,30.205,27.522,23.175,
     &18.595,14.006,8.453,4.341,3.160,4.199,
     &4.720,4.840,4.946,6.183,8.379,-999.000,
     &11.202,-999.000,14.376,15.694,14.454,9.105,
     &2.785,-999.000,-1.390,-999.000,-7.449,-999.000,
     &-12.279,-999.000,-14.370,-999.000,-17.723,-999.000,
     &-18.745,-18.599,-18.156,-16.919,-15.403
     &/
       data (fjrocvz(it, 1,3, 3), it=1,59) /
     &-22.032,-23.356,-23.190,-22.275,-20.936,-19.293,
     &-19.030,-21.760,-27.572,-29.917,-23.388,-10.200,
     &0.540,4.694,7.437,9.553,9.607,8.219,
     &8.345,10.757,13.677,14.771,14.223,11.766,
     &8.537,7.244,7.957,9.233,9.463,8.387,
     &7.685,7.544,7.243,6.230,6.191,-999.000,
     &6.508,-999.000,7.683,-999.000,12.604,15.454,
     &15.494,12.861,4.747,-999.000,-0.949,-999.000,
     &-7.532,-999.000,-10.577,-999.000,-13.061,-999.000,
     &-16.285,-18.904,-20.539,-21.083,-22.032
     &/
       data (fjrocvz(it, 2,3, 3), it=1,59) /
     &-19.806,-20.403,-19.984,-18.875,-17.723,-17.116,
     &-18.321,-21.244,-25.170,-26.860,-19.100,-3.916,
     &8.595,11.637,12.742,13.621,13.495,12.817,
     &13.415,14.737,15.692,16.192,15.179,13.096,
     &11.244,10.683,11.356,12.242,12.972,12.589,
     &11.900,11.637,11.459,10.993,10.468,-999.000,
     &9.358,-999.000,11.196,-999.000,14.478,17.394,
     &16.513,13.339,4.619,-999.000,-1.276,-999.000,
     &-7.856,-999.000,-12.082,-999.000,-14.899,-999.000,
     &-17.627,-19.817,-20.486,-19.747,-19.806
     &/
       data (fjrocvz(it, 3,3, 3), it=1,59) /
     &-18.685,-17.558,-17.012,-16.286,-15.759,-15.401,
     &-16.367,-17.992,-21.332,-25.633,-18.850,-0.064,
     &16.683,18.301,16.809,15.445,13.982,13.410,
     &14.077,14.484,13.438,12.044,10.434,9.469,
     &8.934,8.840,9.750,10.940,12.281,13.431,
     &13.385,13.031,12.878,12.560,12.203,-999.000,
     &12.253,-999.000,12.585,-999.000,13.510,14.262,
     &12.582,11.832,5.374,-999.000,-3.052,-999.000,
     &-9.894,-999.000,-15.017,-999.000,-18.167,-999.000,
     &-21.042,-23.053,-23.177,-20.878,-18.685
     &/
       data (fjrocvz(it, 4,3, 3), it=1,59) /
     &-20.788,-18.743,-18.207,-18.216,-17.642,-16.393,
     &-15.470,-15.980,-19.127,-22.270,-15.113,4.673,
     &19.098,20.894,19.504,18.482,16.728,14.395,
     &12.605,11.254,9.978,8.949,8.573,8.775,
     &8.509,8.767,9.491,10.823,12.371,13.282,
     &13.719,13.844,13.504,12.903,13.766,-999.000,
     &13.696,-999.000,12.619,-999.000,11.914,14.001,
     &13.274,7.899,3.432,-999.000,-5.441,-999.000,
     &-12.588,-999.000,-17.714,-999.000,-21.358,-999.000,
     &-23.587,-25.119,-25.285,-23.243,-20.788
     &/
       data (fjrocvz(it, 5,3, 3), it=1,59) /
     &-23.172,-21.921,-21.183,-19.499,-17.403,-15.650,
     &-14.685,-15.098,-16.379,-17.116,-11.220,2.996,
     &17.484,21.823,21.484,20.599,18.708,16.520,
     &14.670,13.388,13.033,12.891,12.619,11.644,
     &10.239,9.200,8.815,9.406,10.135,10.872,
     &11.803,12.549,12.663,12.590,12.707,-999.000,
     &12.686,-999.000,12.799,12.847,12.846,9.102,
     &8.666,6.757,0.835,-999.000,-7.543,-999.000,
     &-14.587,-999.000,-19.627,-999.000,-23.043,-999.000,
     &-25.705,-27.335,-26.899,-24.820,-23.172
     &/
       data (fjrocvz(it, 6,3, 3), it=1,59) /
     &-25.643,-25.526,-25.329,-23.039,-19.087,-15.657,
     &-14.686,-15.205,-14.711,-13.243,-8.952,1.249,
     &15.020,19.288,18.919,18.129,17.442,16.702,
     &16.128,17.005,17.983,17.996,16.943,15.457,
     &13.629,11.696,10.372,9.500,9.570,9.886,
     &10.487,10.970,11.591,11.734,11.845,-999.000,
     &13.194,-999.000,13.066,13.623,10.998,7.349,
     &2.696,-999.000,-3.112,-999.000,-10.692,-999.000,
     &-18.001,-999.000,-23.068,-999.000,-26.477,-999.000,
     &-29.226,-30.494,-29.108,-26.976,-25.643
     &/
       data (fjrocvz(it, 7,3, 3), it=1,59) /
     &-28.714,-27.095,-26.644,-25.694,-22.934,-20.079,
     &-18.744,-17.779,-16.231,-12.989,-8.696,-1.499,
     &6.226,10.466,11.739,12.620,13.161,13.532,
     &14.606,16.452,18.389,19.447,19.573,19.166,
     &18.316,17.395,15.769,13.822,12.624,11.844,
     &11.468,10.974,10.733,10.465,10.499,-999.000,
     &10.395,-999.000,9.385,-999.000,4.154,-999.000,
     &-0.608,-999.000,-5.373,-999.000,-12.117,-999.000,
     &-19.407,-999.000,-24.853,-999.000,-28.608,-999.000,
     &-31.238,-32.632,-32.111,-30.624,-28.714
     &/
       data (fjrocvz(it, 8,3, 3), it=1,59) /
     &-24.839,-23.891,-24.796,-25.345,-24.151,-20.223,
     &-15.451,-12.401,-10.934,-9.402,-6.398,-2.571,
     &0.096,2.096,3.435,5.050,6.658,7.840,
     &9.263,11.026,13.872,17.028,18.767,19.154,
     &18.964,17.963,16.235,13.905,11.871,11.137,
     &10.537,9.688,8.562,7.310,6.265,-999.000,
     &5.560,-999.000,4.155,-999.000,1.401,-999.000,
     &-1.235,-999.000,-4.808,-999.000,-9.718,-999.000,
     &-15.526,-999.000,-20.377,-999.000,-24.792,-999.000,
     &-27.982,-29.008,-27.373,-25.804,-24.839
     &/
       data (fjrocvz(it, 9,3, 3), it=1,59) /
     &-17.921,-18.198,-18.467,-18.807,-18.480,-15.259,
     &-11.029,-8.713,-8.042,-8.750,-7.816,-5.340,
     &-5.315,-1.656,2.887,7.107,10.975,11.552,
     &12.378,13.138,13.725,13.099,14.242,15.168,
     &15.859,15.447,13.516,10.796,8.754,7.446,
     &6.137,4.718,3.556,3.173,2.952,-999.000,
     &1.771,-999.000,0.587,-999.000,0.671,-999.000,
     &1.282,-999.000,0.909,-999.000,-2.449,-999.000,
     &-6.668,-999.000,-10.185,-999.000,-12.696,-999.000,
     &-15.015,-16.404,-15.980,-16.594,-17.921
     &/
       data (fjrocvz(it,10,3, 3), it=1,59) /
     &-14.995,-15.435,-14.600,-13.540,-13.976,-14.603,
     &-14.013,-12.679,-11.844,-12.031,-11.971,-13.211,
     &-15.197,-7.075,1.473,7.184,12.948,13.866,
     &14.849,15.491,16.035,15.979,15.718,15.507,
     &15.301,15.068,13.376,11.000,8.626,6.308,
     &3.973,1.337,-0.643,-1.454,-2.415,-999.000,
     &-3.851,-999.000,-3.597,-999.000,-2.568,-999.000,
     &0.793,-999.000,2.768,-999.000,2.831,-999.000,
     &0.748,-999.000,-1.970,-999.000,-4.408,-999.000,
     &-6.592,-9.403,-11.316,-13.154,-14.995
     &/
       data (fjrocvz(it,11,3, 3), it=1,59) /
     &-19.062,-20.529,-20.789,-20.077,-19.775,-18.878,
     &-17.276,-16.843,-19.541,-18.343,-17.917,-12.026,
     &-15.486,-9.122,-2.808,3.694,10.238,12.356,
     &14.521,16.263,19.078,20.884,20.221,19.326,
     &18.512,17.540,16.278,14.711,12.264,9.243,
     &6.060,3.216,1.209,-0.844,-2.775,-999.000,
     &-3.932,-999.000,-3.820,-999.000,-2.781,-999.000,
     &3.140,6.651,7.389,5.244,3.444,-999.000,
     &1.603,-999.000,0.071,-999.000,-1.997,-999.000,
     &-4.809,-10.410,-14.287,-16.813,-19.062
     &/
       data (fjrocvz(it,12,3, 3), it=1,59) /
     &-20.480,-22.448,-24.802,-26.538,-26.302,-24.310,
     &-24.202,-28.397,-33.235,-31.093,-24.698,-15.118,
     &-14.356,-11.084,-7.617,-1.495,4.663,8.477,
     &11.757,16.038,20.485,22.298,22.322,21.730,
     &20.803,19.377,17.869,16.853,14.785,11.500,
     &8.510,6.751,5.017,3.323,1.283,-999.000,
     &-0.719,-999.000,0.423,-999.000,5.526,9.463,
     &12.263,13.880,12.063,-999.000,7.706,-999.000,
     &3.508,-999.000,1.414,-999.000,-0.771,-999.000,
     &-4.292,-10.129,-15.132,-18.124,-20.480
     &/
       data (fjrocvz(it,13,3, 3), it=1,59) /
     &-18.827,-20.543,-21.251,-22.294,-24.374,-26.181,
     &-28.605,-31.436,-33.951,-32.355,-25.381,-17.694,
     &-13.948,-12.840,-9.716,-6.244,-2.936,3.985,
     &12.151,18.135,22.362,24.439,24.622,23.381,
     &22.025,20.806,19.345,17.861,15.933,13.305,
     &11.135,8.757,6.801,5.726,4.172,-999.000,
     &3.745,-999.000,6.206,-999.000,9.648,13.447,
     &18.200,20.226,15.692,-999.000,10.696,-999.000,
     &5.723,-999.000,1.802,-999.000,-0.814,-999.000,
     &-3.572,-7.562,-11.781,-15.648,-18.827
     &/
       data (fjrocvz(it,14,3, 3), it=1,59) /
     &-18.666,-20.880,-20.944,-21.842,-24.794,-28.633,
     &-30.969,-33.201,-36.601,-35.695,-27.107,-16.477,
     &-11.858,-11.321,-9.738,-7.089,-4.373,2.140,
     &11.382,17.284,22.182,24.566,24.774,23.091,
     &21.780,21.061,20.226,19.092,17.944,15.900,
     &13.639,11.130,9.295,7.872,7.226,-999.000,
     &7.229,-999.000,9.322,-999.000,13.312,16.256,
     &21.744,22.986,20.412,14.977,11.371,-999.000,
     &5.569,-999.000,2.399,-999.000,-0.411,-999.000,
     &-3.251,-6.862,-10.518,-14.984,-18.666
     &/
       data (fjrocvz(it,15,3, 3), it=1,59) /
     &-18.300,-20.249,-20.668,-21.235,-23.536,-27.191,
     &-30.109,-33.545,-37.728,-38.873,-30.906,-19.423,
     &-13.014,-12.208,-11.450,-9.091,-5.477,0.803,
     &8.745,14.628,19.202,21.483,20.942,18.919,
     &17.243,16.072,15.304,14.170,12.726,11.325,
     &10.038,8.655,6.762,5.752,5.465,-999.000,
     &4.960,-999.000,7.150,-999.000,11.068,14.719,
     &17.549,16.283,12.722,9.533,6.930,-999.000,
     &1.880,-999.000,-1.114,-999.000,-2.949,-999.000,
     &-5.620,-8.805,-11.955,-15.154,-18.300
     &/
       data (fjrocvz(it,16,3, 3), it=1,59) /
     &-16.381,-17.884,-18.000,-18.401,-21.113,-25.529,
     &-29.989,-33.841,-39.165,-41.846,-37.464,-24.867,
     &-15.079,-12.061,-11.159,-8.813,-4.844,0.904,
     &7.892,13.289,17.072,18.296,17.000,15.257,
     &13.591,11.839,9.889,7.307,4.924,3.378,
     &2.691,1.660,0.175,-1.082,-1.406,-999.000,
     &-0.766,-999.000,1.073,-999.000,4.101,7.497,
     &8.572,4.911,2.717,-999.000,-2.500,-999.000,
     &-6.220,-999.000,-8.551,-999.000,-9.157,-999.000,
     &-9.686,-11.329,-12.679,-14.363,-16.381
     &/
       data (fjrocvz(it,17,3, 3), it=1,59) /
     &-17.459,-18.423,-17.981,-17.765,-19.766,-23.190,
     &-27.547,-34.116,-41.714,-44.532,-39.327,-24.230,
     &-12.792,-10.182,-8.257,-5.021,-0.918,3.967,
     &10.153,15.577,18.983,20.019,19.022,16.678,
     &14.223,12.029,9.691,6.607,3.592,1.239,
     &-0.834,-2.164,-3.346,-4.630,-5.404,-999.000,
     &-6.190,-999.000,-5.048,-999.000,-1.951,3.111,
     &4.620,1.443,-3.194,-999.000,-8.024,-999.000,
     &-12.098,-999.000,-14.068,-999.000,-14.456,-999.000,
     &-14.477,-15.490,-16.061,-16.505,-17.459
     &/
       data (fjrocvz(it,18,3, 3), it=1,59) /
     &-24.932,-24.320,-22.712,-20.905,-20.489,-22.052,
     &-24.745,-31.602,-41.562,-44.283,-37.343,-15.767,
     &-7.156,-6.239,-4.395,-1.027,3.102,8.800,
     &15.191,20.218,22.762,23.718,23.166,21.265,
     &18.363,15.195,12.572,9.379,6.392,3.377,
     &0.687,-1.239,-3.020,-4.656,-6.323,-999.000,
     &-7.584,-999.000,-6.115,-999.000,-3.394,-0.526,
     &-1.675,-4.882,-8.021,-999.000,-13.533,-999.000,
     &-17.471,-999.000,-18.982,-999.000,-19.274,-999.000,
     &-19.392,-20.916,-22.392,-23.605,-24.932
     &/
       data (fjrocvz(it,19,3, 3), it=1,59) /
     &-27.283,-26.950,-25.208,-23.164,-22.475,-24.298,
     &-25.549,-28.230,-34.540,-35.988,-30.251,-13.646,
     &-5.238,-3.509,-1.854,1.069,5.622,11.681,
     &18.117,22.952,26.077,26.724,25.975,24.377,
     &22.044,19.251,16.121,13.006,10.079,6.951,
     &4.527,2.793,1.082,-1.398,-3.345,-999.000,
     &-3.725,-999.000,-3.095,-999.000,-0.893,0.386,
     &-1.611,-4.641,-9.157,-999.000,-13.721,-999.000,
     &-16.832,-999.000,-19.475,-999.000,-20.370,-999.000,
     &-20.518,-21.934,-23.413,-25.551,-27.283
     &/
       data (fjrocvz(it,20,3, 3), it=1,59) /
     &-24.814,-25.145,-23.944,-22.190,-22.314,-24.983,
     &-27.635,-30.119,-31.761,-29.462,-23.046,-13.346,
     &-6.015,-2.624,-0.934,1.131,4.974,11.437,
     &18.300,24.585,28.205,28.692,27.370,25.162,
     &22.981,20.954,19.405,17.987,15.594,12.522,
     &9.634,7.411,4.881,2.608,1.249,-999.000,
     &0.437,-999.000,1.291,-999.000,3.555,5.403,
     &4.180,1.279,-3.457,-999.000,-8.239,-999.000,
     &-11.092,-999.000,-13.872,-999.000,-15.431,-999.000,
     &-17.397,-19.240,-20.697,-22.638,-24.814
     &/
       data (fjrocvz(it,21,3, 3), it=1,59) /
     &-21.525,-22.631,-21.893,-20.820,-21.782,-25.372,
     &-28.322,-30.313,-33.475,-35.276,-31.951,-21.627,
     &-10.367,-5.309,-3.088,-1.153,1.882,6.948,
     &13.318,19.005,22.539,23.307,22.691,21.746,
     &20.456,19.105,18.310,16.932,14.981,12.245,
     &9.946,7.782,5.838,4.671,3.838,-999.000,
     &3.412,-999.000,3.292,-999.000,3.314,4.350,
     &5.840,4.926,1.456,-999.000,-3.039,-999.000,
     &-5.064,-999.000,-8.561,-999.000,-10.931,-999.000,
     &-13.327,-15.469,-17.005,-19.071,-21.525
     &/
       data (fjrocvz(it,22,3, 3), it=1,59) /
     &-18.825,-18.811,-17.751,-17.810,-20.198,-23.472,
     &-25.999,-29.612,-35.631,-39.159,-36.540,-26.136,
     &-14.455,-6.916,-4.837,-3.068,-0.633,2.757,
     &7.346,11.631,14.903,15.528,15.017,14.810,
     &14.485,13.638,12.696,11.159,9.102,7.122,
     &5.496,4.420,3.632,3.563,4.464,-999.000,
     &4.643,-999.000,4.125,-999.000,5.041,5.538,
     &9.148,9.239,3.543,-999.000,-1.497,-999.000,
     &-4.368,-999.000,-6.986,-999.000,-9.352,-999.000,
     &-11.636,-14.127,-16.124,-17.924,-18.825
     &/
       data (fjrocvz(it,23,3, 3), it=1,59) /
     &-20.137,-19.747,-18.276,-17.675,-20.095,-23.576,
     &-26.880,-30.005,-33.483,-34.872,-30.930,-21.051,
     &-9.888,-4.510,-3.798,-2.670,-0.718,1.280,
     &4.319,8.596,12.489,14.037,13.101,11.319,
     &9.595,7.940,6.927,5.893,4.312,2.764,
     &2.317,2.032,1.449,1.476,2.684,-999.000,
     &2.815,-999.000,4.108,-999.000,7.181,10.655,
     &13.499,13.300,7.408,-999.000,0.055,-999.000,
     &-4.260,-999.000,-6.982,-999.000,-9.651,-999.000,
     &-12.687,-15.476,-17.615,-19.254,-20.137
     &/
       data (fjrocvz(it,24,3, 3), it=1,59) /
     &-22.537,-23.296,-22.011,-20.257,-20.217,-21.623,
     &-23.570,-25.947,-30.040,-31.295,-26.900,-16.215,
     &-5.869,-1.257,0.482,2.007,3.282,3.820,
     &5.597,8.685,11.906,12.917,12.277,10.737,
     &7.950,5.872,4.885,5.453,5.355,4.682,
     &4.593,4.308,3.841,3.191,3.320,-999.000,
     &4.573,-999.000,6.714,-999.000,10.646,14.421,
     &14.354,12.414,9.523,-999.000,-0.166,-999.000,
     &-6.305,-999.000,-8.437,-999.000,-11.561,-999.000,
     &-15.426,-18.735,-20.574,-21.533,-22.537
     &/
       data (fjrocvz(it,25,3, 3), it=1,59) /
     &-22.032,-23.356,-23.190,-22.275,-20.936,-19.293,
     &-19.030,-21.760,-27.572,-29.917,-23.388,-10.200,
     &0.540,4.694,7.437,9.553,9.607,8.219,
     &8.345,10.757,13.677,14.771,14.223,11.766,
     &8.537,7.244,7.957,9.233,9.463,8.387,
     &7.685,7.544,7.243,6.230,6.191,-999.000,
     &6.508,-999.000,7.683,-999.000,12.604,15.454,
     &15.494,12.861,4.747,-999.000,-0.949,-999.000,
     &-7.532,-999.000,-10.577,-999.000,-13.061,-999.000,
     &-16.285,-18.904,-20.539,-21.083,-22.032
     &/
       data (fjrocvz(it, 1,4, 3), it=1,59) /
     &-15.403,-14.516,-13.213,-12.576,-13.623,-14.603,
     &-15.471,-16.611,-17.434,-16.976,-13.973,-7.617,
     &-8.765,-3.937,1.426,7.655,13.478,22.377,
     &29.380,32.209,31.871,30.205,27.522,23.175,
     &18.595,14.006,8.453,4.341,3.160,4.199,
     &4.720,4.840,4.946,6.183,8.379,-999.000,
     &11.202,-999.000,14.376,15.694,14.454,9.105,
     &2.785,-999.000,-1.390,-999.000,-7.449,-999.000,
     &-12.279,-999.000,-14.370,-999.000,-17.723,-999.000,
     &-18.745,-18.599,-18.156,-16.919,-15.403
     &/
       data (fjrocvz(it, 2,4, 3), it=1,59) /
     &-13.712,-12.989,-12.361,-12.128,-11.640,-10.975,
     &-11.391,-13.376,-14.861,-13.983,-8.514,-0.627,
     &5.158,8.308,12.576,15.241,19.235,24.547,
     &30.002,33.678,34.445,33.071,30.037,25.504,
     &20.787,16.934,12.947,9.026,6.491,6.227,
     &7.189,7.892,8.371,9.450,10.580,-999.000,
     &12.136,-999.000,15.985,17.004,17.329,15.443,
     &9.812,-999.000,-1.276,-999.000,-8.978,-999.000,
     &-10.641,-999.000,-14.622,-999.000,-18.245,-999.000,
     &-18.435,-18.386,-17.035,-14.857,-13.712
     &/
       data (fjrocvz(it, 3,4, 3), it=1,59) /
     &-13.835,-12.605,-11.978,-11.080,-10.278,-10.144,
     &-9.951,-10.316,-10.951,-10.432,-5.857,0.979,
     &7.002,10.295,12.710,15.816,19.495,22.721,
     &26.073,28.685,29.940,28.702,25.486,21.370,
     &17.479,14.507,11.841,9.446,7.295,6.725,
     &7.632,8.580,9.572,10.026,11.295,-999.000,
     &13.800,-999.000,16.137,19.902,22.131,17.684,
     &8.251,1.854,-2.010,-8.662,-12.117,-11.965,
     &-12.474,-999.000,-18.065,-999.000,-20.120,-999.000,
     &-20.327,-19.108,-17.483,-15.565,-13.835
     &/
       data (fjrocvz(it, 4,4, 3), it=1,59) /
     &-15.148,-13.976,-12.459,-11.070,-10.830,-10.559,
     &-9.821,-8.384,-7.417,-6.800,-3.709,2.009,
     &8.012,10.425,11.658,14.266,17.781,21.778,
     &25.134,27.011,26.468,24.051,20.853,17.216,
     &13.673,10.728,8.506,7.099,6.329,6.300,
     &7.025,8.295,9.292,10.800,13.470,-999.000,
     &15.247,-999.000,18.301,22.319,22.473,17.942,
     &11.899,6.928,-1.981,-12.535,-14.942,-999.000,
     &-14.606,-999.000,-19.477,-999.000,-21.957,-999.000,
     &-21.148,-19.021,-17.626,-16.091,-15.148
     &/
       data (fjrocvz(it, 5,4, 3), it=1,59) /
     &-15.959,-14.747,-12.588,-11.055,-11.083,-11.215,
     &-10.782,-9.274,-7.960,-7.128,-4.154,2.037,
     &7.998,10.670,13.175,15.125,18.437,23.010,
     &26.983,29.135,28.599,26.168,22.447,18.399,
     &14.510,11.811,10.011,9.052,8.099,7.656,
     &8.216,9.493,11.013,12.346,13.567,-999.000,
     &17.023,-999.000,20.738,23.039,24.268,23.672,
     &18.542,8.968,-1.553,-9.199,-13.724,-16.753,
     &-19.517,-23.525,-25.237,-24.844,-23.370,-999.000,
     &-21.599,-19.388,-17.719,-16.557,-15.959
     &/
       data (fjrocvz(it, 6,4, 3), it=1,59) /
     &-16.831,-15.990,-14.584,-13.296,-13.050,-12.122,
     &-10.520,-9.231,-7.713,-6.012,-3.971,-2.755,
     &0.323,5.600,11.727,16.366,19.405,23.790,
     &27.779,29.884,30.412,28.879,26.084,22.301,
     &18.921,16.509,14.455,12.646,11.035,9.782,
     &9.525,10.897,12.679,13.133,14.367,-999.000,
     &18.127,-999.000,22.236,26.346,30.428,30.750,
     &26.235,15.909,4.197,-5.730,-12.173,-18.480,
     &-21.430,-999.000,-24.988,-999.000,-24.847,-999.000,
     &-23.014,-19.971,-18.362,-17.666,-16.831
     &/
       data (fjrocvz(it, 7,4, 3), it=1,59) /
     &-18.281,-17.694,-16.627,-15.638,-14.891,-12.554,
     &-10.436,-8.905,-6.630,-5.771,-3.748,-1.541,
     &3.404,5.663,9.248,12.294,15.426,20.490,
     &25.303,28.861,29.623,29.244,27.759,25.259,
     &22.554,19.930,17.616,15.577,13.319,11.171,
     &10.393,11.069,12.347,13.055,13.650,-999.000,
     &16.289,-999.000,20.169,25.953,31.334,35.432,
     &29.213,15.396,3.315,-4.758,-12.063,-20.379,
     &-25.183,-26.917,-27.170,-26.886,-25.951,-999.000,
     &-23.906,-21.906,-20.451,-18.907,-18.281
     &/
       data (fjrocvz(it, 8,4, 3), it=1,59) /
     &-17.873,-16.295,-15.275,-14.717,-13.817,-12.580,
     &-11.547,-10.480,-8.384,-10.628,-12.669,-14.627,
     &-9.798,-2.989,2.372,7.600,10.315,13.771,
     &19.511,23.915,25.836,26.065,25.469,24.103,
     &22.234,20.390,18.104,15.901,13.273,10.856,
     &9.784,10.107,11.279,11.924,12.340,-999.000,
     &15.077,-999.000,19.482,23.180,28.083,28.793,
     &21.961,13.321,4.130,-4.550,-11.227,-18.257,
     &-22.484,-23.393,-21.806,-999.000,-22.381,-999.000,
     &-21.928,-21.771,-20.833,-19.223,-17.873
     &/
       data (fjrocvz(it, 9,4, 3), it=1,59) /
     &-14.891,-12.605,-10.621,-9.292,-9.025,-9.438,
     &-10.763,-11.964,-12.559,-14.563,-17.417,-19.842,
     &-15.164,-5.736,1.997,6.938,8.479,11.902,
     &16.857,20.374,21.831,22.377,22.035,20.520,
     &18.191,15.974,14.119,12.177,9.936,8.650,
     &8.568,9.136,10.247,10.775,11.657,-999.000,
     &13.458,-999.000,18.006,22.817,26.496,29.687,
     &29.340,21.481,10.416,0.846,-5.261,-999.000,
     &-10.033,-999.000,-13.702,-999.000,-16.726,-999.000,
     &-17.495,-17.760,-18.159,-17.098,-14.891
     &/
       data (fjrocvz(it,10,4, 3), it=1,59) /
     &-14.646,-12.351,-10.004,-8.627,-7.499,-8.543,
     &-11.774,-14.361,-15.432,-16.240,-17.113,-17.345,
     &-13.802,-3.602,4.134,8.951,10.758,14.786,
     &19.888,23.038,23.330,22.637,21.170,19.399,
     &17.037,14.343,12.038,9.995,8.554,8.004,
     &8.060,8.384,8.339,8.503,9.855,-999.000,
     &12.092,-999.000,18.841,24.466,30.388,33.751,
     &32.984,23.017,13.102,-999.000,4.227,-999.000,
     &-3.194,-999.000,-9.152,-999.000,-10.872,-999.000,
     &-12.528,-13.919,-15.325,-15.322,-14.646
     &/
       data (fjrocvz(it,11,4, 3), it=1,59) /
     &-16.848,-16.203,-15.904,-15.665,-15.087,-15.729,
     &-16.569,-18.119,-19.582,-21.202,-20.563,-16.744,
     &-11.763,-2.440,4.777,10.476,14.430,17.490,
     &21.741,25.259,26.402,26.016,24.838,22.941,
     &20.861,18.251,15.142,12.875,10.977,10.002,
     &9.330,8.534,7.972,8.112,10.018,-999.000,
     &13.914,-999.000,19.645,25.424,32.799,33.158,
     &28.358,19.927,11.523,-999.000,0.653,-999.000,
     &-7.310,-999.000,-11.625,-999.000,-12.477,-999.000,
     &-15.032,-15.212,-16.480,-17.561,-16.848
     &/
       data (fjrocvz(it,12,4, 3), it=1,59) /
     &-19.203,-20.101,-20.062,-19.929,-20.155,-19.754,
     &-19.449,-21.131,-23.905,-25.140,-25.402,-21.709,
     &-17.314,-8.240,1.275,6.976,11.463,16.143,
     &22.628,27.369,29.051,28.617,27.347,25.914,
     &24.052,20.977,17.604,14.855,13.004,12.198,
     &11.220,9.939,8.966,8.979,11.151,-999.000,
     &14.952,-999.000,20.657,26.196,33.252,36.279,
     &30.915,17.244,7.032,-0.020,-5.830,-999.000,
     &-9.127,-999.000,-11.567,-999.000,-15.431,-999.000,
     &-17.737,-18.013,-18.566,-19.028,-19.203
     &/
       data (fjrocvz(it,13,4, 3), it=1,59) /
     &-19.549,-19.150,-18.971,-19.030,-18.723,-19.365,
     &-20.870,-22.164,-23.775,-23.580,-22.933,-20.696,
     &-17.024,-9.104,-0.484,6.057,10.614,15.933,
     &24.905,30.440,31.033,30.227,29.030,27.687,
     &25.535,22.083,18.778,16.422,15.232,14.871,
     &14.082,12.732,11.230,10.999,12.722,-999.000,
     &15.634,-999.000,23.570,28.425,35.456,39.451,
     &31.920,17.886,3.573,-3.535,-6.138,-7.832,
     &-8.622,-999.000,-11.868,-999.000,-15.168,-999.000,
     &-17.708,-19.940,-20.057,-19.532,-19.549
     &/
       data (fjrocvz(it,14,4, 3), it=1,59) /
     &-18.377,-17.976,-17.661,-17.096,-17.079,-18.642,
     &-21.335,-24.894,-28.142,-27.113,-23.563,-18.842,
     &-13.999,-11.356,-4.636,3.000,8.536,14.598,
     &23.711,29.131,30.457,29.892,28.534,26.876,
     &24.293,21.473,18.934,17.235,16.400,15.929,
     &14.835,13.459,11.923,11.755,13.732,-999.000,
     &17.654,-999.000,25.378,31.132,34.869,35.053,
     &28.160,15.775,7.129,-999.000,-3.304,-999.000,
     &-11.349,-999.000,-14.965,-999.000,-17.864,-999.000,
     &-20.794,-21.649,-20.824,-19.291,-18.377
     &/
       data (fjrocvz(it,15,4, 3), it=1,59) /
     &-16.611,-15.993,-15.286,-14.760,-14.638,-15.888,
     &-19.585,-25.541,-29.488,-30.301,-26.559,-18.845,
     &-13.412,-9.451,-4.501,2.658,8.370,13.944,
     &21.345,26.319,27.969,27.150,25.142,22.366,
     &19.415,17.066,14.825,13.234,12.246,11.394,
     &10.469,9.438,8.867,9.533,12.126,-999.000,
     &15.410,-999.000,21.709,26.131,28.932,30.660,
     &24.991,12.925,4.822,-999.000,-6.661,-999.000,
     &-13.676,-999.000,-18.490,-999.000,-21.993,-999.000,
     &-23.977,-24.210,-21.831,-18.338,-16.611
     &/
       data (fjrocvz(it,16,4, 3), it=1,59) /
     &-16.692,-15.214,-15.261,-15.274,-15.607,-17.171,
     &-20.600,-24.998,-28.037,-30.806,-29.641,-21.923,
     &-12.339,-8.423,-3.282,2.725,7.802,13.196,
     &20.496,25.522,27.209,26.675,25.033,21.475,
     &17.088,13.148,10.316,8.536,7.630,7.138,
     &6.290,5.163,4.446,5.062,7.223,-999.000,
     &10.308,-999.000,16.067,19.733,22.819,24.146,
     &18.180,5.865,1.282,-999.000,-9.266,-999.000,
     &-16.840,-999.000,-21.755,-999.000,-24.730,-999.000,
     &-26.097,-25.504,-22.918,-19.423,-16.692
     &/
       data (fjrocvz(it,17,4, 3), it=1,59) /
     &-19.448,-18.454,-17.582,-16.952,-17.345,-18.256,
     &-20.919,-25.031,-27.896,-29.497,-27.750,-19.588,
     &-10.746,-6.153,-0.228,4.719,8.864,14.501,
     &22.232,27.395,29.626,29.711,28.256,25.051,
     &20.114,15.283,11.240,8.313,7.033,6.260,
     &5.105,3.548,1.856,1.359,3.327,-999.000,
     &7.296,-999.000,13.887,16.342,19.138,18.064,
     &13.247,3.909,-3.841,-11.798,-18.028,-21.641,
     &-23.311,-999.000,-23.959,-999.000,-25.809,-999.000,
     &-26.254,-24.844,-23.081,-21.131,-19.448
     &/
       data (fjrocvz(it,18,4, 3), it=1,59) /
     &-20.392,-18.687,-17.345,-16.871,-16.559,-16.660,
     &-19.841,-23.997,-25.912,-27.465,-24.911,-17.590,
     &-8.673,-3.955,0.385,5.419,10.030,17.057,
     &24.101,29.619,33.103,33.262,30.754,26.507,
     &22.121,17.948,13.547,9.628,7.398,5.825,
     &4.434,2.742,1.310,1.332,3.592,-999.000,
     &6.764,9.240,13.180,15.959,16.680,13.879,
     &6.826,0.873,-7.027,-999.000,-18.805,-999.000,
     &-25.321,-999.000,-27.600,-999.000,-27.613,-999.000,
     &-26.199,-25.348,-23.483,-21.776,-20.392
     &/
       data (fjrocvz(it,19,4, 3), it=1,59) /
     &-18.983,-17.094,-16.786,-16.205,-15.662,-15.399,
     &-17.537,-19.674,-20.785,-22.174,-19.318,-12.131,
     &-3.752,-0.916,3.306,8.488,12.903,19.647,
     &27.447,33.414,36.425,36.340,33.491,29.188,
     &24.414,19.666,15.154,11.163,9.279,8.278,
     &6.725,4.818,3.320,2.981,4.937,-999.000,
     &8.826,-999.000,14.846,16.027,15.588,9.832,
     &6.010,-0.316,-8.613,-17.831,-23.441,-26.764,
     &-27.507,-999.000,-28.119,-999.000,-26.858,-999.000,
     &-25.196,-24.493,-22.929,-21.054,-18.983
     &/
       data (fjrocvz(it,20,4, 3), it=1,59) /
     &-21.283,-19.385,-17.531,-15.659,-14.221,-13.536,
     &-15.152,-17.719,-18.300,-18.764,-16.099,-6.753,
     &1.523,2.299,6.120,9.975,13.700,20.672,
     &28.191,34.942,38.218,38.506,36.300,32.554,
     &28.225,23.775,19.066,15.057,12.653,11.028,
     &9.190,7.133,5.307,5.077,7.472,-999.000,
     &10.899,-999.000,15.400,16.260,18.404,17.044,
     &10.982,0.366,-7.713,-14.040,-19.053,-23.868,
     &-26.477,-999.000,-26.697,-999.000,-26.984,-999.000,
     &-26.008,-25.463,-24.448,-22.896,-21.283
     &/
       data (fjrocvz(it,21,4, 3), it=1,59) /
     &-21.730,-19.246,-17.319,-16.091,-15.376,-14.280,
     &-14.957,-17.171,-18.888,-19.193,-16.204,-8.651,
     &-2.828,0.979,5.274,10.015,14.789,20.333,
     &25.930,32.325,35.334,36.171,34.100,30.176,
     &26.097,22.251,18.163,13.638,10.466,8.418,
     &6.587,5.276,4.392,4.202,5.804,-999.000,
     &9.285,-999.000,15.063,17.860,20.662,19.534,
     &10.710,-999.000,-3.026,-999.000,-15.442,-999.000,
     &-22.618,-999.000,-25.147,-999.000,-25.698,-999.000,
     &-25.263,-24.855,-24.194,-23.228,-21.730
     &/
       data (fjrocvz(it,22,4, 3), it=1,59) /
     &-20.628,-19.718,-18.655,-17.944,-18.180,-17.678,
     &-17.764,-19.808,-23.236,-23.982,-20.891,-13.913,
     &-5.634,-2.740,2.595,7.310,11.872,17.426,
     &23.956,29.719,32.042,31.799,29.397,25.062,
     &19.878,15.172,11.278,7.849,5.200,3.567,
     &2.631,2.846,3.314,3.356,4.830,-999.000,
     &9.735,-999.000,15.320,14.981,16.668,15.221,
     &9.037,-999.000,-5.517,-999.000,-16.228,-999.000,
     &-23.499,-999.000,-24.348,-999.000,-24.142,-999.000,
     &-24.123,-23.565,-22.582,-21.602,-20.628
     &/
       data (fjrocvz(it,23,4, 3), it=1,59) /
     &-18.349,-18.163,-16.838,-15.099,-15.046,-16.117,
     &-17.958,-20.244,-22.504,-23.392,-20.917,-14.784,
     &-9.605,-6.515,-1.028,4.246,9.489,15.018,
     &23.189,28.840,30.538,29.123,26.242,21.811,
     &16.644,11.378,7.416,4.449,2.447,1.474,
     &0.928,0.740,1.798,3.303,5.706,-999.000,
     &8.382,-999.000,13.010,14.696,14.976,14.164,
     &10.223,1.980,-4.722,-999.000,-14.583,-999.000,
     &-20.826,-999.000,-22.979,-999.000,-23.062,-999.000,
     &-22.945,-22.050,-20.236,-18.673,-18.349
     &/
       data (fjrocvz(it,24,4, 3), it=1,59) /
     &-16.126,-14.938,-13.411,-11.773,-12.375,-14.518,
     &-16.664,-18.066,-19.518,-20.483,-18.853,-14.110,
     &-7.746,-2.245,1.325,4.745,9.618,17.787,
     &25.977,29.601,30.212,28.655,25.377,20.679,
     &15.837,10.685,6.175,2.999,1.890,2.492,
     &2.639,2.159,2.945,4.385,6.351,-999.000,
     &8.808,-999.000,13.046,14.359,15.234,13.055,
     &7.017,-999.000,-0.752,-999.000,-8.186,-999.000,
     &-13.835,-999.000,-17.814,-999.000,-18.872,-999.000,
     &-19.939,-19.635,-18.294,-17.244,-16.126
     &/
       data (fjrocvz(it,25,4, 3), it=1,59) /
     &-15.403,-14.516,-13.213,-12.576,-13.623,-14.603,
     &-15.471,-16.611,-17.434,-16.976,-13.973,-7.617,
     &-8.765,-3.937,1.426,7.655,13.478,22.377,
     &29.380,32.209,31.871,30.205,27.522,23.175,
     &18.595,14.006,8.453,4.341,3.160,4.199,
     &4.720,4.840,4.946,6.183,8.379,-999.000,
     &11.202,-999.000,14.376,15.694,14.454,9.105,
     &2.785,-999.000,-1.390,-999.000,-7.449,-999.000,
     &-12.279,-999.000,-14.370,-999.000,-17.723,-999.000,
     &-18.745,-18.599,-18.156,-16.919,-15.403
     &/
       data (fjrocvz(it, 1,1, 4), it=1,59) /
     &-11.848,-11.414,-11.162,-11.384,-11.936,-12.667,
     &-14.149,-15.760,-17.572,-19.197,-18.126,-11.912,
     &-2.408,5.241,11.153,15.542,19.960,24.740,
     &28.876,31.474,31.299,29.544,27.229,24.268,
     &21.181,17.020,12.997,9.911,6.517,3.020,
     &0.937,0.301,0.204,-0.128,-0.388,-999.000,
     &-0.872,-999.000,0.023,-999.000,2.165,-999.000,
     &1.443,-999.000,-0.319,-999.000,-4.321,-999.000,
     &-8.446,-999.000,-11.634,-999.000,-13.354,-999.000,
     &-15.234,-16.041,-15.202,-13.172,-11.848
     &/
       data (fjrocvz(it, 2,1, 4), it=1,59) /
     &-13.649,-12.674,-12.406,-12.433,-13.582,-14.748,
     &-15.281,-16.204,-17.178,-18.279,-14.709,-5.220,
     &3.566,10.165,15.625,21.418,26.271,30.145,
     &34.025,36.481,36.545,34.663,31.468,27.809,
     &23.696,18.803,14.046,10.259,6.661,3.049,
     &-0.041,-1.952,-3.078,-3.128,-3.227,-999.000,
     &-2.734,-999.000,-1.083,-999.000,-1.350,-999.000,
     &-0.964,-999.000,-3.147,-999.000,-7.062,-999.000,
     &-11.189,-999.000,-14.562,-999.000,-16.267,-999.000,
     &-17.156,-17.472,-16.891,-15.326,-13.649
     &/
       data (fjrocvz(it, 3,1, 4), it=1,59) /
     &-14.925,-14.426,-14.324,-14.740,-15.717,-15.538,
     &-14.646,-14.691,-14.447,-13.788,-8.210,0.188,
     &7.136,12.238,17.369,22.520,25.749,29.002,
     &32.698,35.696,35.927,33.636,30.042,26.574,
     &23.512,19.954,16.004,11.963,8.207,4.216,
     &0.851,-0.996,-2.255,-1.986,-2.026,-999.000,
     &-1.112,-999.000,-0.657,-999.000,0.465,2.234,
     &3.684,2.611,-0.493,-2.071,-5.698,-999.000,
     &-9.715,-999.000,-12.902,-999.000,-14.842,-999.000,
     &-16.361,-17.154,-17.209,-16.088,-14.925
     &/
       data (fjrocvz(it, 4,1, 4), it=1,59) /
     &-14.753,-15.628,-15.700,-14.935,-14.153,-14.146,
     &-14.108,-14.324,-13.101,-10.468,-3.484,4.554,
     &9.343,12.912,18.027,22.197,25.329,28.493,
     &32.040,34.176,33.253,30.080,26.336,23.513,
     &21.034,17.272,13.402,10.519,8.512,5.784,
     &2.788,0.347,-1.194,-0.790,0.762,-999.000,
     &1.158,-999.000,2.377,-999.000,1.928,4.930,
     &7.470,4.811,0.494,-999.000,-3.463,-999.000,
     &-7.110,-999.000,-10.548,-999.000,-13.179,-999.000,
     &-15.720,-16.201,-15.608,-14.306,-14.753
     &/
       data (fjrocvz(it, 5,1, 4), it=1,59) /
     &-11.637,-12.408,-12.503,-11.542,-11.004,-12.021,
     &-13.864,-15.011,-14.058,-11.375,-3.949,4.785,
     &8.961,12.178,17.267,21.814,24.856,27.818,
     &31.129,33.241,32.689,29.972,27.063,24.424,
     &21.347,16.821,12.779,10.513,9.323,7.795,
     &5.684,2.906,0.876,0.189,0.890,-999.000,
     &2.342,-999.000,5.018,-999.000,5.622,9.540,
     &10.013,6.533,2.918,-999.000,-1.827,-999.000,
     &-5.983,-999.000,-10.032,-999.000,-13.288,-999.000,
     &-15.207,-14.859,-13.387,-11.618,-11.637
     &/
       data (fjrocvz(it, 6,1, 4), it=1,59) /
     &-10.178,-10.788,-11.450,-12.143,-12.448,-13.272,
     &-15.146,-16.414,-15.643,-12.468,-6.160,1.445,
     &5.250,7.684,11.259,15.023,17.995,21.707,
     &26.005,29.473,29.866,28.386,26.407,24.248,
     &21.494,17.875,14.679,12.136,10.593,9.558,
     &8.684,7.062,4.229,2.327,2.410,-999.000,
     &3.566,-999.000,5.414,-999.000,8.436,-999.000,
     &8.708,-999.000,5.284,-999.000,1.773,-999.000,
     &-2.289,-999.000,-6.189,-999.000,-9.984,-999.000,
     &-12.732,-13.840,-13.004,-10.929,-10.178
     &/
       data (fjrocvz(it, 7,1, 4), it=1,59) /
     &-13.437,-13.949,-14.878,-15.696,-16.102,-16.293,
     &-16.531,-16.403,-15.415,-13.716,-10.205,-6.755,
     &-3.564,0.937,5.828,9.424,12.748,16.386,
     &21.460,25.027,26.029,24.897,22.799,20.784,
     &18.983,16.469,13.454,10.757,8.965,8.471,
     &8.937,8.308,6.135,3.990,3.422,-999.000,
     &3.450,-999.000,5.598,-999.000,10.038,12.439,
     &16.906,16.539,10.377,-999.000,5.387,-999.000,
     &0.977,-999.000,-3.367,-999.000,-7.864,-999.000,
     &-11.444,-13.991,-14.554,-14.017,-13.437
     &/
       data (fjrocvz(it, 8,1, 4), it=1,59) /
     &-17.119,-16.377,-16.047,-15.839,-15.713,-15.725,
     &-16.219,-16.900,-17.359,-17.391,-14.945,-10.029,
     &-5.424,-2.366,0.388,3.014,5.989,10.418,
     &15.994,20.305,22.376,21.790,19.761,17.645,
     &15.610,13.251,10.813,8.364,7.008,6.741,
     &7.953,8.386,7.563,5.864,5.368,-999.000,
     &6.004,-999.000,9.113,-999.000,15.915,23.276,
     &26.757,27.180,21.075,12.399,8.342,-999.000,
     &-1.124,-999.000,-7.458,-999.000,-11.757,-999.000,
     &-15.294,-18.012,-18.872,-18.100,-17.119
     &/
       data (fjrocvz(it, 9,1, 4), it=1,59) /
     &-18.371,-17.037,-16.101,-15.426,-14.879,-15.321,
     &-16.582,-17.931,-18.760,-19.131,-17.225,-13.395,
     &-9.679,-5.691,-1.220,1.914,4.136,7.730,
     &12.839,17.386,19.815,20.202,19.310,17.921,
     &15.929,14.170,12.668,11.195,9.887,8.932,
     &9.163,9.034,8.241,7.250,7.631,7.168,
     &7.057,7.875,11.757,17.118,23.019,31.546,
     &37.541,34.142,20.012,6.078,-1.966,-6.989,
     &-8.996,-999.000,-14.396,-999.000,-17.687,-999.000,
     &-19.922,-21.673,-21.558,-20.026,-18.371
     &/
       data (fjrocvz(it,10,1, 4), it=1,59) /
     &-17.094,-15.701,-14.748,-14.057,-14.294,-15.484,
     &-16.649,-17.482,-18.662,-20.775,-19.897,-15.440,
     &-9.749,-3.761,1.279,5.085,7.731,10.599,
     &14.342,17.784,20.561,22.440,23.042,22.374,
     &20.439,18.627,17.219,15.944,14.300,12.800,
     &12.007,11.239,9.987,8.804,8.213,-999.000,
     &7.934,10.366,17.123,26.046,35.938,42.480,
     &37.064,24.800,10.331,-3.843,-9.032,-12.313,
     &-14.431,-999.000,-18.154,-999.000,-20.947,-999.000,
     &-22.305,-22.273,-21.050,-19.204,-17.094
     &/
       data (fjrocvz(it,11,1, 4), it=1,59) /
     &-16.273,-14.957,-14.538,-14.972,-16.327,-17.840,
     &-19.148,-19.956,-20.738,-21.764,-19.783,-12.916,
     &-5.598,0.763,6.178,10.023,12.186,13.809,
     &16.495,20.545,24.720,27.321,27.906,26.659,
     &24.738,22.674,20.972,18.708,16.354,14.429,
     &12.696,11.451,10.363,9.698,9.839,9.583,
     &10.242,14.130,20.528,27.724,32.201,32.889,
     &21.273,7.530,-2.004,-10.666,-15.964,-999.000,
     &-17.704,-999.000,-20.271,-999.000,-21.808,-999.000,
     &-22.148,-21.433,-19.898,-17.975,-16.273
     &/
       data (fjrocvz(it,12,1, 4), it=1,59) /
     &-18.180,-17.632,-17.899,-19.196,-20.426,-20.638,
     &-20.867,-21.552,-22.482,-22.790,-20.987,-16.024,
     &-8.286,-1.021,7.872,12.154,14.401,16.441,
     &19.134,23.049,26.296,28.291,28.876,28.402,
     &27.279,25.191,22.573,19.460,16.139,13.406,
     &11.728,10.364,9.506,10.114,12.364,-999.000,
     &14.742,14.653,18.879,25.229,29.371,27.009,
     &18.021,8.935,1.299,-999.000,-12.003,-999.000,
     &-19.992,-999.000,-22.732,-999.000,-22.948,-999.000,
     &-22.194,-21.324,-20.359,-19.122,-18.180
     &/
       data (fjrocvz(it,13,1, 4), it=1,59) /
     &-19.294,-19.705,-20.192,-21.011,-21.247,-20.195,
     &-19.956,-21.025,-21.844,-22.484,-21.197,-16.518,
     &-7.313,-1.172,6.099,11.646,14.474,16.789,
     &19.608,22.988,25.557,26.930,27.017,26.282,
     &24.670,22.499,20.443,17.580,13.573,11.022,
     &10.375,10.078,9.670,9.472,9.582,-999.000,
     &11.746,-999.000,15.697,20.133,23.874,20.888,
     &14.766,-999.000,4.549,-999.000,-8.544,-999.000,
     &-18.008,-999.000,-22.366,-999.000,-22.992,-999.000,
     &-22.410,-21.476,-20.393,-19.438,-19.294
     &/
       data (fjrocvz(it,14,1, 4), it=1,59) /
     &-19.280,-20.458,-21.215,-21.300,-20.503,-19.299,
     &-20.315,-21.956,-22.207,-22.303,-19.188,-15.492,
     &-8.984,-4.369,3.077,9.390,12.691,14.951,
     &17.373,20.579,23.730,25.158,24.891,23.417,
     &21.731,20.008,18.072,14.874,11.206,8.723,
     &8.185,8.028,7.082,5.181,4.669,-999.000,
     &6.666,7.845,12.743,15.522,16.889,17.042,
     &14.637,7.954,3.897,-999.000,-6.402,-999.000,
     &-15.419,-999.000,-21.165,-999.000,-23.002,-999.000,
     &-22.894,-21.846,-20.213,-19.214,-19.280
     &/
       data (fjrocvz(it,15,1, 4), it=1,59) /
     &-19.766,-20.409,-21.634,-22.369,-21.767,-20.207,
     &-20.573,-22.075,-23.790,-25.585,-22.979,-17.600,
     &-9.710,-5.176,0.713,6.346,9.975,12.896,
     &15.288,17.082,18.779,19.629,18.950,17.441,
     &16.153,15.050,13.727,11.832,8.919,6.263,
     &5.390,5.015,3.907,3.361,3.694,-999.000,
     &5.769,-999.000,11.489,15.903,19.006,20.444,
     &17.312,10.434,5.724,-999.000,-4.229,-999.000,
     &-13.278,-999.000,-20.233,-999.000,-22.959,-999.000,
     &-22.861,-22.085,-21.038,-20.229,-19.766
     &/
       data (fjrocvz(it,16,1, 4), it=1,59) /
     &-21.960,-21.973,-22.608,-23.369,-22.922,-21.076,
     &-20.794,-21.450,-23.309,-28.621,-29.495,-24.180,
     &-9.461,-2.632,1.392,5.808,9.227,11.842,
     &14.233,16.152,17.977,18.144,17.388,16.009,
     &14.623,13.208,11.689,9.415,6.300,4.168,
     &3.945,4.350,4.409,4.716,5.615,-999.000,
     &8.646,-999.000,14.759,18.898,21.075,20.513,
     &16.566,-999.000,5.489,-999.000,-3.394,-999.000,
     &-12.975,-999.000,-20.169,-999.000,-23.000,-999.000,
     &-23.577,-23.099,-22.593,-22.343,-21.960
     &/
       data (fjrocvz(it,17,1, 4), it=1,59) /
     &-22.918,-22.729,-22.904,-23.681,-24.230,-23.429,
     &-23.293,-21.823,-21.195,-25.752,-28.565,-24.160,
     &-9.302,1.360,4.750,7.541,10.130,12.455,
     &14.775,16.844,18.349,19.204,19.093,17.914,
     &16.526,14.741,12.901,10.448,7.110,4.841,
     &4.448,5.016,4.932,4.747,6.075,-999.000,
     &8.140,10.989,13.656,16.391,17.884,18.813,
     &14.543,-999.000,0.488,-999.000,-9.172,-999.000,
     &-17.996,-999.000,-23.777,-999.000,-26.082,-999.000,
     &-25.627,-24.048,-23.076,-23.008,-22.918
     &/
       data (fjrocvz(it,18,1, 4), it=1,59) /
     &-22.885,-22.601,-22.773,-23.872,-25.474,-25.042,
     &-25.131,-24.396,-22.355,-22.902,-22.173,-17.583,
     &-5.605,4.917,8.974,11.498,14.130,16.427,
     &18.575,19.410,20.065,20.990,20.796,19.358,
     &17.325,15.266,13.133,11.063,8.384,6.440,
     &5.542,5.616,5.761,6.171,7.014,-999.000,
     &9.006,-999.000,12.096,16.412,18.044,16.443,
     &10.182,1.655,-5.240,-999.000,-14.351,-999.000,
     &-21.773,-999.000,-27.310,-999.000,-28.801,-999.000,
     &-27.247,-24.985,-23.805,-23.373,-22.885
     &/
       data (fjrocvz(it,19,1, 4), it=1,59) /
     &-23.661,-22.550,-21.645,-21.849,-22.348,-20.955,
     &-19.868,-19.779,-19.027,-20.537,-20.084,-16.058,
     &-5.697,5.002,11.104,15.003,17.882,20.579,
     &22.935,23.882,24.416,24.886,24.880,23.200,
     &21.105,19.205,16.377,12.641,9.465,8.361,
     &8.139,7.900,7.628,6.887,6.906,-999.000,
     &8.450,-999.000,11.037,14.226,13.422,9.379,
     &5.071,-999.000,-6.070,-999.000,-14.776,-999.000,
     &-22.352,-999.000,-27.225,-999.000,-28.588,-999.000,
     &-27.268,-25.362,-24.320,-24.146,-23.661
     &/
       data (fjrocvz(it,20,1, 4), it=1,59) /
     &-22.506,-21.381,-20.979,-20.957,-21.194,-19.787,
     &-18.672,-18.109,-17.685,-18.764,-20.740,-18.115,
     &-8.566,3.339,9.866,13.335,17.125,20.847,
     &23.921,25.485,25.961,26.481,26.535,25.524,
     &24.677,23.695,21.456,17.608,13.257,11.454,
     &10.989,10.343,8.907,7.461,7.148,-999.000,
     &6.500,-999.000,6.755,9.143,12.482,9.433,
     &2.232,-999.000,-7.348,-999.000,-16.539,-999.000,
     &-23.803,-999.000,-28.503,-999.000,-30.014,-999.000,
     &-29.457,-27.682,-25.791,-24.093,-22.506
     &/
       data (fjrocvz(it,21,1, 4), it=1,59) /
     &-23.778,-23.048,-23.071,-23.587,-24.004,-22.223,
     &-20.185,-19.900,-20.983,-23.510,-26.057,-23.753,
     &-13.270,-0.791,6.127,9.950,14.027,17.233,
     &20.566,23.091,24.332,24.582,24.219,23.904,
     &23.958,23.209,21.085,17.151,12.470,10.278,
     &9.789,9.349,8.117,6.394,5.409,-999.000,
     &6.143,-999.000,7.542,9.808,11.276,7.208,
     &-0.779,-999.000,-10.170,-999.000,-17.576,-999.000,
     &-23.820,-999.000,-28.569,-999.000,-30.224,-999.000,
     &-30.112,-29.017,-27.470,-25.530,-23.778
     &/
       data (fjrocvz(it,22,1, 4), it=1,59) /
     &-22.095,-21.648,-21.981,-23.115,-23.883,-22.657,
     &-21.101,-21.232,-22.515,-24.713,-26.545,-24.293,
     &-14.319,-2.565,4.448,8.348,11.497,15.180,
     &18.843,22.217,23.302,23.342,23.630,23.692,
     &23.183,21.532,18.337,13.392,8.228,5.772,
     &5.463,5.465,4.449,3.178,2.962,-999.000,
     &3.562,-999.000,3.122,-999.000,3.880,-999.000,
     &-0.156,-999.000,-10.804,-999.000,-16.118,-999.000,
     &-21.029,-999.000,-25.420,-999.000,-27.416,-999.000,
     &-27.415,-26.173,-24.398,-23.077,-22.095
     &/
       data (fjrocvz(it,23,1, 4), it=1,59) /
     &-16.494,-16.675,-17.094,-17.988,-19.138,-18.730,
     &-18.440,-18.948,-19.373,-20.194,-20.275,-16.647,
     &-8.558,-0.494,5.328,8.278,10.809,13.712,
     &17.296,20.757,22.862,24.941,26.000,25.384,
     &23.536,20.835,17.897,13.990,9.182,5.953,
     &3.969,2.887,1.949,1.385,1.499,-999.000,
     &2.322,-999.000,5.628,-999.000,6.807,-999.000,
     &4.364,-999.000,-0.246,-999.000,-9.592,-999.000,
     &-15.165,-999.000,-19.786,-999.000,-21.733,-999.000,
     &-21.815,-20.673,-18.822,-17.083,-16.494
     &/
       data (fjrocvz(it,24,1, 4), it=1,59) /
     &-12.644,-12.632,-12.743,-13.510,-14.109,-13.459,
     &-13.532,-14.693,-15.632,-17.359,-17.201,-12.746,
     &-4.569,2.424,7.381,11.093,13.833,16.785,
     &20.200,23.411,25.188,25.797,25.224,23.651,
     &21.384,18.664,15.986,13.180,9.122,5.572,
     &3.096,2.210,2.044,1.598,1.033,-999.000,
     &1.857,-999.000,3.655,-999.000,5.647,-999.000,
     &4.929,-999.000,1.811,-999.000,-3.245,-999.000,
     &-8.678,-999.000,-12.769,-999.000,-14.676,-999.000,
     &-15.709,-15.971,-14.936,-13.466,-12.644
     &/
       data (fjrocvz(it,25,1, 4), it=1,59) /
     &-11.848,-11.414,-11.162,-11.384,-11.936,-12.667,
     &-14.149,-15.760,-17.572,-19.197,-18.126,-11.912,
     &-2.408,5.241,11.153,15.542,19.960,24.740,
     &28.876,31.474,31.299,29.544,27.229,24.268,
     &21.181,17.020,12.997,9.911,6.517,3.020,
     &0.937,0.301,0.204,-0.128,-0.388,-999.000,
     &-0.872,-999.000,0.023,-999.000,2.165,-999.000,
     &1.443,-999.000,-0.319,-999.000,-4.321,-999.000,
     &-8.446,-999.000,-11.634,-999.000,-13.354,-999.000,
     &-15.234,-16.041,-15.202,-13.172,-11.848
     &/
       data (fjrocvz(it, 1,2, 4), it=1,59) /
     &-16.176,-15.275,-13.852,-12.986,-13.884,-14.694,
     &-15.387,-16.355,-16.971,-16.410,-13.264,-6.703,
     &-7.932,-3.525,1.760,7.821,13.328,22.103,
     &29.060,31.826,31.390,29.849,27.238,22.957,
     &18.514,14.213,8.742,4.688,3.778,4.996,
     &5.567,5.769,6.059,7.472,9.843,-999.000,
     &13.180,-999.000,16.586,18.566,18.163,13.184,
     &6.453,-999.000,-1.006,-999.000,-7.729,-999.000,
     &-12.799,-999.000,-14.795,-999.000,-18.567,-999.000,
     &-19.660,-19.492,-19.090,-17.771,-16.176
     &/
       data (fjrocvz(it, 2,2, 4), it=1,59) /
     &-14.420,-13.699,-12.931,-12.592,-12.000,-11.204,
     &-11.435,-13.157,-14.374,-13.526,-8.076,-0.143,
     &5.615,8.586,12.782,15.172,18.974,24.195,
     &29.588,33.241,33.950,32.691,29.728,25.268,
     &20.713,17.117,13.195,9.358,7.018,6.893,
     &7.934,8.686,9.319,10.576,11.870,-999.000,
     &13.898,-999.000,17.954,19.790,20.732,19.481,
     &13.558,-999.000,-1.094,-999.000,-9.744,-999.000,
     &-10.917,-999.000,-15.377,-999.000,-19.332,-999.000,
     &-19.470,-19.363,-17.930,-15.614,-14.420
     &/
       data (fjrocvz(it, 3,2, 4), it=1,59) /
     &-14.605,-13.383,-12.734,-11.716,-10.807,-10.656,
     &-10.283,-10.448,-10.908,-10.310,-5.770,0.889,
     &6.886,10.185,12.497,15.488,19.110,22.307,
     &25.656,28.274,29.513,28.407,25.279,21.287,
     &17.576,14.830,12.288,9.894,7.820,7.363,
     &8.290,9.215,10.325,10.939,12.415,-999.000,
     &15.263,-999.000,17.923,22.418,25.397,22.252,
     &12.345,3.996,-1.679,-9.686,-13.510,-13.247,
     &-13.716,-999.000,-18.938,-999.000,-21.011,-999.000,
     &-21.207,-19.947,-18.289,-16.336,-14.605
     &/
       data (fjrocvz(it, 4,2, 4), it=1,59) /
     &-15.960,-14.800,-13.246,-11.760,-11.452,-11.244,
     &-10.466,-8.993,-8.054,-7.356,-4.289,1.308,
     &7.307,9.725,10.921,13.506,17.026,21.133,
     &24.550,26.492,26.050,23.792,20.723,17.203,
     &13.888,11.189,9.015,7.616,6.958,6.912,
     &7.655,8.986,9.999,11.695,14.650,-999.000,
     &16.476,-999.000,19.670,24.522,25.499,22.778,
     &16.772,9.223,-2.030,-13.646,-16.751,-999.000,
     &-16.097,-999.000,-20.114,-999.000,-22.883,-999.000,
     &-22.084,-19.941,-18.563,-16.941,-15.960
     &/
       data (fjrocvz(it, 5,2, 4), it=1,59) /
     &-16.750,-15.498,-13.285,-11.694,-11.622,-12.024,
     &-11.722,-10.368,-9.408,-8.524,-5.520,0.697,
     &6.695,9.507,12.081,14.110,17.573,22.313,
     &26.398,28.665,28.308,26.033,22.404,18.488,
     &14.836,12.275,10.488,9.530,8.570,8.071,
     &8.660,9.964,11.541,13.023,14.250,-999.000,
     &17.940,-999.000,21.833,25.061,27.243,28.212,
     &23.015,10.430,-1.831,-10.366,-15.809,-18.765,
     &-21.374,-24.684,-25.909,-25.626,-24.154,-999.000,
     &-22.436,-20.252,-18.620,-17.380,-16.750
     &/
       data (fjrocvz(it, 6,2, 4), it=1,59) /
     &-17.597,-16.724,-15.277,-13.929,-13.629,-13.086,
     &-11.679,-10.634,-9.632,-7.872,-5.753,-4.437,
     &-1.240,4.297,10.546,15.317,18.644,23.191,
     &27.261,29.506,30.221,28.802,26.098,22.439,
     &19.199,16.868,14.820,12.905,11.296,9.951,
     &9.513,10.984,12.789,13.350,14.798,-999.000,
     &18.869,-999.000,23.107,28.532,33.310,34.395,
     &29.725,18.146,3.797,-7.004,-14.353,-20.245,
     &-22.797,-999.000,-25.744,-999.000,-25.692,-999.000,
     &-23.918,-20.821,-19.222,-18.475,-17.597
     &/
       data (fjrocvz(it, 7,2, 4), it=1,59) /
     &-19.209,-18.669,-17.592,-16.606,-15.890,-13.846,
     &-11.883,-10.520,-8.612,-7.682,-5.609,-3.327,
     &1.719,4.285,8.010,11.215,14.696,19.903,
     &24.796,28.463,29.413,29.160,27.744,25.291,
     &22.701,20.085,17.702,15.665,13.338,11.023,
     &10.217,10.927,12.164,13.212,14.085,-999.000,
     &16.909,-999.000,21.400,28.163,34.148,38.217,
     &31.134,16.164,2.871,-5.912,-13.925,-21.913,
     &-26.428,-28.029,-28.135,-27.806,-26.913,-999.000,
     &-24.777,-22.765,-21.322,-19.790,-19.209
     &/
       data (fjrocvz(it, 8,2, 4), it=1,59) /
     &-19.008,-17.538,-16.577,-16.071,-15.282,-14.245,
     &-13.326,-12.367,-10.487,-12.592,-14.561,-16.423,
     &-11.442,-4.371,1.117,6.493,9.511,13.086,
     &18.869,23.318,25.457,25.812,25.216,23.930,
     &22.148,20.265,17.964,15.746,13.050,10.623,
     &9.548,9.831,11.069,12.061,12.796,-999.000,
     &15.590,-999.000,20.626,25.034,30.528,31.117,
     &23.424,13.324,3.736,-5.770,-12.506,-19.488,
     &-23.713,-24.669,-23.074,-999.000,-23.572,-999.000,
     &-23.034,-22.904,-21.989,-20.372,-19.008
     &/
       data (fjrocvz(it, 9,2, 4), it=1,59) /
     &-16.274,-14.157,-12.286,-11.094,-11.005,-11.553,
     &-13.009,-14.296,-15.022,-17.011,-19.792,-22.025,
     &-17.139,-7.405,0.466,5.582,7.513,11.015,
     &15.938,19.536,21.176,21.815,21.565,20.139,
     &17.854,15.701,13.827,11.868,9.673,8.442,
     &8.387,9.010,10.223,10.937,12.000,-999.000,
     &13.899,-999.000,19.425,24.674,29.128,31.957,
     &30.908,21.140,10.144,0.227,-6.232,-999.000,
     &-11.548,-999.000,-15.153,-999.000,-18.152,-999.000,
     &-18.905,-19.137,-19.494,-18.457,-16.274
     &/
       data (fjrocvz(it,10,2, 4), it=1,59) /
     &-16.287,-14.127,-11.864,-10.627,-9.855,-11.051,
     &-14.270,-16.945,-18.311,-19.144,-19.917,-19.922,
     &-16.075,-5.483,2.457,7.560,9.731,13.723,
     &18.812,22.013,22.411,21.909,20.572,18.881,
     &16.672,14.054,11.746,9.770,8.438,7.963,
     &8.066,8.443,8.526,8.774,10.212,-999.000,
     &13.095,-999.000,20.182,26.231,32.895,36.630,
     &34.906,24.146,13.446,-999.000,3.393,-999.000,
     &-4.911,-999.000,-10.870,-999.000,-12.745,-999.000,
     &-14.481,-15.702,-16.951,-16.968,-16.287
     &/
       data (fjrocvz(it,11,2, 4), it=1,59) /
     &-18.533,-17.949,-17.646,-17.555,-17.325,-18.054,
     &-18.949,-20.558,-22.358,-24.164,-23.505,-19.703,
     &-14.467,-4.703,2.815,8.812,13.096,16.284,
     &20.582,24.163,25.453,25.228,24.114,22.381,
     &20.500,18.037,15.030,12.860,11.165,10.241,
     &9.605,8.857,8.355,8.491,10.409,-999.000,
     &14.802,-999.000,21.131,27.293,35.089,35.365,
     &30.401,21.290,12.140,-999.000,0.088,-999.000,
     &-8.612,-999.000,-13.424,-999.000,-14.447,-999.000,
     &-16.911,-16.929,-18.012,-19.158,-18.533
     &/
       data (fjrocvz(it,12,2, 4), it=1,59) /
     &-20.640,-21.513,-21.478,-21.399,-21.697,-21.449,
     &-21.335,-23.077,-26.032,-27.659,-28.127,-24.607,
     &-20.225,-10.493,-0.682,5.302,10.240,14.994,
     &21.512,26.294,28.071,27.830,26.682,25.386,
     &23.716,20.918,17.684,15.069,13.474,12.699,
     &11.743,10.472,9.528,9.561,11.782,-999.000,
     &15.916,-999.000,22.045,27.932,35.247,38.774,
     &32.687,17.628,6.598,-0.671,-6.774,-999.000,
     &-10.447,-999.000,-13.456,-999.000,-17.187,-999.000,
     &-19.396,-19.515,-19.918,-20.428,-20.640
     &/
       data (fjrocvz(it,13,2, 4), it=1,59) /
     &-20.581,-20.175,-19.990,-20.032,-19.763,-20.498,
     &-22.067,-23.419,-25.024,-25.230,-25.099,-23.219,
     &-19.909,-11.222,-2.305,4.492,9.396,14.767,
     &23.780,29.342,30.047,29.488,28.394,27.202,
     &25.293,22.100,18.939,16.731,15.796,15.457,
     &14.680,13.338,11.848,11.655,13.458,-999.000,
     &16.656,-999.000,24.914,30.381,38.048,42.198,
     &33.700,18.104,3.202,-4.325,-7.387,-9.180,
     &-10.035,-999.000,-13.482,-999.000,-16.554,-999.000,
     &-18.937,-21.062,-21.080,-20.562,-20.581
     &/
       data (fjrocvz(it,14,2, 4), it=1,59) /
     &-19.264,-18.865,-18.548,-17.986,-17.987,-19.569,
     &-22.297,-25.854,-29.054,-28.420,-25.192,-20.827,
     &-16.461,-13.100,-6.129,1.735,7.613,13.687,
     &22.812,28.250,29.650,29.284,28.043,26.536,
     &24.183,21.599,19.185,17.598,16.987,16.573,
     &15.501,14.157,12.682,12.572,14.623,-999.000,
     &18.843,-999.000,26.857,33.846,38.275,38.158,
     &30.321,16.738,6.960,-999.000,-4.957,-999.000,
     &-12.748,-999.000,-16.234,-999.000,-18.983,-999.000,
     &-21.781,-22.597,-21.707,-20.164,-19.264
     &/
       data (fjrocvz(it,15,2, 4), it=1,59) /
     &-17.530,-16.947,-16.304,-15.802,-15.730,-17.100,
     &-20.881,-26.843,-30.827,-31.914,-28.354,-20.917,
     &-15.805,-11.305,-6.108,1.238,7.390,13.089,
     &20.480,25.503,27.267,26.643,24.749,22.094,
     &19.352,17.185,15.022,13.520,12.714,11.940,
     &11.061,10.074,9.581,10.361,13.114,-999.000,
     &16.690,-999.000,23.291,28.946,32.605,34.500,
     &27.983,14.453,4.780,-999.000,-7.865,-999.000,
     &-14.805,-999.000,-19.564,-999.000,-22.932,-999.000,
     &-24.820,-25.094,-22.728,-19.247,-17.530
     &/
       data (fjrocvz(it,16,2, 4), it=1,59) /
     &-17.641,-16.262,-16.346,-16.441,-16.926,-18.656,
     &-22.182,-26.675,-29.893,-32.728,-31.612,-24.005,
     &-14.494,-9.949,-4.492,1.739,7.281,12.600,
     &19.861,24.826,26.478,26.137,24.579,21.172,
     &16.986,13.220,10.472,8.771,8.027,7.644,
     &6.845,5.756,5.154,5.839,8.074,-999.000,
     &11.631,-999.000,17.849,22.744,26.757,28.353,
     &21.453,7.486,1.364,-999.000,-10.453,-999.000,
     &-17.819,-999.000,-22.668,-999.000,-25.565,-999.000,
     &-26.858,-26.375,-23.872,-20.382,-17.641
     &/
       data (fjrocvz(it,17,2, 4), it=1,59) /
     &-20.491,-19.498,-18.656,-18.061,-18.470,-19.666,
     &-22.509,-26.800,-30.020,-31.665,-29.957,-21.855,
     &-13.030,-7.883,-1.659,3.505,8.081,13.681,
     &21.411,26.616,28.849,29.122,27.800,24.750,
     &20.030,15.394,11.429,8.561,7.461,6.763,
     &5.647,4.153,2.538,2.101,4.191,-999.000,
     &8.742,-999.000,15.905,19.409,22.815,22.216,
     &16.580,5.397,-3.889,-12.133,-18.563,-22.255,
     &-24.007,-999.000,-24.817,-999.000,-26.663,-999.000,
     &-27.102,-25.868,-24.230,-22.216,-20.491
     &/
       data (fjrocvz(it,18,2, 4), it=1,59) /
     &-21.390,-19.689,-18.330,-17.836,-17.419,-17.827,
     &-21.206,-25.563,-27.889,-29.499,-26.998,-19.741,
     &-10.881,-5.761,-1.165,4.063,9.039,16.106,
     &23.193,28.798,32.356,32.743,30.363,26.269,
     &22.143,18.143,13.809,9.958,7.880,6.362,
     &5.021,3.365,1.978,1.858,4.048,-999.000,
     &7.792,10.836,15.306,18.909,20.570,17.562,
     &10.050,2.989,-6.030,-999.000,-19.034,-999.000,
     &-25.800,-999.000,-28.319,-999.000,-28.436,-999.000,
     &-27.110,-26.459,-24.769,-22.922,-21.390
     &/
       data (fjrocvz(it,19,2, 4), it=1,59) /
     &-19.855,-17.929,-17.603,-16.936,-16.162,-16.407,
     &-18.826,-21.244,-22.962,-24.560,-21.874,-14.938,
     &-6.843,-3.515,1.006,6.349,11.123,18.276,
     &26.229,32.419,35.855,35.991,33.257,29.108,
     &24.556,19.977,15.554,11.622,9.891,8.947,
     &7.396,5.511,4.079,3.602,5.421,-999.000,
     &9.788,-999.000,16.728,19.313,19.622,14.433,
     &9.451,1.744,-7.346,-16.696,-23.113,-26.697,
     &-27.679,-999.000,-28.708,-999.000,-27.644,-999.000,
     &-26.163,-25.606,-24.172,-22.099,-19.855
     &/
       data (fjrocvz(it,20,2, 4), it=1,59) /
     &-22.012,-20.174,-18.326,-16.441,-14.958,-14.664,
     &-16.503,-19.300,-20.402,-21.036,-18.468,-9.313,
     &-1.226,-0.064,3.995,8.024,12.007,19.344,
     &27.006,34.008,37.689,38.164,36.069,32.443,
     &28.312,24.057,19.433,15.503,13.314,11.722,
     &9.869,7.846,6.073,5.636,7.899,-999.000,
     &11.759,-999.000,17.067,19.310,22.310,21.196,
     &14.773,2.341,-6.649,-13.403,-19.177,-23.981,
     &-26.602,-999.000,-27.169,-999.000,-27.662,-999.000,
     &-26.883,-26.415,-25.467,-23.761,-22.012
     &/
       data (fjrocvz(it,21,2, 4), it=1,59) /
     &-22.392,-19.905,-17.970,-16.725,-15.936,-15.207,
     &-16.084,-18.516,-20.732,-21.267,-18.391,-10.999,
     &-5.412,-1.229,3.377,8.284,13.275,19.100,
     &24.809,31.314,34.647,35.688,33.736,29.977,
     &26.113,22.507,18.529,14.134,11.221,9.241,
     &7.418,6.115,5.302,5.239,6.956,-999.000,
     &10.639,-999.000,16.585,20.284,23.991,22.860,
     &13.996,-999.000,-2.093,-999.000,-15.360,-999.000,
     &-22.591,-999.000,-25.557,-999.000,-26.315,-999.000,
     &-26.074,-25.685,-25.043,-23.984,-22.392
     &/
       data (fjrocvz(it,22,2, 4), it=1,59) /
     &-21.309,-20.308,-19.201,-18.425,-18.497,-18.114,
     &-18.262,-20.382,-23.964,-24.810,-21.785,-14.880,
     &-6.734,-3.865,1.502,6.258,10.895,16.549,
     &23.010,28.827,31.352,31.300,29.040,24.876,
     &19.903,15.450,11.672,8.351,5.976,4.432,
     &3.506,3.750,4.312,4.621,6.378,-999.000,
     &11.328,-999.000,16.998,18.120,20.381,18.169,
     &11.563,-999.000,-4.712,-999.000,-16.319,-999.000,
     &-23.562,-999.000,-24.600,-999.000,-24.573,-999.000,
     &-24.778,-24.263,-23.353,-22.337,-21.309
     &/
       data (fjrocvz(it,23,2, 4), it=1,59) /
     &-19.136,-18.830,-17.423,-15.581,-15.277,-16.266,
     &-18.111,-20.419,-22.624,-23.511,-21.030,-14.933,
     &-9.861,-6.941,-1.523,3.707,8.885,14.386,
     &22.500,28.175,29.913,28.672,25.892,21.619,
     &16.686,11.672,7.803,4.958,3.246,2.384,
     &1.887,1.738,2.890,4.793,7.594,-999.000,
     &10.835,-999.000,15.437,17.895,18.735,17.906,
     &13.017,3.852,-3.769,-999.000,-14.701,-999.000,
     &-21.023,-999.000,-23.213,-999.000,-23.410,-999.000,
     &-23.463,-22.705,-21.051,-19.477,-19.136
     &/
       data (fjrocvz(it,24,2, 4), it=1,59) /
     &-16.955,-15.643,-14.018,-12.166,-12.403,-14.345,
     &-16.479,-17.814,-19.056,-19.890,-18.036,-13.251,
     &-7.060,-2.025,1.394,4.648,9.217,17.334,
     &25.476,29.054,29.641,28.258,25.049,20.484,
     &15.825,10.923,6.523,3.445,2.588,3.402,
     &3.626,3.221,4.208,5.919,8.170,-999.000,
     &11.218,-999.000,15.293,17.512,19.372,16.870,
     &10.254,-999.000,-0.113,-999.000,-8.309,-999.000,
     &-14.215,-999.000,-18.408,-999.000,-19.497,-999.000,
     &-20.724,-20.465,-19.172,-18.102,-16.955
     &/
       data (fjrocvz(it,25,2, 4), it=1,59) /
     &-16.176,-15.275,-13.852,-12.986,-13.884,-14.694,
     &-15.387,-16.355,-16.971,-16.410,-13.264,-6.703,
     &-7.932,-3.525,1.760,7.821,13.328,22.103,
     &29.060,31.826,31.390,29.849,27.238,22.957,
     &18.514,14.213,8.742,4.688,3.778,4.996,
     &5.567,5.769,6.059,7.472,9.843,-999.000,
     &13.180,-999.000,16.586,18.566,18.163,13.184,
     &6.453,-999.000,-1.006,-999.000,-7.729,-999.000,
     &-12.799,-999.000,-14.795,-999.000,-18.567,-999.000,
     &-19.660,-19.492,-19.090,-17.771,-16.176
     &/
       data (fjrocvz(it, 1,3, 4), it=1,59) /
     &-22.526,-23.823,-23.641,-22.732,-21.393,-20.002,
     &-19.848,-22.849,-29.192,-31.069,-24.247,-10.706,
     &0.577,4.867,7.666,9.830,9.962,8.603,
     &8.750,11.166,14.117,15.081,14.500,12.040,
     &8.712,7.309,7.962,9.124,9.266,8.366,
     &7.701,7.631,7.465,6.639,6.778,-999.000,
     &7.633,-999.000,9.450,-999.000,14.822,17.642,
     &17.851,14.779,6.196,-999.000,-1.635,-999.000,
     &-8.496,-999.000,-11.697,-999.000,-13.944,-999.000,
     &-16.776,-19.147,-20.603,-21.337,-22.526
     &/
       data (fjrocvz(it, 2,3, 4), it=1,59) /
     &-20.258,-20.931,-20.582,-19.400,-18.410,-18.168,
     &-19.491,-22.685,-27.072,-28.288,-20.260,-4.784,
     &8.289,11.605,12.836,13.810,13.887,13.264,
     &13.891,15.253,16.264,16.677,15.661,13.578,
     &11.690,10.936,11.478,12.301,12.910,12.608,
     &12.000,11.720,11.561,11.416,11.132,-999.000,
     &10.443,-999.000,12.862,-999.000,16.856,19.692,
     &18.720,14.433,5.364,-999.000,-1.949,-999.000,
     &-8.774,-999.000,-12.997,-999.000,-15.520,-999.000,
     &-17.952,-19.970,-20.434,-19.956,-20.258
     &/
       data (fjrocvz(it, 3,3, 4), it=1,59) /
     &-19.106,-18.172,-17.683,-16.923,-16.588,-16.480,
     &-17.662,-19.493,-22.931,-26.718,-19.678,-0.683,
     &16.529,18.319,16.840,15.549,14.317,13.839,
     &14.546,15.031,14.122,12.742,11.140,10.206,
     &9.705,9.425,10.254,11.380,12.595,13.799,
     &13.773,13.362,13.206,13.201,13.114,-999.000,
     &13.433,-999.000,14.023,-999.000,15.934,16.420,
     &14.481,12.790,5.416,-999.000,-3.602,-999.000,
     &-10.562,-999.000,-15.537,-999.000,-18.499,-999.000,
     &-21.218,-23.028,-22.958,-20.995,-19.106
     &/
       data (fjrocvz(it, 4,3, 4), it=1,59) /
     &-21.135,-19.234,-18.691,-18.753,-18.274,-17.245,
     &-16.626,-17.270,-20.434,-23.071,-15.694,4.297,
     &19.126,21.038,19.655,18.692,17.127,14.893,
     &13.163,11.907,10.771,9.762,9.394,9.614,
     &9.371,9.540,10.184,11.443,12.916,13.778,
     &14.216,14.337,13.921,13.676,14.919,-999.000,
     &14.804,-999.000,13.983,-999.000,14.757,16.392,
     &15.170,8.647,3.060,-999.000,-5.690,-999.000,
     &-12.962,-999.000,-18.054,-999.000,-21.606,-999.000,
     &-23.750,-25.047,-25.008,-23.254,-21.135
     &/
       data (fjrocvz(it, 5,3, 4), it=1,59) /
     &-23.290,-22.244,-21.595,-19.936,-18.008,-16.514,
     &-15.849,-16.544,-17.851,-18.115,-12.003,2.466,
     &17.348,21.837,21.548,20.746,19.052,16.956,
     &15.155,13.951,13.727,13.575,13.311,12.345,
     &10.956,9.874,9.427,9.957,10.641,11.426,
     &12.408,13.188,13.304,13.531,13.927,-999.000,
     &13.903,-999.000,14.426,15.513,16.001,13.181,
     &10.602,6.075,0.416,-999.000,-7.959,-999.000,
     &-14.933,-999.000,-19.886,-999.000,-23.209,-999.000,
     &-25.768,-27.244,-26.656,-24.749,-23.290
     &/
       data (fjrocvz(it, 6,3, 4), it=1,59) /
     &-25.620,-25.786,-25.694,-23.414,-19.677,-16.474,
     &-15.785,-16.523,-16.155,-14.500,-10.038,0.375,
     &14.433,18.925,18.664,17.980,17.491,16.864,
     &16.353,17.291,18.383,18.424,17.399,15.938,
     &14.125,12.192,10.859,9.904,9.993,10.437,
     &11.079,11.593,12.276,12.546,12.761,-999.000,
     &14.147,-999.000,14.233,15.172,12.963,8.959,
     &3.882,-999.000,-3.533,-999.000,-11.118,-999.000,
     &-18.312,-999.000,-23.339,-999.000,-26.661,-999.000,
     &-29.340,-30.580,-29.125,-26.992,-25.620
     &/
       data (fjrocvz(it, 7,3, 4), it=1,59) /
     &-28.619,-27.201,-26.860,-26.022,-23.387,-20.841,
     &-19.714,-18.730,-17.364,-14.022,-9.655,-2.318,
     &5.557,9.984,11.351,12.318,13.026,13.523,
     &14.659,16.563,18.619,19.737,19.911,19.544,
     &18.747,17.854,16.198,14.234,13.081,12.354,
     &12.005,11.513,11.280,11.020,11.074,-999.000,
     &10.994,-999.000,10.107,-999.000,4.970,-999.000,
     &-0.427,-999.000,-5.857,-999.000,-12.527,-999.000,
     &-19.744,-999.000,-25.101,-999.000,-28.774,-999.000,
     &-31.376,-32.735,-32.161,-30.647,-28.619
     &/
       data (fjrocvz(it, 8,3, 4), it=1,59) /
     &-25.198,-24.223,-25.251,-25.842,-24.773,-21.004,
     &-16.325,-13.337,-11.818,-10.238,-7.180,-3.262,
     &-0.516,1.581,2.967,4.644,6.369,7.660,
     &9.124,10.934,13.892,17.163,18.964,19.403,
     &19.318,18.354,16.636,14.327,12.372,11.721,
     &11.163,10.333,9.226,7.867,6.713,-999.000,
     &5.929,-999.000,4.580,-999.000,1.469,-999.000,
     &-1.403,-999.000,-4.760,-999.000,-9.720,-999.000,
     &-15.636,-999.000,-20.483,-999.000,-25.182,-999.000,
     &-28.556,-29.867,-28.616,-26.746,-25.198
     &/
       data (fjrocvz(it, 9,3, 4), it=1,59) /
     &-18.455,-18.707,-19.011,-19.448,-19.226,-16.213,
     &-12.167,-9.939,-9.389,-9.940,-9.044,-6.595,
     &-6.267,-2.521,1.990,6.221,10.200,10.944,
     &11.774,12.568,13.336,12.840,14.058,15.083,
     &15.841,15.576,13.708,11.028,9.186,7.982,
     &6.709,5.357,4.251,3.749,3.435,-999.000,
     &2.333,-999.000,1.203,-999.000,1.105,-999.000,
     &1.260,-999.000,0.961,-999.000,-2.679,-999.000,
     &-6.928,-999.000,-10.836,-999.000,-13.453,-999.000,
     &-15.797,-17.482,-17.510,-17.791,-18.455
     &/
       data (fjrocvz(it,10,3, 4), it=1,59) /
     &-15.421,-15.831,-14.984,-13.911,-14.279,-14.986,
     &-14.435,-13.129,-12.368,-12.502,-12.371,-13.532,
     &-15.488,-7.664,0.843,6.570,12.284,13.217,
     &14.162,14.817,15.486,15.584,15.403,15.253,
     &15.128,15.068,13.438,11.201,8.997,6.752,
     &4.485,1.918,0.083,-0.754,-1.779,-999.000,
     &-3.123,-999.000,-2.698,-999.000,-1.604,-999.000,
     &1.584,-999.000,3.108,-999.000,2.603,-999.000,
     &-0.134,-999.000,-2.900,-999.000,-5.016,-999.000,
     &-7.723,-10.666,-12.655,-14.074,-15.421
     &/
       data (fjrocvz(it,11,3, 4), it=1,59) /
     &-19.093,-20.443,-20.604,-19.800,-19.308,-18.478,
     &-16.859,-16.455,-19.307,-18.108,-17.744,-11.782,
     &-15.155,-9.112,-2.944,3.500,9.956,12.070,
     &14.143,15.798,18.721,20.660,19.998,19.144,
     &18.455,17.593,16.430,14.943,12.553,9.657,
     &6.540,3.782,1.957,-0.101,-2.070,-999.000,
     &-3.085,-999.000,-2.777,-999.000,-1.018,-999.000,
     &5.331,8.198,8.207,5.467,3.230,-999.000,
     &0.453,-999.000,-1.388,-999.000,-2.997,-999.000,
     &-6.152,-11.473,-15.069,-17.174,-19.093
     &/
       data (fjrocvz(it,12,3, 4), it=1,59) /
     &-20.386,-22.279,-24.549,-26.191,-25.761,-23.626,
     &-23.343,-27.431,-32.308,-30.097,-24.063,-14.445,
     &-13.518,-10.299,-6.888,-0.842,5.233,8.945,
     &12.198,16.407,20.732,22.548,22.553,21.937,
     &21.026,19.683,18.210,17.183,15.144,12.008,
     &9.089,7.411,5.814,4.153,2.144,-999.000,
     &0.277,-999.000,1.537,-999.000,7.639,12.377,
     &16.007,16.994,14.441,-999.000,8.423,-999.000,
     &3.207,-999.000,0.462,-999.000,-1.928,-999.000,
     &-5.335,-10.867,-15.617,-18.290,-20.386
     &/
       data (fjrocvz(it,13,3, 4), it=1,59) /
     &-19.067,-20.842,-21.612,-22.675,-24.778,-26.545,
     &-28.978,-31.790,-34.217,-32.360,-25.240,-17.428,
     &-13.458,-12.113,-8.917,-5.342,-1.873,4.834,
     &12.983,18.889,22.850,24.864,25.006,23.717,
     &22.308,21.133,19.673,18.195,16.267,13.807,
     &11.719,9.429,7.614,6.570,5.153,-999.000,
     &4.902,-999.000,7.371,-999.000,11.954,16.822,
     &22.695,24.653,19.097,-999.000,11.956,-999.000,
     &5.826,-999.000,1.749,-999.000,-1.351,-999.000,
     &-4.562,-8.343,-12.337,-16.071,-19.067
     &/
       data (fjrocvz(it,14,3, 4), it=1,59) /
     &-19.107,-21.380,-21.531,-22.569,-25.607,-29.423,
     &-31.795,-34.043,-37.323,-36.060,-27.389,-16.495,
     &-11.494,-10.598,-8.886,-6.073,-3.123,3.077,
     &12.278,18.090,22.664,24.942,25.092,23.344,
     &21.925,21.208,20.378,19.236,18.054,16.192,
     &14.027,11.601,9.913,8.574,8.001,-999.000,
     &8.258,-999.000,10.570,-999.000,15.399,19.335,
     &25.464,27.315,24.284,17.499,12.738,-999.000,
     &5.705,-999.000,1.794,-999.000,-0.944,-999.000,
     &-4.015,-7.463,-11.022,-15.459,-19.107
     &/
       data (fjrocvz(it,15,3, 4), it=1,59) /
     &-18.663,-20.628,-21.214,-21.944,-24.420,-28.055,
     &-30.995,-34.323,-38.274,-39.097,-30.939,-19.300,
     &-12.488,-11.397,-10.520,-8.052,-4.296,1.698,
     &9.550,15.284,19.516,21.690,21.093,19.004,
     &17.203,16.028,15.292,14.156,12.678,11.407,
     &10.208,8.931,7.198,6.265,6.047,-999.000,
     &6.033,-999.000,8.460,-999.000,12.826,16.869,
     &20.396,20.177,16.514,11.904,8.216,-999.000,
     &1.900,-999.000,-1.471,-999.000,-3.390,-999.000,
     &-6.146,-9.214,-12.248,-15.477,-18.663
     &/
       data (fjrocvz(it,16,3, 4), it=1,59) /
     &-16.585,-18.107,-18.320,-18.915,-21.799,-26.180,
     &-30.637,-34.266,-39.474,-41.992,-37.539,-24.804,
     &-14.544,-11.311,-10.338,-7.950,-3.899,1.617,
     &8.518,13.815,17.349,18.467,17.100,15.322,
     &13.557,11.805,9.898,7.348,4.934,3.500,
     &2.868,1.924,0.619,-0.614,-0.915,-999.000,
     &0.082,-999.000,2.353,-999.000,5.562,9.143,
     &10.543,7.996,5.426,-999.000,-1.081,-999.000,
     &-5.986,-999.000,-8.492,-999.000,-9.268,-999.000,
     &-9.967,-11.585,-12.917,-14.590,-16.585
     &/
       data (fjrocvz(it,17,3, 4), it=1,59) /
     &-17.689,-18.705,-18.306,-18.320,-20.529,-23.927,
     &-28.377,-34.843,-42.223,-44.736,-39.572,-24.253,
     &-12.279,-9.468,-7.498,-4.241,-0.101,4.620,
     &10.759,16.111,19.345,20.275,19.225,16.841,
     &14.305,12.096,9.795,6.728,3.727,1.479,
     &-0.581,-1.797,-2.831,-4.094,-4.846,-999.000,
     &-5.328,-999.000,-3.770,-999.000,-0.367,4.690,
     &6.471,3.597,-0.475,-999.000,-5.609,-999.000,
     &-11.008,-999.000,-14.089,-999.000,-14.666,-999.000,
     &-14.859,-15.817,-16.363,-16.752,-17.689
     &/
       data (fjrocvz(it,18,3, 4), it=1,59) /
     &-25.302,-24.859,-23.420,-22.002,-22.205,-23.878,
     &-26.738,-33.447,-42.854,-44.973,-38.111,-16.373,
     &-7.107,-5.769,-3.762,-0.281,4.050,9.617,
     &15.928,20.883,23.279,24.113,23.495,21.542,
     &18.547,15.342,12.696,9.525,6.551,3.594,
     &0.951,-0.880,-2.505,-4.054,-5.657,-999.000,
     &-6.705,-999.000,-5.018,-999.000,-1.603,1.363,
     &0.375,-2.842,-5.897,-999.000,-11.328,-999.000,
     &-16.357,-999.000,-18.957,-999.000,-19.631,-999.000,
     &-20.119,-21.670,-23.050,-24.140,-25.302
     &/
       data (fjrocvz(it,19,3, 4), it=1,59) /
     &-27.225,-27.184,-25.633,-23.968,-23.866,-25.620,
     &-26.720,-29.317,-35.607,-36.401,-30.533,-13.763,
     &-4.943,-2.905,-1.113,1.921,6.681,12.542,
     &18.889,23.631,26.549,27.036,26.197,24.547,
     &22.094,19.248,16.108,13.007,10.040,6.984,
     &4.585,2.978,1.486,-0.896,-2.742,-999.000,
     &-2.921,-999.000,-2.113,-999.000,1.034,2.299,
     &0.279,-2.870,-7.442,-999.000,-12.023,-999.000,
     &-16.268,-999.000,-19.408,-999.000,-20.584,-999.000,
     &-21.103,-22.481,-23.997,-25.829,-27.225
     &/
       data (fjrocvz(it,20,3, 4), it=1,59) /
     &-24.643,-25.215,-24.165,-22.714,-23.290,-25.942,
     &-28.538,-30.972,-32.521,-29.872,-23.335,-13.470,
     &-5.756,-2.056,-0.239,1.963,6.019,12.255,
     &19.003,25.166,28.554,28.849,27.417,25.135,
     &22.819,20.746,19.200,17.754,15.355,12.373,
     &9.514,7.437,5.060,2.920,1.723,-999.000,
     &1.088,-999.000,2.157,-999.000,5.049,6.940,
     &5.738,2.642,-2.297,-999.000,-7.248,-999.000,
     &-11.087,-999.000,-14.237,-999.000,-15.914,-999.000,
     &-17.787,-19.590,-20.996,-22.687,-24.643
     &/
       data (fjrocvz(it,21,3, 4), it=1,59) /
     &-21.208,-22.302,-21.567,-20.539,-21.562,-25.144,
     &-28.062,-29.945,-33.047,-34.905,-31.619,-21.188,
     &-9.655,-4.484,-2.214,-0.229,2.879,7.745,
     &14.013,19.593,22.953,23.571,22.804,21.798,
     &20.406,19.027,18.194,16.801,14.836,12.188,
     &9.931,7.824,6.036,4.956,4.146,-999.000,
     &3.942,-999.000,4.005,-999.000,4.487,5.735,
     &7.435,6.164,2.303,-999.000,-2.251,-999.000,
     &-5.680,-999.000,-9.264,-999.000,-11.669,-999.000,
     &-13.970,-15.933,-17.313,-19.066,-21.208
     &/
       data (fjrocvz(it,22,3, 4), it=1,59) /
     &-18.872,-18.748,-17.616,-17.688,-19.980,-23.228,
     &-25.769,-29.359,-35.403,-38.964,-36.328,-25.720,
     &-13.756,-6.137,-4.041,-2.255,0.210,3.482,
     &7.986,12.175,15.386,15.900,15.265,15.024,
     &14.659,13.792,12.820,11.238,9.150,7.221,
     &5.596,4.598,3.895,3.877,4.825,-999.000,
     &5.186,-999.000,4.883,-999.000,6.097,6.930,
     &10.619,10.292,4.281,-999.000,-0.869,-999.000,
     &-4.814,-999.000,-7.949,-999.000,-10.317,-999.000,
     &-12.521,-14.724,-16.527,-18.129,-18.872
     &/
       data (fjrocvz(it,23,3, 4), it=1,59) /
     &-20.344,-19.744,-18.228,-17.575,-19.820,-23.302,
     &-26.689,-29.893,-33.416,-34.644,-30.583,-20.491,
     &-9.016,-3.726,-3.072,-1.987,-0.136,1.837,
     &4.833,9.089,13.006,14.484,13.460,11.630,
     &9.890,8.227,7.196,6.132,4.507,2.963,
     &2.538,2.298,1.749,1.886,3.180,-999.000,
     &3.552,-999.000,5.126,-999.000,8.614,12.287,
     &15.331,14.833,8.607,-999.000,0.144,-999.000,
     &-4.895,-999.000,-7.813,-999.000,-10.503,-999.000,
     &-13.427,-16.013,-17.920,-19.490,-20.344
     &/
       data (fjrocvz(it,24,3, 4), it=1,59) /
     &-22.857,-23.500,-22.207,-20.375,-20.215,-21.717,
     &-23.760,-26.347,-30.743,-31.574,-26.986,-15.984,
     &-5.186,-0.658,1.052,2.556,3.738,4.288,
     &6.056,9.125,12.384,13.303,12.593,11.023,
     &8.146,5.993,4.993,5.512,5.299,4.722,
     &4.648,4.387,4.037,3.554,3.811,-999.000,
     &5.470,-999.000,8.087,-999.000,12.514,16.428,
     &16.491,14.408,10.933,-999.000,-0.387,-999.000,
     &-6.925,-999.000,-9.458,-999.000,-12.398,-999.000,
     &-16.091,-19.062,-20.663,-21.745,-22.857
     &/
       data (fjrocvz(it,25,3, 4), it=1,59) /
     &-22.526,-23.823,-23.641,-22.732,-21.393,-20.002,
     &-19.848,-22.849,-29.192,-31.069,-24.247,-10.706,
     &0.577,4.867,7.666,9.830,9.962,8.603,
     &8.750,11.166,14.117,15.081,14.500,12.040,
     &8.712,7.309,7.962,9.124,9.266,8.366,
     &7.701,7.631,7.465,6.639,6.778,-999.000,
     &7.633,-999.000,9.450,-999.000,14.822,17.642,
     &17.851,14.779,6.196,-999.000,-1.635,-999.000,
     &-8.496,-999.000,-11.697,-999.000,-13.944,-999.000,
     &-16.776,-19.147,-20.603,-21.337,-22.526
     &/
       data (fjrocvz(it, 1,4, 4), it=1,59) /
     &-16.176,-15.275,-13.852,-12.986,-13.884,-14.694,
     &-15.387,-16.355,-16.971,-16.410,-13.264,-6.703,
     &-7.932,-3.525,1.760,7.821,13.328,22.103,
     &29.060,31.826,31.390,29.849,27.238,22.957,
     &18.514,14.213,8.742,4.688,3.778,4.996,
     &5.567,5.769,6.059,7.472,9.843,-999.000,
     &13.180,-999.000,16.586,18.566,18.163,13.184,
     &6.453,-999.000,-1.006,-999.000,-7.729,-999.000,
     &-12.799,-999.000,-14.795,-999.000,-18.567,-999.000,
     &-19.660,-19.492,-19.090,-17.771,-16.176
     &/
       data (fjrocvz(it, 2,4, 4), it=1,59) /
     &-14.420,-13.699,-12.931,-12.592,-12.000,-11.204,
     &-11.435,-13.157,-14.374,-13.526,-8.076,-0.143,
     &5.615,8.586,12.782,15.172,18.974,24.195,
     &29.588,33.241,33.950,32.691,29.728,25.268,
     &20.713,17.117,13.195,9.358,7.018,6.893,
     &7.934,8.686,9.319,10.576,11.870,-999.000,
     &13.898,-999.000,17.954,19.790,20.732,19.481,
     &13.558,-999.000,-1.094,-999.000,-9.744,-999.000,
     &-10.917,-999.000,-15.377,-999.000,-19.332,-999.000,
     &-19.470,-19.363,-17.930,-15.614,-14.420
     &/
       data (fjrocvz(it, 3,4, 4), it=1,59) /
     &-14.605,-13.383,-12.734,-11.716,-10.807,-10.656,
     &-10.283,-10.448,-10.908,-10.310,-5.770,0.889,
     &6.886,10.185,12.497,15.488,19.110,22.307,
     &25.656,28.274,29.513,28.407,25.279,21.287,
     &17.576,14.830,12.288,9.894,7.820,7.363,
     &8.290,9.215,10.325,10.939,12.415,-999.000,
     &15.263,-999.000,17.923,22.418,25.397,22.252,
     &12.345,3.996,-1.679,-9.686,-13.510,-13.247,
     &-13.716,-999.000,-18.938,-999.000,-21.011,-999.000,
     &-21.207,-19.947,-18.289,-16.336,-14.605
     &/
       data (fjrocvz(it, 4,4, 4), it=1,59) /
     &-15.960,-14.800,-13.246,-11.760,-11.452,-11.244,
     &-10.466,-8.993,-8.054,-7.356,-4.289,1.308,
     &7.307,9.725,10.921,13.506,17.026,21.133,
     &24.550,26.492,26.050,23.792,20.723,17.203,
     &13.888,11.189,9.015,7.616,6.958,6.912,
     &7.655,8.986,9.999,11.695,14.650,-999.000,
     &16.476,-999.000,19.670,24.522,25.499,22.778,
     &16.772,9.223,-2.030,-13.646,-16.751,-999.000,
     &-16.097,-999.000,-20.114,-999.000,-22.883,-999.000,
     &-22.084,-19.941,-18.563,-16.941,-15.960
     &/
       data (fjrocvz(it, 5,4, 4), it=1,59) /
     &-16.750,-15.498,-13.285,-11.694,-11.622,-12.024,
     &-11.722,-10.368,-9.408,-8.524,-5.520,0.697,
     &6.695,9.507,12.081,14.110,17.573,22.313,
     &26.398,28.665,28.308,26.033,22.404,18.488,
     &14.836,12.275,10.488,9.530,8.570,8.071,
     &8.660,9.964,11.541,13.023,14.250,-999.000,
     &17.940,-999.000,21.833,25.061,27.243,28.212,
     &23.015,10.430,-1.831,-10.366,-15.809,-18.765,
     &-21.374,-24.684,-25.909,-25.626,-24.154,-999.000,
     &-22.436,-20.252,-18.620,-17.380,-16.750
     &/
       data (fjrocvz(it, 6,4, 4), it=1,59) /
     &-17.597,-16.724,-15.277,-13.929,-13.629,-13.086,
     &-11.679,-10.634,-9.632,-7.872,-5.753,-4.437,
     &-1.240,4.297,10.546,15.317,18.644,23.191,
     &27.261,29.506,30.221,28.802,26.098,22.439,
     &19.199,16.868,14.820,12.905,11.296,9.951,
     &9.513,10.984,12.789,13.350,14.798,-999.000,
     &18.869,-999.000,23.107,28.532,33.310,34.395,
     &29.725,18.146,3.797,-7.004,-14.353,-20.245,
     &-22.797,-999.000,-25.744,-999.000,-25.692,-999.000,
     &-23.918,-20.821,-19.222,-18.475,-17.597
     &/
       data (fjrocvz(it, 7,4, 4), it=1,59) /
     &-19.209,-18.669,-17.592,-16.606,-15.890,-13.846,
     &-11.883,-10.520,-8.612,-7.682,-5.609,-3.327,
     &1.719,4.285,8.010,11.215,14.696,19.903,
     &24.796,28.463,29.413,29.160,27.744,25.291,
     &22.701,20.085,17.702,15.665,13.338,11.023,
     &10.217,10.927,12.164,13.212,14.085,-999.000,
     &16.909,-999.000,21.400,28.163,34.148,38.217,
     &31.134,16.164,2.871,-5.912,-13.925,-21.913,
     &-26.428,-28.029,-28.135,-27.806,-26.913,-999.000,
     &-24.777,-22.765,-21.322,-19.790,-19.209
     &/
       data (fjrocvz(it, 8,4, 4), it=1,59) /
     &-19.008,-17.538,-16.577,-16.071,-15.282,-14.245,
     &-13.326,-12.367,-10.487,-12.592,-14.561,-16.423,
     &-11.442,-4.371,1.117,6.493,9.511,13.086,
     &18.869,23.318,25.457,25.812,25.216,23.930,
     &22.148,20.265,17.964,15.746,13.050,10.623,
     &9.548,9.831,11.069,12.061,12.796,-999.000,
     &15.590,-999.000,20.626,25.034,30.528,31.117,
     &23.424,13.324,3.736,-5.770,-12.506,-19.488,
     &-23.713,-24.669,-23.074,-999.000,-23.572,-999.000,
     &-23.034,-22.904,-21.989,-20.372,-19.008
     &/
       data (fjrocvz(it, 9,4, 4), it=1,59) /
     &-16.274,-14.157,-12.286,-11.094,-11.005,-11.553,
     &-13.009,-14.296,-15.022,-17.011,-19.792,-22.025,
     &-17.139,-7.405,0.466,5.582,7.513,11.015,
     &15.938,19.536,21.176,21.815,21.565,20.139,
     &17.854,15.701,13.827,11.868,9.673,8.442,
     &8.387,9.010,10.223,10.937,12.000,-999.000,
     &13.899,-999.000,19.425,24.674,29.128,31.957,
     &30.908,21.140,10.144,0.227,-6.232,-999.000,
     &-11.548,-999.000,-15.153,-999.000,-18.152,-999.000,
     &-18.905,-19.137,-19.494,-18.457,-16.274
     &/
       data (fjrocvz(it,10,4, 4), it=1,59) /
     &-16.287,-14.127,-11.864,-10.627,-9.855,-11.051,
     &-14.270,-16.945,-18.311,-19.144,-19.917,-19.922,
     &-16.075,-5.483,2.457,7.560,9.731,13.723,
     &18.812,22.013,22.411,21.909,20.572,18.881,
     &16.672,14.054,11.746,9.770,8.438,7.963,
     &8.066,8.443,8.526,8.774,10.212,-999.000,
     &13.095,-999.000,20.182,26.231,32.895,36.630,
     &34.906,24.146,13.446,-999.000,3.393,-999.000,
     &-4.911,-999.000,-10.870,-999.000,-12.745,-999.000,
     &-14.481,-15.702,-16.951,-16.968,-16.287
     &/
       data (fjrocvz(it,11,4, 4), it=1,59) /
     &-18.533,-17.949,-17.646,-17.555,-17.325,-18.054,
     &-18.949,-20.558,-22.358,-24.164,-23.505,-19.703,
     &-14.467,-4.703,2.815,8.812,13.096,16.284,
     &20.582,24.163,25.453,25.228,24.114,22.381,
     &20.500,18.037,15.030,12.860,11.165,10.241,
     &9.605,8.857,8.355,8.491,10.409,-999.000,
     &14.802,-999.000,21.131,27.293,35.089,35.365,
     &30.401,21.290,12.140,-999.000,0.088,-999.000,
     &-8.612,-999.000,-13.424,-999.000,-14.447,-999.000,
     &-16.911,-16.929,-18.012,-19.158,-18.533
     &/
       data (fjrocvz(it,12,4, 4), it=1,59) /
     &-20.640,-21.513,-21.478,-21.399,-21.697,-21.449,
     &-21.335,-23.077,-26.032,-27.659,-28.127,-24.607,
     &-20.225,-10.493,-0.682,5.302,10.240,14.994,
     &21.512,26.294,28.071,27.830,26.682,25.386,
     &23.716,20.918,17.684,15.069,13.474,12.699,
     &11.743,10.472,9.528,9.561,11.782,-999.000,
     &15.916,-999.000,22.045,27.932,35.247,38.774,
     &32.687,17.628,6.598,-0.671,-6.774,-999.000,
     &-10.447,-999.000,-13.456,-999.000,-17.187,-999.000,
     &-19.396,-19.515,-19.918,-20.428,-20.640
     &/
       data (fjrocvz(it,13,4, 4), it=1,59) /
     &-20.581,-20.175,-19.990,-20.032,-19.763,-20.498,
     &-22.067,-23.419,-25.024,-25.230,-25.099,-23.219,
     &-19.909,-11.222,-2.305,4.492,9.396,14.767,
     &23.780,29.342,30.047,29.488,28.394,27.202,
     &25.293,22.100,18.939,16.731,15.796,15.457,
     &14.680,13.338,11.848,11.655,13.458,-999.000,
     &16.656,-999.000,24.914,30.381,38.048,42.198,
     &33.700,18.104,3.202,-4.325,-7.387,-9.180,
     &-10.035,-999.000,-13.482,-999.000,-16.554,-999.000,
     &-18.937,-21.062,-21.080,-20.562,-20.581
     &/
       data (fjrocvz(it,14,4, 4), it=1,59) /
     &-19.264,-18.865,-18.548,-17.986,-17.987,-19.569,
     &-22.297,-25.854,-29.054,-28.420,-25.192,-20.827,
     &-16.461,-13.100,-6.129,1.735,7.613,13.687,
     &22.812,28.250,29.650,29.284,28.043,26.536,
     &24.183,21.599,19.185,17.598,16.987,16.573,
     &15.501,14.157,12.682,12.572,14.623,-999.000,
     &18.843,-999.000,26.857,33.846,38.275,38.158,
     &30.321,16.738,6.960,-999.000,-4.957,-999.000,
     &-12.748,-999.000,-16.234,-999.000,-18.983,-999.000,
     &-21.781,-22.597,-21.707,-20.164,-19.264
     &/
       data (fjrocvz(it,15,4, 4), it=1,59) /
     &-17.530,-16.947,-16.304,-15.802,-15.730,-17.100,
     &-20.881,-26.843,-30.827,-31.914,-28.354,-20.917,
     &-15.805,-11.305,-6.108,1.238,7.390,13.089,
     &20.480,25.503,27.267,26.643,24.749,22.094,
     &19.352,17.185,15.022,13.520,12.714,11.940,
     &11.061,10.074,9.581,10.361,13.114,-999.000,
     &16.690,-999.000,23.291,28.946,32.605,34.500,
     &27.983,14.453,4.780,-999.000,-7.865,-999.000,
     &-14.805,-999.000,-19.564,-999.000,-22.932,-999.000,
     &-24.820,-25.094,-22.728,-19.247,-17.530
     &/
       data (fjrocvz(it,16,4, 4), it=1,59) /
     &-17.641,-16.262,-16.346,-16.441,-16.926,-18.656,
     &-22.182,-26.675,-29.893,-32.728,-31.612,-24.005,
     &-14.494,-9.949,-4.492,1.739,7.281,12.600,
     &19.861,24.826,26.478,26.137,24.579,21.172,
     &16.986,13.220,10.472,8.771,8.027,7.644,
     &6.845,5.756,5.154,5.839,8.074,-999.000,
     &11.631,-999.000,17.849,22.744,26.757,28.353,
     &21.453,7.486,1.364,-999.000,-10.453,-999.000,
     &-17.819,-999.000,-22.668,-999.000,-25.565,-999.000,
     &-26.858,-26.375,-23.872,-20.382,-17.641
     &/
       data (fjrocvz(it,17,4, 4), it=1,59) /
     &-20.491,-19.498,-18.656,-18.061,-18.470,-19.666,
     &-22.509,-26.800,-30.020,-31.665,-29.957,-21.855,
     &-13.030,-7.883,-1.659,3.505,8.081,13.681,
     &21.411,26.616,28.849,29.122,27.800,24.750,
     &20.030,15.394,11.429,8.561,7.461,6.763,
     &5.647,4.153,2.538,2.101,4.191,-999.000,
     &8.742,-999.000,15.905,19.409,22.815,22.216,
     &16.580,5.397,-3.889,-12.133,-18.563,-22.255,
     &-24.007,-999.000,-24.817,-999.000,-26.663,-999.000,
     &-27.102,-25.868,-24.230,-22.216,-20.491
     &/
       data (fjrocvz(it,18,4, 4), it=1,59) /
     &-21.390,-19.689,-18.330,-17.836,-17.419,-17.827,
     &-21.206,-25.563,-27.889,-29.499,-26.998,-19.741,
     &-10.881,-5.761,-1.165,4.063,9.039,16.106,
     &23.193,28.798,32.356,32.743,30.363,26.269,
     &22.143,18.143,13.809,9.958,7.880,6.362,
     &5.021,3.365,1.978,1.858,4.048,-999.000,
     &7.792,10.836,15.306,18.909,20.570,17.562,
     &10.050,2.989,-6.030,-999.000,-19.034,-999.000,
     &-25.800,-999.000,-28.319,-999.000,-28.436,-999.000,
     &-27.110,-26.459,-24.769,-22.922,-21.390
     &/
       data (fjrocvz(it,19,4, 4), it=1,59) /
     &-19.855,-17.929,-17.603,-16.936,-16.162,-16.407,
     &-18.826,-21.244,-22.962,-24.560,-21.874,-14.938,
     &-6.843,-3.515,1.006,6.349,11.123,18.276,
     &26.229,32.419,35.855,35.991,33.257,29.108,
     &24.556,19.977,15.554,11.622,9.891,8.947,
     &7.396,5.511,4.079,3.602,5.421,-999.000,
     &9.788,-999.000,16.728,19.313,19.622,14.433,
     &9.451,1.744,-7.346,-16.696,-23.113,-26.697,
     &-27.679,-999.000,-28.708,-999.000,-27.644,-999.000,
     &-26.163,-25.606,-24.172,-22.099,-19.855
     &/
       data (fjrocvz(it,20,4, 4), it=1,59) /
     &-22.012,-20.174,-18.326,-16.441,-14.958,-14.664,
     &-16.503,-19.300,-20.402,-21.036,-18.468,-9.313,
     &-1.226,-0.064,3.995,8.024,12.007,19.344,
     &27.006,34.008,37.689,38.164,36.069,32.443,
     &28.312,24.057,19.433,15.503,13.314,11.722,
     &9.869,7.846,6.073,5.636,7.899,-999.000,
     &11.759,-999.000,17.067,19.310,22.310,21.196,
     &14.773,2.341,-6.649,-13.403,-19.177,-23.981,
     &-26.602,-999.000,-27.169,-999.000,-27.662,-999.000,
     &-26.883,-26.415,-25.467,-23.761,-22.012
     &/
       data (fjrocvz(it,21,4, 4), it=1,59) /
     &-22.392,-19.905,-17.970,-16.725,-15.936,-15.207,
     &-16.084,-18.516,-20.732,-21.267,-18.391,-10.999,
     &-5.412,-1.229,3.377,8.284,13.275,19.100,
     &24.809,31.314,34.647,35.688,33.736,29.977,
     &26.113,22.507,18.529,14.134,11.221,9.241,
     &7.418,6.115,5.302,5.239,6.956,-999.000,
     &10.639,-999.000,16.585,20.284,23.991,22.860,
     &13.996,-999.000,-2.093,-999.000,-15.360,-999.000,
     &-22.591,-999.000,-25.557,-999.000,-26.315,-999.000,
     &-26.074,-25.685,-25.043,-23.984,-22.392
     &/
       data (fjrocvz(it,22,4, 4), it=1,59) /
     &-21.309,-20.308,-19.201,-18.425,-18.497,-18.114,
     &-18.262,-20.382,-23.964,-24.810,-21.785,-14.880,
     &-6.734,-3.865,1.502,6.258,10.895,16.549,
     &23.010,28.827,31.352,31.300,29.040,24.876,
     &19.903,15.450,11.672,8.351,5.976,4.432,
     &3.506,3.750,4.312,4.621,6.378,-999.000,
     &11.328,-999.000,16.998,18.120,20.381,18.169,
     &11.563,-999.000,-4.712,-999.000,-16.319,-999.000,
     &-23.562,-999.000,-24.600,-999.000,-24.573,-999.000,
     &-24.778,-24.263,-23.353,-22.337,-21.309
     &/
       data (fjrocvz(it,23,4, 4), it=1,59) /
     &-19.136,-18.830,-17.423,-15.581,-15.277,-16.266,
     &-18.111,-20.419,-22.624,-23.511,-21.030,-14.933,
     &-9.861,-6.941,-1.523,3.707,8.885,14.386,
     &22.500,28.175,29.913,28.672,25.892,21.619,
     &16.686,11.672,7.803,4.958,3.246,2.384,
     &1.887,1.738,2.890,4.793,7.594,-999.000,
     &10.835,-999.000,15.437,17.895,18.735,17.906,
     &13.017,3.852,-3.769,-999.000,-14.701,-999.000,
     &-21.023,-999.000,-23.213,-999.000,-23.410,-999.000,
     &-23.463,-22.705,-21.051,-19.477,-19.136
     &/
       data (fjrocvz(it,24,4, 4), it=1,59) /
     &-16.955,-15.643,-14.018,-12.166,-12.403,-14.345,
     &-16.479,-17.814,-19.056,-19.890,-18.036,-13.251,
     &-7.060,-2.025,1.394,4.648,9.217,17.334,
     &25.476,29.054,29.641,28.258,25.049,20.484,
     &15.825,10.923,6.523,3.445,2.588,3.402,
     &3.626,3.221,4.208,5.919,8.170,-999.000,
     &11.218,-999.000,15.293,17.512,19.372,16.870,
     &10.254,-999.000,-0.113,-999.000,-8.309,-999.000,
     &-14.215,-999.000,-18.408,-999.000,-19.497,-999.000,
     &-20.724,-20.465,-19.172,-18.102,-16.955
     &/
       data (fjrocvz(it,25,4, 4), it=1,59) /
     &-16.176,-15.275,-13.852,-12.986,-13.884,-14.694,
     &-15.387,-16.355,-16.971,-16.410,-13.264,-6.703,
     &-7.932,-3.525,1.760,7.821,13.328,22.103,
     &29.060,31.826,31.390,29.849,27.238,22.957,
     &18.514,14.213,8.742,4.688,3.778,4.996,
     &5.567,5.769,6.059,7.472,9.843,-999.000,
     &13.180,-999.000,16.586,18.566,18.163,13.184,
     &6.453,-999.000,-1.006,-999.000,-7.729,-999.000,
     &-12.799,-999.000,-14.795,-999.000,-18.567,-999.000,
     &-19.660,-19.492,-19.090,-17.771,-16.176
     &/
       data (fjrocvz(it, 1,1, 5), it=1,59) /
     &-12.189,-11.685,-11.392,-11.576,-12.056,-12.992,
     &-14.579,-16.302,-18.333,-20.018,-18.978,-12.786,
     &-3.289,4.473,10.432,14.894,19.376,24.321,
     &28.528,31.208,31.218,29.438,27.119,24.158,
     &21.080,17.048,13.069,10.045,6.761,3.354,
     &1.322,0.729,0.714,0.425,0.277,-999.000,
     &-0.221,-999.000,0.692,-999.000,2.984,-999.000,
     &2.111,-999.000,-0.015,-999.000,-4.621,-999.000,
     &-8.943,-999.000,-12.166,-999.000,-14.062,-999.000,
     &-15.892,-16.617,-15.700,-13.588,-12.189
     &/
       data (fjrocvz(it, 2,1, 5), it=1,59) /
     &-14.084,-13.046,-12.728,-12.705,-13.789,-15.098,
     &-15.648,-16.611,-17.685,-18.763,-15.185,-5.696,
     &3.079,9.567,15.001,20.792,25.624,29.709,
     &33.654,36.190,36.435,34.534,31.334,27.672,
     &23.581,18.828,14.142,10.426,6.962,3.514,
     &0.495,-1.327,-2.294,-2.279,-2.311,-999.000,
     &-1.846,-999.000,0.008,-999.000,0.086,-999.000,
     &-0.058,-999.000,-2.681,-999.000,-7.258,-999.000,
     &-11.706,-999.000,-15.180,-999.000,-17.055,-999.000,
     &-17.937,-18.129,-17.429,-15.792,-14.084
     &/
       data (fjrocvz(it, 3,1, 5), it=1,59) /
     &-15.460,-14.956,-14.813,-15.185,-16.140,-15.971,
     &-15.068,-15.106,-14.832,-14.095,-8.477,-0.064,
     &6.934,11.888,16.987,22.158,25.482,28.859,
     &32.578,35.568,35.895,33.589,29.994,26.509,
     &23.414,19.983,16.079,12.070,8.424,4.613,
     &1.322,-0.423,-1.502,-1.198,-1.230,-999.000,
     &-0.201,-999.000,0.353,-999.000,2.163,3.673,
     &4.887,3.626,0.306,-1.719,-5.777,-999.000,
     &-10.469,-999.000,-13.782,-999.000,-15.941,-999.000,
     &-17.116,-17.733,-17.628,-16.567,-15.460
     &/
       data (fjrocvz(it, 4,1, 5), it=1,59) /
     &-15.375,-16.261,-16.274,-15.512,-14.727,-14.762,
     &-14.746,-14.984,-13.797,-11.040,-4.005,4.090,
     &8.994,12.511,17.638,21.970,25.266,28.577,
     &32.109,34.181,33.331,30.142,26.408,23.566,
     &21.086,17.391,13.539,10.699,8.748,6.187,
     &3.256,0.878,-0.506,-0.047,1.593,-999.000,
     &2.095,-999.000,3.632,-999.000,3.962,6.730,
     &9.152,6.190,1.562,-999.000,-3.452,-999.000,
     &-8.263,-999.000,-11.808,-999.000,-14.173,-999.000,
     &-16.504,-16.693,-15.941,-14.760,-15.375
     &/
       data (fjrocvz(it, 5,1, 5), it=1,59) /
     &-12.327,-13.124,-13.235,-12.301,-11.838,-12.964,
     &-14.863,-16.066,-15.229,-12.360,-4.837,3.986,
     &8.372,11.735,16.968,21.622,24.887,27.971,
     &31.246,33.283,32.747,30.043,27.137,24.511,
     &21.425,16.959,12.983,10.754,9.631,8.272,
     &6.191,3.442,1.610,0.935,1.594,-999.000,
     &3.378,-999.000,6.636,-999.000,7.691,11.892,
     &12.362,8.687,4.469,-999.000,-1.787,-999.000,
     &-7.323,-999.000,-11.501,-999.000,-14.307,-999.000,
     &-16.035,-15.376,-13.745,-12.145,-12.327
     &/
       data (fjrocvz(it, 6,1, 5), it=1,59) /
     &-10.741,-11.430,-12.126,-12.861,-13.264,-14.173,
     &-16.084,-17.387,-16.683,-13.257,-6.826,0.904,
     &4.962,7.545,11.230,15.101,18.300,21.990,
     &26.239,29.657,29.994,28.499,26.511,24.324,
     &21.554,18.016,14.864,12.363,10.913,10.016,
     &9.160,7.571,4.873,3.027,3.206,-999.000,
     &4.638,-999.000,6.767,-999.000,10.640,-999.000,
     &11.284,-999.000,7.052,-999.000,2.096,-999.000,
     &-3.088,-999.000,-7.399,-999.000,-11.089,-999.000,
     &-13.577,-14.442,-13.385,-11.406,-10.741
     &/
       data (fjrocvz(it, 7,1, 5), it=1,59) /
     &-13.783,-14.426,-15.417,-16.306,-16.854,-17.103,
     &-17.365,-17.263,-16.331,-14.336,-10.689,-7.135,
     &-3.672,0.961,5.954,9.677,13.181,16.758,
     &21.812,25.332,26.251,25.059,22.921,20.881,
     &19.050,16.599,13.625,11.001,9.273,8.890,
     &9.402,8.770,6.693,4.726,4.412,-999.000,
     &4.678,-999.000,7.216,-999.000,12.360,14.955,
     &19.522,19.007,12.312,-999.000,6.037,-999.000,
     &0.702,-999.000,-4.154,-999.000,-8.672,-999.000,
     &-12.269,-14.510,-14.778,-14.301,-13.783
     &/
       data (fjrocvz(it, 8,1, 5), it=1,59) /
     &-17.361,-16.753,-16.503,-16.370,-16.405,-16.475,
     &-16.987,-17.682,-18.180,-17.972,-15.411,-10.384,
     &-5.531,-2.391,0.457,3.163,6.264,10.667,
     &16.226,20.495,22.517,21.867,19.786,17.643,
     &15.572,13.277,10.899,8.502,7.220,7.095,
     &8.345,8.834,8.169,6.675,6.458,-999.000,
     &7.568,-999.000,11.079,-999.000,18.345,25.902,
     &29.536,29.683,22.993,13.726,9.074,-999.000,
     &-1.119,-999.000,-7.830,-999.000,-12.423,-999.000,
     &-15.915,-18.345,-18.996,-18.305,-17.361
     &/
       data (fjrocvz(it, 9,1, 5), it=1,59) /
     &-18.626,-17.426,-16.562,-15.957,-15.571,-16.103,
     &-17.407,-18.789,-19.677,-19.876,-17.884,-13.970,
     &-10.086,-6.076,-1.526,1.734,4.063,7.682,
     &12.764,17.278,19.695,20.051,19.129,17.692,
     &15.721,14.056,12.599,11.188,9.961,9.176,
     &9.480,9.428,8.800,8.068,8.801,8.599,
     &8.715,9.968,14.285,19.883,26.012,34.506,
     &40.462,36.658,21.677,7.145,-1.386,-6.728,
     &-8.957,-999.000,-14.680,-999.000,-18.043,-999.000,
     &-20.334,-21.929,-21.674,-20.212,-18.626
     &/
       data (fjrocvz(it,10,1, 5), it=1,59) /
     &-17.458,-16.145,-15.230,-14.596,-14.951,-16.272,
     &-17.510,-18.389,-19.636,-21.579,-20.646,-16.111,
     &-10.244,-4.374,0.724,4.669,7.436,10.320,
     &14.020,17.415,20.181,22.080,22.685,22.043,
     &20.163,18.450,17.095,15.870,14.318,13.011,
     &12.297,11.582,10.488,9.644,9.359,-999.000,
     &9.925,13.154,20.126,29.134,39.100,45.328,
     &39.601,26.587,11.372,-3.162,-8.716,-12.213,
     &-14.512,-999.000,-18.393,-999.000,-21.224,-999.000,
     &-22.610,-22.509,-21.212,-19.474,-17.458
     &/
       data (fjrocvz(it,11,1, 5), it=1,59) /
     &-16.484,-15.181,-14.767,-15.218,-16.597,-18.251,
     &-19.620,-20.487,-21.387,-22.310,-20.254,-13.312,
     &-5.883,0.274,5.674,9.572,11.843,13.523,
     &16.177,20.202,24.407,27.048,27.674,26.469,
     &24.610,22.655,21.007,18.793,16.545,14.772,
     &13.092,11.941,11.014,10.556,10.841,11.128,
     &12.300,16.691,23.671,30.745,35.118,35.375,
     &23.306,8.790,-1.467,-10.239,-15.742,-999.000,
     &-17.734,-999.000,-20.516,-999.000,-22.049,-999.000,
     &-22.380,-21.597,-19.986,-18.126,-16.484
     &/
       data (fjrocvz(it,12,1, 5), it=1,59) /
     &-18.202,-17.680,-17.949,-19.256,-20.436,-20.864,
     &-21.210,-22.006,-23.187,-23.479,-21.677,-16.651,
     &-8.885,-1.755,7.174,11.581,13.994,16.103,
     &18.840,22.751,26.111,28.189,28.819,28.387,
     &27.340,25.386,22.833,19.788,16.587,13.931,
     &12.348,11.049,10.308,11.142,13.564,-999.000,
     &16.609,16.960,21.807,28.344,31.647,28.622,
     &19.365,9.769,1.672,-999.000,-11.994,-999.000,
     &-20.105,-999.000,-22.965,-999.000,-23.128,-999.000,
     &-22.314,-21.380,-20.351,-19.111,-18.202
     &/
       data (fjrocvz(it,13,1, 5), it=1,59) /
     &-19.072,-19.456,-20.093,-20.913,-21.010,-20.128,
     &-19.922,-21.123,-22.124,-22.843,-21.675,-16.934,
     &-7.916,-2.009,5.227,10.898,14.047,16.456,
     &19.329,22.734,25.425,26.904,27.047,26.365,
     &24.838,22.827,20.854,18.071,14.221,11.763,
     &11.178,10.943,10.676,10.574,10.668,-999.000,
     &13.392,-999.000,18.241,22.760,25.693,22.419,
     &15.874,-999.000,4.908,-999.000,-8.466,-999.000,
     &-18.046,-999.000,-22.558,-999.000,-23.119,-999.000,
     &-22.495,-21.552,-20.423,-19.346,-19.072
     &/
       data (fjrocvz(it,14,1, 5), it=1,59) /
     &-19.064,-20.294,-21.185,-21.271,-20.355,-19.299,
     &-20.362,-21.962,-22.187,-22.474,-19.431,-16.020,
     &-9.812,-5.247,2.219,8.732,12.500,14.854,
     &17.325,20.553,23.789,25.318,25.120,23.691,
     &22.081,20.504,18.640,15.520,12.031,9.661,
     &9.169,9.057,8.223,6.335,5.825,-999.000,
     &8.068,9.839,14.958,17.826,18.879,18.594,
     &15.696,8.602,4.155,-999.000,-6.485,-999.000,
     &-15.588,-999.000,-21.457,-999.000,-23.242,-999.000,
     &-23.165,-22.069,-20.383,-19.180,-19.064
     &/
       data (fjrocvz(it,15,1, 5), it=1,59) /
     &-19.690,-20.258,-21.542,-22.375,-21.694,-20.271,
     &-20.634,-21.930,-23.535,-25.453,-22.862,-17.797,
     &-10.140,-5.740,0.190,6.041,10.057,13.043,
     &15.487,17.284,19.046,19.957,19.356,17.903,
     &16.685,15.682,14.419,12.583,9.798,7.260,
     &6.439,6.116,5.119,4.556,4.873,-999.000,
     &7.140,-999.000,13.377,18.019,21.109,22.114,
     &18.255,11.046,6.029,-999.000,-4.425,-999.000,
     &-13.625,-999.000,-20.655,-999.000,-23.340,-999.000,
     &-23.135,-22.335,-21.227,-20.272,-19.690
     &/
       data (fjrocvz(it,16,1, 5), it=1,59) /
     &-21.864,-21.903,-22.580,-23.372,-22.917,-21.170,
     &-20.822,-21.335,-23.143,-28.371,-29.110,-23.804,
     &-9.212,-2.796,1.195,5.745,9.411,12.183,
     &14.620,16.514,18.368,18.547,17.899,16.628,
     &15.321,13.972,12.496,10.266,7.246,5.184,
     &4.997,5.440,5.572,5.861,6.741,-999.000,
     &9.968,-999.000,16.419,20.820,23.229,22.220,
     &17.539,-999.000,5.782,-999.000,-3.635,-999.000,
     &-13.343,-999.000,-20.584,-999.000,-23.284,-999.000,
     &-23.775,-23.248,-22.661,-22.330,-21.864
     &/
       data (fjrocvz(it,17,1, 5), it=1,59) /
     &-22.639,-22.441,-22.604,-23.413,-23.926,-23.296,
     &-23.153,-21.643,-21.217,-25.720,-28.375,-24.059,
     &-9.285,1.021,4.434,7.370,10.103,12.567,
     &14.950,17.006,18.582,19.524,19.517,18.437,
     &17.191,15.523,13.722,11.338,8.131,5.902,
     &5.524,6.124,6.100,5.951,7.293,-999.000,
     &9.584,12.553,15.339,18.286,20.014,20.509,
     &15.580,-999.000,0.959,-999.000,-9.227,-999.000,
     &-18.152,-999.000,-23.940,-999.000,-26.196,-999.000,
     &-25.683,-24.037,-22.996,-22.825,-22.639
     &/
       data (fjrocvz(it,18,1, 5), it=1,59) /
     &-22.393,-22.070,-22.248,-23.372,-24.916,-24.767,
     &-24.848,-24.096,-22.421,-23.106,-22.308,-17.833,
     &-6.150,4.147,8.285,10.955,13.689,16.217,
     &18.490,19.356,20.141,21.188,21.104,19.781,
     &17.925,16.020,13.955,11.974,9.461,7.542,
     &6.649,6.731,6.903,7.311,8.144,-999.000,
     &10.352,-999.000,13.724,18.292,19.879,17.882,
     &11.025,2.231,-4.835,-999.000,-14.172,-999.000,
     &-21.571,-999.000,-27.146,-999.000,-28.688,-999.000,
     &-27.159,-24.875,-23.679,-23.062,-22.393
     &/
       data (fjrocvz(it,19,1, 5), it=1,59) /
     &-23.167,-22.050,-21.148,-21.338,-21.792,-20.845,
     &-19.917,-19.991,-19.716,-21.558,-21.248,-17.384,
     &-7.353,3.468,9.839,13.989,17.106,20.101,
     &22.634,23.688,24.387,25.023,25.174,23.607,
     &21.690,19.980,17.248,13.613,10.646,9.538,
     &9.315,9.077,8.799,8.036,8.015,-999.000,
     &9.678,-999.000,12.532,15.771,15.017,10.590,
     &5.823,-999.000,-5.679,-999.000,-14.523,-999.000,
     &-22.110,-999.000,-26.990,-999.000,-28.401,-999.000,
     &-27.124,-25.219,-24.187,-23.845,-23.167
     &/
       data (fjrocvz(it,20,1, 5), it=1,59) /
     &-22.012,-20.846,-20.418,-20.403,-20.566,-19.582,
     &-18.652,-18.267,-18.315,-19.830,-22.011,-19.607,
     &-10.484,1.547,8.320,12.055,16.021,20.031,
     &23.280,24.952,25.710,26.416,26.602,25.703,
     &25.042,24.271,22.142,18.406,14.290,12.524,
     &12.068,11.448,10.056,8.573,8.196,-999.000,
     &7.637,-999.000,8.174,10.452,13.672,10.411,
     &2.978,-999.000,-7.048,-999.000,-16.351,-999.000,
     &-23.604,-999.000,-28.288,-999.000,-29.803,-999.000,
     &-29.270,-27.476,-25.569,-23.721,-22.012
     &/
       data (fjrocvz(it,21,1, 5), it=1,59) /
     &-23.230,-22.407,-22.438,-22.934,-23.187,-21.755,
     &-19.868,-19.773,-21.316,-24.223,-26.960,-24.854,
     &-14.753,-2.439,4.613,8.581,12.627,16.198,
     &19.706,22.345,23.887,24.292,24.071,23.901,
     &24.126,23.584,21.569,17.747,13.299,11.156,
     &10.690,10.278,9.091,7.371,6.397,-999.000,
     &7.103,-999.000,8.511,10.733,12.118,7.879,
     &-0.268,-999.000,-9.896,-999.000,-17.516,-999.000,
     &-23.766,-999.000,-28.473,-999.000,-30.055,-999.000,
     &-29.929,-28.751,-27.087,-25.054,-23.230
     &/
       data (fjrocvz(it,22,1, 5), it=1,59) /
     &-21.581,-21.021,-21.328,-22.435,-23.030,-21.985,
     &-20.506,-20.710,-22.278,-24.878,-26.895,-24.823,
     &-15.244,-3.895,3.174,7.171,10.170,14.121,
     &17.973,21.477,22.812,23.016,23.428,23.572,
     &23.210,21.711,18.584,13.714,8.703,6.276,
     &5.985,6.019,5.060,3.783,3.568,-999.000,
     &4.161,-999.000,3.736,-999.000,4.306,-999.000,
     &0.055,-999.000,-10.769,-999.000,-16.292,-999.000,
     &-21.162,-999.000,-25.501,-999.000,-27.401,-999.000,
     &-27.336,-25.991,-24.109,-22.669,-21.581
     &/
       data (fjrocvz(it,23,1, 5), it=1,59) /
     &-16.244,-16.296,-16.712,-17.631,-18.648,-18.437,
     &-18.235,-18.785,-19.430,-20.531,-20.742,-17.232,
     &-9.389,-1.531,4.423,7.546,9.946,13.035,
     &16.777,20.353,22.707,24.825,25.913,25.353,
     &23.596,20.936,18.017,14.149,9.344,6.059,
     &4.075,3.000,2.059,1.525,1.669,-999.000,
     &2.459,-999.000,5.777,-999.000,6.840,-999.000,
     &4.301,-999.000,-0.425,-999.000,-9.888,-999.000,
     &-15.425,-999.000,-19.990,-999.000,-21.882,-999.000,
     &-21.879,-20.660,-18.730,-16.907,-16.244
     &/
       data (fjrocvz(it,24,1, 5), it=1,59) /
     &-12.734,-12.645,-12.743,-13.486,-14.007,-13.604,
     &-13.801,-15.068,-16.280,-18.140,-18.044,-13.655,
     &-5.575,1.514,6.581,10.377,13.230,16.355,
     &19.878,23.212,25.196,25.783,25.217,23.706,
     &21.460,18.790,16.147,13.345,9.302,5.748,
     &3.279,2.360,2.200,1.822,1.315,-999.000,
     &2.185,-999.000,3.983,-999.000,5.915,-999.000,
     &5.098,-999.000,1.708,-999.000,-3.685,-999.000,
     &-9.204,-999.000,-13.222,-999.000,-15.095,-999.000,
     &-16.079,-16.281,-15.182,-13.636,-12.734
     &/
       data (fjrocvz(it,25,1, 5), it=1,59) /
     &-12.189,-11.685,-11.392,-11.576,-12.056,-12.992,
     &-14.579,-16.302,-18.333,-20.018,-18.978,-12.786,
     &-3.289,4.473,10.432,14.894,19.376,24.321,
     &28.528,31.208,31.218,29.438,27.119,24.158,
     &21.080,17.048,13.069,10.045,6.761,3.354,
     &1.322,0.729,0.714,0.425,0.277,-999.000,
     &-0.221,-999.000,0.692,-999.000,2.984,-999.000,
     &2.111,-999.000,-0.015,-999.000,-4.621,-999.000,
     &-8.943,-999.000,-12.166,-999.000,-14.062,-999.000,
     &-15.892,-16.617,-15.700,-13.588,-12.189
     &/
       data (fjrocvz(it, 1,2, 5), it=1,59) /
     &-17.153,-16.303,-14.792,-13.730,-14.541,-15.260,
     &-15.817,-16.651,-17.138,-16.501,-13.226,-6.473,
     &-7.809,-3.650,1.642,7.622,12.987,21.741,
     &28.703,31.459,31.028,29.615,27.079,22.867,
     &18.564,14.575,9.198,5.213,4.598,5.964,
     &6.567,6.836,7.277,8.855,11.389,-999.000,
     &15.024,-999.000,18.739,21.385,21.549,16.637,
     &9.470,-999.000,-0.601,-999.000,-8.367,-999.000,
     &-13.770,-999.000,-15.737,-999.000,-19.763,-999.000,
     &-20.764,-20.569,-20.203,-18.816,-17.153
     &/
       data (fjrocvz(it, 2,2, 5), it=1,59) /
     &-15.419,-14.741,-13.854,-13.428,-12.773,-11.931,
     &-12.020,-13.521,-14.554,-13.779,-8.370,-0.413,
     &5.274,8.258,12.479,14.689,18.494,23.727,
     &29.110,32.792,33.545,32.422,29.540,25.165,
     &20.791,17.489,13.649,9.913,7.802,7.793,
     &8.900,9.689,10.450,11.907,13.386,-999.000,
     &15.675,-999.000,19.982,22.375,23.792,22.918,
     &16.822,-999.000,-0.633,-999.000,-10.869,-999.000,
     &-11.900,-999.000,-16.841,-999.000,-20.894,-999.000,
     &-20.747,-20.547,-18.999,-16.603,-15.419
     &/
       data (fjrocvz(it, 3,2, 5), it=1,59) /
     &-15.634,-14.437,-13.776,-12.647,-11.648,-11.570,
     &-11.060,-11.070,-11.446,-10.789,-6.296,0.177,
     &6.127,9.586,11.872,14.825,18.543,21.808,
     &25.203,27.877,29.199,28.256,25.231,21.380,
     &17.881,15.395,12.995,10.620,8.659,8.339,
     &9.297,10.213,11.464,12.217,13.879,-999.000,
     &16.956,-999.000,19.824,24.796,28.272,26.011,
     &15.908,5.999,-1.154,-10.455,-15.020,-14.764,
     &-15.311,-999.000,-20.425,-999.000,-22.374,-999.000,
     &-22.419,-21.029,-19.248,-17.314,-15.634
     &/
       data (fjrocvz(it, 4,2, 5), it=1,59) /
     &-16.963,-15.829,-14.247,-12.670,-12.308,-12.186,
     &-11.379,-9.881,-8.993,-8.223,-5.184,0.286,
     &6.274,8.798,10.005,12.619,16.245,20.514,
     &24.017,26.052,25.763,23.693,20.768,17.379,
     &14.322,11.920,9.822,8.456,7.961,7.925,
     &8.699,10.104,11.160,13.009,16.215,-999.000,
     &18.065,-999.000,21.375,26.752,28.242,26.657,
     &20.802,11.419,-1.652,-14.502,-18.457,-999.000,
     &-17.838,-999.000,-21.350,-999.000,-24.259,-999.000,
     &-23.320,-21.063,-19.603,-17.938,-16.963
     &/
       data (fjrocvz(it, 5,2, 5), it=1,59) /
     &-17.657,-16.389,-14.133,-12.495,-12.348,-12.941,
     &-12.731,-11.492,-10.808,-9.898,-6.877,-0.649,
     &5.362,8.348,11.009,13.134,16.784,21.721,
     &25.932,28.329,28.180,26.092,22.569,18.802,
     &15.416,13.072,11.336,10.417,9.528,8.991,
     &9.617,10.956,12.609,14.251,15.498,-999.000,
     &19.316,-999.000,23.282,27.113,29.916,31.813,
     &26.741,12.240,-1.714,-11.244,-17.715,-20.719,
     &-23.294,-26.166,-27.162,-26.905,-25.350,-999.000,
     &-23.516,-21.269,-19.586,-18.293,-17.657
     &/
       data (fjrocvz(it, 6,2, 5), it=1,59) /
     &-18.486,-17.613,-16.140,-14.748,-14.429,-14.141,
     &-12.863,-11.997,-11.383,-9.601,-7.424,-6.028,
     &-2.750,3.068,9.451,14.364,18.000,22.740,
     &26.905,29.305,30.238,28.960,26.361,22.840,
     &19.768,17.598,15.595,13.612,12.086,10.706,
     &10.115,11.714,13.600,14.210,15.814,-999.000,
     &20.056,-999.000,24.286,30.613,35.887,37.426,
     &32.415,19.814,3.649,-8.024,-16.277,-22.031,
     &-24.463,-999.000,-27.026,-999.000,-26.889,-999.000,
     &-25.004,-21.801,-20.160,-19.383,-18.486
     &/
       data (fjrocvz(it, 7,2, 5), it=1,59) /
     &-20.217,-19.738,-18.656,-17.679,-17.009,-15.194,
     &-13.353,-12.126,-10.519,-9.542,-7.429,-5.083,
     &0.042,2.956,6.839,10.224,14.092,19.476,
     &24.464,28.257,29.428,29.313,27.973,25.573,
     &23.111,20.563,18.143,16.137,13.803,11.353,
     &10.538,11.298,12.527,13.850,14.936,-999.000,
     &17.825,-999.000,22.683,30.115,36.559,40.607,
     &32.697,16.757,2.435,-7.033,-15.729,-23.616,
     &-28.065,-29.523,-29.472,-29.048,-28.147,-999.000,
     &-25.822,-23.746,-22.264,-20.748,-20.217
     &/
       data (fjrocvz(it, 8,2, 5), it=1,59) /
     &-20.132,-18.748,-17.835,-17.372,-16.673,-15.834,
     &-15.026,-14.175,-12.508,-14.503,-16.413,-18.194,
     &-13.089,-5.704,-0.065,5.485,8.857,12.596,
     &18.444,22.960,25.363,25.829,25.225,24.011,
     &22.302,20.394,18.086,15.859,13.110,10.680,
     &9.606,9.851,11.164,12.402,13.356,-999.000,
     &16.266,-999.000,21.757,26.706,32.644,33.186,
     &24.602,13.373,3.228,-6.962,-13.948,-20.947,
     &-25.234,-26.154,-24.470,-999.000,-24.842,-999.000,
     &-24.171,-24.021,-23.079,-21.482,-20.132
     &/
       data (fjrocvz(it, 9,2, 5), it=1,59) /
     &-17.515,-15.477,-13.675,-12.573,-12.572,-13.270,
     &-14.865,-16.244,-17.117,-19.116,-21.835,-23.888,
     &-18.820,-8.802,-0.806,4.474,6.772,10.397,
     &15.310,19.012,20.879,21.580,21.407,20.052,
     &17.782,15.661,13.754,11.761,9.583,8.344,
     &8.285,8.930,10.182,11.082,12.325,-999.000,
     &14.391,-999.000,20.607,26.243,31.258,33.957,
     &32.226,21.211,9.927,-0.424,-7.321,-999.000,
     &-12.995,-999.000,-16.555,-999.000,-19.482,-999.000,
     &-20.174,-20.358,-20.657,-19.658,-17.515
     &/
       data (fjrocvz(it,10,2, 5), it=1,59) /
     &-17.725,-15.573,-13.333,-12.172,-11.631,-12.949,
     &-16.140,-18.888,-20.518,-21.443,-22.150,-21.962,
     &-17.878,-6.970,1.135,6.487,8.944,12.957,
     &18.061,21.340,21.901,21.549,20.321,18.692,
     &16.595,13.985,11.640,9.696,8.405,7.943,
     &8.061,8.461,8.608,8.959,10.501,-999.000,
     &13.854,-999.000,21.201,27.664,34.940,38.878,
     &36.640,25.239,13.908,-999.000,2.633,-999.000,
     &-6.290,-999.000,-12.371,-999.000,-14.377,-999.000,
     &-16.167,-17.251,-18.377,-18.412,-17.725
     &/
       data (fjrocvz(it,11,2, 5), it=1,59) /
     &-20.062,-19.422,-19.055,-19.054,-19.057,-19.797,
     &-20.710,-22.341,-24.405,-26.426,-25.763,-21.995,
     &-16.536,-6.462,1.291,7.519,12.002,15.359,
     &19.725,23.390,24.868,24.773,23.706,22.121,
     &20.410,18.005,15.056,12.940,11.360,10.452,
     &9.833,9.114,8.635,8.844,10.850,-999.000,
     &15.500,-999.000,22.187,28.677,36.840,37.232,
     &32.303,22.721,13.033,-999.000,-0.256,-999.000,
     &-9.545,-999.000,-14.895,-999.000,-16.166,-999.000,
     &-18.616,-18.538,-19.501,-20.655,-20.062
     &/
       data (fjrocvz(it,12,2, 5), it=1,59) /
     &-22.004,-22.776,-22.706,-22.643,-22.935,-22.760,
     &-22.799,-24.560,-27.617,-29.614,-30.279,-26.920,
     &-22.531,-12.354,-2.353,3.809,8.984,13.909,
     &20.509,25.381,27.349,27.280,26.243,25.074,
     &23.575,20.973,17.836,15.315,13.896,13.143,
     &12.206,10.940,10.018,10.157,12.511,-999.000,
     &16.811,-999.000,23.200,29.302,36.746,40.682,
     &34.546,18.561,6.819,-0.837,-7.402,-999.000,
     &-11.492,-999.000,-15.050,-999.000,-18.750,-999.000,
     &-20.964,-20.995,-21.317,-21.814,-22.004
     &/
       data (fjrocvz(it,13,2, 5), it=1,59) /
     &-21.583,-21.132,-20.923,-20.929,-20.658,-21.440,
     &-23.051,-24.437,-25.992,-26.506,-26.843,-25.275,
     &-22.233,-13.014,-3.916,3.021,8.038,13.582,
     &22.697,28.348,29.286,28.952,27.952,26.899,
     &25.212,22.226,19.185,17.100,16.366,16.066,
     &15.311,13.983,12.524,12.439,14.390,-999.000,
     &17.733,-999.000,26.172,31.951,39.953,44.372,
     &35.639,19.018,3.466,-4.723,-8.487,-10.366,
     &-11.271,-999.000,-14.827,-999.000,-17.758,-999.000,
     &-20.071,-22.127,-22.085,-21.567,-21.583
     &/
       data (fjrocvz(it,14,2, 5), it=1,59) /
     &-20.083,-19.665,-19.336,-18.766,-18.765,-20.380,
     &-23.149,-26.710,-29.875,-29.522,-26.561,-22.494,
     &-18.493,-14.604,-7.479,0.519,6.546,12.755,
     &21.953,27.469,29.065,28.869,27.730,26.358,
     &24.206,21.829,19.525,18.035,17.619,17.285,
     &16.248,14.946,13.557,13.545,15.710,-999.000,
     &20.139,-999.000,28.355,36.100,40.946,40.650,
     &32.483,18.197,7.173,-999.000,-6.366,-999.000,
     &-14.013,-999.000,-17.296,-999.000,-19.940,-999.000,
     &-22.652,-23.449,-22.513,-20.965,-20.083
     &/
       data (fjrocvz(it,15,2, 5), it=1,59) /
     &-18.246,-17.685,-17.101,-16.618,-16.583,-18.097,
     &-21.974,-27.954,-31.997,-33.278,-29.860,-22.658,
     &-17.787,-12.851,-7.460,0.022,6.510,12.372,
     &19.771,24.862,26.778,26.319,24.525,21.975,
     &19.412,17.417,15.329,13.910,13.277,12.617,
     &11.799,10.875,10.493,11.399,14.325,-999.000,
     &18.131,-999.000,24.972,31.406,35.674,37.680,
     &30.829,16.313,5.240,-999.000,-8.924,-999.000,
     &-15.882,-999.000,-20.496,-999.000,-23.752,-999.000,
     &-25.567,-25.843,-23.450,-19.966,-18.246
     &/
       data (fjrocvz(it,16,2, 5), it=1,59) /
     &-18.313,-17.022,-17.140,-17.311,-17.937,-19.845,
     &-23.471,-28.063,-31.471,-34.364,-33.290,-25.792,
     &-16.343,-11.261,-5.534,0.873,6.788,12.101,
     &19.357,24.293,25.978,25.799,24.308,21.034,
     &17.018,13.412,10.741,9.114,8.520,8.276,
     &7.543,6.506,6.051,6.835,9.177,-999.000,
     &13.066,-999.000,19.602,25.348,30.130,31.786,
     &24.330,9.314,1.763,-999.000,-11.333,-999.000,
     &-18.690,-999.000,-23.490,-999.000,-26.350,-999.000,
     &-27.609,-27.169,-24.681,-21.130,-18.313
     &/
       data (fjrocvz(it,17,2, 5), it=1,59) /
     &-21.297,-20.364,-19.582,-19.051,-19.535,-20.964,
     &-23.959,-28.402,-31.925,-33.597,-31.922,-23.873,
     &-15.047,-9.398,-2.900,2.455,7.412,13.029,
     &20.784,26.056,28.343,28.761,27.548,24.631,
     &20.083,15.625,11.728,8.911,7.972,7.380,
     &6.318,4.903,3.397,3.030,5.254,-999.000,
     &10.188,-999.000,17.722,21.995,25.784,25.649,
     &19.597,7.099,-3.404,-12.301,-19.078,-22.842,
     &-24.667,-999.000,-25.620,-999.000,-27.499,-999.000,
     &-27.969,-26.818,-25.214,-23.100,-21.297
     &/
       data (fjrocvz(it,18,2, 5), it=1,59) /
     &-22.301,-20.705,-19.380,-18.915,-18.494,-19.114,
     &-22.646,-27.156,-29.799,-31.429,-28.962,-21.750,
     &-12.913,-7.397,-2.550,2.868,8.203,15.358,
     &22.512,28.228,31.909,32.465,30.183,26.212,
     &22.286,18.448,14.177,10.389,8.454,7.009,
     &5.726,4.115,2.793,2.604,4.798,-999.000,
     &8.970,12.364,17.146,21.324,23.675,20.785,
     &13.138,5.063,-4.979,-999.000,-19.323,-999.000,
     &-26.284,-999.000,-28.987,-999.000,-29.258,-999.000,
     &-28.068,-27.536,-25.942,-23.969,-22.301
     &/
       data (fjrocvz(it,19,2, 5), it=1,59) /
     &-20.771,-18.925,-18.639,-17.944,-17.055,-17.639,
     &-20.254,-22.869,-25.026,-26.745,-24.184,-17.453,
     &-9.555,-5.724,-0.900,4.608,9.750,17.263,
     &25.345,31.732,35.544,35.851,33.207,29.187,
     &24.809,20.402,16.068,12.195,10.619,9.741,
     &8.195,6.337,4.978,4.438,6.197,-999.000,
     &11.013,-999.000,18.551,22.221,23.155,18.476,
     &12.928,4.228,-6.119,-15.862,-23.008,-26.786,
     &-27.942,-999.000,-29.256,-999.000,-28.425,-999.000,
     &-27.161,-26.703,-25.353,-23.136,-20.771
     &/
       data (fjrocvz(it,20,2, 5), it=1,59) /
     &-22.785,-21.063,-19.250,-17.381,-15.907,-15.882,
     &-17.883,-20.850,-22.351,-23.098,-20.600,-11.606,
     &-3.653,-2.099,2.201,6.406,10.652,18.297,
     &26.075,33.300,37.329,37.963,35.965,32.445,
     &28.483,24.442,19.913,16.070,14.113,12.580,
     &10.724,8.748,7.054,6.508,8.738,-999.000,
     &13.061,-999.000,18.909,22.257,25.897,25.274,
     &18.420,4.738,-5.606,-12.864,-19.218,-24.141,
     &-26.902,-999.000,-27.745,-999.000,-28.401,-999.000,
     &-27.777,-27.358,-26.451,-24.632,-22.785
     &/
       data (fjrocvz(it,21,2, 5), it=1,59) /
     &-23.049,-20.600,-18.679,-17.436,-16.615,-16.143,
     &-17.165,-19.760,-22.365,-23.080,-20.290,-13.035,
     &-7.632,-3.078,1.836,6.906,12.108,18.158,
     &23.949,30.535,34.135,35.352,33.506,29.896,
     &26.219,22.849,18.980,14.713,12.054,10.182,
     &8.387,7.111,6.408,6.443,8.245,-999.000,
     &12.148,-999.000,18.471,22.817,27.176,26.030,
     &17.113,-999.000,-1.165,-999.000,-15.286,-999.000,
     &-22.823,-999.000,-26.146,-999.000,-27.057,-999.000,
     &-26.956,-26.555,-25.900,-24.743,-23.049
     &/
       data (fjrocvz(it,22,2, 5), it=1,59) /
     &-21.958,-20.919,-19.794,-18.978,-18.936,-18.643,
     &-18.839,-21.019,-24.726,-25.629,-22.647,-15.794,
     &-7.738,-4.830,0.600,5.428,10.202,15.934,
     &22.313,28.174,30.878,30.984,28.848,24.839,
     &20.045,15.825,12.153,8.932,6.812,5.391,
     &4.492,4.782,5.471,5.949,7.891,-999.000,
     &12.903,-999.000,18.986,20.969,23.638,20.871,
     &13.898,-999.000,-3.867,-999.000,-16.354,-999.000,
     &-23.881,-999.000,-25.089,-999.000,-25.209,-999.000,
     &-25.605,-25.093,-24.216,-23.102,-21.958
     &/
       data (fjrocvz(it,23,2, 5), it=1,59) /
     &-19.925,-19.563,-18.106,-16.192,-15.702,-16.611,
     &-18.462,-20.796,-22.949,-23.780,-21.266,-15.176,
     &-10.158,-7.335,-1.951,3.271,8.456,13.953,
     &22.022,27.733,29.534,28.436,25.740,21.609,
     &16.879,12.093,8.307,5.570,4.127,3.388,
     &2.945,2.839,4.097,6.225,9.251,-999.000,
     &12.809,-999.000,17.800,20.808,22.096,21.060,
     &15.489,5.454,-3.024,-999.000,-14.876,-999.000,
     &-21.335,-999.000,-23.620,-999.000,-23.963,-999.000,
     &-24.216,-23.559,-22.030,-20.364,-19.925
     &/
       data (fjrocvz(it,24,2, 5), it=1,59) /
     &-17.892,-16.528,-14.842,-12.811,-12.758,-14.554,
     &-16.706,-18.001,-19.092,-19.788,-17.707,-12.877,
     &-6.853,-2.126,1.224,4.393,8.817,16.943,
     &25.069,28.632,29.257,28.030,24.881,20.440,
     &15.946,11.301,7.014,4.036,3.438,4.417,
     &4.693,4.341,5.481,7.388,9.847,-999.000,
     &13.235,-999.000,17.555,20.662,23.067,20.346,
     &12.999,-999.000,0.343,-999.000,-8.681,-999.000,
     &-14.889,-999.000,-19.326,-999.000,-20.380,-999.000,
     &-21.702,-21.489,-20.242,-19.110,-17.892
     &/
       data (fjrocvz(it,25,2, 5), it=1,59) /
     &-17.153,-16.303,-14.792,-13.730,-14.541,-15.260,
     &-15.817,-16.651,-17.138,-16.501,-13.226,-6.473,
     &-7.809,-3.650,1.642,7.622,12.987,21.741,
     &28.703,31.459,31.028,29.615,27.079,22.867,
     &18.564,14.575,9.198,5.213,4.598,5.964,
     &6.567,6.836,7.277,8.855,11.389,-999.000,
     &15.024,-999.000,18.739,21.385,21.549,16.637,
     &9.470,-999.000,-0.601,-999.000,-8.367,-999.000,
     &-13.770,-999.000,-15.737,-999.000,-19.763,-999.000,
     &-20.764,-20.569,-20.203,-18.816,-17.153
     &/
       data (fjrocvz(it, 1,3, 5), it=1,59) /
     &-22.481,-23.659,-23.412,-22.464,-21.032,-19.887,
     &-19.840,-23.108,-29.978,-31.573,-24.550,-10.749,
     &0.890,5.213,8.017,10.178,10.283,9.026,
     &9.228,11.685,14.738,15.652,15.077,12.655,
     &9.307,7.797,8.392,9.441,9.498,8.680,
     &8.007,7.960,7.837,7.152,7.423,-999.000,
     &8.919,-999.000,11.124,-999.000,16.697,19.665,
     &19.980,16.450,7.382,-999.000,-2.132,-999.000,
     &-9.105,-999.000,-12.271,-999.000,-14.365,-999.000,
     &-16.890,-19.141,-20.550,-21.263,-22.481
     &/
       data (fjrocvz(it, 2,3, 5), it=1,59) /
     &-20.290,-20.923,-20.587,-19.274,-18.332,-18.468,
     &-19.914,-23.388,-28.249,-29.170,-20.962,-5.285,
     &8.171,11.659,12.967,13.985,14.165,13.672,
     &14.366,15.807,16.950,17.361,16.388,14.347,
     &12.510,11.585,12.007,12.777,13.289,12.994,
     &12.429,12.097,11.885,12.008,11.915,-999.000,
     &11.703,-999.000,14.458,-999.000,18.793,21.605,
     &20.602,15.707,6.005,-999.000,-2.565,-999.000,
     &-9.482,-999.000,-13.523,-999.000,-15.818,-999.000,
     &-18.021,-20.007,-20.409,-19.968,-20.290
     &/
       data (fjrocvz(it, 3,3, 5), it=1,59) /
     &-19.178,-18.327,-17.840,-16.990,-16.735,-16.950,
     &-18.388,-20.462,-24.071,-27.541,-20.339,-1.233,
     &16.249,18.187,16.709,15.478,14.454,14.139,
     &14.921,15.521,14.817,13.536,11.985,11.125,
     &10.742,10.308,11.073,12.150,13.271,14.493,
     &14.468,13.981,13.785,14.026,14.142,-999.000,
     &14.823,-999.000,15.574,-999.000,18.002,18.285,
     &16.152,13.809,5.808,-999.000,-4.093,-999.000,
     &-11.129,-999.000,-15.883,-999.000,-18.701,-999.000,
     &-21.311,-23.062,-22.940,-21.040,-19.178
     &/
       data (fjrocvz(it, 4,3, 5), it=1,59) /
     &-21.143,-19.270,-18.661,-18.717,-18.218,-17.494,
     &-17.221,-18.043,-21.309,-23.670,-16.187,3.895,
     &18.897,20.931,19.559,18.658,17.289,15.198,
     &13.549,12.408,11.456,10.545,10.225,10.501,
     &10.359,10.482,11.065,12.272,13.712,14.529,
     &14.970,15.089,14.602,14.667,16.248,-999.000,
     &16.243,-999.000,15.593,-999.000,17.312,18.600,
     &16.989,9.600,3.176,-999.000,-5.830,-999.000,
     &-13.247,-999.000,-18.321,-999.000,-21.794,-999.000,
     &-23.867,-25.050,-24.928,-23.195,-21.143
     &/
       data (fjrocvz(it, 5,3, 5), it=1,59) /
     &-23.058,-22.045,-21.399,-19.680,-17.747,-16.612,
     &-16.296,-17.323,-18.757,-18.793,-12.587,2.012,
     &17.042,21.672,21.428,20.704,19.199,17.228,
     &15.491,14.383,14.321,14.242,14.028,13.111,
     &11.822,10.737,10.248,10.738,11.416,12.249,
     &13.283,14.097,14.215,14.730,15.394,-999.000,
     &15.558,-999.000,16.445,18.249,19.242,16.528,
     &12.610,6.468,0.531,-999.000,-8.165,-999.000,
     &-15.124,-999.000,-20.046,-999.000,-23.298,-999.000,
     &-25.775,-27.155,-26.470,-24.532,-23.058
     &/
       data (fjrocvz(it, 6,3, 5), it=1,59) /
     &-25.212,-25.484,-25.408,-23.052,-19.353,-16.483,
     &-16.130,-17.140,-17.003,-15.350,-10.813,-0.282,
     &13.873,18.539,18.362,17.760,17.418,16.945,
     &16.516,17.537,18.784,18.921,17.957,16.556,
     &14.827,12.929,11.604,10.585,10.727,11.283,
     &11.958,12.495,13.222,13.630,13.956,-999.000,
     &15.721,-999.000,16.125,17.416,15.594,11.060,
     &5.383,-999.000,-3.544,-999.000,-11.290,-999.000,
     &-18.396,-999.000,-23.409,-999.000,-26.661,-999.000,
     &-29.286,-30.490,-28.955,-26.723,-25.212
     &/
       data (fjrocvz(it, 7,3, 5), it=1,59) /
     &-28.188,-26.812,-26.500,-25.694,-23.026,-20.917,
     &-20.061,-19.123,-18.069,-14.758,-10.383,-2.972,
     &4.917,9.498,10.943,11.978,12.820,13.481,
     &14.698,16.677,18.892,20.109,20.353,20.047,
     &19.344,18.517,16.852,14.890,13.821,13.129,
     &12.799,12.299,12.058,11.834,11.932,-999.000,
     &12.078,-999.000,11.517,-999.000,6.717,-999.000,
     &0.341,-999.000,-5.785,-999.000,-12.578,-999.000,
     &-19.772,-999.000,-25.090,-999.000,-28.709,-999.000,
     &-31.310,-32.620,-31.977,-30.387,-28.188
     &/
       data (fjrocvz(it, 8,3, 5), it=1,59) /
     &-25.136,-24.048,-25.157,-25.749,-24.718,-21.254,
     &-16.740,-13.886,-12.462,-10.902,-7.822,-3.847,
     &-1.091,1.125,2.568,4.318,6.182,7.598,
     &9.111,10.976,14.064,17.461,19.331,19.827,
     &19.861,18.951,17.254,14.975,13.117,12.527,
     &12.000,11.178,10.068,8.617,7.371,-999.000,
     &6.716,-999.000,5.634,-999.000,2.567,-999.000,
     &-0.747,-999.000,-4.036,-999.000,-9.227,-999.000,
     &-15.404,-999.000,-20.376,-999.000,-25.283,-999.000,
     &-28.765,-30.228,-29.226,-27.162,-25.136
     &/
       data (fjrocvz(it, 9,3, 5), it=1,59) /
     &-18.611,-18.815,-19.143,-19.668,-19.528,-16.761,
     &-12.918,-10.798,-10.407,-10.855,-10.024,-7.630,
     &-7.052,-3.127,1.401,5.687,9.871,10.761,
     &11.586,12.404,13.334,12.962,14.251,15.372,
     &16.191,16.043,14.223,11.566,9.895,8.758,
     &7.505,6.202,5.117,4.527,4.150,-999.000,
     &3.085,-999.000,1.968,-999.000,2.021,-999.000,
     &2.067,-999.000,1.877,-999.000,-2.107,-999.000,
     &-6.434,-999.000,-10.837,-999.000,-13.658,-999.000,
     &-16.124,-17.974,-18.326,-18.442,-18.611
     &/
       data (fjrocvz(it,10,3, 5), it=1,59) /
     &-15.572,-16.019,-15.195,-14.141,-14.510,-15.340,
     &-14.847,-13.593,-12.947,-13.021,-12.814,-13.892,
     &-15.810,-8.110,0.443,6.274,12.112,13.053,
     &13.955,14.619,15.403,15.656,15.554,15.466,
     &15.419,15.484,13.892,11.769,9.686,7.471,
     &5.251,2.732,0.998,0.152,-0.919,-999.000,
     &-2.264,-999.000,-1.764,-999.000,-0.514,-999.000,
     &2.731,-999.000,4.136,-999.000,3.344,-999.000,
     &0.136,-999.000,-3.048,-999.000,-5.446,-999.000,
     &-8.431,-11.390,-13.342,-14.531,-15.572
     &/
       data (fjrocvz(it,11,3, 5), it=1,59) /
     &-19.030,-20.386,-20.507,-19.674,-19.115,-18.338,
     &-16.696,-16.316,-19.308,-18.104,-17.799,-11.763,
     &-15.042,-9.161,-3.057,3.409,9.937,12.081,
     &14.077,15.663,18.726,20.819,20.170,19.368,
     &18.825,18.023,16.935,15.502,13.119,10.300,
     &7.225,4.528,2.837,0.767,-1.248,-999.000,
     &-2.257,-999.000,-1.888,-999.000,0.363,-999.000,
     &7.060,9.706,9.409,6.418,4.089,-999.000,
     &0.723,-999.000,-1.628,-999.000,-3.664,-999.000,
     &-7.099,-12.139,-15.452,-17.288,-19.030
     &/
       data (fjrocvz(it,12,3, 5), it=1,59) /
     &-20.180,-22.130,-24.383,-25.997,-25.505,-23.283,
     &-22.853,-26.862,-31.834,-29.563,-23.893,-14.241,
     &-13.156,-9.887,-6.480,-0.459,5.637,9.336,
     &12.606,16.786,21.078,22.943,22.953,22.336,
     &21.489,20.194,18.739,17.686,15.640,12.604,
     &9.731,8.107,6.597,4.941,2.937,-999.000,
     &1.135,-999.000,2.447,-999.000,9.070,14.355,
     &18.561,19.306,16.405,-999.000,9.729,-999.000,
     &3.821,-999.000,0.262,-999.000,-2.562,-999.000,
     &-6.084,-11.301,-15.789,-18.242,-20.180
     &/
       data (fjrocvz(it,13,3, 5), it=1,59) /
     &-18.966,-20.824,-21.669,-22.763,-24.914,-26.664,
     &-29.116,-31.921,-34.281,-32.287,-25.083,-17.208,
     &-13.138,-11.627,-8.394,-4.751,-1.191,5.428,
     &13.623,19.515,23.335,25.349,25.480,24.173,
     &22.773,21.598,20.118,18.625,16.652,14.291,
     &12.252,10.016,8.273,7.204,5.870,-999.000,
     &5.739,-999.000,8.161,-999.000,13.285,18.866,
     &25.543,27.558,21.431,-999.000,13.296,-999.000,
     &6.422,-999.000,1.759,-999.000,-1.797,-999.000,
     &-5.285,-8.799,-12.510,-16.133,-18.966
     &/
       data (fjrocvz(it,14,3, 5), it=1,59) /
     &-19.175,-21.437,-21.639,-22.782,-25.835,-29.637,
     &-32.050,-34.319,-37.487,-36.037,-27.367,-16.293,
     &-11.079,-9.966,-8.196,-5.291,-2.248,3.767,
     &12.993,18.778,23.157,25.390,25.511,23.729,
     &22.262,21.513,20.670,19.504,18.254,16.500,
     &14.395,12.016,10.401,9.095,8.543,-999.000,
     &8.934,-999.000,11.341,-999.000,16.443,20.906,
     &27.578,29.980,26.858,19.287,13.937,-999.000,
     &6.195,-999.000,1.532,-999.000,-1.385,-999.000,
     &-4.604,-7.843,-11.261,-15.615,-19.175
     &/
       data (fjrocvz(it,15,3, 5), it=1,59) /
     &-18.781,-20.611,-21.290,-22.109,-24.610,-28.229,
     &-31.193,-34.417,-38.143,-38.833,-30.579,-18.879,
     &-11.853,-10.618,-9.692,-7.185,-3.429,2.411,
     &10.239,15.890,19.910,22.039,21.416,19.292,
     &17.427,16.230,15.518,14.371,12.839,11.626,
     &10.480,9.272,7.627,6.703,6.484,-999.000,
     &6.717,-999.000,9.138,-999.000,13.606,17.879,
     &21.939,22.597,18.913,13.412,9.152,-999.000,
     &2.142,-999.000,-1.658,-999.000,-3.762,-999.000,
     &-6.702,-9.616,-12.496,-15.655,-18.781
     &/
       data (fjrocvz(it,16,3, 5), it=1,59) /
     &-16.625,-18.005,-18.232,-18.941,-21.834,-26.170,
     &-30.619,-34.019,-39.100,-41.637,-37.203,-24.421,
     &-13.868,-10.554,-9.577,-7.213,-3.214,2.178,
     &9.046,14.296,17.689,18.774,17.373,15.599,
     &13.809,12.054,10.189,7.672,5.224,3.831,
     &3.218,2.326,1.129,-0.130,-0.455,-999.000,
     &0.684,-999.000,3.170,-999.000,6.590,10.256,
     &11.880,9.998,7.256,-999.000,-0.097,-999.000,
     &-5.747,-999.000,-8.476,-999.000,-9.389,-999.000,
     &-10.225,-11.818,-13.130,-14.723,-16.625
     &/
       data (fjrocvz(it,17,3, 5), it=1,59) /
     &-17.735,-18.651,-18.219,-18.387,-20.652,-24.026,
     &-28.572,-34.937,-42.099,-44.489,-39.454,-24.003,
     &-11.675,-8.804,-6.859,-3.651,0.383,5.043,
     &11.186,16.518,19.681,20.602,19.547,17.172,
     &14.651,12.447,10.192,7.151,4.184,1.966,
     &-0.119,-1.258,-2.220,-3.509,-4.284,-999.000,
     &-4.617,-999.000,-2.799,-999.000,1.043,6.082,
     &7.806,5.080,1.419,-999.000,-4.137,-999.000,
     &-10.437,-999.000,-14.112,-999.000,-14.807,-999.000,
     &-15.100,-15.983,-16.483,-16.818,-17.735
     &/
       data (fjrocvz(it,18,3, 5), it=1,59) /
     &-25.343,-24.816,-23.422,-22.266,-22.836,-24.657,
     &-27.704,-34.284,-43.176,-44.912,-38.239,-16.449,
     &-6.750,-5.221,-3.166,0.313,4.614,10.145,
     &16.422,21.352,23.694,24.521,23.896,21.947,
     &18.973,15.771,13.122,9.991,7.070,4.097,
     &1.464,-0.308,-1.851,-3.356,-4.939,-999.000,
     &-5.831,-999.000,-3.985,-999.000,0.197,3.065,
     &2.039,-1.361,-4.506,-999.000,-10.052,-999.000,
     &-15.881,-999.000,-18.992,-999.000,-19.861,-999.000,
     &-20.534,-22.071,-23.317,-24.314,-25.343
     &/
       data (fjrocvz(it,19,3, 5), it=1,59) /
     &-26.994,-27.000,-25.521,-24.113,-24.356,-26.101,
     &-27.082,-29.624,-35.955,-36.238,-30.312,-13.448,
     &-4.359,-2.198,-0.362,2.690,7.471,13.215,
     &19.514,24.202,26.994,27.427,26.551,24.903,
     &22.435,19.583,16.457,13.393,10.433,7.391,
     &4.989,3.480,2.149,-0.196,-2.001,-999.000,
     &-2.063,-999.000,-1.157,-999.000,2.799,4.053,
     &2.010,-1.398,-6.165,-999.000,-11.121,-999.000,
     &-16.156,-999.000,-19.375,-999.000,-20.689,-999.000,
     &-21.437,-22.741,-24.255,-25.857,-26.994
     &/
       data (fjrocvz(it,20,3, 5), it=1,59) /
     &-24.286,-24.868,-23.853,-22.590,-23.387,-26.070,
     &-28.634,-31.041,-32.547,-29.662,-23.060,-13.087,
     &-5.105,-1.278,0.576,2.824,6.911,12.999,
     &19.675,25.755,28.993,29.181,27.683,25.369,
     &23.004,20.914,19.386,17.927,15.551,12.604,
     &9.745,7.786,5.507,3.457,2.378,-999.000,
     &1.810,-999.000,2.983,-999.000,6.406,8.389,
     &7.257,3.977,-1.154,-999.000,-6.760,-999.000,
     &-11.195,-999.000,-14.329,-999.000,-16.150,-999.000,
     &-17.958,-19.770,-21.175,-22.582,-24.286
     &/
       data (fjrocvz(it,21,3, 5), it=1,59) /
     &-20.808,-21.785,-21.001,-19.965,-20.946,-24.523,
     &-27.411,-29.189,-32.235,-34.202,-30.983,-20.470,
     &-8.718,-3.534,-1.266,0.720,3.800,8.543,
     &14.748,20.260,23.523,24.061,23.178,22.146,
     &20.723,19.322,18.454,17.048,15.074,12.454,
     &10.207,8.127,6.434,5.411,4.591,-999.000,
     &4.475,-999.000,4.587,-999.000,5.420,6.872,
     &8.774,7.313,3.229,-999.000,-1.780,-999.000,
     &-6.068,-999.000,-9.558,-999.000,-12.076,-999.000,
     &-14.361,-16.249,-17.575,-18.995,-20.808
     &/
       data (fjrocvz(it,22,3, 5), it=1,59) /
     &-18.713,-18.392,-17.147,-17.189,-19.297,-22.509,
     &-25.057,-28.618,-34.675,-38.360,-35.753,-24.987,
     &-12.833,-5.216,-3.144,-1.383,1.030,4.267,
     &8.730,12.865,16.098,16.556,15.825,15.577,
     &15.229,14.324,13.313,11.677,9.540,7.594,
     &5.937,4.983,4.296,4.305,5.276,-999.000,
     &5.694,-999.000,5.482,-999.000,7.031,8.012,
     &11.794,11.265,5.055,-999.000,-0.439,-999.000,
     &-5.082,-999.000,-8.513,-999.000,-10.927,-999.000,
     &-13.095,-15.114,-16.826,-18.180,-18.713
     &/
       data (fjrocvz(it,23,3, 5), it=1,59) /
     &-20.212,-19.335,-17.741,-17.001,-19.003,-22.495,
     &-25.967,-29.253,-32.827,-34.024,-29.907,-19.667,
     &-8.010,-2.843,-2.265,-1.241,0.475,2.493,
     &5.482,9.752,13.763,15.222,14.136,12.282,
     &10.576,8.878,7.814,6.708,5.013,3.404,
     &2.970,2.740,2.159,2.366,3.705,-999.000,
     &4.273,-999.000,6.021,-999.000,9.847,13.668,
     &16.862,16.124,9.622,-999.000,0.285,-999.000,
     &-5.250,-999.000,-8.369,-999.000,-11.072,-999.000,
     &-13.874,-16.339,-18.095,-19.490,-20.212
     &/
       data (fjrocvz(it,24,3, 5), it=1,59) /
     &-22.687,-23.129,-21.785,-19.833,-19.468,-21.055,
     &-23.188,-25.980,-30.666,-31.243,-26.548,-15.314,
     &-4.234,0.131,1.774,3.217,4.228,4.849,
     &6.638,9.718,13.074,13.958,13.205,11.633,
     &8.722,6.483,5.464,5.930,5.590,5.039,
     &4.947,4.676,4.374,4.022,4.370,-999.000,
     &6.438,-999.000,9.357,-999.000,14.217,18.284,
     &18.493,16.244,12.111,-999.000,-0.439,-999.000,
     &-7.230,-999.000,-10.014,-999.000,-12.819,-999.000,
     &-16.387,-19.131,-20.602,-21.637,-22.687
     &/
       data (fjrocvz(it,25,3, 5), it=1,59) /
     &-22.481,-23.659,-23.412,-22.464,-21.032,-19.887,
     &-19.840,-23.108,-29.978,-31.573,-24.550,-10.749,
     &0.890,5.213,8.017,10.178,10.283,9.026,
     &9.228,11.685,14.738,15.652,15.077,12.655,
     &9.307,7.797,8.392,9.441,9.498,8.680,
     &8.007,7.960,7.837,7.152,7.423,-999.000,
     &8.919,-999.000,11.124,-999.000,16.697,19.665,
     &19.980,16.450,7.382,-999.000,-2.132,-999.000,
     &-9.105,-999.000,-12.271,-999.000,-14.365,-999.000,
     &-16.890,-19.141,-20.550,-21.263,-22.481
     &/
       data (fjrocvz(it, 1,4, 5), it=1,59) /
     &-17.153,-16.303,-14.792,-13.730,-14.541,-15.260,
     &-15.817,-16.651,-17.138,-16.501,-13.226,-6.473,
     &-7.809,-3.650,1.642,7.622,12.987,21.741,
     &28.703,31.459,31.028,29.615,27.079,22.867,
     &18.564,14.575,9.198,5.213,4.598,5.964,
     &6.567,6.836,7.277,8.855,11.389,-999.000,
     &15.024,-999.000,18.739,21.385,21.549,16.637,
     &9.470,-999.000,-0.601,-999.000,-8.367,-999.000,
     &-13.770,-999.000,-15.737,-999.000,-19.763,-999.000,
     &-20.764,-20.569,-20.203,-18.816,-17.153
     &/
       data (fjrocvz(it, 2,4, 5), it=1,59) /
     &-15.419,-14.741,-13.854,-13.428,-12.773,-11.931,
     &-12.020,-13.521,-14.554,-13.779,-8.370,-0.413,
     &5.274,8.258,12.479,14.689,18.494,23.727,
     &29.110,32.792,33.545,32.422,29.540,25.165,
     &20.791,17.489,13.649,9.913,7.802,7.793,
     &8.900,9.689,10.450,11.907,13.386,-999.000,
     &15.675,-999.000,19.982,22.375,23.792,22.918,
     &16.822,-999.000,-0.633,-999.000,-10.869,-999.000,
     &-11.900,-999.000,-16.841,-999.000,-20.894,-999.000,
     &-20.747,-20.547,-18.999,-16.603,-15.419
     &/
       data (fjrocvz(it, 3,4, 5), it=1,59) /
     &-15.634,-14.437,-13.776,-12.647,-11.648,-11.570,
     &-11.060,-11.070,-11.446,-10.789,-6.296,0.177,
     &6.127,9.586,11.872,14.825,18.543,21.808,
     &25.203,27.877,29.199,28.256,25.231,21.380,
     &17.881,15.395,12.995,10.620,8.659,8.339,
     &9.297,10.213,11.464,12.217,13.879,-999.000,
     &16.956,-999.000,19.824,24.796,28.272,26.011,
     &15.908,5.999,-1.154,-10.455,-15.020,-14.764,
     &-15.311,-999.000,-20.425,-999.000,-22.374,-999.000,
     &-22.419,-21.029,-19.248,-17.314,-15.634
     &/
       data (fjrocvz(it, 4,4, 5), it=1,59) /
     &-16.963,-15.829,-14.247,-12.670,-12.308,-12.186,
     &-11.379,-9.881,-8.993,-8.223,-5.184,0.286,
     &6.274,8.798,10.005,12.619,16.245,20.514,
     &24.017,26.052,25.763,23.693,20.768,17.379,
     &14.322,11.920,9.822,8.456,7.961,7.925,
     &8.699,10.104,11.160,13.009,16.215,-999.000,
     &18.065,-999.000,21.375,26.752,28.242,26.657,
     &20.802,11.419,-1.652,-14.502,-18.457,-999.000,
     &-17.838,-999.000,-21.350,-999.000,-24.259,-999.000,
     &-23.320,-21.063,-19.603,-17.938,-16.963
     &/
       data (fjrocvz(it, 5,4, 5), it=1,59) /
     &-17.657,-16.389,-14.133,-12.495,-12.348,-12.941,
     &-12.731,-11.492,-10.808,-9.898,-6.877,-0.649,
     &5.362,8.348,11.009,13.134,16.784,21.721,
     &25.932,28.329,28.180,26.092,22.569,18.802,
     &15.416,13.072,11.336,10.417,9.528,8.991,
     &9.617,10.956,12.609,14.251,15.498,-999.000,
     &19.316,-999.000,23.282,27.113,29.916,31.813,
     &26.741,12.240,-1.714,-11.244,-17.715,-20.719,
     &-23.294,-26.166,-27.162,-26.905,-25.350,-999.000,
     &-23.516,-21.269,-19.586,-18.293,-17.657
     &/
       data (fjrocvz(it, 6,4, 5), it=1,59) /
     &-18.486,-17.613,-16.140,-14.748,-14.429,-14.141,
     &-12.863,-11.997,-11.383,-9.601,-7.424,-6.028,
     &-2.750,3.068,9.451,14.364,18.000,22.740,
     &26.905,29.305,30.238,28.960,26.361,22.840,
     &19.768,17.598,15.595,13.612,12.086,10.706,
     &10.115,11.714,13.600,14.210,15.814,-999.000,
     &20.056,-999.000,24.286,30.613,35.887,37.426,
     &32.415,19.814,3.649,-8.024,-16.277,-22.031,
     &-24.463,-999.000,-27.026,-999.000,-26.889,-999.000,
     &-25.004,-21.801,-20.160,-19.383,-18.486
     &/
       data (fjrocvz(it, 7,4, 5), it=1,59) /
     &-20.217,-19.738,-18.656,-17.679,-17.009,-15.194,
     &-13.353,-12.126,-10.519,-9.542,-7.429,-5.083,
     &0.042,2.956,6.839,10.224,14.092,19.476,
     &24.464,28.257,29.428,29.313,27.973,25.573,
     &23.111,20.563,18.143,16.137,13.803,11.353,
     &10.538,11.298,12.527,13.850,14.936,-999.000,
     &17.825,-999.000,22.683,30.115,36.559,40.607,
     &32.697,16.757,2.435,-7.033,-15.729,-23.616,
     &-28.065,-29.523,-29.472,-29.048,-28.147,-999.000,
     &-25.822,-23.746,-22.264,-20.748,-20.217
     &/
       data (fjrocvz(it, 8,4, 5), it=1,59) /
     &-20.132,-18.748,-17.835,-17.372,-16.673,-15.834,
     &-15.026,-14.175,-12.508,-14.503,-16.413,-18.194,
     &-13.089,-5.704,-0.065,5.485,8.857,12.596,
     &18.444,22.960,25.363,25.829,25.225,24.011,
     &22.302,20.394,18.086,15.859,13.110,10.680,
     &9.606,9.851,11.164,12.402,13.356,-999.000,
     &16.266,-999.000,21.757,26.706,32.644,33.186,
     &24.602,13.373,3.228,-6.962,-13.948,-20.947,
     &-25.234,-26.154,-24.470,-999.000,-24.842,-999.000,
     &-24.171,-24.021,-23.079,-21.482,-20.132
     &/
       data (fjrocvz(it, 9,4, 5), it=1,59) /
     &-17.515,-15.477,-13.675,-12.573,-12.572,-13.270,
     &-14.865,-16.244,-17.117,-19.116,-21.835,-23.888,
     &-18.820,-8.802,-0.806,4.474,6.772,10.397,
     &15.310,19.012,20.879,21.580,21.407,20.052,
     &17.782,15.661,13.754,11.761,9.583,8.344,
     &8.285,8.930,10.182,11.082,12.325,-999.000,
     &14.391,-999.000,20.607,26.243,31.258,33.957,
     &32.226,21.211,9.927,-0.424,-7.321,-999.000,
     &-12.995,-999.000,-16.555,-999.000,-19.482,-999.000,
     &-20.174,-20.358,-20.657,-19.658,-17.515
     &/
       data (fjrocvz(it,10,4, 5), it=1,59) /
     &-17.725,-15.573,-13.333,-12.172,-11.631,-12.949,
     &-16.140,-18.888,-20.518,-21.443,-22.150,-21.962,
     &-17.878,-6.970,1.135,6.487,8.944,12.957,
     &18.061,21.340,21.901,21.549,20.321,18.692,
     &16.595,13.985,11.640,9.696,8.405,7.943,
     &8.061,8.461,8.608,8.959,10.501,-999.000,
     &13.854,-999.000,21.201,27.664,34.940,38.878,
     &36.640,25.239,13.908,-999.000,2.633,-999.000,
     &-6.290,-999.000,-12.371,-999.000,-14.377,-999.000,
     &-16.167,-17.251,-18.377,-18.412,-17.725
     &/
       data (fjrocvz(it,11,4, 5), it=1,59) /
     &-20.062,-19.422,-19.055,-19.054,-19.057,-19.797,
     &-20.710,-22.341,-24.405,-26.426,-25.763,-21.995,
     &-16.536,-6.462,1.291,7.519,12.002,15.359,
     &19.725,23.390,24.868,24.773,23.706,22.121,
     &20.410,18.005,15.056,12.940,11.360,10.452,
     &9.833,9.114,8.635,8.844,10.850,-999.000,
     &15.500,-999.000,22.187,28.677,36.840,37.232,
     &32.303,22.721,13.033,-999.000,-0.256,-999.000,
     &-9.545,-999.000,-14.895,-999.000,-16.166,-999.000,
     &-18.616,-18.538,-19.501,-20.655,-20.062
     &/
       data (fjrocvz(it,12,4, 5), it=1,59) /
     &-22.004,-22.776,-22.706,-22.643,-22.935,-22.760,
     &-22.799,-24.560,-27.617,-29.614,-30.279,-26.920,
     &-22.531,-12.354,-2.353,3.809,8.984,13.909,
     &20.509,25.381,27.349,27.280,26.243,25.074,
     &23.575,20.973,17.836,15.315,13.896,13.143,
     &12.206,10.940,10.018,10.157,12.511,-999.000,
     &16.811,-999.000,23.200,29.302,36.746,40.682,
     &34.546,18.561,6.819,-0.837,-7.402,-999.000,
     &-11.492,-999.000,-15.050,-999.000,-18.750,-999.000,
     &-20.964,-20.995,-21.317,-21.814,-22.004
     &/
       data (fjrocvz(it,13,4, 5), it=1,59) /
     &-21.583,-21.132,-20.923,-20.929,-20.658,-21.440,
     &-23.051,-24.437,-25.992,-26.506,-26.843,-25.275,
     &-22.233,-13.014,-3.916,3.021,8.038,13.582,
     &22.697,28.348,29.286,28.952,27.952,26.899,
     &25.212,22.226,19.185,17.100,16.366,16.066,
     &15.311,13.983,12.524,12.439,14.390,-999.000,
     &17.733,-999.000,26.172,31.951,39.953,44.372,
     &35.639,19.018,3.466,-4.723,-8.487,-10.366,
     &-11.271,-999.000,-14.827,-999.000,-17.758,-999.000,
     &-20.071,-22.127,-22.085,-21.567,-21.583
     &/
       data (fjrocvz(it,14,4, 5), it=1,59) /
     &-20.083,-19.665,-19.336,-18.766,-18.765,-20.380,
     &-23.149,-26.710,-29.875,-29.522,-26.561,-22.494,
     &-18.493,-14.604,-7.479,0.519,6.546,12.755,
     &21.953,27.469,29.065,28.869,27.730,26.358,
     &24.206,21.829,19.525,18.035,17.619,17.285,
     &16.248,14.946,13.557,13.545,15.710,-999.000,
     &20.139,-999.000,28.355,36.100,40.946,40.650,
     &32.483,18.197,7.173,-999.000,-6.366,-999.000,
     &-14.013,-999.000,-17.296,-999.000,-19.940,-999.000,
     &-22.652,-23.449,-22.513,-20.965,-20.083
     &/
       data (fjrocvz(it,15,4, 5), it=1,59) /
     &-18.246,-17.685,-17.101,-16.618,-16.583,-18.097,
     &-21.974,-27.954,-31.997,-33.278,-29.860,-22.658,
     &-17.787,-12.851,-7.460,0.022,6.510,12.372,
     &19.771,24.862,26.778,26.319,24.525,21.975,
     &19.412,17.417,15.329,13.910,13.277,12.617,
     &11.799,10.875,10.493,11.399,14.325,-999.000,
     &18.131,-999.000,24.972,31.406,35.674,37.680,
     &30.829,16.313,5.240,-999.000,-8.924,-999.000,
     &-15.882,-999.000,-20.496,-999.000,-23.752,-999.000,
     &-25.567,-25.843,-23.450,-19.966,-18.246
     &/
       data (fjrocvz(it,16,4, 5), it=1,59) /
     &-18.313,-17.022,-17.140,-17.311,-17.937,-19.845,
     &-23.471,-28.063,-31.471,-34.364,-33.290,-25.792,
     &-16.343,-11.261,-5.534,0.873,6.788,12.101,
     &19.357,24.293,25.978,25.799,24.308,21.034,
     &17.018,13.412,10.741,9.114,8.520,8.276,
     &7.543,6.506,6.051,6.835,9.177,-999.000,
     &13.066,-999.000,19.602,25.348,30.130,31.786,
     &24.330,9.314,1.763,-999.000,-11.333,-999.000,
     &-18.690,-999.000,-23.490,-999.000,-26.350,-999.000,
     &-27.609,-27.169,-24.681,-21.130,-18.313
     &/
       data (fjrocvz(it,17,4, 5), it=1,59) /
     &-21.297,-20.364,-19.582,-19.051,-19.535,-20.964,
     &-23.959,-28.402,-31.925,-33.597,-31.922,-23.873,
     &-15.047,-9.398,-2.900,2.455,7.412,13.029,
     &20.784,26.056,28.343,28.761,27.548,24.631,
     &20.083,15.625,11.728,8.911,7.972,7.380,
     &6.318,4.903,3.397,3.030,5.254,-999.000,
     &10.188,-999.000,17.722,21.995,25.784,25.649,
     &19.597,7.099,-3.404,-12.301,-19.078,-22.842,
     &-24.667,-999.000,-25.620,-999.000,-27.499,-999.000,
     &-27.969,-26.818,-25.214,-23.100,-21.297
     &/
       data (fjrocvz(it,18,4, 5), it=1,59) /
     &-22.301,-20.705,-19.380,-18.915,-18.494,-19.114,
     &-22.646,-27.156,-29.799,-31.429,-28.962,-21.750,
     &-12.913,-7.397,-2.550,2.868,8.203,15.358,
     &22.512,28.228,31.909,32.465,30.183,26.212,
     &22.286,18.448,14.177,10.389,8.454,7.009,
     &5.726,4.115,2.793,2.604,4.798,-999.000,
     &8.970,12.364,17.146,21.324,23.675,20.785,
     &13.138,5.063,-4.979,-999.000,-19.323,-999.000,
     &-26.284,-999.000,-28.987,-999.000,-29.258,-999.000,
     &-28.068,-27.536,-25.942,-23.969,-22.301
     &/
       data (fjrocvz(it,19,4, 5), it=1,59) /
     &-20.771,-18.925,-18.639,-17.944,-17.055,-17.639,
     &-20.254,-22.869,-25.026,-26.745,-24.184,-17.453,
     &-9.555,-5.724,-0.900,4.608,9.750,17.263,
     &25.345,31.732,35.544,35.851,33.207,29.187,
     &24.809,20.402,16.068,12.195,10.619,9.741,
     &8.195,6.337,4.978,4.438,6.197,-999.000,
     &11.013,-999.000,18.551,22.221,23.155,18.476,
     &12.928,4.228,-6.119,-15.862,-23.008,-26.786,
     &-27.942,-999.000,-29.256,-999.000,-28.425,-999.000,
     &-27.161,-26.703,-25.353,-23.136,-20.771
     &/
       data (fjrocvz(it,20,4, 5), it=1,59) /
     &-22.785,-21.063,-19.250,-17.381,-15.907,-15.882,
     &-17.883,-20.850,-22.351,-23.098,-20.600,-11.606,
     &-3.653,-2.099,2.201,6.406,10.652,18.297,
     &26.075,33.300,37.329,37.963,35.965,32.445,
     &28.483,24.442,19.913,16.070,14.113,12.580,
     &10.724,8.748,7.054,6.508,8.738,-999.000,
     &13.061,-999.000,18.909,22.257,25.897,25.274,
     &18.420,4.738,-5.606,-12.864,-19.218,-24.141,
     &-26.902,-999.000,-27.745,-999.000,-28.401,-999.000,
     &-27.777,-27.358,-26.451,-24.632,-22.785
     &/
       data (fjrocvz(it,21,4, 5), it=1,59) /
     &-23.049,-20.600,-18.679,-17.436,-16.615,-16.143,
     &-17.165,-19.760,-22.365,-23.080,-20.290,-13.035,
     &-7.632,-3.078,1.836,6.906,12.108,18.158,
     &23.949,30.535,34.135,35.352,33.506,29.896,
     &26.219,22.849,18.980,14.713,12.054,10.182,
     &8.387,7.111,6.408,6.443,8.245,-999.000,
     &12.148,-999.000,18.471,22.817,27.176,26.030,
     &17.113,-999.000,-1.165,-999.000,-15.286,-999.000,
     &-22.823,-999.000,-26.146,-999.000,-27.057,-999.000,
     &-26.956,-26.555,-25.900,-24.743,-23.049
     &/
       data (fjrocvz(it,22,4, 5), it=1,59) /
     &-21.958,-20.919,-19.794,-18.978,-18.936,-18.643,
     &-18.839,-21.019,-24.726,-25.629,-22.647,-15.794,
     &-7.738,-4.830,0.600,5.428,10.202,15.934,
     &22.313,28.174,30.878,30.984,28.848,24.839,
     &20.045,15.825,12.153,8.932,6.812,5.391,
     &4.492,4.782,5.471,5.949,7.891,-999.000,
     &12.903,-999.000,18.986,20.969,23.638,20.871,
     &13.898,-999.000,-3.867,-999.000,-16.354,-999.000,
     &-23.881,-999.000,-25.089,-999.000,-25.209,-999.000,
     &-25.605,-25.093,-24.216,-23.102,-21.958
     &/
       data (fjrocvz(it,23,4, 5), it=1,59) /
     &-19.925,-19.563,-18.106,-16.192,-15.702,-16.611,
     &-18.462,-20.796,-22.949,-23.780,-21.266,-15.176,
     &-10.158,-7.335,-1.951,3.271,8.456,13.953,
     &22.022,27.733,29.534,28.436,25.740,21.609,
     &16.879,12.093,8.307,5.570,4.127,3.388,
     &2.945,2.839,4.097,6.225,9.251,-999.000,
     &12.809,-999.000,17.800,20.808,22.096,21.060,
     &15.489,5.454,-3.024,-999.000,-14.876,-999.000,
     &-21.335,-999.000,-23.620,-999.000,-23.963,-999.000,
     &-24.216,-23.559,-22.030,-20.364,-19.925
     &/
       data (fjrocvz(it,24,4, 5), it=1,59) /
     &-17.892,-16.528,-14.842,-12.811,-12.758,-14.554,
     &-16.706,-18.001,-19.092,-19.788,-17.707,-12.877,
     &-6.853,-2.126,1.224,4.393,8.817,16.943,
     &25.069,28.632,29.257,28.030,24.881,20.440,
     &15.946,11.301,7.014,4.036,3.438,4.417,
     &4.693,4.341,5.481,7.388,9.847,-999.000,
     &13.235,-999.000,17.555,20.662,23.067,20.346,
     &12.999,-999.000,0.343,-999.000,-8.681,-999.000,
     &-14.889,-999.000,-19.326,-999.000,-20.380,-999.000,
     &-21.702,-21.489,-20.242,-19.110,-17.892
     &/
       data (fjrocvz(it,25,4, 5), it=1,59) /
     &-17.153,-16.303,-14.792,-13.730,-14.541,-15.260,
     &-15.817,-16.651,-17.138,-16.501,-13.226,-6.473,
     &-7.809,-3.650,1.642,7.622,12.987,21.741,
     &28.703,31.459,31.028,29.615,27.079,22.867,
     &18.564,14.575,9.198,5.213,4.598,5.964,
     &6.567,6.836,7.277,8.855,11.389,-999.000,
     &15.024,-999.000,18.739,21.385,21.549,16.637,
     &9.470,-999.000,-0.601,-999.000,-8.367,-999.000,
     &-13.770,-999.000,-15.737,-999.000,-19.763,-999.000,
     &-20.764,-20.569,-20.203,-18.816,-17.153
     &/
       data (fjrocvz(it, 1,1, 6), it=1,59) /
     &-12.547,-11.973,-11.646,-11.795,-12.206,-13.316,
     &-14.991,-16.802,-19.008,-20.647,-19.584,-13.369,
     &-3.827,3.968,9.944,14.423,18.938,24.042,
     &28.328,31.088,31.257,29.454,27.124,24.152,
     &21.051,17.157,13.247,10.292,7.146,3.870,
     &1.904,1.376,1.492,1.221,1.092,-999.000,
     &0.569,-999.000,1.458,-999.000,3.851,-999.000,
     &2.829,-999.000,0.302,-999.000,-4.922,-999.000,
     &-9.446,-999.000,-12.707,-999.000,-14.677,-999.000,
     &-16.581,-17.226,-16.229,-14.031,-12.547
     &/
       data (fjrocvz(it, 2,1, 6), it=1,59) /
     &-14.647,-13.537,-13.184,-13.126,-14.140,-15.618,
     &-16.253,-17.301,-18.546,-19.535,-15.913,-6.381,
     &2.482,9.003,14.454,20.261,25.126,29.389,
     &33.423,36.048,36.471,34.544,31.331,27.657,
     &23.541,18.922,14.303,10.655,7.325,4.073,
     &1.151,-0.574,-1.345,-1.271,-1.243,-999.000,
     &-0.761,-999.000,1.109,-999.000,1.345,-999.000,
     &0.864,-999.000,-2.239,-999.000,-7.513,-999.000,
     &-12.267,-999.000,-15.885,-999.000,-17.854,-999.000,
     &-18.830,-18.901,-18.080,-16.399,-14.647
     &/
       data (fjrocvz(it, 3,1, 6), it=1,59) /
     &-16.250,-15.683,-15.508,-15.849,-16.740,-16.648,
     &-15.783,-15.860,-15.661,-14.760,-9.059,-0.563,
     &6.600,11.660,16.811,22.034,25.463,28.934,
     &32.700,35.737,36.159,33.789,30.162,26.645,
     &23.487,20.128,16.260,12.287,8.712,5.126,
     &1.947,0.314,-0.540,-0.144,-0.083,-999.000,
     &0.982,-999.000,1.651,-999.000,3.435,4.980,
     &6.041,4.535,0.970,-1.486,-5.973,-999.000,
     &-11.458,-999.000,-14.856,-999.000,-16.991,-999.000,
     &-18.142,-18.589,-18.314,-17.305,-16.250
     &/
       data (fjrocvz(it, 4,1, 6), it=1,59) /
     &-16.265,-17.099,-17.086,-16.298,-15.460,-15.576,
     &-15.599,-15.877,-14.769,-11.807,-4.669,3.529,
     &8.638,12.374,17.611,22.052,25.567,28.933,
     &32.493,34.592,33.797,30.535,26.764,23.886,
     &21.333,17.668,13.831,11.006,9.085,6.743,
     &3.923,1.654,0.491,1.033,2.756,-999.000,
     &3.421,-999.000,5.240,-999.000,6.436,8.759,
     &10.737,7.460,2.517,-999.000,-3.656,-999.000,
     &-9.762,-999.000,-13.352,-999.000,-15.707,-999.000,
     &-17.633,-17.598,-16.621,-15.545,-16.265
     &/
       data (fjrocvz(it, 5,1, 6), it=1,59) /
     &-13.199,-13.970,-14.068,-13.121,-12.632,-13.858,
     &-15.807,-17.060,-16.323,-13.219,-5.578,3.363,
     &7.984,11.591,16.946,21.721,25.230,28.326,
     &31.607,33.650,33.125,30.362,27.426,24.771,
     &21.625,17.234,13.295,11.104,10.055,8.910,
     &6.935,4.294,2.676,2.148,2.954,-999.000,
     &5.034,-999.000,8.709,-999.000,10.236,14.532,
     &14.752,10.648,5.824,-999.000,-2.061,-999.000,
     &-9.052,-999.000,-13.260,-999.000,-15.745,-999.000,
     &-17.184,-16.277,-14.397,-12.907,-13.199
     &/
       data (fjrocvz(it, 6,1, 6), it=1,59) /
     &-11.462,-12.167,-12.871,-13.614,-14.033,-15.075,
     &-17.053,-18.423,-17.853,-14.108,-7.518,0.372,
     &4.748,7.559,11.358,15.342,18.767,22.416,
     &26.644,30.041,30.337,28.772,26.749,24.528,
     &21.689,18.259,15.162,12.716,11.374,10.680,
     &9.925,8.437,5.943,4.272,4.625,-999.000,
     &6.471,-999.000,9.056,-999.000,13.239,-999.000,
     &13.873,-999.000,8.719,-999.000,2.161,-999.000,
     &-4.260,-999.000,-8.906,-999.000,-12.546,-999.000,
     &-14.757,-15.306,-13.932,-12.040,-11.462
     &/
       data (fjrocvz(it, 7,1, 6), it=1,59) /
     &-14.300,-15.019,-16.047,-16.974,-17.597,-18.037,
     &-18.394,-18.388,-17.646,-15.264,-11.423,-7.675,
     &-3.825,1.072,6.197,10.051,13.819,17.315,
     &22.328,25.808,26.645,25.367,23.187,21.104,
     &19.188,16.849,13.931,11.362,9.746,9.557,
     &10.165,9.630,7.748,6.086,6.077,-999.000,
     &6.825,-999.000,9.845,-999.000,15.098,17.723,
     &22.254,21.499,14.222,-999.000,6.424,-999.000,
     &0.041,-999.000,-5.335,-999.000,-9.962,-999.000,
     &-13.395,-15.279,-15.191,-14.766,-14.300
     &/
       data (fjrocvz(it, 8,1, 6), it=1,59) /
     &-17.773,-17.256,-17.052,-16.965,-17.092,-17.345,
     &-17.950,-18.737,-19.419,-18.852,-16.112,-10.905,
     &-5.694,-2.307,0.664,3.494,6.840,11.152,
     &16.666,20.889,22.819,22.095,19.976,17.796,
     &15.650,13.443,11.108,8.755,7.561,7.624,
     &8.968,9.550,9.074,7.890,7.983,-999.000,
     &9.609,-999.000,13.637,-999.000,21.040,28.688,
     &32.413,32.245,24.898,15.050,9.816,-999.000,
     &-1.542,-999.000,-8.617,-999.000,-13.733,-999.000,
     &-16.843,-18.978,-19.334,-18.679,-17.773
     &/
       data (fjrocvz(it, 9,1, 6), it=1,59) /
     &-18.943,-17.814,-16.984,-16.415,-16.099,-16.795,
     &-18.181,-19.645,-20.698,-20.634,-18.511,-14.465,
     &-10.319,-6.133,-1.495,1.852,4.357,7.897,
     &12.940,17.414,19.753,20.062,19.117,17.657,
     &15.638,14.048,12.628,11.254,10.102,9.524,
     &9.931,9.983,9.562,9.175,10.254,10.324,
     &10.712,12.238,16.828,22.627,28.957,37.463,
     &43.431,39.185,23.410,8.437,-0.644,-6.645,
     &-9.166,-999.000,-15.133,-999.000,-18.573,-999.000,
     &-20.941,-22.366,-21.941,-20.503,-18.943
     &/
       data (fjrocvz(it,10,1, 6), it=1,59) /
     &-17.765,-16.485,-15.586,-14.968,-15.355,-16.822,
     &-18.131,-19.082,-20.475,-22.192,-21.146,-16.498,
     &-10.404,-4.482,0.642,4.614,7.435,10.259,
     &13.929,17.295,20.002,21.889,22.487,21.839,
     &19.946,18.321,17.011,15.830,14.365,13.390,
     &12.842,12.292,11.530,11.016,11.060,-999.000,
     &12.233,15.730,22.971,32.077,42.141,48.080,
     &42.065,28.404,12.542,-2.361,-8.284,-12.130,
     &-14.677,-999.000,-18.617,-999.000,-21.501,-999.000,
     &-22.940,-22.795,-21.452,-19.748,-17.765
     &/
       data (fjrocvz(it,11,1, 6), it=1,59) /
     &-16.674,-15.386,-14.979,-15.437,-16.830,-18.671,
     &-20.133,-21.093,-22.179,-22.972,-20.851,-13.845,
     &-6.286,-0.032,5.416,9.362,11.729,13.400,
     &16.049,20.070,24.266,26.915,27.545,26.345,
     &24.495,22.656,21.067,18.912,16.781,15.324,
     &13.803,12.810,12.200,12.090,12.723,13.335,
     &14.831,19.550,26.858,33.872,38.186,37.969,
     &25.427,10.334,-0.499,-9.629,-15.490,-999.000,
     &-17.752,-999.000,-20.711,-999.000,-22.255,-999.000,
     &-22.598,-21.809,-20.192,-18.324,-16.674
     &/
       data (fjrocvz(it,12,1, 6), it=1,59) /
     &-18.381,-17.882,-18.162,-19.482,-20.685,-21.357,
     &-21.825,-22.745,-24.170,-24.460,-22.657,-17.629,
     &-9.861,-2.450,6.619,11.166,13.860,16.000,
     &18.752,22.679,26.070,28.181,28.827,28.410,
     &27.396,25.592,23.113,20.142,17.091,14.719,
     &13.278,12.121,11.664,12.767,15.458,-999.000,
     &19.029,19.675,24.925,31.335,34.572,30.717,
     &20.629,10.657,2.183,-999.000,-12.041,-999.000,
     &-20.249,-999.000,-23.205,-999.000,-23.355,-999.000,
     &-22.528,-21.601,-20.579,-19.314,-18.381
     &/
       data (fjrocvz(it,13,1, 6), it=1,59) /
     &-19.322,-19.753,-20.414,-21.257,-21.402,-20.835,
     &-20.787,-22.147,-23.464,-24.287,-23.172,-18.484,
     &-9.571,-3.165,4.321,10.242,13.889,16.367,
     &19.274,22.713,25.473,26.997,27.162,26.503,
     &25.021,23.171,21.278,18.575,14.885,12.601,
     &12.104,11.956,11.863,11.931,12.196,-999.000,
     &15.265,-999.000,20.698,25.439,28.216,24.411,
     &17.087,-999.000,5.489,-999.000,-8.295,-999.000,
     &-18.104,-999.000,-22.844,-999.000,-23.408,-999.000,
     &-22.785,-21.838,-20.704,-19.612,-19.322
     &/
       data (fjrocvz(it,14,1, 6), it=1,59) /
     &-19.446,-20.748,-21.675,-21.798,-20.954,-20.261,
     &-21.506,-23.287,-23.875,-24.219,-21.204,-17.821,
     &-11.670,-6.455,1.335,8.173,12.591,15.008,
     &17.511,20.771,24.070,25.634,25.455,24.043,
     &22.470,20.996,19.184,16.117,12.731,10.515,
     &10.100,10.065,9.385,7.548,7.089,-999.000,
     &9.433,11.338,16.591,19.967,21.266,20.291,
     &16.704,9.332,4.605,-999.000,-6.369,-999.000,
     &-15.771,-999.000,-21.937,-999.000,-23.714,-999.000,
     &-23.629,-22.489,-20.758,-19.558,-19.446
     &/
       data (fjrocvz(it,15,1, 6), it=1,59) /
     &-20.053,-20.689,-22.008,-22.875,-22.262,-21.192,
     &-21.732,-23.204,-25.161,-27.002,-24.372,-19.269,
     &-11.535,-6.516,-0.276,5.885,10.519,13.548,
     &16.013,17.832,19.636,20.569,19.979,18.537,
     &17.341,16.399,15.165,13.360,10.635,8.218,
     &7.457,7.195,6.318,5.738,6.039,-999.000,
     &8.337,-999.000,14.604,19.699,23.268,23.369,
     &19.046,11.647,6.441,-999.000,-4.393,-999.000,
     &-13.935,-999.000,-21.306,-999.000,-23.922,-999.000,
     &-23.648,-22.777,-21.597,-20.639,-20.053
     &/
       data (fjrocvz(it,16,1, 6), it=1,59) /
     &-22.017,-22.123,-22.834,-23.660,-23.273,-21.884,
     &-21.715,-22.406,-24.572,-29.592,-30.227,-24.817,
     &-10.017,-3.099,1.144,5.945,10.113,12.911,
     &15.361,17.267,19.146,19.349,18.713,17.453,
     &16.170,14.861,13.406,11.196,8.216,6.223,
     &6.070,6.547,6.746,6.989,7.823,-999.000,
     &11.003,-999.000,17.408,22.323,25.343,23.475,
     &18.078,-999.000,5.976,-999.000,-3.786,-999.000,
     &-13.666,-999.000,-21.079,-999.000,-23.706,-999.000,
     &-24.124,-23.532,-22.881,-22.516,-22.017
     &/
       data (fjrocvz(it,17,1, 6), it=1,59) /
     &-22.476,-22.315,-22.497,-23.324,-23.875,-23.655,
     &-23.718,-22.412,-22.396,-26.676,-29.220,-24.793,
     &-9.796,0.847,4.428,7.533,10.603,13.108,
     &15.512,17.589,19.205,20.216,20.243,19.198,
     &18.021,16.450,14.698,12.364,9.255,7.086,
     &6.737,7.367,7.402,7.182,8.452,-999.000,
     &10.777,13.762,16.565,19.936,22.053,21.776,
     &16.073,-999.000,1.182,-999.000,-9.276,-999.000,
     &-18.251,-999.000,-24.090,-999.000,-26.307,-999.000,
     &-25.756,-24.041,-22.932,-22.711,-22.476
     &/
       data (fjrocvz(it,18,1, 6), it=1,59) /
     &-22.015,-21.718,-21.910,-23.047,-24.619,-24.993,
     &-25.336,-24.846,-23.695,-24.255,-23.395,-18.858,
     &-7.051,3.562,7.858,10.687,13.736,16.378,
     &18.707,19.631,20.530,21.684,21.653,20.384,
     &18.636,16.869,14.874,12.962,10.588,8.719,
     &7.850,7.958,8.179,8.514,9.275,-999.000,
     &11.728,-999.000,15.347,20.180,21.948,19.199,
     &11.590,2.694,-4.476,-999.000,-13.876,-999.000,
     &-21.302,-999.000,-26.905,-999.000,-28.475,-999.000,
     &-26.974,-24.641,-23.397,-22.731,-22.015
     &/
       data (fjrocvz(it,19,1, 6), it=1,59) /
     &-22.856,-21.749,-20.852,-21.047,-21.510,-21.083,
     &-20.414,-20.748,-20.992,-22.861,-22.564,-18.714,
     &-8.710,2.459,9.005,13.330,16.797,19.990,
     &22.623,23.776,24.674,25.438,25.653,24.149,
     &22.360,20.805,18.150,14.592,11.781,10.679,
     &10.459,10.224,9.952,9.115,9.020,-999.000,
     &10.928,-999.000,14.027,17.314,16.607,11.465,
     &6.319,-999.000,-5.359,-999.000,-14.237,-999.000,
     &-21.847,-999.000,-26.750,-999.000,-28.184,-999.000,
     &-26.929,-24.996,-23.935,-23.563,-22.856
     &/
       data (fjrocvz(it,20,1, 6), it=1,59) /
     &-21.736,-20.534,-20.088,-20.056,-20.182,-19.598,
     &-18.868,-18.683,-19.131,-20.869,-23.161,-20.868,
     &-11.967,0.353,7.270,11.150,15.405,19.635,
     &22.994,24.776,25.754,26.590,26.841,26.007,
     &25.475,24.869,22.823,19.169,15.218,13.478,
     &13.036,12.430,11.065,9.547,9.135,-999.000,
     &8.757,-999.000,9.474,11.695,14.856,11.225,
     &3.423,-999.000,-6.811,-999.000,-16.180,-999.000,
     &-23.438,-999.000,-28.127,-999.000,-29.647,-999.000,
     &-29.119,-27.294,-25.356,-23.477,-21.736
     &/
       data (fjrocvz(it,21,1, 6), it=1,59) /
     &-22.967,-22.083,-22.085,-22.550,-22.744,-21.581,
     &-19.830,-19.869,-21.683,-24.912,-27.809,-25.865,
     &-16.085,-3.565,3.589,7.661,11.912,15.677,
     &19.281,22.017,23.753,24.282,24.122,24.015,
     &24.363,23.995,22.068,18.332,14.058,11.965,
     &11.523,11.136,9.998,8.241,7.231,-999.000,
     &7.901,-999.000,9.273,11.391,12.673,8.330,
     &0.079,-999.000,-9.755,-999.000,-17.582,-999.000,
     &-23.800,-999.000,-28.474,-999.000,-30.024,-999.000,
     &-29.866,-28.637,-26.924,-24.840,-22.967
     &/
       data (fjrocvz(it,22,1, 6), it=1,59) /
     &-21.327,-20.708,-20.986,-22.063,-22.599,-21.775,
     &-20.406,-20.721,-22.509,-25.392,-27.550,-25.620,
     &-16.324,-4.795,2.364,6.450,9.629,13.712,
     &17.630,21.201,22.669,22.961,23.417,23.605,
     &23.332,21.980,18.928,14.131,9.268,6.900,
     &6.638,6.700,5.799,4.469,4.201,-999.000,
     &4.741,-999.000,4.264,-999.000,4.640,-999.000,
     &0.194,-999.000,-10.824,-999.000,-16.542,-999.000,
     &-21.353,-999.000,-25.633,-999.000,-27.474,-999.000,
     &-27.350,-25.938,-23.989,-22.482,-21.327
     &/
       data (fjrocvz(it,23,1, 6), it=1,59) /
     &-16.175,-16.196,-16.596,-17.500,-18.485,-18.464,
     &-18.357,-19.002,-19.836,-21.036,-21.296,-17.835,
     &-10.090,-2.028,4.027,7.253,9.856,13.030,
     &16.815,20.434,22.873,24.998,26.090,25.534,
     &23.784,21.215,18.342,14.519,9.806,6.548,
     &4.579,3.518,2.606,1.995,2.064,-999.000,
     &2.777,-999.000,6.019,-999.000,6.939,-999.000,
     &4.256,-999.000,-0.613,-999.000,-10.220,-999.000,
     &-15.714,-999.000,-20.235,-999.000,-22.085,-999.000,
     &-22.038,-20.762,-18.775,-16.895,-16.175
     &/
       data (fjrocvz(it,24,1, 6), it=1,59) /
     &-12.917,-12.798,-12.881,-13.609,-14.100,-13.934,
     &-14.250,-15.636,-17.086,-18.918,-18.808,-14.405,
     &-6.296,0.963,6.115,9.996,13.020,16.289,
     &19.884,23.290,25.418,25.990,25.417,23.898,
     &21.638,19.062,16.466,13.711,9.762,6.253,
     &3.805,2.909,2.793,2.393,1.864,-999.000,
     &2.689,-999.000,4.442,-999.000,6.261,-999.000,
     &5.305,-999.000,1.613,-999.000,-4.139,-999.000,
     &-9.751,-999.000,-13.698,-999.000,-15.553,-999.000,
     &-16.520,-16.670,-15.518,-13.895,-12.917
     &/
       data (fjrocvz(it,25,1, 6), it=1,59) /
     &-12.547,-11.973,-11.646,-11.795,-12.206,-13.316,
     &-14.991,-16.802,-19.008,-20.647,-19.584,-13.369,
     &-3.827,3.968,9.944,14.423,18.938,24.042,
     &28.328,31.088,31.257,29.454,27.124,24.152,
     &21.051,17.157,13.247,10.292,7.146,3.870,
     &1.904,1.376,1.492,1.221,1.092,-999.000,
     &0.569,-999.000,1.458,-999.000,3.851,-999.000,
     &2.829,-999.000,0.302,-999.000,-4.922,-999.000,
     &-9.446,-999.000,-12.707,-999.000,-14.677,-999.000,
     &-16.581,-17.226,-16.229,-14.031,-12.547
     &/
       data (fjrocvz(it, 1,2, 6), it=1,59) /
     &-18.292,-17.432,-15.915,-14.848,-15.649,-16.432,
     &-17.021,-17.888,-18.439,-17.795,-14.516,-7.760,
     &-9.089,-4.689,0.724,6.825,12.431,21.289,
     &28.302,31.110,30.782,29.484,27.006,22.851,
     &18.663,14.961,9.728,5.886,5.557,7.045,
     &7.710,8.040,8.603,10.331,13.013,-999.000,
     &16.734,-999.000,20.830,24.136,24.564,19.548,
     &11.919,-999.000,-0.171,-999.000,-9.434,-999.000,
     &-15.309,-999.000,-17.476,-999.000,-21.278,-999.000,
     &-22.054,-21.853,-21.480,-20.023,-18.292
     &/
       data (fjrocvz(it, 2,2, 6), it=1,59) /
     &-16.630,-15.937,-15.043,-14.610,-13.939,-13.225,
     &-13.378,-14.943,-16.104,-15.385,-10.004,-2.075,
     &3.555,6.856,11.235,13.604,17.725,23.112,
     &28.571,32.330,33.236,32.256,29.445,25.142,
     &20.911,17.894,14.196,10.603,8.776,8.896,
     &10.068,10.922,11.812,13.462,15.134,-999.000,
     &17.470,-999.000,22.075,24.748,26.445,25.610,
     &19.518,-999.000,0.067,-999.000,-12.320,-999.000,
     &-14.157,-999.000,-19.231,-999.000,-22.867,-999.000,
     &-22.303,-21.942,-20.232,-17.826,-16.630
     &/
       data (fjrocvz(it, 3,2, 6), it=1,59) /
     &-16.828,-15.628,-14.965,-13.834,-12.832,-12.900,
     &-12.464,-12.547,-13.069,-12.427,-7.941,-1.476,
     &4.459,8.216,10.651,13.753,17.769,21.229,
     &24.722,27.493,29.011,28.254,25.322,21.565,
     &18.252,16.020,13.747,11.499,9.791,9.614,
     &10.644,11.630,13.025,13.915,15.713,-999.000,
     &18.898,-999.000,21.874,27.005,30.638,28.699,
     &18.916,7.891,-0.487,-11.029,-16.672,-16.704,
     &-17.538,-999.000,-22.628,-999.000,-24.254,-999.000,
     &-23.975,-22.369,-20.373,-18.473,-16.828
     &/
       data (fjrocvz(it, 4,2, 6), it=1,59) /
     &-18.111,-16.961,-15.370,-13.786,-13.407,-13.418,
     &-12.678,-11.246,-10.491,-9.716,-6.675,-1.203,
     &4.789,7.538,8.858,11.584,15.435,19.920,
     &23.531,25.673,25.599,23.755,20.941,17.666,
     &14.833,12.686,10.715,9.476,9.237,9.334,
     &10.176,11.647,12.838,14.799,18.118,-999.000,
     &20.091,-999.000,23.523,29.018,30.626,29.375,
     &23.853,13.512,-0.848,-15.076,-20.025,-999.000,
     &-20.024,-999.000,-23.563,-999.000,-26.136,-999.000,
     &-24.861,-22.402,-20.741,-19.081,-18.111
     &/
       data (fjrocvz(it, 5,2, 6), it=1,59) /
     &-18.656,-17.362,-15.093,-13.441,-13.268,-13.990,
     &-13.845,-12.669,-12.115,-11.229,-8.219,-2.003,
     &3.984,7.195,9.967,12.205,16.078,21.242,
     &25.566,28.076,28.154,26.291,22.879,19.224,
     &16.063,13.949,12.328,11.523,10.863,10.422,
     &11.096,12.482,14.230,16.033,17.441,-999.000,
     &21.266,-999.000,25.238,29.211,32.157,34.219,
     &29.494,14.488,-1.154,-11.792,-19.369,-22.571,
     &-25.342,-28.205,-29.192,-28.767,-27.045,-999.000,
     &-24.877,-22.472,-20.630,-19.315,-18.656
     &/
       data (fjrocvz(it, 6,2, 6), it=1,59) /
     &-19.489,-18.618,-17.146,-15.754,-15.436,-15.293,
     &-14.088,-13.293,-12.823,-11.039,-8.861,-7.463,
     &-4.182,1.944,8.481,13.547,17.492,22.442,
     &26.712,29.217,30.360,29.267,26.761,23.332,
     &20.445,18.438,16.517,14.617,13.254,11.975,
     &11.434,13.084,15.071,15.838,17.599,-999.000,
     &21.794,-999.000,25.976,32.529,38.029,39.651,
     &34.125,20.762,3.834,-8.719,-17.853,-23.844,
     &-26.514,-999.000,-28.959,-999.000,-28.480,-999.000,
     &-26.253,-22.949,-21.206,-20.407,-19.489
     &/
       data (fjrocvz(it, 7,2, 6), it=1,59) /
     &-21.304,-20.874,-19.816,-18.865,-18.243,-16.608,
     &-14.856,-13.718,-12.290,-11.278,-9.147,-6.784,
     &-1.624,1.690,5.772,9.357,13.625,19.220,
     &24.313,28.213,29.595,29.579,28.288,25.937,
     &23.575,21.122,18.749,16.791,14.553,12.164,
     &11.380,12.170,13.460,14.999,16.301,-999.000,
     &19.187,-999.000,24.040,31.720,38.411,42.449,
     &33.821,17.106,2.010,-8.111,-17.458,-25.524,
     &-30.153,-31.464,-31.266,-30.695,-29.646,-999.000,
     &-27.025,-24.860,-23.288,-21.803,-21.304
     &/
       data (fjrocvz(it, 8,2, 6), it=1,59) /
     &-21.244,-19.928,-19.050,-18.621,-17.991,-17.334,
     &-16.618,-15.858,-14.375,-16.297,-18.171,-19.915,
     &-14.738,-6.963,-1.129,4.616,8.379,12.308,
     &18.251,22.862,25.455,25.945,25.353,24.151,
     &22.465,20.595,18.305,16.097,13.385,10.988,
     &9.930,10.191,11.535,12.942,14.064,-999.000,
     &17.175,-999.000,22.867,28.098,34.319,34.976,
     &25.471,13.481,2.573,-8.116,-15.601,-22.698,
     &-27.083,-27.859,-26.031,-999.000,-26.212,-999.000,
     &-25.349,-25.118,-24.097,-22.547,-21.244
     &/
       data (fjrocvz(it, 9,2, 6), it=1,59) /
     &-18.622,-16.609,-14.819,-13.730,-13.755,-14.540,
     &-16.179,-17.602,-18.561,-20.466,-23.139,-25.145,
     &-19.983,-9.742,-1.634,3.758,6.279,10.046,
     &15.031,18.805,20.814,21.524,21.355,20.005,
     &17.743,15.641,13.743,11.759,9.600,8.335,
     &8.263,8.896,10.122,11.203,12.629,-999.000,
     &14.957,-999.000,21.435,27.419,32.781,35.709,
     &33.334,21.683,9.763,-1.104,-8.517,-999.000,
     &-14.376,-999.000,-17.885,-999.000,-20.695,-999.000,
     &-21.270,-21.400,-21.646,-20.707,-18.622
     &/
       data (fjrocvz(it,10,2, 6), it=1,59) /
     &-19.021,-16.813,-14.545,-13.356,-12.758,-14.040,
     &-17.213,-19.944,-21.538,-22.408,-23.088,-22.873,
     &-18.734,-7.716,0.443,5.850,8.416,12.547,
     &17.709,21.047,21.726,21.418,20.213,18.605,
     &16.553,13.965,11.632,9.699,8.430,7.942,
     &8.046,8.433,8.553,9.043,10.722,-999.000,
     &14.318,-999.000,21.906,28.767,36.441,40.417,
     &38.218,26.308,14.468,-999.000,1.929,-999.000,
     &-7.340,-999.000,-13.716,-999.000,-15.724,-999.000,
     &-17.516,-18.542,-19.609,-19.676,-19.021
     &/
       data (fjrocvz(it,11,2, 6), it=1,59) /
     &-21.497,-20.759,-20.343,-20.293,-20.198,-20.803,
     &-21.648,-23.211,-25.139,-27.125,-26.444,-22.659,
     &-17.164,-7.135,0.595,6.801,11.239,14.744,
     &19.184,22.924,24.550,24.571,23.563,22.036,
     &20.442,18.093,15.172,13.085,11.562,10.640,
     &10.015,9.290,8.798,9.171,11.341,-999.000,
     &16.073,-999.000,22.842,29.608,38.048,38.727,
     &34.087,24.199,14.206,-999.000,-0.367,-999.000,
     &-10.241,-999.000,-16.176,-999.000,-17.566,-999.000,
     &-20.137,-20.025,-20.955,-22.100,-21.497
     &/
       data (fjrocvz(it,12,2, 6), it=1,59) /
     &-23.330,-24.006,-23.888,-23.777,-23.974,-23.679,
     &-23.657,-25.359,-28.296,-30.336,-31.023,-27.686,
     &-23.340,-13.327,-3.408,2.672,7.683,12.879,
     &19.615,24.622,26.862,26.966,26.015,24.932,
     &23.605,21.124,18.047,15.587,14.288,13.544,
     &12.609,11.348,10.433,10.766,13.314,-999.000,
     &17.658,-999.000,24.091,30.320,37.890,42.165,
     &36.495,20.240,7.874,-0.465,-7.713,-999.000,
     &-12.282,-999.000,-16.318,-999.000,-20.124,-999.000,
     &-22.442,-22.451,-22.750,-23.194,-23.330
     &/
       data (fjrocvz(it,13,2, 6), it=1,59) /
     &-22.576,-22.065,-21.825,-21.800,-21.468,-22.175,
     &-23.749,-25.098,-26.578,-27.163,-27.536,-26.003,
     &-23.032,-14.055,-5.078,1.738,6.513,12.383,
     &21.661,27.474,28.737,28.608,27.710,26.759,
     &25.276,22.461,19.505,17.505,16.942,16.697,
     &15.970,14.670,13.267,13.343,15.455,-999.000,
     &18.864,-999.000,27.368,33.242,41.339,46.078,
     &37.756,20.885,4.430,-4.690,-9.387,-11.321,
     &-12.281,-999.000,-15.887,-999.000,-18.805,-999.000,
     &-21.107,-23.140,-23.073,-22.558,-22.576
     &/
       data (fjrocvz(it,14,2, 6), it=1,59) /
     &-20.883,-20.433,-20.088,-19.502,-19.469,-21.077,
     &-23.844,-27.402,-30.560,-30.277,-27.349,-23.317,
     &-19.384,-15.610,-8.541,-0.602,5.312,11.805,
     &21.145,26.803,28.684,28.663,27.611,26.328,
     &24.351,22.148,19.932,18.529,18.289,18.062,
     &17.079,15.831,14.550,14.664,16.956,-999.000,
     &21.516,-999.000,29.864,38.053,43.234,42.876,
     &34.645,20.100,7.853,-999.000,-7.464,-999.000,
     &-15.149,-999.000,-18.175,-999.000,-20.763,-999.000,
     &-23.419,-24.205,-23.258,-21.737,-20.883
     &/
       data (fjrocvz(it,15,2, 6), it=1,59) /
     &-18.911,-18.367,-17.791,-17.316,-17.298,-18.871,
     &-22.777,-28.786,-32.889,-34.190,-30.782,-23.591,
     &-18.741,-13.734,-8.307,-0.789,5.771,11.798,
     &19.280,24.454,26.536,26.211,24.483,22.001,
     &19.571,17.736,15.729,14.389,13.918,13.415,
     &12.677,11.832,11.609,12.649,15.710,-999.000,
     &19.704,-999.000,26.734,33.723,38.545,40.572,
     &33.571,18.474,6.232,-999.000,-9.845,-999.000,
     &-16.896,-999.000,-21.309,-999.000,-24.481,-999.000,
     &-26.210,-26.456,-24.033,-20.590,-18.911
     &/
       data (fjrocvz(it,16,2, 6), it=1,59) /
     &-18.884,-17.649,-17.796,-17.996,-18.679,-20.696,
     &-24.377,-29.024,-32.541,-35.420,-34.339,-26.833,
     &-17.370,-12.094,-6.270,0.234,6.344,11.723,
     &19.013,23.983,25.735,25.675,24.244,21.030,
     &17.134,13.682,11.089,9.538,9.099,9.028,
     &8.382,7.432,7.150,8.055,10.516,-999.000,
     &14.603,-999.000,21.337,27.688,33.075,34.690,
     &26.963,11.276,2.409,-999.000,-11.862,-999.000,
     &-19.444,-999.000,-24.235,-999.000,-27.093,-999.000,
     &-28.350,-27.876,-25.352,-21.751,-18.884
     &/
       data (fjrocvz(it,17,2, 6), it=1,59) /
     &-21.973,-21.152,-20.426,-19.950,-20.546,-22.126,
     &-25.196,-29.715,-33.390,-35.005,-33.301,-25.223,
     &-16.340,-10.413,-3.776,1.719,6.954,12.641,
     &20.431,25.739,28.096,28.642,27.493,24.640,
     &20.220,15.911,12.089,9.346,8.556,8.112,
     &7.124,5.783,4.424,4.178,6.520,-999.000,
     &11.633,-999.000,19.345,24.121,28.238,28.620,
     &22.375,9.009,-2.327,-12.282,-19.575,-23.403,
     &-25.294,-999.000,-26.378,-999.000,-28.320,-999.000,
     &-28.854,-27.681,-26.055,-23.859,-21.973
     &/
       data (fjrocvz(it,18,2, 6), it=1,59) /
     &-23.171,-21.728,-20.480,-20.091,-19.822,-20.596,
     &-24.203,-28.790,-31.585,-33.137,-30.631,-23.380,
     &-14.464,-8.624,-3.616,1.964,7.622,14.916,
     &22.138,27.924,31.743,32.425,30.207,26.299,
     &22.500,18.808,14.610,10.895,9.106,7.767,
     &6.536,4.979,3.761,3.614,5.850,-999.000,
     &10.291,13.819,18.736,23.448,26.332,23.574,
     &16.059,7.083,-3.861,-999.000,-19.674,-999.000,
     &-26.772,-999.000,-29.613,-999.000,-30.078,-999.000,
     &-29.083,-28.580,-27.015,-24.940,-23.171
     &/
       data (fjrocvz(it,19,2, 6), it=1,59) /
     &-21.707,-20.020,-19.813,-19.197,-18.467,-19.203,
     &-21.895,-24.585,-26.895,-28.541,-25.945,-19.178,
     &-11.208,-7.081,-2.109,3.548,8.986,16.729,
     &24.927,31.428,35.471,35.910,33.332,29.378,
     &25.131,20.890,16.640,12.850,11.441,10.636,
     &9.126,7.304,6.019,5.526,7.332,-999.000,
     &12.468,-999.000,20.325,24.790,26.243,22.017,
     &16.438,7.183,-4.944,-15.437,-23.185,-27.048,
     &-28.290,-999.000,-29.773,-999.000,-29.200,-999.000,
     &-28.195,-27.785,-26.482,-24.169,-21.707
     &/
       data (fjrocvz(it,20,2, 6), it=1,59) /
     &-23.570,-21.979,-20.232,-18.428,-17.085,-17.223,
     &-19.305,-22.353,-24.016,-24.735,-22.223,-13.215,
     &-5.234,-3.446,0.970,5.292,9.772,17.647,
     &25.540,32.880,37.139,37.913,35.986,32.536,
     &28.715,24.892,20.471,16.738,14.999,13.572,
     &11.769,9.846,8.258,7.770,10.059,-999.000,
     &14.742,-999.000,20.949,25.101,29.265,29.299,
     &21.929,7.693,-4.592,-12.458,-19.162,-24.349,
     &-27.374,-999.000,-28.415,-999.000,-29.192,-999.000,
     &-28.689,-28.295,-27.412,-25.505,-23.570
     &/
       data (fjrocvz(it,21,2, 6), it=1,59) /
     &-23.705,-21.304,-19.408,-18.189,-17.416,-17.092,
     &-18.187,-20.855,-23.608,-24.323,-21.534,-14.279,
     &-8.877,-4.161,0.835,5.987,11.351,17.558,
     &23.428,30.093,33.851,35.214,33.441,29.904,
     &26.373,23.249,19.502,15.358,12.944,11.215,
     &9.492,8.288,7.728,7.820,9.679,-999.000,
     &13.828,-999.000,20.661,25.440,30.230,29.030,
     &20.060,-999.000,-0.244,-999.000,-15.224,-999.000,
     &-23.337,-999.000,-26.907,-999.000,-27.902,-999.000,
     &-27.885,-27.450,-26.762,-25.502,-23.705
     &/
       data (fjrocvz(it,22,2, 6), it=1,59) /
     &-22.598,-21.536,-20.399,-19.572,-19.507,-19.298,
     &-19.536,-21.758,-25.549,-26.428,-23.434,-16.569,
     &-8.489,-5.484,-0.005,4.872,9.744,15.553,
     &21.972,27.871,30.653,30.901,28.836,24.898,
     &20.246,16.253,12.695,9.587,7.695,6.416,
     &5.587,5.948,6.780,7.342,9.367,-999.000,
     &14.462,-999.000,21.259,23.503,26.434,23.358,
     &16.076,-999.000,-2.975,-999.000,-16.316,-999.000,
     &-24.412,-999.000,-25.859,-999.000,-26.051,-999.000,
     &-26.520,-26.007,-25.128,-23.878,-22.598
     &/
       data (fjrocvz(it,23,2, 6), it=1,59) /
     &-20.715,-20.312,-18.833,-16.899,-16.367,-17.276,
     &-19.127,-21.461,-23.614,-24.350,-21.788,-15.651,
     &-10.538,-7.676,-2.272,2.968,8.192,13.734,
     &21.826,27.560,29.407,28.437,25.805,21.738,
     &17.137,12.580,8.907,6.285,5.070,4.461,
     &4.083,4.042,5.430,7.594,10.655,-999.000,
     &14.322,-999.000,20.094,23.495,25.175,23.645,
     &17.578,6.748,-2.526,-999.000,-15.104,-999.000,
     &-21.756,-999.000,-24.235,-999.000,-24.734,-999.000,
     &-25.145,-24.562,-23.108,-21.298,-20.715
     &/
       data (fjrocvz(it,24,2, 6), it=1,59) /
     &-18.880,-17.469,-15.759,-13.705,-13.605,-15.395,
     &-17.544,-18.836,-19.920,-20.573,-18.470,-13.617,
     &-7.549,-2.723,0.676,3.894,8.417,16.611,
     &24.771,28.368,29.061,27.955,24.867,20.486,
     &16.114,11.738,7.587,4.743,4.416,5.496,
     &5.824,5.522,6.765,8.796,11.379,-999.000,
     &14.810,-999.000,19.834,23.810,26.274,23.464,
     &15.255,-999.000,0.613,-999.000,-9.373,-999.000,
     &-15.928,-999.000,-20.575,-999.000,-21.576,-999.000,
     &-22.845,-22.691,-21.504,-20.235,-18.880
     &/
       data (fjrocvz(it,25,2, 6), it=1,59) /
     &-18.292,-17.432,-15.915,-14.848,-15.649,-16.432,
     &-17.021,-17.888,-18.439,-17.795,-14.516,-7.760,
     &-9.089,-4.689,0.724,6.825,12.431,21.289,
     &28.302,31.110,30.782,29.484,27.006,22.851,
     &18.663,14.961,9.728,5.886,5.557,7.045,
     &7.710,8.040,8.603,10.331,13.013,-999.000,
     &16.734,-999.000,20.830,24.136,24.564,19.548,
     &11.919,-999.000,-0.171,-999.000,-9.434,-999.000,
     &-15.309,-999.000,-17.476,-999.000,-21.278,-999.000,
     &-22.054,-21.853,-21.480,-20.023,-18.292
     &/
       data (fjrocvz(it, 1,3, 6), it=1,59) /
     &-21.930,-22.882,-22.523,-21.462,-19.804,-18.774,
     &-18.786,-22.111,-29.097,-30.760,-23.772,-10.005,
     &1.566,5.759,8.499,10.596,10.572,9.483,
     &9.769,12.310,15.530,16.471,15.909,13.501,
     &10.179,8.579,9.128,10.132,10.098,9.235,
     &8.539,8.470,8.301,7.713,8.080,-999.000,
     &10.265,-999.000,12.720,-999.000,18.326,21.589,
     &21.953,17.928,8.363,-999.000,-2.461,-999.000,
     &-9.388,-999.000,-12.339,-999.000,-14.261,-999.000,
     &-16.614,-18.925,-20.395,-20.910,-21.930
     &/
       data (fjrocvz(it, 2,3, 6), it=1,59) /
     &-19.942,-20.433,-20.026,-18.642,-17.558,-17.861,
     &-19.390,-22.947,-27.974,-28.929,-20.739,-5.079,
     &8.343,11.826,13.132,14.148,14.324,14.042,
     &14.841,16.387,17.740,18.225,17.288,15.284,
     &13.521,12.517,12.900,13.631,14.064,13.685,
     &13.078,12.703,12.407,12.672,12.720,-999.000,
     &13.075,-999.000,16.018,-999.000,20.310,23.139,
     &22.152,17.162,6.574,-999.000,-3.141,-999.000,
     &-10.008,-999.000,-13.688,-999.000,-15.824,-999.000,
     &-17.868,-19.931,-20.408,-19.793,-19.942
     &/
       data (fjrocvz(it, 3,3, 6), it=1,59) /
     &-18.949,-18.040,-17.525,-16.646,-16.335,-16.811,
     &-18.379,-20.583,-24.454,-27.954,-20.768,-1.677,
     &15.774,17.847,16.437,15.273,14.384,14.327,
     &15.239,15.967,15.523,14.396,12.922,12.140,
     &11.910,11.435,12.178,13.234,14.313,15.440,
     &15.367,14.833,14.542,14.939,15.210,-999.000,
     &16.302,-999.000,17.201,-999.000,19.717,19.878,
     &17.622,14.877,6.473,-999.000,-4.546,-999.000,
     &-11.610,-999.000,-16.083,-999.000,-18.806,-999.000,
     &-21.319,-23.156,-23.119,-21.015,-18.949
     &/
       data (fjrocvz(it, 4,3, 6), it=1,59) /
     &-20.788,-18.846,-18.203,-18.225,-17.658,-17.193,
     &-17.050,-18.001,-21.527,-23.998,-16.570,3.457,
     &18.349,20.541,19.248,18.426,17.215,15.352,
     &13.817,12.790,12.065,11.306,11.062,11.415,
     &11.424,11.534,12.111,13.311,14.737,15.503,
     &15.919,16.013,15.474,15.778,17.597,-999.000,
     &17.838,-999.000,17.435,-999.000,19.700,20.664,
     &18.730,10.741,3.717,-999.000,-5.905,-999.000,
     &-13.456,-999.000,-18.522,-999.000,-21.928,-999.000,
     &-23.935,-25.137,-25.032,-23.069,-20.788
     &/
       data (fjrocvz(it, 5,3, 6), it=1,59) /
     &-22.456,-21.362,-20.677,-18.918,-16.905,-16.066,
     &-15.898,-17.072,-18.803,-18.993,-12.864,1.658,
     &16.534,21.315,21.147,20.500,19.146,17.361,
     &15.718,14.703,14.827,14.897,14.757,13.915,
     &12.775,11.720,11.246,11.751,12.459,13.301,
     &14.339,15.157,15.283,16.046,16.957,-999.000,
     &17.462,-999.000,18.690,21.021,22.541,19.215,
     &14.685,7.768,1.056,-999.000,-8.216,-999.000,
     &-15.192,-999.000,-20.133,-999.000,-23.316,-999.000,
     &-25.725,-27.066,-26.342,-24.167,-22.456
     &/
       data (fjrocvz(it, 6,3, 6), it=1,59) /
     &-24.445,-24.640,-24.527,-22.132,-18.356,-15.827,
     &-15.644,-16.824,-17.028,-15.541,-11.087,-0.639,
     &13.350,18.127,18.007,17.460,17.231,16.954,
     &16.624,17.742,19.185,19.468,18.577,17.249,
     &15.666,13.832,12.539,11.552,11.759,12.322,
     &13.002,13.543,14.278,14.844,15.328,-999.000,
     &17.542,-999.000,18.394,20.145,18.782,13.497,
     &7.070,-999.000,-3.185,-999.000,-11.287,-999.000,
     &-18.346,-999.000,-23.311,-999.000,-26.494,-999.000,
     &-29.052,-30.171,-28.552,-26.137,-24.445
     &/
       data (fjrocvz(it, 7,3, 6), it=1,59) /
     &-27.375,-25.925,-25.576,-24.733,-21.991,-20.251,
     &-19.579,-18.826,-18.140,-15.007,-10.720,-3.399,
     &4.313,9.008,10.511,11.603,12.561,13.410,
     &14.722,16.796,19.200,20.550,20.860,20.620,
     &20.049,19.311,17.691,15.773,14.794,14.079,
     &13.738,13.226,12.962,12.793,12.947,-999.000,
     &13.402,-999.000,13.150,-999.000,8.968,-999.000,
     &1.588,-999.000,-5.254,-999.000,-12.445,-999.000,
     &-19.580,-999.000,-24.839,-999.000,-28.420,-999.000,
     &-30.984,-32.196,-31.457,-29.720,-27.375
     &/
       data (fjrocvz(it, 8,3, 6), it=1,59) /
     &-24.532,-23.421,-24.518,-25.098,-24.044,-20.894,
     &-16.538,-13.842,-12.732,-11.262,-8.227,-4.297,
     &-1.631,0.735,2.252,4.076,6.090,7.639,
     &9.219,11.151,14.372,17.882,19.808,20.361,
     &20.507,19.706,18.063,15.838,14.088,13.480,
     &12.944,12.112,10.984,9.476,8.172,-999.000,
     &7.706,-999.000,6.812,-999.000,4.083,-999.000,
     &0.497,-999.000,-2.911,-999.000,-8.435,-999.000,
     &-14.878,-999.000,-20.090,-999.000,-25.014,-999.000,
     &-28.512,-29.887,-28.797,-26.646,-24.532
     &/
       data (fjrocvz(it, 9,3, 6), it=1,59) /
     &-18.302,-18.535,-18.877,-19.416,-19.304,-16.800,
     &-13.089,-11.099,-10.971,-11.408,-10.571,-8.171,
     &-7.582,-3.436,1.203,5.600,10.005,10.963,
     &11.822,12.673,13.670,13.389,14.723,15.890,
     &16.799,16.757,14.989,12.385,10.819,9.683,
     &8.431,7.129,6.046,5.399,4.966,-999.000,
     &3.911,-999.000,2.804,-999.000,3.170,-999.000,
     &3.544,-999.000,3.412,-999.000,-0.791,-999.000,
     &-5.393,-999.000,-10.128,-999.000,-13.249,-999.000,
     &-16.016,-17.816,-18.118,-18.184,-18.302
     &/
       data (fjrocvz(it,10,3, 6), it=1,59) /
     &-15.427,-15.966,-15.187,-14.179,-14.640,-15.648,
     &-15.245,-14.080,-13.612,-13.599,-13.349,-14.384,
     &-16.214,-8.286,0.382,6.327,12.393,13.357,
     &14.271,14.947,15.754,16.102,16.047,16.006,
     &16.054,16.197,14.644,12.560,10.555,8.384,
     &6.187,3.690,2.001,1.109,-0.008,-999.000,
     &-1.332,-999.000,-0.812,-999.000,0.694,-999.000,
     &4.212,-999.000,5.913,-999.000,5.116,-999.000,
     &1.632,-999.000,-2.381,-999.000,-5.685,-999.000,
     &-8.729,-11.528,-13.319,-14.447,-15.427
     &/
       data (fjrocvz(it,11,3, 6), it=1,59) /
     &-18.896,-20.357,-20.531,-19.752,-19.297,-18.622,
     &-17.032,-16.702,-19.796,-18.549,-18.222,-12.164,
     &-15.400,-9.333,-3.137,3.423,10.136,12.335,
     &14.359,15.973,19.093,21.293,20.698,19.950,
     &19.515,18.742,17.669,16.251,13.896,11.125,
     &8.074,5.400,3.756,1.654,-0.395,-999.000,
     &-1.437,-999.000,-1.101,-999.000,1.303,-999.000,
     &8.328,11.173,11.075,8.209,6.005,-999.000,
     &2.432,-999.000,-0.676,-999.000,-3.903,-999.000,
     &-7.580,-12.391,-15.475,-17.233,-18.896
     &/
       data (fjrocvz(it,12,3, 6), it=1,59) /
     &-19.912,-21.996,-24.316,-25.997,-25.639,-23.469,
     &-23.066,-27.101,-32.126,-29.841,-24.165,-14.506,
     &-13.408,-9.961,-6.464,-0.354,5.920,9.674,
     &12.972,17.180,21.527,23.466,23.513,22.933,
     &22.161,20.842,19.375,18.309,16.240,13.264,
     &10.420,8.826,7.375,5.717,3.711,-999.000,
     &1.908,-999.000,3.217,-999.000,9.983,15.338,
     &19.615,20.563,17.867,-999.000,11.452,-999.000,
     &5.183,-999.000,0.754,-999.000,-2.552,-999.000,
     &-6.557,-11.498,-15.711,-18.069,-19.912
     &/
       data (fjrocvz(it,13,3, 6), it=1,59) /
     &-18.648,-20.534,-21.393,-22.501,-24.680,-26.425,
     &-28.876,-31.679,-34.035,-32.083,-24.901,-17.047,
     &-13.020,-11.423,-8.147,-4.461,-0.815,5.839,
     &14.052,19.962,23.818,25.884,26.041,24.760,
     &23.411,22.163,20.645,19.116,17.069,14.764,
     &12.754,10.546,8.860,7.802,6.478,-999.000,
     &6.358,-999.000,8.791,-999.000,14.063,19.718,
     &26.469,28.642,22.673,-999.000,14.686,-999.000,
     &7.418,-999.000,1.830,-999.000,-2.159,-999.000,
     &-5.801,-9.010,-12.416,-15.926,-18.648
     &/
       data (fjrocvz(it,14,3, 6), it=1,59) /
     &-18.955,-21.105,-21.251,-22.339,-25.281,-29.047,
     &-31.441,-33.692,-36.825,-35.432,-26.792,-15.747,
     &-10.590,-9.446,-7.661,-4.741,-1.666,4.323,
     &13.536,19.309,23.663,25.900,26.025,24.245,
     &22.783,21.945,21.058,19.847,18.509,16.821,
     &14.747,12.401,10.851,9.555,9.012,-999.000,
     &9.412,-999.000,11.829,-999.000,17.056,21.581,
     &28.314,30.905,27.971,20.390,15.028,-999.000,
     &6.897,-999.000,1.647,-999.000,-1.747,-999.000,
     &-5.075,-8.112,-11.328,-15.537,-18.955
     &/
       data (fjrocvz(it,15,3, 6), it=1,59) /
     &-18.702,-20.304,-20.868,-21.573,-23.846,-27.394,
     &-30.323,-33.511,-37.165,-37.917,-29.694,-18.025,
     &-11.061,-9.876,-8.975,-6.494,-2.788,3.000,
     &10.801,16.427,20.395,22.518,21.893,19.766,
     &17.896,16.635,15.891,14.712,13.117,11.953,
     &10.832,9.649,8.053,7.112,6.877,-999.000,
     &7.094,-999.000,9.497,-999.000,14.069,18.394,
     &22.505,23.340,19.649,14.143,9.843,-999.000,
     &2.564,-999.000,-1.661,-999.000,-4.051,-999.000,
     &-7.277,-10.013,-12.715,-15.725,-18.702
     &/
       data (fjrocvz(it,16,3, 6), it=1,59) /
     &-16.538,-17.636,-17.721,-18.288,-20.899,-25.163,
     &-29.576,-32.940,-37.949,-40.567,-36.174,-23.434,
     &-12.962,-9.788,-8.881,-6.587,-2.728,2.635,
     &9.489,14.725,18.089,19.206,17.822,16.063,
     &14.306,12.535,10.662,8.138,5.674,4.309,
     &3.710,2.832,1.662,0.360,-0.009,-999.000,
     &1.163,-999.000,3.683,-999.000,7.408,11.099,
     &12.749,10.918,8.077,-999.000,0.380,-999.000,
     &-5.502,-999.000,-8.511,-999.000,-9.522,-999.000,
     &-10.454,-12.031,-13.326,-14.777,-16.538
     &/
       data (fjrocvz(it,17,3, 6), it=1,59) /
     &-17.610,-18.286,-17.735,-17.783,-19.810,-23.127,
     &-27.646,-33.983,-41.089,-43.561,-38.568,-23.159,
     &-10.913,-8.208,-6.346,-3.221,0.647,5.324,
     &11.475,16.815,19.995,20.998,19.983,17.648,
     &15.208,13.032,10.790,7.763,4.824,2.608,
     &0.524,-0.615,-1.574,-2.904,-3.721,-999.000,
     &-3.976,-999.000,-2.080,-999.000,2.346,7.344,
     &8.710,5.852,2.057,-999.000,-4.059,-999.000,
     &-10.489,-999.000,-14.137,-999.000,-14.847,-999.000,
     &-15.155,-15.991,-16.444,-16.735,-17.610
     &/
       data (fjrocvz(it,18,3, 6), it=1,59) /
     &-25.042,-24.267,-22.747,-21.467,-21.789,-23.580,
     &-26.612,-33.178,-42.040,-43.881,-37.261,-15.523,
     &-5.929,-4.577,-2.611,0.780,4.905,10.445,
     &16.727,21.662,24.014,24.943,24.368,22.470,
     &19.598,16.423,13.788,10.671,7.778,4.808,
     &2.176,0.406,-1.135,-2.629,-4.200,-999.000,
     &-4.961,-999.000,-2.983,-999.000,2.003,4.607,
     &3.317,-0.471,-4.004,-999.000,-10.094,-999.000,
     &-16.104,-999.000,-19.112,-999.000,-19.916,-999.000,
     &-20.526,-22.004,-23.189,-24.100,-25.042
     &/
       data (fjrocvz(it,19,3, 6), it=1,59) /
     &-26.599,-26.429,-24.862,-23.366,-23.432,-25.169,
     &-26.145,-28.682,-35.005,-35.287,-29.362,-12.498,
     &-3.409,-1.374,0.399,3.388,8.043,13.737,
     &20.010,24.672,27.414,27.903,27.055,25.435,
     &23.023,20.195,17.082,14.030,11.095,8.080,
     &5.691,4.196,2.892,0.575,-1.201,-999.000,
     &-1.192,-999.000,-0.215,-999.000,4.482,5.696,
     &3.614,-0.205,-5.384,-999.000,-11.168,-999.000,
     &-16.474,-999.000,-19.386,-999.000,-20.668,-999.000,
     &-21.384,-22.666,-24.159,-25.612,-26.599
     &/
       data (fjrocvz(it,20,3, 6), it=1,59) /
     &-23.747,-24.145,-23.038,-21.683,-22.296,-24.987,
     &-27.555,-29.965,-31.479,-28.580,-21.970,-11.991,
     &-3.995,-0.280,1.518,3.711,7.686,13.685,
     &20.316,26.351,29.500,29.680,28.179,25.861,
     &23.487,21.395,19.865,18.405,16.025,13.109,
     &10.265,8.322,6.073,4.084,3.065,-999.000,
     &2.558,-999.000,3.792,-999.000,7.728,9.794,
     &8.746,5.282,-0.031,-999.000,-6.713,-999.000,
     &-11.420,-999.000,-14.206,-999.000,-16.067,-999.000,
     &-17.915,-19.778,-21.235,-22.342,-23.747
     &/
       data (fjrocvz(it,21,3, 6), it=1,59) /
     &-20.336,-21.098,-20.208,-19.065,-19.831,-23.376,
     &-26.248,-28.009,-31.023,-32.995,-29.778,-19.268,
     &-7.520,-2.451,-0.239,1.691,4.657,9.343,
     &15.520,21.003,24.209,24.721,23.825,22.780,
     &21.330,19.901,19.018,17.598,15.595,12.966,
     &10.715,8.630,6.928,5.918,5.111,-999.000,
     &5.008,-999.000,5.134,-999.000,6.267,7.869,
     &9.922,8.386,4.227,-999.000,-1.654,-999.000,
     &-6.230,-999.000,-9.520,-999.000,-12.124,-999.000,
     &-14.496,-16.430,-17.803,-18.873,-20.336
     &/
       data (fjrocvz(it,22,3, 6), it=1,59) /
     &-18.381,-17.767,-16.376,-16.272,-18.088,-21.251,
     &-23.775,-27.312,-33.320,-37.064,-34.486,-23.749,
     &-11.654,-4.163,-2.154,-0.457,1.830,5.103,
     &9.584,13.737,17.007,17.449,16.710,16.453,
     &16.090,15.125,14.084,12.419,10.221,8.200,
     &6.506,5.514,4.752,4.775,5.760,-999.000,
     &6.193,-999.000,5.995,-999.000,7.930,8.962,
     &12.796,12.171,5.867,-999.000,-0.269,-999.000,
     &-5.175,-999.000,-8.671,-999.000,-11.131,-999.000,
     &-13.346,-15.354,-17.056,-18.129,-18.381
     &/
       data (fjrocvz(it,23,3, 6), it=1,59) /
     &-19.788,-18.586,-16.829,-15.926,-17.603,-21.089,
     &-24.557,-27.841,-31.408,-32.697,-28.625,-18.431,
     &-6.865,-1.868,-1.374,-0.436,1.111,3.242,
     &6.287,10.613,14.738,16.204,15.122,13.272,
     &11.574,9.789,8.683,7.534,5.752,4.055,
     &3.577,3.303,2.633,2.867,4.234,-999.000,
     &4.984,-999.000,6.855,-999.000,10.990,14.900,
     &18.183,17.196,10.445,-999.000,0.481,-999.000,
     &-5.288,-999.000,-8.641,-999.000,-11.307,-999.000,
     &-14.072,-16.481,-18.181,-19.321,-19.788
     &/
       data (fjrocvz(it,24,3, 6), it=1,59) /
     &-22.116,-22.220,-20.708,-18.587,-17.884,-19.477,
     &-21.611,-24.406,-29.098,-29.784,-25.144,-13.964,
     &-2.993,1.132,2.654,3.978,4.748,5.493,
     &7.343,10.485,13.964,14.843,14.086,12.512,
     &9.596,7.256,6.187,6.603,6.163,5.556,
     &5.436,5.137,4.779,4.519,4.959,-999.000,
     &7.423,-999.000,10.567,-999.000,15.791,20.024,
     &20.399,17.972,13.118,-999.000,-0.312,-999.000,
     &-7.211,-999.000,-10.104,-999.000,-12.797,-999.000,
     &-16.252,-18.979,-20.433,-21.267,-22.116
     &/
       data (fjrocvz(it,25,3, 6), it=1,59) /
     &-21.930,-22.882,-22.523,-21.462,-19.804,-18.774,
     &-18.786,-22.111,-29.097,-30.760,-23.772,-10.005,
     &1.566,5.759,8.499,10.596,10.572,9.483,
     &9.769,12.310,15.530,16.471,15.909,13.501,
     &10.179,8.579,9.128,10.132,10.098,9.235,
     &8.539,8.470,8.301,7.713,8.080,-999.000,
     &10.265,-999.000,12.720,-999.000,18.326,21.589,
     &21.953,17.928,8.363,-999.000,-2.461,-999.000,
     &-9.388,-999.000,-12.339,-999.000,-14.261,-999.000,
     &-16.614,-18.925,-20.395,-20.910,-21.930
     &/
       data (fjrocvz(it, 1,4, 6), it=1,59) /
     &-18.292,-17.432,-15.915,-14.848,-15.649,-16.432,
     &-17.021,-17.888,-18.439,-17.795,-14.516,-7.760,
     &-9.089,-4.689,0.724,6.825,12.431,21.289,
     &28.302,31.110,30.782,29.484,27.006,22.851,
     &18.663,14.961,9.728,5.886,5.557,7.045,
     &7.710,8.040,8.603,10.331,13.013,-999.000,
     &16.734,-999.000,20.830,24.136,24.564,19.548,
     &11.919,-999.000,-0.171,-999.000,-9.434,-999.000,
     &-15.309,-999.000,-17.476,-999.000,-21.278,-999.000,
     &-22.054,-21.853,-21.480,-20.023,-18.292
     &/
       data (fjrocvz(it, 2,4, 6), it=1,59) /
     &-16.630,-15.937,-15.043,-14.610,-13.939,-13.225,
     &-13.378,-14.943,-16.104,-15.385,-10.004,-2.075,
     &3.555,6.856,11.235,13.604,17.725,23.112,
     &28.571,32.330,33.236,32.256,29.445,25.142,
     &20.911,17.894,14.196,10.603,8.776,8.896,
     &10.068,10.922,11.812,13.462,15.134,-999.000,
     &17.470,-999.000,22.075,24.748,26.445,25.610,
     &19.518,-999.000,0.067,-999.000,-12.320,-999.000,
     &-14.157,-999.000,-19.231,-999.000,-22.867,-999.000,
     &-22.303,-21.942,-20.232,-17.826,-16.630
     &/
       data (fjrocvz(it, 3,4, 6), it=1,59) /
     &-16.828,-15.628,-14.965,-13.834,-12.832,-12.900,
     &-12.464,-12.547,-13.069,-12.427,-7.941,-1.476,
     &4.459,8.216,10.651,13.753,17.769,21.229,
     &24.722,27.493,29.011,28.254,25.322,21.565,
     &18.252,16.020,13.747,11.499,9.791,9.614,
     &10.644,11.630,13.025,13.915,15.713,-999.000,
     &18.898,-999.000,21.874,27.005,30.638,28.699,
     &18.916,7.891,-0.487,-11.029,-16.672,-16.704,
     &-17.538,-999.000,-22.628,-999.000,-24.254,-999.000,
     &-23.975,-22.369,-20.373,-18.473,-16.828
     &/
       data (fjrocvz(it, 4,4, 6), it=1,59) /
     &-18.111,-16.961,-15.370,-13.786,-13.407,-13.418,
     &-12.678,-11.246,-10.491,-9.716,-6.675,-1.203,
     &4.789,7.538,8.858,11.584,15.435,19.920,
     &23.531,25.673,25.599,23.755,20.941,17.666,
     &14.833,12.686,10.715,9.476,9.237,9.334,
     &10.176,11.647,12.838,14.799,18.118,-999.000,
     &20.091,-999.000,23.523,29.018,30.626,29.375,
     &23.853,13.512,-0.848,-15.076,-20.025,-999.000,
     &-20.024,-999.000,-23.563,-999.000,-26.136,-999.000,
     &-24.861,-22.402,-20.741,-19.081,-18.111
     &/
       data (fjrocvz(it, 5,4, 6), it=1,59) /
     &-18.656,-17.362,-15.093,-13.441,-13.268,-13.990,
     &-13.845,-12.669,-12.115,-11.229,-8.219,-2.003,
     &3.984,7.195,9.967,12.205,16.078,21.242,
     &25.566,28.076,28.154,26.291,22.879,19.224,
     &16.063,13.949,12.328,11.523,10.863,10.422,
     &11.096,12.482,14.230,16.033,17.441,-999.000,
     &21.266,-999.000,25.238,29.211,32.157,34.219,
     &29.494,14.488,-1.154,-11.792,-19.369,-22.571,
     &-25.342,-28.205,-29.192,-28.767,-27.045,-999.000,
     &-24.877,-22.472,-20.630,-19.315,-18.656
     &/
       data (fjrocvz(it, 6,4, 6), it=1,59) /
     &-19.489,-18.618,-17.146,-15.754,-15.436,-15.293,
     &-14.088,-13.293,-12.823,-11.039,-8.861,-7.463,
     &-4.182,1.944,8.481,13.547,17.492,22.442,
     &26.712,29.217,30.360,29.267,26.761,23.332,
     &20.445,18.438,16.517,14.617,13.254,11.975,
     &11.434,13.084,15.071,15.838,17.599,-999.000,
     &21.794,-999.000,25.976,32.529,38.029,39.651,
     &34.125,20.762,3.834,-8.719,-17.853,-23.844,
     &-26.514,-999.000,-28.959,-999.000,-28.480,-999.000,
     &-26.253,-22.949,-21.206,-20.407,-19.489
     &/
       data (fjrocvz(it, 7,4, 6), it=1,59) /
     &-21.304,-20.874,-19.816,-18.865,-18.243,-16.608,
     &-14.856,-13.718,-12.290,-11.278,-9.147,-6.784,
     &-1.624,1.690,5.772,9.357,13.625,19.220,
     &24.313,28.213,29.595,29.579,28.288,25.937,
     &23.575,21.122,18.749,16.791,14.553,12.164,
     &11.380,12.170,13.460,14.999,16.301,-999.000,
     &19.187,-999.000,24.040,31.720,38.411,42.449,
     &33.821,17.106,2.010,-8.111,-17.458,-25.524,
     &-30.153,-31.464,-31.266,-30.695,-29.646,-999.000,
     &-27.025,-24.860,-23.288,-21.803,-21.304
     &/
       data (fjrocvz(it, 8,4, 6), it=1,59) /
     &-21.244,-19.928,-19.050,-18.621,-17.991,-17.334,
     &-16.618,-15.858,-14.375,-16.297,-18.171,-19.915,
     &-14.738,-6.963,-1.129,4.616,8.379,12.308,
     &18.251,22.862,25.455,25.945,25.353,24.151,
     &22.465,20.595,18.305,16.097,13.385,10.988,
     &9.930,10.191,11.535,12.942,14.064,-999.000,
     &17.175,-999.000,22.867,28.098,34.319,34.976,
     &25.471,13.481,2.573,-8.116,-15.601,-22.698,
     &-27.083,-27.859,-26.031,-999.000,-26.212,-999.000,
     &-25.349,-25.118,-24.097,-22.547,-21.244
     &/
       data (fjrocvz(it, 9,4, 6), it=1,59) /
     &-18.622,-16.609,-14.819,-13.730,-13.755,-14.540,
     &-16.179,-17.602,-18.561,-20.466,-23.139,-25.145,
     &-19.983,-9.742,-1.634,3.758,6.279,10.046,
     &15.031,18.805,20.814,21.524,21.355,20.005,
     &17.743,15.641,13.743,11.759,9.600,8.335,
     &8.263,8.896,10.122,11.203,12.629,-999.000,
     &14.957,-999.000,21.435,27.419,32.781,35.709,
     &33.334,21.683,9.763,-1.104,-8.517,-999.000,
     &-14.376,-999.000,-17.885,-999.000,-20.695,-999.000,
     &-21.270,-21.400,-21.646,-20.707,-18.622
     &/
       data (fjrocvz(it,10,4, 6), it=1,59) /
     &-19.021,-16.813,-14.545,-13.356,-12.758,-14.040,
     &-17.213,-19.944,-21.538,-22.408,-23.088,-22.873,
     &-18.734,-7.716,0.443,5.850,8.416,12.547,
     &17.709,21.047,21.726,21.418,20.213,18.605,
     &16.553,13.965,11.632,9.699,8.430,7.942,
     &8.046,8.433,8.553,9.043,10.722,-999.000,
     &14.318,-999.000,21.906,28.767,36.441,40.417,
     &38.218,26.308,14.468,-999.000,1.929,-999.000,
     &-7.340,-999.000,-13.716,-999.000,-15.724,-999.000,
     &-17.516,-18.542,-19.609,-19.676,-19.021
     &/
       data (fjrocvz(it,11,4, 6), it=1,59) /
     &-21.497,-20.759,-20.343,-20.293,-20.198,-20.803,
     &-21.648,-23.211,-25.139,-27.125,-26.444,-22.659,
     &-17.164,-7.135,0.595,6.801,11.239,14.744,
     &19.184,22.924,24.550,24.571,23.563,22.036,
     &20.442,18.093,15.172,13.085,11.562,10.640,
     &10.015,9.290,8.798,9.171,11.341,-999.000,
     &16.073,-999.000,22.842,29.608,38.048,38.727,
     &34.087,24.199,14.206,-999.000,-0.367,-999.000,
     &-10.241,-999.000,-16.176,-999.000,-17.566,-999.000,
     &-20.137,-20.025,-20.955,-22.100,-21.497
     &/
       data (fjrocvz(it,12,4, 6), it=1,59) /
     &-23.330,-24.006,-23.888,-23.777,-23.974,-23.679,
     &-23.657,-25.359,-28.296,-30.336,-31.023,-27.686,
     &-23.340,-13.327,-3.408,2.672,7.683,12.879,
     &19.615,24.622,26.862,26.966,26.015,24.932,
     &23.605,21.124,18.047,15.587,14.288,13.544,
     &12.609,11.348,10.433,10.766,13.314,-999.000,
     &17.658,-999.000,24.091,30.320,37.890,42.165,
     &36.495,20.240,7.874,-0.465,-7.713,-999.000,
     &-12.282,-999.000,-16.318,-999.000,-20.124,-999.000,
     &-22.442,-22.451,-22.750,-23.194,-23.330
     &/
       data (fjrocvz(it,13,4, 6), it=1,59) /
     &-22.576,-22.065,-21.825,-21.800,-21.468,-22.175,
     &-23.749,-25.098,-26.578,-27.163,-27.536,-26.003,
     &-23.032,-14.055,-5.078,1.738,6.513,12.383,
     &21.661,27.474,28.737,28.608,27.710,26.759,
     &25.276,22.461,19.505,17.505,16.942,16.697,
     &15.970,14.670,13.267,13.343,15.455,-999.000,
     &18.864,-999.000,27.368,33.242,41.339,46.078,
     &37.756,20.885,4.430,-4.690,-9.387,-11.321,
     &-12.281,-999.000,-15.887,-999.000,-18.805,-999.000,
     &-21.107,-23.140,-23.073,-22.558,-22.576
     &/
       data (fjrocvz(it,14,4, 6), it=1,59) /
     &-20.883,-20.433,-20.088,-19.502,-19.469,-21.077,
     &-23.844,-27.402,-30.560,-30.277,-27.349,-23.317,
     &-19.384,-15.610,-8.541,-0.602,5.312,11.805,
     &21.145,26.803,28.684,28.663,27.611,26.328,
     &24.351,22.148,19.932,18.529,18.289,18.062,
     &17.079,15.831,14.550,14.664,16.956,-999.000,
     &21.516,-999.000,29.864,38.053,43.234,42.876,
     &34.645,20.100,7.853,-999.000,-7.464,-999.000,
     &-15.149,-999.000,-18.175,-999.000,-20.763,-999.000,
     &-23.419,-24.205,-23.258,-21.737,-20.883
     &/
       data (fjrocvz(it,15,4, 6), it=1,59) /
     &-18.911,-18.367,-17.791,-17.316,-17.298,-18.871,
     &-22.777,-28.786,-32.889,-34.190,-30.782,-23.591,
     &-18.741,-13.734,-8.307,-0.789,5.771,11.798,
     &19.280,24.454,26.536,26.211,24.483,22.001,
     &19.571,17.736,15.729,14.389,13.918,13.415,
     &12.677,11.832,11.609,12.649,15.710,-999.000,
     &19.704,-999.000,26.734,33.723,38.545,40.572,
     &33.571,18.474,6.232,-999.000,-9.845,-999.000,
     &-16.896,-999.000,-21.309,-999.000,-24.481,-999.000,
     &-26.210,-26.456,-24.033,-20.590,-18.911
     &/
       data (fjrocvz(it,16,4, 6), it=1,59) /
     &-18.884,-17.649,-17.796,-17.996,-18.679,-20.696,
     &-24.377,-29.024,-32.541,-35.420,-34.339,-26.833,
     &-17.370,-12.094,-6.270,0.234,6.344,11.723,
     &19.013,23.983,25.735,25.675,24.244,21.030,
     &17.134,13.682,11.089,9.538,9.099,9.028,
     &8.382,7.432,7.150,8.055,10.516,-999.000,
     &14.603,-999.000,21.337,27.688,33.075,34.690,
     &26.963,11.276,2.409,-999.000,-11.862,-999.000,
     &-19.444,-999.000,-24.235,-999.000,-27.093,-999.000,
     &-28.350,-27.876,-25.352,-21.751,-18.884
     &/
       data (fjrocvz(it,17,4, 6), it=1,59) /
     &-21.973,-21.152,-20.426,-19.950,-20.546,-22.126,
     &-25.196,-29.715,-33.390,-35.005,-33.301,-25.223,
     &-16.340,-10.413,-3.776,1.719,6.954,12.641,
     &20.431,25.739,28.096,28.642,27.493,24.640,
     &20.220,15.911,12.089,9.346,8.556,8.112,
     &7.124,5.783,4.424,4.178,6.520,-999.000,
     &11.633,-999.000,19.345,24.121,28.238,28.620,
     &22.375,9.009,-2.327,-12.282,-19.575,-23.403,
     &-25.294,-999.000,-26.378,-999.000,-28.320,-999.000,
     &-28.854,-27.681,-26.055,-23.859,-21.973
     &/
       data (fjrocvz(it,18,4, 6), it=1,59) /
     &-23.171,-21.728,-20.480,-20.091,-19.822,-20.596,
     &-24.203,-28.790,-31.585,-33.137,-30.631,-23.380,
     &-14.464,-8.624,-3.616,1.964,7.622,14.916,
     &22.138,27.924,31.743,32.425,30.207,26.299,
     &22.500,18.808,14.610,10.895,9.106,7.767,
     &6.536,4.979,3.761,3.614,5.850,-999.000,
     &10.291,13.819,18.736,23.448,26.332,23.574,
     &16.059,7.083,-3.861,-999.000,-19.674,-999.000,
     &-26.772,-999.000,-29.613,-999.000,-30.078,-999.000,
     &-29.083,-28.580,-27.015,-24.940,-23.171
     &/
       data (fjrocvz(it,19,4, 6), it=1,59) /
     &-21.707,-20.020,-19.813,-19.197,-18.467,-19.203,
     &-21.895,-24.585,-26.895,-28.541,-25.945,-19.178,
     &-11.208,-7.081,-2.109,3.548,8.986,16.729,
     &24.927,31.428,35.471,35.910,33.332,29.378,
     &25.131,20.890,16.640,12.850,11.441,10.636,
     &9.126,7.304,6.019,5.526,7.332,-999.000,
     &12.468,-999.000,20.325,24.790,26.243,22.017,
     &16.438,7.183,-4.944,-15.437,-23.185,-27.048,
     &-28.290,-999.000,-29.773,-999.000,-29.200,-999.000,
     &-28.195,-27.785,-26.482,-24.169,-21.707
     &/
       data (fjrocvz(it,20,4, 6), it=1,59) /
     &-23.570,-21.979,-20.232,-18.428,-17.085,-17.223,
     &-19.305,-22.353,-24.016,-24.735,-22.223,-13.215,
     &-5.234,-3.446,0.970,5.292,9.772,17.647,
     &25.540,32.880,37.139,37.913,35.986,32.536,
     &28.715,24.892,20.471,16.738,14.999,13.572,
     &11.769,9.846,8.258,7.770,10.059,-999.000,
     &14.742,-999.000,20.949,25.101,29.265,29.299,
     &21.929,7.693,-4.592,-12.458,-19.162,-24.349,
     &-27.374,-999.000,-28.415,-999.000,-29.192,-999.000,
     &-28.689,-28.295,-27.412,-25.505,-23.570
     &/
       data (fjrocvz(it,21,4, 6), it=1,59) /
     &-23.705,-21.304,-19.408,-18.189,-17.416,-17.092,
     &-18.187,-20.855,-23.608,-24.323,-21.534,-14.279,
     &-8.877,-4.161,0.835,5.987,11.351,17.558,
     &23.428,30.093,33.851,35.214,33.441,29.904,
     &26.373,23.249,19.502,15.358,12.944,11.215,
     &9.492,8.288,7.728,7.820,9.679,-999.000,
     &13.828,-999.000,20.661,25.440,30.230,29.030,
     &20.060,-999.000,-0.244,-999.000,-15.224,-999.000,
     &-23.337,-999.000,-26.907,-999.000,-27.902,-999.000,
     &-27.885,-27.450,-26.762,-25.502,-23.705
     &/
       data (fjrocvz(it,22,4, 6), it=1,59) /
     &-22.598,-21.536,-20.399,-19.572,-19.507,-19.298,
     &-19.536,-21.758,-25.549,-26.428,-23.434,-16.569,
     &-8.489,-5.484,-0.005,4.872,9.744,15.553,
     &21.972,27.871,30.653,30.901,28.836,24.898,
     &20.246,16.253,12.695,9.587,7.695,6.416,
     &5.587,5.948,6.780,7.342,9.367,-999.000,
     &14.462,-999.000,21.259,23.503,26.434,23.358,
     &16.076,-999.000,-2.975,-999.000,-16.316,-999.000,
     &-24.412,-999.000,-25.859,-999.000,-26.051,-999.000,
     &-26.520,-26.007,-25.128,-23.878,-22.598
     &/
       data (fjrocvz(it,23,4, 6), it=1,59) /
     &-20.715,-20.312,-18.833,-16.899,-16.367,-17.276,
     &-19.127,-21.461,-23.614,-24.350,-21.788,-15.651,
     &-10.538,-7.676,-2.272,2.968,8.192,13.734,
     &21.826,27.560,29.407,28.437,25.805,21.738,
     &17.137,12.580,8.907,6.285,5.070,4.461,
     &4.083,4.042,5.430,7.594,10.655,-999.000,
     &14.322,-999.000,20.094,23.495,25.175,23.645,
     &17.578,6.748,-2.526,-999.000,-15.104,-999.000,
     &-21.756,-999.000,-24.235,-999.000,-24.734,-999.000,
     &-25.145,-24.562,-23.108,-21.298,-20.715
     &/
       data (fjrocvz(it,24,4, 6), it=1,59) /
     &-18.880,-17.469,-15.759,-13.705,-13.605,-15.395,
     &-17.544,-18.836,-19.920,-20.573,-18.470,-13.617,
     &-7.549,-2.723,0.676,3.894,8.417,16.611,
     &24.771,28.368,29.061,27.955,24.867,20.486,
     &16.114,11.738,7.587,4.743,4.416,5.496,
     &5.824,5.522,6.765,8.796,11.379,-999.000,
     &14.810,-999.000,19.834,23.810,26.274,23.464,
     &15.255,-999.000,0.613,-999.000,-9.373,-999.000,
     &-15.928,-999.000,-20.575,-999.000,-21.576,-999.000,
     &-22.845,-22.691,-21.504,-20.235,-18.880
     &/
       data (fjrocvz(it,25,4, 6), it=1,59) /
     &-18.292,-17.432,-15.915,-14.848,-15.649,-16.432,
     &-17.021,-17.888,-18.439,-17.795,-14.516,-7.760,
     &-9.089,-4.689,0.724,6.825,12.431,21.289,
     &28.302,31.110,30.782,29.484,27.006,22.851,
     &18.663,14.961,9.728,5.886,5.557,7.045,
     &7.710,8.040,8.603,10.331,13.013,-999.000,
     &16.734,-999.000,20.830,24.136,24.564,19.548,
     &11.919,-999.000,-0.171,-999.000,-9.434,-999.000,
     &-15.309,-999.000,-17.476,-999.000,-21.278,-999.000,
     &-22.054,-21.853,-21.480,-20.023,-18.292
     &/
       data (fjrocvz(it, 1,1, 7), it=1,59) /
     &-12.906,-12.262,-11.900,-12.015,-12.355,-13.641,
     &-15.403,-17.302,-19.683,-21.276,-20.190,-13.953,
     &-4.365,3.463,9.456,13.952,18.500,23.763,
     &28.129,30.968,31.295,29.470,27.129,24.145,
     &21.022,17.266,13.425,10.540,7.532,4.386,
     &2.486,2.023,2.270,2.018,1.907,-999.000,
     &1.360,-999.000,2.225,-999.000,4.718,-999.000,
     &3.546,-999.000,0.619,-999.000,-5.223,-999.000,
     &-9.948,-999.000,-13.248,-999.000,-15.291,-999.000,
     &-17.270,-17.835,-16.757,-14.474,-12.906
     &/
       data (fjrocvz(it, 2,1, 7), it=1,59) /
     &-15.210,-14.029,-13.641,-13.547,-14.490,-16.139,
     &-16.858,-17.991,-19.406,-20.307,-16.642,-7.066,
     &1.885,8.439,13.906,19.729,24.628,29.068,
     &33.191,35.906,36.506,34.554,31.328,27.641,
     &23.500,19.016,14.465,10.884,7.689,4.631,
     &1.808,0.180,-0.397,-0.262,-0.175,-999.000,
     &0.324,-999.000,2.211,-999.000,2.604,-999.000,
     &1.787,-999.000,-1.797,-999.000,-7.769,-999.000,
     &-12.828,-999.000,-16.589,-999.000,-18.653,-999.000,
     &-19.724,-19.673,-18.731,-17.006,-15.210
     &/
       data (fjrocvz(it, 3,1, 7), it=1,59) /
     &-17.039,-16.409,-16.203,-16.512,-17.341,-17.325,
     &-16.498,-16.613,-16.491,-15.424,-9.640,-1.062,
     &6.267,11.431,16.634,21.910,25.443,29.010,
     &32.822,35.906,36.422,33.989,30.330,26.781,
     &23.560,20.272,16.440,12.503,9.001,5.639,
     &2.573,1.052,0.422,0.911,1.064,-999.000,
     &2.165,-999.000,2.949,-999.000,4.708,6.287,
     &7.195,5.444,1.634,-1.252,-6.169,-999.000,
     &-12.447,-999.000,-15.931,-999.000,-18.042,-999.000,
     &-19.168,-19.445,-19.000,-18.043,-17.039
     &/
       data (fjrocvz(it, 4,1, 7), it=1,59) /
     &-17.155,-17.937,-17.898,-17.084,-16.194,-16.389,
     &-16.452,-16.770,-15.741,-12.574,-5.333,2.967,
     &8.282,12.237,17.584,22.134,25.868,29.289,
     &32.876,35.003,34.262,30.928,27.121,24.206,
     &21.581,17.946,14.123,11.314,9.422,7.300,
     &4.590,2.431,1.487,2.113,3.919,-999.000,
     &4.746,-999.000,6.848,-999.000,8.910,10.788,
     &12.321,8.730,3.472,-999.000,-3.860,-999.000,
     &-11.261,-999.000,-14.896,-999.000,-17.240,-999.000,
     &-18.762,-18.502,-17.300,-16.330,-17.155
     &/
       data (fjrocvz(it, 5,1, 7), it=1,59) /
     &-14.071,-14.816,-14.901,-13.941,-13.426,-14.752,
     &-16.751,-18.055,-17.418,-14.078,-6.319,2.739,
     &7.596,11.446,16.924,21.821,25.573,28.681,
     &31.968,34.016,33.503,30.681,27.716,25.031,
     &21.826,17.509,13.608,11.453,10.479,9.548,
     &7.680,5.145,3.742,3.361,4.313,-999.000,
     &6.690,-999.000,10.781,-999.000,12.780,17.172,
     &17.141,12.609,7.179,-999.000,-2.335,-999.000,
     &-10.780,-999.000,-15.019,-999.000,-17.184,-999.000,
     &-18.333,-17.177,-15.049,-13.669,-14.071
     &/
       data (fjrocvz(it, 6,1, 7), it=1,59) /
     &-12.183,-12.904,-13.615,-14.367,-14.801,-15.977,
     &-18.022,-19.459,-19.023,-14.959,-8.209,-0.161,
     &4.535,7.573,11.485,15.583,19.235,22.842,
     &27.049,30.425,30.679,29.046,26.988,24.733,
     &21.824,18.503,15.460,13.068,11.835,11.343,
     &10.690,9.304,7.012,5.516,6.044,-999.000,
     &8.305,-999.000,11.346,-999.000,15.839,-999.000,
     &16.463,-999.000,10.386,-999.000,2.227,-999.000,
     &-5.431,-999.000,-10.412,-999.000,-14.004,-999.000,
     &-15.938,-16.170,-14.479,-12.674,-12.183
     &/
       data (fjrocvz(it, 7,1, 7), it=1,59) /
     &-14.816,-15.611,-16.677,-17.641,-18.341,-18.971,
     &-19.423,-19.512,-18.961,-16.192,-12.157,-8.215,
     &-3.977,1.182,6.439,10.426,14.457,17.872,
     &22.844,26.283,27.039,25.676,23.453,21.327,
     &19.326,17.098,14.236,11.722,10.218,10.223,
     &10.929,10.491,8.802,7.445,7.741,-999.000,
     &8.972,-999.000,12.474,-999.000,17.837,20.492,
     &24.985,23.991,16.132,-999.000,6.810,-999.000,
     &-0.619,-999.000,-6.515,-999.000,-11.252,-999.000,
     &-14.521,-16.048,-15.603,-15.230,-14.816
     &/
       data (fjrocvz(it, 8,1, 7), it=1,59) /
     &-18.185,-17.760,-17.601,-17.559,-17.778,-18.216,
     &-18.912,-19.792,-20.658,-19.733,-16.813,-11.426,
     &-5.856,-2.223,0.871,3.824,7.417,11.637,
     &17.105,21.282,23.121,22.323,20.167,17.949,
     &15.729,13.609,11.318,9.008,7.901,8.153,
     &9.591,10.267,9.978,9.104,9.508,-999.000,
     &11.650,-999.000,16.194,-999.000,23.735,31.474,
     &35.290,34.807,26.804,16.373,10.558,-999.000,
     &-1.965,-999.000,-9.405,-999.000,-15.044,-999.000,
     &-17.772,-19.611,-19.671,-19.054,-18.185
     &/
       data (fjrocvz(it, 9,1, 7), it=1,59) /
     &-19.260,-18.201,-17.407,-16.873,-16.627,-17.488,
     &-18.956,-20.502,-21.719,-21.392,-19.138,-14.961,
     &-10.551,-6.191,-1.465,1.970,4.650,8.111,
     &13.115,17.550,19.810,20.073,19.104,17.621,
     &15.555,14.039,12.657,11.320,10.242,9.871,
     &10.382,10.538,10.324,10.283,11.706,12.049,
     &12.710,14.508,19.371,25.372,31.903,40.420,
     &46.399,41.712,25.143,9.728,0.099,-6.561,
     &-9.376,-999.000,-15.586,-999.000,-19.103,-999.000,
     &-21.548,-22.803,-22.208,-20.795,-19.260
     &/
       data (fjrocvz(it,10,1, 7), it=1,59) /
     &-18.073,-16.825,-15.942,-15.340,-15.759,-17.371,
     &-18.753,-19.776,-21.314,-22.805,-21.646,-16.885,
     &-10.565,-4.590,0.561,4.560,7.433,10.199,
     &13.839,17.175,19.823,21.697,22.290,21.634,
     &19.729,18.192,16.926,15.789,14.413,13.769,
     &13.387,13.003,12.571,12.387,12.762,-999.000,
     &14.541,18.306,25.817,35.020,45.182,50.833,
     &44.529,30.221,13.711,-1.560,-7.851,-12.047,
     &-14.842,-999.000,-18.840,-999.000,-21.778,-999.000,
     &-23.271,-23.080,-21.692,-20.022,-18.073
     &/
       data (fjrocvz(it,11,1, 7), it=1,59) /
     &-16.865,-15.591,-15.191,-15.656,-17.064,-19.090,
     &-20.645,-21.698,-22.971,-23.634,-21.449,-14.378,
     &-6.689,-0.339,5.157,9.152,11.615,13.277,
     &15.922,19.938,24.125,26.783,27.417,26.220,
     &24.379,22.657,21.127,19.030,17.017,15.877,
     &14.514,13.680,13.387,13.624,14.604,15.541,
     &17.363,22.408,30.044,36.999,41.253,40.563,
     &27.548,11.879,0.468,-9.020,-15.238,-999.000,
     &-17.770,-999.000,-20.907,-999.000,-22.462,-999.000,
     &-22.816,-22.020,-20.397,-18.522,-16.865
     &/
       data (fjrocvz(it,12,1, 7), it=1,59) /
     &-18.559,-18.084,-18.376,-19.707,-20.934,-21.851,
     &-22.441,-23.483,-25.153,-25.440,-23.636,-18.607,
     &-10.837,-3.145,6.064,10.752,13.726,15.897,
     &18.665,22.607,26.028,28.172,28.834,28.434,
     &27.452,25.797,23.393,20.497,17.594,15.506,
     &14.208,13.193,13.020,14.393,17.353,-999.000,
     &21.450,22.390,28.043,34.326,37.498,32.812,
     &21.893,11.544,2.693,-999.000,-12.089,-999.000,
     &-20.393,-999.000,-23.444,-999.000,-23.582,-999.000,
     &-22.743,-21.822,-20.806,-19.517,-18.559
     &/
       data (fjrocvz(it,13,1, 7), it=1,59) /
     &-19.572,-20.051,-20.735,-21.602,-21.794,-21.543,
     &-21.652,-23.170,-24.803,-25.731,-24.669,-20.034,
     &-11.225,-4.320,3.414,9.585,13.732,16.278,
     &19.219,22.692,25.520,27.089,27.277,26.641,
     &25.204,23.514,21.701,19.079,15.549,13.440,
     &13.030,12.969,13.051,13.289,13.724,-999.000,
     &17.138,-999.000,23.156,28.118,30.739,26.403,
     &18.301,-999.000,6.070,-999.000,-8.124,-999.000,
     &-18.162,-999.000,-23.131,-999.000,-23.696,-999.000,
     &-23.076,-22.123,-20.985,-19.878,-19.572
     &/
       data (fjrocvz(it,14,1, 7), it=1,59) /
     &-19.827,-21.202,-22.165,-22.324,-21.553,-21.223,
     &-22.650,-24.612,-25.564,-25.964,-22.977,-19.623,
     &-13.528,-7.664,0.452,7.615,12.681,15.162,
     &17.697,20.988,24.350,25.951,25.789,24.396,
     &22.859,21.489,19.729,16.713,13.431,11.369,
     &11.031,11.073,10.547,8.762,8.354,-999.000,
     &10.799,12.837,18.223,22.108,23.653,21.989,
     &17.712,10.061,5.056,-999.000,-6.254,-999.000,
     &-15.953,-999.000,-22.417,-999.000,-24.185,-999.000,
     &-24.092,-22.908,-21.132,-19.936,-19.827
     &/
       data (fjrocvz(it,15,1, 7), it=1,59) /
     &-20.416,-21.121,-22.473,-23.375,-22.831,-22.113,
     &-22.829,-24.478,-26.788,-28.552,-25.883,-20.741,
     &-12.930,-7.291,-0.743,5.728,10.982,14.053,
     &16.539,18.379,20.226,21.181,20.602,19.171,
     &17.997,17.115,15.911,14.136,11.471,9.175,
     &8.475,8.273,7.517,6.921,7.206,-999.000,
     &9.533,-999.000,15.831,21.379,25.427,24.624,
     &19.837,12.248,6.852,-999.000,-4.361,-999.000,
     &-14.245,-999.000,-21.958,-999.000,-24.504,-999.000,
     &-24.161,-23.218,-21.967,-21.005,-20.416
     &/
       data (fjrocvz(it,16,1, 7), it=1,59) /
     &-22.169,-22.343,-23.088,-23.947,-23.628,-22.597,
     &-22.607,-23.477,-26.001,-30.813,-31.344,-25.830,
     &-10.822,-3.401,1.093,6.146,10.816,13.639,
     &16.101,18.020,19.924,20.151,19.526,18.279,
     &17.019,15.750,14.315,12.126,9.186,7.261,
     &7.142,7.653,7.921,8.117,8.905,-999.000,
     &12.038,-999.000,18.396,23.825,27.457,24.730,
     &18.617,-999.000,6.171,-999.000,-3.936,-999.000,
     &-13.988,-999.000,-21.574,-999.000,-24.128,-999.000,
     &-24.473,-23.817,-23.101,-22.702,-22.169
     &/
       data (fjrocvz(it,17,1, 7), it=1,59) /
     &-22.313,-22.190,-22.390,-23.236,-23.825,-24.014,
     &-24.282,-23.181,-23.576,-27.633,-30.066,-25.526,
     &-10.307,0.673,4.423,7.697,11.104,13.649,
     &16.073,18.171,19.829,20.908,20.970,19.959,
     &18.851,17.378,15.675,13.390,10.379,8.269,
     &7.951,8.610,8.705,8.413,9.612,-999.000,
     &11.970,14.972,17.790,21.586,24.093,23.042,
     &16.567,-999.000,1.404,-999.000,-9.324,-999.000,
     &-18.350,-999.000,-24.241,-999.000,-26.419,-999.000,
     &-25.828,-24.046,-22.869,-22.598,-22.313
     &/
       data (fjrocvz(it,18,1, 7), it=1,59) /
     &-21.636,-21.366,-21.572,-22.723,-24.321,-25.219,
     &-25.824,-25.595,-24.968,-25.404,-24.481,-19.883,
     &-7.951,2.977,7.431,10.418,13.783,16.539,
     &18.925,19.906,20.919,22.180,22.203,20.987,
     &19.347,17.719,15.793,13.951,11.715,9.896,
     &9.052,9.184,9.455,9.718,10.405,-999.000,
     &13.105,-999.000,16.970,22.067,24.018,20.516,
     &12.155,3.156,-4.116,-999.000,-13.579,-999.000,
     &-21.034,-999.000,-26.664,-999.000,-28.261,-999.000,
     &-26.788,-24.407,-23.115,-22.401,-21.636
     &/
       data (fjrocvz(it,19,1, 7), it=1,59) /
     &-22.546,-21.448,-20.556,-20.755,-21.228,-21.320,
     &-20.911,-21.504,-22.268,-24.164,-23.881,-20.044,
     &-10.067,1.451,8.172,12.671,16.487,19.880,
     &22.612,23.864,24.962,25.853,26.131,24.692,
     &23.029,21.629,19.053,15.572,12.916,11.820,
     &11.603,11.371,11.105,10.194,10.025,-999.000,
     &12.179,-999.000,15.523,18.856,18.196,12.341,
     &6.816,-999.000,-5.039,-999.000,-13.951,-999.000,
     &-21.584,-999.000,-26.510,-999.000,-27.966,-999.000,
     &-26.735,-24.772,-23.682,-23.281,-22.546
     &/
       data (fjrocvz(it,20,1, 7), it=1,59) /
     &-21.460,-20.222,-19.758,-19.708,-19.798,-19.614,
     &-19.084,-19.099,-19.948,-21.907,-24.310,-22.129,
     &-13.449,-0.841,6.220,10.245,14.788,19.239,
     &22.708,24.600,25.798,26.764,27.079,26.311,
     &25.909,25.468,23.504,19.932,16.146,14.433,
     &14.004,13.411,12.073,10.521,10.075,-999.000,
     &9.877,-999.000,10.774,12.937,16.040,12.040,
     &3.869,-999.000,-6.575,-999.000,-16.010,-999.000,
     &-23.273,-999.000,-27.966,-999.000,-29.491,-999.000,
     &-28.968,-27.112,-25.142,-23.232,-21.460
     &/
       data (fjrocvz(it,21,1, 7), it=1,59) /
     &-22.703,-21.760,-21.731,-22.167,-22.300,-21.408,
     &-19.792,-19.966,-22.050,-25.601,-28.659,-26.875,
     &-17.417,-4.692,2.566,6.740,11.197,15.155,
     &18.857,21.690,23.619,24.272,24.174,24.128,
     &24.599,24.406,22.566,18.918,14.818,12.773,
     &12.356,11.994,10.905,9.112,8.066,-999.000,
     &8.699,-999.000,10.034,12.049,13.227,8.781,
     &0.427,-999.000,-9.614,-999.000,-17.648,-999.000,
     &-23.834,-999.000,-28.476,-999.000,-29.993,-999.000,
     &-29.803,-28.524,-26.760,-24.627,-22.703
     &/
       data (fjrocvz(it,22,1, 7), it=1,59) /
     &-21.073,-20.395,-20.643,-21.691,-22.168,-21.564,
     &-20.306,-20.732,-22.741,-25.906,-28.205,-26.416,
     &-17.403,-5.696,1.553,5.729,9.087,13.304,
     &17.288,20.925,22.526,22.906,23.406,23.639,
     &23.454,22.250,19.271,14.549,9.833,7.523,
     &7.290,7.381,6.538,5.156,4.835,-999.000,
     &5.322,-999.000,4.792,-999.000,4.973,-999.000,
     &0.333,-999.000,-10.879,-999.000,-16.791,-999.000,
     &-21.543,-999.000,-25.765,-999.000,-27.547,-999.000,
     &-27.364,-25.885,-23.869,-22.295,-21.073
     &/
       data (fjrocvz(it,23,1, 7), it=1,59) /
     &-16.105,-16.096,-16.481,-17.368,-18.323,-18.492,
     &-18.479,-19.218,-20.242,-21.541,-21.850,-18.438,
     &-10.792,-2.526,3.632,6.959,9.766,13.026,
     &16.853,20.515,23.039,25.171,26.267,25.714,
     &23.971,21.493,18.666,14.889,10.267,7.038,
     &5.083,4.037,3.153,2.466,2.458,-999.000,
     &3.096,-999.000,6.262,-999.000,7.038,-999.000,
     &4.212,-999.000,-0.801,-999.000,-10.551,-999.000,
     &-16.003,-999.000,-20.481,-999.000,-22.287,-999.000,
     &-22.198,-20.865,-18.820,-16.883,-16.105
     &/
       data (fjrocvz(it,24,1, 7), it=1,59) /
     &-13.100,-12.951,-13.019,-13.732,-14.192,-14.265,
     &-14.699,-16.204,-17.891,-19.695,-19.572,-15.155,
     &-7.018,0.412,5.650,9.616,12.811,16.223,
     &19.890,23.368,25.639,26.197,25.617,24.091,
     &21.816,19.334,16.785,14.078,10.223,6.758,
     &4.332,3.458,3.386,2.964,2.412,-999.000,
     &3.193,-999.000,4.901,-999.000,6.608,-999.000,
     &5.512,-999.000,1.519,-999.000,-4.594,-999.000,
     &-10.297,-999.000,-14.174,-999.000,-16.012,-999.000,
     &-16.961,-17.058,-15.855,-14.155,-13.100
     &/
       data (fjrocvz(it,25,1, 7), it=1,59) /
     &-12.906,-12.262,-11.900,-12.015,-12.355,-13.641,
     &-15.403,-17.302,-19.683,-21.276,-20.190,-13.953,
     &-4.365,3.463,9.456,13.952,18.500,23.763,
     &28.129,30.968,31.295,29.470,27.129,24.145,
     &21.022,17.266,13.425,10.540,7.532,4.386,
     &2.486,2.023,2.270,2.018,1.907,-999.000,
     &1.360,-999.000,2.225,-999.000,4.718,-999.000,
     &3.546,-999.000,0.619,-999.000,-5.223,-999.000,
     &-9.948,-999.000,-13.248,-999.000,-15.291,-999.000,
     &-17.270,-17.835,-16.757,-14.474,-12.906
     &/
       data (fjrocvz(it, 1,2, 7), it=1,59) /
     &-19.431,-18.560,-17.039,-15.966,-16.756,-17.604,
     &-18.225,-19.124,-19.740,-19.089,-15.807,-9.047,
     &-10.370,-5.728,-0.194,6.028,11.876,20.837,
     &27.902,30.761,30.537,29.354,26.933,22.835,
     &18.763,15.347,10.257,6.558,6.516,8.127,
     &8.853,9.244,9.929,11.806,14.637,-999.000,
     &18.445,-999.000,22.921,26.887,27.578,22.460,
     &14.367,-999.000,0.259,-999.000,-10.500,-999.000,
     &-16.848,-999.000,-19.216,-999.000,-22.793,-999.000,
     &-23.344,-23.137,-22.757,-21.231,-19.431
     &/
       data (fjrocvz(it, 2,2, 7), it=1,59) /
     &-17.841,-17.133,-16.231,-15.791,-15.106,-14.519,
     &-14.736,-16.364,-17.653,-16.991,-11.638,-3.738,
     &1.837,5.454,9.992,12.519,16.956,22.497,
     &28.032,31.868,32.927,32.090,29.351,25.120,
     &21.031,18.298,14.743,11.292,9.750,9.999,
     &11.236,12.155,13.175,15.018,16.883,-999.000,
     &19.264,-999.000,24.168,27.121,29.097,28.302,
     &22.213,-999.000,0.768,-999.000,-13.771,-999.000,
     &-16.414,-999.000,-21.622,-999.000,-24.841,-999.000,
     &-23.860,-23.336,-21.465,-19.048,-17.841
     &/
       data (fjrocvz(it, 3,2, 7), it=1,59) /
     &-18.023,-16.819,-16.154,-15.021,-14.015,-14.230,
     &-13.867,-14.024,-14.693,-14.066,-9.587,-3.128,
     &2.792,6.846,9.430,12.681,16.994,20.650,
     &24.241,27.110,28.823,28.252,25.413,21.749,
     &18.623,16.644,14.498,12.377,10.923,10.889,
     &11.990,13.048,14.586,15.612,17.547,-999.000,
     &20.840,-999.000,23.924,29.213,33.004,31.386,
     &21.925,9.782,0.181,-11.604,-18.324,-18.644,
     &-19.766,-999.000,-24.832,-999.000,-26.134,-999.000,
     &-25.532,-23.710,-21.498,-19.633,-18.023
     &/
       data (fjrocvz(it, 4,2, 7), it=1,59) /
     &-19.259,-18.092,-16.494,-14.901,-14.506,-14.651,
     &-13.976,-12.611,-11.989,-11.210,-8.167,-2.692,
     &3.305,6.278,7.710,10.549,14.625,19.325,
     &23.044,25.294,25.436,23.816,21.115,17.952,
     &15.344,13.452,11.608,10.497,10.512,10.744,
     &11.652,13.191,14.515,16.589,20.020,-999.000,
     &22.116,-999.000,25.671,31.284,33.011,32.092,
     &26.904,15.605,-0.044,-15.650,-21.593,-999.000,
     &-22.210,-999.000,-25.777,-999.000,-28.012,-999.000,
     &-26.401,-23.742,-21.880,-20.224,-19.259
     &/
       data (fjrocvz(it, 5,2, 7), it=1,59) /
     &-19.655,-18.334,-16.052,-14.388,-14.188,-15.039,
     &-14.958,-13.847,-13.422,-12.559,-9.562,-3.357,
     &2.606,6.041,8.926,11.275,15.372,20.762,
     &25.200,27.823,28.127,26.489,23.190,19.647,
     &16.711,14.825,13.319,12.630,12.199,11.852,
     &12.574,14.008,15.850,17.815,19.385,-999.000,
     &23.215,-999.000,27.194,31.310,34.399,36.624,
     &32.247,16.736,-0.595,-12.339,-21.024,-24.422,
     &-27.390,-30.244,-31.221,-30.630,-28.741,-999.000,
     &-26.238,-23.675,-21.675,-20.337,-19.655
     &/
       data (fjrocvz(it, 6,2, 7), it=1,59) /
     &-20.492,-19.622,-18.151,-16.760,-16.444,-16.445,
     &-15.312,-14.590,-14.264,-12.477,-10.297,-8.898,
     &-5.615,0.819,7.510,12.731,16.984,22.143,
     &26.518,29.129,30.483,29.574,27.160,23.824,
     &21.121,19.279,17.440,15.621,14.422,13.245,
     &12.754,14.454,16.542,17.467,19.385,-999.000,
     &23.532,-999.000,27.666,34.446,40.172,41.875,
     &35.835,21.709,4.019,-9.414,-19.428,-25.658,
     &-28.565,-999.000,-30.891,-999.000,-30.071,-999.000,
     &-27.501,-24.096,-22.253,-21.432,-20.492
     &/
       data (fjrocvz(it, 7,2, 7), it=1,59) /
     &-22.391,-22.010,-20.977,-20.050,-19.478,-18.021,
     &-16.358,-15.310,-14.061,-13.014,-10.865,-8.485,
     &-3.290,0.423,4.705,8.490,13.157,18.964,
     &24.163,28.168,29.762,29.845,28.603,26.302,
     &24.038,21.681,19.356,17.446,15.304,12.975,
     &12.221,13.042,14.393,16.148,17.666,-999.000,
     &20.548,-999.000,25.398,33.325,40.264,44.292,
     &34.944,17.455,1.585,-9.188,-19.187,-27.432,
     &-32.240,-33.405,-33.060,-32.341,-31.144,-999.000,
     &-28.229,-25.974,-24.313,-22.859,-22.391
     &/
       data (fjrocvz(it, 8,2, 7), it=1,59) /
     &-22.356,-21.109,-20.265,-19.871,-19.309,-18.835,
     &-18.210,-17.542,-16.241,-18.091,-19.929,-21.637,
     &-16.387,-8.222,-2.193,3.748,7.901,12.020,
     &18.058,22.764,25.548,26.061,25.481,24.290,
     &22.628,20.796,18.525,16.336,13.661,11.295,
     &10.253,10.530,11.906,13.481,14.771,-999.000,
     &18.084,-999.000,23.977,29.491,35.995,36.765,
     &26.341,13.588,1.918,-9.270,-17.254,-24.449,
     &-28.932,-29.564,-27.593,-999.000,-27.581,-999.000,
     &-26.526,-26.216,-25.116,-23.612,-22.356
     &/
       data (fjrocvz(it, 9,2, 7), it=1,59) /
     &-19.729,-17.741,-15.964,-14.888,-14.937,-15.810,
     &-17.493,-18.959,-20.006,-21.817,-24.443,-26.402,
     &-21.146,-10.681,-2.462,3.042,5.786,9.696,
     &14.752,18.597,20.750,21.468,21.304,19.958,
     &17.705,15.621,13.733,11.758,9.616,8.326,
     &8.241,8.861,10.062,11.325,12.933,-999.000,
     &15.523,-999.000,22.264,28.595,34.304,37.461,
     &34.442,22.156,9.600,-1.784,-9.714,-999.000,
     &-15.758,-999.000,-19.215,-999.000,-21.908,-999.000,
     &-22.366,-22.443,-22.635,-21.755,-19.729
     &/
       data (fjrocvz(it,10,2, 7), it=1,59) /
     &-20.317,-18.053,-15.756,-14.539,-13.886,-15.132,
     &-18.287,-20.999,-22.558,-23.374,-24.027,-23.784,
     &-19.591,-8.463,-0.249,5.212,7.889,12.137,
     &17.358,20.755,21.550,21.287,20.104,18.519,
     &16.510,13.945,11.623,9.701,8.456,7.940,
     &8.032,8.405,8.499,9.126,10.943,-999.000,
     &14.781,-999.000,22.611,29.870,37.941,41.957,
     &39.797,27.377,15.028,-999.000,1.225,-999.000,
     &-8.391,-999.000,-15.061,-999.000,-17.070,-999.000,
     &-18.865,-19.833,-20.841,-20.940,-20.317
     &/
       data (fjrocvz(it,11,2, 7), it=1,59) /
     &-22.932,-22.097,-21.632,-21.532,-21.340,-21.809,
     &-22.586,-24.081,-25.873,-27.824,-27.125,-23.323,
     &-17.793,-7.809,-0.101,6.082,10.475,14.129,
     &18.643,22.457,24.232,24.370,23.420,21.951,
     &20.473,18.181,15.289,13.229,11.763,10.828,
     &10.197,9.465,8.960,9.498,11.832,-999.000,
     &16.646,-999.000,23.496,30.539,39.255,40.223,
     &35.870,25.677,15.379,-999.000,-0.478,-999.000,
     &-10.937,-999.000,-17.456,-999.000,-18.967,-999.000,
     &-21.658,-21.512,-22.409,-23.544,-22.932
     &/
       data (fjrocvz(it,12,2, 7), it=1,59) /
     &-24.655,-25.235,-25.070,-24.911,-25.012,-24.597,
     &-24.515,-26.157,-28.974,-31.058,-31.766,-28.451,
     &-24.149,-14.300,-4.463,1.535,6.382,11.850,
     &18.721,23.864,26.375,26.651,25.787,24.790,
     &23.635,21.275,18.259,15.859,14.681,13.944,
     &13.013,11.755,10.848,11.375,14.118,-999.000,
     &18.505,-999.000,24.981,31.337,39.035,43.648,
     &38.444,21.919,8.930,-0.092,-8.024,-999.000,
     &-13.071,-999.000,-17.586,-999.000,-21.497,-999.000,
     &-23.921,-23.907,-24.184,-24.574,-24.655
     &/
       data (fjrocvz(it,13,2, 7), it=1,59) /
     &-23.570,-22.997,-22.726,-22.671,-22.278,-22.910,
     &-24.447,-25.758,-27.163,-27.819,-28.228,-26.731,
     &-23.831,-15.096,-6.240,0.455,4.987,11.183,
     &20.624,26.600,28.189,28.264,27.468,26.619,
     &25.340,22.696,19.825,17.910,17.518,17.329,
     &16.629,15.357,14.009,14.246,16.520,-999.000,
     &19.994,-999.000,28.565,34.534,42.726,47.783,
     &39.874,22.753,5.395,-4.658,-10.287,-12.276,
     &-13.290,-999.000,-16.947,-999.000,-19.853,-999.000,
     &-22.143,-24.152,-24.062,-23.549,-23.570
     &/
       data (fjrocvz(it,14,2, 7), it=1,59) /
     &-21.684,-21.202,-20.840,-20.238,-20.173,-21.775,
     &-24.539,-28.094,-31.246,-31.031,-28.138,-24.139,
     &-20.275,-16.615,-9.604,-1.722,4.077,10.855,
     &20.337,26.138,28.303,28.458,27.493,26.297,
     &24.496,22.468,20.339,19.023,18.958,18.839,
     &17.909,16.716,15.542,15.783,18.201,-999.000,
     &22.894,-999.000,31.374,40.005,45.522,45.102,
     &36.808,22.004,8.532,-999.000,-8.563,-999.000,
     &-16.284,-999.000,-19.053,-999.000,-21.586,-999.000,
     &-24.187,-24.960,-24.002,-22.510,-21.684
     &/
       data (fjrocvz(it,15,2, 7), it=1,59) /
     &-19.576,-19.048,-18.481,-18.014,-18.013,-19.644,
     &-23.580,-29.619,-33.780,-35.102,-31.705,-24.524,
     &-19.695,-14.616,-9.153,-1.600,5.032,11.225,
     &18.789,24.046,26.294,26.102,24.442,22.026,
     &19.730,18.056,16.128,14.869,14.558,14.214,
     &13.555,12.789,12.724,13.899,17.094,-999.000,
     &21.277,-999.000,28.497,36.040,41.416,43.465,
     &36.314,20.636,7.223,-999.000,-10.765,-999.000,
     &-17.909,-999.000,-22.121,-999.000,-25.209,-999.000,
     &-26.854,-27.070,-24.616,-21.214,-19.576
     &/
       data (fjrocvz(it,16,2, 7), it=1,59) /
     &-19.455,-18.277,-18.452,-18.680,-19.420,-21.547,
     &-25.283,-29.985,-33.612,-36.476,-35.388,-27.875,
     &-18.398,-12.927,-7.006,-0.404,5.899,11.346,
     &18.670,23.673,25.492,25.552,24.180,21.026,
     &17.249,13.952,11.436,9.963,9.678,9.781,
     &9.221,8.358,8.250,9.275,11.856,-999.000,
     &16.141,-999.000,23.073,30.029,36.020,37.595,
     &29.596,13.239,3.055,-999.000,-12.391,-999.000,
     &-20.198,-999.000,-24.980,-999.000,-27.836,-999.000,
     &-29.092,-28.582,-26.023,-22.372,-19.455
     &/
       data (fjrocvz(it,17,2, 7), it=1,59) /
     &-22.650,-21.940,-21.269,-20.849,-21.556,-23.287,
     &-26.434,-31.028,-34.854,-36.412,-34.680,-26.574,
     &-17.633,-11.428,-4.652,0.982,6.496,12.253,
     &20.079,25.422,27.849,28.523,27.438,24.650,
     &20.357,16.197,12.449,9.782,9.140,8.844,
     &7.931,6.663,5.452,5.325,7.787,-999.000,
     &13.079,-999.000,20.969,26.247,30.693,31.591,
     &25.153,10.919,-1.250,-12.264,-20.071,-23.965,
     &-25.921,-999.000,-27.136,-999.000,-29.142,-999.000,
     &-29.739,-28.544,-26.897,-24.618,-22.650
     &/
       data (fjrocvz(it,18,2, 7), it=1,59) /
     &-24.042,-22.751,-21.579,-21.266,-21.150,-22.077,
     &-25.761,-30.424,-33.372,-34.845,-32.299,-25.010,
     &-16.015,-9.852,-4.681,1.060,7.042,14.474,
     &21.765,27.619,31.577,32.386,30.231,26.387,
     &22.714,19.169,15.043,11.401,9.757,8.524,
     &7.347,5.842,4.730,4.624,6.901,-999.000,
     &11.612,15.275,20.327,25.572,28.989,26.363,
     &18.980,9.102,-2.743,-999.000,-20.025,-999.000,
     &-27.261,-999.000,-30.239,-999.000,-30.899,-999.000,
     &-30.098,-29.624,-28.087,-25.912,-24.042
     &/
       data (fjrocvz(it,19,2, 7), it=1,59) /
     &-22.643,-21.114,-20.987,-20.450,-19.879,-20.767,
     &-23.535,-26.302,-28.764,-30.338,-27.705,-20.903,
     &-12.861,-8.438,-3.317,2.487,8.221,16.196,
     &24.508,31.125,35.399,35.969,33.457,29.568,
     &25.453,21.379,17.212,13.506,12.263,11.531,
     &10.058,8.272,7.060,6.613,8.466,-999.000,
     &13.922,-999.000,22.100,27.359,29.330,25.558,
     &19.948,10.139,-3.769,-15.013,-23.363,-27.311,
     &-28.637,-999.000,-30.289,-999.000,-29.975,-999.000,
     &-29.229,-28.866,-27.611,-25.201,-22.643
     &/
       data (fjrocvz(it,20,2, 7), it=1,59) /
     &-24.355,-22.895,-21.213,-19.475,-18.263,-18.563,
     &-20.726,-23.856,-25.681,-26.372,-23.845,-14.824,
     &-6.814,-4.793,-0.260,4.179,8.892,16.997,
     &25.005,32.460,36.949,37.864,36.007,32.627,
     &28.947,25.341,21.030,17.405,15.885,14.564,
     &12.814,10.944,9.462,9.033,11.380,-999.000,
     &16.422,-999.000,22.990,27.945,32.633,33.325,
     &25.437,10.649,-3.578,-12.052,-19.106,-24.557,
     &-27.845,-999.000,-29.084,-999.000,-29.983,-999.000,
     &-29.601,-29.231,-28.373,-26.378,-24.355
     &/
       data (fjrocvz(it,21,2, 7), it=1,59) /
     &-24.361,-22.009,-20.136,-18.941,-18.216,-18.040,
     &-19.208,-21.950,-24.850,-25.566,-22.778,-15.524,
     &-10.122,-5.243,-0.166,5.067,10.594,16.959,
     &22.908,29.651,33.567,35.076,33.376,29.912,
     &26.527,23.648,20.024,16.002,13.834,12.248,
     &10.597,9.464,9.047,9.197,11.113,-999.000,
     &15.508,-999.000,22.852,28.062,33.284,32.030,
     &23.006,-999.000,0.678,-999.000,-15.161,-999.000,
     &-23.851,-999.000,-27.668,-999.000,-28.747,-999.000,
     &-28.814,-28.345,-27.624,-26.261,-24.361
     &/
       data (fjrocvz(it,22,2, 7), it=1,59) /
     &-23.238,-22.154,-21.005,-20.167,-20.078,-19.953,
     &-20.233,-22.497,-26.372,-27.227,-24.221,-17.345,
     &-9.241,-6.137,-0.610,4.316,9.286,15.173,
     &21.630,27.569,30.428,30.819,28.824,24.958,
     &20.447,16.681,13.236,10.242,8.577,7.440,
     &6.683,7.115,8.089,8.734,10.843,-999.000,
     &16.021,-999.000,23.533,26.038,29.229,25.845,
     &18.254,-999.000,-2.084,-999.000,-16.278,-999.000,
     &-24.942,-999.000,-26.629,-999.000,-26.894,-999.000,
     &-27.435,-26.921,-26.040,-24.654,-23.238
     &/
       data (fjrocvz(it,23,2, 7), it=1,59) /
     &-21.505,-21.060,-19.561,-17.606,-17.031,-17.941,
     &-19.791,-22.126,-24.279,-24.919,-22.311,-16.125,
     &-10.917,-8.017,-2.594,2.666,7.928,13.516,
     &21.630,27.387,29.279,28.438,25.870,21.868,
     &17.395,13.066,9.508,7.000,6.013,5.534,
     &5.222,5.245,6.763,8.962,12.060,-999.000,
     &15.836,-999.000,22.388,26.181,28.254,26.229,
     &19.668,8.042,-2.028,-999.000,-15.332,-999.000,
     &-22.177,-999.000,-24.849,-999.000,-25.506,-999.000,
     &-26.074,-25.566,-24.186,-22.232,-21.505
     &/
       data (fjrocvz(it,24,2, 7), it=1,59) /
     &-19.867,-18.409,-16.676,-14.599,-14.451,-16.236,
     &-18.382,-19.671,-20.749,-21.358,-19.233,-14.358,
     &-8.246,-3.321,0.128,3.395,8.018,16.280,
     &24.473,28.104,28.865,27.880,24.853,20.532,
     &16.281,12.176,8.159,5.451,5.393,6.576,
     &6.954,6.703,8.048,10.204,12.911,-999.000,
     &16.385,-999.000,22.112,26.958,29.482,26.582,
     &17.510,-999.000,0.883,-999.000,-10.064,-999.000,
     &-16.968,-999.000,-21.824,-999.000,-22.772,-999.000,
     &-23.987,-23.893,-22.766,-21.359,-19.867
     &/
       data (fjrocvz(it,25,2, 7), it=1,59) /
     &-19.431,-18.560,-17.039,-15.966,-16.756,-17.604,
     &-18.225,-19.124,-19.740,-19.089,-15.807,-9.047,
     &-10.370,-5.728,-0.194,6.028,11.876,20.837,
     &27.902,30.761,30.537,29.354,26.933,22.835,
     &18.763,15.347,10.257,6.558,6.516,8.127,
     &8.853,9.244,9.929,11.806,14.637,-999.000,
     &18.445,-999.000,22.921,26.887,27.578,22.460,
     &14.367,-999.000,0.259,-999.000,-10.500,-999.000,
     &-16.848,-999.000,-19.216,-999.000,-22.793,-999.000,
     &-23.344,-23.137,-22.757,-21.231,-19.431
     &/
       data (fjrocvz(it, 1,3, 7), it=1,59) /
     &-21.380,-22.106,-21.633,-20.459,-18.575,-17.661,
     &-17.731,-21.114,-28.216,-29.947,-22.993,-9.261,
     &2.241,6.306,8.982,11.013,10.860,9.939,
     &10.309,12.934,16.322,17.290,16.742,14.347,
     &11.052,9.361,9.864,10.822,10.697,9.789,
     &9.071,8.980,8.766,8.274,8.737,-999.000,
     &11.611,-999.000,14.317,-999.000,19.955,23.512,
     &23.927,19.406,9.345,-999.000,-2.790,-999.000,
     &-9.670,-999.000,-12.408,-999.000,-14.157,-999.000,
     &-16.337,-18.709,-20.239,-20.557,-21.380
     &/
       data (fjrocvz(it, 2,3, 7), it=1,59) /
     &-19.594,-19.944,-19.466,-18.011,-16.785,-17.253,
     &-18.866,-22.506,-27.698,-28.689,-20.516,-4.873,
     &8.514,11.993,13.297,14.311,14.483,14.412,
     &15.316,16.967,18.531,19.089,18.189,16.221,
     &14.531,13.449,13.793,14.484,14.839,14.375,
     &13.726,13.309,12.929,13.335,13.525,-999.000,
     &14.447,-999.000,17.579,-999.000,21.827,24.673,
     &23.703,18.617,7.143,-999.000,-3.717,-999.000,
     &-10.534,-999.000,-13.854,-999.000,-15.831,-999.000,
     &-17.715,-19.854,-20.407,-19.619,-19.594
     &/
       data (fjrocvz(it, 3,3, 7), it=1,59) /
     &-18.720,-17.754,-17.210,-16.303,-15.935,-16.672,
     &-18.370,-20.705,-24.837,-28.367,-21.196,-2.121,
     &15.300,17.507,16.164,15.068,14.313,14.516,
     &15.556,16.414,16.228,15.256,13.859,13.154,
     &13.079,12.561,13.283,14.318,15.355,16.387,
     &16.267,15.686,15.300,15.851,16.277,-999.000,
     &17.782,-999.000,18.827,-999.000,21.433,21.471,
     &19.093,15.945,7.138,-999.000,-5.000,-999.000,
     &-12.092,-999.000,-16.284,-999.000,-18.910,-999.000,
     &-21.327,-23.249,-23.298,-20.989,-18.720
     &/
       data (fjrocvz(it, 4,3, 7), it=1,59) /
     &-20.432,-18.422,-17.745,-17.733,-17.097,-16.892,
     &-16.879,-17.960,-21.745,-24.327,-16.954,3.018,
     &17.800,20.151,18.937,18.195,17.142,15.506,
     &14.084,13.171,12.674,12.067,11.899,12.328,
     &12.489,12.586,13.156,14.350,15.763,16.477,
     &16.868,16.936,16.347,16.889,18.946,-999.000,
     &19.434,-999.000,19.278,-999.000,22.088,22.729,
     &20.471,11.882,4.258,-999.000,-5.980,-999.000,
     &-13.665,-999.000,-18.723,-999.000,-22.063,-999.000,
     &-24.004,-25.223,-25.136,-22.944,-20.432
     &/
       data (fjrocvz(it, 5,3, 7), it=1,59) /
     &-21.853,-20.680,-19.955,-18.155,-16.063,-15.520,
     &-15.500,-16.822,-18.849,-19.193,-13.141,1.304,
     &16.026,20.959,20.867,20.295,19.093,17.494,
     &15.944,15.022,15.333,15.552,15.487,14.719,
     &13.729,12.704,12.245,12.765,13.503,14.352,
     &15.394,16.217,16.351,17.361,18.519,-999.000,
     &19.365,-999.000,20.935,23.794,25.841,21.902,
     &16.759,9.068,1.582,-999.000,-8.267,-999.000,
     &-15.261,-999.000,-20.219,-999.000,-23.334,-999.000,
     &-25.675,-26.977,-26.215,-23.802,-21.853
     &/
       data (fjrocvz(it, 6,3, 7), it=1,59) /
     &-23.677,-23.797,-23.645,-21.212,-17.360,-15.171,
     &-15.158,-16.509,-17.053,-15.732,-11.361,-0.996,
     &12.827,17.716,17.651,17.161,17.043,16.963,
     &16.731,17.947,19.586,20.015,19.197,17.942,
     &16.504,14.734,13.474,12.519,12.790,13.362,
     &14.046,14.591,15.334,16.058,16.700,-999.000,
     &19.363,-999.000,20.664,22.874,21.970,15.935,
     &8.756,-999.000,-2.827,-999.000,-11.284,-999.000,
     &-18.295,-999.000,-23.212,-999.000,-26.328,-999.000,
     &-28.818,-29.853,-28.149,-25.552,-23.677
     &/
       data (fjrocvz(it, 7,3, 7), it=1,59) /
     &-26.562,-25.038,-24.653,-23.772,-20.956,-19.585,
     &-19.097,-18.528,-18.210,-15.256,-11.058,-3.825,
     &3.709,8.519,10.079,11.229,12.301,13.340,
     &14.746,16.915,19.508,20.990,21.366,21.192,
     &20.754,20.105,18.529,16.657,15.767,15.028,
     &14.676,14.152,13.865,13.752,13.963,-999.000,
     &14.727,-999.000,14.783,-999.000,11.219,-999.000,
     &2.835,-999.000,-4.722,-999.000,-12.312,-999.000,
     &-19.387,-999.000,-24.587,-999.000,-28.131,-999.000,
     &-30.658,-31.773,-30.936,-29.053,-26.562
     &/
       data (fjrocvz(it, 8,3, 7), it=1,59) /
     &-23.928,-22.793,-23.878,-24.447,-23.369,-20.534,
     &-16.336,-13.797,-13.002,-11.622,-8.632,-4.747,
     &-2.171,0.344,1.936,3.835,5.998,7.681,
     &9.327,11.326,14.680,18.303,20.286,20.895,
     &21.154,20.461,18.872,16.701,15.059,14.433,
     &13.887,13.046,11.900,10.335,8.974,-999.000,
     &8.695,-999.000,7.990,-999.000,5.599,-999.000,
     &1.740,-999.000,-1.787,-999.000,-7.643,-999.000,
     &-14.353,-999.000,-19.805,-999.000,-24.744,-999.000,
     &-28.259,-29.546,-28.369,-26.130,-23.928
     &/
       data (fjrocvz(it, 9,3, 7), it=1,59) /
     &-17.993,-18.254,-18.611,-19.164,-19.081,-16.839,
     &-13.259,-11.401,-11.534,-11.960,-11.118,-8.713,
     &-8.113,-3.745,1.005,5.513,10.140,11.164,
     &12.057,12.942,14.006,13.816,15.195,16.407,
     &17.408,17.471,15.755,13.203,11.742,10.609,
     &9.358,8.056,6.975,6.272,5.782,-999.000,
     &4.738,-999.000,3.641,-999.000,4.318,-999.000,
     &5.022,-999.000,4.947,-999.000,0.525,-999.000,
     &-4.353,-999.000,-9.419,-999.000,-12.841,-999.000,
     &-15.908,-17.658,-17.909,-17.925,-17.993
     &/
       data (fjrocvz(it,10,3, 7), it=1,59) /
     &-15.282,-15.912,-15.179,-14.217,-14.770,-15.956,
     &-15.642,-14.566,-14.277,-14.177,-13.884,-14.875,
     &-16.619,-8.462,0.320,6.379,12.674,13.662,
     &14.588,15.275,16.105,16.548,16.540,16.547,
     &16.689,16.910,15.396,13.351,11.423,9.298,
     &7.123,4.648,3.004,2.066,0.903,-999.000,
     &-0.401,-999.000,0.140,-999.000,1.902,-999.000,
     &5.692,-999.000,7.689,-999.000,6.889,-999.000,
     &3.128,-999.000,-1.714,-999.000,-5.924,-999.000,
     &-9.027,-11.665,-13.296,-14.363,-15.282
     &/
       data (fjrocvz(it,11,3, 7), it=1,59) /
     &-18.763,-20.329,-20.556,-19.829,-19.480,-18.907,
     &-17.367,-17.088,-20.284,-18.994,-18.645,-12.566,
     &-15.758,-9.505,-3.216,3.436,10.335,12.590,
     &14.642,16.284,19.459,21.767,21.225,20.532,
     &20.204,19.461,18.402,16.999,14.674,11.950,
     &8.922,6.273,4.676,2.540,0.458,-999.000,
     &-0.616,-999.000,-0.314,-999.000,2.242,-999.000,
     &9.597,12.640,12.741,10.000,7.921,-999.000,
     &4.140,-999.000,0.275,-999.000,-4.142,-999.000,
     &-8.060,-12.643,-15.498,-17.178,-18.763
     &/
       data (fjrocvz(it,12,3, 7), it=1,59) /
     &-19.644,-21.863,-24.249,-25.997,-25.773,-23.656,
     &-23.279,-27.341,-32.417,-30.120,-24.437,-14.772,
     &-13.660,-10.034,-6.448,-0.249,6.204,10.013,
     &13.338,17.574,21.975,23.989,24.073,23.530,
     &22.832,21.489,20.011,18.933,16.840,13.923,
     &11.109,9.544,8.153,6.493,4.485,-999.000,
     &2.680,-999.000,3.988,-999.000,10.895,16.321,
     &20.669,21.821,19.328,-999.000,13.176,-999.000,
     &6.546,-999.000,1.245,-999.000,-2.543,-999.000,
     &-7.030,-11.696,-15.634,-17.896,-19.644
     &/
       data (fjrocvz(it,13,3, 7), it=1,59) /
     &-18.330,-20.244,-21.117,-22.239,-24.446,-26.187,
     &-28.636,-31.437,-33.789,-31.880,-24.719,-16.886,
     &-12.901,-11.219,-7.901,-4.172,-0.440,6.250,
     &14.481,20.409,24.301,26.419,26.601,25.346,
     &24.049,22.727,21.173,19.606,17.485,15.238,
     &13.256,11.077,9.448,8.400,7.087,-999.000,
     &6.978,-999.000,9.422,-999.000,14.841,20.570,
     &27.395,29.726,23.914,-999.000,16.077,-999.000,
     &8.414,-999.000,1.901,-999.000,-2.521,-999.000,
     &-6.317,-9.220,-12.321,-15.720,-18.330
     &/
       data (fjrocvz(it,14,3, 7), it=1,59) /
     &-18.734,-20.773,-20.864,-21.896,-24.727,-28.457,
     &-30.833,-33.066,-36.162,-34.828,-26.217,-15.201,
     &-10.102,-8.927,-7.126,-4.190,-1.085,4.879,
     &14.080,19.840,24.169,26.411,26.538,24.761,
     &23.304,22.378,21.446,20.191,18.764,17.141,
     &15.100,12.786,11.302,10.015,9.481,-999.000,
     &9.891,-999.000,12.317,-999.000,17.668,22.255,
     &29.050,31.830,29.085,21.492,16.119,-999.000,
     &7.598,-999.000,1.763,-999.000,-2.109,-999.000,
     &-5.547,-8.381,-11.394,-15.460,-18.734
     &/
       data (fjrocvz(it,15,3, 7), it=1,59) /
     &-18.622,-19.996,-20.446,-21.037,-23.082,-26.559,
     &-29.452,-32.605,-36.188,-37.001,-28.809,-17.171,
     &-10.269,-9.134,-8.259,-5.803,-2.147,3.588,
     &11.364,16.964,20.880,22.998,22.370,20.240,
     &18.364,17.040,16.264,15.054,13.394,12.280,
     &11.184,10.026,8.480,7.522,7.270,-999.000,
     &7.470,-999.000,9.857,-999.000,14.532,18.908,
     &23.071,24.083,20.386,14.873,10.534,-999.000,
     &2.985,-999.000,-1.665,-999.000,-4.340,-999.000,
     &-7.852,-10.410,-12.934,-15.794,-18.622
     &/
       data (fjrocvz(it,16,3, 7), it=1,59) /
     &-16.451,-17.266,-17.210,-17.636,-19.964,-24.156,
     &-28.533,-31.861,-36.798,-39.498,-35.145,-22.446,
     &-12.056,-9.022,-8.186,-5.961,-2.243,3.092,
     &9.932,15.154,18.489,19.638,18.270,16.528,
     &14.802,13.016,11.136,8.603,6.125,4.787,
     &4.202,3.338,2.196,0.849,0.436,-999.000,
     &1.642,-999.000,4.196,-999.000,8.225,11.942,
     &13.617,11.839,8.897,-999.000,0.857,-999.000,
     &-5.257,-999.000,-8.546,-999.000,-9.654,-999.000,
     &-10.684,-12.243,-13.521,-14.831,-16.451
     &/
       data (fjrocvz(it,17,3, 7), it=1,59) /
     &-17.484,-17.922,-17.251,-17.180,-18.968,-22.229,
     &-26.720,-33.029,-40.079,-42.634,-37.682,-22.314,
     &-10.151,-7.612,-5.833,-2.791,0.911,5.605,
     &11.765,17.113,20.310,21.393,20.419,18.125,
     &15.766,13.617,11.389,8.376,5.464,3.250,
     &1.167,0.029,-0.928,-2.299,-3.157,-999.000,
     &-3.334,-999.000,-1.361,-999.000,3.650,8.606,
     &9.615,6.623,2.694,-999.000,-3.982,-999.000,
     &-10.541,-999.000,-14.162,-999.000,-14.887,-999.000,
     &-15.209,-15.999,-16.405,-16.653,-17.484
     &/
       data (fjrocvz(it,18,3, 7), it=1,59) /
     &-24.741,-23.717,-22.073,-20.668,-20.741,-22.504,
     &-25.521,-32.072,-40.905,-42.851,-36.283,-14.598,
     &-5.108,-3.933,-2.055,1.247,5.195,10.745,
     &17.032,21.972,24.333,25.364,24.840,22.992,
     &20.222,17.075,14.455,11.352,8.486,5.519,
     &2.888,1.119,-0.419,-1.901,-3.460,-999.000,
     &-4.090,-999.000,-1.981,-999.000,3.809,6.149,
     &4.595,0.419,-3.502,-999.000,-10.136,-999.000,
     &-16.328,-999.000,-19.231,-999.000,-19.972,-999.000,
     &-20.518,-21.936,-23.061,-23.885,-24.741
     &/
       data (fjrocvz(it,19,3, 7), it=1,59) /
     &-26.204,-25.858,-24.203,-22.619,-22.509,-24.236,
     &-25.208,-27.741,-34.054,-34.337,-28.412,-11.548,
     &-2.460,-0.550,1.160,4.086,8.615,14.258,
     &20.506,25.143,27.834,28.379,27.559,25.967,
     &23.611,20.807,17.707,14.667,11.757,8.769,
     &6.393,4.911,3.634,1.346,-0.402,-999.000,
     &-0.321,-999.000,0.726,-999.000,6.165,7.340,
     &5.218,0.987,-4.603,-999.000,-11.215,-999.000,
     &-16.792,-999.000,-19.398,-999.000,-20.647,-999.000,
     &-21.330,-22.592,-24.064,-25.366,-26.204
     &/
       data (fjrocvz(it,20,3, 7), it=1,59) /
     &-23.208,-23.422,-22.223,-20.776,-21.205,-23.904,
     &-26.475,-28.890,-30.411,-27.498,-20.881,-10.895,
     &-2.884,0.719,2.461,4.597,8.461,14.371,
     &20.957,26.947,30.006,30.179,28.674,26.352,
     &23.971,21.875,20.344,18.882,16.500,13.614,
     &10.786,8.858,6.639,4.710,3.752,-999.000,
     &3.307,-999.000,4.601,-999.000,9.049,11.199,
     &10.234,6.588,1.092,-999.000,-6.666,-999.000,
     &-11.645,-999.000,-14.084,-999.000,-15.984,-999.000,
     &-17.871,-19.786,-21.295,-22.103,-23.208
     &/
       data (fjrocvz(it,21,3, 7), it=1,59) /
     &-19.864,-20.412,-19.414,-18.164,-18.716,-22.229,
     &-25.085,-26.830,-29.811,-31.788,-28.574,-18.066,
     &-6.323,-1.367,0.789,2.661,5.514,10.143,
     &16.291,21.746,24.895,25.382,24.472,23.414,
     &21.938,20.479,19.582,18.147,16.116,13.478,
     &11.222,9.134,7.422,6.425,5.631,-999.000,
     &5.541,-999.000,5.680,-999.000,7.113,8.866,
     &11.071,9.459,5.225,-999.000,-1.528,-999.000,
     &-6.392,-999.000,-9.481,-999.000,-12.171,-999.000,
     &-14.630,-16.612,-18.031,-18.751,-19.864
     &/
       data (fjrocvz(it,22,3, 7), it=1,59) /
     &-18.049,-17.143,-15.605,-15.355,-16.879,-19.994,
     &-22.493,-26.006,-31.966,-35.768,-33.219,-22.512,
     &-10.475,-3.111,-1.165,0.469,2.629,5.939,
     &10.438,14.610,17.916,18.342,17.595,17.330,
     &16.950,15.926,14.855,13.160,10.903,8.807,
     &7.074,6.045,5.208,5.245,6.245,-999.000,
     &6.692,-999.000,6.508,-999.000,8.828,9.912,
     &13.798,13.078,6.678,-999.000,-0.098,-999.000,
     &-5.268,-999.000,-8.829,-999.000,-11.335,-999.000,
     &-13.596,-15.594,-17.286,-18.078,-18.049
     &/
       data (fjrocvz(it,23,3, 7), it=1,59) /
     &-19.365,-17.837,-15.917,-14.852,-16.203,-19.682,
     &-23.148,-26.429,-29.990,-31.369,-27.344,-17.195,
     &-5.719,-0.892,-0.484,0.370,1.746,3.991,
     &7.092,11.475,15.713,17.186,16.108,14.262,
     &12.572,10.701,9.552,8.359,6.492,4.706,
     &4.184,3.866,3.108,3.369,4.763,-999.000,
     &5.695,-999.000,7.690,-999.000,12.133,16.132,
     &19.503,18.268,11.268,-999.000,0.676,-999.000,
     &-5.326,-999.000,-8.913,-999.000,-11.542,-999.000,
     &-14.271,-16.623,-18.267,-19.153,-19.365
     &/
       data (fjrocvz(it,24,3, 7), it=1,59) /
     &-21.546,-21.312,-19.630,-17.341,-16.300,-17.898,
     &-20.035,-22.833,-27.529,-28.325,-23.739,-12.614,
     &-1.753,2.132,3.535,4.738,5.269,6.137,
     &8.049,11.252,14.854,15.727,14.968,13.390,
     &10.469,8.029,6.911,7.276,6.737,6.073,
     &5.925,5.598,5.184,5.015,5.548,-999.000,
     &8.409,-999.000,11.776,-999.000,17.364,21.764,
     &22.306,19.700,14.126,-999.000,-0.184,-999.000,
     &-7.192,-999.000,-10.194,-999.000,-12.775,-999.000,
     &-16.118,-18.828,-20.265,-20.898,-21.546
     &/
       data (fjrocvz(it,25,3, 7), it=1,59) /
     &-21.380,-22.106,-21.633,-20.459,-18.575,-17.661,
     &-17.731,-21.114,-28.216,-29.947,-22.993,-9.261,
     &2.241,6.306,8.982,11.013,10.860,9.939,
     &10.309,12.934,16.322,17.290,16.742,14.347,
     &11.052,9.361,9.864,10.822,10.697,9.789,
     &9.071,8.980,8.766,8.274,8.737,-999.000,
     &11.611,-999.000,14.317,-999.000,19.955,23.512,
     &23.927,19.406,9.345,-999.000,-2.790,-999.000,
     &-9.670,-999.000,-12.408,-999.000,-14.157,-999.000,
     &-16.337,-18.709,-20.239,-20.557,-21.380
     &/
       data (fjrocvz(it, 1,4, 7), it=1,59) /
     &-19.431,-18.560,-17.039,-15.966,-16.756,-17.604,
     &-18.225,-19.124,-19.740,-19.089,-15.807,-9.047,
     &-10.370,-5.728,-0.194,6.028,11.876,20.837,
     &27.902,30.761,30.537,29.354,26.933,22.835,
     &18.763,15.347,10.257,6.558,6.516,8.127,
     &8.853,9.244,9.929,11.806,14.637,-999.000,
     &18.445,-999.000,22.921,26.887,27.578,22.460,
     &14.367,-999.000,0.259,-999.000,-10.500,-999.000,
     &-16.848,-999.000,-19.216,-999.000,-22.793,-999.000,
     &-23.344,-23.137,-22.757,-21.231,-19.431
     &/
       data (fjrocvz(it, 2,4, 7), it=1,59) /
     &-17.841,-17.133,-16.231,-15.791,-15.106,-14.519,
     &-14.736,-16.364,-17.653,-16.991,-11.638,-3.738,
     &1.837,5.454,9.992,12.519,16.956,22.497,
     &28.032,31.868,32.927,32.090,29.351,25.120,
     &21.031,18.298,14.743,11.292,9.750,9.999,
     &11.236,12.155,13.175,15.018,16.883,-999.000,
     &19.264,-999.000,24.168,27.121,29.097,28.302,
     &22.213,-999.000,0.768,-999.000,-13.771,-999.000,
     &-16.414,-999.000,-21.622,-999.000,-24.841,-999.000,
     &-23.860,-23.336,-21.465,-19.048,-17.841
     &/
       data (fjrocvz(it, 3,4, 7), it=1,59) /
     &-18.023,-16.819,-16.154,-15.021,-14.015,-14.230,
     &-13.867,-14.024,-14.693,-14.066,-9.587,-3.128,
     &2.792,6.846,9.430,12.681,16.994,20.650,
     &24.241,27.110,28.823,28.252,25.413,21.749,
     &18.623,16.644,14.498,12.377,10.923,10.889,
     &11.990,13.048,14.586,15.612,17.547,-999.000,
     &20.840,-999.000,23.924,29.213,33.004,31.386,
     &21.925,9.782,0.181,-11.604,-18.324,-18.644,
     &-19.766,-999.000,-24.832,-999.000,-26.134,-999.000,
     &-25.532,-23.710,-21.498,-19.633,-18.023
     &/
       data (fjrocvz(it, 4,4, 7), it=1,59) /
     &-19.259,-18.092,-16.494,-14.901,-14.506,-14.651,
     &-13.976,-12.611,-11.989,-11.210,-8.167,-2.692,
     &3.305,6.278,7.710,10.549,14.625,19.325,
     &23.044,25.294,25.436,23.816,21.115,17.952,
     &15.344,13.452,11.608,10.497,10.512,10.744,
     &11.652,13.191,14.515,16.589,20.020,-999.000,
     &22.116,-999.000,25.671,31.284,33.011,32.092,
     &26.904,15.605,-0.044,-15.650,-21.593,-999.000,
     &-22.210,-999.000,-25.777,-999.000,-28.012,-999.000,
     &-26.401,-23.742,-21.880,-20.224,-19.259
     &/
       data (fjrocvz(it, 5,4, 7), it=1,59) /
     &-19.655,-18.334,-16.052,-14.388,-14.188,-15.039,
     &-14.958,-13.847,-13.422,-12.559,-9.562,-3.357,
     &2.606,6.041,8.926,11.275,15.372,20.762,
     &25.200,27.823,28.127,26.489,23.190,19.647,
     &16.711,14.825,13.319,12.630,12.199,11.852,
     &12.574,14.008,15.850,17.815,19.385,-999.000,
     &23.215,-999.000,27.194,31.310,34.399,36.624,
     &32.247,16.736,-0.595,-12.339,-21.024,-24.422,
     &-27.390,-30.244,-31.221,-30.630,-28.741,-999.000,
     &-26.238,-23.675,-21.675,-20.337,-19.655
     &/
       data (fjrocvz(it, 6,4, 7), it=1,59) /
     &-20.492,-19.622,-18.151,-16.760,-16.444,-16.445,
     &-15.312,-14.590,-14.264,-12.477,-10.297,-8.898,
     &-5.615,0.819,7.510,12.731,16.984,22.143,
     &26.518,29.129,30.483,29.574,27.160,23.824,
     &21.121,19.279,17.440,15.621,14.422,13.245,
     &12.754,14.454,16.542,17.467,19.385,-999.000,
     &23.532,-999.000,27.666,34.446,40.172,41.875,
     &35.835,21.709,4.019,-9.414,-19.428,-25.658,
     &-28.565,-999.000,-30.891,-999.000,-30.071,-999.000,
     &-27.501,-24.096,-22.253,-21.432,-20.492
     &/
       data (fjrocvz(it, 7,4, 7), it=1,59) /
     &-22.391,-22.010,-20.977,-20.050,-19.478,-18.021,
     &-16.358,-15.310,-14.061,-13.014,-10.865,-8.485,
     &-3.290,0.423,4.705,8.490,13.157,18.964,
     &24.163,28.168,29.762,29.845,28.603,26.302,
     &24.038,21.681,19.356,17.446,15.304,12.975,
     &12.221,13.042,14.393,16.148,17.666,-999.000,
     &20.548,-999.000,25.398,33.325,40.264,44.292,
     &34.944,17.455,1.585,-9.188,-19.187,-27.432,
     &-32.240,-33.405,-33.060,-32.341,-31.144,-999.000,
     &-28.229,-25.974,-24.313,-22.859,-22.391
     &/
       data (fjrocvz(it, 8,4, 7), it=1,59) /
     &-22.356,-21.109,-20.265,-19.871,-19.309,-18.835,
     &-18.210,-17.542,-16.241,-18.091,-19.929,-21.637,
     &-16.387,-8.222,-2.193,3.748,7.901,12.020,
     &18.058,22.764,25.548,26.061,25.481,24.290,
     &22.628,20.796,18.525,16.336,13.661,11.295,
     &10.253,10.530,11.906,13.481,14.771,-999.000,
     &18.084,-999.000,23.977,29.491,35.995,36.765,
     &26.341,13.588,1.918,-9.270,-17.254,-24.449,
     &-28.932,-29.564,-27.593,-999.000,-27.581,-999.000,
     &-26.526,-26.216,-25.116,-23.612,-22.356
     &/
       data (fjrocvz(it, 9,4, 7), it=1,59) /
     &-19.729,-17.741,-15.964,-14.888,-14.937,-15.810,
     &-17.493,-18.959,-20.006,-21.817,-24.443,-26.402,
     &-21.146,-10.681,-2.462,3.042,5.786,9.696,
     &14.752,18.597,20.750,21.468,21.304,19.958,
     &17.705,15.621,13.733,11.758,9.616,8.326,
     &8.241,8.861,10.062,11.325,12.933,-999.000,
     &15.523,-999.000,22.264,28.595,34.304,37.461,
     &34.442,22.156,9.600,-1.784,-9.714,-999.000,
     &-15.758,-999.000,-19.215,-999.000,-21.908,-999.000,
     &-22.366,-22.443,-22.635,-21.755,-19.729
     &/
       data (fjrocvz(it,10,4, 7), it=1,59) /
     &-20.317,-18.053,-15.756,-14.539,-13.886,-15.132,
     &-18.287,-20.999,-22.558,-23.374,-24.027,-23.784,
     &-19.591,-8.463,-0.249,5.212,7.889,12.137,
     &17.358,20.755,21.550,21.287,20.104,18.519,
     &16.510,13.945,11.623,9.701,8.456,7.940,
     &8.032,8.405,8.499,9.126,10.943,-999.000,
     &14.781,-999.000,22.611,29.870,37.941,41.957,
     &39.797,27.377,15.028,-999.000,1.225,-999.000,
     &-8.391,-999.000,-15.061,-999.000,-17.070,-999.000,
     &-18.865,-19.833,-20.841,-20.940,-20.317
     &/
       data (fjrocvz(it,11,4, 7), it=1,59) /
     &-22.932,-22.097,-21.632,-21.532,-21.340,-21.809,
     &-22.586,-24.081,-25.873,-27.824,-27.125,-23.323,
     &-17.793,-7.809,-0.101,6.082,10.475,14.129,
     &18.643,22.457,24.232,24.370,23.420,21.951,
     &20.473,18.181,15.289,13.229,11.763,10.828,
     &10.197,9.465,8.960,9.498,11.832,-999.000,
     &16.646,-999.000,23.496,30.539,39.255,40.223,
     &35.870,25.677,15.379,-999.000,-0.478,-999.000,
     &-10.937,-999.000,-17.456,-999.000,-18.967,-999.000,
     &-21.658,-21.512,-22.409,-23.544,-22.932
     &/
       data (fjrocvz(it,12,4, 7), it=1,59) /
     &-24.655,-25.235,-25.070,-24.911,-25.012,-24.597,
     &-24.515,-26.157,-28.974,-31.058,-31.766,-28.451,
     &-24.149,-14.300,-4.463,1.535,6.382,11.850,
     &18.721,23.864,26.375,26.651,25.787,24.790,
     &23.635,21.275,18.259,15.859,14.681,13.944,
     &13.013,11.755,10.848,11.375,14.118,-999.000,
     &18.505,-999.000,24.981,31.337,39.035,43.648,
     &38.444,21.919,8.930,-0.092,-8.024,-999.000,
     &-13.071,-999.000,-17.586,-999.000,-21.497,-999.000,
     &-23.921,-23.907,-24.184,-24.574,-24.655
     &/
       data (fjrocvz(it,13,4, 7), it=1,59) /
     &-23.570,-22.997,-22.726,-22.671,-22.278,-22.910,
     &-24.447,-25.758,-27.163,-27.819,-28.228,-26.731,
     &-23.831,-15.096,-6.240,0.455,4.987,11.183,
     &20.624,26.600,28.189,28.264,27.468,26.619,
     &25.340,22.696,19.825,17.910,17.518,17.329,
     &16.629,15.357,14.009,14.246,16.520,-999.000,
     &19.994,-999.000,28.565,34.534,42.726,47.783,
     &39.874,22.753,5.395,-4.658,-10.287,-12.276,
     &-13.290,-999.000,-16.947,-999.000,-19.853,-999.000,
     &-22.143,-24.152,-24.062,-23.549,-23.570
     &/
       data (fjrocvz(it,14,4, 7), it=1,59) /
     &-21.684,-21.202,-20.840,-20.238,-20.173,-21.775,
     &-24.539,-28.094,-31.246,-31.031,-28.138,-24.139,
     &-20.275,-16.615,-9.604,-1.722,4.077,10.855,
     &20.337,26.138,28.303,28.458,27.493,26.297,
     &24.496,22.468,20.339,19.023,18.958,18.839,
     &17.909,16.716,15.542,15.783,18.201,-999.000,
     &22.894,-999.000,31.374,40.005,45.522,45.102,
     &36.808,22.004,8.532,-999.000,-8.563,-999.000,
     &-16.284,-999.000,-19.053,-999.000,-21.586,-999.000,
     &-24.187,-24.960,-24.002,-22.510,-21.684
     &/
       data (fjrocvz(it,15,4, 7), it=1,59) /
     &-19.576,-19.048,-18.481,-18.014,-18.013,-19.644,
     &-23.580,-29.619,-33.780,-35.102,-31.705,-24.524,
     &-19.695,-14.616,-9.153,-1.600,5.032,11.225,
     &18.789,24.046,26.294,26.102,24.442,22.026,
     &19.730,18.056,16.128,14.869,14.558,14.214,
     &13.555,12.789,12.724,13.899,17.094,-999.000,
     &21.277,-999.000,28.497,36.040,41.416,43.465,
     &36.314,20.636,7.223,-999.000,-10.765,-999.000,
     &-17.909,-999.000,-22.121,-999.000,-25.209,-999.000,
     &-26.854,-27.070,-24.616,-21.214,-19.576
     &/
       data (fjrocvz(it,16,4, 7), it=1,59) /
     &-19.455,-18.277,-18.452,-18.680,-19.420,-21.547,
     &-25.283,-29.985,-33.612,-36.476,-35.388,-27.875,
     &-18.398,-12.927,-7.006,-0.404,5.899,11.346,
     &18.670,23.673,25.492,25.552,24.180,21.026,
     &17.249,13.952,11.436,9.963,9.678,9.781,
     &9.221,8.358,8.250,9.275,11.856,-999.000,
     &16.141,-999.000,23.073,30.029,36.020,37.595,
     &29.596,13.239,3.055,-999.000,-12.391,-999.000,
     &-20.198,-999.000,-24.980,-999.000,-27.836,-999.000,
     &-29.092,-28.582,-26.023,-22.372,-19.455
     &/
       data (fjrocvz(it,17,4, 7), it=1,59) /
     &-22.650,-21.940,-21.269,-20.849,-21.556,-23.287,
     &-26.434,-31.028,-34.854,-36.412,-34.680,-26.574,
     &-17.633,-11.428,-4.652,0.982,6.496,12.253,
     &20.079,25.422,27.849,28.523,27.438,24.650,
     &20.357,16.197,12.449,9.782,9.140,8.844,
     &7.931,6.663,5.452,5.325,7.787,-999.000,
     &13.079,-999.000,20.969,26.247,30.693,31.591,
     &25.153,10.919,-1.250,-12.264,-20.071,-23.965,
     &-25.921,-999.000,-27.136,-999.000,-29.142,-999.000,
     &-29.739,-28.544,-26.897,-24.618,-22.650
     &/
       data (fjrocvz(it,18,4, 7), it=1,59) /
     &-24.042,-22.751,-21.579,-21.266,-21.150,-22.077,
     &-25.761,-30.424,-33.372,-34.845,-32.299,-25.010,
     &-16.015,-9.852,-4.681,1.060,7.042,14.474,
     &21.765,27.619,31.577,32.386,30.231,26.387,
     &22.714,19.169,15.043,11.401,9.757,8.524,
     &7.347,5.842,4.730,4.624,6.901,-999.000,
     &11.612,15.275,20.327,25.572,28.989,26.363,
     &18.980,9.102,-2.743,-999.000,-20.025,-999.000,
     &-27.261,-999.000,-30.239,-999.000,-30.899,-999.000,
     &-30.098,-29.624,-28.087,-25.912,-24.042
     &/
       data (fjrocvz(it,19,4, 7), it=1,59) /
     &-22.643,-21.114,-20.987,-20.450,-19.879,-20.767,
     &-23.535,-26.302,-28.764,-30.338,-27.705,-20.903,
     &-12.861,-8.438,-3.317,2.487,8.221,16.196,
     &24.508,31.125,35.399,35.969,33.457,29.568,
     &25.453,21.379,17.212,13.506,12.263,11.531,
     &10.058,8.272,7.060,6.613,8.466,-999.000,
     &13.922,-999.000,22.100,27.359,29.330,25.558,
     &19.948,10.139,-3.769,-15.013,-23.363,-27.311,
     &-28.637,-999.000,-30.289,-999.000,-29.975,-999.000,
     &-29.229,-28.866,-27.611,-25.201,-22.643
     &/
       data (fjrocvz(it,20,4, 7), it=1,59) /
     &-24.355,-22.895,-21.213,-19.475,-18.263,-18.563,
     &-20.726,-23.856,-25.681,-26.372,-23.845,-14.824,
     &-6.814,-4.793,-0.260,4.179,8.892,16.997,
     &25.005,32.460,36.949,37.864,36.007,32.627,
     &28.947,25.341,21.030,17.405,15.885,14.564,
     &12.814,10.944,9.462,9.033,11.380,-999.000,
     &16.422,-999.000,22.990,27.945,32.633,33.325,
     &25.437,10.649,-3.578,-12.052,-19.106,-24.557,
     &-27.845,-999.000,-29.084,-999.000,-29.983,-999.000,
     &-29.601,-29.231,-28.373,-26.378,-24.355
     &/
       data (fjrocvz(it,21,4, 7), it=1,59) /
     &-24.361,-22.009,-20.136,-18.941,-18.216,-18.040,
     &-19.208,-21.950,-24.850,-25.566,-22.778,-15.524,
     &-10.122,-5.243,-0.166,5.067,10.594,16.959,
     &22.908,29.651,33.567,35.076,33.376,29.912,
     &26.527,23.648,20.024,16.002,13.834,12.248,
     &10.597,9.464,9.047,9.197,11.113,-999.000,
     &15.508,-999.000,22.852,28.062,33.284,32.030,
     &23.006,-999.000,0.678,-999.000,-15.161,-999.000,
     &-23.851,-999.000,-27.668,-999.000,-28.747,-999.000,
     &-28.814,-28.345,-27.624,-26.261,-24.361
     &/
       data (fjrocvz(it,22,4, 7), it=1,59) /
     &-23.238,-22.154,-21.005,-20.167,-20.078,-19.953,
     &-20.233,-22.497,-26.372,-27.227,-24.221,-17.345,
     &-9.241,-6.137,-0.610,4.316,9.286,15.173,
     &21.630,27.569,30.428,30.819,28.824,24.958,
     &20.447,16.681,13.236,10.242,8.577,7.440,
     &6.683,7.115,8.089,8.734,10.843,-999.000,
     &16.021,-999.000,23.533,26.038,29.229,25.845,
     &18.254,-999.000,-2.084,-999.000,-16.278,-999.000,
     &-24.942,-999.000,-26.629,-999.000,-26.894,-999.000,
     &-27.435,-26.921,-26.040,-24.654,-23.238
     &/
       data (fjrocvz(it,23,4, 7), it=1,59) /
     &-21.505,-21.060,-19.561,-17.606,-17.031,-17.941,
     &-19.791,-22.126,-24.279,-24.919,-22.311,-16.125,
     &-10.917,-8.017,-2.594,2.666,7.928,13.516,
     &21.630,27.387,29.279,28.438,25.870,21.868,
     &17.395,13.066,9.508,7.000,6.013,5.534,
     &5.222,5.245,6.763,8.962,12.060,-999.000,
     &15.836,-999.000,22.388,26.181,28.254,26.229,
     &19.668,8.042,-2.028,-999.000,-15.332,-999.000,
     &-22.177,-999.000,-24.849,-999.000,-25.506,-999.000,
     &-26.074,-25.566,-24.186,-22.232,-21.505
     &/
       data (fjrocvz(it,24,4, 7), it=1,59) /
     &-19.867,-18.409,-16.676,-14.599,-14.451,-16.236,
     &-18.382,-19.671,-20.749,-21.358,-19.233,-14.358,
     &-8.246,-3.321,0.128,3.395,8.018,16.280,
     &24.473,28.104,28.865,27.880,24.853,20.532,
     &16.281,12.176,8.159,5.451,5.393,6.576,
     &6.954,6.703,8.048,10.204,12.911,-999.000,
     &16.385,-999.000,22.112,26.958,29.482,26.582,
     &17.510,-999.000,0.883,-999.000,-10.064,-999.000,
     &-16.968,-999.000,-21.824,-999.000,-22.772,-999.000,
     &-23.987,-23.893,-22.766,-21.359,-19.867
     &/
       data (fjrocvz(it,25,4, 7), it=1,59) /
     &-19.431,-18.560,-17.039,-15.966,-16.756,-17.604,
     &-18.225,-19.124,-19.740,-19.089,-15.807,-9.047,
     &-10.370,-5.728,-0.194,6.028,11.876,20.837,
     &27.902,30.761,30.537,29.354,26.933,22.835,
     &18.763,15.347,10.257,6.558,6.516,8.127,
     &8.853,9.244,9.929,11.806,14.637,-999.000,
     &18.445,-999.000,22.921,26.887,27.578,22.460,
     &14.367,-999.000,0.259,-999.000,-10.500,-999.000,
     &-16.848,-999.000,-19.216,-999.000,-22.793,-999.000,
     &-23.344,-23.137,-22.757,-21.231,-19.431
     &/
       data (fjrocvz(it, 1,1, 8), it=1,59) /
     &-13.264,-12.551,-12.154,-12.234,-12.505,-13.966,
     &-15.815,-17.801,-20.357,-21.906,-20.797,-14.536,
     &-4.903,2.959,8.968,13.480,18.062,23.484,
     &27.929,30.847,31.334,29.486,27.134,24.139,
     &20.993,17.375,13.603,10.787,7.917,4.903,
     &3.067,2.670,3.048,2.814,2.722,-999.000,
     &2.150,-999.000,2.991,-999.000,5.584,-999.000,
     &4.264,-999.000,0.935,-999.000,-5.524,-999.000,
     &-10.450,-999.000,-13.788,-999.000,-15.906,-999.000,
     &-17.960,-18.443,-17.286,-14.918,-13.264
     &/
       data (fjrocvz(it, 2,1, 8), it=1,59) /
     &-15.772,-14.521,-14.097,-13.968,-14.841,-16.659,
     &-17.464,-18.682,-20.266,-21.080,-17.370,-7.750,
     &1.288,7.875,13.358,19.198,24.130,28.748,
     &32.960,35.763,36.542,34.564,31.326,27.626,
     &23.459,19.110,14.626,11.113,8.052,5.190,
     &2.464,0.934,0.552,0.746,0.893,-999.000,
     &1.408,-999.000,3.312,-999.000,3.863,-999.000,
     &2.709,-999.000,-1.355,-999.000,-8.024,-999.000,
     &-13.390,-999.000,-17.294,-999.000,-19.452,-999.000,
     &-20.617,-20.446,-19.382,-17.612,-15.772
     &/
       data (fjrocvz(it, 3,1, 8), it=1,59) /
     &-17.829,-17.136,-16.898,-17.176,-17.941,-18.002,
     &-17.213,-17.366,-17.321,-16.088,-10.222,-1.561,
     &5.933,11.202,16.458,21.786,25.424,29.085,
     &32.944,36.076,36.686,34.189,30.498,26.918,
     &23.632,20.417,16.621,12.719,9.289,6.152,
     &3.198,1.789,1.384,1.966,2.212,-999.000,
     &3.349,-999.000,4.247,-999.000,5.981,7.594,
     &8.350,6.353,2.297,-1.018,-6.365,-999.000,
     &-13.435,-999.000,-17.005,-999.000,-19.092,-999.000,
     &-20.195,-20.302,-19.687,-18.781,-17.829
     &/
       data (fjrocvz(it, 4,1, 8), it=1,59) /
     &-18.046,-18.775,-18.709,-17.870,-16.927,-17.202,
     &-17.305,-17.662,-16.713,-13.340,-5.997,2.406,
     &7.927,12.100,17.556,22.217,26.170,29.645,
     &33.259,35.413,34.728,31.321,27.477,24.527,
     &21.828,18.223,14.416,11.621,9.759,7.857,
     &5.256,3.207,2.484,3.192,5.082,-999.000,
     &6.071,-999.000,8.456,-999.000,11.384,12.817,
     &13.906,10.000,4.427,-999.000,-4.063,-999.000,
     &-12.760,-999.000,-16.439,-999.000,-18.774,-999.000,
     &-19.891,-19.406,-17.980,-17.115,-18.046
     &/
       data (fjrocvz(it, 5,1, 8), it=1,59) /
     &-14.944,-15.662,-15.734,-14.761,-14.220,-15.646,
     &-17.695,-19.049,-18.512,-14.937,-7.061,2.116,
     &7.208,11.302,16.901,21.920,25.917,29.036,
     &32.329,34.383,33.881,31.000,28.005,25.291,
     &22.027,17.784,13.920,11.803,10.902,10.186,
     &8.425,5.997,4.808,4.573,5.672,-999.000,
     &8.346,-999.000,12.854,-999.000,15.324,19.811,
     &19.531,14.570,8.533,-999.000,-2.609,-999.000,
     &-12.508,-999.000,-16.778,-999.000,-18.623,-999.000,
     &-19.482,-18.078,-15.701,-14.431,-14.944
     &/
       data (fjrocvz(it, 6,1, 8), it=1,59) /
     &-12.904,-13.641,-14.360,-15.119,-15.569,-16.879,
     &-18.991,-20.495,-20.193,-15.810,-8.901,-0.693,
     &4.322,7.587,11.612,15.823,19.703,23.268,
     &27.454,30.809,31.022,29.319,27.227,24.937,
     &21.959,18.747,15.758,13.420,12.296,12.007,
     &11.455,10.170,8.081,6.760,7.463,-999.000,
     &10.139,-999.000,13.635,-999.000,18.438,-999.000,
     &19.052,-999.000,12.053,-999.000,2.292,-999.000,
     &-6.602,-999.000,-11.919,-999.000,-15.462,-999.000,
     &-17.119,-17.034,-15.027,-13.309,-12.904
     &/
       data (fjrocvz(it, 7,1, 8), it=1,59) /
     &-15.333,-16.203,-17.308,-18.309,-19.084,-19.905,
     &-20.453,-20.637,-20.276,-17.119,-12.891,-8.755,
     &-4.130,1.293,6.682,10.800,15.095,18.428,
     &23.360,26.758,27.433,25.984,23.718,21.550,
     &19.463,17.347,14.541,12.083,10.690,10.890,
     &11.692,11.352,9.857,8.805,9.406,-999.000,
     &11.119,-999.000,15.103,-999.000,20.576,23.260,
     &27.717,26.484,18.042,-999.000,7.197,-999.000,
     &-1.280,-999.000,-7.696,-999.000,-12.542,-999.000,
     &-15.647,-16.817,-16.015,-15.695,-15.333
     &/
       data (fjrocvz(it, 8,1, 8), it=1,59) /
     &-18.597,-18.263,-18.151,-18.154,-18.464,-19.086,
     &-19.875,-20.847,-21.898,-20.613,-17.514,-11.948,
     &-6.019,-2.139,1.078,4.154,7.993,12.122,
     &17.544,21.676,23.424,22.550,20.357,18.103,
     &15.807,13.775,11.527,9.262,8.242,8.681,
     &10.213,10.983,10.882,10.319,11.033,-999.000,
     &13.692,-999.000,18.752,-999.000,26.430,34.260,
     &38.167,37.370,28.709,17.697,11.300,-999.000,
     &-2.388,-999.000,-10.192,-999.000,-16.354,-999.000,
     &-18.700,-20.244,-20.009,-19.429,-18.597
     &/
       data (fjrocvz(it, 9,1, 8), it=1,59) /
     &-19.577,-18.588,-17.829,-17.331,-17.156,-18.180,
     &-19.730,-21.358,-22.739,-22.150,-19.764,-15.456,
     &-10.784,-6.248,-1.435,2.088,4.943,8.326,
     &13.290,17.686,19.868,20.083,19.092,17.585,
     &15.473,14.031,12.686,11.386,10.382,10.219,
     &10.833,11.093,11.087,11.390,13.159,13.774,
     &14.708,16.779,21.914,28.116,34.848,43.377,
     &49.367,44.239,26.877,11.019,0.842,-6.478,
     &-9.585,-999.000,-16.040,-999.000,-19.634,-999.000,
     &-22.155,-23.240,-22.474,-21.087,-19.577
     &/
       data (fjrocvz(it,10,1, 8), it=1,59) /
     &-18.381,-17.165,-16.298,-15.712,-16.164,-17.920,
     &-19.374,-20.470,-22.152,-23.418,-22.145,-17.272,
     &-10.726,-4.698,0.480,4.505,7.431,10.138,
     &13.749,17.055,19.644,21.505,22.092,21.430,
     &19.512,18.063,16.841,15.748,14.460,14.148,
     &13.931,13.713,13.613,13.759,14.464,-999.000,
     &16.848,20.883,28.662,37.963,48.223,53.585,
     &46.994,32.038,14.881,-0.759,-7.419,-11.964,
     &-15.007,-999.000,-19.064,-999.000,-22.055,-999.000,
     &-23.601,-23.365,-21.932,-20.295,-18.381
     &/
       data (fjrocvz(it,11,1, 8), it=1,59) /
     &-17.055,-15.796,-15.403,-15.875,-17.297,-19.510,
     &-21.158,-22.304,-23.762,-24.297,-22.046,-14.910,
     &-7.092,-0.646,4.899,8.942,11.502,13.154,
     &15.794,19.806,23.984,26.650,27.288,26.096,
     &24.263,22.659,21.187,19.149,17.252,16.430,
     &15.225,14.550,14.573,15.158,16.486,17.747,
     &19.894,25.267,33.231,40.126,44.321,43.157,
     &29.669,13.423,1.436,-8.410,-14.986,-999.000,
     &-17.788,-999.000,-21.102,-999.000,-22.669,-999.000,
     &-23.034,-22.232,-20.602,-18.720,-17.055
     &/
       data (fjrocvz(it,12,1, 8), it=1,59) /
     &-18.738,-18.286,-18.590,-19.933,-21.183,-22.344,
     &-23.057,-24.221,-26.136,-26.421,-24.615,-19.586,
     &-11.812,-3.840,5.509,10.337,13.591,15.794,
     &18.577,22.534,25.987,28.163,28.842,28.458,
     &27.508,26.002,23.673,20.851,18.098,16.294,
     &15.137,14.265,14.375,16.018,19.247,-999.000,
     &23.870,25.105,31.161,37.317,40.424,34.907,
     &23.156,12.431,3.204,-999.000,-12.137,-999.000,
     &-20.537,-999.000,-23.684,-999.000,-23.809,-999.000,
     &-22.957,-22.043,-21.033,-19.720,-18.738
     &/
       data (fjrocvz(it,13,1, 8), it=1,59) /
     &-19.822,-20.348,-21.056,-21.947,-22.185,-22.250,
     &-22.518,-24.193,-26.142,-27.176,-26.166,-21.583,
     &-12.880,-5.476,2.508,8.928,13.574,16.188,
     &19.164,22.671,25.568,27.182,27.393,26.779,
     &25.387,23.858,22.125,19.582,16.213,14.278,
     &13.956,13.982,14.238,14.647,15.252,-999.000,
     &19.011,-999.000,25.614,30.798,33.261,28.395,
     &19.514,-999.000,6.651,-999.000,-7.953,-999.000,
     &-18.220,-999.000,-23.418,-999.000,-23.985,-999.000,
     &-23.366,-22.409,-21.266,-20.143,-19.822
     &/
       data (fjrocvz(it,14,1, 8), it=1,59) /
     &-20.208,-21.656,-22.655,-22.851,-22.152,-22.185,
     &-23.793,-25.937,-27.252,-27.709,-24.751,-21.424,
     &-15.385,-8.872,-0.432,7.056,12.772,15.316,
     &17.882,21.205,24.631,26.267,26.124,24.749,
     &23.247,21.981,20.273,17.309,14.132,12.223,
     &11.962,12.081,11.709,9.975,9.619,-999.000,
     &12.164,14.336,19.856,24.249,26.040,23.687,
     &18.720,10.790,5.506,-999.000,-6.139,-999.000,
     &-16.135,-999.000,-22.897,-999.000,-24.657,-999.000,
     &-24.556,-23.327,-21.507,-20.314,-20.208
     &/
       data (fjrocvz(it,15,1, 8), it=1,59) /
     &-20.779,-21.552,-22.939,-23.874,-23.399,-23.034,
     &-23.926,-25.752,-28.414,-30.101,-27.393,-22.213,
     &-14.325,-8.067,-1.209,5.572,11.445,14.558,
     &17.065,18.926,20.815,21.792,21.224,19.805,
     &18.652,17.831,16.657,14.912,12.308,10.132,
     &9.493,9.351,8.716,8.103,8.372,-999.000,
     &10.730,-999.000,17.058,23.058,27.585,25.879,
     &20.628,12.850,7.264,-999.000,-4.329,-999.000,
     &-14.554,-999.000,-22.609,-999.000,-25.087,-999.000,
     &-24.674,-23.660,-22.337,-21.372,-20.779
     &/
       data (fjrocvz(it,16,1, 8), it=1,59) /
     &-22.322,-22.564,-23.342,-24.235,-23.984,-23.310,
     &-23.499,-24.548,-27.430,-32.034,-32.461,-26.843,
     &-11.627,-3.703,1.042,6.346,11.518,14.367,
     &16.842,18.773,20.702,20.953,20.340,19.104,
     &17.867,16.639,15.225,13.055,10.157,8.299,
     &8.215,8.760,9.096,9.245,9.986,-999.000,
     &13.074,-999.000,19.385,25.328,29.572,25.985,
     &19.156,-999.000,6.365,-999.000,-4.086,-999.000,
     &-14.311,-999.000,-22.069,-999.000,-24.550,-999.000,
     &-24.822,-24.101,-23.320,-22.888,-22.322
     &/
       data (fjrocvz(it,17,1, 8), it=1,59) /
     &-22.150,-22.064,-22.283,-23.148,-23.774,-24.373,
     &-24.846,-23.951,-24.755,-28.589,-30.911,-26.260,
     &-10.818,0.500,4.418,7.860,11.604,14.191,
     &16.635,18.754,20.452,21.601,21.697,20.720,
     &19.680,18.306,16.652,14.415,11.503,9.453,
     &9.164,9.853,10.008,9.644,10.771,-999.000,
     &13.162,16.181,19.016,23.236,26.132,24.308,
     &17.060,-999.000,1.627,-999.000,-9.372,-999.000,
     &-18.450,-999.000,-24.391,-999.000,-26.530,-999.000,
     &-25.900,-24.050,-22.805,-22.485,-22.150
     &/
       data (fjrocvz(it,18,1, 8), it=1,59) /
     &-21.258,-21.015,-21.234,-22.398,-24.023,-25.445,
     &-26.311,-26.345,-26.241,-26.553,-25.568,-20.907,
     &-8.851,2.393,7.005,10.149,13.830,16.700,
     &19.143,20.181,21.307,22.676,22.753,21.591,
     &20.058,18.569,16.712,14.939,12.842,11.073,
     &10.254,10.411,10.732,10.921,11.535,-999.000,
     &14.482,-999.000,18.593,23.955,26.087,21.834,
     &12.720,3.618,-3.757,-999.000,-13.283,-999.000,
     &-20.765,-999.000,-26.423,-999.000,-28.048,-999.000,
     &-26.602,-24.173,-22.833,-22.071,-21.258
     &/
       data (fjrocvz(it,19,1, 8), it=1,59) /
     &-22.235,-21.146,-20.259,-20.464,-20.946,-21.557,
     &-21.407,-22.261,-23.544,-25.467,-25.197,-21.375,
     &-11.425,0.443,7.338,12.012,16.177,19.769,
     &22.601,23.952,25.249,26.267,26.609,25.234,
     &23.699,22.454,19.955,16.552,14.050,12.961,
     &12.747,12.518,12.258,11.273,11.031,-999.000,
     &13.429,-999.000,17.019,20.399,19.786,13.216,
     &7.312,-999.000,-4.719,-999.000,-13.666,-999.000,
     &-21.321,-999.000,-26.269,-999.000,-27.749,-999.000,
     &-26.541,-24.549,-23.429,-23.000,-22.235
     &/
       data (fjrocvz(it,20,1, 8), it=1,59) /
     &-21.185,-19.910,-19.428,-19.360,-19.414,-19.631,
     &-19.300,-19.515,-20.764,-22.946,-25.460,-23.389,
     &-14.932,-2.035,5.171,9.340,14.172,18.843,
     &22.422,24.424,25.842,26.938,27.318,26.614,
     &26.342,26.066,24.184,20.695,17.074,15.388,
     &14.972,14.392,13.081,11.495,11.015,-999.000,
     &10.997,-999.000,12.075,14.179,17.224,12.855,
     &4.315,-999.000,-6.338,-999.000,-15.839,-999.000,
     &-23.107,-999.000,-27.806,-999.000,-29.335,-999.000,
     &-28.817,-26.930,-24.929,-22.988,-21.185
     &/
       data (fjrocvz(it,21,1, 8), it=1,59) /
     &-22.440,-21.437,-21.378,-21.784,-21.856,-21.235,
     &-19.753,-20.063,-22.417,-26.289,-29.509,-27.886,
     &-18.749,-5.818,1.542,5.819,10.482,14.634,
     &18.433,21.362,23.486,24.261,24.225,24.241,
     &24.836,24.817,23.064,19.503,15.578,13.582,
     &13.190,12.852,11.813,9.983,8.900,-999.000,
     &9.497,-999.000,10.796,12.707,13.782,9.232,
     &0.774,-999.000,-9.474,-999.000,-17.715,-999.000,
     &-23.867,-999.000,-28.477,-999.000,-29.962,-999.000,
     &-29.739,-28.411,-26.597,-24.414,-22.440
     &/
       data (fjrocvz(it,22,1, 8), it=1,59) /
     &-20.819,-20.082,-20.301,-21.320,-21.738,-21.354,
     &-20.207,-20.742,-22.972,-26.421,-28.861,-27.213,
     &-18.483,-6.596,0.742,5.008,8.545,12.895,
     &16.945,20.649,22.383,22.852,23.396,23.672,
     &23.576,22.520,19.615,14.966,10.398,8.146,
     &7.942,8.063,7.277,5.842,5.468,-999.000,
     &5.903,-999.000,5.320,-999.000,5.307,-999.000,
     &0.473,-999.000,-10.934,-999.000,-17.041,-999.000,
     &-21.734,-999.000,-25.896,-999.000,-27.620,-999.000,
     &-27.379,-25.832,-23.749,-22.108,-20.819
     &/
       data (fjrocvz(it,23,1, 8), it=1,59) /
     &-16.036,-15.995,-16.365,-17.237,-18.160,-18.519,
     &-18.600,-19.435,-20.649,-22.045,-22.404,-19.041,
     &-11.493,-3.024,3.236,6.665,9.676,13.021,
     &16.891,20.595,23.205,25.345,26.443,25.894,
     &24.158,21.772,18.990,15.259,10.728,7.528,
     &5.587,4.555,3.699,2.937,2.853,-999.000,
     &3.414,-999.000,6.504,-999.000,7.137,-999.000,
     &4.167,-999.000,-0.989,-999.000,-10.883,-999.000,
     &-16.291,-999.000,-20.726,-999.000,-22.490,-999.000,
     &-22.357,-20.967,-18.865,-16.871,-16.036
     &/
       data (fjrocvz(it,24,1, 8), it=1,59) /
     &-13.283,-13.104,-13.157,-13.855,-14.285,-14.595,
     &-15.149,-16.772,-18.697,-20.473,-20.336,-15.905,
     &-7.740,-0.139,5.184,9.236,12.601,16.158,
     &19.896,23.446,25.861,26.405,25.817,24.283,
     &21.994,19.607,17.104,14.444,10.683,7.262,
     &4.859,4.007,3.979,3.535,2.961,-999.000,
     &3.696,-999.000,5.360,-999.000,6.955,-999.000,
     &5.719,-999.000,1.425,-999.000,-5.048,-999.000,
     &-10.844,-999.000,-14.649,-999.000,-16.470,-999.000,
     &-17.401,-17.447,-16.192,-14.415,-13.283
     &/
       data (fjrocvz(it,25,1, 8), it=1,59) /
     &-13.264,-12.551,-12.154,-12.234,-12.505,-13.966,
     &-15.815,-17.801,-20.357,-21.906,-20.797,-14.536,
     &-4.903,2.959,8.968,13.480,18.062,23.484,
     &27.929,30.847,31.334,29.486,27.134,24.139,
     &20.993,17.375,13.603,10.787,7.917,4.903,
     &3.067,2.670,3.048,2.814,2.722,-999.000,
     &2.150,-999.000,2.991,-999.000,5.584,-999.000,
     &4.264,-999.000,0.935,-999.000,-5.524,-999.000,
     &-10.450,-999.000,-13.788,-999.000,-15.906,-999.000,
     &-17.960,-18.443,-17.286,-14.918,-13.264
     &/
       data (fjrocvz(it, 1,2, 8), it=1,59) /
     &-20.570,-19.688,-18.162,-17.084,-17.863,-18.776,
     &-19.429,-20.360,-21.041,-20.383,-17.098,-10.335,
     &-11.650,-6.767,-1.111,5.231,11.321,20.385,
     &27.501,30.412,30.291,29.223,26.860,22.820,
     &18.862,15.733,10.786,7.231,7.475,9.208,
     &9.995,10.448,11.256,13.281,16.261,-999.000,
     &20.156,-999.000,25.012,29.638,30.593,25.371,
     &16.816,-999.000,0.688,-999.000,-11.566,-999.000,
     &-18.387,-999.000,-20.955,-999.000,-24.308,-999.000,
     &-24.635,-24.420,-24.034,-22.439,-20.570
     &/
       data (fjrocvz(it, 2,2, 8), it=1,59) /
     &-19.053,-18.329,-17.420,-16.972,-16.272,-15.813,
     &-16.094,-17.786,-19.202,-18.597,-13.272,-5.400,
     &0.118,4.052,8.748,11.433,16.187,21.881,
     &27.494,31.405,32.619,31.924,29.257,25.097,
     &21.151,18.703,15.290,11.981,10.724,11.103,
     &12.404,13.387,14.537,16.573,18.631,-999.000,
     &21.058,-999.000,26.261,29.494,31.750,30.994,
     &24.909,-999.000,1.468,-999.000,-15.222,-999.000,
     &-18.672,-999.000,-24.012,-999.000,-26.814,-999.000,
     &-25.416,-24.731,-22.698,-20.270,-19.053
     &/
       data (fjrocvz(it, 3,2, 8), it=1,59) /
     &-19.217,-18.009,-17.343,-16.208,-15.198,-15.561,
     &-15.271,-15.501,-16.317,-15.704,-11.233,-4.781,
     &1.125,5.477,8.210,11.609,16.220,20.071,
     &23.760,26.726,28.634,28.250,25.505,21.933,
     &18.993,17.269,15.249,13.255,12.054,12.164,
     &13.336,14.466,16.146,17.309,19.381,-999.000,
     &22.782,-999.000,25.974,31.421,35.371,34.073,
     &24.933,11.674,0.849,-12.178,-19.976,-20.584,
     &-21.993,-999.000,-27.035,-999.000,-28.014,-999.000,
     &-27.088,-25.051,-22.623,-20.793,-19.217
     &/
       data (fjrocvz(it, 4,2, 8), it=1,59) /
     &-20.406,-19.224,-17.617,-16.017,-15.606,-15.883,
     &-15.275,-13.976,-13.486,-12.703,-9.658,-4.181,
     &1.820,5.018,6.563,9.514,13.815,18.731,
     &22.557,24.915,25.272,23.878,21.289,18.238,
     &15.855,14.218,12.501,11.517,11.787,12.153,
     &13.129,14.734,16.193,18.379,21.923,-999.000,
     &24.141,-999.000,27.819,33.550,35.395,34.810,
     &29.955,17.698,0.760,-16.225,-23.160,-999.000,
     &-24.397,-999.000,-27.990,-999.000,-29.889,-999.000,
     &-27.941,-25.081,-23.019,-21.368,-20.406
     &/
       data (fjrocvz(it, 5,2, 8), it=1,59) /
     &-20.654,-19.307,-17.012,-15.334,-15.108,-16.088,
     &-16.071,-15.025,-14.729,-13.889,-10.904,-4.711,
     &1.229,4.887,7.884,10.345,14.667,20.283,
     &24.834,27.571,28.101,26.687,23.500,20.070,
     &17.358,15.702,14.311,13.736,13.535,13.283,
     &14.052,15.534,17.471,19.598,21.328,-999.000,
     &25.165,-999.000,29.149,33.409,36.641,39.029,
     &35.000,18.984,-0.036,-12.887,-22.678,-26.273,
     &-29.439,-32.283,-33.251,-32.492,-30.436,-999.000,
     &-27.600,-24.878,-22.720,-21.359,-20.654
     &/
       data (fjrocvz(it, 6,2, 8), it=1,59) /
     &-21.495,-20.627,-19.156,-17.766,-17.452,-17.597,
     &-16.536,-15.886,-15.705,-13.914,-11.734,-10.333,
     &-7.047,-0.305,6.540,11.915,16.475,21.845,
     &26.325,29.041,30.605,29.881,27.559,24.316,
     &21.798,20.119,18.362,16.625,15.591,14.514,
     &14.074,15.824,18.013,19.095,21.170,-999.000,
     &25.270,-999.000,29.357,36.362,42.314,44.100,
     &37.545,22.657,4.205,-10.109,-21.004,-27.471,
     &-30.616,-999.000,-32.824,-999.000,-31.661,-999.000,
     &-28.750,-25.244,-23.300,-22.457,-21.495
     &/
       data (fjrocvz(it, 7,2, 8), it=1,59) /
     &-23.477,-23.146,-22.138,-21.235,-20.712,-19.434,
     &-17.861,-16.902,-15.831,-14.750,-12.584,-10.186,
     &-4.956,-0.843,3.638,7.623,12.690,18.708,
     &24.013,28.124,29.929,30.111,28.918,26.666,
     &24.501,22.239,19.963,18.100,16.054,13.786,
     &13.063,13.915,15.326,17.297,19.031,-999.000,
     &21.909,-999.000,26.756,34.930,42.116,46.134,
     &36.067,17.804,1.159,-10.265,-20.916,-29.341,
     &-34.328,-35.346,-34.854,-33.987,-32.643,-999.000,
     &-29.432,-27.088,-25.337,-23.914,-23.477
     &/
       data (fjrocvz(it, 8,2, 8), it=1,59) /
     &-23.468,-22.290,-21.480,-21.120,-20.627,-20.336,
     &-19.802,-19.226,-18.107,-19.885,-21.687,-23.359,
     &-18.037,-9.481,-3.256,2.879,7.423,11.732,
     &17.866,22.667,25.640,26.177,25.609,24.430,
     &22.791,20.996,18.744,16.574,13.937,11.602,
     &10.576,10.869,12.277,14.020,15.478,-999.000,
     &18.993,-999.000,25.087,30.884,37.670,38.554,
     &27.211,13.695,1.263,-10.424,-18.907,-26.200,
     &-30.780,-31.269,-29.154,-999.000,-28.951,-999.000,
     &-27.704,-27.314,-26.134,-24.677,-23.468
     &/
       data (fjrocvz(it, 9,2, 8), it=1,59) /
     &-20.836,-18.873,-17.109,-16.045,-16.120,-17.080,
     &-18.806,-20.317,-21.451,-23.168,-25.747,-27.659,
     &-22.309,-11.621,-3.290,2.325,5.293,9.346,
     &14.473,18.390,20.685,21.412,21.252,19.911,
     &17.667,15.601,13.722,11.756,9.633,8.317,
     &8.219,8.827,10.002,11.447,13.237,-999.000,
     &16.090,-999.000,23.092,29.770,35.827,39.213,
     &35.550,22.628,9.436,-2.464,-10.911,-999.000,
     &-17.139,-999.000,-20.544,-999.000,-23.121,-999.000,
     &-23.462,-23.486,-23.625,-22.803,-20.836
     &/
       data (fjrocvz(it,10,2, 8), it=1,59) /
     &-21.613,-19.292,-16.968,-15.723,-15.013,-16.223,
     &-19.360,-22.055,-23.577,-24.339,-24.965,-24.695,
     &-20.448,-9.210,-0.941,4.575,7.361,11.726,
     &17.006,20.462,21.375,21.156,19.995,18.432,
     &16.468,13.925,11.614,9.704,8.481,7.939,
     &8.017,8.377,8.444,9.209,11.164,-999.000,
     &15.244,-999.000,23.316,30.973,39.442,43.496,
     &41.375,28.446,15.588,-999.000,0.521,-999.000,
     &-9.442,-999.000,-16.406,-999.000,-18.417,-999.000,
     &-20.214,-21.124,-22.074,-22.204,-21.613
     &/
       data (fjrocvz(it,11,2, 8), it=1,59) /
     &-24.367,-23.434,-22.920,-22.772,-22.481,-22.814,
     &-23.524,-24.951,-26.607,-28.523,-27.807,-23.986,
     &-18.421,-8.482,-0.797,5.364,9.712,13.514,
     &18.102,21.991,23.914,24.168,23.276,21.866,
     &20.504,18.269,15.405,13.374,11.964,11.016,
     &10.378,9.641,9.123,9.825,12.324,-999.000,
     &17.219,-999.000,24.151,31.470,40.462,41.718,
     &37.654,27.156,16.552,-999.000,-0.589,-999.000,
     &-11.633,-999.000,-18.736,-999.000,-20.368,-999.000,
     &-23.178,-23.000,-23.862,-24.989,-24.367
     &/
       data (fjrocvz(it,12,2, 8), it=1,59) /
     &-25.981,-26.465,-26.252,-26.045,-26.050,-25.515,
     &-25.373,-26.955,-29.653,-31.780,-32.510,-29.217,
     &-24.958,-15.273,-5.518,0.398,5.081,10.820,
     &17.827,23.106,25.888,26.337,25.559,24.648,
     &23.665,21.426,18.470,16.131,15.074,14.344,
     &13.417,12.163,11.263,11.984,14.921,-999.000,
     &19.352,-999.000,25.871,32.354,40.179,45.131,
     &40.393,23.598,9.985,0.280,-8.335,-999.000,
     &-13.861,-999.000,-18.855,-999.000,-22.870,-999.000,
     &-25.399,-25.363,-25.618,-25.953,-25.981
     &/
       data (fjrocvz(it,13,2, 8), it=1,59) /
     &-24.563,-23.929,-23.628,-23.542,-23.088,-23.646,
     &-25.145,-26.419,-27.749,-28.476,-28.921,-27.459,
     &-24.630,-16.137,-7.402,-0.828,3.462,9.984,
     &19.587,25.726,27.640,27.920,27.226,26.479,
     &25.405,22.931,20.145,18.315,18.093,17.960,
     &17.288,16.044,14.752,15.150,17.585,-999.000,
     &21.125,-999.000,29.761,35.825,44.112,49.489,
     &41.992,24.620,6.360,-4.626,-11.187,-13.230,
     &-14.300,-999.000,-18.006,-999.000,-20.901,-999.000,
     &-23.179,-25.165,-25.051,-24.540,-24.563
     &/
       data (fjrocvz(it,14,2, 8), it=1,59) /
     &-22.484,-21.970,-21.592,-20.974,-20.876,-22.473,
     &-25.233,-28.785,-31.932,-31.785,-28.926,-24.962,
     &-21.166,-17.621,-10.667,-2.842,2.842,9.905,
     &19.529,25.472,27.922,28.252,27.375,26.267,
     &24.640,22.787,20.746,19.518,19.627,19.616,
     &18.740,17.600,16.535,16.902,19.447,-999.000,
     &24.271,-999.000,32.883,41.958,47.810,47.329,
     &38.971,23.907,9.211,-999.000,-9.662,-999.000,
     &-17.420,-999.000,-19.932,-999.000,-22.409,-999.000,
     &-24.954,-25.716,-24.746,-23.282,-22.484
     &/
       data (fjrocvz(it,15,2, 8), it=1,59) /
     &-20.242,-19.730,-19.172,-18.713,-18.728,-20.418,
     &-24.383,-30.451,-34.671,-36.014,-32.627,-25.457,
     &-20.648,-15.498,-10.000,-2.411,4.292,10.651,
     &18.299,23.638,26.052,25.994,24.400,22.051,
     &19.889,18.375,16.528,15.349,15.198,15.013,
     &14.433,13.746,13.839,15.149,18.479,-999.000,
     &22.851,-999.000,30.259,38.357,44.287,46.358,
     &39.057,22.797,8.215,-999.000,-11.686,-999.000,
     &-18.923,-999.000,-22.934,-999.000,-25.937,-999.000,
     &-27.497,-27.683,-25.199,-21.839,-20.242
     &/
       data (fjrocvz(it,16,2, 8), it=1,59) /
     &-20.025,-18.905,-19.109,-19.365,-20.162,-22.398,
     &-26.189,-30.946,-34.682,-37.532,-36.437,-28.917,
     &-19.425,-13.760,-7.742,-1.043,5.455,10.969,
     &18.326,23.363,25.249,25.428,24.116,21.022,
     &17.364,14.222,11.783,10.388,10.258,10.534,
     &10.061,9.284,9.349,10.494,13.196,-999.000,
     &17.679,-999.000,24.808,32.369,38.965,40.499,
     &32.229,15.202,3.702,-999.000,-12.919,-999.000,
     &-20.952,-999.000,-25.724,-999.000,-28.579,-999.000,
     &-29.833,-29.288,-26.695,-22.993,-20.025
     &/
       data (fjrocvz(it,17,2, 8), it=1,59) /
     &-23.326,-22.727,-22.112,-21.748,-22.567,-24.449,
     &-27.671,-32.341,-36.319,-37.819,-36.059,-27.924,
     &-18.927,-12.443,-5.527,0.246,6.038,11.866,
     &19.727,25.105,27.602,28.404,27.383,24.659,
     &20.494,16.483,12.810,10.217,9.725,9.576,
     &8.737,7.543,6.480,6.472,9.054,-999.000,
     &14.524,-999.000,22.593,28.373,33.148,34.562,
     &27.930,12.829,-0.173,-12.245,-20.567,-24.527,
     &-26.548,-999.000,-27.894,-999.000,-29.963,-999.000,
     &-30.623,-29.407,-27.738,-25.376,-23.326
     &/
       data (fjrocvz(it,18,2, 8), it=1,59) /
     &-24.912,-23.774,-22.679,-22.442,-22.479,-23.558,
     &-27.318,-32.058,-35.158,-36.553,-33.968,-26.639,
     &-17.566,-11.079,-5.747,0.156,6.461,14.031,
     &21.391,27.315,31.410,32.346,30.255,26.474,
     &22.929,19.529,15.476,11.907,10.409,9.282,
     &8.157,6.705,5.699,5.634,7.953,-999.000,
     &12.933,16.731,21.918,27.695,31.646,29.152,
     &21.901,11.122,-1.625,-999.000,-20.376,-999.000,
     &-27.750,-999.000,-30.865,-999.000,-31.719,-999.000,
     &-31.112,-30.668,-29.160,-26.883,-24.912
     &/
       data (fjrocvz(it,19,2, 8), it=1,59) /
     &-23.580,-22.209,-22.161,-21.703,-21.291,-22.331,
     &-25.175,-28.018,-30.632,-32.135,-29.466,-22.628,
     &-14.514,-9.795,-4.526,1.426,7.456,15.662,
     &24.089,30.822,35.326,36.028,33.582,29.759,
     &25.775,21.868,17.784,14.161,13.085,12.426,
     &10.989,9.240,8.101,7.701,9.601,-999.000,
     &15.377,-999.000,23.874,29.927,32.418,29.099,
     &23.459,13.094,-2.594,-14.589,-23.540,-27.573,
     &-28.984,-999.000,-30.806,-999.000,-30.751,-999.000,
     &-30.263,-29.947,-28.740,-26.234,-23.580
     &/
       data (fjrocvz(it,20,2, 8), it=1,59) /
     &-25.140,-23.811,-22.195,-20.522,-19.441,-19.904,
     &-22.148,-25.358,-27.346,-28.009,-25.468,-16.432,
     &-8.395,-6.140,-1.491,3.065,8.011,16.347,
     &24.470,32.040,36.759,37.815,36.028,32.719,
     &29.179,25.791,21.589,18.073,16.771,15.556,
     &13.859,12.042,10.666,10.295,12.701,-999.000,
     &18.103,-999.000,25.030,30.789,36.000,37.351,
     &28.946,13.604,-2.564,-11.646,-19.049,-24.765,
     &-28.317,-999.000,-29.754,-999.000,-30.774,-999.000,
     &-30.513,-30.168,-29.333,-27.251,-25.140
     &/
       data (fjrocvz(it,21,2, 8), it=1,59) /
     &-25.018,-22.713,-20.865,-19.694,-19.017,-18.988,
     &-20.230,-23.045,-26.093,-26.810,-24.022,-16.768,
     &-11.367,-6.325,-1.167,4.147,9.837,16.360,
     &22.387,29.209,33.282,34.938,33.311,29.920,
     &26.681,24.047,20.545,16.647,14.724,13.282,
     &11.701,10.640,10.367,10.573,12.547,-999.000,
     &17.188,-999.000,25.042,30.684,36.339,35.030,
     &25.952,-999.000,1.599,-999.000,-15.099,-999.000,
     &-24.365,-999.000,-28.428,-999.000,-29.592,-999.000,
     &-29.743,-29.241,-28.486,-27.020,-25.018
     &/
       data (fjrocvz(it,22,2, 8), it=1,59) /
     &-23.878,-22.771,-21.611,-20.761,-20.650,-20.609,
     &-20.931,-23.236,-27.195,-28.026,-25.008,-18.120,
     &-9.992,-6.791,-1.214,3.760,8.827,14.793,
     &21.289,27.266,30.203,30.736,28.812,25.017,
     &20.648,17.110,13.778,10.897,9.459,8.465,
     &7.778,8.281,9.397,10.126,12.319,-999.000,
     &17.580,-999.000,25.806,28.573,32.025,28.332,
     &20.432,-999.000,-1.192,-999.000,-16.241,-999.000,
     &-25.473,-999.000,-27.399,-999.000,-27.736,-999.000,
     &-28.350,-27.834,-26.952,-25.430,-23.878
     &/
       data (fjrocvz(it,23,2, 8), it=1,59) /
     &-22.295,-21.808,-20.288,-18.312,-17.696,-18.606,
     &-20.456,-22.791,-24.944,-25.489,-22.833,-16.600,
     &-11.297,-8.358,-2.916,2.364,7.665,13.297,
     &21.434,27.214,29.151,28.438,25.935,21.997,
     &17.653,13.553,10.109,7.715,6.957,6.607,
     &6.360,6.448,8.096,10.331,13.464,-999.000,
     &17.350,-999.000,24.682,28.868,31.333,28.814,
     &21.758,9.336,-1.530,-999.000,-15.560,-999.000,
     &-22.598,-999.000,-25.463,-999.000,-26.278,-999.000,
     &-27.004,-26.569,-25.264,-23.166,-22.295
     &/
       data (fjrocvz(it,24,2, 8), it=1,59) /
     &-20.855,-19.350,-17.593,-15.493,-15.298,-17.077,
     &-19.220,-20.506,-21.578,-22.143,-19.995,-15.099,
     &-8.942,-3.918,-0.420,2.897,7.618,15.948,
     &24.176,27.841,28.669,27.806,24.838,20.578,
     &16.448,12.613,8.732,6.158,6.371,7.655,
     &8.085,7.885,9.331,11.611,14.443,-999.000,
     &17.960,-999.000,24.391,30.106,32.689,29.700,
     &19.765,-999.000,1.152,-999.000,-10.755,-999.000,
     &-18.007,-999.000,-23.073,-999.000,-23.967,-999.000,
     &-25.129,-25.095,-24.027,-22.484,-20.855
     &/
       data (fjrocvz(it,25,2, 8), it=1,59) /
     &-20.570,-19.688,-18.162,-17.084,-17.863,-18.776,
     &-19.429,-20.360,-21.041,-20.383,-17.098,-10.335,
     &-11.650,-6.767,-1.111,5.231,11.321,20.385,
     &27.501,30.412,30.291,29.223,26.860,22.820,
     &18.862,15.733,10.786,7.231,7.475,9.208,
     &9.995,10.448,11.256,13.281,16.261,-999.000,
     &20.156,-999.000,25.012,29.638,30.593,25.371,
     &16.816,-999.000,0.688,-999.000,-11.566,-999.000,
     &-18.387,-999.000,-20.955,-999.000,-24.308,-999.000,
     &-24.635,-24.420,-24.034,-22.439,-20.570
     &/
       data (fjrocvz(it, 1,3, 8), it=1,59) /
     &-20.829,-21.329,-20.744,-19.457,-17.347,-16.549,
     &-16.676,-20.117,-27.334,-29.135,-22.215,-8.517,
     &2.917,6.852,9.464,11.431,11.149,10.396,
     &10.850,13.559,17.115,18.109,17.574,15.193,
     &11.924,10.142,10.600,11.513,11.297,10.344,
     &9.603,9.489,9.231,8.835,9.395,-999.000,
     &12.957,-999.000,15.913,-999.000,21.584,25.436,
     &25.901,20.884,10.326,-999.000,-3.120,-999.000,
     &-9.953,-999.000,-12.477,-999.000,-14.054,-999.000,
     &-16.061,-18.493,-20.084,-20.204,-20.829
     &/
       data (fjrocvz(it, 2,3, 8), it=1,59) /
     &-19.247,-19.454,-18.905,-17.379,-16.011,-16.646,
     &-18.341,-22.064,-27.423,-28.448,-20.293,-4.667,
     &8.685,12.160,13.462,14.474,14.642,14.781,
     &15.791,17.547,19.322,19.954,19.090,17.159,
     &15.542,14.381,14.686,15.338,15.614,15.066,
     &14.375,13.916,13.451,13.999,14.330,-999.000,
     &15.819,-999.000,19.139,-999.000,23.345,26.207,
     &25.254,20.072,7.711,-999.000,-4.293,-999.000,
     &-11.060,-999.000,-14.019,-999.000,-15.837,-999.000,
     &-17.563,-19.777,-20.407,-19.445,-19.247
     &/
       data (fjrocvz(it, 3,3, 8), it=1,59) /
     &-18.491,-17.468,-16.896,-15.960,-15.535,-16.533,
     &-18.362,-20.827,-25.219,-28.780,-21.625,-2.564,
     &14.826,17.168,15.892,14.863,14.243,14.704,
     &15.874,16.861,16.934,16.116,14.796,14.168,
     &14.248,13.688,14.389,15.402,16.397,17.334,
     &17.167,16.538,16.058,16.764,17.344,-999.000,
     &19.261,-999.000,20.454,-999.000,23.148,23.064,
     &20.564,17.013,7.804,-999.000,-5.453,-999.000,
     &-12.574,-999.000,-16.484,-999.000,-19.014,-999.000,
     &-21.335,-23.343,-23.477,-20.964,-18.491
     &/
       data (fjrocvz(it, 4,3, 8), it=1,59) /
     &-20.077,-17.998,-17.287,-17.241,-16.537,-16.592,
     &-16.708,-17.919,-21.964,-24.655,-17.337,2.580,
     &17.252,19.761,18.627,17.963,17.069,15.660,
     &14.352,13.553,13.283,12.828,12.736,13.241,
     &13.554,13.638,14.201,15.388,16.788,17.452,
     &17.817,17.859,17.219,18.000,20.296,-999.000,
     &21.030,-999.000,21.120,-999.000,24.476,24.794,
     &22.212,13.023,4.799,-999.000,-6.055,-999.000,
     &-13.874,-999.000,-18.924,-999.000,-22.198,-999.000,
     &-24.072,-25.309,-25.240,-22.818,-20.077
     &/
       data (fjrocvz(it, 5,3, 8), it=1,59) /
     &-21.251,-19.998,-19.232,-17.393,-15.221,-14.974,
     &-15.102,-16.572,-18.894,-19.393,-13.418,0.950,
     &15.518,20.602,20.586,20.090,19.040,17.628,
     &16.171,15.342,15.839,16.207,16.217,15.524,
     &14.682,13.687,13.243,13.778,14.546,15.404,
     &16.450,17.276,17.419,18.676,20.082,-999.000,
     &21.269,-999.000,23.180,26.566,29.141,24.589,
     &18.834,10.368,2.108,-999.000,-8.318,-999.000,
     &-15.330,-999.000,-20.305,-999.000,-23.352,-999.000,
     &-25.625,-26.889,-26.088,-23.437,-21.251
     &/
       data (fjrocvz(it, 6,3, 8), it=1,59) /
     &-22.910,-22.953,-22.763,-20.292,-16.364,-14.515,
     &-14.673,-16.193,-17.078,-15.923,-11.635,-1.353,
     &12.303,17.305,17.296,16.862,16.856,16.972,
     &16.838,18.153,19.988,20.562,19.817,18.634,
     &17.342,15.637,14.408,13.486,13.821,14.402,
     &15.089,15.639,16.390,17.272,18.072,-999.000,
     &21.183,-999.000,22.934,25.603,25.159,18.372,
     &10.442,-999.000,-2.468,-999.000,-11.282,-999.000,
     &-18.244,-999.000,-23.113,-999.000,-26.161,-999.000,
     &-28.584,-29.534,-27.746,-24.967,-22.910
     &/
       data (fjrocvz(it, 7,3, 8), it=1,59) /
     &-25.749,-24.151,-23.729,-22.812,-19.921,-18.918,
     &-18.615,-18.230,-18.281,-15.504,-11.395,-4.252,
     &3.104,8.029,9.647,10.854,12.041,13.269,
     &14.770,17.033,19.815,21.430,21.872,21.765,
     &21.458,20.899,19.368,17.540,16.739,15.978,
     &15.614,15.079,14.768,14.712,14.978,-999.000,
     &16.051,-999.000,16.416,-999.000,13.471,-999.000,
     &4.083,-999.000,-4.191,-999.000,-12.180,-999.000,
     &-19.195,-999.000,-24.336,-999.000,-27.842,-999.000,
     &-30.332,-31.350,-30.416,-28.387,-25.749
     &/
       data (fjrocvz(it, 8,3, 8), it=1,59) /
     &-23.324,-22.166,-23.239,-23.796,-22.695,-20.174,
     &-16.133,-13.752,-13.272,-11.982,-9.037,-5.197,
     &-2.711,-0.046,1.620,3.594,5.906,7.722,
     &9.436,11.501,14.988,18.724,20.763,21.429,
     &21.801,21.216,19.681,17.565,16.031,15.386,
     &14.831,13.981,12.816,11.194,9.776,-999.000,
     &9.685,-999.000,9.168,-999.000,7.115,-999.000,
     &2.984,-999.000,-0.662,-999.000,-6.851,-999.000,
     &-13.827,-999.000,-19.520,-999.000,-24.475,-999.000,
     &-28.006,-29.205,-27.940,-25.613,-23.324
     &/
       data (fjrocvz(it, 9,3, 8), it=1,59) /
     &-17.684,-17.974,-18.345,-18.913,-18.858,-16.878,
     &-13.429,-11.702,-12.098,-12.513,-11.665,-9.254,
     &-8.643,-4.054,0.807,5.426,10.274,11.366,
     &12.292,13.210,14.342,14.243,15.668,16.925,
     &18.016,18.184,16.521,14.022,12.666,11.534,
     &10.284,8.983,7.904,7.145,6.599,-999.000,
     &5.564,-999.000,4.478,-999.000,5.467,-999.000,
     &6.500,-999.000,6.481,-999.000,1.841,-999.000,
     &-3.312,-999.000,-8.710,-999.000,-12.432,-999.000,
     &-15.800,-17.499,-17.701,-17.666,-17.684
     &/
       data (fjrocvz(it,10,3, 8), it=1,59) /
     &-15.137,-15.859,-15.172,-14.255,-14.899,-16.264,
     &-16.039,-15.053,-14.942,-14.755,-14.419,-15.366,
     &-17.023,-8.638,0.258,6.432,12.955,13.966,
     &14.904,15.603,16.456,16.994,17.033,17.087,
     &17.324,17.622,16.147,14.141,12.291,10.211,
     &8.059,5.607,4.007,3.023,1.814,-999.000,
     &0.531,-999.000,1.092,-999.000,3.109,-999.000,
     &7.173,-999.000,9.465,-999.000,8.661,-999.000,
     &4.624,-999.000,-1.047,-999.000,-6.163,-999.000,
     &-9.325,-11.803,-13.273,-14.279,-15.137
     &/
       data (fjrocvz(it,11,3, 8), it=1,59) /
     &-18.629,-20.301,-20.580,-19.906,-19.662,-19.191,
     &-17.703,-17.475,-20.773,-19.439,-19.068,-12.967,
     &-16.116,-9.678,-3.296,3.449,10.534,12.845,
     &14.925,16.594,19.825,22.241,21.753,21.114,
     &20.894,20.180,19.136,17.748,15.452,12.776,
     &9.771,7.145,5.596,3.427,1.312,-999.000,
     &0.204,-999.000,0.473,-999.000,3.182,-999.000,
     &10.865,14.107,14.407,11.792,9.837,-999.000,
     &5.849,-999.000,1.227,-999.000,-4.381,-999.000,
     &-8.540,-12.894,-15.521,-17.123,-18.629
     &/
       data (fjrocvz(it,12,3, 8), it=1,59) /
     &-19.376,-21.729,-24.182,-25.997,-25.907,-23.843,
     &-23.492,-27.580,-32.709,-30.398,-24.708,-15.037,
     &-13.912,-10.108,-6.432,-0.144,6.487,10.352,
     &13.704,17.967,22.424,24.512,24.633,24.127,
     &23.504,22.137,20.647,19.557,17.440,14.582,
     &11.798,10.263,8.931,7.269,5.260,-999.000,
     &3.453,-999.000,4.758,-999.000,11.807,17.304,
     &21.722,23.078,20.789,-999.000,14.900,-999.000,
     &7.909,-999.000,1.737,-999.000,-2.533,-999.000,
     &-7.503,-11.893,-15.556,-17.724,-19.376
     &/
       data (fjrocvz(it,13,3, 8), it=1,59) /
     &-18.012,-19.953,-20.841,-21.977,-24.212,-25.949,
     &-28.396,-31.194,-33.543,-31.676,-24.536,-16.726,
     &-12.783,-11.015,-7.654,-3.882,-0.065,6.661,
     &14.910,20.856,24.784,26.953,27.162,25.933,
     &24.687,23.291,21.700,20.096,17.901,15.711,
     &13.757,11.607,10.035,8.998,7.696,-999.000,
     &7.597,-999.000,10.052,-999.000,15.619,21.422,
     &28.321,30.810,25.156,-999.000,17.467,-999.000,
     &9.410,-999.000,1.972,-999.000,-2.882,-999.000,
     &-6.832,-9.431,-12.226,-15.513,-18.012
     &/
       data (fjrocvz(it,14,3, 8), it=1,59) /
     &-18.513,-20.441,-20.476,-21.453,-24.172,-27.867,
     &-30.225,-32.440,-35.500,-34.224,-25.641,-14.654,
     &-9.614,-8.408,-6.592,-3.640,-0.504,5.436,
     &14.623,20.370,24.674,26.922,27.051,25.277,
     &23.825,22.810,21.834,20.535,19.019,17.461,
     &15.453,13.171,11.752,10.474,9.950,-999.000,
     &10.369,-999.000,12.805,-999.000,18.280,22.929,
     &29.786,32.755,30.198,22.594,17.211,-999.000,
     &8.300,-999.000,1.878,-999.000,-2.471,-999.000,
     &-6.018,-8.650,-11.461,-15.383,-18.513
     &/
       data (fjrocvz(it,15,3, 8), it=1,59) /
     &-18.543,-19.689,-20.025,-20.502,-22.318,-25.724,
     &-28.581,-31.699,-35.210,-36.086,-27.925,-16.317,
     &-9.477,-8.393,-7.542,-5.112,-1.506,4.177,
     &11.927,17.501,21.365,23.477,22.846,20.714,
     &18.833,17.444,16.637,15.395,13.672,12.607,
     &11.536,10.402,8.906,7.932,7.663,-999.000,
     &7.846,-999.000,10.216,-999.000,14.994,19.422,
     &23.637,24.826,21.123,15.603,11.225,-999.000,
     &3.407,-999.000,-1.668,-999.000,-4.630,-999.000,
     &-8.427,-10.807,-13.153,-15.864,-18.543
     &/
       data (fjrocvz(it,16,3, 8), it=1,59) /
     &-16.363,-16.896,-16.699,-16.984,-19.029,-23.149,
     &-27.490,-30.782,-35.647,-38.428,-34.117,-21.458,
     &-11.149,-8.256,-7.490,-5.335,-1.758,3.549,
     &10.375,15.582,18.889,20.071,18.718,16.992,
     &15.299,13.497,11.609,9.069,6.575,5.265,
     &4.694,3.843,2.729,1.338,0.882,-999.000,
     &2.122,-999.000,4.708,-999.000,9.042,12.785,
     &14.486,12.759,9.718,-999.000,1.334,-999.000,
     &-5.013,-999.000,-8.582,-999.000,-9.786,-999.000,
     &-10.913,-12.455,-13.716,-14.885,-16.363
     &/
       data (fjrocvz(it,17,3, 8), it=1,59) /
     &-17.359,-17.558,-16.767,-16.577,-18.125,-21.331,
     &-25.794,-32.074,-39.069,-41.707,-36.796,-21.469,
     &-9.389,-7.016,-5.320,-2.361,1.175,5.886,
     &12.054,17.411,20.625,21.789,20.855,18.601,
     &16.323,14.202,11.988,8.988,6.104,3.891,
     &1.810,0.673,-0.283,-1.695,-2.593,-999.000,
     &-2.693,-999.000,-0.642,-999.000,4.954,9.868,
     &10.520,7.394,3.332,-999.000,-3.905,-999.000,
     &-10.593,-999.000,-14.188,-999.000,-14.927,-999.000,
     &-15.264,-16.007,-16.365,-16.571,-17.359
     &/
       data (fjrocvz(it,18,3, 8), it=1,59) /
     &-24.440,-23.167,-21.399,-19.870,-19.694,-21.427,
     &-24.430,-30.966,-39.770,-41.820,-35.305,-13.672,
     &-4.287,-3.289,-1.499,1.715,5.486,11.045,
     &17.337,22.282,24.653,25.785,25.312,23.515,
     &20.846,17.727,15.121,12.032,9.194,6.229,
     &3.600,1.832,0.297,-1.173,-2.721,-999.000,
     &-3.219,-999.000,-0.978,-999.000,5.615,7.691,
     &5.873,1.309,-3.000,-999.000,-10.178,-999.000,
     &-16.552,-999.000,-19.350,-999.000,-20.027,-999.000,
     &-20.509,-21.868,-22.933,-23.671,-24.440
     &/
       data (fjrocvz(it,19,3, 8), it=1,59) /
     &-25.809,-25.287,-23.543,-21.872,-21.586,-23.304,
     &-24.271,-26.799,-33.103,-33.387,-27.461,-10.598,
     &-1.510,0.274,1.921,4.784,9.187,14.780,
     &21.002,25.614,28.254,28.855,28.063,26.499,
     &24.198,21.420,18.331,15.304,12.418,9.457,
     &7.095,5.627,4.377,2.118,0.398,-999.000,
     &0.549,-999.000,1.668,-999.000,7.848,8.983,
     &6.822,2.180,-3.822,-999.000,-11.262,-999.000,
     &-17.110,-999.000,-19.409,-999.000,-20.626,-999.000,
     &-21.276,-22.517,-23.968,-25.121,-25.809
     &/
       data (fjrocvz(it,20,3, 8), it=1,59) /
     &-22.669,-22.699,-21.408,-19.869,-20.114,-22.820,
     &-25.395,-27.814,-29.342,-26.415,-19.792,-9.798,
     &-1.774,1.718,3.404,5.484,9.236,15.056,
     &21.598,27.543,30.513,30.678,29.169,26.843,
     &24.455,22.356,20.823,19.360,16.975,14.119,
     &11.306,9.393,7.206,5.337,4.439,-999.000,
     &4.055,-999.000,5.410,-999.000,10.371,12.604,
     &11.723,7.893,2.214,-999.000,-6.619,-999.000,
     &-11.870,-999.000,-13.961,-999.000,-15.901,-999.000,
     &-17.828,-19.795,-21.355,-21.864,-22.669
     &/
       data (fjrocvz(it,21,3, 8), it=1,59) /
     &-19.392,-19.726,-18.621,-17.264,-17.601,-21.082,
     &-23.922,-25.650,-28.599,-30.581,-27.369,-16.864,
     &-5.125,-0.283,1.816,3.632,6.371,10.943,
     &17.063,22.490,25.582,26.042,25.119,24.048,
     &22.546,21.058,20.147,18.697,16.636,13.990,
     &11.730,9.637,7.916,6.932,6.152,-999.000,
     &6.075,-999.000,6.226,-999.000,7.959,9.864,
     &12.219,10.532,6.222,-999.000,-1.402,-999.000,
     &-6.554,-999.000,-9.442,-999.000,-12.219,-999.000,
     &-14.764,-16.793,-18.259,-18.629,-19.392
     &/
       data (fjrocvz(it,22,3, 8), it=1,59) /
     &-17.717,-16.518,-14.835,-14.438,-15.670,-18.736,
     &-21.212,-24.700,-30.611,-34.472,-31.953,-21.274,
     &-9.296,-2.058,-0.176,1.395,3.429,6.775,
     &11.293,15.482,18.825,19.235,18.480,18.207,
     &17.811,16.727,15.627,13.901,11.585,9.413,
     &7.643,6.576,5.664,5.715,6.729,-999.000,
     &7.191,-999.000,7.021,-999.000,9.727,10.863,
     &14.800,13.984,7.489,-999.000,0.072,-999.000,
     &-5.361,-999.000,-8.987,-999.000,-11.539,-999.000,
     &-13.847,-15.835,-17.516,-18.027,-17.717
     &/
       data (fjrocvz(it,23,3, 8), it=1,59) /
     &-18.942,-17.088,-15.006,-13.777,-14.803,-18.276,
     &-21.739,-25.017,-28.572,-30.042,-26.062,-15.958,
     &-4.574,0.083,0.406,1.175,2.382,4.739,
     &7.897,12.337,16.687,18.169,17.094,15.252,
     &13.570,11.612,10.420,9.185,7.231,5.357,
     &4.791,4.428,3.582,3.871,5.292,-999.000,
     &6.406,-999.000,8.524,-999.000,13.276,17.363,
     &20.824,19.340,12.092,-999.000,0.872,-999.000,
     &-5.364,-999.000,-9.185,-999.000,-11.778,-999.000,
     &-14.470,-16.766,-18.353,-18.984,-18.942
     &/
       data (fjrocvz(it,24,3, 8), it=1,59) /
     &-20.975,-20.404,-18.553,-16.095,-14.717,-16.319,
     &-18.459,-21.259,-25.961,-26.866,-22.335,-11.264,
     &-0.513,3.133,4.416,5.499,5.790,6.781,
     &8.754,12.019,15.744,16.612,15.850,14.269,
     &11.342,8.802,7.634,7.950,7.310,6.591,
     &6.414,6.059,5.588,5.512,6.137,-999.000,
     &9.394,-999.000,12.985,-999.000,18.938,23.504,
     &24.212,21.428,15.133,-999.000,-0.057,-999.000,
     &-7.173,-999.000,-10.284,-999.000,-12.752,-999.000,
     &-15.984,-18.676,-20.096,-20.528,-20.975
     &/
       data (fjrocvz(it,25,3, 8), it=1,59) /
     &-20.829,-21.329,-20.744,-19.457,-17.347,-16.549,
     &-16.676,-20.117,-27.334,-29.135,-22.215,-8.517,
     &2.917,6.852,9.464,11.431,11.149,10.396,
     &10.850,13.559,17.115,18.109,17.574,15.193,
     &11.924,10.142,10.600,11.513,11.297,10.344,
     &9.603,9.489,9.231,8.835,9.395,-999.000,
     &12.957,-999.000,15.913,-999.000,21.584,25.436,
     &25.901,20.884,10.326,-999.000,-3.120,-999.000,
     &-9.953,-999.000,-12.477,-999.000,-14.054,-999.000,
     &-16.061,-18.493,-20.084,-20.204,-20.829
     &/
       data (fjrocvz(it, 1,4, 8), it=1,59) /
     &-20.570,-19.688,-18.162,-17.084,-17.863,-18.776,
     &-19.429,-20.360,-21.041,-20.383,-17.098,-10.335,
     &-11.650,-6.767,-1.111,5.231,11.321,20.385,
     &27.501,30.412,30.291,29.223,26.860,22.820,
     &18.862,15.733,10.786,7.231,7.475,9.208,
     &9.995,10.448,11.256,13.281,16.261,-999.000,
     &20.156,-999.000,25.012,29.638,30.593,25.371,
     &16.816,-999.000,0.688,-999.000,-11.566,-999.000,
     &-18.387,-999.000,-20.955,-999.000,-24.308,-999.000,
     &-24.635,-24.420,-24.034,-22.439,-20.570
     &/
       data (fjrocvz(it, 2,4, 8), it=1,59) /
     &-19.053,-18.329,-17.420,-16.972,-16.272,-15.813,
     &-16.094,-17.786,-19.202,-18.597,-13.272,-5.400,
     &0.118,4.052,8.748,11.433,16.187,21.881,
     &27.494,31.405,32.619,31.924,29.257,25.097,
     &21.151,18.703,15.290,11.981,10.724,11.103,
     &12.404,13.387,14.537,16.573,18.631,-999.000,
     &21.058,-999.000,26.261,29.494,31.750,30.994,
     &24.909,-999.000,1.468,-999.000,-15.222,-999.000,
     &-18.672,-999.000,-24.012,-999.000,-26.814,-999.000,
     &-25.416,-24.731,-22.698,-20.270,-19.053
     &/
       data (fjrocvz(it, 3,4, 8), it=1,59) /
     &-19.217,-18.009,-17.343,-16.208,-15.198,-15.561,
     &-15.271,-15.501,-16.317,-15.704,-11.233,-4.781,
     &1.125,5.477,8.210,11.609,16.220,20.071,
     &23.760,26.726,28.634,28.250,25.505,21.933,
     &18.993,17.269,15.249,13.255,12.054,12.164,
     &13.336,14.466,16.146,17.309,19.381,-999.000,
     &22.782,-999.000,25.974,31.421,35.371,34.073,
     &24.933,11.674,0.849,-12.178,-19.976,-20.584,
     &-21.993,-999.000,-27.035,-999.000,-28.014,-999.000,
     &-27.088,-25.051,-22.623,-20.793,-19.217
     &/
       data (fjrocvz(it, 4,4, 8), it=1,59) /
     &-20.406,-19.224,-17.617,-16.017,-15.606,-15.883,
     &-15.275,-13.976,-13.486,-12.703,-9.658,-4.181,
     &1.820,5.018,6.563,9.514,13.815,18.731,
     &22.557,24.915,25.272,23.878,21.289,18.238,
     &15.855,14.218,12.501,11.517,11.787,12.153,
     &13.129,14.734,16.193,18.379,21.923,-999.000,
     &24.141,-999.000,27.819,33.550,35.395,34.810,
     &29.955,17.698,0.760,-16.225,-23.160,-999.000,
     &-24.397,-999.000,-27.990,-999.000,-29.889,-999.000,
     &-27.941,-25.081,-23.019,-21.368,-20.406
     &/
       data (fjrocvz(it, 5,4, 8), it=1,59) /
     &-20.654,-19.307,-17.012,-15.334,-15.108,-16.088,
     &-16.071,-15.025,-14.729,-13.889,-10.904,-4.711,
     &1.229,4.887,7.884,10.345,14.667,20.283,
     &24.834,27.571,28.101,26.687,23.500,20.070,
     &17.358,15.702,14.311,13.736,13.535,13.283,
     &14.052,15.534,17.471,19.598,21.328,-999.000,
     &25.165,-999.000,29.149,33.409,36.641,39.029,
     &35.000,18.984,-0.036,-12.887,-22.678,-26.273,
     &-29.439,-32.283,-33.251,-32.492,-30.436,-999.000,
     &-27.600,-24.878,-22.720,-21.359,-20.654
     &/
       data (fjrocvz(it, 6,4, 8), it=1,59) /
     &-21.495,-20.627,-19.156,-17.766,-17.452,-17.597,
     &-16.536,-15.886,-15.705,-13.914,-11.734,-10.333,
     &-7.047,-0.305,6.540,11.915,16.475,21.845,
     &26.325,29.041,30.605,29.881,27.559,24.316,
     &21.798,20.119,18.362,16.625,15.591,14.514,
     &14.074,15.824,18.013,19.095,21.170,-999.000,
     &25.270,-999.000,29.357,36.362,42.314,44.100,
     &37.545,22.657,4.205,-10.109,-21.004,-27.471,
     &-30.616,-999.000,-32.824,-999.000,-31.661,-999.000,
     &-28.750,-25.244,-23.300,-22.457,-21.495
     &/
       data (fjrocvz(it, 7,4, 8), it=1,59) /
     &-23.477,-23.146,-22.138,-21.235,-20.712,-19.434,
     &-17.861,-16.902,-15.831,-14.750,-12.584,-10.186,
     &-4.956,-0.843,3.638,7.623,12.690,18.708,
     &24.013,28.124,29.929,30.111,28.918,26.666,
     &24.501,22.239,19.963,18.100,16.054,13.786,
     &13.063,13.915,15.326,17.297,19.031,-999.000,
     &21.909,-999.000,26.756,34.930,42.116,46.134,
     &36.067,17.804,1.159,-10.265,-20.916,-29.341,
     &-34.328,-35.346,-34.854,-33.987,-32.643,-999.000,
     &-29.432,-27.088,-25.337,-23.914,-23.477
     &/
       data (fjrocvz(it, 8,4, 8), it=1,59) /
     &-23.468,-22.290,-21.480,-21.120,-20.627,-20.336,
     &-19.802,-19.226,-18.107,-19.885,-21.687,-23.359,
     &-18.037,-9.481,-3.256,2.879,7.423,11.732,
     &17.866,22.667,25.640,26.177,25.609,24.430,
     &22.791,20.996,18.744,16.574,13.937,11.602,
     &10.576,10.869,12.277,14.020,15.478,-999.000,
     &18.993,-999.000,25.087,30.884,37.670,38.554,
     &27.211,13.695,1.263,-10.424,-18.907,-26.200,
     &-30.780,-31.269,-29.154,-999.000,-28.951,-999.000,
     &-27.704,-27.314,-26.134,-24.677,-23.468
     &/
       data (fjrocvz(it, 9,4, 8), it=1,59) /
     &-20.836,-18.873,-17.109,-16.045,-16.120,-17.080,
     &-18.806,-20.317,-21.451,-23.168,-25.747,-27.659,
     &-22.309,-11.621,-3.290,2.325,5.293,9.346,
     &14.473,18.390,20.685,21.412,21.252,19.911,
     &17.667,15.601,13.722,11.756,9.633,8.317,
     &8.219,8.827,10.002,11.447,13.237,-999.000,
     &16.090,-999.000,23.092,29.770,35.827,39.213,
     &35.550,22.628,9.436,-2.464,-10.911,-999.000,
     &-17.139,-999.000,-20.544,-999.000,-23.121,-999.000,
     &-23.462,-23.486,-23.625,-22.803,-20.836
     &/
       data (fjrocvz(it,10,4, 8), it=1,59) /
     &-21.613,-19.292,-16.968,-15.723,-15.013,-16.223,
     &-19.360,-22.055,-23.577,-24.339,-24.965,-24.695,
     &-20.448,-9.210,-0.941,4.575,7.361,11.726,
     &17.006,20.462,21.375,21.156,19.995,18.432,
     &16.468,13.925,11.614,9.704,8.481,7.939,
     &8.017,8.377,8.444,9.209,11.164,-999.000,
     &15.244,-999.000,23.316,30.973,39.442,43.496,
     &41.375,28.446,15.588,-999.000,0.521,-999.000,
     &-9.442,-999.000,-16.406,-999.000,-18.417,-999.000,
     &-20.214,-21.124,-22.074,-22.204,-21.613
     &/
       data (fjrocvz(it,11,4, 8), it=1,59) /
     &-24.367,-23.434,-22.920,-22.772,-22.481,-22.814,
     &-23.524,-24.951,-26.607,-28.523,-27.807,-23.986,
     &-18.421,-8.482,-0.797,5.364,9.712,13.514,
     &18.102,21.991,23.914,24.168,23.276,21.866,
     &20.504,18.269,15.405,13.374,11.964,11.016,
     &10.378,9.641,9.123,9.825,12.324,-999.000,
     &17.219,-999.000,24.151,31.470,40.462,41.718,
     &37.654,27.156,16.552,-999.000,-0.589,-999.000,
     &-11.633,-999.000,-18.736,-999.000,-20.368,-999.000,
     &-23.178,-23.000,-23.862,-24.989,-24.367
     &/
       data (fjrocvz(it,12,4, 8), it=1,59) /
     &-25.981,-26.465,-26.252,-26.045,-26.050,-25.515,
     &-25.373,-26.955,-29.653,-31.780,-32.510,-29.217,
     &-24.958,-15.273,-5.518,0.398,5.081,10.820,
     &17.827,23.106,25.888,26.337,25.559,24.648,
     &23.665,21.426,18.470,16.131,15.074,14.344,
     &13.417,12.163,11.263,11.984,14.921,-999.000,
     &19.352,-999.000,25.871,32.354,40.179,45.131,
     &40.393,23.598,9.985,0.280,-8.335,-999.000,
     &-13.861,-999.000,-18.855,-999.000,-22.870,-999.000,
     &-25.399,-25.363,-25.618,-25.953,-25.981
     &/
       data (fjrocvz(it,13,4, 8), it=1,59) /
     &-24.563,-23.929,-23.628,-23.542,-23.088,-23.646,
     &-25.145,-26.419,-27.749,-28.476,-28.921,-27.459,
     &-24.630,-16.137,-7.402,-0.828,3.462,9.984,
     &19.587,25.726,27.640,27.920,27.226,26.479,
     &25.405,22.931,20.145,18.315,18.093,17.960,
     &17.288,16.044,14.752,15.150,17.585,-999.000,
     &21.125,-999.000,29.761,35.825,44.112,49.489,
     &41.992,24.620,6.360,-4.626,-11.187,-13.230,
     &-14.300,-999.000,-18.006,-999.000,-20.901,-999.000,
     &-23.179,-25.165,-25.051,-24.540,-24.563
     &/
       data (fjrocvz(it,14,4, 8), it=1,59) /
     &-22.484,-21.970,-21.592,-20.974,-20.876,-22.473,
     &-25.233,-28.785,-31.932,-31.785,-28.926,-24.962,
     &-21.166,-17.621,-10.667,-2.842,2.842,9.905,
     &19.529,25.472,27.922,28.252,27.375,26.267,
     &24.640,22.787,20.746,19.518,19.627,19.616,
     &18.740,17.600,16.535,16.902,19.447,-999.000,
     &24.271,-999.000,32.883,41.958,47.810,47.329,
     &38.971,23.907,9.211,-999.000,-9.662,-999.000,
     &-17.420,-999.000,-19.932,-999.000,-22.409,-999.000,
     &-24.954,-25.716,-24.746,-23.282,-22.484
     &/
       data (fjrocvz(it,15,4, 8), it=1,59) /
     &-20.242,-19.730,-19.172,-18.713,-18.728,-20.418,
     &-24.383,-30.451,-34.671,-36.014,-32.627,-25.457,
     &-20.648,-15.498,-10.000,-2.411,4.292,10.651,
     &18.299,23.638,26.052,25.994,24.400,22.051,
     &19.889,18.375,16.528,15.349,15.198,15.013,
     &14.433,13.746,13.839,15.149,18.479,-999.000,
     &22.851,-999.000,30.259,38.357,44.287,46.358,
     &39.057,22.797,8.215,-999.000,-11.686,-999.000,
     &-18.923,-999.000,-22.934,-999.000,-25.937,-999.000,
     &-27.497,-27.683,-25.199,-21.839,-20.242
     &/
       data (fjrocvz(it,16,4, 8), it=1,59) /
     &-20.025,-18.905,-19.109,-19.365,-20.162,-22.398,
     &-26.189,-30.946,-34.682,-37.532,-36.437,-28.917,
     &-19.425,-13.760,-7.742,-1.043,5.455,10.969,
     &18.326,23.363,25.249,25.428,24.116,21.022,
     &17.364,14.222,11.783,10.388,10.258,10.534,
     &10.061,9.284,9.349,10.494,13.196,-999.000,
     &17.679,-999.000,24.808,32.369,38.965,40.499,
     &32.229,15.202,3.702,-999.000,-12.919,-999.000,
     &-20.952,-999.000,-25.724,-999.000,-28.579,-999.000,
     &-29.833,-29.288,-26.695,-22.993,-20.025
     &/
       data (fjrocvz(it,17,4, 8), it=1,59) /
     &-23.326,-22.727,-22.112,-21.748,-22.567,-24.449,
     &-27.671,-32.341,-36.319,-37.819,-36.059,-27.924,
     &-18.927,-12.443,-5.527,0.246,6.038,11.866,
     &19.727,25.105,27.602,28.404,27.383,24.659,
     &20.494,16.483,12.810,10.217,9.725,9.576,
     &8.737,7.543,6.480,6.472,9.054,-999.000,
     &14.524,-999.000,22.593,28.373,33.148,34.562,
     &27.930,12.829,-0.173,-12.245,-20.567,-24.527,
     &-26.548,-999.000,-27.894,-999.000,-29.963,-999.000,
     &-30.623,-29.407,-27.738,-25.376,-23.326
     &/
       data (fjrocvz(it,18,4, 8), it=1,59) /
     &-24.912,-23.774,-22.679,-22.442,-22.479,-23.558,
     &-27.318,-32.058,-35.158,-36.553,-33.968,-26.639,
     &-17.566,-11.079,-5.747,0.156,6.461,14.031,
     &21.391,27.315,31.410,32.346,30.255,26.474,
     &22.929,19.529,15.476,11.907,10.409,9.282,
     &8.157,6.705,5.699,5.634,7.953,-999.000,
     &12.933,16.731,21.918,27.695,31.646,29.152,
     &21.901,11.122,-1.625,-999.000,-20.376,-999.000,
     &-27.750,-999.000,-30.865,-999.000,-31.719,-999.000,
     &-31.112,-30.668,-29.160,-26.883,-24.912
     &/
       data (fjrocvz(it,19,4, 8), it=1,59) /
     &-23.580,-22.209,-22.161,-21.703,-21.291,-22.331,
     &-25.175,-28.018,-30.632,-32.135,-29.466,-22.628,
     &-14.514,-9.795,-4.526,1.426,7.456,15.662,
     &24.089,30.822,35.326,36.028,33.582,29.759,
     &25.775,21.868,17.784,14.161,13.085,12.426,
     &10.989,9.240,8.101,7.701,9.601,-999.000,
     &15.377,-999.000,23.874,29.927,32.418,29.099,
     &23.459,13.094,-2.594,-14.589,-23.540,-27.573,
     &-28.984,-999.000,-30.806,-999.000,-30.751,-999.000,
     &-30.263,-29.947,-28.740,-26.234,-23.580
     &/
       data (fjrocvz(it,20,4, 8), it=1,59) /
     &-25.140,-23.811,-22.195,-20.522,-19.441,-19.904,
     &-22.148,-25.358,-27.346,-28.009,-25.468,-16.432,
     &-8.395,-6.140,-1.491,3.065,8.011,16.347,
     &24.470,32.040,36.759,37.815,36.028,32.719,
     &29.179,25.791,21.589,18.073,16.771,15.556,
     &13.859,12.042,10.666,10.295,12.701,-999.000,
     &18.103,-999.000,25.030,30.789,36.000,37.351,
     &28.946,13.604,-2.564,-11.646,-19.049,-24.765,
     &-28.317,-999.000,-29.754,-999.000,-30.774,-999.000,
     &-30.513,-30.168,-29.333,-27.251,-25.140
     &/
       data (fjrocvz(it,21,4, 8), it=1,59) /
     &-25.018,-22.713,-20.865,-19.694,-19.017,-18.988,
     &-20.230,-23.045,-26.093,-26.810,-24.022,-16.768,
     &-11.367,-6.325,-1.167,4.147,9.837,16.360,
     &22.387,29.209,33.282,34.938,33.311,29.920,
     &26.681,24.047,20.545,16.647,14.724,13.282,
     &11.701,10.640,10.367,10.573,12.547,-999.000,
     &17.188,-999.000,25.042,30.684,36.339,35.030,
     &25.952,-999.000,1.599,-999.000,-15.099,-999.000,
     &-24.365,-999.000,-28.428,-999.000,-29.592,-999.000,
     &-29.743,-29.241,-28.486,-27.020,-25.018
     &/
       data (fjrocvz(it,22,4, 8), it=1,59) /
     &-23.878,-22.771,-21.611,-20.761,-20.650,-20.609,
     &-20.931,-23.236,-27.195,-28.026,-25.008,-18.120,
     &-9.992,-6.791,-1.214,3.760,8.827,14.793,
     &21.289,27.266,30.203,30.736,28.812,25.017,
     &20.648,17.110,13.778,10.897,9.459,8.465,
     &7.778,8.281,9.397,10.126,12.319,-999.000,
     &17.580,-999.000,25.806,28.573,32.025,28.332,
     &20.432,-999.000,-1.192,-999.000,-16.241,-999.000,
     &-25.473,-999.000,-27.399,-999.000,-27.736,-999.000,
     &-28.350,-27.834,-26.952,-25.430,-23.878
     &/
       data (fjrocvz(it,23,4, 8), it=1,59) /
     &-22.295,-21.808,-20.288,-18.312,-17.696,-18.606,
     &-20.456,-22.791,-24.944,-25.489,-22.833,-16.600,
     &-11.297,-8.358,-2.916,2.364,7.665,13.297,
     &21.434,27.214,29.151,28.438,25.935,21.997,
     &17.653,13.553,10.109,7.715,6.957,6.607,
     &6.360,6.448,8.096,10.331,13.464,-999.000,
     &17.350,-999.000,24.682,28.868,31.333,28.814,
     &21.758,9.336,-1.530,-999.000,-15.560,-999.000,
     &-22.598,-999.000,-25.463,-999.000,-26.278,-999.000,
     &-27.004,-26.569,-25.264,-23.166,-22.295
     &/
       data (fjrocvz(it,24,4, 8), it=1,59) /
     &-20.855,-19.350,-17.593,-15.493,-15.298,-17.077,
     &-19.220,-20.506,-21.578,-22.143,-19.995,-15.099,
     &-8.942,-3.918,-0.420,2.897,7.618,15.948,
     &24.176,27.841,28.669,27.806,24.838,20.578,
     &16.448,12.613,8.732,6.158,6.371,7.655,
     &8.085,7.885,9.331,11.611,14.443,-999.000,
     &17.960,-999.000,24.391,30.106,32.689,29.700,
     &19.765,-999.000,1.152,-999.000,-10.755,-999.000,
     &-18.007,-999.000,-23.073,-999.000,-23.967,-999.000,
     &-25.129,-25.095,-24.027,-22.484,-20.855
     &/
       data (fjrocvz(it,25,4, 8), it=1,59) /
     &-20.570,-19.688,-18.162,-17.084,-17.863,-18.776,
     &-19.429,-20.360,-21.041,-20.383,-17.098,-10.335,
     &-11.650,-6.767,-1.111,5.231,11.321,20.385,
     &27.501,30.412,30.291,29.223,26.860,22.820,
     &18.862,15.733,10.786,7.231,7.475,9.208,
     &9.995,10.448,11.256,13.281,16.261,-999.000,
     &20.156,-999.000,25.012,29.638,30.593,25.371,
     &16.816,-999.000,0.688,-999.000,-11.566,-999.000,
     &-18.387,-999.000,-20.955,-999.000,-24.308,-999.000,
     &-24.635,-24.420,-24.034,-22.439,-20.570
     &/
       data (fjrocvz(it, 1,1, 9), it=1,59) /
     &-13.622,-12.840,-12.408,-12.453,-12.655,-14.291,
     &-16.228,-18.301,-21.032,-22.535,-21.403,-15.120,
     &-5.441,2.454,8.480,13.009,17.624,23.205,
     &27.729,30.727,31.372,29.502,27.138,24.132,
     &20.964,17.484,13.782,11.034,8.303,5.419,
     &3.649,3.317,3.826,3.611,3.537,-999.000,
     &2.941,-999.000,3.757,-999.000,6.451,-999.000,
     &4.981,-999.000,1.252,-999.000,-5.825,-999.000,
     &-10.952,-999.000,-14.329,-999.000,-16.521,-999.000,
     &-18.649,-19.052,-17.814,-15.361,-13.622
     &/
       data (fjrocvz(it, 2,1, 9), it=1,59) /
     &-16.335,-15.013,-14.554,-14.389,-15.191,-17.179,
     &-18.069,-19.372,-21.126,-21.852,-18.099,-8.435,
     &0.691,7.311,12.811,18.667,23.632,28.428,
     &32.729,35.621,36.577,34.574,31.323,27.611,
     &23.419,19.204,14.788,11.342,8.416,5.748,
     &3.120,1.687,1.501,1.755,1.961,-999.000,
     &2.493,-999.000,4.413,-999.000,5.122,-999.000,
     &3.632,-999.000,-0.912,-999.000,-8.280,-999.000,
     &-13.951,-999.000,-17.998,-999.000,-20.251,-999.000,
     &-21.511,-21.218,-20.033,-18.219,-16.335
     &/
       data (fjrocvz(it, 3,1, 9), it=1,59) /
     &-18.619,-17.863,-17.593,-17.839,-18.541,-18.679,
     &-17.929,-18.119,-18.150,-16.753,-10.803,-2.060,
     &5.600,10.973,16.282,21.662,25.405,29.160,
     &33.067,36.245,36.950,34.389,30.667,27.054,
     &23.705,20.561,16.801,12.936,9.578,6.665,
     &3.823,2.527,2.346,3.020,3.359,-999.000,
     &4.532,-999.000,5.545,-999.000,7.253,8.901,
     &9.504,7.262,2.961,-0.784,-6.561,-999.000,
     &-14.424,-999.000,-18.079,-999.000,-20.142,-999.000,
     &-21.221,-21.158,-20.373,-19.519,-18.619
     &/
       data (fjrocvz(it, 4,1, 9), it=1,59) /
     &-18.936,-19.613,-19.521,-18.655,-17.661,-18.015,
     &-18.158,-18.555,-17.686,-14.107,-6.661,1.845,
     &7.571,11.963,17.529,22.299,26.471,30.001,
     &33.643,35.824,35.194,31.714,27.834,24.847,
     &22.076,18.500,14.708,11.928,10.096,8.413,
     &5.923,3.984,3.480,4.272,6.245,-999.000,
     &7.397,-999.000,10.064,-999.000,13.857,14.846,
     &15.490,11.270,5.382,-999.000,-4.267,-999.000,
     &-14.259,-999.000,-17.983,-999.000,-20.308,-999.000,
     &-21.020,-20.311,-18.659,-17.900,-18.936
     &/
       data (fjrocvz(it, 5,1, 9), it=1,59) /
     &-15.816,-16.508,-16.567,-15.581,-15.013,-16.541,
     &-18.639,-20.043,-19.607,-15.796,-7.802,1.492,
     &6.820,11.158,16.879,22.020,26.260,29.391,
     &32.689,34.749,34.260,31.319,28.295,25.551,
     &22.228,18.059,14.232,12.152,11.326,10.823,
     &9.170,6.849,5.874,5.786,7.032,-999.000,
     &10.002,-999.000,14.926,-999.000,17.868,22.451,
     &21.920,16.531,9.888,-999.000,-2.882,-999.000,
     &-14.237,-999.000,-18.537,-999.000,-20.062,-999.000,
     &-20.631,-18.978,-16.352,-15.193,-15.816
     &/
       data (fjrocvz(it, 6,1, 9), it=1,59) /
     &-13.626,-14.378,-15.105,-15.872,-16.337,-17.781,
     &-19.960,-21.531,-21.363,-16.661,-9.592,-1.225,
     &4.109,7.601,11.740,16.064,20.171,23.694,
     &27.859,31.194,31.364,29.592,27.466,25.141,
     &22.094,18.990,16.056,13.772,12.756,12.671,
     &12.220,11.037,9.151,8.004,8.882,-999.000,
     &11.972,-999.000,15.925,-999.000,21.037,-999.000,
     &21.642,-999.000,13.720,-999.000,2.357,-999.000,
     &-7.773,-999.000,-13.426,-999.000,-16.920,-999.000,
     &-18.299,-17.898,-15.574,-13.943,-13.626
     &/
       data (fjrocvz(it, 7,1, 9), it=1,59) /
     &-15.849,-16.795,-17.938,-18.977,-19.828,-20.839,
     &-21.482,-21.761,-21.591,-18.047,-13.625,-9.295,
     &-4.282,1.404,6.925,11.174,15.733,18.985,
     &23.876,27.233,27.827,26.293,23.984,21.773,
     &19.601,17.596,14.846,12.444,11.163,11.556,
     &12.456,12.212,10.912,10.165,11.071,-999.000,
     &13.266,-999.000,17.732,-999.000,23.314,26.029,
     &30.448,28.976,19.952,-999.000,7.583,-999.000,
     &-1.941,-999.000,-8.876,-999.000,-13.832,-999.000,
     &-16.773,-17.586,-16.428,-16.159,-15.849
     &/
       data (fjrocvz(it, 8,1, 9), it=1,59) /
     &-19.009,-18.767,-18.700,-18.749,-19.150,-19.957,
     &-20.838,-21.902,-23.137,-21.493,-18.215,-12.469,
     &-6.181,-2.055,1.285,4.484,8.569,12.607,
     &17.984,22.070,23.726,22.778,20.547,18.256,
     &15.885,13.941,11.737,9.515,8.583,9.210,
     &10.836,11.700,11.787,11.534,12.558,-999.000,
     &15.733,-999.000,21.309,-999.000,29.125,37.046,
     &41.044,39.932,30.614,19.021,12.042,-999.000,
     &-2.811,-999.000,-10.980,-999.000,-17.665,-999.000,
     &-19.629,-20.877,-20.346,-19.804,-19.009
     &/
       data (fjrocvz(it, 9,1, 9), it=1,59) /
     &-19.894,-18.976,-18.252,-17.789,-17.684,-18.872,
     &-20.505,-22.215,-23.760,-22.908,-20.391,-15.951,
     &-11.017,-6.305,-1.404,2.206,5.236,8.541,
     &13.466,17.822,19.926,20.094,19.079,17.549,
     &15.390,14.023,12.714,11.452,10.523,10.567,
     &11.285,11.648,11.849,12.497,14.611,15.500,
     &16.706,19.049,24.457,30.860,37.794,46.334,
     &52.335,46.766,28.610,12.311,1.585,-6.395,
     &-9.795,-999.000,-16.493,-999.000,-20.164,-999.000,
     &-22.763,-23.677,-22.741,-21.379,-19.894
     &/
       data (fjrocvz(it,10,1, 9), it=1,59) /
     &-18.689,-17.505,-16.654,-16.084,-16.568,-18.469,
     &-19.996,-21.164,-22.991,-24.030,-22.645,-17.658,
     &-10.887,-4.806,0.399,4.450,7.429,10.077,
     &13.658,16.935,19.465,21.314,21.894,21.226,
     &19.295,17.934,16.756,15.708,14.508,14.527,
     &14.476,14.424,14.655,15.131,16.165,-999.000,
     &19.156,23.459,31.507,40.906,51.263,56.338,
     &49.458,33.855,16.050,0.041,-6.987,-11.881,
     &-15.171,-999.000,-19.287,-999.000,-22.332,-999.000,
     &-23.932,-23.650,-22.172,-20.569,-18.689
     &/
       data (fjrocvz(it,11,1, 9), it=1,59) /
     &-17.246,-16.000,-15.615,-16.094,-17.531,-19.929,
     &-21.670,-22.910,-24.554,-24.959,-22.644,-15.443,
     &-7.495,-0.952,4.640,8.731,11.388,13.031,
     &15.667,19.674,23.842,26.517,27.160,25.972,
     &24.148,22.660,21.247,19.267,17.488,16.982,
     &15.936,15.419,15.760,16.692,18.367,19.954,
     &22.425,28.126,36.417,43.253,47.388,45.752,
     &31.790,14.967,2.404,-7.800,-14.734,-999.000,
     &-17.806,-999.000,-21.298,-999.000,-22.875,-999.000,
     &-23.251,-22.443,-20.808,-18.917,-17.246
     &/
       data (fjrocvz(it,12,1, 9), it=1,59) /
     &-18.916,-18.488,-18.804,-20.158,-21.432,-22.838,
     &-23.673,-24.959,-27.119,-27.401,-25.595,-20.564,
     &-12.788,-4.535,4.955,9.922,13.457,15.691,
     &18.490,22.462,25.946,28.155,28.849,28.481,
     &27.564,26.207,23.952,21.205,18.601,17.081,
     &16.067,15.336,15.731,17.643,21.142,-999.000,
     &26.291,27.820,34.279,40.307,43.350,37.001,
     &24.420,13.318,3.715,-999.000,-12.185,-999.000,
     &-20.680,-999.000,-23.924,-999.000,-24.036,-999.000,
     &-23.172,-22.264,-21.261,-19.923,-18.916
     &/
       data (fjrocvz(it,13,1, 9), it=1,59) /
     &-20.072,-20.646,-21.377,-22.291,-22.577,-22.958,
     &-23.383,-25.217,-27.481,-28.620,-27.663,-23.133,
     &-14.535,-6.632,1.602,8.272,13.417,16.099,
     &19.109,22.651,25.615,27.275,27.508,26.916,
     &25.570,24.201,22.548,20.086,16.877,15.117,
     &14.881,14.995,15.426,16.005,16.780,-999.000,
     &20.884,-999.000,28.072,33.477,35.784,30.387,
     &20.728,-999.000,7.232,-999.000,-7.782,-999.000,
     &-18.278,-999.000,-23.704,-999.000,-24.273,-999.000,
     &-23.656,-22.695,-21.547,-20.409,-20.072
     &/
       data (fjrocvz(it,14,1, 9), it=1,59) /
     &-20.589,-22.109,-23.146,-23.377,-22.751,-23.148,
     &-24.937,-27.263,-28.941,-29.454,-26.524,-23.225,
     &-17.243,-10.080,-1.315,6.497,12.863,15.471,
     &18.068,21.422,24.911,26.584,26.458,25.101,
     &23.636,22.474,20.817,17.906,14.832,13.077,
     &12.893,13.089,12.871,11.189,10.884,-999.000,
     &13.530,15.835,21.488,26.390,28.427,25.384,
     &19.728,11.519,5.956,-999.000,-6.023,-999.000,
     &-16.318,-999.000,-23.377,-999.000,-25.129,-999.000,
     &-25.020,-23.746,-21.881,-20.692,-20.589
     &/
       data (fjrocvz(it,15,1, 9), it=1,59) /
     &-21.143,-21.984,-23.405,-24.374,-23.967,-23.955,
     &-25.024,-27.025,-30.041,-31.650,-28.904,-23.685,
     &-15.719,-8.843,-1.675,5.415,11.907,15.063,
     &17.591,19.474,21.405,22.404,21.847,20.439,
     &19.308,18.547,17.404,15.688,13.144,11.090,
     &10.510,10.429,9.916,9.286,9.538,-999.000,
     &11.927,-999.000,18.285,24.738,29.744,27.134,
     &21.419,13.451,7.675,-999.000,-4.297,-999.000,
     &-14.864,-999.000,-23.260,-999.000,-25.669,-999.000,
     &-25.188,-24.101,-22.707,-21.738,-21.143
     &/
       data (fjrocvz(it,16,1, 9), it=1,59) /
     &-22.474,-22.784,-23.596,-24.523,-24.340,-24.024,
     &-24.391,-25.620,-28.859,-33.255,-33.578,-27.856,
     &-12.432,-4.006,0.991,6.546,12.221,15.095,
     &17.582,19.526,21.481,21.755,21.153,19.929,
     &18.716,17.529,16.134,13.985,11.127,9.338,
     &9.287,9.866,10.270,10.374,11.068,-999.000,
     &14.109,-999.000,20.373,26.831,31.686,27.240,
     &19.695,-999.000,6.559,-999.000,-4.237,-999.000,
     &-14.633,-999.000,-22.564,-999.000,-24.972,-999.000,
     &-25.171,-24.385,-23.540,-23.074,-22.474
     &/
       data (fjrocvz(it,17,1, 9), it=1,59) /
     &-21.987,-21.938,-22.176,-23.059,-23.723,-24.733,
     &-25.410,-24.720,-25.934,-29.546,-31.756,-26.994,
     &-11.329,0.326,4.413,8.023,12.105,14.732,
     &17.197,19.336,21.076,22.293,22.423,21.481,
     &20.510,19.234,17.629,15.441,12.626,10.636,
     &10.377,11.096,11.311,10.875,11.931,-999.000,
     &14.355,17.390,20.241,24.886,28.172,25.574,
     &17.553,-999.000,1.849,-999.000,-9.420,-999.000,
     &-18.549,-999.000,-24.542,-999.000,-26.642,-999.000,
     &-25.973,-24.055,-22.742,-22.371,-21.987
     &/
       data (fjrocvz(it,18,1, 9), it=1,59) /
     &-20.879,-20.663,-20.896,-22.073,-23.726,-25.671,
     &-26.799,-27.095,-27.515,-27.702,-26.655,-21.932,
     &-9.752,1.808,6.578,9.880,13.877,16.861,
     &19.361,20.455,21.696,23.172,23.302,22.194,
     &20.769,19.418,17.631,15.927,13.969,12.249,
     &11.456,11.637,12.008,12.125,12.666,-999.000,
     &15.858,-999.000,20.216,25.842,28.156,23.151,
     &13.285,4.080,-3.398,-999.000,-12.987,-999.000,
     &-20.496,-999.000,-26.182,-999.000,-27.835,-999.000,
     &-26.417,-23.939,-22.551,-21.740,-20.879
     &/
       data (fjrocvz(it,19,1, 9), it=1,59) /
     &-21.924,-20.845,-19.963,-20.172,-20.664,-21.795,
     &-21.904,-23.017,-24.820,-26.770,-26.514,-22.705,
     &-12.782,-0.565,6.505,11.354,15.867,19.658,
     &22.589,24.041,25.536,26.682,27.088,25.776,
     &24.368,23.278,20.857,17.531,15.185,14.102,
     &13.891,13.665,13.411,12.352,12.036,-999.000,
     &14.680,-999.000,18.515,21.942,21.375,14.091,
     &7.808,-999.000,-4.400,-999.000,-13.380,-999.000,
     &-21.058,-999.000,-26.029,-999.000,-27.532,-999.000,
     &-26.346,-24.325,-23.177,-22.718,-21.924
     &/
       data (fjrocvz(it,20,1, 9), it=1,59) /
     &-20.909,-19.598,-19.098,-19.012,-19.030,-19.647,
     &-19.517,-19.931,-21.580,-23.984,-26.609,-24.650,
     &-16.415,-3.229,4.121,8.434,13.556,18.447,
     &22.136,24.248,25.887,27.112,27.557,26.918,
     &26.776,26.664,24.865,21.458,18.002,16.342,
     &15.940,15.374,14.089,12.468,11.955,-999.000,
     &12.117,-999.000,13.375,15.421,18.408,13.670,
     &4.760,-999.000,-6.101,-999.000,-15.668,-999.000,
     &-22.941,-999.000,-27.645,-999.000,-29.179,-999.000,
     &-28.666,-26.747,-24.716,-22.743,-20.909
     &/
       data (fjrocvz(it,21,1, 9), it=1,59) /
     &-22.177,-21.113,-21.025,-21.400,-21.413,-21.061,
     &-19.715,-20.160,-22.785,-26.978,-30.358,-28.896,
     &-20.081,-6.944,0.519,4.899,9.767,14.113,
     &18.008,21.035,23.352,24.251,24.277,24.355,
     &25.073,25.228,23.562,20.088,16.337,14.391,
     &14.023,13.710,12.720,10.854,9.734,-999.000,
     &10.295,-999.000,11.557,13.365,14.336,9.683,
     &1.122,-999.000,-9.333,-999.000,-17.781,-999.000,
     &-23.901,-999.000,-28.479,-999.000,-29.931,-999.000,
     &-29.676,-28.297,-26.433,-24.200,-22.177
     &/
       data (fjrocvz(it,22,1, 9), it=1,59) /
     &-20.565,-19.770,-19.959,-20.948,-21.307,-21.144,
     &-20.107,-20.753,-23.203,-26.935,-29.516,-28.010,
     &-19.562,-7.496,-0.069,4.287,8.003,12.486,
     &16.603,20.373,22.240,22.797,23.385,23.706,
     &23.697,22.789,19.958,15.383,10.964,8.769,
     &8.594,8.744,8.016,6.528,6.102,-999.000,
     &6.484,-999.000,5.848,-999.000,5.641,-999.000,
     &0.612,-999.000,-10.990,-999.000,-17.290,-999.000,
     &-21.925,-999.000,-26.028,-999.000,-27.693,-999.000,
     &-27.393,-25.779,-23.629,-21.921,-20.565
     &/
       data (fjrocvz(it,23,1, 9), it=1,59) /
     &-15.967,-15.895,-16.249,-17.106,-17.998,-18.546,
     &-18.722,-19.652,-21.055,-22.550,-22.958,-19.644,
     &-12.195,-3.521,2.840,6.371,9.586,13.016,
     &16.929,20.676,23.371,25.518,26.620,26.074,
     &24.346,22.051,19.314,15.629,11.190,8.018,
     &6.091,5.073,4.246,3.408,3.248,-999.000,
     &3.733,-999.000,6.747,-999.000,7.236,-999.000,
     &4.123,-999.000,-1.177,-999.000,-11.214,-999.000,
     &-16.580,-999.000,-20.972,-999.000,-22.692,-999.000,
     &-22.516,-21.069,-18.911,-16.858,-15.967
     &/
       data (fjrocvz(it,24,1, 9), it=1,59) /
     &-13.466,-13.257,-13.295,-13.978,-14.378,-14.926,
     &-15.598,-17.340,-19.503,-21.251,-21.099,-16.655,
     &-8.461,-0.690,4.718,8.856,12.392,16.092,
     &19.902,23.524,26.083,26.612,26.017,24.476,
     &22.172,19.879,17.423,14.810,11.143,7.767,
     &5.385,4.555,4.572,4.105,3.510,-999.000,
     &4.200,-999.000,5.819,-999.000,7.301,-999.000,
     &5.926,-999.000,1.331,-999.000,-5.503,-999.000,
     &-11.390,-999.000,-15.125,-999.000,-16.928,-999.000,
     &-17.842,-17.836,-16.528,-14.675,-13.466
     &/
       data (fjrocvz(it,25,1, 9), it=1,59) /
     &-13.622,-12.840,-12.408,-12.453,-12.655,-14.291,
     &-16.228,-18.301,-21.032,-22.535,-21.403,-15.120,
     &-5.441,2.454,8.480,13.009,17.624,23.205,
     &27.729,30.727,31.372,29.502,27.138,24.132,
     &20.964,17.484,13.782,11.034,8.303,5.419,
     &3.649,3.317,3.826,3.611,3.537,-999.000,
     &2.941,-999.000,3.757,-999.000,6.451,-999.000,
     &4.981,-999.000,1.252,-999.000,-5.825,-999.000,
     &-10.952,-999.000,-14.329,-999.000,-16.521,-999.000,
     &-18.649,-19.052,-17.814,-15.361,-13.622
     &/
       data (fjrocvz(it, 1,2, 9), it=1,59) /
     &-21.709,-20.817,-19.285,-18.202,-18.971,-19.948,
     &-20.633,-21.597,-22.342,-21.677,-18.388,-11.622,
     &-12.931,-7.805,-2.029,4.434,10.765,19.933,
     &27.100,30.063,30.045,29.092,26.786,22.804,
     &18.961,16.119,11.315,7.903,8.434,10.289,
     &11.138,11.652,12.582,14.757,17.886,-999.000,
     &21.867,-999.000,27.103,32.389,33.608,28.283,
     &19.265,-999.000,1.118,-999.000,-12.633,-999.000,
     &-19.926,-999.000,-22.695,-999.000,-25.823,-999.000,
     &-25.925,-25.704,-25.312,-23.647,-21.709
     &/
       data (fjrocvz(it, 2,2, 9), it=1,59) /
     &-20.264,-19.525,-18.609,-18.153,-17.438,-17.107,
     &-17.452,-19.208,-20.752,-20.203,-14.906,-7.062,
     &-1.601,2.649,7.504,10.348,15.419,21.266,
     &26.955,30.943,32.310,31.759,29.163,25.074,
     &21.272,19.108,15.837,12.671,11.698,12.206,
     &13.572,14.620,15.899,18.129,20.380,-999.000,
     &22.852,-999.000,28.355,31.867,34.403,33.686,
     &27.605,-999.000,2.168,-999.000,-16.673,-999.000,
     &-20.929,-999.000,-26.403,-999.000,-28.787,-999.000,
     &-26.972,-26.126,-23.931,-21.492,-20.264
     &/
       data (fjrocvz(it, 3,2, 9), it=1,59) /
     &-20.412,-19.200,-18.532,-17.395,-16.382,-16.891,
     &-16.675,-16.978,-17.941,-17.343,-12.878,-6.434,
     &-0.543,4.107,6.989,10.537,15.446,19.492,
     &23.279,26.343,28.446,28.248,25.596,22.118,
     &19.364,17.893,16.000,14.133,13.186,13.438,
     &14.683,15.884,17.707,19.007,21.215,-999.000,
     &24.724,-999.000,28.024,33.629,37.737,36.761,
     &27.941,13.566,1.517,-12.752,-21.628,-22.523,
     &-24.220,-999.000,-29.239,-999.000,-29.894,-999.000,
     &-28.645,-26.392,-23.748,-21.953,-20.412
     &/
       data (fjrocvz(it, 4,2, 9), it=1,59) /
     &-21.554,-20.356,-18.741,-17.132,-16.705,-17.115,
     &-16.573,-15.341,-14.984,-14.197,-11.149,-5.670,
     &0.335,3.758,5.415,8.479,13.005,18.136,
     &22.071,24.536,25.109,23.939,21.463,18.524,
     &16.367,14.983,13.395,12.538,13.062,13.562,
     &14.605,16.278,17.870,20.169,23.826,-999.000,
     &26.167,-999.000,29.967,35.816,37.779,37.527,
     &33.006,19.790,1.564,-16.799,-24.728,-999.000,
     &-26.583,-999.000,-30.203,-999.000,-31.766,-999.000,
     &-29.481,-26.421,-24.157,-22.511,-21.554
     &/
       data (fjrocvz(it, 5,2, 9), it=1,59) /
     &-21.653,-20.280,-17.971,-16.281,-16.028,-17.137,
     &-17.185,-16.203,-16.035,-15.220,-12.246,-6.065,
     &-0.149,3.734,6.842,9.416,13.961,19.804,
     &24.468,27.318,28.075,26.885,23.811,20.493,
     &18.006,16.579,15.303,14.843,14.871,14.714,
     &15.530,17.059,19.092,21.380,23.272,-999.000,
     &27.115,-999.000,31.105,35.507,38.882,41.434,
     &37.752,21.232,0.523,-13.434,-24.333,-28.125,
     &-31.487,-34.322,-35.280,-34.355,-32.132,-999.000,
     &-28.961,-26.081,-23.764,-22.380,-21.653
     &/
       data (fjrocvz(it, 6,2, 9), it=1,59) /
     &-22.498,-21.631,-20.162,-18.773,-18.459,-18.749,
     &-17.760,-17.183,-17.145,-15.352,-13.170,-11.768,
     &-8.479,-1.429,5.570,11.098,15.967,21.547,
     &26.132,28.953,30.727,30.188,27.959,24.808,
     &22.475,20.960,19.285,17.630,16.759,15.783,
     &15.394,17.194,19.484,20.723,22.956,-999.000,
     &27.008,-999.000,31.047,38.279,44.457,46.325,
     &39.254,23.605,4.390,-10.805,-22.580,-29.285,
     &-32.668,-999.000,-34.757,-999.000,-33.252,-999.000,
     &-29.998,-26.391,-24.346,-23.482,-22.498
     &/
       data (fjrocvz(it, 7,2, 9), it=1,59) /
     &-24.564,-24.282,-23.298,-22.420,-21.947,-20.848,
     &-19.364,-18.494,-17.602,-16.486,-14.302,-11.887,
     &-6.622,-2.110,2.572,6.756,12.222,18.452,
     &23.863,28.079,30.096,30.376,29.233,27.031,
     &24.964,22.798,20.570,18.755,16.804,14.598,
     &13.905,14.787,16.259,18.446,20.396,-999.000,
     &23.271,-999.000,28.114,36.535,43.968,47.977,
     &37.191,18.154,0.734,-11.342,-22.645,-31.249,
     &-36.416,-37.287,-36.649,-35.634,-34.142,-999.000,
     &-30.635,-28.202,-26.362,-24.970,-24.564
     &/
       data (fjrocvz(it, 8,2, 9), it=1,59) /
     &-24.581,-23.471,-22.695,-22.370,-21.946,-21.837,
     &-21.395,-20.909,-19.974,-21.679,-23.444,-25.081,
     &-19.686,-10.740,-4.320,2.011,6.945,11.444,
     &17.673,22.569,25.733,26.293,25.737,24.570,
     &22.955,21.197,18.964,16.812,14.212,11.910,
     &10.900,11.208,12.648,14.559,16.186,-999.000,
     &19.902,-999.000,26.198,32.276,39.345,40.344,
     &28.081,13.803,0.608,-11.578,-20.560,-27.951,
     &-32.629,-32.974,-30.716,-999.000,-30.320,-999.000,
     &-28.881,-28.412,-27.152,-25.742,-24.581
     &/
       data (fjrocvz(it, 9,2, 9), it=1,59) /
     &-21.942,-20.006,-18.254,-17.203,-17.303,-18.350,
     &-20.120,-21.674,-22.896,-24.519,-27.051,-28.916,
     &-23.472,-12.561,-4.118,1.609,4.800,8.996,
     &14.194,18.182,20.620,21.357,21.201,19.864,
     &17.629,15.582,13.711,11.754,9.649,8.308,
     &8.198,8.792,9.942,11.569,13.541,-999.000,
     &16.656,-999.000,23.920,30.946,37.350,40.965,
     &36.658,23.100,9.272,-3.144,-12.107,-999.000,
     &-18.521,-999.000,-21.874,-999.000,-24.334,-999.000,
     &-24.558,-24.529,-24.614,-23.851,-21.942
     &/
       data (fjrocvz(it,10,2, 9), it=1,59) /
     &-22.909,-20.532,-18.179,-16.906,-16.140,-17.314,
     &-20.434,-23.110,-24.597,-25.304,-25.903,-25.606,
     &-21.304,-9.957,-1.634,3.938,6.834,11.316,
     &16.655,20.169,21.200,21.025,19.886,18.345,
     &16.425,13.905,11.606,9.707,8.506,7.938,
     &8.002,8.349,8.389,9.292,11.385,-999.000,
     &15.707,-999.000,24.021,32.076,40.943,45.036,
     &42.953,29.515,16.147,-999.000,-0.183,-999.000,
     &-10.493,-999.000,-17.751,-999.000,-19.764,-999.000,
     &-21.563,-22.414,-23.306,-23.468,-22.909
     &/
       data (fjrocvz(it,11,2, 9), it=1,59) /
     &-25.803,-24.771,-24.208,-24.011,-23.623,-23.820,
     &-24.461,-25.821,-27.342,-29.222,-28.488,-24.650,
     &-19.049,-9.156,-1.493,4.645,8.948,12.898,
     &17.561,21.524,23.596,23.967,23.133,21.781,
     &20.536,18.357,15.521,13.518,12.165,11.205,
     &10.560,9.816,9.286,10.152,12.815,-999.000,
     &17.792,-999.000,24.806,32.401,41.669,43.213,
     &39.437,28.634,17.725,-999.000,-0.700,-999.000,
     &-12.328,-999.000,-20.017,-999.000,-21.768,-999.000,
     &-24.699,-24.487,-25.316,-26.434,-25.803
     &/
       data (fjrocvz(it,12,2, 9), it=1,59) /
     &-27.306,-27.695,-27.434,-27.179,-27.088,-26.434,
     &-26.232,-27.754,-30.331,-32.502,-33.254,-29.982,
     &-25.767,-16.246,-6.573,-0.739,3.780,9.791,
     &16.933,22.348,25.401,26.022,25.330,24.506,
     &23.696,21.577,18.682,16.403,15.467,14.744,
     &13.821,12.570,11.677,12.593,15.724,-999.000,
     &20.199,-999.000,26.762,33.372,41.323,46.613,
     &42.342,25.278,11.040,0.652,-8.646,-999.000,
     &-14.650,-999.000,-20.123,-999.000,-24.244,-999.000,
     &-26.878,-26.820,-27.051,-27.333,-27.306
     &/
       data (fjrocvz(it,13,2, 9), it=1,59) /
     &-25.557,-24.861,-24.530,-24.413,-23.898,-24.381,
     &-25.843,-27.079,-28.335,-29.133,-29.613,-28.187,
     &-25.429,-17.177,-8.564,-2.111,1.937,8.784,
     &18.550,24.852,27.092,27.576,26.984,26.340,
     &25.469,23.166,20.465,18.721,18.669,18.591,
     &17.947,16.731,15.494,16.054,18.650,-999.000,
     &22.256,-999.000,30.957,37.116,45.498,51.195,
     &44.110,26.488,7.325,-4.593,-12.087,-14.185,
     &-15.309,-999.000,-19.066,-999.000,-21.949,-999.000,
     &-24.215,-26.177,-26.040,-25.532,-25.557
     &/
       data (fjrocvz(it,14,2, 9), it=1,59) /
     &-23.285,-22.738,-22.344,-21.710,-21.580,-23.171,
     &-25.928,-29.477,-32.618,-32.539,-29.714,-25.784,
     &-22.057,-18.626,-11.730,-3.962,1.608,8.955,
     &18.721,24.807,27.541,28.046,27.257,26.236,
     &24.785,23.107,21.152,20.012,20.296,20.393,
     &19.571,18.485,17.528,18.021,20.692,-999.000,
     &25.649,-999.000,34.392,43.910,50.098,49.555,
     &41.133,25.811,9.890,-999.000,-10.761,-999.000,
     &-18.556,-999.000,-20.810,-999.000,-23.232,-999.000,
     &-25.721,-26.472,-25.490,-24.054,-23.285
     &/
       data (fjrocvz(it,15,2, 9), it=1,59) /
     &-20.907,-20.412,-19.862,-19.411,-19.442,-21.192,
     &-25.186,-31.284,-35.563,-36.926,-33.550,-26.390,
     &-21.602,-16.380,-10.846,-3.222,3.553,10.077,
     &17.808,23.230,25.809,25.886,24.359,22.076,
     &20.048,18.695,16.928,15.829,15.839,15.811,
     &15.311,14.703,14.955,16.399,19.863,-999.000,
     &24.424,-999.000,32.022,40.674,47.159,49.250,
     &41.799,24.958,9.206,-999.000,-12.607,-999.000,
     &-19.936,-999.000,-23.747,-999.000,-26.665,-999.000,
     &-28.141,-28.296,-25.781,-22.463,-20.907
     &/
       data (fjrocvz(it,16,2, 9), it=1,59) /
     &-20.596,-19.532,-19.765,-20.050,-20.903,-23.249,
     &-27.095,-31.906,-35.752,-38.587,-37.485,-29.958,
     &-20.452,-14.593,-8.478,-1.682,5.010,10.591,
     &17.982,23.053,25.006,25.305,24.053,21.018,
     &17.479,14.492,12.131,10.812,10.837,11.286,
     &10.900,10.210,10.449,11.714,14.536,-999.000,
     &19.216,-999.000,26.543,34.709,41.910,43.403,
     &34.863,17.164,4.348,-999.000,-13.448,-999.000,
     &-21.706,-999.000,-26.469,-999.000,-29.322,-999.000,
     &-30.575,-29.995,-27.366,-23.614,-20.596
     &/
       data (fjrocvz(it,17,2, 9), it=1,59) /
     &-24.003,-23.515,-22.956,-22.648,-23.577,-25.611,
     &-28.909,-33.654,-37.783,-39.227,-37.438,-29.274,
     &-20.220,-13.457,-6.403,-0.491,5.580,11.478,
     &19.374,24.788,27.356,28.285,27.329,24.668,
     &20.631,16.769,13.171,10.652,10.309,10.308,
     &9.543,8.423,7.508,7.620,10.321,-999.000,
     &15.969,-999.000,24.217,30.499,35.603,37.533,
     &30.708,14.740,0.905,-12.226,-21.063,-25.088,
     &-27.175,-999.000,-28.652,-999.000,-30.785,-999.000,
     &-31.508,-30.270,-28.580,-26.135,-24.003
     &/
       data (fjrocvz(it,18,2, 9), it=1,59) /
     &-25.782,-24.797,-23.778,-23.618,-23.807,-25.039,
     &-28.876,-33.692,-36.945,-38.261,-35.637,-28.269,
     &-19.117,-12.307,-6.813,-0.748,5.881,13.589,
     &21.018,27.011,31.244,32.307,30.279,26.562,
     &23.143,19.889,15.909,12.413,11.061,10.039,
     &8.967,7.568,6.667,6.644,9.004,-999.000,
     &14.255,18.187,23.508,29.819,34.303,31.941,
     &24.823,13.142,-0.507,-999.000,-20.727,-999.000,
     &-28.238,-999.000,-31.491,-999.000,-32.540,-999.000,
     &-32.127,-31.711,-30.233,-27.855,-25.782
     &/
       data (fjrocvz(it,19,2, 9), it=1,59) /
     &-24.516,-23.304,-23.335,-22.957,-22.703,-23.895,
     &-26.816,-29.735,-32.501,-33.932,-31.227,-24.353,
     &-16.167,-11.152,-5.735,0.366,6.692,15.128,
     &23.671,30.519,35.254,36.087,33.707,29.949,
     &26.097,22.357,18.356,14.816,13.906,13.321,
     &11.921,10.208,9.142,8.789,10.736,-999.000,
     &16.831,-999.000,25.648,32.496,35.505,32.640,
     &26.969,16.049,-1.419,-14.165,-23.718,-27.835,
     &-29.331,-999.000,-31.322,-999.000,-31.526,-999.000,
     &-31.297,-31.029,-29.869,-27.266,-24.516
     &/
       data (fjrocvz(it,20,2, 9), it=1,59) /
     &-25.925,-24.727,-23.176,-21.569,-20.619,-21.244,
     &-23.569,-26.861,-29.011,-29.646,-27.091,-18.041,
     &-9.975,-7.488,-2.721,1.951,7.131,15.697,
     &23.935,31.620,36.569,37.765,36.048,32.810,
     &29.410,26.241,22.148,18.741,17.656,16.548,
     &14.904,13.140,11.870,11.558,14.022,-999.000,
     &19.783,-999.000,27.070,33.633,39.368,41.377,
     &32.455,16.559,-1.550,-11.240,-18.993,-24.972,
     &-28.789,-999.000,-30.424,-999.000,-31.565,-999.000,
     &-31.425,-31.104,-30.294,-28.124,-25.925
     &/
       data (fjrocvz(it,21,2, 9), it=1,59) /
     &-25.674,-23.418,-21.594,-20.447,-19.818,-19.936,
     &-21.252,-24.141,-27.335,-28.053,-25.266,-18.012,
     &-12.612,-7.408,-2.168,3.228,9.080,15.760,
     &21.866,28.768,32.998,34.800,33.246,29.928,
     &26.835,24.447,21.067,17.291,15.614,14.315,
     &12.806,11.816,11.686,11.950,13.982,-999.000,
     &18.868,-999.000,27.233,33.307,39.393,38.031,
     &28.899,-999.000,2.520,-999.000,-15.037,-999.000,
     &-24.879,-999.000,-29.189,-999.000,-30.436,-999.000,
     &-30.671,-30.136,-29.348,-27.779,-25.674
     &/
       data (fjrocvz(it,22,2, 9), it=1,59) /
     &-24.518,-23.388,-22.216,-21.355,-21.221,-21.264,
     &-21.628,-23.975,-28.018,-28.825,-25.796,-18.895,
     &-10.744,-7.445,-1.819,3.204,8.369,14.412,
     &20.948,26.964,29.979,30.653,28.801,25.076,
     &20.849,17.538,14.320,11.553,10.341,9.489,
     &8.874,9.448,10.706,11.518,13.794,-999.000,
     &19.139,-999.000,28.080,31.107,34.821,30.819,
     &22.610,-999.000,-0.301,-999.000,-16.203,-999.000,
     &-26.004,-999.000,-28.169,-999.000,-28.579,-999.000,
     &-29.265,-28.748,-27.865,-26.206,-24.518
     &/
       data (fjrocvz(it,23,2, 9), it=1,59) /
     &-23.085,-22.557,-21.016,-19.019,-18.361,-19.271,
     &-21.121,-23.456,-25.609,-26.059,-23.355,-17.075,
     &-11.676,-8.699,-3.237,2.061,7.401,13.079,
     &21.239,27.040,29.023,28.439,26.000,22.127,
     &17.911,14.039,10.710,8.430,7.900,7.680,
     &7.498,7.651,9.429,11.700,14.869,-999.000,
     &18.864,-999.000,26.976,31.554,34.412,31.398,
     &23.847,10.629,-1.032,-999.000,-15.788,-999.000,
     &-23.019,-999.000,-26.077,-999.000,-27.050,-999.000,
     &-27.933,-27.573,-26.341,-24.100,-23.085
     &/
       data (fjrocvz(it,24,2, 9), it=1,59) /
     &-21.843,-20.291,-18.511,-16.387,-16.145,-17.918,
     &-20.057,-21.340,-22.407,-22.927,-20.758,-15.840,
     &-9.639,-4.516,-0.968,2.398,7.219,15.617,
     &23.878,27.577,28.474,27.731,24.824,20.625,
     &16.615,13.050,9.304,6.866,7.349,8.735,
     &9.215,9.066,10.614,13.019,15.975,-999.000,
     &19.535,-999.000,26.670,33.253,35.897,32.818,
     &22.020,-999.000,1.422,-999.000,-11.446,-999.000,
     &-19.047,-999.000,-24.322,-999.000,-25.163,-999.000,
     &-26.272,-26.297,-25.289,-23.609,-21.843
     &/
       data (fjrocvz(it,25,2, 9), it=1,59) /
     &-21.709,-20.817,-19.285,-18.202,-18.971,-19.948,
     &-20.633,-21.597,-22.342,-21.677,-18.388,-11.622,
     &-12.931,-7.805,-2.029,4.434,10.765,19.933,
     &27.100,30.063,30.045,29.092,26.786,22.804,
     &18.961,16.119,11.315,7.903,8.434,10.289,
     &11.138,11.652,12.582,14.757,17.886,-999.000,
     &21.867,-999.000,27.103,32.389,33.608,28.283,
     &19.265,-999.000,1.118,-999.000,-12.633,-999.000,
     &-19.926,-999.000,-22.695,-999.000,-25.823,-999.000,
     &-25.925,-25.704,-25.312,-23.647,-21.709
     &/
       data (fjrocvz(it, 1,3, 9), it=1,59) /
     &-20.279,-20.553,-19.854,-18.454,-16.118,-15.436,
     &-15.621,-19.120,-26.453,-28.322,-21.436,-7.773,
     &3.592,7.399,9.946,11.849,11.437,10.852,
     &11.390,14.183,17.907,18.928,18.407,16.039,
     &12.797,10.924,11.336,12.204,11.897,10.898,
     &10.135,9.999,9.696,9.396,10.052,-999.000,
     &14.303,-999.000,17.510,-999.000,23.213,27.359,
     &27.875,22.361,11.308,-999.000,-3.449,-999.000,
     &-10.236,-999.000,-12.545,-999.000,-13.950,-999.000,
     &-15.784,-18.277,-19.928,-19.851,-20.279
     &/
       data (fjrocvz(it, 2,3, 9), it=1,59) /
     &-18.899,-18.965,-18.344,-16.748,-15.238,-16.039,
     &-17.817,-21.623,-27.148,-28.208,-20.069,-4.461,
     &8.856,12.328,13.627,14.637,14.801,15.151,
     &16.266,18.127,20.112,20.818,19.990,18.096,
     &16.553,15.313,15.579,16.192,16.389,15.757,
     &15.023,14.522,13.973,14.662,15.135,-999.000,
     &17.191,-999.000,20.700,-999.000,24.862,27.741,
     &26.805,21.527,8.280,-999.000,-4.869,-999.000,
     &-11.586,-999.000,-14.185,-999.000,-15.843,-999.000,
     &-17.410,-19.701,-20.406,-19.271,-18.899
     &/
       data (fjrocvz(it, 3,3, 9), it=1,59) /
     &-18.262,-17.182,-16.581,-15.617,-15.135,-16.394,
     &-18.353,-20.949,-25.602,-29.194,-22.053,-3.008,
     &14.352,16.828,15.620,14.658,14.172,14.892,
     &16.191,17.308,17.639,16.976,15.733,15.183,
     &15.416,14.814,15.494,16.486,17.438,18.282,
     &18.067,17.391,16.815,17.676,18.411,-999.000,
     &20.741,-999.000,22.081,-999.000,24.863,24.657,
     &22.035,18.081,8.469,-999.000,-5.906,-999.000,
     &-13.056,-999.000,-16.684,-999.000,-19.118,-999.000,
     &-21.343,-23.436,-23.656,-20.939,-18.262
     &/
       data (fjrocvz(it, 4,3, 9), it=1,59) /
     &-19.721,-17.575,-16.829,-16.749,-15.977,-16.291,
     &-16.537,-17.877,-22.182,-24.983,-17.720,2.141,
     &16.703,19.371,18.316,17.731,16.996,15.814,
     &14.620,13.934,13.892,13.589,13.573,14.154,
     &14.619,14.690,15.247,16.427,17.814,18.426,
     &18.766,18.783,18.091,19.110,21.645,-999.000,
     &22.626,-999.000,22.962,-999.000,26.864,26.858,
     &23.953,14.164,5.340,-999.000,-6.130,-999.000,
     &-14.083,-999.000,-19.124,-999.000,-22.332,-999.000,
     &-24.141,-25.395,-25.344,-22.692,-19.721
     &/
       data (fjrocvz(it, 5,3, 9), it=1,59) /
     &-20.648,-19.315,-18.510,-16.631,-14.379,-14.428,
     &-14.703,-16.322,-18.940,-19.593,-13.695,0.596,
     &15.010,20.246,20.306,19.886,18.987,17.761,
     &16.397,15.661,16.344,16.862,16.946,16.328,
     &15.636,14.670,14.241,14.791,15.589,16.455,
     &17.506,18.336,18.487,19.992,21.644,-999.000,
     &23.173,-999.000,25.425,29.339,32.440,27.276,
     &20.908,11.668,2.633,-999.000,-8.369,-999.000,
     &-15.398,-999.000,-20.391,-999.000,-23.370,-999.000,
     &-25.575,-26.800,-25.961,-23.072,-20.648
     &/
       data (fjrocvz(it, 6,3, 9), it=1,59) /
     &-22.143,-22.109,-21.881,-19.372,-15.367,-13.859,
     &-14.187,-15.878,-17.102,-16.114,-11.909,-1.710,
     &11.780,16.894,16.941,16.563,16.669,16.981,
     &16.945,18.358,20.389,21.109,20.436,19.327,
     &18.180,16.540,15.343,14.453,14.853,15.441,
     &16.133,16.687,17.447,18.486,19.444,-999.000,
     &23.004,-999.000,25.203,28.332,28.347,20.809,
     &12.129,-999.000,-2.109,-999.000,-11.279,-999.000,
     &-18.193,-999.000,-23.015,-999.000,-25.995,-999.000,
     &-28.349,-29.216,-27.344,-24.382,-22.143
     &/
       data (fjrocvz(it, 7,3, 9), it=1,59) /
     &-24.936,-23.264,-22.805,-21.851,-18.887,-18.252,
     &-18.133,-17.932,-18.352,-15.753,-11.733,-4.678,
     &2.500,7.540,9.215,10.479,11.781,13.199,
     &14.794,17.152,20.123,21.870,22.379,22.337,
     &22.163,21.694,20.207,18.424,17.712,16.928,
     &16.552,16.005,15.671,15.671,15.993,-999.000,
     &17.375,-999.000,18.049,-999.000,15.722,-999.000,
     &5.330,-999.000,-3.660,-999.000,-12.047,-999.000,
     &-19.003,-999.000,-24.085,-999.000,-27.554,-999.000,
     &-30.005,-30.927,-29.896,-27.720,-24.936
     &/
       data (fjrocvz(it, 8,3, 9), it=1,59) /
     &-22.720,-21.539,-22.600,-23.145,-22.020,-19.815,
     &-15.931,-13.707,-13.542,-12.342,-9.442,-5.647,
     &-3.251,-0.437,1.304,3.353,5.814,7.764,
     &9.544,11.676,15.296,19.145,21.241,21.963,
     &22.448,21.971,20.490,18.428,17.002,16.338,
     &15.775,14.915,13.732,12.053,10.578,-999.000,
     &10.675,-999.000,10.346,-999.000,8.631,-999.000,
     &4.228,-999.000,0.463,-999.000,-6.060,-999.000,
     &-13.301,-999.000,-19.235,-999.000,-24.206,-999.000,
     &-27.753,-28.865,-27.512,-25.097,-22.720
     &/
       data (fjrocvz(it, 9,3, 9), it=1,59) /
     &-17.375,-17.693,-18.079,-18.661,-18.634,-16.917,
     &-13.599,-12.003,-12.662,-13.066,-12.212,-9.796,
     &-9.173,-4.362,0.609,5.339,10.409,11.568,
     &12.527,13.479,14.678,14.670,16.140,17.443,
     &18.625,18.898,17.287,14.841,13.589,12.459,
     &11.210,9.911,8.833,8.017,7.415,-999.000,
     &6.391,-999.000,5.314,-999.000,6.615,-999.000,
     &7.977,-999.000,8.016,-999.000,3.158,-999.000,
     &-2.271,-999.000,-8.001,-999.000,-12.024,-999.000,
     &-15.692,-17.341,-17.492,-17.408,-17.375
     &/
       data (fjrocvz(it,10,3, 9), it=1,59) /
     &-14.992,-15.805,-15.164,-14.294,-15.029,-16.572,
     &-16.436,-15.539,-15.607,-15.334,-14.953,-15.858,
     &-17.428,-8.814,0.197,6.484,13.236,14.271,
     &15.220,15.930,16.808,17.439,17.526,17.627,
     &17.959,18.335,16.899,14.932,13.160,11.124,
     &8.995,6.565,5.010,3.980,2.724,-999.000,
     &1.462,-999.000,2.044,-999.000,4.317,-999.000,
     &8.653,-999.000,11.241,-999.000,10.434,-999.000,
     &6.120,-999.000,-0.380,-999.000,-6.402,-999.000,
     &-9.623,-11.940,-13.250,-14.195,-14.992
     &/
       data (fjrocvz(it,11,3, 9), it=1,59) /
     &-18.495,-20.273,-20.605,-19.983,-19.845,-19.475,
     &-18.038,-17.861,-21.261,-19.883,-19.491,-13.368,
     &-16.474,-9.850,-3.375,3.463,10.732,13.099,
     &15.207,16.905,20.192,22.715,22.281,21.695,
     &21.583,20.899,19.870,18.496,16.230,13.601,
     &10.620,8.017,6.515,4.313,2.165,-999.000,
     &1.024,-999.000,1.260,-999.000,4.121,-999.000,
     &12.134,15.575,16.073,13.583,11.753,-999.000,
     &7.557,-999.000,2.179,-999.000,-4.620,-999.000,
     &-9.021,-13.146,-15.544,-17.067,-18.495
     &/
       data (fjrocvz(it,12,3, 9), it=1,59) /
     &-19.108,-21.595,-24.116,-25.997,-26.041,-24.030,
     &-23.705,-27.819,-33.001,-30.677,-24.980,-15.302,
     &-14.164,-10.181,-6.417,-0.039,6.771,10.690,
     &14.071,18.361,22.873,25.035,25.193,24.725,
     &24.175,22.785,21.282,20.181,18.040,15.242,
     &12.487,10.982,9.709,8.045,6.034,-999.000,
     &4.225,-999.000,5.529,-999.000,12.719,18.287,
     &22.776,24.336,22.251,-999.000,16.623,-999.000,
     &9.272,-999.000,2.229,-999.000,-2.524,-999.000,
     &-7.976,-12.091,-15.478,-17.551,-19.108
     &/
       data (fjrocvz(it,13,3, 9), it=1,59) /
     &-17.694,-19.663,-20.565,-21.715,-23.978,-25.711,
     &-28.155,-30.952,-33.297,-31.472,-24.354,-16.565,
     &-12.665,-10.811,-7.407,-3.593,0.310,7.072,
     &15.339,21.303,25.267,27.488,27.723,26.519,
     &25.326,23.855,22.227,20.586,18.317,16.184,
     &14.259,12.137,10.622,9.596,8.304,-999.000,
     &8.217,-999.000,10.682,-999.000,16.397,22.275,
     &29.248,31.894,26.398,-999.000,18.858,-999.000,
     &10.406,-999.000,2.043,-999.000,-3.244,-999.000,
     &-7.348,-9.641,-12.132,-15.307,-17.694
     &/
       data (fjrocvz(it,14,3, 9), it=1,59) /
     &-18.293,-20.109,-20.089,-21.010,-23.618,-27.276,
     &-29.617,-31.813,-34.838,-33.619,-25.066,-14.108,
     &-9.126,-7.889,-6.057,-3.090,0.077,5.992,
     &15.167,20.901,25.180,27.433,27.565,25.793,
     &24.346,23.243,22.222,20.879,19.274,17.781,
     &15.806,13.557,12.202,10.934,10.420,-999.000,
     &10.848,-999.000,13.293,-999.000,18.892,23.603,
     &30.523,33.680,31.312,23.697,18.302,-999.000,
     &9.002,-999.000,1.993,-999.000,-2.832,-999.000,
     &-6.490,-8.919,-11.527,-15.306,-18.293
     &/
       data (fjrocvz(it,15,3, 9), it=1,59) /
     &-18.464,-19.381,-19.603,-19.966,-21.554,-24.889,
     &-27.711,-30.792,-34.233,-35.170,-27.040,-15.463,
     &-8.685,-7.651,-6.826,-4.420,-0.866,4.766,
     &12.490,18.038,21.850,23.957,23.323,21.188,
     &19.301,17.849,17.010,15.736,13.949,12.934,
     &11.888,10.779,9.332,8.341,8.056,-999.000,
     &8.223,-999.000,10.576,-999.000,15.457,19.937,
     &24.203,25.569,21.859,16.333,11.916,-999.000,
     &3.828,-999.000,-1.671,-999.000,-4.919,-999.000,
     &-9.002,-11.204,-13.372,-15.934,-18.464
     &/
       data (fjrocvz(it,16,3, 9), it=1,59) /
     &-16.276,-16.526,-16.188,-16.331,-18.094,-22.142,
     &-26.447,-29.703,-34.496,-37.359,-33.088,-20.470,
     &-10.243,-7.490,-6.794,-4.710,-1.272,4.006,
     &10.818,16.011,19.289,20.503,19.166,17.456,
     &15.795,13.978,12.083,9.535,7.026,5.743,
     &5.186,4.349,3.262,1.828,1.327,-999.000,
     &2.601,-999.000,5.221,-999.000,9.859,13.627,
     &15.355,13.679,10.538,-999.000,1.811,-999.000,
     &-4.768,-999.000,-8.617,-999.000,-9.919,-999.000,
     &-11.143,-12.668,-13.912,-14.940,-16.276
     &/
       data (fjrocvz(it,17,3, 9), it=1,59) /
     &-17.233,-17.193,-16.283,-15.973,-17.283,-20.433,
     &-24.867,-31.120,-38.059,-40.779,-35.910,-20.625,
     &-8.627,-6.420,-4.807,-1.931,1.439,6.166,
     &12.344,17.709,20.940,22.185,21.291,19.078,
     &16.881,14.787,12.586,9.600,6.744,4.533,
     &2.452,1.316,0.363,-1.090,-2.030,-999.000,
     &-2.052,-999.000,0.077,-999.000,6.258,11.129,
     &11.424,8.165,3.969,-999.000,-3.827,-999.000,
     &-10.646,-999.000,-14.213,-999.000,-14.967,-999.000,
     &-15.318,-16.014,-16.326,-16.488,-17.233
     &/
       data (fjrocvz(it,18,3, 9), it=1,59) /
     &-24.139,-22.617,-20.725,-19.071,-18.646,-20.350,
     &-23.338,-29.860,-38.635,-40.790,-34.327,-12.746,
     &-3.466,-2.645,-0.944,2.182,5.776,11.345,
     &17.642,22.592,24.973,26.206,25.784,24.038,
     &21.470,18.379,15.787,12.712,9.902,6.940,
     &4.312,2.545,1.013,-0.446,-1.981,-999.000,
     &-2.348,-999.000,0.024,-999.000,7.421,9.233,
     &7.151,2.199,-2.498,-999.000,-10.220,-999.000,
     &-16.776,-999.000,-19.469,-999.000,-20.083,-999.000,
     &-20.501,-21.800,-22.806,-23.456,-24.139
     &/
       data (fjrocvz(it,19,3, 9), it=1,59) /
     &-25.414,-24.716,-22.884,-21.125,-20.663,-22.372,
     &-23.334,-25.858,-32.152,-32.436,-26.511,-9.648,
     &-0.560,1.098,2.682,5.482,9.759,15.301,
     &21.499,26.085,28.674,29.331,28.567,27.031,
     &24.786,22.032,18.956,15.941,13.080,10.146,
     &7.797,6.342,5.119,2.889,1.198,-999.000,
     &1.420,-999.000,2.609,-999.000,9.531,10.627,
     &8.426,3.372,-3.041,-999.000,-11.309,-999.000,
     &-17.427,-999.000,-19.421,-999.000,-20.605,-999.000,
     &-21.223,-22.442,-23.872,-24.876,-25.414
     &/
       data (fjrocvz(it,20,3, 9), it=1,59) /
     &-22.130,-21.976,-20.593,-18.962,-19.023,-21.737,
     &-24.316,-26.738,-28.274,-25.333,-18.702,-8.702,
     &-0.663,2.716,4.347,6.371,10.011,15.742,
     &22.238,28.140,31.020,31.177,29.664,27.335,
     &24.938,22.837,21.302,19.838,17.449,14.625,
     &11.827,9.929,7.772,5.963,5.126,-999.000,
     &4.803,-999.000,6.220,-999.000,11.693,14.010,
     &13.211,9.199,3.337,-999.000,-6.572,-999.000,
     &-12.094,-999.000,-13.839,-999.000,-15.818,-999.000,
     &-17.785,-19.803,-21.415,-21.624,-22.130
     &/
       data (fjrocvz(it,21,3, 9), it=1,59) /
     &-18.921,-19.040,-17.828,-16.363,-16.487,-19.935,
     &-22.758,-24.471,-27.387,-29.374,-26.164,-15.661,
     &-3.928,0.801,2.843,4.602,7.228,11.743,
     &17.834,23.233,26.268,26.702,25.766,24.681,
     &23.153,21.637,20.711,19.247,17.157,14.502,
     &12.237,10.140,8.411,7.440,6.672,-999.000,
     &6.608,-999.000,6.773,-999.000,8.805,10.861,
     &13.368,11.605,7.220,-999.000,-1.276,-999.000,
     &-6.716,-999.000,-9.404,-999.000,-12.267,-999.000,
     &-14.899,-16.974,-18.487,-18.507,-18.921
     &/
       data (fjrocvz(it,22,3, 9), it=1,59) /
     &-17.385,-15.894,-14.064,-13.522,-14.461,-17.479,
     &-19.930,-23.394,-29.256,-33.176,-30.686,-20.037,
     &-8.117,-1.006,0.813,2.321,4.228,7.611,
     &12.147,16.355,19.734,20.128,19.365,19.084,
     &18.672,17.528,16.398,14.643,12.266,10.019,
     &8.212,7.107,6.120,6.185,7.214,-999.000,
     &7.689,-999.000,7.534,-999.000,10.625,11.813,
     &15.802,14.891,8.300,-999.000,0.243,-999.000,
     &-5.454,-999.000,-9.145,-999.000,-11.744,-999.000,
     &-14.097,-16.075,-17.746,-17.976,-17.385
     &/
       data (fjrocvz(it,23,3, 9), it=1,59) /
     &-18.519,-16.339,-14.094,-12.702,-13.403,-16.870,
     &-20.330,-23.605,-27.154,-28.715,-24.780,-14.722,
     &-3.429,1.058,1.297,1.981,3.017,5.488,
     &8.702,13.198,17.662,19.151,18.081,16.242,
     &14.567,12.524,11.289,10.011,7.971,6.009,
     &5.397,4.991,4.056,4.372,5.821,-999.000,
     &7.117,-999.000,9.358,-999.000,14.419,18.595,
     &22.144,20.411,12.915,-999.000,1.068,-999.000,
     &-5.402,-999.000,-9.456,-999.000,-12.013,-999.000,
     &-14.668,-16.908,-18.439,-18.816,-18.519
     &/
       data (fjrocvz(it,24,3, 9), it=1,59) /
     &-20.404,-19.495,-17.476,-14.849,-13.133,-14.741,
     &-16.883,-19.686,-24.392,-25.406,-20.930,-9.914,
     &0.728,4.133,5.296,6.260,6.311,7.425,
     &9.460,12.786,16.634,17.496,16.731,15.148,
     &12.215,9.576,8.357,8.623,7.884,7.108,
     &6.903,6.520,5.993,6.009,6.726,-999.000,
     &10.380,-999.000,14.195,-999.000,20.512,25.244,
     &26.118,23.157,16.140,-999.000,0.071,-999.000,
     &-7.154,-999.000,-10.374,-999.000,-12.730,-999.000,
     &-15.849,-18.525,-19.928,-20.159,-20.404
     &/
       data (fjrocvz(it,25,3, 9), it=1,59) /
     &-20.279,-20.553,-19.854,-18.454,-16.118,-15.436,
     &-15.621,-19.120,-26.453,-28.322,-21.436,-7.773,
     &3.592,7.399,9.946,11.849,11.437,10.852,
     &11.390,14.183,17.907,18.928,18.407,16.039,
     &12.797,10.924,11.336,12.204,11.897,10.898,
     &10.135,9.999,9.696,9.396,10.052,-999.000,
     &14.303,-999.000,17.510,-999.000,23.213,27.359,
     &27.875,22.361,11.308,-999.000,-3.449,-999.000,
     &-10.236,-999.000,-12.545,-999.000,-13.950,-999.000,
     &-15.784,-18.277,-19.928,-19.851,-20.279
     &/
       data (fjrocvz(it, 1,4, 9), it=1,59) /
     &-21.709,-20.817,-19.285,-18.202,-18.971,-19.948,
     &-20.633,-21.597,-22.342,-21.677,-18.388,-11.622,
     &-12.931,-7.805,-2.029,4.434,10.765,19.933,
     &27.100,30.063,30.045,29.092,26.786,22.804,
     &18.961,16.119,11.315,7.903,8.434,10.289,
     &11.138,11.652,12.582,14.757,17.886,-999.000,
     &21.867,-999.000,27.103,32.389,33.608,28.283,
     &19.265,-999.000,1.118,-999.000,-12.633,-999.000,
     &-19.926,-999.000,-22.695,-999.000,-25.823,-999.000,
     &-25.925,-25.704,-25.312,-23.647,-21.709
     &/
       data (fjrocvz(it, 2,4, 9), it=1,59) /
     &-20.264,-19.525,-18.609,-18.153,-17.438,-17.107,
     &-17.452,-19.208,-20.752,-20.203,-14.906,-7.062,
     &-1.601,2.649,7.504,10.348,15.419,21.266,
     &26.955,30.943,32.310,31.759,29.163,25.074,
     &21.272,19.108,15.837,12.671,11.698,12.206,
     &13.572,14.620,15.899,18.129,20.380,-999.000,
     &22.852,-999.000,28.355,31.867,34.403,33.686,
     &27.605,-999.000,2.168,-999.000,-16.673,-999.000,
     &-20.929,-999.000,-26.403,-999.000,-28.787,-999.000,
     &-26.972,-26.126,-23.931,-21.492,-20.264
     &/
       data (fjrocvz(it, 3,4, 9), it=1,59) /
     &-20.412,-19.200,-18.532,-17.395,-16.382,-16.891,
     &-16.675,-16.978,-17.941,-17.343,-12.878,-6.434,
     &-0.543,4.107,6.989,10.537,15.446,19.492,
     &23.279,26.343,28.446,28.248,25.596,22.118,
     &19.364,17.893,16.000,14.133,13.186,13.438,
     &14.683,15.884,17.707,19.007,21.215,-999.000,
     &24.724,-999.000,28.024,33.629,37.737,36.761,
     &27.941,13.566,1.517,-12.752,-21.628,-22.523,
     &-24.220,-999.000,-29.239,-999.000,-29.894,-999.000,
     &-28.645,-26.392,-23.748,-21.953,-20.412
     &/
       data (fjrocvz(it, 4,4, 9), it=1,59) /
     &-21.554,-20.356,-18.741,-17.132,-16.705,-17.115,
     &-16.573,-15.341,-14.984,-14.197,-11.149,-5.670,
     &0.335,3.758,5.415,8.479,13.005,18.136,
     &22.071,24.536,25.109,23.939,21.463,18.524,
     &16.367,14.983,13.395,12.538,13.062,13.562,
     &14.605,16.278,17.870,20.169,23.826,-999.000,
     &26.167,-999.000,29.967,35.816,37.779,37.527,
     &33.006,19.790,1.564,-16.799,-24.728,-999.000,
     &-26.583,-999.000,-30.203,-999.000,-31.766,-999.000,
     &-29.481,-26.421,-24.157,-22.511,-21.554
     &/
       data (fjrocvz(it, 5,4, 9), it=1,59) /
     &-21.653,-20.280,-17.971,-16.281,-16.028,-17.137,
     &-17.185,-16.203,-16.035,-15.220,-12.246,-6.065,
     &-0.149,3.734,6.842,9.416,13.961,19.804,
     &24.468,27.318,28.075,26.885,23.811,20.493,
     &18.006,16.579,15.303,14.843,14.871,14.714,
     &15.530,17.059,19.092,21.380,23.272,-999.000,
     &27.115,-999.000,31.105,35.507,38.882,41.434,
     &37.752,21.232,0.523,-13.434,-24.333,-28.125,
     &-31.487,-34.322,-35.280,-34.355,-32.132,-999.000,
     &-28.961,-26.081,-23.764,-22.380,-21.653
     &/
       data (fjrocvz(it, 6,4, 9), it=1,59) /
     &-22.498,-21.631,-20.162,-18.773,-18.459,-18.749,
     &-17.760,-17.183,-17.145,-15.352,-13.170,-11.768,
     &-8.479,-1.429,5.570,11.098,15.967,21.547,
     &26.132,28.953,30.727,30.188,27.959,24.808,
     &22.475,20.960,19.285,17.630,16.759,15.783,
     &15.394,17.194,19.484,20.723,22.956,-999.000,
     &27.008,-999.000,31.047,38.279,44.457,46.325,
     &39.254,23.605,4.390,-10.805,-22.580,-29.285,
     &-32.668,-999.000,-34.757,-999.000,-33.252,-999.000,
     &-29.998,-26.391,-24.346,-23.482,-22.498
     &/
       data (fjrocvz(it, 7,4, 9), it=1,59) /
     &-24.564,-24.282,-23.298,-22.420,-21.947,-20.848,
     &-19.364,-18.494,-17.602,-16.486,-14.302,-11.887,
     &-6.622,-2.110,2.572,6.756,12.222,18.452,
     &23.863,28.079,30.096,30.376,29.233,27.031,
     &24.964,22.798,20.570,18.755,16.804,14.598,
     &13.905,14.787,16.259,18.446,20.396,-999.000,
     &23.271,-999.000,28.114,36.535,43.968,47.977,
     &37.191,18.154,0.734,-11.342,-22.645,-31.249,
     &-36.416,-37.287,-36.649,-35.634,-34.142,-999.000,
     &-30.635,-28.202,-26.362,-24.970,-24.564
     &/
       data (fjrocvz(it, 8,4, 9), it=1,59) /
     &-24.581,-23.471,-22.695,-22.370,-21.946,-21.837,
     &-21.395,-20.909,-19.974,-21.679,-23.444,-25.081,
     &-19.686,-10.740,-4.320,2.011,6.945,11.444,
     &17.673,22.569,25.733,26.293,25.737,24.570,
     &22.955,21.197,18.964,16.812,14.212,11.910,
     &10.900,11.208,12.648,14.559,16.186,-999.000,
     &19.902,-999.000,26.198,32.276,39.345,40.344,
     &28.081,13.803,0.608,-11.578,-20.560,-27.951,
     &-32.629,-32.974,-30.716,-999.000,-30.320,-999.000,
     &-28.881,-28.412,-27.152,-25.742,-24.581
     &/
       data (fjrocvz(it, 9,4, 9), it=1,59) /
     &-21.942,-20.006,-18.254,-17.203,-17.303,-18.350,
     &-20.120,-21.674,-22.896,-24.519,-27.051,-28.916,
     &-23.472,-12.561,-4.118,1.609,4.800,8.996,
     &14.194,18.182,20.620,21.357,21.201,19.864,
     &17.629,15.582,13.711,11.754,9.649,8.308,
     &8.198,8.792,9.942,11.569,13.541,-999.000,
     &16.656,-999.000,23.920,30.946,37.350,40.965,
     &36.658,23.100,9.272,-3.144,-12.107,-999.000,
     &-18.521,-999.000,-21.874,-999.000,-24.334,-999.000,
     &-24.558,-24.529,-24.614,-23.851,-21.942
     &/
       data (fjrocvz(it,10,4, 9), it=1,59) /
     &-22.909,-20.532,-18.179,-16.906,-16.140,-17.314,
     &-20.434,-23.110,-24.597,-25.304,-25.903,-25.606,
     &-21.304,-9.957,-1.634,3.938,6.834,11.316,
     &16.655,20.169,21.200,21.025,19.886,18.345,
     &16.425,13.905,11.606,9.707,8.506,7.938,
     &8.002,8.349,8.389,9.292,11.385,-999.000,
     &15.707,-999.000,24.021,32.076,40.943,45.036,
     &42.953,29.515,16.147,-999.000,-0.183,-999.000,
     &-10.493,-999.000,-17.751,-999.000,-19.764,-999.000,
     &-21.563,-22.414,-23.306,-23.468,-22.909
     &/
       data (fjrocvz(it,11,4, 9), it=1,59) /
     &-25.803,-24.771,-24.208,-24.011,-23.623,-23.820,
     &-24.461,-25.821,-27.342,-29.222,-28.488,-24.650,
     &-19.049,-9.156,-1.493,4.645,8.948,12.898,
     &17.561,21.524,23.596,23.967,23.133,21.781,
     &20.536,18.357,15.521,13.518,12.165,11.205,
     &10.560,9.816,9.286,10.152,12.815,-999.000,
     &17.792,-999.000,24.806,32.401,41.669,43.213,
     &39.437,28.634,17.725,-999.000,-0.700,-999.000,
     &-12.328,-999.000,-20.017,-999.000,-21.768,-999.000,
     &-24.699,-24.487,-25.316,-26.434,-25.803
     &/
       data (fjrocvz(it,12,4, 9), it=1,59) /
     &-27.306,-27.695,-27.434,-27.179,-27.088,-26.434,
     &-26.232,-27.754,-30.331,-32.502,-33.254,-29.982,
     &-25.767,-16.246,-6.573,-0.739,3.780,9.791,
     &16.933,22.348,25.401,26.022,25.330,24.506,
     &23.696,21.577,18.682,16.403,15.467,14.744,
     &13.821,12.570,11.677,12.593,15.724,-999.000,
     &20.199,-999.000,26.762,33.372,41.323,46.613,
     &42.342,25.278,11.040,0.652,-8.646,-999.000,
     &-14.650,-999.000,-20.123,-999.000,-24.244,-999.000,
     &-26.878,-26.820,-27.051,-27.333,-27.306
     &/
       data (fjrocvz(it,13,4, 9), it=1,59) /
     &-25.557,-24.861,-24.530,-24.413,-23.898,-24.381,
     &-25.843,-27.079,-28.335,-29.133,-29.613,-28.187,
     &-25.429,-17.177,-8.564,-2.111,1.937,8.784,
     &18.550,24.852,27.092,27.576,26.984,26.340,
     &25.469,23.166,20.465,18.721,18.669,18.591,
     &17.947,16.731,15.494,16.054,18.650,-999.000,
     &22.256,-999.000,30.957,37.116,45.498,51.195,
     &44.110,26.488,7.325,-4.593,-12.087,-14.185,
     &-15.309,-999.000,-19.066,-999.000,-21.949,-999.000,
     &-24.215,-26.177,-26.040,-25.532,-25.557
     &/
       data (fjrocvz(it,14,4, 9), it=1,59) /
     &-23.285,-22.738,-22.344,-21.710,-21.580,-23.171,
     &-25.928,-29.477,-32.618,-32.539,-29.714,-25.784,
     &-22.057,-18.626,-11.730,-3.962,1.608,8.955,
     &18.721,24.807,27.541,28.046,27.257,26.236,
     &24.785,23.107,21.152,20.012,20.296,20.393,
     &19.571,18.485,17.528,18.021,20.692,-999.000,
     &25.649,-999.000,34.392,43.910,50.098,49.555,
     &41.133,25.811,9.890,-999.000,-10.761,-999.000,
     &-18.556,-999.000,-20.810,-999.000,-23.232,-999.000,
     &-25.721,-26.472,-25.490,-24.054,-23.285
     &/
       data (fjrocvz(it,15,4, 9), it=1,59) /
     &-20.907,-20.412,-19.862,-19.411,-19.442,-21.192,
     &-25.186,-31.284,-35.563,-36.926,-33.550,-26.390,
     &-21.602,-16.380,-10.846,-3.222,3.553,10.077,
     &17.808,23.230,25.809,25.886,24.359,22.076,
     &20.048,18.695,16.928,15.829,15.839,15.811,
     &15.311,14.703,14.955,16.399,19.863,-999.000,
     &24.424,-999.000,32.022,40.674,47.159,49.250,
     &41.799,24.958,9.206,-999.000,-12.607,-999.000,
     &-19.936,-999.000,-23.747,-999.000,-26.665,-999.000,
     &-28.141,-28.296,-25.781,-22.463,-20.907
     &/
       data (fjrocvz(it,16,4, 9), it=1,59) /
     &-20.596,-19.532,-19.765,-20.050,-20.903,-23.249,
     &-27.095,-31.906,-35.752,-38.587,-37.485,-29.958,
     &-20.452,-14.593,-8.478,-1.682,5.010,10.591,
     &17.982,23.053,25.006,25.305,24.053,21.018,
     &17.479,14.492,12.131,10.812,10.837,11.286,
     &10.900,10.210,10.449,11.714,14.536,-999.000,
     &19.216,-999.000,26.543,34.709,41.910,43.403,
     &34.863,17.164,4.348,-999.000,-13.448,-999.000,
     &-21.706,-999.000,-26.469,-999.000,-29.322,-999.000,
     &-30.575,-29.995,-27.366,-23.614,-20.596
     &/
       data (fjrocvz(it,17,4, 9), it=1,59) /
     &-24.003,-23.515,-22.956,-22.648,-23.577,-25.611,
     &-28.909,-33.654,-37.783,-39.227,-37.438,-29.274,
     &-20.220,-13.457,-6.403,-0.491,5.580,11.478,
     &19.374,24.788,27.356,28.285,27.329,24.668,
     &20.631,16.769,13.171,10.652,10.309,10.308,
     &9.543,8.423,7.508,7.620,10.321,-999.000,
     &15.969,-999.000,24.217,30.499,35.603,37.533,
     &30.708,14.740,0.905,-12.226,-21.063,-25.088,
     &-27.175,-999.000,-28.652,-999.000,-30.785,-999.000,
     &-31.508,-30.270,-28.580,-26.135,-24.003
     &/
       data (fjrocvz(it,18,4, 9), it=1,59) /
     &-25.782,-24.797,-23.778,-23.618,-23.807,-25.039,
     &-28.876,-33.692,-36.945,-38.261,-35.637,-28.269,
     &-19.117,-12.307,-6.813,-0.748,5.881,13.589,
     &21.018,27.011,31.244,32.307,30.279,26.562,
     &23.143,19.889,15.909,12.413,11.061,10.039,
     &8.967,7.568,6.667,6.644,9.004,-999.000,
     &14.255,18.187,23.508,29.819,34.303,31.941,
     &24.823,13.142,-0.507,-999.000,-20.727,-999.000,
     &-28.238,-999.000,-31.491,-999.000,-32.540,-999.000,
     &-32.127,-31.711,-30.233,-27.855,-25.782
     &/
       data (fjrocvz(it,19,4, 9), it=1,59) /
     &-24.516,-23.304,-23.335,-22.957,-22.703,-23.895,
     &-26.816,-29.735,-32.501,-33.932,-31.227,-24.353,
     &-16.167,-11.152,-5.735,0.366,6.692,15.128,
     &23.671,30.519,35.254,36.087,33.707,29.949,
     &26.097,22.357,18.356,14.816,13.906,13.321,
     &11.921,10.208,9.142,8.789,10.736,-999.000,
     &16.831,-999.000,25.648,32.496,35.505,32.640,
     &26.969,16.049,-1.419,-14.165,-23.718,-27.835,
     &-29.331,-999.000,-31.322,-999.000,-31.526,-999.000,
     &-31.297,-31.029,-29.869,-27.266,-24.516
     &/
       data (fjrocvz(it,20,4, 9), it=1,59) /
     &-25.925,-24.727,-23.176,-21.569,-20.619,-21.244,
     &-23.569,-26.861,-29.011,-29.646,-27.091,-18.041,
     &-9.975,-7.488,-2.721,1.951,7.131,15.697,
     &23.935,31.620,36.569,37.765,36.048,32.810,
     &29.410,26.241,22.148,18.741,17.656,16.548,
     &14.904,13.140,11.870,11.558,14.022,-999.000,
     &19.783,-999.000,27.070,33.633,39.368,41.377,
     &32.455,16.559,-1.550,-11.240,-18.993,-24.972,
     &-28.789,-999.000,-30.424,-999.000,-31.565,-999.000,
     &-31.425,-31.104,-30.294,-28.124,-25.925
     &/
       data (fjrocvz(it,21,4, 9), it=1,59) /
     &-25.674,-23.418,-21.594,-20.447,-19.818,-19.936,
     &-21.252,-24.141,-27.335,-28.053,-25.266,-18.012,
     &-12.612,-7.408,-2.168,3.228,9.080,15.760,
     &21.866,28.768,32.998,34.800,33.246,29.928,
     &26.835,24.447,21.067,17.291,15.614,14.315,
     &12.806,11.816,11.686,11.950,13.982,-999.000,
     &18.868,-999.000,27.233,33.307,39.393,38.031,
     &28.899,-999.000,2.520,-999.000,-15.037,-999.000,
     &-24.879,-999.000,-29.189,-999.000,-30.436,-999.000,
     &-30.671,-30.136,-29.348,-27.779,-25.674
     &/
       data (fjrocvz(it,22,4, 9), it=1,59) /
     &-24.518,-23.388,-22.216,-21.355,-21.221,-21.264,
     &-21.628,-23.975,-28.018,-28.825,-25.796,-18.895,
     &-10.744,-7.445,-1.819,3.204,8.369,14.412,
     &20.948,26.964,29.979,30.653,28.801,25.076,
     &20.849,17.538,14.320,11.553,10.341,9.489,
     &8.874,9.448,10.706,11.518,13.794,-999.000,
     &19.139,-999.000,28.080,31.107,34.821,30.819,
     &22.610,-999.000,-0.301,-999.000,-16.203,-999.000,
     &-26.004,-999.000,-28.169,-999.000,-28.579,-999.000,
     &-29.265,-28.748,-27.865,-26.206,-24.518
     &/
       data (fjrocvz(it,23,4, 9), it=1,59) /
     &-23.085,-22.557,-21.016,-19.019,-18.361,-19.271,
     &-21.121,-23.456,-25.609,-26.059,-23.355,-17.075,
     &-11.676,-8.699,-3.237,2.061,7.401,13.079,
     &21.239,27.040,29.023,28.439,26.000,22.127,
     &17.911,14.039,10.710,8.430,7.900,7.680,
     &7.498,7.651,9.429,11.700,14.869,-999.000,
     &18.864,-999.000,26.976,31.554,34.412,31.398,
     &23.847,10.629,-1.032,-999.000,-15.788,-999.000,
     &-23.019,-999.000,-26.077,-999.000,-27.050,-999.000,
     &-27.933,-27.573,-26.341,-24.100,-23.085
     &/
       data (fjrocvz(it,24,4, 9), it=1,59) /
     &-21.843,-20.291,-18.511,-16.387,-16.145,-17.918,
     &-20.057,-21.340,-22.407,-22.927,-20.758,-15.840,
     &-9.639,-4.516,-0.968,2.398,7.219,15.617,
     &23.878,27.577,28.474,27.731,24.824,20.625,
     &16.615,13.050,9.304,6.866,7.349,8.735,
     &9.215,9.066,10.614,13.019,15.975,-999.000,
     &19.535,-999.000,26.670,33.253,35.897,32.818,
     &22.020,-999.000,1.422,-999.000,-11.446,-999.000,
     &-19.047,-999.000,-24.322,-999.000,-25.163,-999.000,
     &-26.272,-26.297,-25.289,-23.609,-21.843
     &/
       data (fjrocvz(it,25,4, 9), it=1,59) /
     &-21.709,-20.817,-19.285,-18.202,-18.971,-19.948,
     &-20.633,-21.597,-22.342,-21.677,-18.388,-11.622,
     &-12.931,-7.805,-2.029,4.434,10.765,19.933,
     &27.100,30.063,30.045,29.092,26.786,22.804,
     &18.961,16.119,11.315,7.903,8.434,10.289,
     &11.138,11.652,12.582,14.757,17.886,-999.000,
     &21.867,-999.000,27.103,32.389,33.608,28.283,
     &19.265,-999.000,1.118,-999.000,-12.633,-999.000,
     &-19.926,-999.000,-22.695,-999.000,-25.823,-999.000,
     &-25.925,-25.704,-25.312,-23.647,-21.709
     &/
       data (fjrocvz(it, 1,1,10), it=1,59) /
     &-13.981,-13.129,-12.662,-12.673,-12.805,-14.615,
     &-16.640,-18.801,-21.707,-23.164,-22.009,-15.703,
     &-5.979,1.949,7.992,12.538,17.186,22.926,
     &27.529,30.607,31.411,29.518,27.143,24.126,
     &20.935,17.594,13.960,11.282,8.688,5.935,
     &4.231,3.964,4.604,4.407,4.352,-999.000,
     &3.732,-999.000,4.523,-999.000,7.318,-999.000,
     &5.699,-999.000,1.569,-999.000,-6.126,-999.000,
     &-11.454,-999.000,-14.869,-999.000,-17.136,-999.000,
     &-19.338,-19.661,-18.343,-15.805,-13.981
     &/
       data (fjrocvz(it, 2,1,10), it=1,59) /
     &-16.897,-15.505,-15.010,-14.811,-15.541,-17.700,
     &-18.674,-20.062,-21.986,-22.625,-18.827,-9.120,
     &0.094,6.747,12.263,18.136,23.133,28.107,
     &32.497,35.478,36.613,34.585,31.321,27.595,
     &23.378,19.298,14.949,11.570,8.779,6.307,
     &3.776,2.441,2.450,2.763,3.029,-999.000,
     &3.578,-999.000,5.514,-999.000,6.381,-999.000,
     &4.555,-999.000,-0.470,-999.000,-8.535,-999.000,
     &-14.513,-999.000,-18.703,-999.000,-21.050,-999.000,
     &-22.404,-21.990,-20.683,-18.826,-16.897
     &/
       data (fjrocvz(it, 3,1,10), it=1,59) /
     &-19.409,-18.589,-18.288,-18.503,-19.142,-19.355,
     &-18.644,-18.873,-18.980,-17.417,-11.385,-2.558,
     &5.267,10.745,16.105,21.538,25.386,29.235,
     &33.189,36.415,37.214,34.589,30.835,27.190,
     &23.777,20.706,16.982,13.152,9.866,7.178,
     &4.448,3.264,3.308,4.075,4.506,-999.000,
     &5.715,-999.000,6.842,-999.000,8.526,10.208,
     &10.658,8.171,3.625,-0.550,-6.757,-999.000,
     &-15.413,-999.000,-19.153,-999.000,-21.192,-999.000,
     &-22.247,-22.014,-21.059,-20.257,-19.409
     &/
       data (fjrocvz(it, 4,1,10), it=1,59) /
     &-19.826,-20.451,-20.333,-19.441,-18.394,-18.828,
     &-19.011,-19.448,-18.658,-14.874,-7.325,1.284,
     &7.215,11.827,17.502,22.381,26.772,30.357,
     &34.026,36.235,35.659,32.107,28.191,25.167,
     &22.323,18.778,15.000,12.235,10.433,8.970,
     &6.590,4.760,4.476,5.352,7.408,-999.000,
     &8.722,-999.000,11.671,-999.000,16.331,16.875,
     &17.075,12.540,6.338,-999.000,-4.471,-999.000,
     &-15.758,-999.000,-19.527,-999.000,-21.842,-999.000,
     &-22.150,-21.215,-19.339,-18.685,-19.826
     &/
       data (fjrocvz(it, 5,1,10), it=1,59) /
     &-16.688,-17.355,-17.400,-16.401,-15.807,-17.435,
     &-19.584,-21.038,-20.701,-16.655,-8.543,0.868,
     &6.431,11.013,16.856,22.119,26.604,29.746,
     &33.050,35.116,34.638,31.638,28.584,25.811,
     &22.429,18.334,14.545,12.501,11.750,11.461,
     &9.915,7.701,6.940,6.999,8.391,-999.000,
     &11.658,-999.000,16.999,-999.000,20.413,25.091,
     &24.309,18.492,11.242,-999.000,-3.156,-999.000,
     &-15.965,-999.000,-20.296,-999.000,-21.501,-999.000,
     &-21.780,-19.879,-17.004,-15.956,-16.688
     &/
       data (fjrocvz(it, 6,1,10), it=1,59) /
     &-14.347,-15.115,-15.850,-16.624,-17.105,-18.683,
     &-20.930,-22.567,-22.533,-17.512,-10.284,-1.757,
     &3.896,7.614,11.867,16.305,20.639,24.120,
     &28.264,31.578,31.706,29.865,27.704,25.345,
     &22.229,19.234,16.354,14.124,13.217,13.334,
     &12.985,11.903,10.220,9.248,10.301,-999.000,
     &13.806,-999.000,18.214,-999.000,23.637,-999.000,
     &24.231,-999.000,15.387,-999.000,2.422,-999.000,
     &-8.945,-999.000,-14.933,-999.000,-18.378,-999.000,
     &-19.480,-18.762,-16.121,-14.577,-14.347
     &/
       data (fjrocvz(it, 7,1,10), it=1,59) /
     &-16.366,-17.388,-18.568,-19.645,-20.571,-21.773,
     &-22.511,-22.886,-22.906,-18.974,-14.358,-9.835,
     &-4.435,1.515,7.167,11.549,16.371,19.542,
     &24.392,27.709,28.221,26.601,24.250,21.996,
     &19.739,17.846,15.151,12.805,11.635,12.223,
     &13.219,13.073,11.966,11.524,12.735,-999.000,
     &15.413,-999.000,20.361,-999.000,26.053,28.797,
     &33.180,31.469,21.862,-999.000,7.970,-999.000,
     &-2.602,-999.000,-10.057,-999.000,-15.122,-999.000,
     &-17.899,-18.355,-16.840,-16.624,-16.366
     &/
       data (fjrocvz(it, 8,1,10), it=1,59) /
     &-19.421,-19.270,-19.249,-19.344,-19.836,-20.827,
     &-21.801,-22.957,-24.376,-22.374,-18.916,-12.991,
     &-6.343,-1.971,1.492,4.814,9.146,13.092,
     &18.423,22.463,24.028,23.006,20.738,18.409,
     &15.964,14.107,11.947,9.768,8.924,9.739,
     &11.458,12.416,12.691,12.749,14.084,-999.000,
     &17.774,-999.000,23.866,-999.000,31.820,39.832,
     &43.921,42.494,32.520,20.344,12.784,-999.000,
     &-3.234,-999.000,-11.768,-999.000,-18.976,-999.000,
     &-20.557,-21.510,-20.684,-20.179,-19.421
     &/
       data (fjrocvz(it, 9,1,10), it=1,59) /
     &-20.211,-19.363,-18.675,-18.247,-18.213,-19.565,
     &-21.279,-23.071,-24.780,-23.666,-21.017,-16.447,
     &-11.249,-6.363,-1.374,2.324,5.529,8.755,
     &13.641,17.959,19.983,20.105,19.067,17.513,
     &15.307,14.014,12.743,11.518,10.663,10.914,
     &11.736,12.202,12.611,13.605,16.064,17.225,
     &18.703,21.319,27.000,33.605,40.740,49.291,
     &55.303,49.293,30.343,13.602,2.328,-6.312,
     &-10.005,-999.000,-16.947,-999.000,-20.694,-999.000,
     &-23.370,-24.114,-23.007,-21.670,-20.211
     &/
       data (fjrocvz(it,10,1,10), it=1,59) /
     &-18.997,-17.845,-17.010,-16.456,-16.972,-19.018,
     &-20.617,-21.858,-23.830,-24.643,-23.145,-18.045,
     &-11.048,-4.914,0.317,4.395,7.428,10.016,
     &13.568,16.816,19.287,21.122,21.696,21.021,
     &19.078,17.805,16.672,15.667,14.555,14.906,
     &15.021,15.134,15.697,16.503,17.867,-999.000,
     &21.463,26.036,34.353,43.849,54.304,59.090,
     &51.922,35.672,17.220,0.842,-6.555,-11.797,
     &-15.336,-999.000,-19.511,-999.000,-22.609,-999.000,
     &-24.262,-23.935,-22.411,-20.843,-18.997
     &/
       data (fjrocvz(it,11,1,10), it=1,59) /
     &-17.436,-16.205,-15.827,-16.313,-17.764,-20.349,
     &-22.183,-23.515,-25.346,-25.621,-23.241,-15.975,
     &-7.898,-1.259,4.382,8.521,11.274,12.908,
     &15.539,19.541,23.701,26.384,27.031,25.848,
     &24.032,22.662,21.307,19.386,17.723,17.535,
     &16.648,16.289,16.946,18.226,20.248,22.160,
     &24.957,30.985,39.603,46.380,50.456,48.346,
     &33.911,16.512,3.371,-7.190,-14.482,-999.000,
     &-17.824,-999.000,-21.493,-999.000,-23.082,-999.000,
     &-23.469,-22.655,-21.013,-19.115,-17.436
     &/
       data (fjrocvz(it,12,1,10), it=1,59) /
     &-19.095,-18.690,-19.017,-20.384,-21.681,-23.331,
     &-24.289,-25.698,-28.102,-28.382,-26.574,-21.542,
     &-13.764,-5.230,4.400,9.508,13.323,15.588,
     &18.402,22.390,25.905,28.146,28.856,28.505,
     &27.620,26.412,24.232,21.560,19.105,17.869,
     &16.997,16.408,17.087,19.268,23.036,-999.000,
     &28.712,30.535,37.397,43.298,46.276,39.096,
     &25.684,14.205,4.225,-999.000,-12.233,-999.000,
     &-20.824,-999.000,-24.163,-999.000,-24.264,-999.000,
     &-23.387,-22.485,-21.488,-20.126,-19.095
     &/
       data (fjrocvz(it,13,1,10), it=1,59) /
     &-20.323,-20.943,-21.698,-22.636,-22.969,-23.665,
     &-24.249,-26.240,-28.820,-30.064,-29.160,-24.682,
     &-16.190,-7.787,0.696,7.615,13.259,16.010,
     &19.054,22.630,25.663,27.368,27.623,27.054,
     &25.753,24.544,22.972,20.590,17.541,15.955,
     &15.807,16.008,16.613,17.362,18.308,-999.000,
     &22.758,-999.000,30.530,36.157,38.307,32.380,
     &21.942,-999.000,7.813,-999.000,-7.611,-999.000,
     &-18.336,-999.000,-23.991,-999.000,-24.562,-999.000,
     &-23.947,-22.981,-21.829,-20.675,-20.323
     &/
       data (fjrocvz(it,14,1,10), it=1,59) /
     &-20.971,-22.563,-23.636,-23.904,-23.351,-24.110,
     &-26.081,-28.588,-30.629,-31.199,-28.297,-25.027,
     &-19.101,-11.288,-2.199,5.938,12.954,15.625,
     &18.254,21.640,25.192,26.901,26.793,25.454,
     &24.025,22.966,21.362,18.502,15.532,13.931,
     &13.824,14.097,14.033,12.402,12.149,-999.000,
     &14.895,17.334,23.121,28.531,30.814,27.082,
     &20.737,12.249,6.406,-999.000,-5.908,-999.000,
     &-16.500,-999.000,-23.856,-999.000,-25.601,-999.000,
     &-25.484,-24.165,-22.256,-21.070,-20.971
     &/
       data (fjrocvz(it,15,1,10), it=1,59) /
     &-21.506,-22.416,-23.871,-24.874,-24.536,-24.876,
     &-26.121,-28.299,-31.667,-33.199,-30.415,-25.157,
     &-17.114,-9.618,-2.141,5.259,12.370,15.568,
     &18.118,20.021,21.995,23.016,22.470,21.072,
     &19.964,19.263,18.150,16.465,13.981,12.047,
     &11.528,11.508,11.115,10.469,10.704,-999.000,
     &13.123,-999.000,19.512,26.417,31.903,28.390,
     &22.210,14.052,8.086,-999.000,-4.266,-999.000,
     &-15.174,-999.000,-23.912,-999.000,-26.251,-999.000,
     &-25.701,-24.543,-23.076,-22.105,-21.506
     &/
       data (fjrocvz(it,16,1,10), it=1,59) /
     &-22.626,-23.004,-23.850,-24.811,-24.695,-24.737,
     &-25.284,-26.691,-30.288,-34.475,-34.694,-28.869,
     &-13.236,-4.308,0.939,6.746,12.924,15.823,
     &18.322,20.279,22.259,22.556,21.967,20.755,
     &19.565,18.418,17.044,14.915,12.097,10.376,
     &10.360,10.973,11.445,11.502,12.150,-999.000,
     &15.144,-999.000,21.362,28.334,33.800,28.496,
     &20.235,-999.000,6.754,-999.000,-4.387,-999.000,
     &-14.956,-999.000,-23.059,-999.000,-25.393,-999.000,
     &-25.520,-24.670,-23.759,-23.260,-22.626
     &/
       data (fjrocvz(it,17,1,10), it=1,59) /
     &-21.823,-21.812,-22.069,-22.971,-23.672,-25.092,
     &-25.975,-25.489,-27.114,-30.503,-32.601,-27.728,
     &-11.840,0.152,4.407,8.187,12.605,15.274,
     &17.759,19.919,21.699,22.985,23.150,22.242,
     &21.340,20.162,18.606,16.467,13.750,11.820,
     &11.591,12.339,12.614,12.107,13.090,-999.000,
     &15.547,18.599,21.467,26.536,30.211,26.841,
     &18.046,-999.000,2.072,-999.000,-9.468,-999.000,
     &-18.648,-999.000,-24.692,-999.000,-26.753,-999.000,
     &-26.045,-24.059,-22.678,-22.258,-21.823
     &/
       data (fjrocvz(it,18,1,10), it=1,59) /
     &-20.500,-20.311,-20.558,-21.749,-23.428,-25.897,
     &-27.287,-27.844,-28.788,-28.851,-27.742,-22.957,
     &-10.652,1.224,6.152,9.612,13.924,17.022,
     &19.579,20.730,22.084,23.668,23.852,22.798,
     &21.479,20.268,18.549,16.915,15.096,13.426,
     &12.657,12.864,13.285,13.328,13.796,-999.000,
     &17.235,-999.000,21.839,27.729,30.226,24.468,
     &13.850,4.542,-3.038,-999.000,-12.691,-999.000,
     &-20.227,-999.000,-25.941,-999.000,-27.621,-999.000,
     &-26.231,-23.705,-22.268,-21.410,-20.500
     &/
       data (fjrocvz(it,19,1,10), it=1,59) /
     &-21.614,-20.544,-19.667,-19.881,-20.382,-22.032,
     &-22.401,-23.774,-26.095,-28.073,-27.830,-24.035,
     &-14.139,-1.573,5.671,10.695,15.557,19.547,
     &22.578,24.129,25.824,27.097,27.566,26.318,
     &25.038,24.103,21.759,18.511,16.320,15.243,
     &15.035,14.812,14.565,13.432,13.041,-999.000,
     &15.931,-999.000,20.011,23.484,22.964,14.967,
     &8.304,-999.000,-4.080,-999.000,-13.094,-999.000,
     &-20.795,-999.000,-25.789,-999.000,-27.314,-999.000,
     &-26.152,-24.102,-22.924,-22.437,-21.614
     &/
       data (fjrocvz(it,20,1,10), it=1,59) /
     &-20.633,-19.287,-18.769,-18.664,-18.646,-19.663,
     &-19.733,-20.348,-22.397,-25.022,-27.759,-25.910,
     &-17.897,-4.422,3.072,7.529,12.939,18.051,
     &21.850,24.072,25.931,27.286,27.796,27.222,
     &27.209,27.263,25.546,22.221,18.930,17.297,
     &16.908,16.355,15.097,13.442,12.894,-999.000,
     &13.237,-999.000,14.676,16.663,19.592,14.485,
     &5.206,-999.000,-5.864,-999.000,-15.497,-999.000,
     &-22.775,-999.000,-27.484,-999.000,-29.024,-999.000,
     &-28.515,-26.565,-24.502,-22.499,-20.633
     &/
       data (fjrocvz(it,21,1,10), it=1,59) /
     &-21.914,-20.790,-20.671,-21.017,-20.969,-20.888,
     &-19.677,-20.257,-23.152,-27.667,-31.208,-29.907,
     &-21.413,-8.071,-0.505,3.978,9.053,13.592,
     &17.584,20.708,23.218,24.241,24.328,24.468,
     &25.310,25.639,24.060,20.674,17.097,15.200,
     &14.857,14.568,13.627,11.725,10.569,-999.000,
     &11.093,-999.000,12.318,14.023,14.891,10.134,
     &1.469,-999.000,-9.193,-999.000,-17.848,-999.000,
     &-23.935,-999.000,-28.480,-999.000,-29.900,-999.000,
     &-29.612,-28.184,-26.270,-23.987,-21.914
     &/
       data (fjrocvz(it,22,1,10), it=1,59) /
     &-20.312,-19.457,-19.616,-20.576,-20.876,-20.934,
     &-20.007,-20.764,-23.435,-27.449,-30.172,-28.807,
     &-20.642,-8.397,-0.879,3.566,7.461,12.077,
     &16.261,20.097,22.097,22.742,23.374,23.739,
     &23.819,23.059,20.302,15.801,11.529,9.392,
     &9.246,9.425,8.755,7.215,6.736,-999.000,
     &7.065,-999.000,6.376,-999.000,5.974,-999.000,
     &0.751,-999.000,-11.045,-999.000,-17.539,-999.000,
     &-22.115,-999.000,-26.160,-999.000,-27.765,-999.000,
     &-27.407,-25.726,-23.509,-21.734,-20.312
     &/
       data (fjrocvz(it,23,1,10), it=1,59) /
     &-15.898,-15.795,-16.133,-16.974,-17.835,-18.573,
     &-18.844,-19.868,-21.461,-23.055,-23.512,-20.247,
     &-12.896,-4.019,2.444,6.077,9.496,13.012,
     &16.967,20.757,23.537,25.691,26.797,26.255,
     &24.533,22.329,19.639,15.999,11.651,8.508,
     &6.595,5.592,4.793,3.878,3.642,-999.000,
     &4.052,-999.000,6.990,-999.000,7.335,-999.000,
     &4.078,-999.000,-1.365,-999.000,-11.546,-999.000,
     &-16.868,-999.000,-21.217,-999.000,-22.894,-999.000,
     &-22.676,-21.171,-18.956,-16.846,-15.898
     &/
       data (fjrocvz(it,24,1,10), it=1,59) /
     &-13.649,-13.410,-13.433,-14.100,-14.470,-15.256,
     &-16.047,-17.909,-20.309,-22.029,-21.863,-17.404,
     &-9.183,-1.241,4.253,8.475,12.182,16.026,
     &19.908,23.602,26.304,26.819,26.217,24.669,
     &22.350,20.151,17.743,15.176,11.604,8.271,
     &5.912,5.104,5.165,4.676,4.058,-999.000,
     &4.704,-999.000,6.278,-999.000,7.648,-999.000,
     &6.133,-999.000,1.236,-999.000,-5.957,-999.000,
     &-11.937,-999.000,-15.601,-999.000,-17.386,-999.000,
     &-18.283,-18.225,-16.865,-14.935,-13.649
     &/
       data (fjrocvz(it,25,1,10), it=1,59) /
     &-13.981,-13.129,-12.662,-12.673,-12.805,-14.615,
     &-16.640,-18.801,-21.707,-23.164,-22.009,-15.703,
     &-5.979,1.949,7.992,12.538,17.186,22.926,
     &27.529,30.607,31.411,29.518,27.143,24.126,
     &20.935,17.594,13.960,11.282,8.688,5.935,
     &4.231,3.964,4.604,4.407,4.352,-999.000,
     &3.732,-999.000,4.523,-999.000,7.318,-999.000,
     &5.699,-999.000,1.569,-999.000,-6.126,-999.000,
     &-11.454,-999.000,-14.869,-999.000,-17.136,-999.000,
     &-19.338,-19.661,-18.343,-15.805,-13.981
     &/
       data (fjrocvz(it, 1,2,10), it=1,59) /
     &-22.847,-21.945,-20.408,-19.319,-20.078,-21.119,
     &-21.837,-22.833,-23.643,-22.971,-19.679,-12.909,
     &-14.211,-8.844,-2.947,3.637,10.210,19.481,
     &26.700,29.714,29.799,28.961,26.713,22.788,
     &19.061,16.505,11.844,8.576,9.393,11.371,
     &12.281,12.856,13.908,16.232,19.510,-999.000,
     &23.578,-999.000,29.193,35.140,36.622,31.194,
     &21.714,-999.000,1.547,-999.000,-13.699,-999.000,
     &-21.464,-999.000,-24.434,-999.000,-27.338,-999.000,
     &-27.215,-26.988,-26.589,-24.855,-22.847
     &/
       data (fjrocvz(it, 2,2,10), it=1,59) /
     &-21.475,-20.721,-19.797,-19.334,-18.604,-18.401,
     &-18.809,-20.629,-22.301,-21.809,-16.541,-8.725,
     &-3.320,1.247,6.260,9.262,14.650,20.651,
     &26.416,30.481,32.001,31.593,29.068,25.051,
     &21.392,19.513,16.384,13.360,12.671,13.309,
     &14.740,15.853,17.261,19.684,22.129,-999.000,
     &24.646,-999.000,30.448,34.240,37.056,36.378,
     &30.300,-999.000,2.868,-999.000,-18.124,-999.000,
     &-23.186,-999.000,-28.794,-999.000,-30.761,-999.000,
     &-28.528,-27.520,-25.164,-22.714,-21.475
     &/
       data (fjrocvz(it, 3,2,10), it=1,59) /
     &-21.606,-20.391,-19.720,-18.582,-17.565,-18.221,
     &-18.078,-18.455,-19.565,-18.981,-14.524,-8.087,
     &-2.210,2.738,5.768,9.465,14.672,18.914,
     &22.797,25.959,28.258,28.246,25.687,22.302,
     &19.734,18.517,16.752,15.011,14.318,14.713,
     &16.029,17.302,19.268,20.704,23.049,-999.000,
     &26.666,-999.000,30.074,35.837,40.103,39.448,
     &30.949,15.458,2.185,-13.326,-23.280,-24.463,
     &-26.448,-999.000,-31.442,-999.000,-31.774,-999.000,
     &-30.201,-27.732,-24.873,-23.112,-21.606
     &/
       data (fjrocvz(it, 4,2,10), it=1,59) /
     &-22.702,-21.487,-19.864,-18.248,-17.804,-18.347,
     &-17.872,-16.706,-16.482,-15.690,-12.641,-7.160,
     &-1.150,2.498,4.268,7.445,12.195,17.542,
     &21.584,24.157,24.946,24.001,21.637,18.811,
     &16.878,15.749,14.288,13.558,14.338,14.972,
     &16.081,17.821,19.548,21.959,25.728,-999.000,
     &28.192,-999.000,32.115,38.082,40.163,40.245,
     &36.056,21.883,2.368,-17.373,-26.296,-999.000,
     &-28.769,-999.000,-32.417,-999.000,-33.643,-999.000,
     &-31.021,-27.760,-25.296,-23.654,-22.702
     &/
       data (fjrocvz(it, 5,2,10), it=1,59) /
     &-22.651,-21.252,-18.931,-17.227,-16.949,-18.186,
     &-18.298,-17.381,-17.342,-16.550,-13.588,-7.419,
     &-1.526,2.580,5.801,8.486,13.255,19.325,
     &24.102,27.065,28.048,27.084,24.121,20.916,
     &18.653,17.456,16.294,15.949,16.207,16.145,
     &17.009,18.585,20.712,23.162,25.215,-999.000,
     &29.064,-999.000,33.060,37.606,41.124,43.840,
     &40.505,23.480,1.083,-13.982,-25.987,-29.976,
     &-33.535,-36.361,-37.310,-36.217,-33.827,-999.000,
     &-30.322,-27.284,-24.809,-23.402,-22.651
     &/
       data (fjrocvz(it, 6,2,10), it=1,59) /
     &-23.501,-22.636,-21.167,-19.779,-19.467,-19.901,
     &-18.984,-18.479,-18.586,-16.790,-14.606,-13.203,
     &-9.911,-2.554,4.599,10.282,15.458,21.249,
     &25.939,28.865,30.849,30.495,28.358,25.300,
     &23.152,21.800,20.207,18.634,17.927,17.052,
     &16.713,18.564,20.955,22.352,24.742,-999.000,
     &28.746,-999.000,32.737,40.195,46.600,48.550,
     &40.964,24.552,4.576,-11.500,-24.156,-31.098,
     &-34.719,-999.000,-36.689,-999.000,-34.842,-999.000,
     &-31.247,-27.539,-25.393,-24.506,-23.501
     &/
       data (fjrocvz(it, 7,2,10), it=1,59) /
     &-25.651,-25.418,-24.459,-23.606,-23.181,-22.261,
     &-20.866,-20.086,-19.373,-18.222,-16.021,-13.588,
     &-8.288,-3.376,1.505,5.889,11.755,18.196,
     &23.712,28.035,30.263,30.642,29.548,27.395,
     &25.427,23.357,21.176,19.409,17.554,15.409,
     &14.746,15.659,17.192,19.595,21.761,-999.000,
     &24.632,-999.000,29.472,38.140,45.821,49.819,
     &38.314,18.503,0.309,-12.419,-24.374,-33.157,
     &-38.503,-39.227,-38.443,-37.280,-35.641,-999.000,
     &-31.838,-29.316,-27.386,-26.026,-25.651
     &/
       data (fjrocvz(it, 8,2,10), it=1,59) /
     &-25.693,-24.651,-23.910,-23.619,-23.264,-23.338,
     &-22.987,-22.593,-21.840,-23.473,-25.202,-26.802,
     &-21.336,-11.999,-5.384,1.142,6.467,11.157,
     &17.480,22.471,25.825,26.409,25.865,24.709,
     &23.118,21.398,19.183,17.050,14.488,12.217,
     &11.223,11.548,13.019,15.098,16.893,-999.000,
     &20.810,-999.000,27.308,33.669,41.020,42.133,
     &28.951,13.910,-0.047,-12.732,-22.213,-29.702,
     &-34.478,-34.680,-32.277,-999.000,-31.690,-999.000,
     &-30.059,-29.510,-28.170,-26.808,-25.693
     &/
       data (fjrocvz(it, 9,2,10), it=1,59) /
     &-23.049,-21.138,-19.399,-18.360,-18.486,-19.620,
     &-21.434,-23.031,-24.341,-25.870,-28.355,-30.173,
     &-24.635,-13.500,-4.946,0.893,4.307,8.645,
     &13.915,17.975,20.555,21.301,21.149,19.817,
     &17.591,15.562,13.700,11.752,9.666,8.299,
     &8.176,8.758,9.882,11.691,13.844,-999.000,
     &17.222,-999.000,24.749,32.122,38.873,42.717,
     &37.767,23.573,9.109,-3.824,-13.304,-999.000,
     &-19.902,-999.000,-23.204,-999.000,-25.547,-999.000,
     &-25.654,-25.571,-25.604,-24.899,-23.049
     &/
       data (fjrocvz(it,10,2,10), it=1,59) /
     &-24.205,-21.772,-19.391,-18.089,-17.267,-18.406,
     &-21.507,-24.166,-25.616,-26.269,-26.841,-26.517,
     &-22.161,-10.704,-2.326,3.301,6.306,10.906,
     &16.304,19.876,21.024,20.894,19.777,18.258,
     &16.382,13.885,11.597,9.709,8.532,7.936,
     &7.988,8.320,8.335,9.375,11.606,-999.000,
     &16.170,-999.000,24.727,33.179,42.443,46.575,
     &44.532,30.584,16.707,-999.000,-0.887,-999.000,
     &-11.544,-999.000,-19.095,-999.000,-21.111,-999.000,
     &-22.913,-23.705,-24.538,-24.733,-24.205
     &/
       data (fjrocvz(it,11,2,10), it=1,59) /
     &-27.238,-26.109,-25.497,-25.251,-24.764,-24.826,
     &-25.399,-26.691,-28.076,-29.921,-29.169,-25.314,
     &-19.678,-9.829,-2.189,3.927,8.184,12.283,
     &17.020,21.057,23.278,23.765,22.990,21.696,
     &20.567,18.445,15.637,13.663,12.366,11.393,
     &10.742,9.992,9.448,10.479,13.306,-999.000,
     &18.365,-999.000,25.461,33.332,42.876,44.709,
     &41.221,30.112,18.898,-999.000,-0.811,-999.000,
     &-13.024,-999.000,-21.297,-999.000,-23.169,-999.000,
     &-26.220,-25.974,-26.770,-27.878,-27.238
     &/
       data (fjrocvz(it,12,2,10), it=1,59) /
     &-28.632,-28.925,-28.616,-28.313,-28.126,-27.352,
     &-27.090,-28.552,-31.009,-33.224,-33.998,-30.748,
     &-26.576,-17.219,-7.628,-1.876,2.479,8.761,
     &16.040,21.590,24.915,25.708,25.102,24.363,
     &23.726,21.728,18.893,16.675,15.860,15.144,
     &14.225,12.978,12.092,13.202,16.527,-999.000,
     &21.045,-999.000,27.652,34.389,42.468,48.096,
     &44.291,26.957,12.096,1.025,-8.956,-999.000,
     &-15.440,-999.000,-21.391,-999.000,-25.617,-999.000,
     &-28.356,-28.276,-28.485,-28.712,-28.632
     &/
       data (fjrocvz(it,13,2,10), it=1,59) /
     &-26.550,-25.794,-25.431,-25.284,-24.708,-25.116,
     &-26.541,-27.740,-28.921,-29.790,-30.305,-28.915,
     &-26.227,-18.218,-9.726,-3.394,0.412,7.584,
     &17.513,23.978,26.544,27.232,26.742,26.200,
     &25.534,23.401,20.785,19.126,19.245,19.222,
     &18.606,17.418,16.237,16.958,19.715,-999.000,
     &23.386,-999.000,32.154,38.408,46.885,52.900,
     &46.227,28.356,8.290,-4.561,-12.988,-15.140,
     &-16.319,-999.000,-20.125,-999.000,-22.997,-999.000,
     &-25.251,-27.190,-27.029,-26.523,-26.550
     &/
       data (fjrocvz(it,14,2,10), it=1,59) /
     &-24.085,-23.506,-23.096,-22.446,-22.284,-23.868,
     &-26.623,-30.169,-33.304,-33.293,-30.503,-26.607,
     &-22.948,-19.632,-12.793,-5.082,0.373,8.005,
     &17.914,24.141,27.160,27.840,27.139,26.205,
     &24.929,23.426,21.559,20.506,20.965,21.170,
     &20.402,19.370,18.520,19.140,21.938,-999.000,
     &27.026,-999.000,35.902,45.862,52.386,51.781,
     &43.296,27.714,10.570,-999.000,-11.860,-999.000,
     &-19.691,-999.000,-21.689,-999.000,-24.055,-999.000,
     &-26.489,-27.228,-26.234,-24.827,-24.085
     &/
       data (fjrocvz(it,15,2,10), it=1,59) /
     &-21.573,-21.094,-20.552,-20.110,-20.157,-21.965,
     &-25.989,-32.116,-36.454,-37.838,-34.472,-27.322,
     &-22.556,-17.263,-11.693,-4.033,2.813,9.504,
     &17.317,22.823,25.567,25.777,24.317,22.102,
     &20.207,19.014,17.327,16.309,16.479,16.610,
     &16.189,15.660,16.070,17.649,21.247,-999.000,
     &25.998,-999.000,33.784,42.991,50.030,52.143,
     &44.542,27.120,10.198,-999.000,-13.528,-999.000,
     &-20.950,-999.000,-24.559,-999.000,-27.393,-999.000,
     &-28.785,-28.909,-26.364,-23.087,-21.573
     &/
       data (fjrocvz(it,16,2,10), it=1,59) /
     &-21.166,-20.160,-20.421,-20.734,-21.645,-24.101,
     &-28.001,-32.867,-36.823,-39.643,-38.534,-31.000,
     &-21.480,-15.426,-9.214,-2.321,4.566,10.214,
     &17.638,22.743,24.763,25.181,23.989,21.014,
     &17.595,14.762,12.478,11.237,11.416,12.039,
     &11.740,11.136,11.548,12.934,15.876,-999.000,
     &20.754,-999.000,28.279,37.049,44.855,46.307,
     &37.496,19.127,4.994,-999.000,-13.977,-999.000,
     &-22.461,-999.000,-27.214,-999.000,-30.065,-999.000,
     &-31.316,-30.701,-28.038,-24.235,-21.166
     &/
       data (fjrocvz(it,17,2,10), it=1,59) /
     &-24.679,-24.303,-23.799,-23.547,-24.588,-26.773,
     &-30.146,-34.968,-39.248,-40.634,-38.817,-30.625,
     &-21.513,-14.472,-7.279,-1.227,5.122,11.091,
     &19.022,24.470,27.109,28.167,27.274,24.677,
     &20.768,17.055,13.532,11.087,10.893,11.040,
     &10.349,9.303,8.536,8.767,11.588,-999.000,
     &17.415,-999.000,25.841,32.626,38.058,40.504,
     &33.485,16.650,1.982,-12.207,-21.559,-25.650,
     &-27.802,-999.000,-29.410,-999.000,-31.606,-999.000,
     &-32.393,-31.134,-29.421,-26.894,-24.679
     &/
       data (fjrocvz(it,18,2,10), it=1,59) /
     &-26.653,-25.820,-24.877,-24.794,-25.136,-26.520,
     &-30.433,-35.325,-38.731,-39.969,-37.306,-29.898,
     &-20.668,-13.534,-7.879,-1.652,5.301,13.146,
     &20.645,26.706,31.077,32.267,30.303,26.649,
     &23.358,20.249,16.343,12.919,11.713,10.797,
     &9.777,8.431,7.636,7.654,10.055,-999.000,
     &15.576,19.643,25.099,31.943,36.960,34.730,
     &27.744,15.161,0.611,-999.000,-21.078,-999.000,
     &-28.727,-999.000,-32.117,-999.000,-33.360,-999.000,
     &-33.142,-32.755,-31.305,-28.826,-26.653
     &/
       data (fjrocvz(it,19,2,10), it=1,59) /
     &-25.452,-24.399,-24.509,-24.210,-24.115,-25.459,
     &-28.456,-31.451,-34.370,-35.728,-32.988,-26.077,
     &-17.820,-12.508,-6.944,-0.695,5.927,14.594,
     &23.252,30.216,35.181,36.146,33.831,30.140,
     &26.419,22.845,18.928,15.472,14.728,14.215,
     &12.852,11.176,10.183,9.877,11.870,-999.000,
     &18.286,-999.000,27.423,35.064,38.593,36.182,
     &30.480,19.005,-0.245,-13.741,-23.896,-28.098,
     &-29.678,-999.000,-31.839,-999.000,-32.301,-999.000,
     &-32.330,-32.110,-30.997,-28.299,-25.452
     &/
       data (fjrocvz(it,20,2,10), it=1,59) /
     &-26.710,-25.643,-24.158,-22.616,-21.798,-22.584,
     &-24.991,-28.364,-30.676,-31.283,-28.714,-19.650,
     &-11.556,-8.835,-3.951,0.838,6.251,15.047,
     &23.400,31.200,36.379,37.716,36.069,32.901,
     &29.642,26.691,22.706,19.409,18.542,17.539,
     &15.948,14.238,13.074,12.820,15.342,-999.000,
     &21.464,-999.000,29.111,36.476,42.735,45.402,
     &35.964,19.514,-0.536,-10.834,-18.937,-25.180,
     &-29.260,-999.000,-31.094,-999.000,-32.356,-999.000,
     &-32.337,-32.040,-31.255,-28.996,-26.710
     &/
       data (fjrocvz(it,21,2,10), it=1,59) /
     &-26.330,-24.122,-22.322,-21.199,-20.619,-20.884,
     &-22.273,-25.236,-28.578,-29.297,-26.509,-19.257,
     &-13.858,-8.490,-3.170,2.308,8.324,15.161,
     &21.346,28.326,32.714,34.662,33.181,29.936,
     &26.989,24.846,21.589,17.936,16.504,15.348,
     &13.911,12.993,13.006,13.327,15.416,-999.000,
     &20.549,-999.000,29.423,35.929,42.447,41.031,
     &31.845,-999.000,3.442,-999.000,-14.974,-999.000,
     &-25.393,-999.000,-29.950,-999.000,-31.281,-999.000,
     &-31.600,-31.032,-30.210,-28.538,-26.330
     &/
       data (fjrocvz(it,22,2,10), it=1,59) /
     &-25.158,-24.005,-22.822,-21.949,-21.792,-21.919,
     &-22.325,-24.714,-28.841,-29.625,-26.583,-19.671,
     &-11.495,-8.098,-2.424,2.649,7.911,14.032,
     &20.606,26.661,29.754,30.570,28.789,25.135,
     &21.050,17.966,14.862,12.208,11.224,10.514,
     &9.969,10.615,12.015,12.911,15.270,-999.000,
     &20.698,-999.000,30.353,33.642,37.616,33.305,
     &24.788,-999.000,0.591,-999.000,-16.165,-999.000,
     &-26.535,-999.000,-28.939,-999.000,-29.421,-999.000,
     &-30.180,-29.661,-28.777,-26.982,-25.158
     &/
       data (fjrocvz(it,23,2,10), it=1,59) /
     &-23.875,-23.305,-21.743,-19.726,-19.026,-19.936,
     &-21.786,-24.121,-26.274,-26.629,-23.877,-17.549,
     &-12.056,-9.040,-3.559,1.759,7.137,12.861,
     &21.043,26.867,28.895,28.440,26.065,22.256,
     &18.169,14.526,11.310,9.145,8.843,8.753,
     &8.636,8.854,10.761,13.068,16.273,-999.000,
     &20.377,-999.000,29.270,34.241,37.491,33.982,
     &25.937,11.923,-0.534,-999.000,-16.016,-999.000,
     &-23.440,-999.000,-26.691,-999.000,-27.821,-999.000,
     &-28.862,-28.577,-27.419,-25.034,-23.875
     &/
       data (fjrocvz(it,24,2,10), it=1,59) /
     &-22.830,-21.232,-19.428,-17.280,-16.992,-18.759,
     &-20.895,-22.175,-23.236,-23.712,-21.521,-16.580,
     &-10.336,-5.113,-1.516,1.900,6.819,15.285,
     &23.581,27.313,28.278,27.656,24.810,20.671,
     &16.782,13.488,9.877,7.573,8.326,9.815,
     &10.346,10.248,11.898,14.426,17.507,-999.000,
     &21.111,-999.000,28.948,36.401,39.104,35.936,
     &24.276,-999.000,1.692,-999.000,-12.137,-999.000,
     &-20.086,-999.000,-25.571,-999.000,-26.359,-999.000,
     &-27.414,-27.499,-26.551,-24.733,-22.830
     &/
       data (fjrocvz(it,25,2,10), it=1,59) /
     &-22.847,-21.945,-20.408,-19.319,-20.078,-21.119,
     &-21.837,-22.833,-23.643,-22.971,-19.679,-12.909,
     &-14.211,-8.844,-2.947,3.637,10.210,19.481,
     &26.700,29.714,29.799,28.961,26.713,22.788,
     &19.061,16.505,11.844,8.576,9.393,11.371,
     &12.281,12.856,13.908,16.232,19.510,-999.000,
     &23.578,-999.000,29.193,35.140,36.622,31.194,
     &21.714,-999.000,1.547,-999.000,-13.699,-999.000,
     &-21.464,-999.000,-24.434,-999.000,-27.338,-999.000,
     &-27.215,-26.988,-26.589,-24.855,-22.847
     &/
       data (fjrocvz(it, 1,3,10), it=1,59) /
     &-19.729,-19.777,-18.965,-17.452,-14.890,-14.323,
     &-14.566,-18.123,-25.572,-27.509,-20.658,-7.029,
     &4.268,7.946,10.428,12.266,11.726,11.309,
     &11.931,14.807,18.699,19.747,19.239,16.884,
     &13.669,11.705,12.072,12.894,12.496,11.453,
     &10.667,10.509,10.160,9.957,10.709,-999.000,
     &15.648,-999.000,19.106,-999.000,24.842,29.283,
     &29.849,23.839,12.290,-999.000,-3.778,-999.000,
     &-10.519,-999.000,-12.614,-999.000,-13.846,-999.000,
     &-15.508,-18.061,-19.773,-19.498,-19.729
     &/
       data (fjrocvz(it, 2,3,10), it=1,59) /
     &-18.552,-18.475,-17.784,-16.116,-14.464,-15.431,
     &-17.293,-21.182,-26.873,-27.967,-19.846,-4.256,
     &9.027,12.495,13.793,14.800,14.960,15.520,
     &16.741,18.708,20.903,21.682,20.891,19.034,
     &17.563,16.245,16.472,17.045,17.164,16.447,
     &15.672,15.128,14.495,15.326,15.939,-999.000,
     &18.563,-999.000,22.261,-999.000,26.379,29.275,
     &28.356,22.982,8.849,-999.000,-5.445,-999.000,
     &-12.111,-999.000,-14.350,-999.000,-15.850,-999.000,
     &-17.257,-19.624,-20.406,-19.097,-18.552
     &/
       data (fjrocvz(it, 3,3,10), it=1,59) /
     &-18.032,-16.896,-16.266,-15.274,-14.735,-16.255,
     &-18.344,-21.071,-25.985,-29.607,-22.482,-3.452,
     &13.878,16.489,15.347,14.453,14.102,15.080,
     &16.509,17.755,18.345,17.836,16.671,16.197,
     &16.585,15.940,16.599,17.571,18.480,19.229,
     &18.967,18.243,17.573,18.589,19.479,-999.000,
     &22.220,-999.000,23.708,-999.000,26.578,26.250,
     &23.506,19.149,9.134,-999.000,-6.359,-999.000,
     &-13.538,-999.000,-16.885,-999.000,-19.223,-999.000,
     &-21.351,-23.530,-23.835,-20.914,-18.032
     &/
       data (fjrocvz(it, 4,3,10), it=1,59) /
     &-19.366,-17.151,-16.372,-16.257,-15.416,-15.990,
     &-16.366,-17.836,-22.400,-25.311,-18.104,1.703,
     &16.155,18.981,18.005,17.500,16.922,15.968,
     &14.888,14.316,14.501,14.350,14.410,15.067,
     &15.685,15.742,16.292,17.466,18.839,19.401,
     &19.714,19.706,18.963,20.221,22.994,-999.000,
     &24.222,-999.000,24.805,-999.000,29.252,28.923,
     &25.694,15.305,5.881,-999.000,-6.205,-999.000,
     &-14.292,-999.000,-19.325,-999.000,-22.467,-999.000,
     &-24.210,-25.482,-25.448,-22.567,-19.366
     &/
       data (fjrocvz(it, 5,3,10), it=1,59) /
     &-20.046,-18.633,-17.788,-15.869,-13.537,-13.882,
     &-14.305,-16.072,-18.986,-19.793,-13.972,0.242,
     &14.501,19.889,20.025,19.681,18.935,17.894,
     &16.624,15.981,16.850,17.517,17.676,17.132,
     &16.589,15.654,15.240,15.805,16.633,17.507,
     &18.561,19.396,19.555,21.307,23.207,-999.000,
     &25.077,-999.000,27.670,32.111,35.740,29.963,
     &22.983,12.968,3.159,-999.000,-8.421,-999.000,
     &-15.467,-999.000,-20.477,-999.000,-23.388,-999.000,
     &-25.524,-26.711,-25.834,-22.708,-20.046
     &/
       data (fjrocvz(it, 6,3,10), it=1,59) /
     &-21.375,-21.266,-20.999,-18.452,-14.371,-13.203,
     &-13.701,-15.562,-17.127,-16.305,-12.183,-2.067,
     &11.257,16.482,16.586,16.263,16.482,16.990,
     &17.052,18.563,20.791,21.656,21.056,20.019,
     &19.018,17.442,16.278,15.420,15.884,16.481,
     &17.177,17.735,18.503,19.700,20.815,-999.000,
     &24.825,-999.000,27.473,31.061,31.536,23.247,
     &13.815,-999.000,-1.751,-999.000,-11.276,-999.000,
     &-18.143,-999.000,-22.916,-999.000,-25.829,-999.000,
     &-28.115,-28.897,-26.941,-23.797,-21.375
     &/
       data (fjrocvz(it, 7,3,10), it=1,59) /
     &-24.123,-22.377,-21.881,-20.890,-17.852,-17.586,
     &-17.651,-17.634,-18.422,-16.001,-12.070,-5.105,
     &1.895,7.050,8.782,10.105,11.521,13.128,
     &14.818,17.271,20.431,22.310,22.885,22.910,
     &22.868,22.488,21.046,19.307,18.685,17.877,
     &17.490,16.932,16.575,16.630,17.008,-999.000,
     &18.699,-999.000,19.683,-999.000,17.974,-999.000,
     &6.578,-999.000,-3.129,-999.000,-11.914,-999.000,
     &-18.811,-999.000,-23.833,-999.000,-27.265,-999.000,
     &-29.679,-30.503,-29.375,-27.053,-24.123
     &/
       data (fjrocvz(it, 8,3,10), it=1,59) /
     &-22.116,-20.911,-21.961,-22.494,-21.345,-19.455,
     &-15.729,-13.662,-13.812,-12.702,-9.847,-6.097,
     &-3.790,-0.827,0.988,3.112,5.722,7.805,
     &9.652,11.850,15.604,19.566,21.718,22.497,
     &23.095,22.726,21.299,19.291,17.973,17.291,
     &16.718,15.850,14.648,12.912,11.380,-999.000,
     &11.665,-999.000,11.524,-999.000,10.147,-999.000,
     &5.472,-999.000,1.588,-999.000,-5.268,-999.000,
     &-12.775,-999.000,-18.950,-999.000,-23.937,-999.000,
     &-27.500,-28.524,-27.083,-24.581,-22.116
     &/
       data (fjrocvz(it, 9,3,10), it=1,59) /
     &-17.066,-17.413,-17.812,-18.409,-18.411,-16.956,
     &-13.770,-12.305,-13.226,-13.619,-12.759,-10.337,
     &-9.704,-4.671,0.411,5.251,10.543,11.769,
     &12.763,13.748,15.014,15.097,16.612,17.961,
     &19.234,19.611,18.053,15.659,14.513,13.385,
     &12.137,10.838,9.762,8.890,8.231,-999.000,
     &7.217,-999.000,6.151,-999.000,7.764,-999.000,
     &9.455,-999.000,9.551,-999.000,4.474,-999.000,
     &-1.230,-999.000,-7.292,-999.000,-11.615,-999.000,
     &-15.584,-17.183,-17.284,-17.149,-17.066
     &/
       data (fjrocvz(it,10,3,10), it=1,59) /
     &-14.847,-15.752,-15.156,-14.332,-15.158,-16.880,
     &-16.834,-16.026,-16.272,-15.912,-15.488,-16.349,
     &-17.832,-8.990,0.135,6.537,13.517,14.575,
     &15.536,16.258,17.159,17.885,18.019,18.168,
     &18.594,19.048,17.651,15.722,14.028,12.038,
     &9.930,7.523,6.014,4.937,3.635,-999.000,
     &2.394,-999.000,2.996,-999.000,5.525,-999.000,
     &10.134,-999.000,13.017,-999.000,12.206,-999.000,
     &7.617,-999.000,0.286,-999.000,-6.641,-999.000,
     &-9.920,-12.077,-13.227,-14.111,-14.847
     &/
       data (fjrocvz(it,11,3,10), it=1,59) /
     &-18.362,-20.245,-20.629,-20.060,-20.027,-19.760,
     &-18.373,-18.248,-21.749,-20.328,-19.914,-13.770,
     &-16.831,-10.022,-3.455,3.476,10.931,13.354,
     &15.490,17.215,20.558,23.189,22.809,22.277,
     &22.273,21.618,20.603,19.244,17.008,14.426,
     &11.469,8.890,7.435,5.200,3.019,-999.000,
     &1.845,-999.000,2.048,-999.000,5.061,-999.000,
     &13.402,17.042,17.739,15.374,13.670,-999.000,
     &9.266,-999.000,3.130,-999.000,-4.859,-999.000,
     &-9.501,-13.398,-15.567,-17.012,-18.362
     &/
       data (fjrocvz(it,12,3,10), it=1,59) /
     &-18.840,-21.461,-24.049,-25.998,-26.176,-24.216,
     &-23.918,-28.058,-33.293,-30.955,-25.252,-15.567,
     &-14.416,-10.255,-6.401,0.066,7.054,11.029,
     &14.437,18.755,23.321,25.558,25.753,25.322,
     &24.847,23.432,21.918,20.805,18.640,15.901,
     &13.176,11.700,10.487,8.822,6.808,-999.000,
     &4.998,-999.000,6.300,-999.000,13.631,19.270,
     &23.830,25.593,23.712,-999.000,18.347,-999.000,
     &10.635,-999.000,2.721,-999.000,-2.514,-999.000,
     &-8.448,-12.289,-15.401,-17.378,-18.840
     &/
       data (fjrocvz(it,13,3,10), it=1,59) /
     &-17.375,-19.373,-20.289,-21.453,-23.744,-25.473,
     &-27.915,-30.710,-33.050,-31.269,-24.172,-16.404,
     &-12.546,-10.607,-7.160,-3.303,0.685,7.483,
     &15.769,21.750,25.750,28.023,28.283,27.106,
     &25.964,24.420,22.754,21.077,18.733,16.657,
     &14.761,12.667,11.209,10.194,8.913,-999.000,
     &8.836,-999.000,11.312,-999.000,17.176,23.127,
     &30.174,32.978,27.639,-999.000,20.249,-999.000,
     &11.403,-999.000,2.114,-999.000,-3.606,-999.000,
     &-7.864,-9.852,-12.037,-15.101,-17.375
     &/
       data (fjrocvz(it,14,3,10), it=1,59) /
     &-18.072,-19.778,-19.702,-20.567,-23.064,-26.686,
     &-29.008,-31.187,-34.175,-33.015,-24.491,-13.562,
     &-8.638,-7.369,-5.522,-2.539,0.659,6.548,
     &15.710,21.432,25.686,27.944,28.078,26.309,
     &24.867,23.675,22.610,21.222,19.529,18.101,
     &16.158,13.942,12.653,11.394,10.889,-999.000,
     &11.327,-999.000,13.781,-999.000,19.504,24.277,
     &31.259,34.605,32.425,24.799,19.393,-999.000,
     &9.704,-999.000,2.109,-999.000,-3.194,-999.000,
     &-6.961,-9.188,-11.594,-15.229,-18.072
     &/
       data (fjrocvz(it,15,3,10), it=1,59) /
     &-18.384,-19.074,-19.182,-19.430,-20.791,-24.054,
     &-26.840,-29.886,-33.255,-34.254,-26.155,-14.609,
     &-7.892,-6.909,-6.109,-3.729,-0.225,5.355,
     &13.053,18.575,22.335,24.436,23.800,21.662,
     &19.770,18.254,17.383,16.077,14.227,13.262,
     &12.240,11.156,9.759,8.751,8.449,-999.000,
     &8.599,-999.000,10.935,-999.000,15.920,20.451,
     &24.769,26.312,22.596,17.064,12.607,-999.000,
     &4.249,-999.000,-1.674,-999.000,-5.208,-999.000,
     &-9.577,-11.600,-13.590,-16.004,-18.384
     &/
       data (fjrocvz(it,16,3,10), it=1,59) /
     &-16.189,-16.157,-15.677,-15.679,-17.159,-21.135,
     &-25.404,-28.624,-33.345,-36.290,-32.059,-19.482,
     &-9.337,-6.724,-6.098,-4.084,-0.787,4.463,
     &11.260,16.439,19.690,20.935,19.615,17.921,
     &16.291,14.460,12.556,10.001,7.476,6.221,
     &5.678,4.854,3.795,2.317,1.773,-999.000,
     &3.080,-999.000,5.734,-999.000,10.676,14.470,
     &16.223,14.600,11.359,-999.000,2.288,-999.000,
     &-4.523,-999.000,-8.652,-999.000,-10.051,-999.000,
     &-11.372,-12.880,-14.107,-14.994,-16.189
     &/
       data (fjrocvz(it,17,3,10), it=1,59) /
     &-17.108,-16.829,-15.799,-15.370,-16.441,-19.535,
     &-23.941,-30.166,-37.049,-39.852,-35.024,-19.780,
     &-7.865,-5.824,-4.294,-1.501,1.703,6.447,
     &12.633,18.007,21.254,22.580,21.727,19.554,
     &17.438,15.372,13.185,10.213,7.384,5.175,
     &3.095,1.960,1.008,-0.486,-1.466,-999.000,
     &-1.410,-999.000,0.796,-999.000,7.561,12.391,
     &12.329,8.936,4.607,-999.000,-3.750,-999.000,
     &-10.698,-999.000,-14.239,-999.000,-15.007,-999.000,
     &-15.373,-16.022,-16.287,-16.406,-17.108
     &/
       data (fjrocvz(it,18,3,10), it=1,59) /
     &-23.838,-22.067,-20.050,-18.272,-17.599,-19.273,
     &-22.247,-28.754,-37.500,-39.760,-33.349,-11.821,
     &-2.645,-2.001,-0.388,2.649,6.066,11.645,
     &17.947,22.902,25.292,26.627,26.256,24.561,
     &22.094,19.032,16.453,13.392,10.610,7.651,
     &5.024,3.259,1.729,0.282,-1.242,-999.000,
     &-1.477,-999.000,1.026,-999.000,9.227,10.775,
     &8.429,3.089,-1.996,-999.000,-10.262,-999.000,
     &-16.999,-999.000,-19.588,-999.000,-20.138,-999.000,
     &-20.493,-21.732,-22.678,-23.242,-23.838
     &/
       data (fjrocvz(it,19,3,10), it=1,59) /
     &-25.019,-24.145,-22.225,-20.378,-19.740,-21.439,
     &-22.397,-24.916,-31.202,-31.486,-25.561,-8.698,
     &0.390,1.922,3.443,6.180,10.332,15.823,
     &21.995,26.556,29.095,29.807,29.071,27.563,
     &25.374,22.645,19.581,16.578,13.742,10.835,
     &8.499,7.058,5.862,3.660,1.998,-999.000,
     &2.291,-999.000,3.551,-999.000,11.214,12.270,
     &10.029,4.565,-2.260,-999.000,-11.356,-999.000,
     &-17.745,-999.000,-19.432,-999.000,-20.584,-999.000,
     &-21.169,-22.367,-23.777,-24.630,-25.019
     &/
       data (fjrocvz(it,20,3,10), it=1,59) /
     &-21.591,-21.253,-19.778,-18.055,-17.932,-20.653,
     &-23.236,-25.662,-27.206,-24.251,-17.613,-7.605,
     &0.447,3.715,5.289,7.258,10.786,16.427,
     &22.879,28.736,31.526,31.676,30.159,27.826,
     &25.422,23.317,21.782,20.315,17.924,15.130,
     &12.347,10.465,8.338,6.590,5.813,-999.000,
     &5.551,-999.000,7.029,-999.000,13.014,15.415,
     &14.700,10.504,4.459,-999.000,-6.525,-999.000,
     &-12.319,-999.000,-13.717,-999.000,-15.736,-999.000,
     &-17.741,-19.811,-21.475,-21.385,-21.591
     &/
       data (fjrocvz(it,21,3,10), it=1,59) /
     &-18.449,-18.354,-17.035,-15.463,-15.372,-18.788,
     &-21.595,-23.291,-26.176,-28.167,-24.960,-14.459,
     &-2.730,1.885,3.870,5.572,8.085,12.543,
     &18.606,23.976,26.954,27.362,26.413,25.315,
     &23.761,22.215,21.275,19.796,17.678,15.013,
     &12.745,10.643,8.905,7.947,7.192,-999.000,
     &7.142,-999.000,7.319,-999.000,9.651,11.858,
     &14.516,12.678,8.218,-999.000,-1.150,-999.000,
     &-6.878,-999.000,-9.365,-999.000,-12.315,-999.000,
     &-15.033,-17.155,-18.715,-18.385,-18.449
     &/
       data (fjrocvz(it,22,3,10), it=1,59) /
     &-17.053,-15.270,-13.294,-12.605,-13.252,-16.221,
     &-18.648,-22.088,-27.902,-31.880,-29.419,-18.799,
     &-6.938,0.047,1.803,3.247,5.028,8.447,
     &13.001,17.227,20.643,21.020,20.250,19.961,
     &19.533,18.329,17.169,15.384,12.948,10.626,
     &8.780,7.638,6.575,6.655,7.698,-999.000,
     &8.188,-999.000,8.047,-999.000,11.524,12.763,
     &16.804,15.797,9.111,-999.000,0.413,-999.000,
     &-5.547,-999.000,-9.303,-999.000,-11.948,-999.000,
     &-14.348,-16.315,-17.976,-17.925,-17.053
     &/
       data (fjrocvz(it,23,3,10), it=1,59) /
     &-18.096,-15.590,-13.182,-11.628,-12.002,-15.464,
     &-18.921,-22.192,-25.736,-27.388,-23.498,-13.486,
     &-2.283,2.034,2.187,2.786,3.653,6.236,
     &9.508,14.060,18.637,20.134,19.067,17.232,
     &15.565,13.436,12.157,10.836,8.710,6.660,
     &6.004,5.554,4.531,4.874,6.349,-999.000,
     &7.829,-999.000,10.192,-999.000,15.562,19.827,
     &23.464,21.483,13.738,-999.000,1.264,-999.000,
     &-5.440,-999.000,-9.728,-999.000,-12.248,-999.000,
     &-14.867,-17.050,-18.525,-18.647,-18.096
     &/
       data (fjrocvz(it,24,3,10), it=1,59) /
     &-19.834,-18.587,-16.399,-13.603,-11.549,-13.162,
     &-15.307,-18.112,-22.823,-23.947,-19.526,-8.565,
     &1.968,5.134,6.177,7.020,6.831,8.068,
     &10.165,13.553,17.525,18.381,17.613,16.027,
     &13.088,10.349,9.080,9.297,8.457,7.625,
     &7.393,6.981,6.398,6.506,7.315,-999.000,
     &11.366,-999.000,15.404,-999.000,22.085,26.984,
     &28.025,24.885,17.147,-999.000,0.198,-999.000,
     &-7.136,-999.000,-10.464,-999.000,-12.708,-999.000,
     &-15.715,-18.373,-19.759,-19.789,-19.834
     &/
       data (fjrocvz(it,25,3,10), it=1,59) /
     &-19.729,-19.777,-18.965,-17.452,-14.890,-14.323,
     &-14.566,-18.123,-25.572,-27.509,-20.658,-7.029,
     &4.268,7.946,10.428,12.266,11.726,11.309,
     &11.931,14.807,18.699,19.747,19.239,16.884,
     &13.669,11.705,12.072,12.894,12.496,11.453,
     &10.667,10.509,10.160,9.957,10.709,-999.000,
     &15.648,-999.000,19.106,-999.000,24.842,29.283,
     &29.849,23.839,12.290,-999.000,-3.778,-999.000,
     &-10.519,-999.000,-12.614,-999.000,-13.846,-999.000,
     &-15.508,-18.061,-19.773,-19.498,-19.729
     &/
       data (fjrocvz(it, 1,4,10), it=1,59) /
     &-22.847,-21.945,-20.408,-19.319,-20.078,-21.119,
     &-21.837,-22.833,-23.643,-22.971,-19.679,-12.909,
     &-14.211,-8.844,-2.947,3.637,10.210,19.481,
     &26.700,29.714,29.799,28.961,26.713,22.788,
     &19.061,16.505,11.844,8.576,9.393,11.371,
     &12.281,12.856,13.908,16.232,19.510,-999.000,
     &23.578,-999.000,29.193,35.140,36.622,31.194,
     &21.714,-999.000,1.547,-999.000,-13.699,-999.000,
     &-21.464,-999.000,-24.434,-999.000,-27.338,-999.000,
     &-27.215,-26.988,-26.589,-24.855,-22.847
     &/
       data (fjrocvz(it, 2,4,10), it=1,59) /
     &-21.475,-20.721,-19.797,-19.334,-18.604,-18.401,
     &-18.809,-20.629,-22.301,-21.809,-16.541,-8.725,
     &-3.320,1.247,6.260,9.262,14.650,20.651,
     &26.416,30.481,32.001,31.593,29.068,25.051,
     &21.392,19.513,16.384,13.360,12.671,13.309,
     &14.740,15.853,17.261,19.684,22.129,-999.000,
     &24.646,-999.000,30.448,34.240,37.056,36.378,
     &30.300,-999.000,2.868,-999.000,-18.124,-999.000,
     &-23.186,-999.000,-28.794,-999.000,-30.761,-999.000,
     &-28.528,-27.520,-25.164,-22.714,-21.475
     &/
       data (fjrocvz(it, 3,4,10), it=1,59) /
     &-21.606,-20.391,-19.720,-18.582,-17.565,-18.221,
     &-18.078,-18.455,-19.565,-18.981,-14.524,-8.087,
     &-2.210,2.738,5.768,9.465,14.672,18.914,
     &22.797,25.959,28.258,28.246,25.687,22.302,
     &19.734,18.517,16.752,15.011,14.318,14.713,
     &16.029,17.302,19.268,20.704,23.049,-999.000,
     &26.666,-999.000,30.074,35.837,40.103,39.448,
     &30.949,15.458,2.185,-13.326,-23.280,-24.463,
     &-26.448,-999.000,-31.442,-999.000,-31.774,-999.000,
     &-30.201,-27.732,-24.873,-23.112,-21.606
     &/
       data (fjrocvz(it, 4,4,10), it=1,59) /
     &-22.702,-21.487,-19.864,-18.248,-17.804,-18.347,
     &-17.872,-16.706,-16.482,-15.690,-12.641,-7.160,
     &-1.150,2.498,4.268,7.445,12.195,17.542,
     &21.584,24.157,24.946,24.001,21.637,18.811,
     &16.878,15.749,14.288,13.558,14.338,14.972,
     &16.081,17.821,19.548,21.959,25.728,-999.000,
     &28.192,-999.000,32.115,38.082,40.163,40.245,
     &36.056,21.883,2.368,-17.373,-26.296,-999.000,
     &-28.769,-999.000,-32.417,-999.000,-33.643,-999.000,
     &-31.021,-27.760,-25.296,-23.654,-22.702
     &/
       data (fjrocvz(it, 5,4,10), it=1,59) /
     &-22.651,-21.252,-18.931,-17.227,-16.949,-18.186,
     &-18.298,-17.381,-17.342,-16.550,-13.588,-7.419,
     &-1.526,2.580,5.801,8.486,13.255,19.325,
     &24.102,27.065,28.048,27.084,24.121,20.916,
     &18.653,17.456,16.294,15.949,16.207,16.145,
     &17.009,18.585,20.712,23.162,25.215,-999.000,
     &29.064,-999.000,33.060,37.606,41.124,43.840,
     &40.505,23.480,1.083,-13.982,-25.987,-29.976,
     &-33.535,-36.361,-37.310,-36.217,-33.827,-999.000,
     &-30.322,-27.284,-24.809,-23.402,-22.651
     &/
       data (fjrocvz(it, 6,4,10), it=1,59) /
     &-23.501,-22.636,-21.167,-19.779,-19.467,-19.901,
     &-18.984,-18.479,-18.586,-16.790,-14.606,-13.203,
     &-9.911,-2.554,4.599,10.282,15.458,21.249,
     &25.939,28.865,30.849,30.495,28.358,25.300,
     &23.152,21.800,20.207,18.634,17.927,17.052,
     &16.713,18.564,20.955,22.352,24.742,-999.000,
     &28.746,-999.000,32.737,40.195,46.600,48.550,
     &40.964,24.552,4.576,-11.500,-24.156,-31.098,
     &-34.719,-999.000,-36.689,-999.000,-34.842,-999.000,
     &-31.247,-27.539,-25.393,-24.506,-23.501
     &/
       data (fjrocvz(it, 7,4,10), it=1,59) /
     &-25.651,-25.418,-24.459,-23.606,-23.181,-22.261,
     &-20.866,-20.086,-19.373,-18.222,-16.021,-13.588,
     &-8.288,-3.376,1.505,5.889,11.755,18.196,
     &23.712,28.035,30.263,30.642,29.548,27.395,
     &25.427,23.357,21.176,19.409,17.554,15.409,
     &14.746,15.659,17.192,19.595,21.761,-999.000,
     &24.632,-999.000,29.472,38.140,45.821,49.819,
     &38.314,18.503,0.309,-12.419,-24.374,-33.157,
     &-38.503,-39.227,-38.443,-37.280,-35.641,-999.000,
     &-31.838,-29.316,-27.386,-26.026,-25.651
     &/
       data (fjrocvz(it, 8,4,10), it=1,59) /
     &-25.693,-24.651,-23.910,-23.619,-23.264,-23.338,
     &-22.987,-22.593,-21.840,-23.473,-25.202,-26.802,
     &-21.336,-11.999,-5.384,1.142,6.467,11.157,
     &17.480,22.471,25.825,26.409,25.865,24.709,
     &23.118,21.398,19.183,17.050,14.488,12.217,
     &11.223,11.548,13.019,15.098,16.893,-999.000,
     &20.810,-999.000,27.308,33.669,41.020,42.133,
     &28.951,13.910,-0.047,-12.732,-22.213,-29.702,
     &-34.478,-34.680,-32.277,-999.000,-31.690,-999.000,
     &-30.059,-29.510,-28.170,-26.808,-25.693
     &/
       data (fjrocvz(it, 9,4,10), it=1,59) /
     &-23.049,-21.138,-19.399,-18.360,-18.486,-19.620,
     &-21.434,-23.031,-24.341,-25.870,-28.355,-30.173,
     &-24.635,-13.500,-4.946,0.893,4.307,8.645,
     &13.915,17.975,20.555,21.301,21.149,19.817,
     &17.591,15.562,13.700,11.752,9.666,8.299,
     &8.176,8.758,9.882,11.691,13.844,-999.000,
     &17.222,-999.000,24.749,32.122,38.873,42.717,
     &37.767,23.573,9.109,-3.824,-13.304,-999.000,
     &-19.902,-999.000,-23.204,-999.000,-25.547,-999.000,
     &-25.654,-25.571,-25.604,-24.899,-23.049
     &/
       data (fjrocvz(it,10,4,10), it=1,59) /
     &-24.205,-21.772,-19.391,-18.089,-17.267,-18.406,
     &-21.507,-24.166,-25.616,-26.269,-26.841,-26.517,
     &-22.161,-10.704,-2.326,3.301,6.306,10.906,
     &16.304,19.876,21.024,20.894,19.777,18.258,
     &16.382,13.885,11.597,9.709,8.532,7.936,
     &7.988,8.320,8.335,9.375,11.606,-999.000,
     &16.170,-999.000,24.727,33.179,42.443,46.575,
     &44.532,30.584,16.707,-999.000,-0.887,-999.000,
     &-11.544,-999.000,-19.095,-999.000,-21.111,-999.000,
     &-22.913,-23.705,-24.538,-24.733,-24.205
     &/
       data (fjrocvz(it,11,4,10), it=1,59) /
     &-27.238,-26.109,-25.497,-25.251,-24.764,-24.826,
     &-25.399,-26.691,-28.076,-29.921,-29.169,-25.314,
     &-19.678,-9.829,-2.189,3.927,8.184,12.283,
     &17.020,21.057,23.278,23.765,22.990,21.696,
     &20.567,18.445,15.637,13.663,12.366,11.393,
     &10.742,9.992,9.448,10.479,13.306,-999.000,
     &18.365,-999.000,25.461,33.332,42.876,44.709,
     &41.221,30.112,18.898,-999.000,-0.811,-999.000,
     &-13.024,-999.000,-21.297,-999.000,-23.169,-999.000,
     &-26.220,-25.974,-26.770,-27.878,-27.238
     &/
       data (fjrocvz(it,12,4,10), it=1,59) /
     &-28.632,-28.925,-28.616,-28.313,-28.126,-27.352,
     &-27.090,-28.552,-31.009,-33.224,-33.998,-30.748,
     &-26.576,-17.219,-7.628,-1.876,2.479,8.761,
     &16.040,21.590,24.915,25.708,25.102,24.363,
     &23.726,21.728,18.893,16.675,15.860,15.144,
     &14.225,12.978,12.092,13.202,16.527,-999.000,
     &21.045,-999.000,27.652,34.389,42.468,48.096,
     &44.291,26.957,12.096,1.025,-8.956,-999.000,
     &-15.440,-999.000,-21.391,-999.000,-25.617,-999.000,
     &-28.356,-28.276,-28.485,-28.712,-28.632
     &/
       data (fjrocvz(it,13,4,10), it=1,59) /
     &-26.550,-25.794,-25.431,-25.284,-24.708,-25.116,
     &-26.541,-27.740,-28.921,-29.790,-30.305,-28.915,
     &-26.227,-18.218,-9.726,-3.394,0.412,7.584,
     &17.513,23.978,26.544,27.232,26.742,26.200,
     &25.534,23.401,20.785,19.126,19.245,19.222,
     &18.606,17.418,16.237,16.958,19.715,-999.000,
     &23.386,-999.000,32.154,38.408,46.885,52.900,
     &46.227,28.356,8.290,-4.561,-12.988,-15.140,
     &-16.319,-999.000,-20.125,-999.000,-22.997,-999.000,
     &-25.251,-27.190,-27.029,-26.523,-26.550
     &/
       data (fjrocvz(it,14,4,10), it=1,59) /
     &-24.085,-23.506,-23.096,-22.446,-22.284,-23.868,
     &-26.623,-30.169,-33.304,-33.293,-30.503,-26.607,
     &-22.948,-19.632,-12.793,-5.082,0.373,8.005,
     &17.914,24.141,27.160,27.840,27.139,26.205,
     &24.929,23.426,21.559,20.506,20.965,21.170,
     &20.402,19.370,18.520,19.140,21.938,-999.000,
     &27.026,-999.000,35.902,45.862,52.386,51.781,
     &43.296,27.714,10.570,-999.000,-11.860,-999.000,
     &-19.691,-999.000,-21.689,-999.000,-24.055,-999.000,
     &-26.489,-27.228,-26.234,-24.827,-24.085
     &/
       data (fjrocvz(it,15,4,10), it=1,59) /
     &-21.573,-21.094,-20.552,-20.110,-20.157,-21.965,
     &-25.989,-32.116,-36.454,-37.838,-34.472,-27.322,
     &-22.556,-17.263,-11.693,-4.033,2.813,9.504,
     &17.317,22.823,25.567,25.777,24.317,22.102,
     &20.207,19.014,17.327,16.309,16.479,16.610,
     &16.189,15.660,16.070,17.649,21.247,-999.000,
     &25.998,-999.000,33.784,42.991,50.030,52.143,
     &44.542,27.120,10.198,-999.000,-13.528,-999.000,
     &-20.950,-999.000,-24.559,-999.000,-27.393,-999.000,
     &-28.785,-28.909,-26.364,-23.087,-21.573
     &/
       data (fjrocvz(it,16,4,10), it=1,59) /
     &-21.166,-20.160,-20.421,-20.734,-21.645,-24.101,
     &-28.001,-32.867,-36.823,-39.643,-38.534,-31.000,
     &-21.480,-15.426,-9.214,-2.321,4.566,10.214,
     &17.638,22.743,24.763,25.181,23.989,21.014,
     &17.595,14.762,12.478,11.237,11.416,12.039,
     &11.740,11.136,11.548,12.934,15.876,-999.000,
     &20.754,-999.000,28.279,37.049,44.855,46.307,
     &37.496,19.127,4.994,-999.000,-13.977,-999.000,
     &-22.461,-999.000,-27.214,-999.000,-30.065,-999.000,
     &-31.316,-30.701,-28.038,-24.235,-21.166
     &/
       data (fjrocvz(it,17,4,10), it=1,59) /
     &-24.679,-24.303,-23.799,-23.547,-24.588,-26.773,
     &-30.146,-34.968,-39.248,-40.634,-38.817,-30.625,
     &-21.513,-14.472,-7.279,-1.227,5.122,11.091,
     &19.022,24.470,27.109,28.167,27.274,24.677,
     &20.768,17.055,13.532,11.087,10.893,11.040,
     &10.349,9.303,8.536,8.767,11.588,-999.000,
     &17.415,-999.000,25.841,32.626,38.058,40.504,
     &33.485,16.650,1.982,-12.207,-21.559,-25.650,
     &-27.802,-999.000,-29.410,-999.000,-31.606,-999.000,
     &-32.393,-31.134,-29.421,-26.894,-24.679
     &/
       data (fjrocvz(it,18,4,10), it=1,59) /
     &-26.653,-25.820,-24.877,-24.794,-25.136,-26.520,
     &-30.433,-35.325,-38.731,-39.969,-37.306,-29.898,
     &-20.668,-13.534,-7.879,-1.652,5.301,13.146,
     &20.645,26.706,31.077,32.267,30.303,26.649,
     &23.358,20.249,16.343,12.919,11.713,10.797,
     &9.777,8.431,7.636,7.654,10.055,-999.000,
     &15.576,19.643,25.099,31.943,36.960,34.730,
     &27.744,15.161,0.611,-999.000,-21.078,-999.000,
     &-28.727,-999.000,-32.117,-999.000,-33.360,-999.000,
     &-33.142,-32.755,-31.305,-28.826,-26.653
     &/
       data (fjrocvz(it,19,4,10), it=1,59) /
     &-25.452,-24.399,-24.509,-24.210,-24.115,-25.459,
     &-28.456,-31.451,-34.370,-35.728,-32.988,-26.077,
     &-17.820,-12.508,-6.944,-0.695,5.927,14.594,
     &23.252,30.216,35.181,36.146,33.831,30.140,
     &26.419,22.845,18.928,15.472,14.728,14.215,
     &12.852,11.176,10.183,9.877,11.870,-999.000,
     &18.286,-999.000,27.423,35.064,38.593,36.182,
     &30.480,19.005,-0.245,-13.741,-23.896,-28.098,
     &-29.678,-999.000,-31.839,-999.000,-32.301,-999.000,
     &-32.330,-32.110,-30.997,-28.299,-25.452
     &/
       data (fjrocvz(it,20,4,10), it=1,59) /
     &-26.710,-25.643,-24.158,-22.616,-21.798,-22.584,
     &-24.991,-28.364,-30.676,-31.283,-28.714,-19.650,
     &-11.556,-8.835,-3.951,0.838,6.251,15.047,
     &23.400,31.200,36.379,37.716,36.069,32.901,
     &29.642,26.691,22.706,19.409,18.542,17.539,
     &15.948,14.238,13.074,12.820,15.342,-999.000,
     &21.464,-999.000,29.111,36.476,42.735,45.402,
     &35.964,19.514,-0.536,-10.834,-18.937,-25.180,
     &-29.260,-999.000,-31.094,-999.000,-32.356,-999.000,
     &-32.337,-32.040,-31.255,-28.996,-26.710
     &/
       data (fjrocvz(it,21,4,10), it=1,59) /
     &-26.330,-24.122,-22.322,-21.199,-20.619,-20.884,
     &-22.273,-25.236,-28.578,-29.297,-26.509,-19.257,
     &-13.858,-8.490,-3.170,2.308,8.324,15.161,
     &21.346,28.326,32.714,34.662,33.181,29.936,
     &26.989,24.846,21.589,17.936,16.504,15.348,
     &13.911,12.993,13.006,13.327,15.416,-999.000,
     &20.549,-999.000,29.423,35.929,42.447,41.031,
     &31.845,-999.000,3.442,-999.000,-14.974,-999.000,
     &-25.393,-999.000,-29.950,-999.000,-31.281,-999.000,
     &-31.600,-31.032,-30.210,-28.538,-26.330
     &/
       data (fjrocvz(it,22,4,10), it=1,59) /
     &-25.158,-24.005,-22.822,-21.949,-21.792,-21.919,
     &-22.325,-24.714,-28.841,-29.625,-26.583,-19.671,
     &-11.495,-8.098,-2.424,2.649,7.911,14.032,
     &20.606,26.661,29.754,30.570,28.789,25.135,
     &21.050,17.966,14.862,12.208,11.224,10.514,
     &9.969,10.615,12.015,12.911,15.270,-999.000,
     &20.698,-999.000,30.353,33.642,37.616,33.305,
     &24.788,-999.000,0.591,-999.000,-16.165,-999.000,
     &-26.535,-999.000,-28.939,-999.000,-29.421,-999.000,
     &-30.180,-29.661,-28.777,-26.982,-25.158
     &/
       data (fjrocvz(it,23,4,10), it=1,59) /
     &-23.875,-23.305,-21.743,-19.726,-19.026,-19.936,
     &-21.786,-24.121,-26.274,-26.629,-23.877,-17.549,
     &-12.056,-9.040,-3.559,1.759,7.137,12.861,
     &21.043,26.867,28.895,28.440,26.065,22.256,
     &18.169,14.526,11.310,9.145,8.843,8.753,
     &8.636,8.854,10.761,13.068,16.273,-999.000,
     &20.377,-999.000,29.270,34.241,37.491,33.982,
     &25.937,11.923,-0.534,-999.000,-16.016,-999.000,
     &-23.440,-999.000,-26.691,-999.000,-27.821,-999.000,
     &-28.862,-28.577,-27.419,-25.034,-23.875
     &/
       data (fjrocvz(it,24,4,10), it=1,59) /
     &-22.830,-21.232,-19.428,-17.280,-16.992,-18.759,
     &-20.895,-22.175,-23.236,-23.712,-21.521,-16.580,
     &-10.336,-5.113,-1.516,1.900,6.819,15.285,
     &23.581,27.313,28.278,27.656,24.810,20.671,
     &16.782,13.488,9.877,7.573,8.326,9.815,
     &10.346,10.248,11.898,14.426,17.507,-999.000,
     &21.111,-999.000,28.948,36.401,39.104,35.936,
     &24.276,-999.000,1.692,-999.000,-12.137,-999.000,
     &-20.086,-999.000,-25.571,-999.000,-26.359,-999.000,
     &-27.414,-27.499,-26.551,-24.733,-22.830
     &/
       data (fjrocvz(it,25,4,10), it=1,59) /
     &-22.847,-21.945,-20.408,-19.319,-20.078,-21.119,
     &-21.837,-22.833,-23.643,-22.971,-19.679,-12.909,
     &-14.211,-8.844,-2.947,3.637,10.210,19.481,
     &26.700,29.714,29.799,28.961,26.713,22.788,
     &19.061,16.505,11.844,8.576,9.393,11.371,
     &12.281,12.856,13.908,16.232,19.510,-999.000,
     &23.578,-999.000,29.193,35.140,36.622,31.194,
     &21.714,-999.000,1.547,-999.000,-13.699,-999.000,
     &-21.464,-999.000,-24.434,-999.000,-27.338,-999.000,
     &-27.215,-26.988,-26.589,-24.855,-22.847
     &/
       data (fjrocvz(it, 1,1,11), it=1,59) /
     &-14.339,-13.417,-12.916,-12.892,-12.955,-14.940,
     &-17.052,-19.301,-22.381,-23.793,-22.616,-16.287,
     &-6.517,1.445,7.504,12.066,16.748,22.646,
     &27.330,30.486,31.449,29.534,27.148,24.119,
     &20.906,17.703,14.138,11.529,9.074,6.452,
     &4.812,4.611,5.382,5.204,5.167,-999.000,
     &4.522,-999.000,5.289,-999.000,8.184,-999.000,
     &6.416,-999.000,1.886,-999.000,-6.427,-999.000,
     &-11.957,-999.000,-15.410,-999.000,-17.751,-999.000,
     &-20.027,-20.270,-18.871,-16.248,-14.339
     &/
       data (fjrocvz(it, 2,1,11), it=1,59) /
     &-17.460,-15.997,-15.467,-15.232,-15.892,-18.220,
     &-19.279,-20.752,-22.846,-23.397,-19.556,-9.804,
     &-0.503,6.183,11.716,17.605,22.635,27.787,
     &32.266,35.336,36.648,34.595,31.318,27.580,
     &23.337,19.392,15.110,11.799,9.143,6.866,
     &4.432,3.195,3.399,3.772,4.097,-999.000,
     &4.662,-999.000,6.615,-999.000,7.640,-999.000,
     &5.477,-999.000,-0.028,-999.000,-8.791,-999.000,
     &-15.074,-999.000,-19.407,-999.000,-21.849,-999.000,
     &-23.298,-22.762,-21.334,-19.433,-17.460
     &/
       data (fjrocvz(it, 3,1,11), it=1,59) /
     &-20.198,-19.316,-18.983,-19.166,-19.742,-20.032,
     &-19.359,-19.626,-19.810,-18.081,-11.966,-3.057,
     &4.933,10.516,15.929,21.414,25.366,29.310,
     &33.311,36.584,37.478,34.789,31.003,27.327,
     &23.850,20.850,17.162,13.369,10.154,7.691,
     &5.074,4.002,4.271,5.130,5.654,-999.000,
     &6.898,-999.000,8.140,-999.000,9.798,11.515,
     &11.812,9.080,4.288,-0.316,-6.953,-999.000,
     &-16.401,-999.000,-20.227,-999.000,-22.242,-999.000,
     &-23.274,-22.870,-21.745,-20.995,-20.198
     &/
       data (fjrocvz(it, 4,1,11), it=1,59) /
     &-20.716,-21.289,-21.145,-20.227,-19.128,-19.641,
     &-19.864,-20.340,-19.630,-15.641,-7.989,0.722,
     &6.860,11.690,17.474,22.463,27.073,30.713,
     &34.409,36.646,36.125,32.500,28.547,25.487,
     &22.571,19.055,15.292,12.542,10.770,9.527,
     &7.256,5.537,5.473,6.431,8.570,-999.000,
     &10.048,-999.000,13.279,-999.000,18.805,18.904,
     &18.659,13.809,7.293,-999.000,-4.674,-999.000,
     &-17.257,-999.000,-21.071,-999.000,-23.375,-999.000,
     &-23.279,-22.119,-20.019,-19.470,-20.716
     &/
       data (fjrocvz(it, 5,1,11), it=1,59) /
     &-17.560,-18.201,-18.233,-17.222,-16.601,-18.329,
     &-20.528,-22.032,-21.796,-17.514,-9.285,0.245,
     &6.043,10.869,16.834,22.219,26.947,30.101,
     &33.411,35.482,35.016,31.957,28.874,26.071,
     &22.630,18.610,14.857,12.851,12.174,12.099,
     &10.660,8.553,8.006,8.212,9.751,-999.000,
     &13.314,-999.000,19.071,-999.000,22.957,27.731,
     &26.699,20.453,12.597,-999.000,-3.430,-999.000,
     &-17.694,-999.000,-22.055,-999.000,-22.939,-999.000,
     &-22.929,-20.779,-17.656,-16.718,-17.560
     &/
       data (fjrocvz(it, 6,1,11), it=1,59) /
     &-15.068,-15.851,-16.594,-17.377,-17.874,-19.586,
     &-21.899,-23.603,-23.703,-18.363,-10.975,-2.289,
     &3.683,7.628,11.995,16.546,21.106,24.546,
     &28.669,31.962,32.049,30.139,27.943,25.549,
     &22.364,19.477,16.652,14.477,13.678,13.998,
     &13.750,12.770,11.290,10.492,11.719,-999.000,
     &15.640,-999.000,20.504,-999.000,26.236,-999.000,
     &26.821,-999.000,17.054,-999.000,2.487,-999.000,
     &-10.116,-999.000,-16.440,-999.000,-19.835,-999.000,
     &-20.661,-19.626,-16.668,-15.211,-15.068
     &/
       data (fjrocvz(it, 7,1,11), it=1,59) /
     &-16.883,-17.980,-19.198,-20.313,-21.315,-22.707,
     &-23.541,-24.010,-24.221,-19.902,-15.092,-10.375,
     &-4.588,1.626,7.410,11.923,17.009,20.098,
     &24.908,28.184,28.615,26.910,24.516,22.219,
     &19.877,18.095,15.456,13.166,12.108,12.889,
     &13.983,13.933,13.021,12.884,14.400,-999.000,
     &17.560,-999.000,22.990,-999.000,28.792,31.566,
     &35.911,33.961,23.772,-999.000,8.357,-999.000,
     &-3.263,-999.000,-11.238,-999.000,-16.412,-999.000,
     &-19.024,-19.124,-17.253,-17.088,-16.883
     &/
       data (fjrocvz(it, 8,1,11), it=1,59) /
     &-19.833,-19.773,-19.798,-19.939,-20.523,-21.698,
     &-22.763,-24.012,-25.615,-23.254,-19.617,-13.512,
     &-6.506,-1.887,1.699,5.144,9.722,13.577,
     &18.862,22.857,24.331,23.233,20.928,18.562,
     &16.042,14.272,12.156,10.022,9.265,10.267,
     &12.081,13.133,13.596,13.964,15.609,-999.000,
     &19.816,-999.000,26.424,-999.000,34.515,42.618,
     &46.798,45.056,34.425,21.668,13.526,-999.000,
     &-3.657,-999.000,-12.555,-999.000,-20.286,-999.000,
     &-21.486,-22.143,-21.021,-20.553,-19.833
     &/
       data (fjrocvz(it, 9,1,11), it=1,59) /
     &-20.528,-19.751,-19.097,-18.705,-18.741,-20.257,
     &-22.054,-23.928,-25.801,-24.424,-21.644,-16.942,
     &-11.482,-6.420,-1.344,2.442,5.823,8.970,
     &13.817,18.095,20.041,20.116,19.054,17.477,
     &15.224,14.006,12.772,11.584,10.803,11.262,
     &12.187,12.757,13.373,14.712,17.516,18.950,
     &20.701,23.589,29.543,36.349,43.685,52.248,
     &58.272,51.820,32.076,14.893,3.070,-6.229,
     &-10.214,-999.000,-17.400,-999.000,-21.224,-999.000,
     &-23.977,-24.551,-23.274,-21.962,-20.528
     &/
       data (fjrocvz(it,10,1,11), it=1,59) /
     &-19.304,-18.185,-17.366,-16.828,-17.376,-19.567,
     &-21.239,-22.552,-24.668,-25.256,-23.645,-18.432,
     &-11.209,-5.021,0.236,4.341,7.426,9.956,
     &13.478,16.696,19.108,20.931,21.498,20.817,
     &18.861,17.677,16.587,15.626,14.603,15.285,
     &15.565,15.844,16.739,17.874,19.569,-999.000,
     &23.771,28.612,37.198,46.792,57.345,61.843,
     &54.386,37.488,18.390,1.643,-6.123,-11.714,
     &-15.501,-999.000,-19.734,-999.000,-22.886,-999.000,
     &-24.593,-24.220,-22.651,-21.117,-19.304
     &/
       data (fjrocvz(it,11,1,11), it=1,59) /
     &-17.627,-16.410,-16.039,-16.532,-17.997,-20.768,
     &-22.696,-24.121,-26.138,-26.283,-23.838,-16.508,
     &-8.301,-1.566,4.124,8.311,11.160,12.785,
     &15.412,19.409,23.560,26.252,26.903,25.723,
     &23.916,22.663,21.367,19.504,17.959,18.087,
     &17.359,17.158,18.133,19.760,22.130,24.366,
     &27.488,33.844,42.790,49.507,53.523,50.940,
     &36.032,18.056,4.339,-6.580,-14.229,-999.000,
     &-17.842,-999.000,-21.689,-999.000,-23.289,-999.000,
     &-23.687,-22.866,-21.218,-19.313,-17.627
     &/
       data (fjrocvz(it,12,1,11), it=1,59) /
     &-19.274,-18.892,-19.231,-20.609,-21.929,-23.825,
     &-24.905,-26.436,-29.085,-29.362,-27.553,-22.520,
     &-14.740,-5.926,3.845,9.093,13.189,15.484,
     &18.314,22.318,25.864,28.137,28.864,28.528,
     &27.676,26.617,24.512,21.914,19.608,18.657,
     &17.927,17.480,18.443,20.893,24.931,-999.000,
     &31.132,33.250,40.515,46.289,49.201,41.191,
     &26.948,15.093,4.736,-999.000,-12.280,-999.000,
     &-20.968,-999.000,-24.403,-999.000,-24.491,-999.000,
     &-23.601,-22.706,-21.716,-20.329,-19.274
     &/
       data (fjrocvz(it,13,1,11), it=1,59) /
     &-20.573,-21.240,-22.019,-22.980,-23.360,-24.373,
     &-25.114,-27.263,-30.159,-31.509,-30.657,-26.232,
     &-17.845,-8.943,-0.210,6.958,13.101,15.921,
     &18.999,22.609,25.710,27.460,27.739,27.192,
     &25.937,24.888,23.396,21.093,18.205,16.794,
     &16.733,17.021,17.801,18.720,19.836,-999.000,
     &24.631,-999.000,32.987,38.836,40.830,34.372,
     &23.155,-999.000,8.393,-999.000,-7.440,-999.000,
     &-18.394,-999.000,-24.278,-999.000,-24.850,-999.000,
     &-24.237,-23.266,-22.110,-20.940,-20.573
     &/
       data (fjrocvz(it,14,1,11), it=1,59) /
     &-21.352,-23.017,-24.126,-24.430,-23.950,-25.072,
     &-27.225,-29.913,-32.317,-32.943,-30.070,-26.828,
     &-20.959,-12.497,-3.082,5.380,13.045,15.779,
     &18.439,21.857,25.472,27.217,27.128,25.807,
     &24.413,23.459,21.906,19.098,16.232,14.785,
     &14.756,15.105,15.195,13.616,13.413,-999.000,
     &16.261,18.833,24.753,30.672,33.201,28.779,
     &21.745,12.978,6.857,-999.000,-5.793,-999.000,
     &-16.682,-999.000,-24.336,-999.000,-26.073,-999.000,
     &-25.948,-24.585,-22.631,-21.448,-21.352
     &/
       data (fjrocvz(it,15,1,11), it=1,59) /
     &-21.869,-22.847,-24.336,-25.374,-25.104,-25.797,
     &-27.218,-29.573,-33.293,-34.748,-31.925,-26.629,
     &-18.509,-10.394,-2.607,5.103,12.833,16.073,
     &18.644,20.569,22.584,23.628,23.093,21.706,
     &20.620,19.979,18.896,17.241,14.817,13.004,
     &12.546,12.586,12.314,11.651,11.870,-999.000,
     &14.320,-999.000,20.739,28.097,34.062,29.645,
     &23.001,14.653,8.498,-999.000,-4.234,-999.000,
     &-15.484,-999.000,-24.563,-999.000,-26.834,-999.000,
     &-26.214,-24.984,-23.446,-22.471,-21.869
     &/
       data (fjrocvz(it,16,1,11), it=1,59) /
     &-22.779,-23.224,-24.104,-25.099,-25.051,-25.450,
     &-26.176,-27.762,-31.716,-35.696,-35.811,-29.882,
     &-14.041,-4.611,0.888,6.946,13.626,16.550,
     &19.063,21.032,23.037,23.358,22.780,21.580,
     &20.414,19.307,17.953,15.845,13.068,11.415,
     &11.432,12.080,12.620,12.630,13.231,-999.000,
     &16.179,-999.000,22.351,29.837,35.914,29.751,
     &20.774,-999.000,6.948,-999.000,-4.537,-999.000,
     &-15.279,-999.000,-23.554,-999.000,-25.815,-999.000,
     &-25.869,-24.954,-23.979,-23.446,-22.779
     &/
       data (fjrocvz(it,17,1,11), it=1,59) /
     &-21.660,-21.687,-21.962,-22.883,-23.621,-25.451,
     &-26.539,-26.259,-28.293,-31.459,-33.446,-28.461,
     &-12.351,-0.022,4.402,8.350,13.105,15.815,
     &18.321,20.501,22.323,23.677,23.877,23.003,
     &22.170,21.089,19.582,17.493,14.874,13.003,
     &12.804,13.582,13.916,13.338,14.250,-999.000,
     &16.740,19.808,22.693,28.186,32.250,28.107,
     &18.539,-999.000,2.294,-999.000,-9.516,-999.000,
     &-18.748,-999.000,-24.843,-999.000,-26.865,-999.000,
     &-26.118,-24.064,-22.615,-22.145,-21.660
     &/
       data (fjrocvz(it,18,1,11), it=1,59) /
     &-20.122,-19.960,-20.219,-21.424,-23.130,-26.123,
     &-27.775,-28.594,-30.061,-30.000,-28.829,-23.981,
     &-11.552,0.639,5.725,9.343,13.971,17.183,
     &19.796,21.005,22.473,24.164,24.402,23.401,
     &22.190,21.117,19.468,17.904,16.223,14.603,
     &13.859,14.091,14.561,14.531,14.927,-999.000,
     &18.612,-999.000,23.462,29.617,32.295,25.785,
     &14.415,5.004,-2.679,-999.000,-12.394,-999.000,
     &-19.959,-999.000,-25.700,-999.000,-27.408,-999.000,
     &-26.045,-23.471,-21.986,-21.079,-20.122
     &/
       data (fjrocvz(it,19,1,11), it=1,59) /
     &-21.303,-20.243,-19.370,-19.589,-20.100,-22.269,
     &-22.898,-24.530,-27.371,-29.376,-29.147,-25.365,
     &-15.496,-2.581,4.838,10.036,15.248,19.437,
     &22.567,24.217,26.111,27.511,28.045,26.860,
     &25.707,24.928,22.661,19.491,17.455,16.384,
     &16.179,15.959,15.718,14.511,14.046,-999.000,
     &17.181,-999.000,21.506,25.027,24.554,15.842,
     &8.800,-999.000,-3.760,-999.000,-12.808,-999.000,
     &-20.532,-999.000,-25.549,-999.000,-27.097,-999.000,
     &-25.957,-23.878,-22.672,-22.155,-21.303
     &/
       data (fjrocvz(it,20,1,11), it=1,59) /
     &-20.357,-18.975,-18.439,-18.316,-18.262,-19.679,
     &-19.949,-20.764,-23.213,-26.061,-28.908,-27.171,
     &-19.380,-5.616,2.022,6.624,12.323,17.654,
     &21.564,23.896,25.975,27.460,28.035,27.526,
     &27.643,27.861,26.227,22.985,19.858,18.252,
     &17.876,17.337,16.105,14.416,13.834,-999.000,
     &14.358,-999.000,15.976,17.905,20.776,15.299,
     &5.652,-999.000,-5.627,-999.000,-15.326,-999.000,
     &-22.609,-999.000,-27.323,-999.000,-28.868,-999.000,
     &-28.364,-26.383,-24.289,-22.254,-20.357
     &/
       data (fjrocvz(it,21,1,11), it=1,59) /
     &-21.651,-20.467,-20.318,-20.633,-20.526,-20.715,
     &-19.639,-20.354,-23.519,-28.356,-32.058,-30.917,
     &-22.745,-9.197,-1.528,3.057,8.338,13.071,
     &17.160,20.380,23.085,24.231,24.380,24.581,
     &25.547,26.050,24.559,21.259,17.856,16.009,
     &15.690,15.426,14.535,12.596,11.403,-999.000,
     &11.891,-999.000,13.080,14.681,15.445,10.585,
     &1.817,-999.000,-9.052,-999.000,-17.914,-999.000,
     &-23.969,-999.000,-28.482,-999.000,-29.869,-999.000,
     &-29.549,-28.070,-26.107,-23.774,-21.651
     &/
       data (fjrocvz(it,22,1,11), it=1,59) /
     &-20.058,-19.144,-19.274,-20.204,-20.445,-20.724,
     &-19.908,-20.774,-23.666,-27.963,-30.827,-29.604,
     &-21.721,-9.297,-1.690,2.845,6.919,11.668,
     &15.918,19.821,21.954,22.687,23.364,23.773,
     &23.941,23.328,20.645,16.218,12.094,10.015,
     &9.898,10.106,9.494,7.901,7.369,-999.000,
     &7.645,-999.000,6.904,-999.000,6.308,-999.000,
     &0.891,-999.000,-11.100,-999.000,-17.789,-999.000,
     &-22.306,-999.000,-26.291,-999.000,-27.838,-999.000,
     &-27.421,-25.673,-23.389,-21.548,-20.058
     &/
       data (fjrocvz(it,23,1,11), it=1,59) /
     &-15.829,-15.694,-16.017,-16.843,-17.673,-18.600,
     &-18.966,-20.085,-21.868,-23.559,-24.066,-20.850,
     &-13.598,-4.517,2.049,5.783,9.406,13.007,
     &17.005,20.838,23.703,25.864,26.973,26.435,
     &24.720,22.608,19.963,16.369,12.112,8.997,
     &7.099,6.110,5.340,4.349,4.037,-999.000,
     &4.370,-999.000,7.232,-999.000,7.434,-999.000,
     &4.034,-999.000,-1.553,-999.000,-11.878,-999.000,
     &-17.157,-999.000,-21.463,-999.000,-23.097,-999.000,
     &-22.835,-21.274,-19.001,-16.834,-15.829
     &/
       data (fjrocvz(it,24,1,11), it=1,59) /
     &-13.832,-13.562,-13.570,-14.223,-14.563,-15.587,
     &-16.496,-18.477,-21.115,-22.807,-22.627,-18.154,
     &-9.905,-1.792,3.787,8.095,11.972,15.960,
     &19.915,23.680,26.526,27.026,26.416,24.861,
     &22.528,20.423,18.062,15.543,12.064,8.776,
     &6.439,5.653,5.758,5.247,4.607,-999.000,
     &5.208,-999.000,6.736,-999.000,7.994,-999.000,
     &6.340,-999.000,1.142,-999.000,-6.412,-999.000,
     &-12.483,-999.000,-16.077,-999.000,-17.845,-999.000,
     &-18.724,-18.613,-17.202,-15.195,-13.832
     &/
       data (fjrocvz(it,25,1,11), it=1,59) /
     &-14.339,-13.417,-12.916,-12.892,-12.955,-14.940,
     &-17.052,-19.301,-22.381,-23.793,-22.616,-16.287,
     &-6.517,1.445,7.504,12.066,16.748,22.646,
     &27.330,30.486,31.449,29.534,27.148,24.119,
     &20.906,17.703,14.138,11.529,9.074,6.452,
     &4.812,4.611,5.382,5.204,5.167,-999.000,
     &4.522,-999.000,5.289,-999.000,8.184,-999.000,
     &6.416,-999.000,1.886,-999.000,-6.427,-999.000,
     &-11.957,-999.000,-15.410,-999.000,-17.751,-999.000,
     &-20.027,-20.270,-18.871,-16.248,-14.339
     &/
       data (fjrocvz(it, 1,2,11), it=1,59) /
     &-23.986,-23.073,-21.531,-20.437,-21.186,-22.291,
     &-23.041,-24.070,-24.944,-24.265,-20.970,-14.196,
     &-15.492,-9.883,-3.865,2.840,9.655,19.028,
     &26.299,29.365,29.553,28.831,26.640,22.773,
     &19.160,16.891,12.374,9.248,10.352,12.452,
     &13.423,14.060,15.235,17.707,21.134,-999.000,
     &25.289,-999.000,31.284,37.891,39.637,34.106,
     &24.163,-999.000,1.977,-999.000,-14.765,-999.000,
     &-23.003,-999.000,-26.174,-999.000,-28.853,-999.000,
     &-28.506,-28.272,-27.866,-26.063,-23.986
     &/
       data (fjrocvz(it, 2,2,11), it=1,59) /
     &-22.686,-21.917,-20.986,-20.515,-19.770,-19.695,
     &-20.167,-22.051,-23.851,-23.415,-18.175,-10.387,
     &-5.039,-0.155,5.017,8.177,13.881,20.035,
     &25.878,30.019,31.693,31.427,28.974,25.028,
     &21.512,19.917,16.931,14.049,13.645,14.413,
     &15.908,17.086,18.623,21.239,23.877,-999.000,
     &26.441,-999.000,32.541,36.613,39.709,39.070,
     &32.996,-999.000,3.568,-999.000,-19.575,-999.000,
     &-25.443,-999.000,-31.184,-999.000,-32.734,-999.000,
     &-30.084,-28.915,-26.397,-23.936,-22.686
     &/
       data (fjrocvz(it, 3,2,11), it=1,59) /
     &-22.801,-21.582,-20.909,-19.769,-18.749,-19.551,
     &-19.482,-19.932,-21.189,-20.619,-16.169,-9.740,
     &-3.877,1.368,4.547,8.394,13.898,18.335,
     &22.316,25.576,28.070,28.244,25.778,22.486,
     &20.105,19.142,17.503,15.889,15.450,15.988,
     &17.376,18.720,20.829,22.402,24.882,-999.000,
     &28.608,-999.000,32.124,38.045,42.470,42.135,
     &33.958,17.349,2.852,-13.900,-24.932,-26.403,
     &-28.675,-999.000,-33.646,-999.000,-33.654,-999.000,
     &-31.758,-29.073,-25.998,-24.272,-22.801
     &/
       data (fjrocvz(it, 4,2,11), it=1,59) /
     &-23.850,-22.619,-20.988,-19.363,-18.904,-19.579,
     &-19.170,-18.071,-17.980,-17.184,-14.132,-8.649,
     &-2.635,1.238,3.120,6.410,11.385,16.947,
     &21.097,23.778,24.782,24.062,21.811,19.097,
     &17.389,16.515,15.181,14.579,15.613,16.381,
     &17.558,19.365,21.225,23.749,27.631,-999.000,
     &30.218,-999.000,34.263,40.348,42.547,42.962,
     &39.107,23.976,3.172,-17.947,-27.864,-999.000,
     &-30.955,-999.000,-34.630,-999.000,-35.519,-999.000,
     &-32.561,-29.099,-26.435,-24.797,-23.850
     &/
       data (fjrocvz(it, 5,2,11), it=1,59) /
     &-23.650,-22.225,-19.890,-18.173,-17.869,-19.236,
     &-19.412,-18.559,-18.649,-17.880,-14.930,-8.773,
     &-2.904,1.426,4.759,7.556,12.550,18.845,
     &23.736,26.812,28.022,27.282,24.432,21.339,
     &19.301,18.333,17.286,17.056,17.543,17.576,
     &18.487,20.111,22.333,24.944,27.159,-999.000,
     &31.014,-999.000,35.016,39.704,43.365,46.245,
     &43.258,25.728,1.642,-14.529,-27.641,-31.827,
     &-35.584,-38.400,-39.340,-38.080,-35.522,-999.000,
     &-31.683,-28.487,-25.853,-24.424,-23.650
     &/
       data (fjrocvz(it, 6,2,11), it=1,59) /
     &-24.503,-23.640,-22.172,-20.785,-20.475,-21.053,
     &-20.209,-19.775,-20.027,-18.228,-16.043,-14.638,
     &-11.343,-3.678,3.629,9.466,14.950,20.951,
     &25.746,28.777,30.971,30.802,28.758,25.791,
     &23.828,22.641,21.130,19.639,19.095,18.322,
     &18.033,19.935,22.426,23.980,26.527,-999.000,
     &30.484,-999.000,34.428,42.112,48.742,50.775,
     &42.674,25.500,4.761,-12.195,-25.732,-32.912,
     &-36.770,-999.000,-38.622,-999.000,-36.433,-999.000,
     &-32.495,-28.686,-26.439,-25.531,-24.503
     &/
       data (fjrocvz(it, 7,2,11), it=1,59) /
     &-26.737,-26.554,-25.619,-24.791,-24.416,-23.674,
     &-22.369,-21.678,-21.144,-19.957,-17.739,-15.289,
     &-9.954,-4.642,0.438,5.022,11.287,17.940,
     &23.562,27.990,30.430,30.908,29.864,27.760,
     &25.891,23.916,21.783,20.064,18.304,16.220,
     &15.588,16.531,18.125,20.744,23.126,-999.000,
     &25.994,-999.000,30.829,39.745,47.673,51.661,
     &39.438,18.852,-0.117,-13.496,-26.103,-35.066,
     &-40.591,-41.168,-40.237,-38.927,-37.139,-999.000,
     &-33.042,-30.430,-28.411,-27.081,-26.737
     &/
       data (fjrocvz(it, 8,2,11), it=1,59) /
     &-26.805,-25.832,-25.126,-24.868,-24.582,-24.839,
     &-24.579,-24.276,-23.706,-25.267,-26.960,-28.524,
     &-22.985,-13.258,-6.448,0.274,5.989,10.869,
     &17.287,22.374,25.918,26.525,25.993,24.849,
     &23.281,21.598,19.402,17.288,14.763,12.525,
     &11.546,11.887,13.390,15.638,17.600,-999.000,
     &21.719,-999.000,28.418,35.062,42.695,43.922,
     &29.821,14.018,-0.703,-13.886,-23.866,-31.452,
     &-36.326,-36.385,-33.839,-999.000,-33.059,-999.000,
     &-31.236,-30.607,-29.188,-27.873,-26.805
     &/
       data (fjrocvz(it, 9,2,11), it=1,59) /
     &-24.156,-22.270,-20.543,-19.518,-19.669,-20.890,
     &-22.748,-24.389,-25.785,-27.221,-29.659,-31.430,
     &-25.798,-14.440,-5.774,0.176,3.814,8.295,
     &13.637,17.767,20.490,21.245,21.098,19.770,
     &17.552,15.542,13.689,11.751,9.682,8.290,
     &8.154,8.723,9.822,11.813,14.148,-999.000,
     &17.788,-999.000,25.577,33.298,40.396,44.469,
     &38.875,24.045,8.945,-4.505,-14.500,-999.000,
     &-21.284,-999.000,-24.534,-999.000,-26.760,-999.000,
     &-26.750,-26.614,-26.593,-25.947,-24.156
     &/
       data (fjrocvz(it,10,2,11), it=1,59) /
     &-25.501,-23.012,-20.603,-19.273,-18.394,-19.497,
     &-22.580,-25.221,-26.636,-27.235,-27.779,-27.428,
     &-23.017,-11.451,-3.018,2.664,5.779,10.496,
     &15.952,19.584,20.849,20.763,19.668,18.171,
     &16.340,13.865,11.588,9.712,8.557,7.935,
     &7.973,8.292,8.280,9.459,11.828,-999.000,
     &16.634,-999.000,25.432,34.282,43.944,48.115,
     &46.110,31.653,17.267,-999.000,-1.591,-999.000,
     &-12.594,-999.000,-20.440,-999.000,-22.458,-999.000,
     &-24.262,-24.996,-25.771,-25.997,-25.501
     &/
       data (fjrocvz(it,11,2,11), it=1,59) /
     &-28.673,-27.446,-26.785,-26.490,-25.905,-25.831,
     &-26.337,-27.561,-28.810,-30.620,-29.851,-25.977,
     &-20.306,-10.502,-2.885,3.208,7.421,11.668,
     &16.480,20.591,22.960,23.563,22.846,21.611,
     &20.599,18.533,15.754,13.807,12.567,11.581,
     &10.924,10.167,9.611,10.805,13.797,-999.000,
     &18.938,-999.000,26.115,34.263,44.084,46.204,
     &43.004,31.590,20.071,-999.000,-0.922,-999.000,
     &-13.720,-999.000,-22.578,-999.000,-24.569,-999.000,
     &-27.741,-27.461,-28.224,-29.323,-28.673
     &/
       data (fjrocvz(it,12,2,11), it=1,59) /
     &-29.958,-30.155,-29.798,-29.447,-29.165,-28.270,
     &-27.948,-29.350,-31.688,-33.946,-34.741,-31.514,
     &-27.386,-18.192,-8.683,-3.013,1.178,7.732,
     &15.146,20.832,24.428,25.393,24.874,24.221,
     &23.756,21.879,19.105,16.947,16.252,15.544,
     &14.628,13.385,12.507,13.811,17.330,-999.000,
     &21.892,-999.000,28.543,35.406,43.612,49.579,
     &46.240,28.636,13.151,1.397,-9.267,-999.000,
     &-16.230,-999.000,-22.659,-999.000,-26.991,-999.000,
     &-29.835,-29.732,-29.918,-30.092,-29.958
     &/
       data (fjrocvz(it,13,2,11), it=1,59) /
     &-27.544,-26.726,-26.333,-26.155,-25.518,-25.851,
     &-27.239,-28.400,-29.507,-30.447,-30.998,-29.642,
     &-27.026,-19.259,-10.888,-4.677,-1.113,6.385,
     &16.477,23.104,25.995,26.887,26.500,26.060,
     &25.598,23.635,21.105,19.531,19.820,19.854,
     &19.265,18.105,16.979,17.862,20.781,-999.000,
     &24.517,-999.000,33.350,39.699,48.271,54.606,
     &48.345,30.223,9.255,-4.529,-13.888,-16.095,
     &-17.328,-999.000,-21.185,-999.000,-24.045,-999.000,
     &-26.287,-28.202,-28.018,-27.514,-27.544
     &/
       data (fjrocvz(it,14,2,11), it=1,59) /
     &-24.886,-24.275,-23.849,-23.182,-22.988,-24.566,
     &-27.318,-30.861,-33.989,-34.047,-31.291,-27.430,
     &-23.839,-20.637,-13.855,-6.202,-0.862,7.055,
     &17.106,23.475,26.779,27.634,27.020,26.175,
     &25.074,23.746,21.966,21.001,21.635,21.947,
     &21.233,20.255,19.513,20.259,23.183,-999.000,
     &28.404,-999.000,37.411,47.815,54.674,54.008,
     &45.459,29.618,11.249,-999.000,-12.958,-999.000,
     &-20.827,-999.000,-22.567,-999.000,-24.877,-999.000,
     &-27.256,-27.983,-26.978,-25.599,-24.886
     &/
       data (fjrocvz(it,15,2,11), it=1,59) /
     &-22.238,-21.776,-21.242,-20.808,-20.872,-22.739,
     &-26.792,-32.949,-37.345,-38.750,-35.395,-28.255,
     &-23.509,-18.145,-12.539,-4.843,2.074,8.930,
     &16.826,22.415,25.325,25.669,24.275,22.127,
     &20.366,19.333,17.727,16.789,17.119,17.409,
     &17.066,16.617,17.186,18.899,22.632,-999.000,
     &27.571,-999.000,35.547,45.307,52.901,55.036,
     &47.285,29.281,11.189,-999.000,-14.448,-999.000,
     &-21.963,-999.000,-25.372,-999.000,-28.121,-999.000,
     &-29.428,-29.522,-26.947,-23.711,-22.238
     &/
       data (fjrocvz(it,16,2,11), it=1,59) /
     &-21.737,-20.787,-21.077,-21.419,-22.387,-24.952,
     &-28.907,-33.828,-37.893,-40.699,-39.583,-32.041,
     &-22.507,-16.259,-9.950,-2.959,4.121,9.837,
     &17.295,22.432,24.520,25.058,23.925,21.010,
     &17.710,15.032,12.825,11.662,11.996,12.792,
     &12.579,12.062,12.648,14.153,17.215,-999.000,
     &22.291,-999.000,30.014,39.389,47.800,49.211,
     &40.129,21.089,5.640,-999.000,-14.506,-999.000,
     &-23.215,-999.000,-27.959,-999.000,-30.809,-999.000,
     &-32.057,-31.408,-28.709,-24.856,-21.737
     &/
       data (fjrocvz(it,17,2,11), it=1,59) /
     &-25.355,-25.091,-24.643,-24.446,-25.598,-27.935,
     &-31.384,-36.281,-40.712,-42.042,-40.196,-31.975,
     &-22.807,-15.487,-8.154,-1.963,4.664,10.703,
     &18.669,24.153,26.862,28.048,27.219,24.686,
     &20.905,17.342,13.892,11.522,11.477,11.773,
     &11.155,10.183,9.563,9.914,12.854,-999.000,
     &18.860,-999.000,27.464,34.752,40.512,43.475,
     &36.263,18.560,3.059,-12.188,-22.055,-26.211,
     &-28.429,-999.000,-30.168,-999.000,-32.428,-999.000,
     &-33.278,-31.997,-30.263,-27.653,-25.355
     &/
       data (fjrocvz(it,18,2,11), it=1,59) /
     &-27.523,-26.843,-25.977,-25.969,-26.464,-28.001,
     &-31.990,-36.959,-40.518,-41.677,-38.975,-31.528,
     &-22.220,-14.762,-8.944,-2.556,4.720,12.704,
     &20.271,26.402,30.911,32.228,30.327,26.736,
     &23.572,20.610,16.776,13.426,12.365,11.554,
     &10.588,9.294,8.605,8.664,11.107,-999.000,
     &16.897,21.099,26.690,34.067,39.617,37.519,
     &30.665,17.181,1.729,-999.000,-21.430,-999.000,
     &-29.215,-999.000,-32.743,-999.000,-34.180,-999.000,
     &-34.157,-33.799,-32.378,-29.798,-27.523
     &/
       data (fjrocvz(it,19,2,11), it=1,59) /
     &-26.388,-25.493,-25.683,-25.463,-25.527,-27.024,
     &-30.096,-33.167,-36.238,-37.525,-34.749,-27.802,
     &-19.473,-13.865,-8.152,-1.756,5.163,14.060,
     &22.834,29.912,35.109,36.205,33.956,30.331,
     &26.742,23.334,19.500,16.127,15.550,15.110,
     &13.783,12.144,11.224,10.964,13.005,-999.000,
     &19.740,-999.000,29.197,37.633,41.681,39.723,
     &33.990,21.960,0.930,-13.317,-24.073,-28.360,
     &-30.025,-999.000,-32.355,-999.000,-33.076,-999.000,
     &-33.364,-33.191,-32.126,-29.331,-26.388
     &/
       data (fjrocvz(it,20,2,11), it=1,59) /
     &-27.495,-26.559,-25.139,-23.663,-22.976,-23.925,
     &-26.412,-29.867,-32.341,-32.919,-30.336,-21.258,
     &-13.136,-10.182,-5.182,-0.276,5.370,14.396,
     &22.865,30.780,36.189,37.666,36.090,32.992,
     &29.874,27.140,23.265,20.077,19.428,18.531,
     &16.993,15.336,14.278,14.082,16.663,-999.000,
     &23.145,-999.000,31.151,39.320,46.103,49.428,
     &39.473,22.470,0.478,-10.428,-18.881,-25.388,
     &-29.732,-999.000,-31.764,-999.000,-33.147,-999.000,
     &-33.249,-32.977,-32.215,-29.869,-27.495
     &/
       data (fjrocvz(it,21,2,11), it=1,59) /
     &-26.987,-24.827,-23.051,-21.952,-21.420,-21.832,
     &-23.295,-26.331,-29.820,-30.540,-27.753,-20.501,
     &-15.103,-9.573,-4.171,1.388,7.567,14.562,
     &20.825,27.884,32.430,34.524,33.115,29.944,
     &27.143,25.245,22.111,18.581,17.394,16.381,
     &15.016,14.169,14.325,14.704,16.850,-999.000,
     &22.229,-999.000,31.614,38.552,45.501,44.031,
     &34.791,-999.000,4.363,-999.000,-14.912,-999.000,
     &-25.907,-999.000,-30.710,-999.000,-32.126,-999.000,
     &-32.529,-31.927,-31.072,-29.297,-26.987
     &/
       data (fjrocvz(it,22,2,11), it=1,59) /
     &-25.798,-24.622,-23.428,-22.543,-22.363,-22.574,
     &-23.022,-25.453,-29.663,-30.424,-27.370,-20.446,
     &-12.247,-8.752,-3.029,2.093,7.453,13.652,
     &20.265,26.358,29.529,30.487,28.777,25.194,
     &21.251,18.394,15.403,12.863,12.106,11.538,
     &11.065,11.781,13.324,14.303,16.746,-999.000,
     &22.257,-999.000,32.627,36.176,40.412,35.792,
     &26.966,-999.000,1.482,-999.000,-16.127,-999.000,
     &-27.066,-999.000,-29.708,-999.000,-30.264,-999.000,
     &-31.095,-30.575,-29.689,-27.759,-25.798
     &/
       data (fjrocvz(it,23,2,11), it=1,59) /
     &-24.665,-24.053,-22.471,-20.432,-19.691,-20.600,
     &-22.451,-24.786,-26.939,-27.199,-24.400,-18.024,
     &-12.435,-9.381,-3.880,1.457,6.874,12.642,
     &20.847,26.694,28.768,28.441,26.131,22.386,
     &18.427,15.012,11.911,9.860,9.787,9.826,
     &9.774,10.057,12.094,14.437,17.678,-999.000,
     &21.891,-999.000,31.564,36.927,40.570,36.567,
     &28.027,13.217,-0.036,-999.000,-16.244,-999.000,
     &-23.861,-999.000,-27.306,-999.000,-28.593,-999.000,
     &-29.791,-29.580,-28.497,-25.968,-24.665
     &/
       data (fjrocvz(it,24,2,11), it=1,59) /
     &-23.818,-22.172,-20.345,-18.174,-17.839,-19.599,
     &-21.733,-23.010,-24.065,-24.497,-22.284,-17.321,
     &-11.032,-5.711,-2.064,1.401,6.420,14.953,
     &23.283,27.050,28.082,27.581,24.796,20.717,
     &16.950,13.925,10.449,8.281,9.304,10.894,
     &11.476,11.429,13.181,15.834,19.039,-999.000,
     &22.686,-999.000,31.227,39.549,42.312,39.054,
     &26.531,-999.000,1.961,-999.000,-12.829,-999.000,
     &-21.126,-999.000,-26.820,-999.000,-27.554,-999.000,
     &-28.556,-28.701,-27.812,-25.858,-23.818
     &/
       data (fjrocvz(it,25,2,11), it=1,59) /
     &-23.986,-23.073,-21.531,-20.437,-21.186,-22.291,
     &-23.041,-24.070,-24.944,-24.265,-20.970,-14.196,
     &-15.492,-9.883,-3.865,2.840,9.655,19.028,
     &26.299,29.365,29.553,28.831,26.640,22.773,
     &19.160,16.891,12.374,9.248,10.352,12.452,
     &13.423,14.060,15.235,17.707,21.134,-999.000,
     &25.289,-999.000,31.284,37.891,39.637,34.106,
     &24.163,-999.000,1.977,-999.000,-14.765,-999.000,
     &-23.003,-999.000,-26.174,-999.000,-28.853,-999.000,
     &-28.506,-28.272,-27.866,-26.063,-23.986
     &/
       data (fjrocvz(it, 1,3,11), it=1,59) /
     &-19.178,-19.000,-18.076,-16.449,-13.661,-13.210,
     &-13.511,-17.126,-24.690,-26.696,-19.880,-6.284,
     &4.943,8.492,10.910,12.684,12.015,11.765,
     &12.471,15.432,19.492,20.566,20.071,17.730,
     &14.542,12.487,12.808,13.585,13.096,12.008,
     &11.200,11.018,10.625,10.518,11.367,-999.000,
     &16.994,-999.000,20.703,-999.000,26.471,31.206,
     &31.822,25.317,13.271,-999.000,-4.108,-999.000,
     &-10.802,-999.000,-12.683,-999.000,-13.742,-999.000,
     &-15.231,-17.845,-19.617,-19.145,-19.178
     &/
       data (fjrocvz(it, 2,3,11), it=1,59) /
     &-18.204,-17.986,-17.223,-15.485,-13.691,-14.824,
     &-16.768,-20.740,-26.598,-27.727,-19.623,-4.050,
     &9.199,12.662,13.958,14.963,15.119,15.890,
     &17.215,19.288,21.694,22.546,21.792,19.971,
     &18.574,17.177,17.364,17.899,17.939,17.138,
     &16.320,15.735,15.017,15.989,16.744,-999.000,
     &19.935,-999.000,23.821,-999.000,27.897,30.810,
     &29.906,24.437,9.418,-999.000,-6.021,-999.000,
     &-12.637,-999.000,-14.516,-999.000,-15.856,-999.000,
     &-17.105,-19.547,-20.405,-18.923,-18.204
     &/
       data (fjrocvz(it, 3,3,11), it=1,59) /
     &-17.803,-16.610,-15.952,-14.931,-14.335,-16.116,
     &-18.336,-21.193,-26.368,-30.020,-22.910,-3.896,
     &13.403,16.149,15.075,14.248,14.032,15.268,
     &16.826,18.202,19.050,18.696,17.608,17.211,
     &17.754,17.067,17.705,18.655,19.522,20.176,
     &19.866,19.095,18.331,19.501,20.546,-999.000,
     &23.700,-999.000,25.334,-999.000,28.293,27.843,
     &24.977,20.217,9.799,-999.000,-6.812,-999.000,
     &-14.019,-999.000,-17.085,-999.000,-19.327,-999.000,
     &-21.359,-23.623,-24.014,-20.889,-17.803
     &/
       data (fjrocvz(it, 4,3,11), it=1,59) /
     &-19.011,-16.727,-15.914,-15.765,-14.856,-15.689,
     &-16.195,-17.795,-22.618,-25.640,-18.487,1.265,
     &15.606,18.591,17.694,17.268,16.849,16.123,
     &15.156,14.698,15.110,15.112,15.248,15.980,
     &16.750,16.794,17.337,18.504,19.865,20.375,
     &20.663,20.629,19.836,21.332,24.344,-999.000,
     &25.817,-999.000,26.647,-999.000,31.640,30.987,
     &27.435,16.446,6.421,-999.000,-6.280,-999.000,
     &-14.501,-999.000,-19.526,-999.000,-22.602,-999.000,
     &-24.278,-25.568,-25.552,-22.441,-19.011
     &/
       data (fjrocvz(it, 5,3,11), it=1,59) /
     &-19.443,-17.951,-17.065,-15.106,-12.695,-13.335,
     &-13.907,-15.822,-19.032,-19.993,-14.249,-0.112,
     &13.993,19.533,19.744,19.476,18.882,18.028,
     &16.850,16.301,17.356,18.172,18.406,17.936,
     &17.543,16.637,16.238,16.818,17.676,18.559,
     &19.617,20.456,20.623,22.622,24.770,-999.000,
     &26.981,-999.000,29.915,34.883,39.040,32.650,
     &25.057,14.268,3.684,-999.000,-8.472,-999.000,
     &-15.536,-999.000,-20.564,-999.000,-23.406,-999.000,
     &-25.474,-26.622,-25.706,-22.343,-19.443
     &/
       data (fjrocvz(it, 6,3,11), it=1,59) /
     &-20.608,-20.422,-20.117,-17.532,-13.375,-12.547,
     &-13.215,-15.247,-17.152,-16.496,-12.457,-2.425,
     &10.734,16.071,16.230,15.964,16.295,16.999,
     &17.159,18.768,21.192,22.203,21.676,20.712,
     &19.857,18.345,17.213,16.387,16.916,17.521,
     &18.221,18.783,19.559,20.914,22.187,-999.000,
     &26.646,-999.000,29.743,33.790,34.724,25.684,
     &15.501,-999.000,-1.392,-999.000,-11.273,-999.000,
     &-18.092,-999.000,-22.817,-999.000,-25.662,-999.000,
     &-27.881,-28.579,-26.538,-23.212,-20.608
     &/
       data (fjrocvz(it, 7,3,11), it=1,59) /
     &-23.310,-21.490,-20.957,-19.929,-16.817,-16.920,
     &-17.169,-17.337,-18.493,-16.250,-12.408,-5.531,
     &1.291,6.561,8.350,9.730,11.262,13.057,
     &14.842,17.389,20.739,22.750,23.391,23.482,
     &23.572,23.282,21.885,20.191,19.658,18.827,
     &18.428,17.858,17.478,17.590,18.024,-999.000,
     &20.023,-999.000,21.316,-999.000,20.225,-999.000,
     &7.825,-999.000,-2.598,-999.000,-11.781,-999.000,
     &-18.619,-999.000,-23.582,-999.000,-26.976,-999.000,
     &-29.353,-30.080,-28.855,-26.387,-23.310
     &/
       data (fjrocvz(it, 8,3,11), it=1,59) /
     &-21.512,-20.284,-21.321,-21.843,-20.671,-19.095,
     &-15.527,-13.618,-14.082,-13.062,-10.252,-6.547,
     &-4.330,-1.218,0.673,2.870,5.630,7.847,
     &9.760,12.025,15.913,19.987,22.196,23.031,
     &23.741,23.481,22.108,20.154,18.945,18.244,
     &17.662,16.784,15.564,13.771,12.182,-999.000,
     &12.655,-999.000,12.702,-999.000,11.663,-999.000,
     &6.715,-999.000,2.713,-999.000,-4.476,-999.000,
     &-12.249,-999.000,-18.664,-999.000,-23.668,-999.000,
     &-27.247,-28.183,-26.655,-24.065,-21.512
     &/
       data (fjrocvz(it, 9,3,11), it=1,59) /
     &-16.758,-17.133,-17.546,-18.157,-18.188,-16.995,
     &-13.940,-12.606,-13.790,-14.171,-13.306,-10.879,
     &-10.234,-4.980,0.213,5.164,10.677,11.971,
     &12.998,14.017,15.350,15.524,17.085,18.479,
     &19.842,20.325,18.819,16.478,15.436,14.310,
     &13.063,11.765,10.691,9.763,9.047,-999.000,
     &8.043,-999.000,6.987,-999.000,8.913,-999.000,
     &10.933,-999.000,11.085,-999.000,5.790,-999.000,
     &-0.189,-999.000,-6.583,-999.000,-11.207,-999.000,
     &-15.475,-17.024,-17.075,-16.890,-16.758
     &/
       data (fjrocvz(it,10,3,11), it=1,59) /
     &-14.702,-15.698,-15.148,-14.370,-15.288,-17.188,
     &-17.231,-16.512,-16.937,-16.490,-16.023,-16.840,
     &-18.237,-9.166,0.073,6.589,13.799,14.880,
     &15.852,16.586,17.510,18.331,18.513,18.708,
     &19.229,19.761,18.402,16.513,14.897,12.951,
     &10.866,8.481,7.017,5.895,4.546,-999.000,
     &3.325,-999.000,3.948,-999.000,6.732,-999.000,
     &11.615,-999.000,14.794,-999.000,13.978,-999.000,
     &9.113,-999.000,0.953,-999.000,-6.880,-999.000,
     &-10.218,-12.215,-13.204,-14.027,-14.702
     &/
       data (fjrocvz(it,11,3,11), it=1,59) /
     &-18.228,-20.216,-20.653,-20.137,-20.210,-20.044,
     &-18.709,-18.634,-22.238,-20.773,-20.337,-14.171,
     &-17.189,-10.194,-3.534,3.489,11.130,13.609,
     &15.772,17.526,20.924,23.663,23.337,22.859,
     &22.962,22.337,21.337,19.993,17.786,15.251,
     &12.317,9.762,8.354,6.086,3.872,-999.000,
     &2.665,-999.000,2.835,-999.000,6.000,-999.000,
     &14.671,18.509,19.405,17.165,15.586,-999.000,
     &10.975,-999.000,4.082,-999.000,-5.098,-999.000,
     &-9.982,-13.650,-15.590,-16.957,-18.228
     &/
       data (fjrocvz(it,12,3,11), it=1,59) /
     &-18.572,-21.327,-23.982,-25.998,-26.310,-24.403,
     &-24.131,-28.298,-33.585,-31.234,-25.524,-15.833,
     &-14.668,-10.328,-6.385,0.171,7.338,11.367,
     &14.803,19.148,23.770,26.081,26.313,25.919,
     &25.518,24.080,22.554,21.429,19.240,16.560,
     &13.864,12.419,11.265,9.598,7.582,-999.000,
     &5.770,-999.000,7.070,-999.000,14.544,20.253,
     &24.883,26.851,25.173,-999.000,20.071,-999.000,
     &11.997,-999.000,3.213,-999.000,-2.505,-999.000,
     &-8.921,-12.486,-15.323,-17.205,-18.572
     &/
       data (fjrocvz(it,13,3,11), it=1,59) /
     &-17.057,-19.083,-20.012,-21.191,-23.510,-25.235,
     &-27.675,-30.468,-32.804,-31.065,-23.990,-16.243,
     &-12.428,-10.403,-6.914,-3.013,1.061,7.894,
     &16.198,22.198,26.233,28.558,28.844,27.692,
     &26.602,24.984,23.282,21.567,19.150,17.131,
     &15.263,13.197,11.796,10.792,9.522,-999.000,
     &9.455,-999.000,11.942,-999.000,17.954,23.979,
     &31.100,34.062,28.881,-999.000,21.639,-999.000,
     &12.399,-999.000,2.185,-999.000,-3.968,-999.000,
     &-8.379,-10.062,-11.942,-14.894,-17.057
     &/
       data (fjrocvz(it,14,3,11), it=1,59) /
     &-17.851,-19.446,-19.314,-20.124,-22.510,-26.096,
     &-28.400,-30.561,-33.513,-32.411,-23.916,-13.016,
     &-8.149,-6.850,-4.987,-1.989,1.240,7.104,
     &16.254,21.963,26.191,28.455,28.592,26.825,
     &25.388,24.107,22.998,21.566,19.784,18.421,
     &16.511,14.327,13.103,11.854,11.358,-999.000,
     &11.805,-999.000,14.269,-999.000,20.116,24.952,
     &31.995,35.530,33.539,25.901,20.484,-999.000,
     &10.405,-999.000,2.224,-999.000,-3.556,-999.000,
     &-7.433,-9.458,-11.661,-15.152,-17.851
     &/
       data (fjrocvz(it,15,3,11), it=1,59) /
     &-18.305,-18.766,-18.760,-18.895,-20.027,-23.219,
     &-25.969,-28.980,-32.278,-33.339,-25.270,-13.755,
     &-7.100,-6.167,-5.393,-3.038,0.416,5.944,
     &13.616,19.112,22.820,24.916,24.277,22.136,
     &20.238,18.659,17.756,16.418,14.505,13.589,
     &12.592,11.533,10.185,9.161,8.842,-999.000,
     &8.975,-999.000,11.295,-999.000,16.383,20.966,
     &25.335,27.055,23.333,17.794,13.298,-999.000,
     &4.671,-999.000,-1.677,-999.000,-5.497,-999.000,
     &-10.152,-11.997,-13.809,-16.073,-18.305
     &/
       data (fjrocvz(it,16,3,11), it=1,59) /
     &-16.102,-15.787,-15.166,-15.027,-16.224,-20.128,
     &-24.361,-27.545,-32.194,-35.220,-31.031,-18.494,
     &-8.430,-5.958,-5.402,-3.458,-0.301,4.920,
     &11.703,16.868,20.090,21.368,20.063,18.385,
     &16.788,14.941,13.030,10.467,7.927,6.699,
     &6.170,5.360,4.328,2.806,2.218,-999.000,
     &3.559,-999.000,6.247,-999.000,11.493,15.313,
     &17.092,15.520,12.179,-999.000,2.765,-999.000,
     &-4.278,-999.000,-8.687,-999.000,-10.183,-999.000,
     &-11.602,-13.093,-14.302,-15.048,-16.102
     &/
       data (fjrocvz(it,17,3,11), it=1,59) /
     &-16.982,-16.464,-15.315,-14.767,-15.599,-18.637,
     &-23.015,-29.212,-36.039,-38.925,-34.138,-18.935,
     &-7.103,-5.228,-3.781,-1.071,1.967,6.728,
     &12.922,18.304,21.569,22.976,22.163,20.031,
     &17.996,15.956,13.784,10.825,8.023,5.817,
     &3.738,2.604,1.654,0.119,-0.903,-999.000,
     &-0.769,-999.000,1.515,-999.000,8.865,13.653,
     &13.233,9.707,5.244,-999.000,-3.672,-999.000,
     &-10.750,-999.000,-14.264,-999.000,-15.047,-999.000,
     &-15.428,-16.030,-16.248,-16.324,-16.982
     &/
       data (fjrocvz(it,18,3,11), it=1,59) /
     &-23.537,-21.517,-19.376,-17.474,-16.551,-18.197,
     &-21.156,-27.648,-36.365,-38.729,-32.371,-10.895,
     &-1.824,-1.357,0.168,3.116,6.357,11.946,
     &18.253,23.212,25.612,27.049,26.728,25.083,
     &22.719,19.684,17.119,14.072,11.318,8.361,
     &5.736,3.972,2.445,1.010,-0.503,-999.000,
     &-0.607,-999.000,2.028,-999.000,11.033,12.317,
     &9.707,3.979,-1.494,-999.000,-10.304,-999.000,
     &-17.223,-999.000,-19.707,-999.000,-20.194,-999.000,
     &-20.485,-21.664,-22.550,-23.028,-23.537
     &/
       data (fjrocvz(it,19,3,11), it=1,59) /
     &-24.624,-23.574,-21.566,-19.631,-18.816,-20.507,
     &-21.460,-23.975,-30.251,-30.536,-24.611,-7.748,
     &1.339,2.746,4.204,6.878,10.904,16.344,
     &22.491,27.027,29.515,30.283,29.575,28.095,
     &25.961,23.257,20.206,17.216,14.404,11.523,
     &9.201,7.774,6.604,4.431,2.798,-999.000,
     &3.162,-999.000,4.492,-999.000,12.897,13.913,
     &11.633,5.757,-1.478,-999.000,-11.403,-999.000,
     &-18.063,-999.000,-19.444,-999.000,-20.563,-999.000,
     &-21.115,-22.293,-23.681,-24.385,-24.624
     &/
       data (fjrocvz(it,20,3,11), it=1,59) /
     &-21.052,-20.530,-18.963,-17.148,-16.841,-19.570,
     &-22.157,-24.586,-26.138,-23.168,-16.524,-6.509,
     &1.558,4.714,6.232,8.145,11.561,17.113,
     &23.520,29.332,32.033,32.175,30.655,28.317,
     &25.906,23.798,22.261,20.793,18.398,15.635,
     &12.867,11.001,8.905,7.217,6.500,-999.000,
     &6.299,-999.000,7.838,-999.000,14.336,16.820,
     &16.188,11.810,5.582,-999.000,-6.478,-999.000,
     &-12.544,-999.000,-13.594,-999.000,-15.653,-999.000,
     &-17.698,-19.820,-21.535,-21.145,-21.052
     &/
       data (fjrocvz(it,21,3,11), it=1,59) /
     &-17.977,-17.668,-16.241,-14.563,-14.257,-17.641,
     &-20.432,-22.112,-24.964,-26.960,-23.755,-13.257,
     &-1.533,2.969,4.897,6.543,8.941,13.344,
     &19.378,24.719,27.641,28.022,27.060,25.949,
     &24.369,22.794,21.839,20.346,18.198,15.525,
     &13.252,11.146,9.399,8.454,7.712,-999.000,
     &7.675,-999.000,7.866,-999.000,10.498,12.856,
     &15.664,13.752,9.216,-999.000,-1.024,-999.000,
     &-7.040,-999.000,-9.326,-999.000,-12.363,-999.000,
     &-15.168,-17.337,-18.943,-18.264,-17.977
     &/
       data (fjrocvz(it,22,3,11), it=1,59) /
     &-16.721,-14.645,-12.523,-11.688,-12.043,-14.964,
     &-17.366,-20.782,-26.547,-30.584,-28.152,-17.561,
     &-5.759,1.099,2.792,4.173,5.827,9.283,
     &13.855,18.100,21.552,21.913,21.134,20.838,
     &20.394,19.130,17.940,16.125,13.629,11.232,
     &9.349,8.169,7.031,7.125,8.182,-999.000,
     &8.687,-999.000,8.559,-999.000,12.422,13.713,
     &17.806,16.704,9.923,-999.000,0.584,-999.000,
     &-5.640,-999.000,-9.462,-999.000,-12.152,-999.000,
     &-14.598,-16.555,-18.206,-17.874,-16.721
     &/
       data (fjrocvz(it,23,3,11), it=1,59) /
     &-17.672,-14.841,-12.270,-10.553,-10.602,-14.058,
     &-17.511,-20.780,-24.317,-26.061,-22.217,-12.250,
     &-1.138,3.009,3.078,3.591,4.288,6.985,
     &10.313,14.922,19.612,21.116,20.053,18.222,
     &16.563,14.347,13.026,11.662,9.450,7.311,
     &6.611,6.117,5.005,5.375,6.878,-999.000,
     &8.540,-999.000,11.026,-999.000,16.705,21.058,
     &24.785,22.555,14.561,-999.000,1.460,-999.000,
     &-5.478,-999.000,-10.000,-999.000,-12.483,-999.000,
     &-15.065,-17.193,-18.611,-18.478,-17.672
     &/
       data (fjrocvz(it,24,3,11), it=1,59) /
     &-19.263,-17.679,-15.322,-12.357,-9.965,-11.583,
     &-13.731,-16.538,-21.255,-22.488,-18.121,-7.215,
     &3.209,6.134,7.057,7.781,7.352,8.712,
     &10.871,14.320,18.415,19.265,18.495,16.906,
     &13.961,11.122,9.804,9.970,9.031,8.143,
     &7.882,7.442,6.803,7.003,7.904,-999.000,
     &12.351,-999.000,16.614,-999.000,23.659,28.724,
     &29.931,26.613,18.154,-999.000,0.326,-999.000,
     &-7.117,-999.000,-10.554,-999.000,-12.686,-999.000,
     &-15.581,-18.222,-19.590,-19.419,-19.263
     &/
       data (fjrocvz(it,25,3,11), it=1,59) /
     &-19.178,-19.000,-18.076,-16.449,-13.661,-13.210,
     &-13.511,-17.126,-24.690,-26.696,-19.880,-6.284,
     &4.943,8.492,10.910,12.684,12.015,11.765,
     &12.471,15.432,19.492,20.566,20.071,17.730,
     &14.542,12.487,12.808,13.585,13.096,12.008,
     &11.200,11.018,10.625,10.518,11.367,-999.000,
     &16.994,-999.000,20.703,-999.000,26.471,31.206,
     &31.822,25.317,13.271,-999.000,-4.108,-999.000,
     &-10.802,-999.000,-12.683,-999.000,-13.742,-999.000,
     &-15.231,-17.845,-19.617,-19.145,-19.178
     &/
       data (fjrocvz(it, 1,4,11), it=1,59) /
     &-23.986,-23.073,-21.531,-20.437,-21.186,-22.291,
     &-23.041,-24.070,-24.944,-24.265,-20.970,-14.196,
     &-15.492,-9.883,-3.865,2.840,9.655,19.028,
     &26.299,29.365,29.553,28.831,26.640,22.773,
     &19.160,16.891,12.374,9.248,10.352,12.452,
     &13.423,14.060,15.235,17.707,21.134,-999.000,
     &25.289,-999.000,31.284,37.891,39.637,34.106,
     &24.163,-999.000,1.977,-999.000,-14.765,-999.000,
     &-23.003,-999.000,-26.174,-999.000,-28.853,-999.000,
     &-28.506,-28.272,-27.866,-26.063,-23.986
     &/
       data (fjrocvz(it, 2,4,11), it=1,59) /
     &-22.686,-21.917,-20.986,-20.515,-19.770,-19.695,
     &-20.167,-22.051,-23.851,-23.415,-18.175,-10.387,
     &-5.039,-0.155,5.017,8.177,13.881,20.035,
     &25.878,30.019,31.693,31.427,28.974,25.028,
     &21.512,19.917,16.931,14.049,13.645,14.413,
     &15.908,17.086,18.623,21.239,23.877,-999.000,
     &26.441,-999.000,32.541,36.613,39.709,39.070,
     &32.996,-999.000,3.568,-999.000,-19.575,-999.000,
     &-25.443,-999.000,-31.184,-999.000,-32.734,-999.000,
     &-30.084,-28.915,-26.397,-23.936,-22.686
     &/
       data (fjrocvz(it, 3,4,11), it=1,59) /
     &-22.801,-21.582,-20.909,-19.769,-18.749,-19.551,
     &-19.482,-19.932,-21.189,-20.619,-16.169,-9.740,
     &-3.877,1.368,4.547,8.394,13.898,18.335,
     &22.316,25.576,28.070,28.244,25.778,22.486,
     &20.105,19.142,17.503,15.889,15.450,15.988,
     &17.376,18.720,20.829,22.402,24.882,-999.000,
     &28.608,-999.000,32.124,38.045,42.470,42.135,
     &33.958,17.349,2.852,-13.900,-24.932,-26.403,
     &-28.675,-999.000,-33.646,-999.000,-33.654,-999.000,
     &-31.758,-29.073,-25.998,-24.272,-22.801
     &/
       data (fjrocvz(it, 4,4,11), it=1,59) /
     &-23.850,-22.619,-20.988,-19.363,-18.904,-19.579,
     &-19.170,-18.071,-17.980,-17.184,-14.132,-8.649,
     &-2.635,1.238,3.120,6.410,11.385,16.947,
     &21.097,23.778,24.782,24.062,21.811,19.097,
     &17.389,16.515,15.181,14.579,15.613,16.381,
     &17.558,19.365,21.225,23.749,27.631,-999.000,
     &30.218,-999.000,34.263,40.348,42.547,42.962,
     &39.107,23.976,3.172,-17.947,-27.864,-999.000,
     &-30.955,-999.000,-34.630,-999.000,-35.519,-999.000,
     &-32.561,-29.099,-26.435,-24.797,-23.850
     &/
       data (fjrocvz(it, 5,4,11), it=1,59) /
     &-23.650,-22.225,-19.890,-18.173,-17.869,-19.236,
     &-19.412,-18.559,-18.649,-17.880,-14.930,-8.773,
     &-2.904,1.426,4.759,7.556,12.550,18.845,
     &23.736,26.812,28.022,27.282,24.432,21.339,
     &19.301,18.333,17.286,17.056,17.543,17.576,
     &18.487,20.111,22.333,24.944,27.159,-999.000,
     &31.014,-999.000,35.016,39.704,43.365,46.245,
     &43.258,25.728,1.642,-14.529,-27.641,-31.827,
     &-35.584,-38.400,-39.340,-38.080,-35.522,-999.000,
     &-31.683,-28.487,-25.853,-24.424,-23.650
     &/
       data (fjrocvz(it, 6,4,11), it=1,59) /
     &-24.503,-23.640,-22.172,-20.785,-20.475,-21.053,
     &-20.209,-19.775,-20.027,-18.228,-16.043,-14.638,
     &-11.343,-3.678,3.629,9.466,14.950,20.951,
     &25.746,28.777,30.971,30.802,28.758,25.791,
     &23.828,22.641,21.130,19.639,19.095,18.322,
     &18.033,19.935,22.426,23.980,26.527,-999.000,
     &30.484,-999.000,34.428,42.112,48.742,50.775,
     &42.674,25.500,4.761,-12.195,-25.732,-32.912,
     &-36.770,-999.000,-38.622,-999.000,-36.433,-999.000,
     &-32.495,-28.686,-26.439,-25.531,-24.503
     &/
       data (fjrocvz(it, 7,4,11), it=1,59) /
     &-26.737,-26.554,-25.619,-24.791,-24.416,-23.674,
     &-22.369,-21.678,-21.144,-19.957,-17.739,-15.289,
     &-9.954,-4.642,0.438,5.022,11.287,17.940,
     &23.562,27.990,30.430,30.908,29.864,27.760,
     &25.891,23.916,21.783,20.064,18.304,16.220,
     &15.588,16.531,18.125,20.744,23.126,-999.000,
     &25.994,-999.000,30.829,39.745,47.673,51.661,
     &39.438,18.852,-0.117,-13.496,-26.103,-35.066,
     &-40.591,-41.168,-40.237,-38.927,-37.139,-999.000,
     &-33.042,-30.430,-28.411,-27.081,-26.737
     &/
       data (fjrocvz(it, 8,4,11), it=1,59) /
     &-26.805,-25.832,-25.126,-24.868,-24.582,-24.839,
     &-24.579,-24.276,-23.706,-25.267,-26.960,-28.524,
     &-22.985,-13.258,-6.448,0.274,5.989,10.869,
     &17.287,22.374,25.918,26.525,25.993,24.849,
     &23.281,21.598,19.402,17.288,14.763,12.525,
     &11.546,11.887,13.390,15.638,17.600,-999.000,
     &21.719,-999.000,28.418,35.062,42.695,43.922,
     &29.821,14.018,-0.703,-13.886,-23.866,-31.452,
     &-36.326,-36.385,-33.839,-999.000,-33.059,-999.000,
     &-31.236,-30.607,-29.188,-27.873,-26.805
     &/
       data (fjrocvz(it, 9,4,11), it=1,59) /
     &-24.156,-22.270,-20.543,-19.518,-19.669,-20.890,
     &-22.748,-24.389,-25.785,-27.221,-29.659,-31.430,
     &-25.798,-14.440,-5.774,0.176,3.814,8.295,
     &13.637,17.767,20.490,21.245,21.098,19.770,
     &17.552,15.542,13.689,11.751,9.682,8.290,
     &8.154,8.723,9.822,11.813,14.148,-999.000,
     &17.788,-999.000,25.577,33.298,40.396,44.469,
     &38.875,24.045,8.945,-4.505,-14.500,-999.000,
     &-21.284,-999.000,-24.534,-999.000,-26.760,-999.000,
     &-26.750,-26.614,-26.593,-25.947,-24.156
     &/
       data (fjrocvz(it,10,4,11), it=1,59) /
     &-25.501,-23.012,-20.603,-19.273,-18.394,-19.497,
     &-22.580,-25.221,-26.636,-27.235,-27.779,-27.428,
     &-23.017,-11.451,-3.018,2.664,5.779,10.496,
     &15.952,19.584,20.849,20.763,19.668,18.171,
     &16.340,13.865,11.588,9.712,8.557,7.935,
     &7.973,8.292,8.280,9.459,11.828,-999.000,
     &16.634,-999.000,25.432,34.282,43.944,48.115,
     &46.110,31.653,17.267,-999.000,-1.591,-999.000,
     &-12.594,-999.000,-20.440,-999.000,-22.458,-999.000,
     &-24.262,-24.996,-25.771,-25.997,-25.501
     &/
       data (fjrocvz(it,11,4,11), it=1,59) /
     &-28.673,-27.446,-26.785,-26.490,-25.905,-25.831,
     &-26.337,-27.561,-28.810,-30.620,-29.851,-25.977,
     &-20.306,-10.502,-2.885,3.208,7.421,11.668,
     &16.480,20.591,22.960,23.563,22.846,21.611,
     &20.599,18.533,15.754,13.807,12.567,11.581,
     &10.924,10.167,9.611,10.805,13.797,-999.000,
     &18.938,-999.000,26.115,34.263,44.084,46.204,
     &43.004,31.590,20.071,-999.000,-0.922,-999.000,
     &-13.720,-999.000,-22.578,-999.000,-24.569,-999.000,
     &-27.741,-27.461,-28.224,-29.323,-28.673
     &/
       data (fjrocvz(it,12,4,11), it=1,59) /
     &-29.958,-30.155,-29.798,-29.447,-29.165,-28.270,
     &-27.948,-29.350,-31.688,-33.946,-34.741,-31.514,
     &-27.386,-18.192,-8.683,-3.013,1.178,7.732,
     &15.146,20.832,24.428,25.393,24.874,24.221,
     &23.756,21.879,19.105,16.947,16.252,15.544,
     &14.628,13.385,12.507,13.811,17.330,-999.000,
     &21.892,-999.000,28.543,35.406,43.612,49.579,
     &46.240,28.636,13.151,1.397,-9.267,-999.000,
     &-16.230,-999.000,-22.659,-999.000,-26.991,-999.000,
     &-29.835,-29.732,-29.918,-30.092,-29.958
     &/
       data (fjrocvz(it,13,4,11), it=1,59) /
     &-27.544,-26.726,-26.333,-26.155,-25.518,-25.851,
     &-27.239,-28.400,-29.507,-30.447,-30.998,-29.642,
     &-27.026,-19.259,-10.888,-4.677,-1.113,6.385,
     &16.477,23.104,25.995,26.887,26.500,26.060,
     &25.598,23.635,21.105,19.531,19.820,19.854,
     &19.265,18.105,16.979,17.862,20.781,-999.000,
     &24.517,-999.000,33.350,39.699,48.271,54.606,
     &48.345,30.223,9.255,-4.529,-13.888,-16.095,
     &-17.328,-999.000,-21.185,-999.000,-24.045,-999.000,
     &-26.287,-28.202,-28.018,-27.514,-27.544
     &/
       data (fjrocvz(it,14,4,11), it=1,59) /
     &-24.886,-24.275,-23.849,-23.182,-22.988,-24.566,
     &-27.318,-30.861,-33.989,-34.047,-31.291,-27.430,
     &-23.839,-20.637,-13.855,-6.202,-0.862,7.055,
     &17.106,23.475,26.779,27.634,27.020,26.175,
     &25.074,23.746,21.966,21.001,21.635,21.947,
     &21.233,20.255,19.513,20.259,23.183,-999.000,
     &28.404,-999.000,37.411,47.815,54.674,54.008,
     &45.459,29.618,11.249,-999.000,-12.958,-999.000,
     &-20.827,-999.000,-22.567,-999.000,-24.877,-999.000,
     &-27.256,-27.983,-26.978,-25.599,-24.886
     &/
       data (fjrocvz(it,15,4,11), it=1,59) /
     &-22.238,-21.776,-21.242,-20.808,-20.872,-22.739,
     &-26.792,-32.949,-37.345,-38.750,-35.395,-28.255,
     &-23.509,-18.145,-12.539,-4.843,2.074,8.930,
     &16.826,22.415,25.325,25.669,24.275,22.127,
     &20.366,19.333,17.727,16.789,17.119,17.409,
     &17.066,16.617,17.186,18.899,22.632,-999.000,
     &27.571,-999.000,35.547,45.307,52.901,55.036,
     &47.285,29.281,11.189,-999.000,-14.448,-999.000,
     &-21.963,-999.000,-25.372,-999.000,-28.121,-999.000,
     &-29.428,-29.522,-26.947,-23.711,-22.238
     &/
       data (fjrocvz(it,16,4,11), it=1,59) /
     &-21.737,-20.787,-21.077,-21.419,-22.387,-24.952,
     &-28.907,-33.828,-37.893,-40.699,-39.583,-32.041,
     &-22.507,-16.259,-9.950,-2.959,4.121,9.837,
     &17.295,22.432,24.520,25.058,23.925,21.010,
     &17.710,15.032,12.825,11.662,11.996,12.792,
     &12.579,12.062,12.648,14.153,17.215,-999.000,
     &22.291,-999.000,30.014,39.389,47.800,49.211,
     &40.129,21.089,5.640,-999.000,-14.506,-999.000,
     &-23.215,-999.000,-27.959,-999.000,-30.809,-999.000,
     &-32.057,-31.408,-28.709,-24.856,-21.737
     &/
       data (fjrocvz(it,17,4,11), it=1,59) /
     &-25.355,-25.091,-24.643,-24.446,-25.598,-27.935,
     &-31.384,-36.281,-40.712,-42.042,-40.196,-31.975,
     &-22.807,-15.487,-8.154,-1.963,4.664,10.703,
     &18.669,24.153,26.862,28.048,27.219,24.686,
     &20.905,17.342,13.892,11.522,11.477,11.773,
     &11.155,10.183,9.563,9.914,12.854,-999.000,
     &18.860,-999.000,27.464,34.752,40.512,43.475,
     &36.263,18.560,3.059,-12.188,-22.055,-26.211,
     &-28.429,-999.000,-30.168,-999.000,-32.428,-999.000,
     &-33.278,-31.997,-30.263,-27.653,-25.355
     &/
       data (fjrocvz(it,18,4,11), it=1,59) /
     &-27.523,-26.843,-25.977,-25.969,-26.464,-28.001,
     &-31.990,-36.959,-40.518,-41.677,-38.975,-31.528,
     &-22.220,-14.762,-8.944,-2.556,4.720,12.704,
     &20.271,26.402,30.911,32.228,30.327,26.736,
     &23.572,20.610,16.776,13.426,12.365,11.554,
     &10.588,9.294,8.605,8.664,11.107,-999.000,
     &16.897,21.099,26.690,34.067,39.617,37.519,
     &30.665,17.181,1.729,-999.000,-21.430,-999.000,
     &-29.215,-999.000,-32.743,-999.000,-34.180,-999.000,
     &-34.157,-33.799,-32.378,-29.798,-27.523
     &/
       data (fjrocvz(it,19,4,11), it=1,59) /
     &-26.388,-25.493,-25.683,-25.463,-25.527,-27.024,
     &-30.096,-33.167,-36.238,-37.525,-34.749,-27.802,
     &-19.473,-13.865,-8.152,-1.756,5.163,14.060,
     &22.834,29.912,35.109,36.205,33.956,30.331,
     &26.742,23.334,19.500,16.127,15.550,15.110,
     &13.783,12.144,11.224,10.964,13.005,-999.000,
     &19.740,-999.000,29.197,37.633,41.681,39.723,
     &33.990,21.960,0.930,-13.317,-24.073,-28.360,
     &-30.025,-999.000,-32.355,-999.000,-33.076,-999.000,
     &-33.364,-33.191,-32.126,-29.331,-26.388
     &/
       data (fjrocvz(it,20,4,11), it=1,59) /
     &-27.495,-26.559,-25.139,-23.663,-22.976,-23.925,
     &-26.412,-29.867,-32.341,-32.919,-30.336,-21.258,
     &-13.136,-10.182,-5.182,-0.276,5.370,14.396,
     &22.865,30.780,36.189,37.666,36.090,32.992,
     &29.874,27.140,23.265,20.077,19.428,18.531,
     &16.993,15.336,14.278,14.082,16.663,-999.000,
     &23.145,-999.000,31.151,39.320,46.103,49.428,
     &39.473,22.470,0.478,-10.428,-18.881,-25.388,
     &-29.732,-999.000,-31.764,-999.000,-33.147,-999.000,
     &-33.249,-32.977,-32.215,-29.869,-27.495
     &/
       data (fjrocvz(it,21,4,11), it=1,59) /
     &-26.987,-24.827,-23.051,-21.952,-21.420,-21.832,
     &-23.295,-26.331,-29.820,-30.540,-27.753,-20.501,
     &-15.103,-9.573,-4.171,1.388,7.567,14.562,
     &20.825,27.884,32.430,34.524,33.115,29.944,
     &27.143,25.245,22.111,18.581,17.394,16.381,
     &15.016,14.169,14.325,14.704,16.850,-999.000,
     &22.229,-999.000,31.614,38.552,45.501,44.031,
     &34.791,-999.000,4.363,-999.000,-14.912,-999.000,
     &-25.907,-999.000,-30.710,-999.000,-32.126,-999.000,
     &-32.529,-31.927,-31.072,-29.297,-26.987
     &/
       data (fjrocvz(it,22,4,11), it=1,59) /
     &-25.798,-24.622,-23.428,-22.543,-22.363,-22.574,
     &-23.022,-25.453,-29.663,-30.424,-27.370,-20.446,
     &-12.247,-8.752,-3.029,2.093,7.453,13.652,
     &20.265,26.358,29.529,30.487,28.777,25.194,
     &21.251,18.394,15.403,12.863,12.106,11.538,
     &11.065,11.781,13.324,14.303,16.746,-999.000,
     &22.257,-999.000,32.627,36.176,40.412,35.792,
     &26.966,-999.000,1.482,-999.000,-16.127,-999.000,
     &-27.066,-999.000,-29.708,-999.000,-30.264,-999.000,
     &-31.095,-30.575,-29.689,-27.759,-25.798
     &/
       data (fjrocvz(it,23,4,11), it=1,59) /
     &-24.665,-24.053,-22.471,-20.432,-19.691,-20.600,
     &-22.451,-24.786,-26.939,-27.199,-24.400,-18.024,
     &-12.435,-9.381,-3.880,1.457,6.874,12.642,
     &20.847,26.694,28.768,28.441,26.131,22.386,
     &18.427,15.012,11.911,9.860,9.787,9.826,
     &9.774,10.057,12.094,14.437,17.678,-999.000,
     &21.891,-999.000,31.564,36.927,40.570,36.567,
     &28.027,13.217,-0.036,-999.000,-16.244,-999.000,
     &-23.861,-999.000,-27.306,-999.000,-28.593,-999.000,
     &-29.791,-29.580,-28.497,-25.968,-24.665
     &/
       data (fjrocvz(it,24,4,11), it=1,59) /
     &-23.818,-22.172,-20.345,-18.174,-17.839,-19.599,
     &-21.733,-23.010,-24.065,-24.497,-22.284,-17.321,
     &-11.032,-5.711,-2.064,1.401,6.420,14.953,
     &23.283,27.050,28.082,27.581,24.796,20.717,
     &16.950,13.925,10.449,8.281,9.304,10.894,
     &11.476,11.429,13.181,15.834,19.039,-999.000,
     &22.686,-999.000,31.227,39.549,42.312,39.054,
     &26.531,-999.000,1.961,-999.000,-12.829,-999.000,
     &-21.126,-999.000,-26.820,-999.000,-27.554,-999.000,
     &-28.556,-28.701,-27.812,-25.858,-23.818
     &/
       data (fjrocvz(it,25,4,11), it=1,59) /
     &-23.986,-23.073,-21.531,-20.437,-21.186,-22.291,
     &-23.041,-24.070,-24.944,-24.265,-20.970,-14.196,
     &-15.492,-9.883,-3.865,2.840,9.655,19.028,
     &26.299,29.365,29.553,28.831,26.640,22.773,
     &19.160,16.891,12.374,9.248,10.352,12.452,
     &13.423,14.060,15.235,17.707,21.134,-999.000,
     &25.289,-999.000,31.284,37.891,39.637,34.106,
     &24.163,-999.000,1.977,-999.000,-14.765,-999.000,
     &-23.003,-999.000,-26.174,-999.000,-28.853,-999.000,
     &-28.506,-28.272,-27.866,-26.063,-23.986
     &/

      MSOL=11
      MSN=4
      MLON1=25
      MLT1=59
      MLT=58
c 
      do 40 is= 1,MSOL
        do 40 isn= 1,MSN
          do 40 il= 1,MLON1
            do 30 it= 1,MLT1
              vzmx=fjrocvz(it,il,isn,is)
	          if(vzmx.lt.-900. ) then
	            itm=it-1
	            if(itm.lt.1) itm=itm+MLT
	            vzmx=fjrocvz(itm,il,isn,is)
	            endif
	            vzm(it,il,isn,is)=vzmx
30	          continue
40      continue
	return
	end subroutine vfjmodelrocstart
