## @file

import os
import numpy as np
from math import sin, cos, pi
from pkg_resources import resource_filename

from gym import core, spaces, logger
from gym.utils import seeding

from jiminy_py import core as jiminy
from jiminy_py.engine_asynchronous import EngineAsynchronous

from ..common.robots import RobotJiminyEnv

DT = 2.0e-3      ## Stepper update period
MAX_FORCE = 40.0 ## Max force of the motor

class JiminyCartPoleEnv(RobotJiminyEnv):
    """
    @brief      Implementation of a Gym environment for the Cartpole which is using
                Jiminy Engine to perform physics computations and Gepetto-viewer for
                rendering. It is a specialization of RobotJiminyGoalEnv. The Cartpole
                is a pole attached by an un-actuated joint to a cart. The goal is to
                prevent the pendulum from falling over by increasing and reducing the
                cart's velocity.

    @details    **OBSERVATION:**
                Type: Box(4)
                Num	Observation                Min         Max
                0	Cart Position             -1.5         1.5
                1	Cart Velocity             -Inf         Inf
                2	Pole Angle                -50 deg      50 deg
                3	Pole Velocity At Tip      -Inf         Inf

                **ACTIONS:**
                Type: Discrete(2)
                Num	Action
                0	Push cart to the left
                1	Push cart to the right

                Note that the amount the velocity that is reduced or increased is not
                fixed, it depends on the angle the pole is pointing. This is because the
                center of gravity of the pole increases the amount of energy needed to
                move the cart underneath it.

                **REWARD:**
                Reward is 1 for every step taken, including the termination step.
                move the cart underneath it.

                **STARTING STATE:**
                All observations are assigned a uniform random value in [-0.05..0.05]

                **EPISODE TERMINATION:**
                If any of these conditions is satisfied:
                    - Pole Angle is more than 25 degrees Cart
                    - Position is more than 75 cm
                    - Episode length is greater than 200

                **SOLVED REQUIREMENTS:**
                Considered solved when the average reward is greater than or equal to
                195.0 over 100 consecutive trials.
    """

    ## @var metadata
    # @copydoc RobotJiminyEnv::metadata

    metadata = {
        'render.modes': ['human']
    }

    def __init__(self):
        """
        @brief      Constructor

        @return     Instance of the environment.
        """

        #  @copydoc RobotJiminyEnv::__init__
        # ## @var state_random_high
        #  @copydoc RobotJiminyEnv::state_random_high
        ## @var state_random_low
        #  @copydoc RobotJiminyEnv::state_random_low

        # ############################### Initialize Jiminy ####################################

        os.environ["JIMINY_MESH_PATH"] = resource_filename('gym_jiminy.envs', 'data')
        urdf_path = os.path.join(os.environ["JIMINY_MESH_PATH"], "cartpole/cartpole.urdf")

        robot = jiminy.Robot()
        robot.initialize(urdf_path)

        motor_joint_name = "slider_to_cart"
        encoder_sensors_def = {"slider": "slider_to_cart", "pole": "cart_to_pole"}
        motor = jiminy.SimpleMotor(motor_joint_name)
        robot.attach_motor(motor)
        motor.initialize(motor_joint_name)
        for sensor_name, joint_name in encoder_sensors_def.items():
            encoder = jiminy.EncoderSensor(sensor_name)
            robot.attach_sensor(encoder)
            encoder.initialize(joint_name)

        engine_py = EngineAsynchronous(robot)

        # ############################### Configure Jiminy #####################################

        robot_options = robot.get_options()

        # Set the effort limit of the motor
        robot_options["motors"][motor_joint_name]["effortLimitFromUrdf"] = False
        robot_options["motors"][motor_joint_name]["effortLimit"] = MAX_FORCE

        robot.set_options(robot_options)

        # ##################### Define some problem-specific variables #########################

        ## Map between discrete actions and actual motor force
        self.AVAIL_FORCE = [-MAX_FORCE, MAX_FORCE]

        ## Maximum absolute angle of the pole before considering the episode failed
        self.theta_threshold_radians = 25 * pi / 180

        ## Maximum absolute position of the cart before considering the episode failed
        self.x_threshold = 0.75

        # Bounds of the hypercube associated with the initial state of the robot
        self.state_random_high = np.array([0.5, 0.15, 0.1, 0.1])
        self.state_random_low = -self.state_random_high

        # ####################### Configure the learning environment ###########################

        super().__init__("cartpole", engine_py, DT)

        # #################### Overwrite some problem-generic variables ########################

        # Replace the observation space, which is the spa space instead of the sensor space.
        # Note that the Angle limit set to 2 * theta_threshold_radians, thus observations of
        # failure are still within bounds.
        high = np.array([self.x_threshold * 2,
                         self.theta_threshold_radians * 2,
                         *robot.velocity_limit])

        self.observation_space = spaces.Box(low=-high, high=high, dtype=np.float64)

        self.action_space = spaces.Discrete(2) # Force using a discrete action space

        ## Current observation of the robot
        self.observation = None

    def _sample_state(self):
        # @copydoc RobotJiminyEnv::_sample_state
        return self.np_random.uniform(low=self.state_random_low,
                                      high=self.state_random_high)

    def _update_observation(self, obs):
        # @copydoc RobotJiminyEnv::_update_observation
        obs = self.engine_py.state

    def _is_done(self):
        # @copydoc RobotJiminyEnv::_is_done
        x, theta, _, _ = self.observation
        return        x < -self.x_threshold \
               or     x >  self.x_threshold \
               or theta < -self.theta_threshold_radians \
               or theta >  self.theta_threshold_radians

    def _compute_reward(self):
        # @copydoc RobotJiminyEnv::_compute_reward

        # Add a small positive reward as long as the terminal condition
        # has never been reached during the same episode.
        reward = 0.0
        if self._steps_beyond_done is None:
            done = self._is_done()
            if not done:
                reward += 1.0
        return reward

    def step(self, action):
        # @copydoc RobotJiminyEnv::step
        assert self.action_space.contains(action), "%r (%s) invalid" % (action, type(action))

        # Compute the force to apply
        action = self.AVAIL_FORCE[action]

        # Perform the step
        return super().step(action)
