"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceFactory = void 0;
const table_1 = require("cloudform-types/types/dynamoDb/table");
const resolver_1 = __importDefault(require("cloudform-types/types/appSync/resolver"));
const cloudform_types_1 = require("cloudform-types");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_transformer_core_1 = require("graphql-transformer-core");
class ResourceFactory {
    makeParams() {
        return {};
    }
    initTemplate() {
        return {
            Parameters: this.makeParams(),
            Resources: {},
            Outputs: {},
        };
    }
    updateTableForConnection(table, connectionName, connectionAttributeName, sortField = null) {
        const gsis = table.Properties.GlobalSecondaryIndexes || [];
        if (gsis.length >= 20) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`Cannot create connection ${connectionName}. Table ${table.Properties.TableName} out of GSI capacity.`);
        }
        const connectionGSIName = `gsi-${connectionName}`;
        const existingGSI = gsis.find(gsi => gsi.IndexName === connectionGSIName);
        if (!existingGSI) {
            const keySchema = [new table_1.KeySchema({ AttributeName: connectionAttributeName, KeyType: 'HASH' })];
            if (sortField) {
                keySchema.push(new table_1.KeySchema({ AttributeName: sortField.name, KeyType: 'RANGE' }));
            }
            gsis.push(new table_1.GlobalSecondaryIndex({
                IndexName: connectionGSIName,
                KeySchema: keySchema,
                Projection: new table_1.Projection({
                    ProjectionType: 'ALL',
                }),
                ProvisionedThroughput: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePayPerRequestBilling, cloudform_types_1.Refs.NoValue, {
                    ReadCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableReadIOPS),
                    WriteCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableWriteIOPS),
                }),
            }));
        }
        const attributeDefinitions = table.Properties.AttributeDefinitions;
        const existingAttribute = attributeDefinitions.find(attr => attr.AttributeName === connectionAttributeName);
        if (!existingAttribute) {
            attributeDefinitions.push(new table_1.AttributeDefinition({
                AttributeName: connectionAttributeName,
                AttributeType: 'S',
            }));
        }
        if (sortField) {
            const existingSortAttribute = attributeDefinitions.find(attr => attr.AttributeName === sortField.name);
            if (!existingSortAttribute) {
                const scalarType = graphql_transformer_common_1.DEFAULT_SCALARS[sortField.type];
                const attributeType = scalarType === 'String' ? 'S' : 'N';
                attributeDefinitions.push(new table_1.AttributeDefinition({ AttributeName: sortField.name, AttributeType: attributeType }));
            }
        }
        table.Properties.GlobalSecondaryIndexes = gsis;
        table.Properties.AttributeDefinitions = attributeDefinitions;
        return table;
    }
    makeGetItemConnectionResolver(type, field, relatedType, connectionAttribute, idFieldName, sortFieldInfo) {
        let keyObj = graphql_mapping_template_1.obj({
            [`${idFieldName}`]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNullOrBlank($ctx.source.${connectionAttribute}, "${graphql_transformer_common_1.NONE_VALUE}"))`),
        });
        if (sortFieldInfo) {
            if (sortFieldInfo.sortFieldIsStringLike) {
                keyObj.attributes.push([
                    sortFieldInfo.primarySortFieldName,
                    graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNullOrBlank($ctx.source.${sortFieldInfo.sortFieldName}, "${graphql_transformer_common_1.NONE_VALUE}"))`),
                ]);
            }
            else {
                keyObj.attributes.push([
                    sortFieldInfo.primarySortFieldName,
                    graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNull($ctx.source.${sortFieldInfo.sortFieldName}, ${graphql_transformer_common_1.NONE_INT_VALUE}))`),
                ]);
            }
        }
        return new resolver_1.default({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(relatedType), 'Name'),
            FieldName: field,
            TypeName: type,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.ifElse(graphql_mapping_template_1.or([
                graphql_mapping_template_1.raw(`$util.isNull($ctx.source.${connectionAttribute})`),
                ...(sortFieldInfo ? [graphql_mapping_template_1.raw(`$util.isNull($ctx.source.${connectionAttribute})`)] : []),
            ]), graphql_mapping_template_1.raw('#return'), graphql_mapping_template_1.DynamoDBMappingTemplate.getItem({
                key: keyObj,
            }))),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(false)),
        }).dependsOn(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID);
    }
    makeQueryConnectionResolver(type, field, relatedType, connectionAttribute, connectionName, idFieldName, sortKeyInfo, limit) {
        const pageLimit = limit || graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT;
        const setup = [
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${pageLimit})`)),
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('query'), graphql_mapping_template_1.obj({
                expression: graphql_mapping_template_1.str('#connectionAttribute = :connectionAttribute'),
                expressionNames: graphql_mapping_template_1.obj({
                    '#connectionAttribute': graphql_mapping_template_1.str(connectionAttribute),
                }),
                expressionValues: graphql_mapping_template_1.obj({
                    ':connectionAttribute': graphql_mapping_template_1.obj({
                        S: graphql_mapping_template_1.str(`$context.source.${idFieldName}`),
                    }),
                }),
            })),
        ];
        if (sortKeyInfo) {
            setup.push(graphql_transformer_common_1.applyKeyConditionExpression(sortKeyInfo.fieldName, sortKeyInfo.attributeType, 'query'));
        }
        return new resolver_1.default({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(relatedType), 'Name'),
            FieldName: field,
            TypeName: type,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`$util.isNull($context.source.${idFieldName})`), graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.set(graphql_mapping_template_1.ref('result'), graphql_mapping_template_1.obj({ items: graphql_mapping_template_1.list([]) })), graphql_mapping_template_1.raw('#return($result)')]), graphql_mapping_template_1.compoundExpression([
                ...setup,
                graphql_mapping_template_1.DynamoDBMappingTemplate.query({
                    query: graphql_mapping_template_1.raw('$util.toJson($query)'),
                    scanIndexForward: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.ifElse(graphql_mapping_template_1.equals(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.str('ASC')), graphql_mapping_template_1.bool(true), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.bool(true)),
                    filter: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.ref('util.transform.toDynamoDBFilterExpression($ctx.args.filter)'), graphql_mapping_template_1.nul()),
                    limit: graphql_mapping_template_1.ref('limit'),
                    nextToken: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.ref('util.toJson($context.args.nextToken)'), graphql_mapping_template_1.nul()),
                    index: graphql_mapping_template_1.str(`gsi-${connectionName}`),
                }),
            ]))),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(false, graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.iff(graphql_mapping_template_1.raw('!$result'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('result'), graphql_mapping_template_1.ref('ctx.result'))), graphql_mapping_template_1.raw('$util.toJson($result)')]))),
        }).dependsOn(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID);
    }
    makeGetItemConnectionWithKeyResolver(type, field, relatedType, connectionAttributes, keySchema) {
        const partitionKeyName = keySchema[0].AttributeName;
        let keyObj = graphql_mapping_template_1.obj({
            [partitionKeyName]: graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNullOrBlank($ctx.source.${connectionAttributes[0]}, "${graphql_transformer_common_1.NONE_VALUE}"))`),
        });
        if (connectionAttributes.length > 2) {
            const rangeKeyFields = connectionAttributes.slice(1);
            const sortKeyName = keySchema[1].AttributeName;
            const condensedSortKeyValue = this.condenseRangeKey(rangeKeyFields.map(keyField => `\${ctx.source.${keyField}}`));
            keyObj.attributes.push([
                sortKeyName,
                graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNullOrBlank("${condensedSortKeyValue}", "${graphql_transformer_common_1.NONE_VALUE}"))`),
            ]);
        }
        else if (connectionAttributes[1]) {
            const sortKeyName = keySchema[1].AttributeName;
            keyObj.attributes.push([
                sortKeyName,
                graphql_mapping_template_1.ref(`util.dynamodb.toDynamoDBJson($util.defaultIfNullOrBlank($ctx.source.${connectionAttributes[1]}, "${graphql_transformer_common_1.NONE_VALUE}"))`),
            ]);
        }
        return new resolver_1.default({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(relatedType), 'Name'),
            FieldName: field,
            TypeName: type,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.ifElse(graphql_mapping_template_1.or(connectionAttributes.map(ca => graphql_mapping_template_1.raw(`$util.isNull($ctx.source.${ca})`))), graphql_mapping_template_1.raw('#return'), graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.DynamoDBMappingTemplate.getItem({
                    key: keyObj,
                }),
            ]))),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(false)),
        }).dependsOn(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID);
    }
    makeQueryConnectionWithKeyResolver(type, field, relatedType, connectionAttributes, keySchema, indexName, limit) {
        const pageLimit = limit || graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT;
        const setup = [
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${pageLimit})`)),
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('query'), this.makeExpression(keySchema, connectionAttributes)),
        ];
        if (keySchema[1] && !connectionAttributes[1]) {
            const sortKeyField = relatedType.fields.find(f => f.name.value === keySchema[1].AttributeName);
            if (sortKeyField) {
                setup.push(graphql_transformer_common_1.applyKeyConditionExpression(String(keySchema[1].AttributeName), graphql_transformer_common_1.attributeTypeFromScalar(sortKeyField.type), 'query'));
            }
            else {
                setup.push(graphql_transformer_common_1.applyCompositeKeyConditionExpression(this.getSortKeyNames(String(keySchema[1].AttributeName)), 'query', this.makeCompositeSortKeyName(String(keySchema[1].AttributeName)), String(keySchema[1].AttributeName)));
            }
        }
        let queryArguments = {
            query: graphql_mapping_template_1.raw('$util.toJson($query)'),
            scanIndexForward: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.ifElse(graphql_mapping_template_1.equals(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.str('ASC')), graphql_mapping_template_1.bool(true), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.bool(true)),
            filter: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.ref('util.transform.toDynamoDBFilterExpression($ctx.args.filter)'), graphql_mapping_template_1.nul()),
            limit: graphql_mapping_template_1.ref('limit'),
            nextToken: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.ref('util.toJson($context.args.nextToken)'), graphql_mapping_template_1.nul()),
            index: indexName ? graphql_mapping_template_1.str(indexName) : undefined,
        };
        if (!indexName) {
            const indexArg = 'index';
            delete queryArguments[indexArg];
        }
        const queryObj = graphql_mapping_template_1.DynamoDBMappingTemplate.query(queryArguments);
        return new resolver_1.default({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(relatedType.name.value), 'Name'),
            FieldName: field,
            TypeName: type,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`$util.isNull($ctx.source.${connectionAttributes[0]})`), graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.set(graphql_mapping_template_1.ref('result'), graphql_mapping_template_1.obj({ items: graphql_mapping_template_1.list([]) })), graphql_mapping_template_1.raw('#return($result)')]), graphql_mapping_template_1.compoundExpression([...setup, queryObj]))),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(false, graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.iff(graphql_mapping_template_1.raw('!$result'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('result'), graphql_mapping_template_1.ref('ctx.result'))), graphql_mapping_template_1.raw('$util.toJson($result)')]))),
        }).dependsOn(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLSchemaLogicalID);
    }
    makeExpression(keySchema, connectionAttributes) {
        if (keySchema[1] && connectionAttributes[1]) {
            let condensedSortKeyValue = undefined;
            if (connectionAttributes.length > 2) {
                const rangeKeyFields = connectionAttributes.slice(1);
                condensedSortKeyValue = this.condenseRangeKey(rangeKeyFields.map(keyField => `\${context.source.${keyField}}`));
            }
            return graphql_mapping_template_1.obj({
                expression: graphql_mapping_template_1.str('#partitionKey = :partitionKey AND #sortKey = :sortKey'),
                expressionNames: graphql_mapping_template_1.obj({
                    '#partitionKey': graphql_mapping_template_1.str(String(keySchema[0].AttributeName)),
                    '#sortKey': graphql_mapping_template_1.str(String(keySchema[1].AttributeName)),
                }),
                expressionValues: graphql_mapping_template_1.obj({
                    ':partitionKey': graphql_mapping_template_1.obj({
                        S: graphql_mapping_template_1.str(`$context.source.${connectionAttributes[0]}`),
                    }),
                    ':sortKey': graphql_mapping_template_1.obj({
                        S: graphql_mapping_template_1.str(condensedSortKeyValue || `$context.source.${connectionAttributes[1]}`),
                    }),
                }),
            });
        }
        return graphql_mapping_template_1.obj({
            expression: graphql_mapping_template_1.str('#partitionKey = :partitionKey'),
            expressionNames: graphql_mapping_template_1.obj({
                '#partitionKey': graphql_mapping_template_1.str(String(keySchema[0].AttributeName)),
            }),
            expressionValues: graphql_mapping_template_1.obj({
                ':partitionKey': graphql_mapping_template_1.obj({
                    S: graphql_mapping_template_1.str(`$context.source.${connectionAttributes[0]}`),
                }),
            }),
        });
    }
    condenseRangeKey(fields) {
        return fields.join(graphql_transformer_common_1.ModelResourceIDs.ModelCompositeKeySeparator());
    }
    makeCompositeSortKeyName(sortKeyName) {
        const attributeNames = sortKeyName.split(graphql_transformer_common_1.ModelResourceIDs.ModelCompositeKeySeparator());
        return graphql_transformer_common_1.toCamelCase(attributeNames);
    }
    getSortKeyNames(compositeSK) {
        return compositeSK.split(graphql_transformer_common_1.ModelResourceIDs.ModelCompositeKeySeparator());
    }
}
exports.ResourceFactory = ResourceFactory;
//# sourceMappingURL=resources.js.map