"""AbstractDaemon class.
Retrieved from: http://www.jejik.com/articles/2007/02/
                    a_simple_unix_linux_daemon_in_python/ (2011.06.01)
Written by: Sandel Marechal (http://www.jejik.com/authors/sander_marechal/)

Adapted for Python 3 by Luke Sneeringer <luke@sneeringer.com>
"""
from __future__ import absolute_import, print_function, unicode_literals
import atexit
import os
import signal
import six
import sys
import time
 

class AbstractDaemon(object):
    """A generic daemon class.
       
    Usage: subclass the AbstractDaemon class and override the `run` method.
    """
    def __init__(self, pidfile, stdin='/dev/null', stdout='/dev/null',
                       stderr='/dev/null'):
        self.stdin = stdin
        self.stdout = stdout
        self.stderr = stderr
        self.pidfile = pidfile
       
    def daemonize(self):
        """Do the UNIX double-fork magic.

        See Stevens' "Advanced Programming in the UNIX Environment"
        for details (ISBN 0201563177).

        http://www.erlenstar.demon.co.uk/unix/faq_2.html#SEC16
        """
        try:
            pid = os.fork()
            if pid > 0:
                # Exit the first parent.
                sys.exit(0)
        except OSError as e:
            sys.stderr.write('fork #1 failed: %d (%s)\n' %
                             (e.errno, e.strerror))
            sys.exit(1)
       
        # Decouple from the parent environment.
        os.chdir('/')
        os.setsid()
        os.umask(0)
       
        # Do second fork.
        try:
            pid = os.fork()
            if pid > 0:
                # Exit from second parent.
                sys.exit(0)
        except OSError as e:
            sys.stderr.write('fork #2 failed: %d (%s)\n' %
                             (e.errno, e.strerror))
            sys.exit(1)
       
        # Redirect standard file descriptors.
        sys.stdout.flush()
        sys.stderr.flush()
        si = open(self.stdin, 'r')
        so = open(self.stdout, 'a+')
        se = open(self.stderr, 'a+')
        os.dup2(si.fileno(), sys.stdin.fileno())
        os.dup2(so.fileno(), sys.stdout.fileno())
        os.dup2(se.fileno(), sys.stderr.fileno())
       
        # Write the PID file.
        atexit.register(self.delpid)
        pid = str(os.getpid())
        open(self.pidfile, 'w+').write("%s\n" % pid)
       
    def delpid(self):
        os.remove(self.pidfile)
 
    def start(self):
        """
        Start the daemon
        """
        # Check for a pidfile to see if the daemon already runs.
        try:
            pf = open(self.pidfile,'r')
            pid = int(pf.read().strip())
            pf.close()
        except IOError:
            pid = None
       
        if pid:
            message = "pidfile %s already exists. Daemon already running?\n"
            sys.stderr.write(message % self.pidfile)
            sys.exit(1)
           
        # Start the daemon.
        self.daemonize()
        self.run()
 
    def stop(self):
        """
        Stop the daemon
        """
        # Get the PID from the PID file.
        try:
            pf = open(self.pidfile,'r')
            pid = int(pf.read().strip())
            pf.close()
        except IOError:
            pid = None
       
        if not pid:
            message = 'pidfile %s does not exist. Daemon not running?\n'
            sys.stderr.write(message % self.pidfile)
            return  # This is not an error in a restart case.
 
        # Try killing the daemon process.
        try:
            while 1:
                os.kill(pid, signal.SIGTERM)
                time.sleep(0.1)
        except OSError as err:
            err = str(err)
            if err.find('No such process') > 0:
                if os.path.exists(self.pidfile):
                    os.remove(self.pidfile)
            else:
                print(six.text_type(err))
                sys.exit(1)
 
    def restart(self):
        """Restart the daemon."""

        self.stop()
        self.start()
 
    def run(self):
        """You should override this method when you subclass AbstractDaemon.

        It will be called after the process has been daemonized by `start`
        or `restart`.
        """
