from types import SimpleNamespace
from collections import namedtuple
import dateutil.parser as dp
import warnings
import json

#########################
## Upgrade definitions ##
#########################

current_magic = "E!vIA5L86J2I"

dead_end_magic = []

UpgradeInstruction = namedtuple("UpgradeInstruction", ["result", "issues", "prs"])

upgrades = {
    "B8n2c@e8kvfx": UpgradeInstruction(
        result="E!vIA5L86J2I",
        issues=None,
        prs="""
            fragment prFields on PullRequest {
                number
                baseRepository { nameWithOwner }
                baseRefName
                baseRefOid
                headRepository { nameWithOwner }
                headRefName
                headRefOid
                mergeCommit { oid }
            }""",
    )
}


def get_current_magic():
    return current_magic


def is_current_magic(magic):
    return magic == current_magic


def is_legacy_magic(magic):
    return magic in upgrades


def is_dead_end_magic(magic):
    return magic in dead_end_magic


def is_valid_magic(magic):
    return is_current_magic(magic) or is_legacy_magic(magic) or is_dead_end_magic(magic)


def get_upgrade_instructions(magic):
    return upgrades[magic]


###############
## Functions ##
###############


def newReferenceFile():
    return SimpleNamespace(
        magic=current_magic,
        issues=dict(),
        prs=dict(),
        issues_only=True,
        lastSuccess=None,
        canCopy=False,
        filename=None,
    )


def loadReference(filename, repo):
    fileIsValid = False
    reference = newReferenceFile()
    reference.filename = filename

    if filename:
        try:
            with open(filename, "r") as ref_file:
                raw_reference = json.load(ref_file)

            fileIsValid = True
            for element in ("magic", "timestamp", "issues", "repo"):
                if element not in raw_reference:
                    fileIsValid = False
                    break

            if fileIsValid and is_valid_magic(raw_reference["magic"]):
                reference.magic = raw_reference["magic"]
            else:
                warnings.warn("Input file does not appear to be generated by this tool")
                fileIsValid = False

            if fileIsValid and raw_reference["repo"] != repo:
                warnings.warn("Input file was generated from a different repo")
                fileIsValid = False

            if fileIsValid:
                reference.lastSuccess = dp.parse(raw_reference["timestamp"])
                reference.issues = dict(
                    [(issue["number"], issue) for issue in raw_reference["issues"]]
                )
                if "pulls" in raw_reference:
                    reference.issues_only = False
                    reference.prs = dict(
                        [(pr["number"], pr) for pr in raw_reference["pulls"]]
                    )
                else:
                    reference.issues_only = True
                    reference.prs = dict()

        except:
            warnings.warn("Unable to read input file; proceeding without it")
            pass

    if not fileIsValid:
        return newReferenceFile()

    return reference
