# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['nipype_generate_fieldmaps']
install_requires = \
['nibabel>=3.0.0,<4.0.0', 'nipype>=1.1.0,<2.0.0']

entry_points = \
{'console_scripts': ['nipype-generate-fieldmaps = '
                     'nipype_generate_fieldmaps:cli']}

setup_kwargs = {
    'name': 'nipype-generate-fieldmaps',
    'version': '0.2.2',
    'description': 'Nipype workflow to generate fieldmaps from EPI acquisitions with differing phase-encoding directions',
    'long_description': "# nipype-generate-fieldmaps\n\nNipype workflow to generate fieldmaps from EPI acquisitions with differing phase-encoding directions\n\n## Installation\n\n```bash\npip install nipype-generate-fieldmaps\n```\n\n## Usage\n\n### As a stand-alone workflow\n\n```python\n# create the workflow\nwf = create_generate_fieldmaps_wf()\n\n# wire-up the inputs\nwf.inputs.inputnode.se_epi_pe1_file = my_se_epi_pe1_file  # type: str | Path\nwf.inputs.inputnode.se_epi_pe2_file = my_se_epi_pe2_file  # type: str | Path\nwf.inputs.inputnode.se_epi_pe1_sidecar_file = my_se_epi_pe1_sidecar_file  # type: str | Path\nwf.inputs.inputnode.se_epi_pe2_sidecar_file = my_se_epi_pe2_sidecar_file  # type: str | Path\n\n# set the output directory\nwf.base_dir = my_output_dir  # type: str | Path\n\n# run it\nwf.run()\n```\n\n### As a nested workflow\n\nThe nodes `node1`, `node2`, `some_other_node`, `maybe_a_4th_node`, `epi_node`, and `anat_node` are made up for demonstration purposes\n\n```python\nfrom nipype import Workflow\nfrom nipype.interfaces.fsl import EpiReg\nfrom nipype_generate_fieldmaps import create_generate_fieldmaps_wf\n\n# parent workflow defined elsewhere\nwf = Workflow(...)\n\n# create the (sub-)workflow\nfmap_wf = create_generate_fieldmaps_wf()\n\n# connect the various nodes form the parent workflow to the nested fieldmap workflow\nwf.connect(node1, 'out_file', fmap_wf, 'inputnode.se_epi_pe1_file')\nwf.connect(node2, 'out', fmap_wf, 'inputnode.se_epi_pe2_file')\nwf.connect(some_other_node, 'output_file', fmap_wf, 'inputnode.se_epi_pe1_sidecar_file')\nwf.connect(maybe_a_4th_node, 'sidecar_file', fmap_wf, 'inputnode.se_epi_pe2_sidecar_file')\n\n# connect the fieldmap workflow outputs to one (or more) node(s) in the parent workflow\n# for example: EpiReg()\nepireg = Node(EpiReg(out_base='epi2str.nii.gz'), name='epi_reg')\n# from elsewhere\nwf.connect(epi_node, 'my_epi_file' epireg, 'epi')\nwf.connect(anat_node, 'my_t1_file', epireg, 't1_head')\nwf.connect(anat_node, 'my_t1_brain_file', epireg, 't1_brain')\n# from the fieldmap workflow!\nwf.connect(fmap_wf, 'outputnode.fmap_rads_file', epireg, 'fmap')\nwf.connect(fmap_wf, 'outputnode.fmap_mag_file', epireg, 'fmapmag')\nwf.connect(fmap_wf, 'outputnode.fmap_mag_brain_file', epireg, 'fmapmagbrain')\nwf.connect(fmap_wf, 'outputnode.echospacing', epireg, 'echospacing')\nwf.connect(fmap_wf, 'outputnode.pedir', epireg, 'pedir')\n```\n\n### From the command line\n\n```bash\n$ nipype-generate-fieldmaps --help\nusage: nipype-generate-fieldmaps [-h] [-v] se_epi_pe1 se_epi_pe2 se_epi_pe1_sidecar se_epi_pe2_sidecar out_dir\n\nGenerate fieldmaps from EPI acquisitions with differing phase-encoding directions\n\npositional arguments:\n  se_epi_pe1          The spin-echo EPI file acquired in the 'first' phase-encoding direction\n  se_epi_pe2          The spin-echo EPI file acquired in the 'second' phase-encoding direction\n  se_epi_pe1_sidecar  The JSON sidecar for the first spin-echo EPI file\n  se_epi_pe2_sidecar  The JSON sidecar for the second spin-echo EPI file\n  out_dir             The directory into which outputs are written\n\noptional arguments:\n  -h, --help          show this help message and exit\n  -v, --version       show program's version number and exit\n```\n\n## Prerequisites\n\nThis workflow has a few requirements:\n\n1. There are **two** acquisitions (i.e. `.nii.gz` files) acquired with **different** phase encodings, usually opposite phase encodings but this need not be the case.\n2. The number of volumes in acquisition 1 (the first phase encoding direction) **equals** the number of volumes in acquisition 2 (the second phase encoding direction)\n3. Each acquisition has a JSON sidecar. Specifically, this workflow requires that _each_ sidecar contain one of the following sets of properties. These properties are listed in the order in which the workflow will search:\n\n   - `PhaseEncodingDirection` and `TotalReadoutTime`, or\n   - `PhaseEncodingDirection`, `ReconMatrixPE`, and `EffectiveEchoSpacing`, or\n   - `PhaseEncodingDirection`, `ReconMatrixPE`, and `BandwidthPerPixelPhaseEncode`\n\n   If either JSON sidecar fails to contain at least one of the above sets of parameters the workflow will produce an error.\n\n## I/O\n\nThis workflow requires 4 inputs to be connected to the node named `inputnode`:\n\n- **`se_epi_pe1_file`**\n\n  The spin-echo EPI file acquired in the 'first' phase-encoding direction\n\n- **`se_epi_pe2_file`**\n\n  The spin-echo EPI file acquired in the 'second' phase-encoding direction\n\n- **`se_epi_pe1_sidecar_file`**\n\n  The JSON sidecar for the first spin-echo EPI file\n\n- **`se_epi_pe2_sidecar_file`**\n\n  The JSON sidecar for the second spin-echo EPI file\n\nThis workflow also exposes the following outputs via the node named `outputnode`:\n\n- **`acq_params_file`**\n\n  The computed file passed to the `--datain` option of `topup`\n\n- **`corrected_se_epi_file`**\n\n  The `.nii.gz` image containing all _distortion corrected_ volumes from the two input acquisitions\n\n- **`fmap_hz_file`**\n\n  The fieldmap in hertz (Hz)\n\n- **`fmap_rads_file`**\n\n  The fieldmap in radians per second (rad/s)\n\n- **`fmap_mag_file`**\n\n  The 'magnitude' image (mean image) computed by averaging all volumes in `corrected_se_epi_file`\n\n- **`fmap_mag_brain_file`**\n\n  The result of applying brain-extraction to `fmap_mag_file`\n\n- **`fmap_mag_brain_mask_file`**\n\n  The brain mask produced during the brain-extraction of `fmap_mag_file`\n\n## Contributing\n\n1. Have or install a recent version of `poetry` (version >= 1.1)\n1. Fork the repo\n1. Setup a virtual environment (however you prefer)\n1. Run `poetry install`\n1. Run `pre-commit install`\n1. Add your changes\n1. Commit your changes + push to your fork\n1. Open a PR\n",
    'author': 'Andrew Ross',
    'author_email': 'andrew.ross.mail@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/andrewrosss/nipype-generate-fieldmaps',
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
