"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types,
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value,
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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