"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjenUpgrade = void 0;
const node_package_1 = require("./node-package");
const tasks_1 = require("./tasks");
/**
 * Checks for new versions of projen and creates a PR with an upgrade change.
 */
class ProjenUpgrade {
    constructor(project, options = {}) {
        var _a;
        const upgradeTask = project.addTask('projen:upgrade', {
            description: 'upgrades projen to the latest version',
            category: tasks_1.TaskCategory.MAINTAIN,
        });
        switch (project.package.packageManager) {
            case node_package_1.NodePackageManager.NPM:
                upgradeTask.exec('npm update projen');
                upgradeTask.exec('CI="" npm run projen');
                break;
            case node_package_1.NodePackageManager.YARN:
                upgradeTask.exec('yarn upgrade -L projen');
                upgradeTask.exec('CI="" yarn projen');
                break;
            case node_package_1.NodePackageManager.PNPM:
                upgradeTask.exec('pnpm update -L projen');
                upgradeTask.exec('CI="" pnpm run projen');
                break;
            default: throw new Error(`unexpected package manager ${project.package.packageManager}`);
        }
        if (options.autoUpgradeSecret) {
            if (!project.github) {
                throw new Error('github workflows are required in order for auto-update');
            }
            const workflow = project.github.addWorkflow('ProjenUpgrade');
            workflow.on({
                schedule: options.autoUpgradeSchedule
                    ? options.autoUpgradeSchedule.map(s => ({ cron: s }))
                    : [{ cron: '0 6 * * *' }],
                workflow_dispatch: {},
            });
            const withOptions = {
                'token': '${{ secrets.' + options.autoUpgradeSecret + ' }}',
                'commit-message': 'chore: upgrade projen',
                'branch': 'auto/projen-upgrade',
                'title': 'chore: upgrade projen',
                'body': 'This PR upgrades projen to the latest version',
            };
            if ((_a = options.labels) === null || _a === void 0 ? void 0 : _a.length) {
                withOptions.labels = options.labels.join(',');
            }
            workflow.addJobs({
                upgrade: {
                    'runs-on': 'ubuntu-latest',
                    'steps': [
                        // check out sources.
                        {
                            name: 'Checkout',
                            uses: 'actions/checkout@v2',
                        },
                        // install dependencies (and runs projen)
                        ...project.installWorkflowSteps,
                        // upgrade
                        {
                            name: 'Upgrade projen',
                            run: project.runTaskCommand(upgradeTask),
                        },
                        // submit a PR
                        {
                            name: 'Create pull request',
                            uses: 'peter-evans/create-pull-request@v3',
                            with: withOptions,
                        },
                    ],
                },
            });
        }
    }
}
exports.ProjenUpgrade = ProjenUpgrade;
//# sourceMappingURL=data:application/json;base64,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