"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve('../../package.json'));
async function synth(runtime, options) {
    var _a;
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, (_a = options.rcfile) !== null && _a !== void 0 ? _a : common_1.PROJEN_RC); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.PROJEN_RC)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find(t => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn('Default task skipped. Trying legacy synthesis since --rc is specified');
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        logging.info(`Watching for changes in ${workdir}...`);
        const watch = fs.watch(workdir, { recursive: true });
        watch.on('change', event => {
            // we only care about "change" events
            if (event !== 'change') {
                return;
            }
            process.stdout.write('\x1Bc'); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, 'node_modules');
        const projenModulePath = path.resolve(nodeModules, 'projen');
        if (!fs.existsSync(path.join(projenModulePath, 'package.json')) || !fs.statSync(projenModulePath).isDirectory()) {
            fs.removeSync(projenModulePath);
            fs.mkdirpSync(nodeModules);
            fs.symlinkSync(projenModule, projenModulePath, (os.platform() === 'win32') ? 'junction' : null);
        }
        child_process_1.spawnSync(process.execPath, [rcfile], { stdio: 'inherit' });
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,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