"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const inquirer = require("inquirer");
const yargs = require("yargs");
const inventory = require("../../inventory");
const render_options_1 = require("../../javascript/render-options");
const logging = require("../../logging");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(true);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
                            continue; // we don't support non-primitive fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(option.default);
                            }
                        }
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: option.type,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectType) {
            console.log(`Invalid project type ${args.projectType}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const js = render_options_1.renderJavaScriptOptions({
        bootstrap: true,
        comments: false,
        type: opts.type,
        args: opts.params,
    });
    const newProjectCode = `const project = new ${opts.type.typename}(${js});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // collect projects by looking up all .jsii modules in `node_modules`.
    const modulesDir = path.join(baseDir, 'node_modules');
    const modules = fs.readdirSync(modulesDir).map(file => path.join(modulesDir, file));
    const projects = inventory
        .discover(...modules)
        .filter(x => x.moduleName !== 'projen'); // filter built-in project types
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectType;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: [specDependencyInfo],
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments,
        synth: args.synth,
        post: args.post,
    });
    // interactive git and github setup
    const gitFolder = path.resolve(baseDir, '.git');
    let pushInitialToGithub = false;
    if (!fs.existsSync(gitFolder)) {
        pushInitialToGithub = await askAboutGit(baseDir);
    }
    if (pushInitialToGithub) {
        util_1.exec('git add .', { cwd: baseDir });
        util_1.exec('git commit -m \'Initial commit generated by projen\'', { cwd: baseDir });
        util_1.exec('git branch -M main', { cwd: baseDir });
        util_1.exec('git push --set-upstream origin main', { cwd: baseDir });
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const packageDir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        const copy = path.join(packageDir, 'pkg.tgz');
        fs.copyFileSync(spec, copy);
        spec = copy;
        dependencyInfo = `${moduleName}@${spec}`;
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
/**
 * Returns the last path element for use as a repository name default.
 */
function repoName() {
    return path.basename(path.basename(process.cwd()));
}
async function askAboutGit(cwd) {
    logging.info('We notice that you do not have a local git repository.');
    const { setUpGit } = await inquirer.prompt([
        {
            type: 'confirm',
            name: 'setUpGit',
            message: 'Do you want to set that up now?',
        },
    ]);
    if (setUpGit) {
        const { plan } = await inquirer.prompt(githubPlanOptions);
        const { gh, git } = plan;
        if (!git && !gh) {
            util_1.exec('git init', { cwd });
            logging.info('Ok, we have run \'git init\' for you! Have a great day.');
        }
        if (git) {
            const { gitRepoURL } = await inquirer.prompt([
                {
                    type: 'input',
                    name: 'gitRepoURL',
                    message: 'What is the repo? (example: https://github.com/projen/projen)',
                },
            ]);
            util_1.exec('git init', { cwd });
            let formattedGitRepoURL = gitRepoURL;
            if (!gitRepoURL.includes('https')) {
                formattedGitRepoURL = `https://github.com/${gitRepoURL}`;
            }
            util_1.exec(`git remote add origin ${formattedGitRepoURL}`, { cwd });
            logging.info(`Great! We have run 'git init' for you and set the remote to ${formattedGitRepoURL}`);
        }
        if (!git && gh) {
            logging.info('Ok! We will make you a repository on GitHub.');
            const ghCLIPath = util_1.execOrUndefined(`${os.platform() === 'win32' ? 'where' : 'which'} gh`, { cwd });
            if (!ghCLIPath) {
                logging.warn('Looks like you do not have the GitHub CLI installed. Please go to https://cli.github.com/ to install and try again.');
            }
            else {
                const { gitProjectName } = await inquirer.prompt([
                    {
                        type: 'input',
                        name: 'gitProjectName',
                        message: 'What would you like to name it?',
                        default: repoName(),
                    },
                ]);
                logging.info(`Wow! ${gitProjectName} is such a great name!`);
                util_1.exec('git init', { cwd });
                util_1.exec(`gh repo create ${gitProjectName}`, { cwd });
                return true;
            }
        }
    }
    return false;
}
const githubPlanOptions = [
    {
        type: 'list',
        name: 'plan',
        message: 'We\'ll need some more info. Please choose one:',
        choices: [
            {
                value: {
                    git: true,
                },
                name: 'I already have a git repository',
            },
            {
                value: {
                    gh: true,
                    git: false,
                },
                name: 'I don\'t have a git repository and want to make one on GitHub',
            },
            {
                value: {
                    gh: false,
                    git: false,
                },
                name: 'I don\'t have a git repository and I don\'t want to use GitHub',
            },
        ],
    },
];
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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